/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef CALIB_CHAN_LIST_HH
#define CALIB_CHAN_LIST_HH
#include <string>

//======================================  XML file field names
#define CALXML_CHANNEL_PARAM    "Channel"
#define CALXML_VERSION_PARAM    "Version"
#define CALXML_VERSNID_PARAM    "VersionID"
#define CALXML_COMMENT_PARAM    "Comment"
#define CALXML_START_TIME_PARAM "StartTime"
#define CALXML_CALIB_TIME_PARAM "CalibrationTime"
#define CALXML_DURATION_PARAM   "Duration"

//======================================  Frame file field names
#define CALFRAME_VERSION_NAME "CalibrationVersion"
#define CALFRAME_COMMENT_NAME "CalibrationComment"


/**  The CalibChanList class generates a set of channel names to be used for 
  *  in calibration frames. A channel of the form \c Xn:LSC-\<chan\> is 
  *  specified and the derived channel names of the form 
  *  \c Xn:CAL-\<type\>_\<chan\> are generated.
  *  \brief Calibration frame channel name list.
  *  \author John Zweizig
  *  \version 1.0; Last Moodified: August 25, 2005
  */
class CalibChanList {
public:
    /**  Output format enumerator.
      */
    enum cal_format {
        cfmt_default,  ///<  Default clibration format
	cfmt_xml,      ///<  XML calibration file
	cfmt_framev1,  ///<  Frame v1 calibration file.
	cfmt_framev2   ///<  Frame v2 calibration file.
    };

    /**  Construct a channel list.
      *  \brief Channel constructor.
      *  \param chan Channel name string.
      */
    CalibChanList(const std::string& chan);

    /**  Destroy the channel list.
      *  \brief Destructor
      */
    ~CalibChanList(void);

    /**  Get the Sensing transfer function Channel name.
      *  \brief Get sensing transfer function channel
      *  \return Pointer to sensing transfer function channel name string.
      */
    const char* getSensing(void) const;
 
    /**  Get the response transfer functionchannel name
      *  \brief Get response transfer function channel
      *  \return Pointer to response transfer function channel name string.
      */
    const char* getResponse(void) const;

    /**  Get the open loop gain transfer function channel name.
      *  \brief Get OLG transfer function channel
      *  \return Pointer to OLG transfer function channel name string.
      */
    const char* getOLoopGain(void) const;

    /**  Get the cavity factor (alpha) time series channel name.
      *  \brief Get cavity factor channel
      *  \return Pointer to cavity factor channel name string.
      */
    const char* getCavFac(void) const;

    /**  Get the open loop gain factor (alpha*beta) time series channel name.
      *  \brief Get OLG factor channel
      *  \return Pointer to OLG factor channel name string.
      */
    const char* getOLoopFac(void) const;

    /**  Set the name of the channel to be calibrated and generate all the
      *  associated channel names.
      *  \brief Set channel name
      *  \param chan Channel name string
      */
    void setChannel(const std::string& chan);

    /**  Test whether the name of the channel to be calibrated has been 
      *  specified.
      *  \brief Test if defined.
      *  \return Trues if the channel has been defined.
      */
    bool operator!(void) const;
private:
    std::string mChannel;
    std::string mSensing;
    std::string mResponse;
    std::string mOLoopGain;
    std::string mCavFac;
    std::string mOLoopFac;
};

//======================================  Inline accessors
inline const char*
CalibChanList::getSensing(void) const {
    return mSensing.c_str();
}

inline const char*
CalibChanList::getOLoopGain(void) const {
    return mOLoopGain.c_str();
}

inline const char* 
CalibChanList::getResponse(void) const {
    return mResponse.c_str();
}

inline const char* 
CalibChanList::getCavFac(void) const {
    return mCavFac.c_str();
}

inline const char* 
CalibChanList::getOLoopFac(void) const {
    return mOLoopFac.c_str();
}

inline bool 
CalibChanList::operator!(void) const {
    return !mChannel.empty();
}

#endif  // !defined(CALIB_CHAN_LIST_HH)
