/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DMT base class methods.
//

//-------------------------------------  Header files.
#include <time.h>
#include <iostream>
#include "SigFlag.hh"
#include "DMTBase.hh"
#include "Interval.hh"
#include "Time.hh"

#include "framecpp/FrameH.hh"
#include "ldas/ldasconfig.hh"
#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif
#include "Dacc.hh"

using namespace std;

//-------------------------------------  DMTBase constructor.
DMTBase::DMTBase(int argc, const char *argv[]) 
  : mActive(false), mDebug(0), mContig(false), mNFrames(0), mNContig(0), 
    mEOF(false)
{
    mIn = new Dacc;

    char parfile[64];
    strcpy(parfile, DACC_ONLDEV);
    if (getenv("LIGOSMPART")) strcat(parfile, getenv("LIGOSMPART"));
    else                      strcat(parfile, "LIGO_Online");
    const char* file = getenv("DMTINPUT");
    if (!file)  file = parfile;

    //---------------------------------  Parse the arguments
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi == "-partition") {
	    strcpy(parfile+strlen(DACC_ONLDEV), argv[++i]);
	    mIn->addFile(parfile);
	    file = 0;
	} else if (argi == "-infile") {
	    if (++i >= argc) return;
	    mIn->addFile(argv[i]);
	    file = 0;
	} else if (argi == "-inlist") {
	    if (++i >= argc) return;
	    mIn->addFileList(argv[i]);
	    file = 0;
	} else if (argi == "-debug") {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	    mIn->setDebug(mDebug);
	}
    }
    if (file) mIn->addFile(file);

    //---------------------------------  Handle signals
    mTerm = new SigFlag(SIGTERM);
    if (!mTerm) return;
    mTerm->add(SIGHUP);
    mTerm->add(SIGINT);

    mAttn = new SigFlag(SIGUSR1, SigFlag::kBlock);

    //---------------------------------  Open a frame stream (FrameCPP)
    mIn->setDebug(Debug());
    // if (mIn->open()) return;
    mActive = true;
}

bool
DMTBase::isEnvArg(const char* arg) const {
    if (!strcmp("-partition", arg)) return true;
    if (!strcmp("-infile", arg))    return true;
    if (!strcmp("-inlist", arg))    return true;
    if (!strcmp("-debug", arg))     return true;
    return false;
}

//-------------------------------------  DMTBase object destructor.
DMTBase::~DMTBase() 
{
    //---------------------------------  DMT has terminater
    cout << "DMT has terminated with Term/Attn/finish/EOF =" << bool(*mTerm) 
	 << "/" << bool(*mAttn) << "/" << !mActive << "/" << mEOF << endl;

    //---------------------------------  Release signals
    if (mTerm) delete mTerm;
    mTerm = 0;

    if (mAttn) delete mAttn;
    mAttn = 0;

    //---------------------------------  Close the Input file/partition.
    try {
        mIn->DaccIn::close();
    } catch (exception& e) {
        cerr << "Exception in frame destructors:" << e.what() << endl;
    } catch (...) {
        cerr << "Unknown exception in frame destructors." << endl;
    }

    //---------------------------------  Print statistics
    if (mDebug != 999) {
        cout << "Frame reading terminated after " << mNFrames 
	     << " frames and " << mNContig << " boundaries." << endl;
    }
}

//-------------------------------------  Main processing loop function.
void
DMTBase::MainLoop(void) 
{
    //---------------------------------  Loop until terminated.
    int  FNext = 0;
    Time TNext(0);
    while(!(*mTerm) && mActive) {

        //-----------------------------  Check for an attention interrupt.
        if (*mAttn) {
	    mAttn->clear();
	    Attention();
	    if (!mActive) continue;
	}

        //-----------------------------  Read a frame
	mIn->nextFrame();
        frame_ptr_type frame = mIn->getFrame();
	if (frame) {

	    //-------------------------  Check continuity
	    int      FrameID = frame->GetFrame();
	    GPSTime TimeCPP=frame->GetGTime();
	    Time     FTime(TimeCPP.getSec(), TimeCPP.getNSec());
	    Interval dT(frame->GetDt());
	    mContig = !mNFrames || (FNext != FrameID) || (TNext != FTime);
	    if (mContig) {
	        mNContig++;
	        if (Debug() && mNFrames) {
		    cout << "Frame " << FrameID << " (GPS " << FTime 
			 << ") found when frame " << FNext << " (GPS " 
			 << TNext << ") was expected." << endl;
		}
	    }
	    FNext = FrameID + 1;
	    TNext = FTime   + dT;

	    //------------------------  Process a frame.
	    mNFrames++;
	    try {
	        ProcessFrame(frame);
	    } catch (exception& e) {
	        cerr << "Error in ProcessFrame. " << e.what() << endl;
	    } catch (...) {
	        cerr << "Error in ProcessFrame caught." << endl;
	    }

	    //------------------------  Delete the frame.
	    mIn->endFrame();
	} else {
	    cerr << "MainLoop: Error while reading frame." << endl;
	    mEOF = true;
	    break;
	}
    }
}
void 
DMTBase::setBuffer(int nbuf) {
    mIn->setBuffer(nbuf);
}
