/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "Histogram2.hh"
#include <iomanip>
#include <cmath>
#include <cstring>
#include <iostream>

   using namespace std;

//-------------------------------------- Data Constructor
   Histogram2::Histogram2(const char* name, int nbinx, xbin_t xmin, xbin_t xmax,
                     int nbiny, xbin_t ymin, xbin_t ymax, const char* xlabel, 
                     const char* ylabel,const char* nlabel):
   fArray(0),fSumw2(0),fXbins(0),fYbins(0)
   {
      Reset();
   
      fTitle = name;
      fXLabel = xlabel;
      fYLabel = ylabel;
      fNLabel = nlabel;
   
      SetBinLowEdges(nbinx,xmin,xmax,nbiny,ymin,ymax);
   }

//-------------------------------------- Data Constructor
   Histogram2:: Histogram2(const char* name, int nbinx, xbin_t* xbins,
                     int nbiny, xbin_t* ybins, const char* xlabel, 
                     const char* ylabel,const char* nlabel):
   fArray(0),fSumw2(0),fXbins(0),fYbins(0)
   {
      Reset();
   
      fTitle = name;
      fXLabel = xlabel;
      fYLabel = ylabel;
      fNLabel = nlabel;
   
      SetBinLowEdges(nbinx,xbins,nbiny,ybins);
   
   }

//-------------------------------------- Destructor
   Histogram2::~Histogram2()
   {
      if(fArray) delete[] fArray;
      if(fSumw2) delete[] fSumw2;
      if(fXbins) delete[] fXbins;
      if(fYbins) delete[] fYbins;
   }

//-------------------------------------- Find bin number (internal use only)
   int Histogram2::SearchBin(int min, int max, xbin_t val, int axis) const
   {
   
      double* edge;
      if (axis) edge = fYbins;
      else edge = fXbins;
   
      if ( val >= edge[min] && val < edge[min+1] ) 
         return min;
      else if ( val >= edge[min] && val < edge[(max+min)/2+1] ){
         return SearchBin(min , (max+min)/2, val, axis);
      }
      return  SearchBin((max+min)/2+1 , max, val, axis);
   }

//-------------------------------------- Allocate memory (internal use only)
   void Histogram2::Allocate(int nbinx, int nbiny)
   {
      if (fArray) delete[] fArray;
      if (fXbins) delete[] fXbins;
      if (fYbins) delete[] fYbins;
      if (fSumw2) delete[] fSumw2;
   
      fArray = 0;
      fSumw2 = 0;
      fXbins = 0;
      fYbins = 0;
      fNbinx = nbinx;
      fNbiny = nbiny;
   
      if(nbinx && nbiny){
         fArray = new histdata_t[(nbinx+2) * (nbiny+2)];
         fXbins = new xbin_t[nbinx+1];
         fYbins = new xbin_t[nbiny+1];
         memset(fArray,0,((nbinx+2) * (nbiny+2)) * sizeof(histdata_t));
         memset(fXbins,0,(nbinx+1) * sizeof(xbin_t));
         memset(fYbins,0,(nbiny+1) * sizeof(xbin_t)); 
      }
   }

//======================================

   void Histogram2::Clear()
   {
      if (fNbinx * fNbiny) {
         memset(fArray,0,(fNbinx+2)*(fNbiny+2)*sizeof(histdata_t));
         if (fBinErrorFlag) {
            memset(fSumw2,0,(fNbinx+2)*(fNbiny+2)*sizeof(stat_t));
         }
      }
   
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0;
      fTsumwy = fTsumwy2 = fTsumwxy = 0;
   
      fNEntries = 0;
      fTime = Time(0);
   }

   ostream&
   Histogram2::Dump(ostream& out) const
   {
      out<<"Title                      : "<<fTitle<<endl
         <<"XLabel                     : "<<fXLabel<<endl
         <<"XLabel                     : "<<fYLabel<<endl
         <<"NLabel                     : "<<fNLabel<<endl
         <<"GPS Time                   : "<<fTime.totalS()<<endl
         <<"# of Entries               : "<<fNEntries<<endl
         <<"# of X Bins                : "<<fNbinx<<endl
         <<"# of Y Bins                : "<<fNbiny<<endl
         <<"Bin Type                   : ";
      switch (fBinType) {
         case kUndefinedBin:
            out<<"Undefined"<<endl;
            break;
         case kFixedBin:
            out<<"Fixed"<<endl;
            break;
         default:
            out<<"Variable"<<endl;
      }
      out<<"Sum of Weights             : "<<fTsumw<<endl
         <<"Sum of Weights^2           : "<<fTsumw2<<endl
         <<"Sum of Weights*xdata       : "<<fTsumwx<<endl
         <<"Sum of Weights*xdata^2     : "<<fTsumwx2<<endl
         <<"Sum of Weights*ydata       : "<<fTsumwy<<endl
         <<"Sum of Weights*ydata^2     : "<<fTsumwy2<<endl
         <<"Sum of Weights*xdata*ydata : "<<fTsumwxy<<endl;
      if (fBinErrorFlag) out<<"Bin Error ON"<<endl;
      else out<<"Bin Error OFF"<<endl;
   
      if (fNbinx && fNbiny) {
         out << "Contents:" << endl;
         for (int j = fNbiny+1; j >= 0; j--){
            for (int i = 0; i < fNbinx+2; i++){
               if ( i == 0 ) {
                  if ( j == 0 )
                     out << setw(5) << "UF" << "|";
                  else out << setw(5) << fYbins[j - 1] << "|";
               }
               out << setw(5) << fArray[i + j * (fNbinx+2)] << " ";
               if ( i == fNbinx+1) out << endl;
            }
         }
         for (int i = 0; i <= fNbinx+2; i++) {
            out << "-----+";
            if ( i == fNbinx+2 ) out << endl;
         }
         for (int i = 0; i <= fNbinx+2; i++) {
            if ( i == 0 ) out << setw(5) << "Y/X" << "|";
            else if ( i == 1 ) out << setw(5) << "UF" << " ";
            else out << setw(5) << fXbins[i - 2] << " ";
            if ( i == fNbinx+2 ) out << endl;
         }
      
         if (fBinErrorFlag) {
            out << setw(8) << "Error^2:" << endl;
            for (int j = fNbiny+1; j >= 0; j--){
               for (int i = 0; i < fNbinx+2; i++){
                  if ( i == 0 ) {
                     if ( j == 0 )
                        out << setw(5) << "UF" << "|";
                     else out << setw(5) << fYbins[j - 1] << "|";
                  }
                  out << setw(5) << fSumw2[i + j * (fNbinx+2)] << " ";
                  if ( i == fNbinx+1) out << endl;
               }
            }
            for (int i = 0; i <= fNbinx+2; i++) {
               out << "-----+";
               if ( i == fNbinx+2 ) out << endl;
            }
            for (int i = 0; i <= fNbinx+2; i++) {
               if ( i == 0 ) out << setw(5) << "Y/X" << "|";
               else if ( i == 1 ) out << setw(5) << "UF" << " ";
               else out << setw(5) << fXbins[i - 2] << " ";
               if ( i == fNbinx+2 ) out << endl;
            }
         }
      }
      else out<<"No Data"<<endl;
   
      return out;
   }

   void Histogram2::Fill(xbin_t x, xbin_t y, double w)
   {
      int xindex, yindex;
   
      if (fBinType == kFixedBin){  // fixed bin
         if (x >= fXbins[fNbinx]) xindex = fNbinx+1;
         else if (x < fXbins[0])  xindex = 0;
         else {
            xindex = 1 + (int)(( x - fXbins[0] ) * 
                              fNbinx / ( fXbins[fNbinx] - fXbins[0] ));
         }
         if (y >= fYbins[fNbiny]) yindex = fNbiny+1;
         else if (y < fYbins[0])  yindex = 0;
         else {
            yindex = 1 + (int)(( y - fYbins[0] ) * 
                              fNbiny / ( fYbins[fNbiny] - fYbins[0] ));
         }
      } // variable bin
      else if (fBinType == kVariableBin){
         xindex = GetBinNumber(x,kXAxis);
         yindex = GetBinNumber(y,kYAxis);
      }
      else 
         return;
   
      fArray[xindex + yindex * (fNbinx + 2)] += w;
      if (fBinErrorFlag) fSumw2[xindex + yindex * (fNbinx + 2)] += w*w;
      fNEntries++;
      if (xindex > 0 && xindex < fNbinx+1 &&
         yindex > 0 && yindex < fNbiny+1){
         fTsumw += w;
         fTsumw2 += w*w;
         fTsumwx += w*x;
         fTsumwx2 += w*x*x;
         fTsumwy += w*y;
         fTsumwy2 += w*y*y;
         fTsumwxy += w*x*y;
      }
   
   }

   void Histogram2::FillN(int ntimes, const xbin_t* x, const xbin_t* y)
   {
      for ( int i = 0; i < ntimes; i++) Fill(x[i], y[i]);
   }

   void Histogram2::FillN(int ntimes, const xbin_t* x, 
                     const xbin_t* y, const double* w)
   {
      for ( int i = 0; i < ntimes; i++) Fill(x[i], y[i], w[i]);
   }

   void Histogram2::GetBinContents(histdata_t* data) const
   {
      for( int i = 0 ; i < (fNbinx+2) * (fNbiny+2); i++) data[i] = fArray[i];
   }

   Histogram2::histdata_t 
   Histogram2::GetBinContent(int xbin, int ybin) const
   {
      int xindex, yindex;
   
      if (xbin < 0) xindex = 0;
      else if (xbin > fNbinx+2) xindex = fNbinx+1;
      else xindex = xbin;
   
      if (ybin < 0) yindex = 0;
      else if (ybin > fNbiny+2) yindex = fNbiny+1;
      else yindex = ybin;
   
      return fArray[xindex + yindex * (fNbinx + 2)];
   }

   Histogram2::xbin_t 
   Histogram2::GetBinCenter(int n, int axis) const
   {
      if (axis == kXAxis) {
         if (n < 1) n = 1;
         else if (n >= fNbinx+1) n = fNbinx;
      }
      else if (axis == kYAxis) {
         if (n < 1) n = 1;
         else if (n >= fNbiny+1) n = fNbiny;
      }
   
      return (fXbins[n-1] + fXbins[n])/2; 
   }

   void Histogram2::GetBinLowEdges(xbin_t* data, int axis) const
   {
      if (axis == kXAxis) {
         for( int i = 0 ; i < fNbinx+1 ; i++) data[i] = fXbins[i];
      }
      else if (axis == kYAxis) {
         for( int i = 0 ; i < fNbiny+1 ; i++) data[i] = fYbins[i];
      }
      else data = 0;
   }

   Histogram2::xbin_t 
   Histogram2::GetBinLowEdge(int n, int axis) const
   {
      if (axis == kXAxis) {
         if (n < 1) n = 1;
         else if (n >= fNbinx+2) n = fNbinx;
         return fXbins[n - 1];
      }
      else if (axis == kYAxis) {
         if (n < 1) n = 1;
         else if (n >= fNbiny+2) n = fNbiny;
         return fYbins[n - 1];
      }
   
      return 0;
   }

   bool Histogram2::GetBinErrors(stat_t* err) const
   {
      if (fBinErrorFlag) {
         for( int i = 0 ; i < (fNbinx+2) * (fNbiny+2); i++) { 
            err[i] = sqrt(fSumw2[i]);
         }
      }
      return fBinErrorFlag;
   }

   Histogram2::stat_t 
   Histogram2::GetBinError(int xbin, int ybin) const
   {
      int xindex, yindex;
   
      if (xbin < 0) xindex = 0;
      else if (xbin > fNbinx+2) xindex = fNbinx+1;
      else xindex = xbin;
   
      if (ybin < 0) yindex = 0;
      else if (ybin > fNbiny+2) yindex = fNbiny+1;
      else yindex = ybin;
   
      return sqrt(fSumw2[xindex + yindex * (fNbinx+2)]);
   }

   int Histogram2::GetBinNumber(xbin_t val, int axis) const
   {
      if (axis == kXAxis) {
         if ( val < fXbins[0] ) 
            return 0;
         else if ( val > fXbins[fNbinx]) 
            return fNbinx+1;
         return SearchBin(0,fNbinx-1,val,axis) + 1;
      }
      else if (axis == kYAxis) {
         if ( val < fYbins[0] ) 
            return 0;
         else if ( val > fYbins[fNbiny]) 
            return fNbiny+1;
         return SearchBin(0,fNbiny-1,val,axis) + 1;
      }
   
      return 0;
   }

   Histogram2::xbin_t 
   Histogram2::GetBinSpacing(int axis) const
   {
      if (fBinType == kFixedBin) {
         if (axis == kXAxis) 
            return (fXbins[fNbinx] - fXbins[0])/fNbinx;
         else if (axis == kYAxis) 
            return (fYbins[fNbiny] - fYbins[0])/fNbiny;
      }
   
      return 0;
   }

   int Histogram2::GetNBins(int axis) const
   {
      if (axis == kXAxis) 
         return fNbinx;
      else if (axis == kYAxis) 
         return fNbiny;
   
      return 0;
   }

   const char* Histogram2::GetTitle(void) const 
   {
      if ( fTitle.size() ) 
         return fTitle.c_str();
      else 
         return 0;
   }
   const char* Histogram2::GetXLabel(void) const 
   {
      if ( fXLabel.size() ) 
         return fXLabel.c_str();
      else 
         return 0;
   }

   const char* Histogram2::GetYLabel(void) const 
   {
      if ( fYLabel.size() ) 
         return fYLabel.c_str();
      else 
         return 0;
   }

   const char* Histogram2::GetNLabel(void) const 
   {
      if ( fNLabel.size() ) 
         return fNLabel.c_str();
      else 
         return 0;
   }

   Histogram2::histdata_t 
   Histogram2::GetMinContent(void) const 
   {
      int xbin, ybin;
      return GetMinContentBin(xbin, ybin);
   }

   Histogram2::histdata_t 
   Histogram2::GetMaxContent(void) const
   {
      int xbin, ybin;
      return GetMaxContentBin(xbin, ybin);
   }

   Histogram2::histdata_t 
   Histogram2::GetMinContentBin(int& xbin, int& ybin) const
   {
      histdata_t min = fArray[1 + (fNbinx + 2)];
      xbin = ybin = 1;
      for (int j = 1; j < fNbiny+1; j++) {
         for ( int i = 1; i < fNbinx+1; i++) {
            if ( fArray[i + j * (fNbinx+2)] < min ) {
               min = fArray[i + j * (fNbinx+2)];
               xbin = i;
               ybin = j;
            }
         }
      }
      return min;
   }

   Histogram2::histdata_t 
   Histogram2::GetMaxContentBin(int& xbin, int& ybin) const
   {
      histdata_t max = fArray[1 + (fNbinx + 2)];
      xbin = ybin = 1;
      for (int j = 1; j < fNbiny+1; j++) {
         for ( int i = 1; i < fNbinx+1; i++) {
            if ( fArray[i + j * (fNbinx+2)] > max ) {
               max = fArray[i + j * (fNbinx+2)];
               xbin = i;
               ybin = j;
            }
         }
      }
      return max;
   }

   Histogram2::stat_t 
   Histogram2::GetMean(int axis) const
   {
      stat_t stats[7];
      GetStats(stats);
      if (stats[0] == 0) 
         return 0;
   
      if (axis == kXAxis) {
         return stats[2] / stats[0];
      }
      else if (axis == kYAxis) {
         return stats[4] / stats[0];
      }
      else 
         return 0;
   }

   Histogram2::stat_t 
   Histogram2::GetSdev(int axis) const
   {
      stat_t stats[7];
      GetStats(stats);
      if (stats[0] == 0) 
         return 0;
      if (axis == kXAxis) {
         return sqrt(stats[3]/stats[0]-stats[2]*stats[2]/(stats[0]*stats[0]));
      }
      else if (axis == kYAxis) {
         return sqrt(stats[5]/stats[0]-stats[4]*stats[4]/(stats[0]*stats[0]));
      }
      else 
         return 0;
   }

   void Histogram2::GetStats(stat_t *stats) const 
   {
      histdata_t x,y,w;
   
      if(fTsumw == 0 ){
         for(int i = 0; i < 7; i++ ) stats[i]=0;
         for (int j = 1; j < fNbiny+1; j++) {
            for (int i = 1; i < fNbinx+1; i++ ){
               x = GetBinCenter(i,kXAxis);
               y = GetBinCenter(j,kYAxis);
               w = GetBinContent(i,j);
               stats[0] += w;
               stats[1] += w*w;
               stats[2] += w*x;
               stats[3] += w*x*x;
               stats[4] += w*y;
               stats[5] += w*y*y;
               stats[6] += w*x*y;
            }
         }
      }
      else {
         stats[0] = fTsumw;
         stats[1] = fTsumw2;
         stats[2] = fTsumwx;
         stats[3] = fTsumwx2;
         stats[4] = fTsumwy;
         stats[5] = fTsumwy2;
         stats[6] = fTsumwxy;
      }
   }

   void Histogram2::PutStats(const stat_t *stats)
   {
      fTsumw   = stats[0];
      fTsumw2  = stats[1];
      fTsumwx  = stats[2];
      fTsumwx2 = stats[3];
      fTsumwy  = stats[4];
      fTsumwy2 = stats[5];
      fTsumwxy = stats[6];
   }

   Histogram1* Histogram2::Projection(const char* title, int axis,
                     int firstbin, int lastbin)
   {
      int nbin;
      xbin_t* edge;
   
      std::string axislabel;
   
      if (axis == kXAxis) {
         nbin = fNbinx;
         edge = fXbins;
         if (firstbin < 0) firstbin = 0;
         if (lastbin > fNbiny+1) lastbin = fNbiny+1;
         axislabel = fXLabel.c_str();
      }
      else if (axis == kYAxis) {
         nbin = fNbiny;
         edge = fYbins;
         if (firstbin < 0) firstbin = 0;
         if (lastbin > fNbinx+1) lastbin = fNbinx+1;
         axislabel = fYLabel.c_str();
      }
      else 
         return 0;
   
      Histogram1* h1;
      if (fBinType == kFixedBin) {
         h1 = new Histogram1(title,nbin,edge[0],edge[fNbinx],
                            axislabel.c_str(),fNLabel.c_str());
      }
      else if (fBinType == kVariableBin) {
         h1 = new Histogram1(title,nbin,edge,
                            axislabel.c_str(),fNLabel.c_str());
      }
      else h1 = 0;
   
      if (h1) {
         if (fBinErrorFlag) h1->Sumw2();
         for (int j = firstbin; j <= lastbin; j++) {
            for (int i = 0; i < nbin+2; i++) {
               int index = (axis == kXAxis) ? i + j * (nbin+2) : j + i * (nbin+2);
               if (fArray[index]) {
                  if (i == 0 ) h1->Fill(edge[0]-1,fArray[index]);
                  else if (i == nbin+1 ) h1->Fill(edge[nbin]+1,fArray[index]);
                  else 
                     h1->Fill(GetBinCenter(i,axis),fArray[index]);
               }
            }
         }
      
         stat_t stat[7];
         if (axis == kXAxis) GetStats(stat);
         else {
            GetStats(stat);
            stat[2] = fTsumwy;
            stat[3] = fTsumwy2;
         }
         h1->PutStats(stat);
      
         h1->SetNEntries(fNEntries);
      
      }
   
      return h1;
   }

   void Histogram2::SetBinLowEdges(int nbinx, xbin_t xmin, xbin_t xmax,
                     int nbiny, xbin_t ymin, xbin_t ymax)
   {
      fNEntries = 0;
      // fNbinx = nbinx;
      // fNbiny = nbiny;
   
      Allocate(nbinx,nbiny);
      if ( !(fNbinx * fNbiny) ) 
         return;
   
      double dx = (xmax - xmin)/nbinx;
      for(int i=0;i<fNbinx+1;i++) fXbins[i] = xmin + dx * i;
   
      double dy = (ymax - ymin)/nbiny;
      for(int i=0;i<fNbiny+1;i++) fYbins[i] = ymin + dy * i;
   
      fBinType = kFixedBin;
   
   }

   void Histogram2::SetBinLowEdges(int nbinx, const xbin_t* xbins, 
                     int nbiny, const xbin_t* ybins)
   {
      fNEntries = 0;
      // fNbinx = nbinx;
      // fNbiny = nbiny;
   
      Allocate(nbinx,nbiny);
      if ( !(fNbinx * fNbiny) ) 
         return;
   
      memcpy(fXbins,xbins,(nbinx+1)*sizeof(xbin_t));
      memcpy(fYbins,ybins,(nbiny+1)*sizeof(xbin_t));
   
      fBinType = kVariableBin;
   }

   void Histogram2::SetBinContents(const histdata_t* data) 
   {
      memcpy(fArray,data,(fNbinx+2) * (fNbiny+2) * sizeof(histdata_t));
   }

   bool Histogram2::SetBinContent(int x, int y, histdata_t content)
   {
      if ((x < 0 || x > fNbinx+1) || (y < 0 || y > fNbiny+1))
         return false;
   
      fArray[x + y * (fNbinx+2)] = content;
      return true;
   }

   void Histogram2::SetBinErrors(const stat_t* err)
   {
      Sumw2();
      for (int i = 0; i < fNbinx+2; i++){
         for (int j = 0; j < fNbiny+2; j++){
            SetBinError(i,j,err[i + j * (fNbinx+2)]);
         }
      }
   }

   bool Histogram2::SetBinError(int x, int y, stat_t err)
   {
      if (fBinErrorFlag){
         if ((x < 0 || x > fNbinx+1) || (y < 0 || y > fNbiny+1))
            return false;
         fSumw2[x + y * (fNbinx+2)] = err * err;
         return true;
      }
      return false;
   }

   void Histogram2::SetNBins(int nbin, int axis)
   {
      if (axis == kXAxis) fNbinx = nbin;
      else if (axis == kYAxis) fNbiny = nbin;
   
      return;
   }

   void Histogram2::SetBinType(int type)
   {
      switch(type){
         case 1:
            fBinType = kFixedBin;
            break;
         case 2:
            fBinType = kVariableBin;
            break;
         default:
            fBinType = kUndefinedBin;
      }
   }

   void Histogram2::Sumw2(bool reset)
   {
      if (fSumw2) delete[] fSumw2;
      if ( !(fNbinx * fNbiny) ) 
         return;
   
      fSumw2 = new stat_t[(fNbinx+2) * (fNbiny+2)];
   
      fBinErrorFlag = 1;
   
      if (reset) memset(fSumw2,0,(fNbinx+2) * (fNbiny+2) * sizeof(stat_t));
      else memcpy(fSumw2, fArray,(fNbinx+2) * (fNbiny+2) * sizeof(stat_t));
   }

   void Histogram2::Reset(void)
   {
      Allocate();
   
      fTitle = "";
      fNEntries = 0;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0.;
      fTsumwy = fTsumwy2 = fTsumwxy = 0.;
      fXLabel = "";
      fYLabel = "";
      fNLabel = "";
      fBinType = kUndefinedBin;
      fTime = Time(0);
      fBinErrorFlag = 0;
   }

   Histogram2& Histogram2::operator = (const Histogram2& h) 
   {
      if (this != &h){
         Reset();
      
         fTitle = h.fTitle;
      
         if ( fNbinx * fNbiny ) {
            Allocate ( h.fNbinx,  h.fNbiny );
            memcpy(fArray,h.fArray,(fNbinx+2) * (fNbiny+2) * sizeof(histdata_t));
         
            if (h.IsErrorFlagON()) {
               Sumw2();
               memcpy(fSumw2,h.fSumw2,(fNbinx+2) * (fNbiny+2) * sizeof(stat_t));
            }
         
            memcpy(fXbins,h.fXbins,(fNbinx+1)*sizeof(xbin_t));
            memcpy(fYbins,h.fYbins,(fNbiny+1)*sizeof(xbin_t));
         }
      
      
         fNEntries = h.fNEntries;
         fTsumw = h.fTsumw;
         fTsumw2 = h.fTsumw2;
         fTsumwx = h.fTsumwx;
         fTsumwx2 = h.fTsumwx2;
         fTsumwy = h.fTsumwy;
         fTsumwy2 = h.fTsumwy2;
         fTsumwxy = h.fTsumwxy;
         fXLabel = h.fXLabel;
         fYLabel = h.fYLabel;
         fNLabel = h.fNLabel;
         fBinType = h.fBinType;
         fTime = h.fTime;
      }
   
      return (*this);
   }

   Histogram2& Histogram2::operator += (const Histogram2& h)
   {
      Histogram2 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int i = 0; i < (fNbinx+2) * (fNbiny+2); i++) { 
         fArray[i] += h0.fArray[i];
         if (fBinErrorFlag) fSumw2[i] += h0.fSumw2[i];
      }
   
      stat_t s1[7],s2[7];
      GetStats(s1);
      h0.GetStats(s2);
   
      for (int i=0; i< 7; i++) { s1[i] += s2[i]; }
   
      fNEntries += h0.fNEntries;
   
      PutStats(s1);
   
      return (*this);
   }

   Histogram2& Histogram2::operator += (histdata_t bias)
   {
      for (int i = 0; i < (fNbinx+2) * (fNbiny+2); i++) fArray[i] += bias;
   
      fNEntries = fNbinx+2;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0.;
      fTsumwy = fTsumwy2 = fTsumwxy = 0.;
      stat_t stats[7];
      GetStats(stats);
      PutStats(stats);
   
      return (*this);
   }

   Histogram2& Histogram2::operator -= (const Histogram2& h)
   {
      Histogram2 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int i = 0; i < (fNbinx+2) * (fNbiny+2); i++) {
         fArray[i] -= h0.fArray[i];
         if (fBinErrorFlag) fSumw2[i] += h0.fSumw2[i];
      }
   
      stat_t s1[7],s2[7];
      GetStats(s1);
      h0.GetStats(s2);
   
      for (int i=0; i<7; i++) s1[i] += s2[i];
   
      fNEntries -= h0.fNEntries;
   
      PutStats(s1);
   
      return (*this);
   }

   Histogram2& Histogram2::operator *= (double scale)
   {
      stat_t stats[7];
      GetStats(stats);
      stats[0] *= scale;
      stats[1] *= ( scale * scale );
      stats[2] *= scale;
      stats[3] *= scale;
      stats[4] *= scale;
      stats[5] *= scale;
      stats[6] *= scale;
   
      PutStats(stats);
   
      for (int i = 0; i < (fNbinx+2) * (fNbiny+2); i++) {
         fArray[i] *= scale;
         if (fBinErrorFlag) fSumw2[i] *= ( scale * scale );
      }
      return (*this);
   }

   Histogram2& Histogram2::operator *= (const Histogram2& h)
   {
      Histogram2 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int j = 0; j < fNbiny+2; j++) {
         for (int i = 0; i < fNbinx+2; i++) {
            histdata_t c0 = fArray[i + j * (fNbinx+2)];
            histdata_t c1 = h0.fArray[i + j * (fNbinx+2)];
            stat_t e0 = GetBinError(i,j);
            stat_t e1 = h0.GetBinError(i,j);
            if (fBinErrorFlag) 
               fSumw2[i + j * (fNbinx+2)] = e0*e0*c1*c1+e1*e1*c0*c0;
            fArray[i + j * (fNbinx+2)] = c0*c1;
         }
      }
   
      fNEntries = fNbinx+2;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0.;
      fTsumwy = fTsumwy2 = fTsumwxy = 0.;
      stat_t stats[7];
      GetStats(stats); 
      PutStats(stats);
   
      return (*this);
   }

   Histogram2& Histogram2::operator /= (const Histogram2& h)
   {
      Histogram2 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int j = 0; j < fNbiny+2; j++) {
         for (int i = 0; i < fNbinx+2; i++) {
            histdata_t c0 = fArray[i + j * (fNbinx+2)];
            histdata_t c1 = h0.fArray[i + j * (fNbinx+2)];
            stat_t e0 = GetBinError(i,j);
            stat_t e1 = h0.GetBinError(i,j);
            if (!c1) { 
               if (fBinErrorFlag) fSumw2[i + j * (fNbinx+2)]=0;
               fArray[i + j * (fNbinx+2)]=0;
            }
            else {
               if (fBinErrorFlag) 
                  fSumw2[i + j * (fNbinx+2)] = (e0*e0*c1*c1+e1*e1*c0*c0)/(c1*c1*c1*c1);
               fArray[i + j * (fNbinx+2)] = c0/c1;
            }
         }
      }
   
      fNEntries = fNbinx+2;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0.;
      fTsumwy = fTsumwy2 = fTsumwxy = 0.;
      stat_t stats[7];
      GetStats(stats); 
      PutStats(stats);
   
      return (*this);
   }

//=================External Operator=====================


   Histogram2 operator + (Histogram2::histdata_t bias, const Histogram2& h)
   {
      Histogram2 hbias(h);
      hbias += bias;
   
      return hbias;
   
   }

   Histogram2 operator + (const Histogram2& h, Histogram2::histdata_t bias)
   {
      Histogram2 hbias(h);
      hbias += bias;
   
      return hbias;
   
   }

   Histogram2 operator + (const Histogram2& h1, const Histogram2& h2)
   {
      Histogram2 hsum(h1);
      hsum += h2;
   
      return hsum;
   }

   Histogram2 operator - (const Histogram2& h1, const Histogram2& h2)
   {
      Histogram2 hsub(h1);
      hsub -= h2;
   
      return hsub;
   }

   Histogram2 operator * (double scale, const Histogram2& h)
   {
      Histogram2 hscl(h);
      hscl *= scale;
   
      return hscl;
   }


   Histogram2 operator * (const Histogram2& h, double scale)
   {
      Histogram2 hscl(h);
      hscl *= scale;
   
      return hscl;
   }

   Histogram2 operator * (const Histogram2& h1, const Histogram2& h2)
   {
      Histogram2 hmlt(h1);
      hmlt *= h2;
   
      return hmlt;
   }

   Histogram2 operator / (const Histogram2& h1, const Histogram2& h2)
   {
      Histogram2 hdiv(h1);
      hdiv /= h2;
   
      return hdiv;
   }
