/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: pipe_exec						*/
/*                                                         		*/
/* Module Description: Executes a child process using pipes		*/
/*                     for communications                		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 17Apr01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: pipe_exec.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_PIPEEXEC_H
#define _LIGO_PIPEEXEC_H

#include "PConfig.h"
#ifdef P__SOLARIS
#include <time.h>
#endif
#include <iostream>
#include <string>


namespace std {


/** @name Child process with pipe communications. 
    This header defines a class which provides the p2open functionality.
   
    @memo Pipe Exec
    @author Written April 2001 by Daniel Sigg
    @version 1.0
 ************************************************************************/

//@{

/** Prog_exec.
    Starts a child process with fork and executes the string cmd
    with execlp.
    
    @memo system equivalent
 ************************************************************************/
   class prog_exec {
   public:
      /** Constructor.
          Sets up the pipes and starts the child process.
        */
      explicit prog_exec (const char* cmd);
      /** Destuctor.
          Sends a kill signal to the child 
        */
      virtual ~prog_exec();
      /// Returns true on error
      virtual bool operator!() {
         return fError; }
   
      /// Returns the comand string
      virtual const char* cmd() const {
         return fCmd.c_str(); }
      /// Return the child process ID
      virtual int pid() const {
         return fPID; }
      /** Wait for child to terminate (returns exit code)
          If poll is true, the method will not block and return
          0 in case the child process is still alive. This
          method will return the same as waitpid and set the
          return status if stat_loc is non-zero.
        */
      virtual int wait (int* stat_loc = 0, bool poll = false);
      /** Wait for child to terminate with timeout.
          If the time expires and the child process is still alive,
          the mothod will return 0. Otherwise, this 
          method will return the same as waitpid and set the
          return status if stat_loc is non-zero. The timeout is
          in seconds. A timeout of 0 is equal to polling and a negative
          timeout will block.
        */
      virtual int wait (double timeout, int* stat_loc = 0);
      /// Send a signal to the child
      virtual bool kill (int sig);
   
   protected:
      /// Default constructor
      prog_exec ();
      /// parse command line
      virtual bool parse (const char* cmd);
      /// set the command
      virtual void setcmd (const char* cmd) {
         fCmd = cmd ? cmd : ""; }
      /// set the pid
      virtual void setpid (int pid) {
         fPID = pid; }
      /// Return path
      virtual const char* path() const {
         return fFilename.c_str(); }
      /// Return args
      virtual char* const* args() const {
         return fArgs; }
   
   private:
      /// Command string
      std::string	fCmd;
      /// Process ID
      int		fPID;
      /// Error?
      bool		fError;
      /// filename
      string		fFilename;
      /// Argument list
      char**		fArgs;
   };


/** Pipe_exec.
    Starts a child process with fork, executes the string cmd
    with execlp and creates a bidirectional pipe to and from
    the child's standard input and standard output, respectively.
    pipe_exec inherits from iostream and the << and >> operators
    can be used to receive and send data from and to the child,
    respectively.
    This implementation uses non buffered output to make
    sure that the all characters are written to the child
    process before any read operation is performed. Child processes
    which use buffered IO might block the communication when they
    are trying to write a result back in the middle of receiving 
    a block of data.
    
    @memo p2open equivalent
 ************************************************************************/
   class pipe_exec : public std::iostream, public prog_exec {
   public:
      /** Constructor.
          Sets up the pipes and starts the child process.
        */
      explicit pipe_exec (const char* cmd, const char* mode = "rw");
      /** Destuctor.
          Sends a kill signal to the child 
        */
      virtual ~pipe_exec();
      /// Closes the pipes
      virtual void close();
   
      /// Returns true on error
      virtual bool operator!() {
         return prog_exec::operator!() || fail(); }
      /** Wait for child to terminate (returns exit code)
          If poll is true, the method will not block and return
          0 in case the child process is still alive. This
          method will return the same as waitpid and set the
          return status if stat_loc is non-zero.
        */
      virtual int wait (int* stat_loc = 0, bool poll = false);
      /** Wait for child to terminate with timeout.
          If the time expires and the child process is still alive,
          the mothod will return 0. Otherwise, this 
          method will return the same as waitpid and set the
          return status if stat_loc is non-zero. The timeout is
          in seconds. A timeout of 0 is equal to polling and a negative
          timeout will block.
        */
      virtual int wait (double timeout, int* stat_loc = 0);
      /// Returns the mode string
      virtual const char* getmode() const {
         return fMode.c_str(); }
   
   protected:
      /// set the mode
      virtual void setmode (const char* mode) {
         fMode = mode ? mode : ""; }
   
   private:
      /// IO mode
      std::string	fMode;
      /// Input pipe
      int 		fIn[2];
      /// Output pipe
      int 		fOut[2];
      /// Stream buffer
      std::streambuf*	fSBuf;
   
      /// Init: fork, execl and open pipes
      virtual void initcmd(); 
      /// Closes the pipe
      virtual void closepipe();
   };

//@}

}

#endif // _LIGO_PIPEEXEC_H

