/*
 * variable.c - mkcf variable info storage.
 * 
 * include LICENSE
 */
#include <stdio.h>
#include <string.h>

#include <variable.h>
#include <duprintf.h>
#include <stutil.h>


/*
 *** \brief Allocates memory for a new Variable object.
 */

Variable *var_new( char *varname, char *vartype, char *varcmt, int tini,
		   int lineno )
{
   Variable *var;

   var =  app_new0(Variable, 1);
   var_construct( var, varname, vartype,  varcmt, tini, lineno );
   app_class_overload_destroy( (AppClass *) var, var_destroy );
   return var;
}

/** \brief Constructor for the Variable object. */

void var_construct( Variable *var, char *varname, char *vartype,
		    char *varcmt, int tini, int lineno )
{
   app_class_construct( (AppClass *) var );

   var->lineno = lineno;
   var->varname = app_strdup(varname);
   var->vartype = app_strdup(vartype);
   varcmt = stu_rm_blank(varcmt);
   var->varcmt = app_strdup(varcmt);
   var_set_desctype(var, vartype); 
   var->tini = tini;
   if ( tini ) {
      var->desctype = "TINI";
   }
}

/** \brief Destructor for the Variable object. */

void var_destroy(void *var)
{
   Variable *this = (Variable *) var;

   if (var == NULL) {
      return;
   }
   app_free(this->varname);
   app_free(this->vartype);
   app_free(this->varcmt);
   app_free(this->varfunc);

   app_class_destroy( var );
}

void var_dup_varfunc( Variable *var, char *prefix)
{
   if ( var->tini ) {
      var->varfunc = app_strdup_printf("up_init_%s", var->varname );
   } else {
      var->varfunc = app_strdup_printf("up_addr_%s_%s", prefix, var->varname);
   }
}

/*
 * this function return 0 to avoid deleting the node
 */
int var_iter_print_var( AppClass *data, void *user_data )
{
   Variable *var = (Variable *) data;
   char *cmt;
   
   if ( !  var->varcmt ){
      cmt = app_strdup("");
   } else {
      cmt = app_strdup_printf("  /* %s */", var->varcmt );
   }
   
   fprintf(stderr, "    %s%s%s\n",
	 var->vartype, var->varname, cmt ) ;
   app_free(cmt);
   return 0;
}

/*
 * used in dlist_lookup
 */
int var_name_cmp(AppClass *d1, AppClass *d2 )
{
   Variable *var = (Variable *) d1 ;
   char *name = (char *) d2 ;
   return app_strcmp( var->varname, name );
}

/*
 * used in dlist_add
 */
int var_varname_cmp(AppClass *d1, AppClass *d2 )
{
   Variable *var    = (Variable *) d1 ;
   Variable *newvar = (Variable *) d2 ;
   return app_strcmp( var->varname, newvar->varname );
}

void var_set_desctype(Variable *var, char *vartype)
{
   var->desctype = var_text2type(var->vartype); /* not mallocated */
   if ( ! var->desctype ){
      msg_warning( "Unknown type '%s' in variable table at line %d",
		   vartype, var->lineno);
      var->desctype = "TCMT"; /* not mallocated */
   }
}

char *var_text2type( char *ctype)
{
   if ( app_strstr( ctype, "int" ) ) {
      return "TDECI";
   } else if (  app_strstr( ctype, "char" ) ) {
      return "TPTR";
   } else if (  app_strstr( ctype, "BOOL" ) ) {
      return "TBOOL";
   } else if (  app_strstr( ctype, "DList" ) ) {
      return "TLIST";
   } else if ( app_strstr( ctype, "Array" ) ) {
      return "TARYSTR";
   } else {
      return NULL;
   }
}
