/*
 * descrec.c - mkcf object for conf desc tables and elems
 * 
 * include LICENSE
 */
#include <stdio.h>
#include <string.h>

#include <strcatdup.h>
#include <pcreg_ex.h>
#include <descrec.h>
#include <descelem.h>
#include <strmem.h>


/*
 *** \brief Allocates memory for a new DescRecord object.
 */

DescRecord *descrec_new( char *name, char *proto, char *prefix )
{
   DescRecord *desc;

   desc =  app_new0(DescRecord, 1);
   descrec_construct( desc, name, proto, prefix );
   app_class_overload_destroy( (AppClass *) desc, descrec_destroy );
   return desc;
}

/** \brief Constructor for the DescRecord object. */

void descrec_construct( DescRecord *desc, char *name, char *proto, char *prefix )
{
   app_class_construct( (AppClass *) desc );
   
   desc->name = app_strdup(name);
   desc->proto = app_strdup(proto);
   desc->prefix = app_strdup(prefix);
}

/** \brief Destructor for the DescRecord object. */

void descrec_destroy(void *desc)
{
   DescRecord *this = (DescRecord *) desc;

   if (desc == NULL) {
      return;
   }
   app_free(this->name);
   app_free(this->proto);
   app_free(this->prefix);
   dlist_delete_all( this->elems );
   
   app_class_destroy( desc );
}

/*
 * varfunc is something like :
 *   (char *) &prog_debug   (old)
 *   &userPrefs->npanels    (old)
 *   up_addr_up_prog_debug  (new)
 *   up_init_services       (new)
 */
void descrec_extract_varname(DescRecord *desc, DescElem *elem, char *varfunc)
{
   Pcregex *re = NULL;
   char *varname = NULL;
   char *pattern = NULL;
   
   if ( app_strstr(varfunc,"->") ) {
      pattern = app_strdup("->([\\w]+)\\s*");
   } else if (app_strstr(varfunc, "up_addr") ){
      pattern = app_strcatdup("up_addr_", desc->prefix,"_([\\w]+)\\s*", NULL);
   } else if (app_strstr(varfunc, "up_init") ){
      pattern = app_strcatdup("up_init_", "([\\w]+)\\s*", NULL);
   } else if (strchr(varfunc, '&') ){
      pattern = app_strdup("&\\s*([\\w]+)\\s*");
   }
   if ( pattern ){
      re = pcregex_new( pattern, 0, 2 );
      if ( pcregex_exec(re, varfunc, 0 ) > 0 ){
	 varname = array_strPtr_get( re->matchStr, 1 ),
	 descelem_dup_varname(elem, varname);
      }
      app_free(pattern);
      pcregex_destroy(re);
   }
}

void descrec_add_elem( DescRecord *desc, char *extTok, char *varfunc,
		       char *desctype, char *desccmt, int lineno )
{
   int res = 1; 

   DescElem *elem = descelem_new( extTok, varfunc, desctype, desccmt, lineno );
   descrec_extract_varname(desc, elem, varfunc);

   desc->elems = dlist_add( desc->elems, (AppClass *) elem,
                            descelem_extTok_cmp, &res );
   if ( res ) {
      /* extTok already in list - elem is destroyed by dlist_add */
      msg_warningl( 2, "element '%s' at line %d  already in list;\n"
                        "this record is discarded",
                    extTok, lineno) ;
   }
}

/*
 * this function return 0 to avoid deleting the node
 */
int descrec_iter_print_elems( AppClass *data, void *user_data )
{
   DescRecord *desc = (DescRecord *) data;
   
   fprintf(stderr, "Name '%s'\n",
         desc->name) ;

   dlist_iterator( desc->elems, descelem_iter_print_elem, user_data );
   return 0;
}

/*
 * used in dlist_lookup
 */
int descrec_proto_cmp(AppClass *d1, AppClass *d2 )
{
   DescRecord *desc = (DescRecord *) d1 ;
   char *name = (char *) d2 ;
   return app_strcmp( desc->proto, name );
}

