/*
 * dbuf.c - dynamic buffer functions
 * 
 * include LICENSE
 */

#define _GNU_SOURCE      /* for vasprintf */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include <dbuf.h>
 
#ifdef TRACE_MEM
#include <tracemem.h>
#endif
        

/** \brief Allocates memory for a new DBuf object. */

DBuf *
dbuf_new( int size)
{
   DBuf *dbuf;
   
   dbuf = app_new0(DBuf, 1 );
   dbuf_construct( dbuf, size);
   app_class_overload_destroy( (AppClass *) dbuf, dbuf_destroy );
   
   return dbuf;
}

/** \brief Constructor for the DBuf object. */

void
dbuf_construct( DBuf *dbuf, int size)
{
   app_class_construct( (AppClass *) dbuf );

   if (size == 0 ){
      size = BUFSIZ;
   } else {
      size = app_power_of_2(size);
   }
   dbuf->s =  app_new(char, size );
   *(dbuf->s) = 0; 
   dbuf->size = size ;
}

/** \brief Destructor for the DBuf object. */

void dbuf_destroy( void *dbuf)
{
   DBuf *this = (DBuf *) dbuf;

   if ( dbuf == NULL ) {
      return;
   }
   app_free(this->s);
   app_class_destroy( dbuf );
}

void dbuf_renew( DBuf *dbuf, int newsize)
{
   dbuf->size = app_power_of_2(newsize + 1);
   dbuf->nRenew++;
   dbuf->s = app_renew( char, dbuf->s , dbuf->size);
}

void dbuf_set_flags( DBuf *dest, int flags)
{
   dest->flags = flags ;
}

void dbuf_set_pos( DBuf *dest, int pos)
{
   dest->pos = pos ;
}

int dbuf_get_pos( DBuf *dest)
{
   return dest->pos ;
}

void dbuf_set_len( DBuf *dest, int len)
{
   dest->len = len ;
   if ( dest->len > dest->size - 1) {
      dbuf_renew( dest, dest->len );
   }
   char *p = dest->s + dest->len ;
   *p = 0;
}

int dbuf_get_len( DBuf *dest)
{
   return dest->len ;
}

void dbuf_inc_len( DBuf *dest)
{
   dbuf_set_len( dest, dest->len + 1);
}

void dbuf_clear( DBuf *dest)
{
   dest->pos = 0 ;
   dest->len = 0 ;
   *(dest->s) = 0;  
}

void dbuf_ncat( DBuf *dest, DBuf *src, int slen)
{
   int len = dest->len ;
   
   dest->len += slen ;
   if ( dest->len >= dest->size - 1) {
      dbuf_renew( dest, dest->len );
   }
   memcpy(dest->s + len, src->s + src->pos, slen );
   *(dest->s + dest->len) = 0;
   src->pos += slen;
}
/*
 * Warning : src start at src->s + src->pos and not at start of buffer
 */

void dbuf_cat( DBuf *dest, DBuf *src)
{
   int slen = src->len - src->pos;

   dbuf_ncat( dest, src, slen);
}

void dbuf_ncopy( DBuf *dest, DBuf *src, int slen)
{
   dest->len = 0;
   dbuf_ncat( dest, src, slen);
}

void dbuf_copy( DBuf *dest, DBuf *src)
{
   dest->len = 0;
   dbuf_cat( dest, src);
}

void dbuf_strncpy( DBuf *dest, char *src, int slen)
{
   dest->len = slen ;
   if ( dest->len >= dest->size - 1) {
      dbuf_renew( dest, dest->len );
   }
   memcpy(dest->s, src,  dest->len);  
   *(dest->s + dest->len) = 0;
}

void dbuf_strcpy( DBuf *dest, char *src)
{
   dbuf_strncpy( dest, src, app_strlen(src));
}

void dbuf_strncat( DBuf *dest, char *src, int slen)
{
   int dlen = dest->len ;
   
   dest->len += slen ;
   if ( dest->len >= dest->size - 1) {
      dbuf_renew( dest, dest->len );
   }
   memcpy(dest->s + dlen, src, slen );
   *(dest->s + dest->len) = 0;
}

void dbuf_strcat( DBuf *dest, char *src)
{
   dbuf_strncat( dest, src, app_strlen(src));
}

/*
 * concatanate all arguments of a NULL terminated list
 */

void dbuf_concat( DBuf *dest, ...)
{
   va_list args;
   const char *p;

   if ((p = ( const char *) dest) == NULL) {
      return ;
   }

   va_start (args, dest);
   p = va_arg (args, const char *);
   while (p != NULL) {
      dbuf_strcat( dest, (char *) p);
      p = va_arg (args, const char *);
   }
   va_end (args);
}

int dbuf_get_char(DBuf *sbuf)
{
   if (sbuf->pos >= sbuf->len) {
      return -1;
   }
   return (unsigned char) sbuf->s[sbuf->pos++];
}

/*
 * return len chars from buf
 *   may countain 0
 */
char *dbuf_get_dupchars(DBuf *sbuf, int len)
{
   if (sbuf->pos + len >= sbuf->len) {
      return NULL;
   }
   char *ptr = app_new0( char, len + 1) ;
   memcpy(ptr, sbuf->s + sbuf->pos, len);
   sbuf->pos += len;
   return ptr;
}

char *dbuf_get_line(DBuf *dest, DBuf *src)
{
   int c;

   dbuf_clear(dest);
   while ((c = dbuf_get_char(src)) >= 0) {
      if (c == '\n' || c == 0 ) {
         break;
      }
      dbuf_put_char( dest,  c);
   }
   if ( c < 0 && dest->len == 0) {
     return NULL;
   }
   if ( dest->flags & DB_KEEP_LF) {
      dbuf_put_char( dest,  c);
   }
   return dest->s;
}

int dbuf_unget_char(DBuf *sbuf)
{
   if (--sbuf->len < 0) {
      return -1;
   }
   char c = sbuf->s[sbuf->len];
   sbuf->s[sbuf->len] = 0;
   return (unsigned char) c;
}

void dbuf_put_char( DBuf *dest, char c)
{
   if ( dest->len >= dest->size - 2) {
      dbuf_renew( dest, dest->len + 1 );
   }
   char *p = dest->s + dest->len++ ;
   *p++ = c;
   *p = 0;
}

/*
 * dbuf_printf is like sprintf but to dbuf
 */
void dbuf_printf( DBuf *dest, const char *format, ...)
{
   char *buffer = NULL;
   va_list args;

   va_start (args, format);
   int len = vasprintf (&buffer, format, args);
   va_end (args);
   if ( len < 0 ){
      msg_error("vasprintf returned -1.");
      return;
   }
   dbuf_strncat(dest, buffer, len);
   app_free (buffer);
}
