// -*- mode: c++; c-indent-level: 4; c++-member-init-indent: 8; comment-column: 35; -*-

//-----------------------------------------------------------------------------
// eoEasyPSO.h
// (c) OPAC 2007
/*
    Contact: paradiseo-help@lists.gforge.inria.fr
 */
//-----------------------------------------------------------------------------

#ifndef _EOEASYPSO_H
#define _EOEASYPSO_H

//-----------------------------------------------------------------------------
#include <eoContinue.h>
#include <eoPSO.h>
#include <eoVelocity.h>
#include <eoFlight.h>
#include <eoDummyFlight.h>
//-----------------------------------------------------------------------------

/** An easy-to-use particle swarm algorithm; you can use any particle,
*   any flight, any topology... The main steps are :
* 	  (The population  is expected to be already evaluated)
* 	 - for each generation and each particle pi
* 	 - evaluate the velocities
* 	 	-- perform the fligth of pi
*    	-- evaluate pi
*    	-- update the neighborhoods
*/
template < class POT > class eoEasyPSO:public eoPSO < POT >
{
public:

    /** Full constructor
    * @param _continuator - An eoContinue that manages the stopping criterion and the checkpointing system
    * @param _eval - An eoEvalFunc: the evaluation performer
    * @param _velocity - An eoVelocity that defines how to compute the velocities
    * @param _flight - An eoFlight that defines how to make the particle flying: that means how 
    * to modify the positions according to the velocities
    */
    eoEasyPSO (
        eoContinue < POT > &_continuator,
        eoEvalFunc < POT > &_eval,
        eoVelocity < POT > &_velocity,
        eoFlight < POT > &_flight):
            continuator (_continuator),
            eval (_eval),
            velocity (_velocity),
            flight (_flight){}


    /** Constructor without eoFlight. For special cases when the flight is performed withing the velocity.
       * @param _continuator - An eoContinue that manages the stopping criterion and the checkpointing system
       * @param _eval - An eoEvalFunc: the evaluation performer
       * @param _velocity - An eoVelocity that defines how to compute the velocities
    */
    eoEasyPSO (
        eoContinue < POT > &_continuator,
        eoEvalFunc < POT > &_eval,
        eoVelocity < POT > &_velocity):
            continuator (_continuator),
            eval (_eval),
            velocity (_velocity),
            flight (dummyFlight){}



    /// Apply a few iteration of flight to the population (=swarm).
    virtual void operator  () (eoPop < POT > &_pop)
    {
        try
        {
            do
            {
                // loop over all the particles for the current iteration
                for (unsigned idx = 0; idx < _pop.size (); idx++)
                {
                    // perform velocity evaluation
                    velocity (_pop[idx],idx);

                    // apply the flight
                    flight (_pop[idx]);

                    // evaluate the position
                    eval (_pop[idx]);

                    // update the topology (particle and local/global best(s))
                    velocity.updateNeighborhood(_pop[idx],idx);
                }

            }while (continuator (_pop));

        }
        catch (std::exception & e)
        {
            std::string s = e.what ();
            s.append (" in eoEasyPSO");
            throw std::runtime_error (s);
        }

    }

private:
    eoContinue < POT > &continuator;
    eoEvalFunc < POT > &eval;
    eoVelocity < POT > &velocity;
    eoFlight < POT > &flight;

    // if the flight does not need to be used, use the dummy flight instance
    eoDummyFlight<POT> dummyFlight;
};


#endif /*_EOEASYPSO_H*/
