note
	description: "[
		Protection information, just stores information, does not know how to get an activation key.
		Note:
				The data is layed out as follows:
				[x][14 20 21][xx][1 2 3][x][reg. date][7 8 9][exec. count][4 5 6][xx][12 11 10 13][xxx][15 16 17][x][22 18 19 23 24 25][check. length][checksum][xxxx]
				 1  2  3  4   56  7 8 9 10  11...  22 232425  26       27 282930 3132 33 34 35 36 37383940 41 42 43  44 45 46 47 48 49  50         51  52...

			where:
				- x is a random number
				- i is the i-th character of the activation key
				- reg. date is the registration date compact representation coded on 12 characters
				- exec. count is the number of executions left coded on 2 characters
				- check. length is the length of the checksum in characters
				- checksum is the checksum
			]"
	legal: "See notice at end of class."
	status: "See notice at end of class."
	date: "$Date: 2012-12-20 23:36:30 +0000 (Thu, 20 Dec 2012) $"
	revision: "$Revision: 96469 $"

class
	KG_PRIVATE_INFO_IMP

inherit
	KG_PRIVATE_INFO

	KG_ENVIRONMENT_VARIABLES_IMP
		export
			{NONE} all
		end

create
	retrieve,
	initialize

feature {NONE} -- Initialization

	retrieve (l_product, l_version: STRING)
			-- Retrieve license info for product `product' version `version'.
		require
			non_void_product_name: l_product /= Void
			valid_product_name: not l_product.is_empty
			non_void_version: l_version /= Void
			valid_version: not l_version.is_empty
		local
			now: DATE
		do
			data := user_stored_value (data_storage (l_product, l_version))
			if data = Void or else data.count < Min_data_length then
				data := global_stored_value (data_storage (l_product, l_version))
				if data = Void or else data.count < Min_data_length then
					initialize_user (l_product, l_version)
					set_user_stored_value (data_storage (l_product, l_version), data)
					set_remaining_executions (0)
				else
					create now.make_now
					if
						registration_date = Void or else
						(now < registration_date or
						now >= (registration_date + (create {DATE_DURATION}.make_by_days (Trial_days_count))))
					then
						initialize_user (l_product, l_version)
						set_user_stored_value (data_storage (l_product, l_version), data)
						set_remaining_executions (0)
					else
						initialize_user (l_product, l_version)
						set_user_stored_value (data_storage (l_product, l_version), data)
					end
				end
			else
				product := l_product
				version := l_version
			end
		ensure
			product_set: product = l_product
			version_set: version = l_version
		end

	initialize (l_product, l_version: STRING)
			-- Set `data' to default value for product `product' version `version'.
		require
			non_void_product_name: l_product /= Void
			valid_product_name: not l_product.is_empty
			non_void_version: l_version /= Void
			valid_version: not l_version.is_empty
		do
			create data.make_filled ('0', Min_data_length)
			data.replace_substring ("50", Execution_count_index, Execution_count_index + 1)
			data.put ('0', last_remaining_days_digit_index)
			product := l_product
			version := l_version
			set_initial_key (Default_activation_key, create {DATE}.make_now)
		ensure
			product_set: product = l_product
			version_set: version = l_version
			is_consistent: is_consistent
		end

feature -- Access

	product: STRING
			-- Product with current private data

	version: STRING
			-- Product version with current private data

	is_consistent: BOOLEAN
			-- Is protection info consistent?
		local
			s: STRING
			count: INTEGER
			retried: BOOLEAN
			l_int_str: STRING
		do
			if not retried then
				s := data.twin
				s.keep_head (Checksum_length_index - 1)
				l_int_str := data.substring (Checksum_length_index, Checksum_length_index + 1)
				if l_int_str.is_integer then
					count := l_int_str.to_integer
					l_int_str := data.substring (Checksum_index, Checksum_index + count - 1)
					if l_int_str.is_integer then
						Result := checksum (s) = l_int_str.to_integer
					else
						Result := False
					end
				else
					Result := False
				end
			else
				Result := False
			end
		rescue
			retried := True
			retry
		end

	remaining_executions: INTEGER
			-- Number of times system has been executed without registration
		local
			retried: BOOLEAN
			l_string: STRING
		do
			if not retried then
				l_string := data.substring (Execution_count_index, Execution_count_index + 1)
				l_string.append_character (data.item (last_remaining_days_digit_index))
				if l_string.is_integer then
					Result := l_string.to_integer
				else
					Result := 0
				end
			else
				Result := 0
			end
		rescue
			retried := True
			retry
		end

	activation_key: STRING
			-- Activation key
		do
			create Result.make (29)
			Result.append (data.substring (7, 9))
			Result.append (data.substring (28, 29))
			Result.append (Key_separator)
			Result.append (data.substring (30, 30))
			Result.append (data.substring (23, 25))
			Result.append (data.substring (35, 35))
			Result.append (Key_separator)
			Result.append (data.substring (34, 34))
			Result.append (data.substring (33, 33))
			Result.append (data.substring (36, 36))
			Result.append (data.substring (2, 2))
			Result.append (data.substring (40, 40))
			Result.append (Key_separator)
			Result.append (data.substring (41, 42))
			Result.append (data.substring (45, 46))
			Result.append (data.substring (3, 3))
			Result.append (Key_separator)
			Result.append (data.substring (4, 4))
			Result.append (data.substring (44, 44))
			Result.append (data.substring (47, 49))
		end

	registration_date: DATE
			-- Date when product was registered
		local
			retried: BOOLEAN
		do
			if not retried then
				create Result.make_by_ordered_compact_date (data.substring (Registration_date_index, Registration_date_index + Registration_date_length - 1).to_integer)
			else
				create Result.make_now_utc
			end
		rescue
			retried := True
			retry
		end

	checksum (value: STRING): INTEGER
			-- Data integrity checksum for `value'.
		local
			i, val: INTEGER
			random: RANDOM
		do
			from
				i := 1
			until
				i > value.count
			loop
				Result := Result + value.item (i).code * i
				i := i + 1
			end
			create random.set_seed (Result)
			from
				i := 1
			until
				i > value.count
			loop
				val := ((-1)^i * random.i_th (i)).truncated_to_integer // 100
				if val > 0 and then Result > {INTEGER}.Max_value - val then
					Result := 1
				elseif val < 0 and then Result < {INTEGER}.Min_value - val then
					Result := -1
				else
					Result := Result + val
				end
				i := i + 1
			end
		end

feature -- Basic Operation

	set_remaining_executions (times: INTEGER)
			-- Set `remaining_executions' with `times'.
		local
			s: STRING
		do
			s := times.out
			if s.count = 1 then
				s.prepend ("00")
			elseif s.count = 2 then
				s.prepend ("0")
			end
			data.replace_substring (s.substring (1,2), Execution_count_index, Execution_count_index + 1)
			data.put (s.item (3), last_remaining_days_digit_index)
			set_random_values
			update_checksum
			set_user_stored_value (data_storage (product, version), data)
		end

	set_key (key: like activation_key; date: like registration_date)
			-- Set `activation_key' with `key'.
			-- Set `registration_date' with `date'.
			-- Reset checksum so that `is_consistent' is `True'.
		do
			internal_set_key (key, date, False)
		end

	set_initial_key (key: like activation_key; date: like registration_date)
			-- Set `activation_key' with `key'.
			-- Set `registration_date' with `date'.
			-- Reset checksum so that `is_consistent' is `True'.
			-- Store information globally.
		do
			internal_set_key (key, date, True)
		end

feature {KG_TESTER} -- Implementation

	initialize_user (l_product, l_version: STRING)
			-- Set user specific `data' to default value for product `product' version `version'.
		local
			s: STRING
		do
			product := l_product
			version := l_version
			create data.make_filled ('0', Min_data_length)
			data.replace_substring ("50", Execution_count_index, Execution_count_index + 1)
			data.put ('0', last_remaining_days_digit_index)
			set_key (Default_activation_key, create {DATE}.make_now)
			s := global_username (l_product, l_version)
			if not s.is_empty then
				set_user_username (s, l_product, l_version)
			end
			s := global_cd_key (l_product, l_version)
			if not s.is_empty then
				set_user_cd_key (s, l_product, l_version)
			end
		ensure
			product_set: product = l_product
			version_set: version = l_version
			is_consistent: is_consistent
		end

	internal_set_key (key: like activation_key; date: like registration_date; global: BOOLEAN)
			-- Set `activation_key' with `key'.
			-- Set `registration_date' with `date'.
			-- Reset checksum so that `is_consistent' is `True'.
			-- Store value globally if `global', user-specific otherwise
		local
			s, d, l_key: STRING
		do
			d := date.ordered_compact_date.out
			create s.make_filled ('0', registration_date_length - d.count)
			s.append (d)
			data.replace_substring (s, registration_date_index, registration_date_index + registration_date_length - 1)
			l_key := key.twin
			l_key.prune_all ('-')
			data.replace_substring (l_key.substring (14, 14), 2, 2)
			data.replace_substring (l_key.substring (20, 21), 3, 4)
			data.replace_substring (l_key.substring (1, 3), 7, 9)
			data.replace_substring (l_key.substring (7, 9), 23, 25)
			data.replace_substring (l_key.substring (4, 6), 28, 30)
			data.replace_substring (l_key.substring (12, 12), 33, 33)
			data.replace_substring (l_key.substring (11, 11), 34, 34)
			data.replace_substring (l_key.substring (10, 10), 35, 35)
			data.replace_substring (l_key.substring (13, 13), 36, 36)
			data.replace_substring (l_key.substring (15, 17), 40, 42)
			data.replace_substring (l_key.substring (22, 22), 44, 44)
			data.replace_substring (l_key.substring (18, 19), 45, 46)
			data.replace_substring (l_key.substring (23, 25), 47, 49)
			set_random_values
			update_checksum
			if global then
				set_global_stored_value (data_storage (product, version), data)
			else
				set_user_stored_value (data_storage (product, version), data)
			end
		end

	hack (value: like data)
			-- Set `data' with `value'.
			-- Use only for testing.
		require
			non_void_data: value /= Void
			valid_data: data.count > Min_data_length
		do
			data := value
		end

	set_random_values
			-- Fills in `data' with new random values.
		local
			rand: RANDOM
		do
			create rand.set_seed ((create {C_DATE}).millisecond_now * (create {C_DATE}).second_now)
			data.replace_substring (rand.i_th (1).abs.out.item (1).out, 1, 1)
			data.replace_substring (rand.i_th (2).abs.out.item (1).out, 5, 5)
			data.replace_substring (rand.i_th (3).abs.out.item (1).out, 6, 6)
			data.replace_substring (rand.i_th (4).abs.out.item (1).out, 10, 10)
			data.replace_substring (rand.i_th (5).abs.out.item (1).out, 31, 31)
			data.replace_substring (rand.i_th (6).abs.out.item (1).out, 32, 32)
			data.replace_substring (rand.i_th (7).abs.out.item (1).out, 37, 37)
			data.replace_substring (rand.i_th (8).abs.out.item (1).out, 38, 38)
			data.replace_substring (rand.i_th (9).abs.out.item (1).out, 39, 39)
			data.replace_substring (rand.i_th (10).abs.out.item (1).out, 43, 43)
		end

	update_checksum
			-- Update checksum and last 4 digits.
		local
			ck: INTEGER
			s: STRING
			rand: RANDOM
		do
			data.keep_head (checksum_length_index - 1)
			ck := checksum (data)
			s := ck.out.count.out
			if s.count = 1 then
				s.prepend ("0")
			end
			data.append (s)
			data.append (ck.out)
			create rand.set_seed ((create {C_DATE}).millisecond_now * (create {C_DATE}).second_now)
			data.append (rand.i_th (1).abs.out.item (1).out)
			data.append (rand.i_th (2).abs.out.item (1).out)
			data.append (rand.i_th (3).abs.out.item (1).out)
			data.append (rand.i_th (4).abs.out.item (1).out)
		end

	data: STRING
			-- Private data

	registration_date_index: INTEGER = 11
			-- Index in `data' of first character of registration date

	registration_date_length: INTEGER = 12
			-- Length of string representing registration date

	last_remaining_days_digit_index: INTEGER = 50
			-- 3rd digit of the `remaining_executions'.

	checksum_length_index: INTEGER = 51
			-- Index in `data' of first character of checksum length

	checksum_index: INTEGER = 53
			-- Index in `data' of first character of checksum

	Execution_count_index: INTEGER = 26
			-- Index in `data' of first character of remaining execution count

	Min_data_length: INTEGER = 57
			-- Minimum length of `data' (with a checksum size of 1 character)

invariant
	non_void_data: data /= Void
	valid_data: data.count >= Min_data_length

note
	copyright:	"Copyright (c) 1984-2006, Eiffel Software and others"
	license:	"Eiffel Forum License v2 (see http://www.eiffel.com/licensing/forum.txt)"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"




end -- class KG_PRIVATE_INFO
