#define GE_USE_THREADS

/*
	description:

		"C declarations for the Gobo Eiffel runtime."

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2005-2020, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_EIFFEL_H
#define GE_EIFFEL_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

/* Class name mapping as defined in the FreeELKS library. */
#ifndef EIF_INTEGER
#define EIF_INTEGER EIF_INTEGER_32
#endif
#ifndef EIF_CHARACTER
#define EIF_CHARACTER EIF_CHARACTER_8
#endif
#ifndef EIF_REAL
#define EIF_REAL EIF_REAL_32
#endif
#ifndef EIF_DOUBLE
#define EIF_DOUBLE EIF_REAL_64
#endif
#ifndef GE_ms
#if EIF_CHARACTER == EIF_CHARACTER_8
#define GE_ms(s,c) GE_ms8((s),(c))
#else
#define GE_ms(s,c) GE_ms32((s),(c))
#endif
#endif

#if defined(__USE_POSIX) || defined(__unix__) || defined(_POSIX_C_SOURCE)
#include <unistd.h>
#endif
#if !defined(WIN32) && \
	(defined(WINVER) || defined(_WIN32_WINNT) || defined(_WIN32) || \
	defined(__WIN32__) || defined(__TOS_WIN__) || defined(_MSC_VER) || \
	defined(__MINGW32__))
#define WIN32 1
#endif
#ifdef WIN32
#define EIF_WINDOWS 1
#include <windows.h>
#endif

/* See https://sourceforge.net/p/predef/wiki/OperatingSystems/ */
#if (defined(macintosh) || defined(Macintosh))
#define EIF_MAC 1
#define EIF_MACOSX 1
#endif
#if (defined(__APPLE__) && defined(__MACH__))
/* Apparently ISE does not define EIF_MASOSX for Mac OS X >=10.4 (see EXECUTION_ENVIRONMENT.available_cpu_count) */
#define EIF_MAC 1
#endif

#if (defined(VMS) || defined(__VMS))
#define EIF_VMS 1
#endif

#if (defined(__VXWORKS__) || defined(__vxworks))
#define EIF_VXWORKS 1
#endif

#define BYTEORDER 0x1234

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stddef.h>

#define EIF_OS_WINNT 	1
#define EIF_OS_LINUX 	2
#define EIF_OS_DARWIN 	4
#define EIF_OS_VXWORKS	11
#define EIF_OS_VMS	12

/* Platform definition */
/* Unix definition */
#define EIF_IS_UNIX EIF_TRUE
#define EIF_OS EIF_OS_LINUX
/* Windows definition */
#ifdef EIF_WINDOWS
#define EIF_IS_WINDOWS EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_WINNT
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_WINDOWS EIF_FALSE
#endif
/* VMS definition */
#ifdef EIF_VMS
#define EIF_IS_VMS EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_VMS
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VMS EIF_FALSE
#endif
/* MAC definition */
#ifdef EIF_MAC
#define EIF_IS_MAC EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_DARWIN
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_MAC EIF_FALSE
#endif
/* VxWorks definition */
#ifdef EIF_VXWORKS
#define EIF_IS_VXWORKS EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_VXWORKS
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VXWORKS EIF_FALSE
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef _MSC_VER /* MSVC */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#if defined (__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed long int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned long int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#include <inttypes.h>
#endif
#endif

/* Portable integer pointers */
#ifdef EIF_WINDOWS
#ifndef _INTPTR_T_DEFINED
#define _INTPTR_T_DEFINED
#ifdef _WIN64
typedef __int64 intptr_t;
#else
typedef int intptr_t;
#endif
#endif
#ifndef _UINTPTR_T_DEFINED
#define _UINTPTR_T_DEFINED
#ifdef _WIN64
  typedef unsigned __int64 uintptr_t;
#else
  typedef unsigned int uintptr_t;
#endif
#endif
#endif

/* C type for underlying integer type identifying object's dynamic type. */
typedef uint16_t EIF_TYPE_INDEX;

/*
 * Abstraction representing an Eiffel type.
 * It is made of a compiler type-id,
 * and of some annotations (attached/detachable/separate/variant/frozen).
 */
typedef struct eif_type {
	EIF_TYPE_INDEX id;
	EIF_TYPE_INDEX annotations;
} EIF_TYPE;

/*
 * Since EIF_TYPE and EIF_ENCODED_TYPE have the same size, the encoded version
 * is basically a memcpy version of the EIF_TYPE representation.
 * It is used to provide backward compatibility to most Eiffel and
 * C APIs manipulating types as an INTEGER.
 */
typedef int32_t EIF_ENCODED_TYPE;
typedef EIF_ENCODED_TYPE EIF_TYPE_ID;
#define EIF_NO_TYPE (EIF_TYPE_ID)(-1)

/* Basic Eiffel types */
typedef char EIF_BOOLEAN;
typedef unsigned char EIF_CHARACTER_8;
typedef uint32_t EIF_CHARACTER_32;
typedef int8_t EIF_INTEGER_8;
typedef int16_t EIF_INTEGER_16;
typedef int32_t EIF_INTEGER_32;
typedef int64_t EIF_INTEGER_64;
typedef uint8_t EIF_NATURAL_8;
typedef uint16_t EIF_NATURAL_16;
typedef uint32_t EIF_NATURAL_32;
typedef uint64_t EIF_NATURAL_64;
typedef void* EIF_POINTER;
typedef float EIF_REAL_32;
typedef double EIF_REAL_64;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags;} EIF_ANY;
typedef EIF_ANY* EIF_REFERENCE;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags; EIF_REFERENCE area; EIF_INTEGER count;} EIF_STRING;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags; uint32_t offset; EIF_INTEGER count; EIF_INTEGER capacity;} EIF_SPECIAL;

#ifdef EIF_WINDOWS
typedef wchar_t EIF_NATIVE_CHAR;
#else
typedef char EIF_NATIVE_CHAR;
#endif
typedef EIF_NATIVE_CHAR* EIF_FILENAME;

#define EIF_VOID ((EIF_REFERENCE)0)
#define EIF_FALSE ((EIF_BOOLEAN)'\0')
#define EIF_TRUE ((EIF_BOOLEAN)'\1')
#define EIF_TEST(x) ((x) ? EIF_TRUE : EIF_FALSE)

#define EIF_IS_WORKBENCH EIF_FALSE
#define EIF_POINTER_DISPLAY "lX"

/* For INTEGER and NATURAL manifest constants */
#define GE_int8(x) x
#define GE_nat8(x) x
#define GE_int16(x) x
#define GE_nat16(x) x
#define GE_int32(x) x##L
#define GE_nat32(x) x##U
#if defined (_MSC_VER) && (_MSC_VER < 1400) /* MSC older than v8 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#elif defined(__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#else /* ISO C 99 */
#define GE_int64(x) x##LL
#define GE_nat64(x) x##ULL
#endif
#ifdef __LCC__
/* lcc-win32 reports a constant overflow for -21474836478. */
#define GE_min_int32 (-GE_int32(2147483647)-GE_int32(1))
#else
#define GE_min_int32 GE_int32(-2147483648)
#endif
#define GE_max_int32 GE_int32(2147483647)
#if defined(__LCC__) || defined(__GNUC__) || defined(__MINGW32__)
/* lcc-win32 reports a constant overflow for -9223372036854775808. */
/* gcc and mingw-win64 warn that integer constant is so large that it is unsigned. */
#define GE_min_int64 (-GE_int64(9223372036854775807)-GE_int64(1))
#else
#define GE_min_int64 GE_int64(-9223372036854775808)
#endif
#if defined(__LCC__) && !defined(_WIN64)
/* lcc-win32 does not consider 64 bit constants as constants in case statement. */
#define GE_case_int64(x) ((int32_t)(x))
#define GE_case_nat64(x) ((uint32_t)(x))
#else
#define GE_case_int64(x) (x)
#define GE_case_nat64(x) (x)
#endif

#ifdef _WIN64
#define GE_IS_64_BITS EIF_TRUE
#else
#define GE_IS_64_BITS EIF_TEST(sizeof(void*)==64)
#endif

/* Posix threads */
#if !defined(EIF_WINDOWS)
#define GE_USE_POSIX_THREADS
#endif

#ifdef _MSC_VER /* MSVC */
/* MSVC does not support ISO C 99's 'snprintf' from stdio.h */
#define snprintf(a,b,c,d) sprintf(a,c,d)
#endif

/*
 * Gobo compiler version.
 * Starts with 6080 (looks like GOBO) followed by 5 digits.
 */
#define GE_compiler_version() 608000001

/*
	Interoperability with ISE.
*/
#define RTI64C(x) GE_int64(x)
#define EIF_OBJECT EIF_REFERENCE
#define EIF_OBJ EIF_OBJECT
#define OVERHEAD sizeof(EIF_ANY)
/* Function pointer call to make sure all arguments are correctly pushed onto stack. */
/* FUNCTION_CAST is for standard C calls. */
/* FUNCTION_CAST_TYPE is for non-standard C calls. */
#define FUNCTION_CAST(r_type,arg_types) (r_type (*) arg_types)
#define FUNCTION_CAST_TYPE(r_type,call_type,arg_types) (r_type (call_type *) arg_types)
#define SIGBLOCK
#define SIGRESUME
#define rt_public				/* default C scope */
#define rt_private static		/* static outside a block means private */
#define rt_shared				/* data shared between modules, but not public */
typedef intptr_t rt_int_ptr;
typedef uintptr_t rt_uint_ptr;
#define RTMS(s) GE_str8(s)
#define RTMS_EX(s,c) GE_ms8((s),(c))

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C types used to implement class THREAD and related threading facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_THREAD_TYPES_H
#define GE_THREAD_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef GE_USE_THREADS

#define EIF_THREADS
#ifdef GE_USE_POSIX_THREADS
#define EIF_POSIX_THREADS
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef GE_USE_POSIX_THREADS
#include <pthread.h>
#include <semaphore.h>
#elif defined EIF_WINDOWS
#include <windows.h>
#include <process.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GE_USE_POSIX_THREADS

#define EIF_THR_TYPE            pthread_t
#define EIF_CS_TYPE				pthread_mutex_t
#define EIF_MUTEX_TYPE          pthread_mutex_t
#define EIF_COND_TYPE			pthread_cond_t
#define EIF_SEM_TYPE			sem_t
#define EIF_RWL_TYPE			pthread_rwlock_t

#elif defined EIF_WINDOWS

#define EIF_THR_TYPE            HANDLE
#define EIF_CS_TYPE				CRITICAL_SECTION
#define EIF_MUTEX_TYPE          CRITICAL_SECTION
#define EIF_SEM_TYPE            HANDLE

typedef struct {
		/* Semaphore used to queue up threads waiting for the condition to become signaled. */
	EIF_SEM_TYPE* semaphore;
		/* Serialize access to fields of Current. */
	EIF_CS_TYPE* csection;
		/* Number of waiters. */
	unsigned long num_waiting;
		/* Number of already awoken. */
	unsigned long num_wake;
		/* Number of time we signaled/broadcasted for improving fairness.
		 * This ensures one thread won't steal wakeups from other threads in queue. */
	unsigned long generation;
} EIF_COND_TYPE;

typedef struct {
	EIF_MUTEX_TYPE* m; /* Internal monitor lock. */
	int rwlock; /* >0 = # readers, <0 = writer, 0 = none */
	EIF_COND_TYPE* readers_ok; /* Start waiting readers. */
	unsigned int waiting_writers; /* Number of waiting writers. */
	EIF_COND_TYPE* writers_ok; /* Start a waiting writer. */
} EIF_RWL_TYPE;

#endif

typedef struct {
	unsigned int priority;
	unsigned int stack_size;
} EIF_THR_ATTR_TYPE;

/* Struct for thread context. */
typedef struct GE_thread_context_struct GE_thread_context;
struct GE_thread_context_struct {
	EIF_THR_TYPE thread_id; /* Thread identifier for associated thread. */
	EIF_REFERENCE current; /* Eiffel root object. */
	void (*routine)(EIF_REFERENCE); /* Eiffel routine. */
	void (*set_terminated)(EIF_REFERENCE,EIF_BOOLEAN); /* Eiffel routine to set {THREAD}.terminated. */
	unsigned int initial_priority; /* Initial priority. */
	EIF_THR_TYPE last_thread_id; /* Last thread created from current thread. */
	volatile int n_children; /* Number of direct thread children. */
	EIF_MUTEX_TYPE* children_mutex; /* Mutex to wait for thread children. */
	EIF_COND_TYPE* children_cond; /* Condition variable to wait for thread children. */
	GE_thread_context* parent_context;	/* Context of parent thread, NULL if main thread. */
	int thread_exiting; /* Has current thread already called GE_thread_exit? */
	volatile int is_alive; /* Is Current thread still alive? */
};

#ifdef __cplusplus
}
#endif

#endif
#endif

/*
	description:

		"C functions used to implement once features"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_ONCE_H
#define GE_ONCE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Struct to keep track of the call status
 * and results of once features.
 */
typedef struct {
	EIF_BOOLEAN* boolean_value;
	EIF_REFERENCE* boolean_exception;
	unsigned char* boolean_status;
	unsigned int boolean_count;
	EIF_CHARACTER_8* character_8_value;
	EIF_REFERENCE* character_8_exception;
	unsigned char* character_8_status;
	unsigned int character_8_count;
	EIF_CHARACTER_32* character_32_value;
	EIF_REFERENCE* character_32_exception;
	unsigned char* character_32_status;
	unsigned int character_32_count;
	EIF_INTEGER_8* integer_8_value;
	EIF_REFERENCE* integer_8_exception;
	unsigned char* integer_8_status;
	unsigned int integer_8_count;
	EIF_INTEGER_16* integer_16_value;
	EIF_REFERENCE* integer_16_exception;
	unsigned char* integer_16_status;
	unsigned int integer_16_count;
	EIF_INTEGER_32* integer_32_value;
	EIF_REFERENCE* integer_32_exception;
	unsigned char* integer_32_status;
	unsigned int integer_32_count;
	EIF_INTEGER_64* integer_64_value;
	EIF_REFERENCE* integer_64_exception;
	unsigned char* integer_64_status;
	unsigned int integer_64_count;
	EIF_NATURAL_8* natural_8_value;
	EIF_REFERENCE* natural_8_exception;
	unsigned char* natural_8_status;
	unsigned int natural_8_count;
	EIF_NATURAL_16* natural_16_value;
	EIF_REFERENCE* natural_16_exception;
	unsigned char* natural_16_status;
	unsigned int natural_16_count;
	EIF_NATURAL_32* natural_32_value;
	EIF_REFERENCE* natural_32_exception;
	unsigned char* natural_32_status;
	unsigned int natural_32_count;
	EIF_NATURAL_64* natural_64_value;
	EIF_REFERENCE* natural_64_exception;
	unsigned char* natural_64_status;
	unsigned int natural_64_count;
	EIF_POINTER* pointer_value;
	EIF_REFERENCE* pointer_exception;
	unsigned char* pointer_status;
	unsigned int pointer_count;
	EIF_REAL_32* real_32_value;
	EIF_REFERENCE* real_32_exception;
	unsigned char* real_32_status;
	unsigned int real_32_count;
	EIF_REAL_64* real_64_value;
	EIF_REFERENCE* real_64_exception;
	unsigned char* real_64_status;
	unsigned int real_64_count;
	EIF_REFERENCE* reference_value;
	EIF_REFERENCE* reference_exception;
	unsigned char* reference_status;
	unsigned int reference_count;
	EIF_REFERENCE* procedure_exception;
	unsigned char* procedure_status;
	unsigned int procedure_count;
} GE_onces;

/*
 * Variable to keep track of the call status
 * and results of once-per-process features.
 */
extern GE_onces* GE_process_onces;

/*
 * Initialize `GE_process_onces'.
 */
extern void GE_init_onces(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Create a new 'GE_onces' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_onces* GE_new_onces(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Free memory allocated by `a_onces'.
 */
extern void GE_free_onces(GE_onces* a_onces);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class EXCEPTION"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_EXCEPTION_H
#define GE_EXCEPTION_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_ONCE_H
#include "ge_once.h"
#endif
#ifdef GE_USE_THREADS
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#endif

#include <setjmp.h>

/*
 * On Linux glibc systems, we need to use sig* versions of jmp_buf,
 * setjmp and longjmp to preserve the signal handling context.
 * One way to detect this is if _SIGSET_H_types has
 * been defined in /usr/include/setjmp.h.
 * NOTE: ANSI only recognizes the non-sig versions.
 */
#if (defined(_SIGSET_H_types) && !defined(__STRICT_ANSI__))
#define GE_jmp_buf sigjmp_buf
#define GE_setjmp(x) sigsetjmp((x),1)
#define GE_longjmp(x,y) siglongjmp((x),(y))
#else
#define GE_jmp_buf jmp_buf
#define GE_setjmp(x) setjmp((x))
#define GE_longjmp(x,y) longjmp((x),(y))
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Predefined exception codes.
 */
#define GE_EX_VOID		1			/* Feature applied to void reference */
#define GE_EX_MEM		2			/* No more memory */
#define GE_EX_PRE		3			/* Pre-condition violated */
#define GE_EX_POST		4			/* Post-condition violated */
#define GE_EX_FLOAT		5			/* Floating point exception (signal SIGFPE) */
#define GE_EX_CINV		6			/* Class invariant violated */
#define GE_EX_CHECK		7			/* Check instruction violated */
#define GE_EX_FAIL		8			/* Routine failure */
#define GE_EX_WHEN		9			/* Unmatched inspect value */
#define GE_EX_VAR		10			/* Non-decreasing loop variant */
#define GE_EX_LINV		11			/* Loop invariant violated */
#define GE_EX_SIG		12			/* Operating system signal */
#define GE_EX_BYE		13			/* Eiffel run-time panic */
#define GE_EX_RESC		14			/* Exception in rescue clause */
#define GE_EX_OMEM		15			/* Out of memory (cannot be ignored) */
#define GE_EX_RES		16			/* Resumption failed (retry did not succeed) */
#define GE_EX_CDEF		17			/* Create on deferred */
#define GE_EX_EXT		18			/* External event */
#define GE_EX_VEXP		19			/* Void assigned to expanded */
#define GE_EX_HDLR		20			/* Exception in signal handler */
#define GE_EX_IO		21			/* I/O error */
#define GE_EX_SYS		22			/* Operating system error */
#define GE_EX_RETR		23			/* Retrieval error */
#define GE_EX_PROG		24			/* Developer exception */
#define GE_EX_FATAL		25			/* Eiffel run-time fatal error */
#define GE_EX_DOL		26			/* $ applied to melted feature */
#define GE_EX_ISE_IO	27			/* I/O error raised by the ISE Eiffel runtime */
#define GE_EX_COM		28			/* COM error raised by EiffelCOM runtime */
#define GE_EX_RT_CHECK	29			/* Runtime check error such as out-of-bound array access */
#define GE_EX_OLD		30			/* Old violation */
#define GE_EX_SEL		31			/* Serialization failure */
#define GE_EX_DIRTY		32			/* SCOOP processor dirty exception. */
#define GE_EX_NEX		32			/* Number of internal exceptions */

/*
 * String buffer used to build the exception trace.
 */
typedef struct GE_exception_trace_buffer_struct GE_exception_trace_buffer;
struct GE_exception_trace_buffer_struct {
	char* area;
	uint32_t count;
	uint32_t capacity;
};

/*
 * Information about the feature being executed.
 */
typedef struct GE_call_struct GE_call;
struct GE_call_struct {
#ifdef GE_USE_CURRENT_IN_EXCEPTION_TRACE
	void* object; /* Current object */
#endif
	const char* class_name;
	const char* feature_name;
	GE_call* caller; /* previous feature in the call chain */
};

/*
 * Context of features containing a rescue clause.
 */
typedef struct GE_rescue_struct GE_rescue;
struct GE_rescue_struct {
	GE_jmp_buf jb;
	GE_rescue* previous; /* previous context in the call chain */
};

/*
 * Information about the execution context.
 * One such struct per thread.
 */
typedef struct GE_context_struct GE_context;
struct GE_context_struct {
	GE_call* call; /* Call stack */
	uint32_t in_assertion; /* Is an assertion evaluated? */
	GE_rescue* last_rescue; /* Context of last feature entered containing a rescue clause */
	uint32_t in_rescue; /* Number of rescue clauses currently being executed */
	EIF_REFERENCE exception_manager; /* Exception manager */
	char raising_exception; /* Is an exception currently being raised? */
	char exception_trace_enabled; /* Should exception trace be displayed? */
	long exception_code; /* Code of the exception currently being raised, 0 otherwise */
	const char* exception_tag; /* Tag of the exception currently being raised, NULL otherwise */
	GE_exception_trace_buffer exception_trace_buffer; /* String buffer used to build the exception trace */
	GE_exception_trace_buffer last_exception_trace; /* Last non-routine-failure exception trace */
	int pre_ecma_mapping_status; /* Do we map old names to new name? (i.e. STRING to STRING_8, INTEGER to INTEGER_32, ...). */
#ifdef GE_USE_THREADS
	GE_thread_context* thread; /* Thread context */
	GE_onces* process_onces; /* Cache for status and results of onces-per-process */
	GE_onces* thread_onces; /* Status and results of onces-per-thread */
	void* wel_per_thread_data; /* WEL private data */
#endif
};

/*
 * Default initialization for `GE_context'.
 */
extern GE_context GE_default_context;

/*
 * Execution context of main thread.
 */
extern GE_context* GE_main_context;

/*
 * Execution context of current thread.
 */
extern GE_context* GE_current_context(void);

/*
 * Initialization of exception handling.
 */
extern void GE_init_exception(GE_context* context);

/*
 * Free memory allocated in `a_context' for exception handling.
 */
extern void GE_free_exception(GE_context* a_context);

/*
 * Pointer to function to create a new exception manager object
 * (of type ISE_EXCEPTION_MANAGER).
 */
extern EIF_REFERENCE (*GE_new_exception_manager)(EIF_BOOLEAN);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.init_exception_manager.
 */
extern void (*GE_init_exception_manager)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.last_exception.
 */
extern EIF_REFERENCE (*GE_last_exception)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.once_raise.
 */
extern void (*GE_once_raise)(GE_context*, EIF_REFERENCE, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.set_exception_data.
 */
extern void (*GE_set_exception_data)(GE_context*, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN, EIF_INTEGER_32, EIF_INTEGER_32, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN);

/*
 * Exception tag associated with `a_code'.
 */
extern char* GE_exception_tag(long a_code);

/*
 * Raise an exception with code `a_code'.
 */
extern void GE_raise(long a_code);

/*
 * Raise an exception with code `a_code' and message `msg'.
 */
extern void GE_raise_with_message(long a_code, const char* msg);

/*
 * Raise an exception from EXCEPTION_MANAGER.
 */
extern void GE_developer_raise(long a_code, char* a_meaning, char* a_message);

/*
 * Raise exception which was raised the first time a once routine
 * was executed when executing it again.
 */
extern void GE_raise_once_exception(GE_context* a_context, EIF_REFERENCE a_exception);

/*
 * Exception, if any, which was last raised in `a_context'.
 */
extern EIF_REFERENCE GE_last_exception_raised(GE_context* a_context);

/*
 * Jump to execute the rescue of the last routine with a rescue
 * in the call stack.
 */
extern void GE_jump_to_last_rescue(GE_context* a_context);

/*
 * Set `in_assertion' to 'not b'.
 * Return the opposite of previous value.
 */
extern EIF_BOOLEAN GE_check_assert(EIF_BOOLEAN b);

/*
 * Check whether the type id of `obj' is not in `type_ids'.
 * If it is, then raise a CAT-call exception. Don't do anything if `obj' is Void.
 * `nb' is the number of ids in `type_ids' and is expected to be >0.
 * `type_ids' is sorted in increasing order.
 * Return `obj'.
 */
#define GE_catcall(obj,type_ids,nb) GE_check_catcall((obj),(type_ids),(nb))
extern EIF_REFERENCE GE_check_catcall(EIF_REFERENCE obj, EIF_TYPE_INDEX type_ids[], int nb);

/*
 * Check whether `obj' is Void.
 * If it is, then raise a call-on-void-target exception.
 * Return `obj'.
 */
#define GE_void(obj) (!(obj)?GE_check_void(obj):(obj))
extern EIF_REFERENCE GE_check_void(EIF_REFERENCE obj);

/*
 * Check whether `ptr' is a null pointer.
 * If it is, then raise a no-more-memory exception.
 * Return `ptr'.
 */
#define GE_null(ptr) GE_check_null(ptr)
extern void* GE_check_null(void* ptr);

#ifdef EIF_WINDOWS
/*
 * Set default exception handler.
 */
extern void GE_set_windows_exception_filter(void);
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class THREAD and related threading facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_THREAD_H
#define GE_THREAD_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef GE_USE_THREADS

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GE_USE_POSIX_THREADS
#include <unistd.h>
#endif

#ifdef GE_USE_POSIX_THREADS
#	define EIF_TSD_VAL_TYPE        void*
#	define EIF_TSD_TYPE            pthread_key_t
#	define EIF_TSD_CREATE(key,msg) \
		if (pthread_key_create(&(key),NULL)) \
	        GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_SET(key,val,msg) \
		if (pthread_setspecific((key), (EIF_TSD_VAL_TYPE)(val))) \
			GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_GET0(val_type,key,val) (val = pthread_getspecific(key))
#	define EIF_TSD_GET(val_type,key,val,msg) \
		if (EIF_TSD_GET0(val_type,key,val) == (void*) 0) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_DESTROY(key,msg) if (pthread_key_delete(key)) GE_raise_with_message(GE_EX_EXT, msg)
#elif defined EIF_WINDOWS
#	define EIF_TSD_VAL_TYPE        LPVOID
#	define EIF_TSD_TYPE            DWORD
#	define EIF_TSD_CREATE(key,msg) \
		if ((key=TlsAlloc())==TLS_OUT_OF_INDEXES) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_SET(key,val,msg) \
		if (!TlsSetValue((key),(EIF_TSD_VAL_TYPE)(val))) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_GET0(val_type,key,val) \
		val=(val_type) TlsGetValue(key)
#	define EIF_TSD_GET(val_type,key,val,msg) \
		EIF_TSD_GET0(val_type,key,val); \
		if (GetLastError() != NO_ERROR) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_DESTROY(key,msg) \
		if (!TlsFree(key)) GE_raise_with_message(GE_EX_EXT, msg)
#endif

/* Thread priority levels. */
#define EIF_MIN_THR_PRIORITY			0L
#define EIF_BELOW_NORMAL_THR_PRIORITY	100L
#define EIF_DEFAULT_THR_PRIORITY		127L
#define EIF_ABOVE_NORMAL_THR_PRIORITY	154L
#define EIF_MAX_THR_PRIORITY			255L

/*
 * Mutexes used to protect the calls to once-per-process features.
 */
typedef struct {
	EIF_POINTER* boolean_mutex;
	EIF_POINTER* character_8_mutex;
	EIF_POINTER* character_32_mutex;
	EIF_POINTER* integer_8_mutex;
	EIF_POINTER* integer_16_mutex;
	EIF_POINTER* integer_32_mutex;
	EIF_POINTER* integer_64_mutex;
	EIF_POINTER* natural_8_mutex;
	EIF_POINTER* natural_16_mutex;
	EIF_POINTER* natural_32_mutex;
	EIF_POINTER* natural_64_mutex;
	EIF_POINTER* pointer_mutex;
	EIF_POINTER* real_32_mutex;
	EIF_POINTER* real_64_mutex;
	EIF_POINTER* reference_mutex;
	EIF_POINTER* procedure_mutex;
} GE_once_mutexes;

/*
 * Mutexes used to protect the calls to once-per-process features.
 */
extern GE_once_mutexes* GE_process_once_mutexes;

/*
 * Create a new 'GE_once_mutexes' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_once_mutexes* GE_new_once_mutexes(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Keep track of the numbers of once-per-thread features.
 */
extern void GE_thread_onces_set_counts(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/* Global mutex to protect creation of once-per-object data. */
extern EIF_POINTER GE_once_per_object_data_mutex;

/*
 * Initialize data to handle threads.
 * To be called at the beginning of the main function
 * on the main thread.
 */
extern void GE_init_thread(GE_context* a_context);

/*
 * Create a new thread with attributes `attr' and execute
 * Eiffel routine `routine' on object `current'.
 */
extern void GE_thread_create_with_attr(EIF_REFERENCE current, void (*routine)(EIF_REFERENCE), void (*set_terminated)(EIF_REFERENCE,EIF_BOOLEAN), EIF_THR_ATTR_TYPE* attr);

/*
 * Execution context of current thread.
 */
extern GE_context* GE_thread_current_context(void);

/*
 * Thead ID of current thread.
 */
extern EIF_POINTER GE_thread_id(void);

/*
 * Thread ID of last thread created from current thread.
 */
extern EIF_POINTER GE_last_thread_created(void);

#ifdef EIF_WINDOWS
/*
 * Support for Windows GUI that requires that all GUI operations are performed in the same thread.
 * Allocate new structure of the given size `a_size', assign it to `wel_per_thread_data'.
 * Return newly allocated memory block. It will be freed automatically on thread termination.
 */
extern void* GE_thread_create_wel_per_thread_data(size_t a_size);
#endif

/*
 * Waits until a child thread sets `terminated' from `obj' to True,
 * which means it is terminated. The calling thread must be the
 * direct parent of the thread, or the function might loop indefinitely.
 */
extern void GE_thread_wait(EIF_REFERENCE obj, EIF_BOOLEAN (*get_terminated)(EIF_REFERENCE));

/*
 * Waits until a child thread sets `terminated' from `obj' to True,
 * which means it is terminated, or reaching `a_timeout_ms'.
 * The calling thread must be the direct parent of the thread,
 * or the function might loop indefinitely.
 */
extern EIF_BOOLEAN GE_thread_wait_with_timeout(EIF_REFERENCE obj, EIF_BOOLEAN (*get_terminated)(EIF_REFERENCE), EIF_NATURAL_64 a_timeout_ms);

/*
 * Yields execution to other threads.
 */
extern void GE_thread_yield(void);

/*
 * The calling thread waits for all other children threads to terminate.
 */
extern void GE_thread_join_all(void);

/*
 * Function called to terminate a thread launched by Eiffel with `GE_thread_create_with_attr'.
 * This function must be called from the thread itself (not the parent).
 */
extern void GE_thread_exit(void);

/*
 * Default thread priority level.
 */
extern EIF_INTEGER GE_thread_default_priority(void);

/*
 * Minimum thread priority level.
 */
extern EIF_INTEGER GE_thread_min_priority(void);

/*
 * Maximum thread priority level.
 */
extern EIF_INTEGER GE_thread_max_priority(void);

/*
 * Create a new mutex.
 */
extern EIF_POINTER GE_mutex_create(void);

/*
 * Lock mutex.
 */
extern void GE_mutex_lock(EIF_POINTER a_mutex);

/*
 * Try to lock mutex. Return True on success.
 */
extern EIF_BOOLEAN GE_mutex_try_lock(EIF_POINTER a_mutex);

/*
 * Unlock mutex.
 */
extern void GE_mutex_unlock(EIF_POINTER a_mutex);

/*
 * Destroy and free all resources used by mutex.
 */
extern void GE_mutex_destroy(EIF_POINTER a_mutex);

/*
 * Create a new semaphore allowing `a_count' threads
 * to go into a critical section.
 */
extern EIF_POINTER GE_semaphore_create(EIF_INTEGER a_count);

/*
 * Decrement semaphore count, waiting if necessary
 * until that becomes possible.
 */
extern void GE_semaphore_wait(EIF_POINTER a_semaphore);

/*
 * Has client been successful in decrementing semaphore
 * count without waiting?
 */
extern EIF_BOOLEAN GE_semaphore_try_wait(EIF_POINTER a_semaphore);

/*
 * Increment semaphore count.
 */
extern void GE_semaphore_post(EIF_POINTER a_semaphore);

/*
 * Destroy and free all resources used by semaphore.
 */
extern void GE_semaphore_destroy(EIF_POINTER a_semaphore);

/*
 * Create a new condition variable.
 */
extern EIF_POINTER GE_condition_variable_create(void);

/*
 * Unblock all threads blocked on condition variable.
 */
extern void GE_condition_variable_broadcast(EIF_POINTER a_condition_variable);

/*
 * Unblock one thread blocked on condition variable.
 */
extern void GE_condition_variable_signal(EIF_POINTER a_condition_variable);

/*
 * Block calling thread on condition variable.
 */
extern void GE_condition_variable_wait(EIF_POINTER a_condition_variable, EIF_POINTER a_mutex);

/*
 * Block calling thread on condition variable for at most `a_timeout' milliseconds.
 * Return 1 is we got the condition variable on time, otherwise return 0.
 */
extern EIF_INTEGER GE_condition_variable_wait_with_timeout(EIF_POINTER a_condition_variable, EIF_POINTER a_mutex, EIF_INTEGER a_timeout);

/*
 * Destroy and free all resources used by condition variable.
 */
extern void GE_condition_variable_destroy(EIF_POINTER a_condition_variable);

/*
 * Create a new read-write lock.
 */
extern EIF_POINTER GE_read_write_lock_create(void);

/*
 * Acquire a read lock. Multiple readers can go if there are no writer.
 */
extern void GE_read_write_lock_read_lock(EIF_POINTER a_read_write_lock);

/*
 * Acquire a write lock. Only a single write can proceed.
 */
extern void GE_read_write_lock_write_lock(EIF_POINTER a_read_write_lock);

/*
 * Unlock a read or write lock.
 */
extern void GE_read_write_lock_unlock(EIF_POINTER a_read_write_lock);

/*
 * Destroy and free all resources used by read-write lock.
 */
extern void GE_read_write_lock_destroy(EIF_POINTER a_read_write_lock);

#ifdef __cplusplus
}
#endif

#endif
#endif

/*
	description:

		"C functions used to manipulate native strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2013-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_NATIVE_STRING_H
#define GE_NATIVE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#include <string.h>
#ifdef EIF_WINDOWS
#ifdef __LCC__
/* With lcc-win32, stat.h should be included before wchar.h. */
#include <sys/stat.h>
#endif
#include <wchar.h>
#else
#include <sys/types.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef EIF_WINDOWS

/* Macro used to manipulate native strings, i.e: (wchar_t*) */
#define GE_nstrlen wcslen /* size of string */
#define GE_nstrncpy wcsncpy /* Copy n characters of one string to another */
#define GE_nstrcpy wcscpy /* Copy one string to another */
#define GE_nstrncat wcsncat /* Append characters of a string */
#define GE_nstrcat wcscat /* Append a string */
#define GE_nstrstr wcsstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) L##quote /* Manifest Native string declaration */
#define GE_nstr_fopen _wfopen /* Open file using native string name */
#define GE_nstrcmp wcscmp /* Compare two strings. */
#define GE_nstrdup _wcsdup /* Duplicate string. */
#define GE_nstr_cat_ascii(dest, src) {							\
		int i;													\
		size_t dest_len, src_len;								\
		dest_len = rt_nstrlen (dest);							\
		src_len = strlen (src);									\
		for (i = 0; i < src_len; i++) {							\
			dest[dest_len + i] = (EIF_NATIVE_CHAR)src[i];		\
		}														\
		dest[dest_len + src_len] = (EIF_NATIVE_CHAR)0;			\
	}

#else /* not EIF_WINDOWS */

/* Macro used to manipulate native strings, i.e: (char*) */
#define GE_nstrlen strlen /* size of string */
#define GE_nstrncpy strncpy /* Copy n characters of one string to another */
#define GE_nstrcpy strcpy /* Copy one string to another */
#define GE_nstrncat strncat /* Append characters of a string */
#define GE_nstrcat strcat /* Append a string */
#define GE_nstrstr strstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) quote /* Manifest Native string declaration */
#define GE_nstr_fopen fopen /* Open file using native string name */
#define GE_nstrcmp strcmp /* Compare two strings. */
#define GE_nstrdup strdup /* Duplicate string. */
#define GE_nstr_cat_ascii strcat

#endif


#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class ARGUMENTS"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_ARGUMENTS_H
#define GE_ARGUMENTS_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_argc;
extern EIF_NATIVE_CHAR** GE_argv;

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement type information"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_TYPES_H
#define GE_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Type annotations.
 * When a type has no annotation, it means a detachable, non-separate, variant type.
 * In all other cases, there will be an annotation.
 */
#define ANNOTATION_MASK			0x007F	/* All possible annotations. */
#define ATTACHED_FLAG			0x0001
#define DETACHABLE_FLAG			0x0002	/* Only present when overriding an attached type. */
#define SEPARATE_FLAG			0x0004
#define VARIANT_FLAG			0x0008	/* Only present when overriding a frozen/poly type. */
#define UNUSABLE_FLAG			0x0010	/* Reserved for backward compatibility for storables. */
#define FROZEN_FLAG				0x0020
#define POLY_FLAG				0x0040

/*
 * Type flags.
 */
#define GE_TYPE_FLAG_SPECIAL		0x0010
#define GE_TYPE_FLAG_TUPLE			0x0020
#define GE_TYPE_FLAG_EXPANDED		0x0040
#define GE_TYPE_FLAG_DEFERRED		0x0080
#define GE_TYPE_FLAG_NONE			0x0100
#define GE_TYPE_FLAG_BASIC_MASK		0x000F /* One of "BOOLEAN", "CHARACTER_8", "CHARACTER_32", "INTEGER_8", "INTEGER_16", "INTEGER_32", "INTEGER_64", "NATURAL_8", "NATURAL_16", "NATURAL_32", "NATURAL_64", "POINTER", "REAL_32", "REAL_64" */
#define GE_TYPE_FLAG_BOOLEAN		0x0001
#define GE_TYPE_FLAG_CHARACTER_8	0x0002
#define GE_TYPE_FLAG_CHARACTER_32	0x0003
#define GE_TYPE_FLAG_INTEGER_8		0x0004
#define GE_TYPE_FLAG_INTEGER_16		0x0005
#define GE_TYPE_FLAG_INTEGER_32		0x0006
#define GE_TYPE_FLAG_INTEGER_64		0x0007
#define GE_TYPE_FLAG_NATURAL_8		0x0008
#define GE_TYPE_FLAG_NATURAL_16		0x0009
#define GE_TYPE_FLAG_NATURAL_32		0x000A
#define GE_TYPE_FLAG_NATURAL_64		0x000B
#define GE_TYPE_FLAG_POINTER		0x000C
#define GE_TYPE_FLAG_REAL_32		0x000D
#define GE_TYPE_FLAG_REAL_64		0x000E

/*
 * Convention for attribute types.
 * The values are in sync with REFLECTOR_CONSTANTS.
 */
#define GE_TYPE_KIND_INVALID		-1
#define GE_TYPE_KIND_POINTER		0
#define GE_TYPE_KIND_REFERENCE		1
#define GE_TYPE_KIND_CHARACTER_8	2
#define GE_TYPE_KIND_BOOLEAN		3
#define GE_TYPE_KIND_INTEGER_32		4
#define GE_TYPE_KIND_REAL_32		5
#define GE_TYPE_KIND_REAL_64		6
#define GE_TYPE_KIND_EXPANDED		7
#define GE_TYPE_KIND_INTEGER_8		9
#define GE_TYPE_KIND_INTEGER_16		10
#define GE_TYPE_KIND_INTEGER_64 	11
#define GE_TYPE_KIND_CHARACTER_32	12
#define GE_TYPE_KIND_NATURAL_8		13
#define GE_TYPE_KIND_NATURAL_16		14
#define GE_TYPE_KIND_NATURAL_32 	15
#define GE_TYPE_KIND_NATURAL_64 	16

/*
 * Object flags.
 */
#define GE_OBJECT_FLAG_MARKED		0x0001

/*
 * Ancestor relationship between two types X and Y.
 */
#ifdef GE_USE_ANCESTORS
typedef struct {
	EIF_TYPE_INDEX type_id; /* Type id of Y */
	EIF_BOOLEAN conforms; /* Does X conform to Y? */
	void (**qualified_calls)(); /* Function pointers, indexed by call id, when the static type of the target is Y and the dynamic type is X */
} GE_ancestor;
#endif

/*
 * Attribute.
 */
#ifdef GE_USE_ATTRIBUTES
typedef struct {
#ifdef GE_USE_ATTRIBUTE_NAME
	const char* name; /* Attribute name */
#endif
#ifdef GE_USE_ATTRIBUTE_TYPE_ID
	EIF_ENCODED_TYPE type_id; /* Static type id */
#endif
#ifdef GE_USE_ATTRIBUTE_OFFSET
	uint32_t offset; /* Address offset in object */
#endif
} GE_attribute;
#endif

/*
 * Type information.
 */
typedef struct {
	EIF_TYPE_INDEX type_id;
	uint16_t flags;
#ifdef GE_USE_TYPE_GENERATOR
	const char* generator; /* Generator class name */
#endif
#ifdef GE_USE_TYPE_NAME
	const char* name; /* Full type name */
#endif
#ifdef GE_USE_TYPE_GENERIC_PARAMETERS
	EIF_ENCODED_TYPE* generic_parameters;
	uint32_t generic_parameter_count;
#endif
#ifdef GE_USE_ANCESTORS
	GE_ancestor** ancestors;
	uint32_t ancestor_count;
#endif
#ifdef GE_USE_ATTRIBUTES
	GE_attribute** attributes;
	uint32_t attribute_count;
#endif
#ifdef GE_USE_TYPE_OBJECT_SIZE
	uint64_t object_size;
#endif
	EIF_REFERENCE (*new_instance)();
	void (*dispose)(GE_context*, EIF_REFERENCE);
} GE_type_info;

typedef struct {
	EIF_TYPE_INDEX id; /* Type id of the "TYPE [X]" object */
	EIF_INTEGER type_id; /* Type id of the type "X" */
	EIF_BOOLEAN is_special;
	void (*dispose)(GE_context*, EIF_REFERENCE);
	EIF_REFERENCE a1; /* internal_name */
	EIF_REFERENCE a2; /* internal_name_32 */
} EIF_TYPE_OBJ;

/*
 * Types indexed by type id.
 * Generated by the compiler.
 */
extern EIF_TYPE_OBJ GE_types[][2];
extern GE_type_info GE_type_infos[];

/*
 * Number of type infos in `GE_type_infos'.
 * Do not take into account the fake item at index 0.
 */
extern int GE_type_info_count;

/*
 * Encode a EIF_TYPE into a EIF_ENCODED_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_ENCODED_TYPE GE_encoded_type(EIF_TYPE a_type);

/*
 * Decode a EIF_ENCODED_TYPE into a EIF_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_TYPE GE_decoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Type with `a_id' and `a_annotations'.
 */
extern EIF_TYPE GE_new_type(EIF_TYPE_INDEX a_id, EIF_TYPE_INDEX a_annotations);

/*
 * Type of object `obj'.
 */
#define GE_object_type(obj)	GE_new_type(((EIF_REFERENCE)(obj))->id, 0x0)
#define GE_object_encoded_type(obj) GE_encoded_type(GE_object_type(obj))

/*
 * Attachment status of `a_type'.
 */
#define GE_is_attached_type(a_type) EIF_TEST(((a_type).annotations & ATTACHED_FLAG) || GE_is_expanded_type_index((a_type).id))
#define GE_is_attached_encoded_type(a_type) GE_is_attached_type(GE_decoded_type(a_type))

/*
 * Associated detachable type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_non_attached_type(EIF_TYPE a_type);
#define GE_non_attached_encoded_type(a_type) GE_encoded_type(GE_non_attached_type(GE_decoded_type(a_type)))

/*
 * Associated attached type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_attached_type(EIF_TYPE a_type);
#define GE_attached_encoded_type(t) GE_encoded_type(GE_attached_type(GE_decoded_type(t)))

/*
 * Is `a_type' a SPECIAL type?
 */
#define GE_is_special_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_SPECIAL)
#define GE_is_special_encoded_type(a_type) GE_is_special_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_object(obj) GE_is_special_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of user-defined expanded type?
 */
extern EIF_BOOLEAN GE_is_special_of_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_expanded_encoded_type(a_type) GE_is_special_of_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_expanded_object(obj) GE_is_special_of_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type?
 */
extern EIF_BOOLEAN GE_is_special_of_reference_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_encoded_type(a_type) GE_is_special_of_reference_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_object(obj) GE_is_special_of_reference_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type or basic expanded type?
 * (Note that user-defined expanded types are excluded.)
 */
extern EIF_BOOLEAN GE_is_special_of_reference_or_basic_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_or_basic_expanded_encoded_type(a_type) GE_is_special_of_reference_or_basic_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_or_basic_expanded_object(obj) GE_is_special_of_reference_or_basic_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a TUPLE type?
 */
#define GE_is_tuple_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_TUPLE)
#define GE_is_tuple_encoded_type(a_type) GE_is_tuple_type_index(GE_decoded_type(a_type).id)
#define GE_is_tuple_object(obj) GE_is_tuple_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' an expanded type?
 */
#define GE_is_expanded_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_EXPANDED)
#define GE_is_expanded_encoded_type(a_type) GE_is_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_expanded_object(obj) GE_is_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a type whose base class is deferred?
 */
#define GE_is_deferred_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_DEFERRED)
#define GE_is_deferred_encoded_type(a_type) GE_is_deferred_type_index(GE_decoded_type(a_type).id)

/*
 * Does `i'-th field of `a_object + a_physical_offset' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_copy_semantics_field(EIF_INTEGER i, EIF_POINTER a_object, EIF_INTEGER a_physical_offset);

/*
 * Does `i'-th item of special `a_object' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_special_copy_semantics_item(EIF_INTEGER i, EIF_POINTER a_object);

/*
 * Generator class name of `a_type'.
 */
extern EIF_REFERENCE GE_generator_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generator_of_encoded_type(a_type) GE_generator_of_type_index(GE_decoded_type(a_type).id)
extern EIF_REFERENCE GE_generator_8_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generator_8_of_encoded_type(a_type) GE_generator_8_of_type_index(GE_decoded_type(a_type).id)

/*
 * Full name of `a_type'.
 */
extern EIF_REFERENCE GE_generating_type_of_encoded_type(EIF_ENCODED_TYPE a_type);
extern EIF_REFERENCE GE_generating_type_8_of_encoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Encoded type whose name is `a_name'.
 * -1 if no such type.
 */
extern EIF_ENCODED_TYPE GE_encoded_type_from_name(EIF_POINTER a_name);

/*
 * Does `a_type_1' conform to `a_type_2'?
 */
extern EIF_BOOLEAN GE_encoded_type_conforms_to(EIF_ENCODED_TYPE a_type_1, EIF_ENCODED_TYPE a_type_2);

/*
 * Number of generic parameters.
 */
extern EIF_INTEGER GE_generic_parameter_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generic_parameter_count_of_encoded_type(a_type) GE_generic_parameter_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Type of `i'-th generic parameter of `a_type'.
 */
extern EIF_INTEGER GE_generic_parameter_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER i);
#define GE_generic_parameter_of_encoded_type(a_type,i) GE_generic_parameter_of_type_index(GE_decoded_type(a_type).id, (i))

/*
 * Number of fields of an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_field_count_of_encoded_type(a_type) GE_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Physical offset of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_offset_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_offset_of_encoded_type(i, a_type) GE_field_offset_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Name of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_POINTER GE_field_name_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_name_of_encoded_type(i, a_type) GE_field_name_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Static type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_static_type_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_static_type_of_encoded_type(i, a_type) GE_field_static_type_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Kind of type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_type_kind_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_type_kind_of_encoded_type(i, a_type) GE_field_type_kind_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Physical size of `a_object'.
 */
extern EIF_NATURAL_64 GE_object_size(EIF_POINTER a_object);

/*
 * Is `i'-th field of objects of type `a_type' a user-defined expanded attribute?
 */
extern EIF_BOOLEAN GE_is_field_expanded_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_is_field_expanded_of_encoded_type(i, a_type) GE_is_field_expanded_of_type_index((i), GE_decoded_type(a_type).id)

#define GE_field_address_at(a_field_offset, a_object, a_physical_offset) ((char*)(a_object) + (a_physical_offset) + (a_field_offset))
#define GE_object_at_offset(a_enclosing, a_physical_offset) (EIF_REFERENCE)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_raw_object_at_offset(a_enclosing, a_physical_offset) (EIF_POINTER)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_object_encoded_type_at_offset(a_enclosing, a_physical_offset) GE_object_encoded_type(GE_raw_object_at_offset((a_enclosing), (a_physical_offset)))
#define GE_boolean_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_pointer_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_raw_reference_field_at(a_field_offset, a_object, a_physical_offset) (EIF_POINTER)*(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_reference_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_set_boolean_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_boolean_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_pointer_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_pointer_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_reference_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_reference_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value

#if defined(GE_USE_ATTRIBUTES) && defined(GE_USE_ATTRIBUTE_OFFSET)
#define GE_field_address(i, a_object, a_physical_offset) GE_field_address_at(GE_type_infos[((EIF_REFERENCE)(a_object))->id].attributes[(i) - 1]->offset, (a_object), (a_physical_offset))
#define GE_boolean_field(i, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_8_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_32_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_8_field(i, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_16_field(i, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_32_field(i, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_64_field(i, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_8_field(i, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_16_field(i, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_32_field(i, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_64_field(i, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_pointer_field(i, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_32_field(i, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_64_field(i, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_reference_field(i, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value) GE_boolean_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value) GE_character_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value) GE_character_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value) GE_integer_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value) GE_integer_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value) GE_integer_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value) GE_integer_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value) GE_natural_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value) GE_natural_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value) GE_natural_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value) GE_natural_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value) GE_pointer_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value) GE_real_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value) GE_real_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value) GE_reference_field((i), (a_object), (a_physical_offset)) = (a_value)
#else
#define GE_boolean_field(i, a_object, a_physical_offset) (EIF_BOOLEAN)0
#define GE_character_8_field(i, a_object, a_physical_offset) (EIF_CHARACTER_8)0
#define GE_character_32_field(i, a_object, a_physical_offset) (EIF_CHARACTER_32)0
#define GE_integer_8_field(i, a_object, a_physical_offset) (EIF_INTEGER_8)0
#define GE_integer_16_field(i, a_object, a_physical_offset) (EIF_INTEGER_16)0
#define GE_integer_32_field(i, a_object, a_physical_offset) (EIF_INTEGER_32)0
#define GE_integer_64_field(i, a_object, a_physical_offset) (EIF_INTEGER_64)0
#define GE_natural_8_field(i, a_object, a_physical_offset) (EIF_NATURAL_8)0
#define GE_natural_16_field(i, a_object, a_physical_offset) (EIF_NATURAL_16)0
#define GE_natural_32_field(i, a_object, a_physical_offset) (EIF_NATURAL_32)0
#define GE_natural_64_field(i, a_object, a_physical_offset) (EIF_NATURAL_64)0
#define GE_pointer_field(i, a_object, a_physical_offset) (EIF_POINTER)0
#define GE_real_32_field(i, a_object, a_physical_offset) (EIF_REAL_32)0
#define GE_real_64_field(i, a_object, a_physical_offset) (EIF_REAL_64)0
#define GE_reference_field(i, a_object, a_physical_offset) (EIF_REFERENCE)0
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value)
#endif

/*
 * Number of non-transient fields of an object of dynamic type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_persistent_field_count_of_type_index(a_type) GE_field_count_of_type_index(a_type)
#define GE_persistent_field_count_of_encoded_type(a_type) GE_persistent_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Is `i'-th field of objects of type `a_type' a transient field?
 * TODO: storable not implemented yet.
 */
#define GE_is_field_transient_of_type_index(i, a_type) EIF_FALSE
#define GE_is_field_transient_of_encoded_type(i, a_type) GE_is_field_transient_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Storable version of objects of type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_storable_version_of_type_index(a_type) EIF_VOID
#define GE_storable_version_of_encoded_type(a_type) GE_storable_version_of_type_index(GE_decoded_type(a_type).id)

/*
 * Get a lock on `GE_mark_object' and `GE_unmark_object' routines so that
 * 2 threads cannot `GE_mark_object' and `GE_unmark_object' at the same time.
 */
extern void GE_lock_marking(void);

/*
 * Release a lock on `GE_mark_object' and `GE_unmark_object', so that another
 * thread can use `GE_mark_object' and `GE_unmark_object'.
 */
extern void GE_unlock_marking(void);

/*
 * Is `obj' marked?
 */
extern EIF_BOOLEAN GE_is_object_marked(EIF_POINTER obj);

/*
 * Mark `obj'.
 */
extern void GE_mark_object(EIF_POINTER obj);

/*
 * Unmark `obj'.
 */
extern void GE_unmark_object(EIF_POINTER obj);

/*
 * New instance of dynamic `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 * `a_type' cannot represent a SPECIAL type, use
 * `GE_new_special_of_reference_instance_of_type_index' instead.
 */
extern EIF_REFERENCE GE_new_instance_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_new_instance_of_encoded_type(a_type) GE_new_instance_of_type_index(GE_decoded_type(a_type).id)

/*
 * New instance of dynamic `a_type' that represents
 * a SPECIAL with can contain `a_capacity' elements of reference type.
 * To create a SPECIAL of basic type, use class SPECIAL directly.
 */
extern EIF_REFERENCE GE_new_special_of_reference_instance_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER a_capacity);
#define GE_new_special_of_reference_instance_of_encoded_type(a_type, a_capacity) GE_new_special_of_reference_instance_of_type_index(GE_decoded_type(a_type).id, (a_capacity))

/*
 * New instance of tuple of type `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 */
#define GE_new_tuple_instance_of_type_index(a_type) GE_new_instance_of_type_index(a_type)
#define GE_new_tuple_instance_of_encoded_type(a_type) GE_new_tuple_instance_of_type_index(GE_decoded_type(a_type).id)

/*
 * New instance of TYPE for object of type `a_type'.
 */
extern EIF_REFERENCE GE_new_type_instance_of_encoded_type(EIF_ENCODED_TYPE a_type);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to manipulate strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2019, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_STRING_H
#define GE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * New Eiffel empty string of type "STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str32(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr32(EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ms8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_str8(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_8" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ims8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ms32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' 32-bit characters built from `s' by reading
 * groups of four bytes with little-endian byte order.
 */
extern EIF_REFERENCE GE_ms32_from_utf32le(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_str32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing
 * the first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ims32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `c' 32-bit characters built from `s' by reading
 * groups of four bytes with little-endian byte order.
 */
extern EIF_REFERENCE GE_ms32_from_utf32le(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_istr32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `n' native characters found in native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_ims32_from_nstr(EIF_NATIVE_CHAR* s, EIF_INTEGER n);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_istr32_from_nstr(EIF_NATIVE_CHAR* s);

/*
 * New Eiffel string of type "STRING" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'
 */
extern EIF_REFERENCE GE_str(const char* s);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class CONSOLE"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_CONSOLE_H
#define GE_CONSOLE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize mutex to determine whether a new
 * console needs to be created.
 */
#ifdef EIF_WINDOWS
extern void GE_init_console(void);
#else
#define GE_init_console()
#endif

/*
 * Create a new DOS console if needed (i.e. in case of a Windows application).
 */
#ifdef EIF_WINDOWS
extern void GE_show_console(void);
#else
#define GE_show_console()
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement the program initialization"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_MAIN_H
#define GE_MAIN_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_main(int argc, EIF_NATIVE_CHAR** argv);

/*
 * System name.
 */
extern char* GE_system_name;

/*
 * Root class name.
 */
extern char* GE_root_class_name;

#ifdef EIF_WINDOWS

#include <windows.h>

/*
 * Used in WEL.
 */
extern HINSTANCE eif_hInstance;
extern HINSTANCE eif_hPrevInstance;
extern LPWSTR eif_lpCmdLine;
extern int eif_nCmdShow;

/*
 * Main entry point when compiling a Windows application.
 * See:
 *    http://en.wikipedia.org/wiki/WinMain
 *    http://msdn2.microsoft.com/en-us/library/ms633559.aspx
 */
extern int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);

#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to access garbage collector facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_GC_H
#define GE_GC_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef GE_USE_BOEHM_GC
/*
 *	Use the Boehm garbage collector.
 *	See:
 *		http://en.wikipedia.org/wiki/Boehm_GC
 *		http://www.hpl.hp.com/personal/Hans_Boehm/gc/
 */
#define GC_IGNORE_WARN
#ifdef GE_USE_THREADS
#define GC_THREADS
#undef GC_NO_THREAD_REDIRECTS
#endif
#include "gc.h"
#endif

/*
 * GC initialization.
 */

#ifdef GE_USE_BOEHM_GC
#ifdef GE_USE_THREADS
#define GE_init_gc() \
	GC_INIT(); \
	GC_allow_register_threads(); \
	GC_enable_incremental()
#else
#define GE_init_gc() \
	GC_INIT(); \
	GC_enable_incremental()
#endif
#else /* No GC */
#define GE_init_gc() /* do nothing */
#endif

/*
 * Memory allocation.
 */

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc(size) GE_null(GC_MALLOC(size))
#else /* No GC */
#define GE_malloc(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc(size) GC_MALLOC(size)
#else /* No GC */
#define GE_unprotected_malloc(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_atomic(size) GE_null(GC_MALLOC_ATOMIC(size))
#else /* No GC */
#define GE_malloc_atomic(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic(size) GC_MALLOC_ATOMIC(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc(nelem, elsize) GE_null(GC_MALLOC((nelem) * (elsize)))
#else /* No GC */
#define GE_calloc(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc(nelem, elsize) GC_MALLOC((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic(nelem, elsize) memset(GE_null(GC_MALLOC_ATOMIC((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_uncollectable(size) GE_null(GC_MALLOC_UNCOLLECTABLE(size))
#else /* No GC */
#define GE_malloc_uncollectable(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_uncollectable(size) GC_MALLOC_UNCOLLECTABLE(size)
#else /* No GC */
#define GE_unprotected_malloc_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_atomic_uncollectable(size) GE_null(GC_malloc_atomic_uncollectable(size))
#else /* No GC */
#define GE_malloc_atomic_uncollectable(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic_uncollectable(size) GC_malloc_atomic_uncollectable(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_uncollectable(nelem, elsize) GE_null(GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize)))
#else /* No GC */
#define GE_calloc_uncollectable(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc_uncollectable(nelem, elsize) GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic_uncollectable(nelem, elsize) memset(GE_null(GC_malloc_atomic_uncollectable((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc_atomic_uncollectable(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
extern void* GE_unprotected_calloc_atomic_uncollectable(size_t nelem, size_t elsize);
#else /* No GC */
#define GE_unprotected_calloc_atomic_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_realloc(p, size) GE_null(GC_REALLOC((p), (size)))
#else /* No GC */
#define GE_realloc(p, size) GE_null(realloc((p), (size)))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_realloc(p, size) GC_REALLOC((p), (size))
#else /* No GC */
#define GE_unprotected_realloc(p, size) realloc((p), (size))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Raise an exception when no-more-memory.
 */
extern void* GE_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Do not raise an exception when no-more-memory.
 */
extern void* GE_unprotected_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Explicitly deallocate an object.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_free(p) GC_FREE(p)
#else /* No GC */
#define GE_free(p) free(p)
#endif

/*
 * When defined, `GE_malloc_cleared' means that `GE_malloc' and
 * `GE_malloc_uncollectable' make sure that the allocated memory
 * is zeroed.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_cleared
#else /* No GC */
#endif

/*
 * When defined, `GE_malloc_atomic_cleared' means that `GE_malloc_atomic'
 * and `GE_malloc_atomic_uncollectable' make sure that the allocated memory
 * is zeroed.
 */
#ifdef GE_USE_BOEHM_GC
/* #define GE_malloc_atomic_cleared */
#else /* No GC */
#endif

/*
 * Dispose
 */

/*
 * Register dispose routine `disp' to be called on object `obj' when it will be collected.
 */
#ifdef GE_USE_BOEHM_GC
extern void GE_boehm_dispose(void* C, void* disp); /* Call dispose routine on object `C'. */
#define GE_register_dispose(obj, disp) GC_REGISTER_FINALIZER_NO_ORDER((void*)(obj), (void (*) (void*, void*)) &GE_boehm_dispose, (void*)(disp), NULL, NULL)
#else /* No GC */
#define GE_register_dispose(obj, disp) /* do nothing */
#endif

/*
 * Register dispose routine `disp' to be called on once-per-object `data' when it will be collected.
 */
#ifdef GE_USE_BOEHM_GC
extern void GE_boehm_dispose_once_per_object_data(void* data, void* disp); /* Call dispose routine `disp' on once-per-object data `data'. */
#define GE_register_dispose_once_per_object_data(data, disp) GC_REGISTER_FINALIZER_NO_ORDER((void*)(data), (void (*) (void*, void*)) &GE_boehm_dispose_once_per_object_data, (void*)(disp), NULL, NULL)
#else /* No GC */
#define GE_register_dispose_once_per_object_data(data, disp) /* do nothing */
#endif

/*
 * Access to objects, useful with GCs which move objects in memory.
 * This is not the case here, since the Boehm GC is not a moving GC.
 */

/* Access object through hector. */
#define eif_access(obj) (obj)
/* Freeze memory address. */
#define eif_freeze(obj) (obj)
/* The C side adopts an object. */
#define eif_adopt(obj) (obj)
/* The C side protects an object. */
#define eif_protect(obj) (obj)
/* The C side weans adopted object. */
#define eif_wean(obj) (obj)
/* Forget a frozen memory address. */
#define eif_unfreeze(obj)
/* Always frozen since they do not move. */
#define eif_frozen(obj) 1
/* Always frozen since they do not move. */
#define spfrozen(obj) 1

#endif

/*
	description:

		"C functions used to implement class IDENTIFIED"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_IDENTIFIED_H
#define GE_IDENTIFIED_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize data to keep track of object ids.
 */
extern void GE_init_identified(void);

/*
 * Get a new id for `object', assuming it is NOT in the stack.
 */
extern EIF_INTEGER_32 GE_object_id(EIF_OBJECT object);

/*
 * Return the object associated with `id'.
 */
extern EIF_REFERENCE GE_id_object(EIF_INTEGER_32 id);

/*
 * Remove the object associated with `id' from the stack.
 */
extern void GE_object_id_free(EIF_INTEGER_32 id);

#ifdef __cplusplus
}
#endif

#endif

#ifdef __cplusplus
extern "C" {
#endif

#define T0 EIF_ANY

/* CHARACTER */
#define EIF_CHARACTER EIF_CHARACTER_8

/* WIDE_CHARACTER */
#define EIF_WIDE_CHAR EIF_CHARACTER_32

/* INTEGER */
#define EIF_INTEGER EIF_INTEGER_32

/* NATURAL */
#define EIF_NATURAL EIF_NATURAL_32

/* REAL */
#define EIF_REAL EIF_REAL_32

/* DOUBLE */
#define EIF_DOUBLE EIF_REAL_64

/* BOOLEAN */
#define T1 EIF_BOOLEAN
extern T0* GE_boxed1(T1 a1);
extern T0* GE_boxed_pointer1(T1* a1);
typedef struct Sb1 Tb1;

/* CHARACTER_8 */
#define T2 EIF_CHARACTER_8
extern T0* GE_boxed2(T2 a1);
extern T0* GE_boxed_pointer2(T2* a1);
typedef struct Sb2 Tb2;

/* CHARACTER_32 */
#define T3 EIF_CHARACTER_32
extern T0* GE_boxed3(T3 a1);
extern T0* GE_boxed_pointer3(T3* a1);
typedef struct Sb3 Tb3;

/* INTEGER_8 */
#define T4 EIF_INTEGER_8
extern T0* GE_boxed4(T4 a1);
extern T0* GE_boxed_pointer4(T4* a1);
typedef struct Sb4 Tb4;

/* INTEGER_16 */
#define T5 EIF_INTEGER_16
extern T0* GE_boxed5(T5 a1);
extern T0* GE_boxed_pointer5(T5* a1);
typedef struct Sb5 Tb5;

/* INTEGER_32 */
#define T6 EIF_INTEGER_32
extern T0* GE_boxed6(T6 a1);
extern T0* GE_boxed_pointer6(T6* a1);
typedef struct Sb6 Tb6;

/* INTEGER_64 */
#define T7 EIF_INTEGER_64
extern T0* GE_boxed7(T7 a1);
extern T0* GE_boxed_pointer7(T7* a1);
typedef struct Sb7 Tb7;

/* NATURAL_8 */
#define T8 EIF_NATURAL_8
extern T0* GE_boxed8(T8 a1);
extern T0* GE_boxed_pointer8(T8* a1);
typedef struct Sb8 Tb8;

/* NATURAL_16 */
#define T9 EIF_NATURAL_16
extern T0* GE_boxed9(T9 a1);
extern T0* GE_boxed_pointer9(T9* a1);
typedef struct Sb9 Tb9;

/* NATURAL_32 */
#define T10 EIF_NATURAL_32
extern T0* GE_boxed10(T10 a1);
extern T0* GE_boxed_pointer10(T10* a1);
typedef struct Sb10 Tb10;

/* NATURAL_64 */
#define T11 EIF_NATURAL_64
extern T0* GE_boxed11(T11 a1);
extern T0* GE_boxed_pointer11(T11* a1);
typedef struct Sb11 Tb11;

/* REAL_32 */
#define T12 EIF_REAL_32
extern T0* GE_boxed12(T12 a1);
extern T0* GE_boxed_pointer12(T12* a1);
typedef struct Sb12 Tb12;

/* REAL_64 */
#define T13 EIF_REAL_64
extern T0* GE_boxed13(T13 a1);
extern T0* GE_boxed_pointer13(T13* a1);
typedef struct Sb13 Tb13;

/* POINTER */
#define T14 EIF_POINTER
extern T0* GE_boxed14(T14 a1);
extern T0* GE_boxed_pointer14(T14* a1);
typedef struct Sb14 Tb14;

/* [detachable] SPECIAL [CHARACTER_8] */
typedef struct S15 T15;

/* [detachable] SPECIAL [CHARACTER_32] */
typedef struct S16 T16;

/* [detachable] STRING_8 */
typedef struct S17 T17;

/* [detachable] STRING_32 */
typedef struct S18 T18;

/* [detachable] IMMUTABLE_STRING_32 */
typedef struct S20 T20;

/* [detachable] ISE_EXCEPTION_MANAGER */
typedef struct S21 T21;

/* [detachable] GEC */
typedef struct S25 T25;

/* [detachable] CELL [detachable EXCEPTION] */
typedef struct S26 T26;

/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
typedef struct S27 T27;

/* [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
typedef struct S28 T28;

/* [detachable] CELL [[attached] NO_MORE_MEMORY] */
typedef struct S29 T29;

/* detachable C_STRING */
typedef struct S30 T30;

/* [detachable] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
typedef struct S31 T31;

/* [detachable] KL_ARGUMENTS */
typedef struct S32 T32;

/* [detachable] ARRAY [[attached] STRING_8] */
typedef struct S33 T33;

/* [detachable] SPECIAL [[attached] STRING_8] */
typedef struct S34 T34;

/* [detachable] KL_EXCEPTIONS */
typedef struct S35 T35;

/* [detachable] ET_ERROR_HANDLER */
typedef struct S36 T36;

/* [detachable] KL_TEXT_INPUT_FILE */
typedef struct S37 T37;

/* [detachable] ET_ISE_VARIABLES */
typedef struct S38 T38;

/* [detachable] AP_FLAG */
typedef struct S39 T39;

/* detachable ET_SYSTEM */
typedef struct S40 T40;

/* [detachable] AP_PARSER */
typedef struct S42 T42;

/* [detachable] AP_ALTERNATIVE_OPTIONS_LIST */
typedef struct S43 T43;

/* [detachable] AP_STRING_OPTION */
typedef struct S44 T44;

/* [detachable] UT_VERSION */
typedef struct S46 T46;

/* [detachable] AP_ENUMERATION_OPTION */
typedef struct S47 T47;

/* [detachable] AP_BOOLEAN_OPTION */
typedef struct S48 T48;

/* [detachable] AP_INTEGER_OPTION */
typedef struct S49 T49;

/* [detachable] ET_NULL_ERROR_HANDLER */
typedef struct S53 T53;

/* [detachable] ET_ECF_SYSTEM_PARSER */
typedef struct S55 T55;

/* [detachable] ET_ECF_ERROR_HANDLER */
typedef struct S56 T56;

/* detachable ET_ECF_SETTINGS */
typedef struct S57 T57;

/* [detachable] ET_ECF_SYSTEM */
typedef struct S58 T58;

/* [detachable] ET_ECF_TARGET */
typedef struct S59 T59;

/* detachable ET_ECF_CAPABILITIES */
typedef struct S61 T61;

/* detachable ET_ECF_VARIABLES */
typedef struct S62 T62;

/* [detachable] ET_DYNAMIC_SYSTEM */
typedef struct S63 T63;

/* [detachable] ET_SYSTEM_PROCESSOR */
typedef struct S64 T64;

/* [detachable] ET_C_GENERATOR */
typedef struct S67 T67;

/* detachable DT_DATE_TIME */
typedef struct S68 T68;

/* detachable DS_HASH_SET [[attached] STRING_8] */
typedef struct S70 T70;

/* [detachable] ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
typedef struct S71 T71;

/* [detachable] ET_CLASS_TYPE */
typedef struct S72 T72;

/* [detachable] ET_TOKEN_CONSTANTS */
typedef struct S73 T73;

/* [detachable] ET_CLASS */
typedef struct S74 T74;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S76 T76;

/* [detachable] UT_CANNOT_READ_FILE_ERROR */
typedef struct S77 T77;

/* [detachable] UT_VERSION_NUMBER */
typedef struct S79 T79;

/* [detachable] UT_MESSAGE */
typedef struct S80 T80;

/* [detachable] RX_PCRE_REGULAR_EXPRESSION */
typedef struct S81 T81;

/* [detachable] KL_STRING_ROUTINES */
typedef struct S82 T82;

/* [detachable] KL_EXECUTION_ENVIRONMENT */
typedef struct S86 T86;

/* [detachable] AP_ERROR */
typedef struct S87 T87;

/* [detachable] KL_SHELL_COMMAND */
typedef struct S88 T88;

/* [detachable] GECC */
typedef struct S89 T89;

/* [detachable] KL_OPERATING_SYSTEM */
typedef struct S90 T90;

/* [detachable] VOID_TARGET */
typedef struct S93 T93;

/* [detachable] TYPE [[attached] VOID_TARGET] */
#define T94 EIF_TYPE_OBJ

/* [detachable] ROUTINE_FAILURE */
typedef struct S95 T95;

/* [detachable] TYPE [[attached] ROUTINE_FAILURE] */
#define T96 EIF_TYPE_OBJ

/* [detachable] OLD_VIOLATION */
typedef struct S97 T97;

/* [detachable] TYPE [[attached] OLD_VIOLATION] */
#define T98 EIF_TYPE_OBJ

/* [detachable] NO_MORE_MEMORY */
typedef struct S99 T99;

/* [detachable] INVARIANT_VIOLATION */
typedef struct S100 T100;

/* [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE */
typedef struct S101 T101;

/* [detachable] IO_FAILURE */
typedef struct S102 T102;

/* [detachable] OPERATING_SYSTEM_FAILURE */
typedef struct S103 T103;

/* [detachable] COM_FAILURE */
typedef struct S104 T104;

/* [detachable] EIFFEL_RUNTIME_PANIC */
typedef struct S105 T105;

/* [detachable] PRECONDITION_VIOLATION */
typedef struct S107 T107;

/* [detachable] POSTCONDITION_VIOLATION */
typedef struct S108 T108;

/* [detachable] FLOATING_POINT_FAILURE */
typedef struct S109 T109;

/* [detachable] CHECK_VIOLATION */
typedef struct S110 T110;

/* [detachable] BAD_INSPECT_VALUE */
typedef struct S111 T111;

/* [detachable] VARIANT_VIOLATION */
typedef struct S112 T112;

/* [detachable] LOOP_INVARIANT_VIOLATION */
typedef struct S113 T113;

/* [detachable] RESCUE_FAILURE */
typedef struct S114 T114;

/* [detachable] RESUMPTION_FAILURE */
typedef struct S115 T115;

/* [detachable] CREATE_ON_DEFERRED */
typedef struct S116 T116;

/* [detachable] EXTERNAL_FAILURE */
typedef struct S117 T117;

/* [detachable] VOID_ASSIGNED_TO_EXPANDED */
typedef struct S118 T118;

/* [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
typedef struct S119 T119;

/* [detachable] MISMATCH_FAILURE */
typedef struct S120 T120;

/* [detachable] DEVELOPER_EXCEPTION */
typedef struct S121 T121;

/* [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE */
typedef struct S122 T122;

/* [detachable] SERIALIZATION_FAILURE */
typedef struct S123 T123;

/* [detachable] EXECUTION_ENVIRONMENT */
typedef struct S124 T124;

/* [detachable] KL_WINDOWS_FILE_SYSTEM */
typedef struct S125 T125;

/* [detachable] KL_UNIX_FILE_SYSTEM */
typedef struct S126 T126;

/* [detachable] PRIMES */
typedef struct S127 T127;

/* [detachable] SPECIAL [[attached] INTEGER_32] */
typedef struct S128 T128;

/* [detachable] SPECIAL [[attached] BOOLEAN] */
typedef struct S129 T129;

/* [detachable] ARGUMENTS_32 */
typedef struct S130 T130;

/* [detachable] MUTEX */
typedef struct S131 T131;

/* [detachable] UT_ERROR_HANDLER */
typedef struct S132 T132;

/* [detachable] KL_STANDARD_FILES */
typedef struct S133 T133;

/* [detachable] KL_STDERR_FILE */
typedef struct S134 T134;

/* [detachable] KL_STDOUT_FILE */
typedef struct S135 T135;

/* detachable KL_LINKABLE [[attached] CHARACTER_8] */
typedef struct S137 T137;

/* [detachable] MANAGED_POINTER */
typedef struct S140 T140;

/* [detachable] FILE_INFO */
typedef struct S141 T141;

/* [detachable] ST_WORD_WRAPPER */
typedef struct S142 T142;

/* [detachable] AP_DISPLAY_HELP_FLAG */
typedef struct S144 T144;

/* [detachable] DS_ARRAYED_LIST [[attached] AP_OPTION] */
typedef struct S145 T145;

/* [detachable] DS_ARRAYED_LIST [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S146 T146;

/* [detachable] DS_ARRAYED_LIST [[attached] STRING_8] */
typedef struct S147 T147;

/* [detachable] AP_ERROR_HANDLER */
typedef struct S148 T148;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] AP_OPTION] */
typedef struct S149 T149;

/* [detachable] DS_ARRAYED_LIST [detachable STRING_8] */
typedef struct S150 T150;

/* [detachable] DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
typedef struct S151 T151;

/* [detachable] DS_LINKED_LIST [[attached] STRING_8] */
typedef struct S152 T152;

/* [detachable] KL_STRING_EQUALITY_TESTER */
typedef struct S153 T153;

/* detachable KL_EQUALITY_TESTER [[attached] STRING_8] */
typedef struct S154 T154;

/* [detachable] DS_LINKED_LIST [[attached] BOOLEAN] */
typedef struct S157 T157;

/* [detachable] DS_LINKED_LIST [[attached] INTEGER_32] */
typedef struct S160 T160;

/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM */
typedef struct S161 T161;

/* [detachable] ET_ECF_AST_FACTORY */
typedef struct S162 T162;

/* [detachable] DS_CELL [detachable ET_ECF_SYSTEM] */
typedef struct S163 T163;

/* [detachable] TUPLE [[attached] ET_ECF_TARGET] */
typedef struct S164 T164;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]] */
typedef struct S165 T165;

/* detachable TUPLE */
typedef struct S166 T166;

/* [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER] */
typedef struct S167 T167;

/* [detachable] XM_ELEMENT */
typedef struct S168 T168;

/* detachable XM_POSITION_TABLE */
typedef struct S169 T169;

/* [detachable] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8] */
typedef struct S170 T170;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8]] */
typedef struct S171 T171;

/* [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [detachable] STRING_8, [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]], [attached] DS_CELL [detachable ET_ECF_SYSTEM]] */
typedef struct S172 T172;

/* [detachable] DS_HASH_TABLE [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
typedef struct S174 T174;

/* [detachable] KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
typedef struct S175 T175;

/* [detachable] DS_HASH_TABLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
typedef struct S176 T176;

/* [detachable] XM_EIFFEL_PARSER */
typedef struct S178 T178;

/* [detachable] XM_TREE_CALLBACKS_PIPE */
typedef struct S179 T179;

/* [detachable] XM_CALLBACKS_TO_TREE_FILTER */
typedef struct S182 T182;

/* [detachable] ET_ECF_STATE */
typedef struct S183 T183;

/* [detachable] ET_IDENTIFIER */
typedef struct S184 T184;

/* [detachable] ET_ECF_SYSTEM_CONFIG */
typedef struct S185 T185;

/* [detachable] ET_ECF_LIBRARY */
typedef struct S187 T187;

/* [detachable] ET_ADAPTED_LIBRARIES */
typedef struct S188 T188;

/* [detachable] TUPLE [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S190 T190;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_LIBRARY]] */
typedef struct S191 T191;

/* [detachable] XM_DOCUMENT */
typedef struct S192 T192;

/* [detachable] ET_COMPRESSED_POSITION */
typedef struct S193 T193;

/* [detachable] XM_STOP_ON_ERROR_FILTER */
typedef struct S195 T195;

/* detachable XM_ATTRIBUTE */
typedef struct S196 T196;

/* [detachable] DS_HASH_TABLE [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
typedef struct S198 T198;

/* [detachable] DS_HASH_TABLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
typedef struct S199 T199;

/* detachable ET_ECF_TARGET_PARENT */
typedef struct S200 T200;

/* [detachable] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
typedef struct S201 T201;

/* [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [attached] ET_ECF_INTERNAL_UNIVERSE, [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
typedef struct S202 T202;

/* detachable ET_ECF_TARGETS */
typedef struct S203 T203;

/* [detachable] DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S204 T204;

/* [detachable] DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S205 T205;

/* [detachable] DS_CELL [detachable ET_ECF_LIBRARY] */
typedef struct S206 T206;

/* [detachable] ET_ECF_ADAPTED_LIBRARY */
typedef struct S207 T207;

/* [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [attached] ET_ECF_ADAPTED_LIBRARY, [attached] DS_CELL [detachable ET_ECF_LIBRARY]] */
typedef struct S208 T208;

/* [detachable] ET_ADAPTED_DOTNET_ASSEMBLIES */
typedef struct S210 T210;

/* [detachable] ET_ECF_DOTNET_ASSEMBLY */
typedef struct S211 T211;

/* [detachable] ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
typedef struct S213 T213;

/* [detachable] ET_ECF_ERROR */
typedef struct S214 T214;

/* [detachable] KL_AGENT_HASH_FUNCTION [[attached] STRING_8] */
typedef struct S215 T215;

/* [detachable] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8] */
typedef struct S216 T216;

/* [detachable] TUPLE [[attached] STRING_8] */
typedef struct S217 T217;

/* [detachable] FUNCTION [[attached] TUPLE [[attached] STRING_8], [attached] INTEGER_32] */
typedef struct S218 T218;

/* [detachable] TUPLE [[attached] KL_STRING_ROUTINES] */
typedef struct S219 T219;

/* [detachable] DS_HASH_TABLE [[attached] ET_IDENTIFIER, [attached] STRING_8] */
typedef struct S221 T221;

/* [detachable] ET_DYNAMIC_PRIMARY_TYPE */
typedef struct S223 T223;

/* [detachable] ET_DYNAMIC_FEATURE_LIST */
typedef struct S224 T224;

/* [detachable] DS_HASH_TABLE [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
typedef struct S225 T225;

/* [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] STRING_8] */
typedef struct S226 T226;

/* [detachable] TUPLE [[attached] ET_CLASS] */
typedef struct S227 T227;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]] */
typedef struct S228 T228;

/* [detachable] TUPLE [INTEGER_32] */
typedef struct S229 T229;

/* [detachable] ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
typedef struct S230 T230;

/* [detachable] TUPLE [[attached] ET_DYNAMIC_SYSTEM] */
typedef struct S231 T231;

/* detachable ET_DYNAMIC_FEATURE */
typedef struct S236 T236;

/* [detachable] ET_ACTUAL_PARAMETER_LIST */
typedef struct S240 T240;

/* [detachable] ET_NESTED_TYPE_CONTEXT */
typedef struct S241 T241;

/* [detachable] ET_FORMAL_PARAMETER_TYPE */
typedef struct S242 T242;

/* [detachable] ARRAY [[attached] ET_TYPE] */
typedef struct S244 T244;

/* [detachable] SPECIAL [[attached] ET_TYPE] */
typedef struct S245 T245;

/* [detachable] ET_EXTERNAL_FUNCTION */
typedef struct S253 T253;

/* [detachable] ET_TUPLE_TYPE */
typedef struct S254 T254;

/* detachable ET_FORMAL_ARGUMENT_LIST */
typedef struct S255 T255;

/* [detachable] ET_FORMAL_ARGUMENT */
typedef struct S256 T256;

/* [detachable] ET_QUERY_LIST */
typedef struct S257 T257;

/* [detachable] ET_PROCEDURE_LIST */
typedef struct S258 T258;

/* [detachable] ET_DYNAMIC_TYPE_SET_LIST */
typedef struct S259 T259;

/* [detachable] KL_STRING_OUTPUT_STREAM */
typedef struct S260 T260;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_IDENTIFIER] */
typedef struct S261 T261;

/* [detachable] DS_ARRAYED_LIST [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S262 T262;

/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32] */
typedef struct S263 T263;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef struct S264 T264;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S265 T265;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S266 T266;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_RESULT] */
typedef struct S267 T267;

/* [detachable] ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST */
typedef struct S268 T268;

/* [detachable] ET_DYNAMIC_STANDALONE_TYPE_SET */
typedef struct S269 T269;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef struct S270 T270;

/* [detachable] DS_ARRAYED_STACK [[attached] ET_EXPRESSION] */
typedef struct S271 T271;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_EXPRESSION] */
typedef struct S272 T272;

/* [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] INTEGER_32] */
typedef struct S273 T273;

/* [detachable] ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
typedef struct S274 T274;

/* [detachable] DS_HASH_SET [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S275 T275;

/* [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S276 T276;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_OBJECT_TEST] */
typedef struct S277 T277;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S278 T278;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S279 T279;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_AGENT] */
typedef struct S280 T280;

/* [detachable] ET_CURRENT */
typedef struct S282 T282;

/* [detachable] ET_ACTUAL_ARGUMENT_LIST */
typedef struct S283 T283;

/* [detachable] ET_QUALIFIED_CALL_INSTRUCTION */
typedef struct S284 T284;

/* [detachable] ET_QUALIFIED_CALL_EXPRESSION */
typedef struct S287 T287;

/* [detachable] ET_MANIFEST_TUPLE */
typedef struct S288 T288;

/* [detachable] DS_HASH_SET [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S289 T289;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FEATURE] */
typedef struct S290 T290;

/* [detachable] ARRAY [[attached] INTEGER_32] */
typedef struct S291 T291;

/* [detachable] DS_HASH_TABLE [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
typedef struct S292 T292;

/* [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] ET_INLINE_CONSTANT] */
typedef struct S293 T293;

/* [detachable] DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S294 T294;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S295 T295;

/* [detachable] DS_HASH_TABLE [[attached] BOOLEAN, [attached] STRING_8] */
typedef struct S296 T296;

/* [detachable] DS_HASH_SET [[attached] ET_IDENTIFIER] */
typedef struct S297 T297;

/* [detachable] ET_IDENTIFIER_TESTER */
typedef struct S298 T298;

/* [detachable] KL_TEXT_OUTPUT_FILE */
typedef struct S300 T300;

/* [detachable] ET_IMPLICIT_TYPE_MARK */
typedef struct S301 T301;

/* [detachable] ET_DYNAMIC_TUPLE_TYPE */
typedef struct S302 T302;

/* [detachable] DS_STRING_HASH_TABLE */
typedef struct S303 T303;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] STRING_8, [attached] STRING_8] */
typedef struct S304 T304;

/* [detachable] UT_CANNOT_WRITE_TO_FILE_ERROR */
typedef struct S305 T305;

/* detachable ET_DYNAMIC_PRECURSOR */
typedef struct S306 T306;

/* detachable ET_DYNAMIC_PRECURSOR_LIST */
typedef struct S307 T307;

/* [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S308 T308;

/* [detachable] ET_RESULT */
typedef struct S309 T309;

/* [detachable] ET_OBJECT_TEST */
typedef struct S311 T311;

/* [detachable] ET_OBJECT_EQUALITY_EXPRESSION */
typedef struct S312 T312;

/* [detachable] ET_EQUALITY_EXPRESSION */
typedef struct S313 T313;

/* detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL */
typedef struct S314 T314;

/* detachable ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
typedef struct S318 T318;

/* [detachable] ET_DYNAMIC_SPECIAL_TYPE */
typedef struct S319 T319;

/* [detachable] DS_QUICK_SORTER [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S322 T322;

/* detachable ET_DYNAMIC_SECONDARY_TYPE */
typedef struct S323 T323;

/* [detachable] ET_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID */
typedef struct S324 T324;

/* [detachable] ET_CREATE_EXPRESSION */
typedef struct S328 T328;

/* [detachable] ET_QUALIFIED_CALL */
typedef struct S329 T329;

/* [detachable] KL_DIRECTORY */
typedef struct S333 T333;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S334 T334;

/* [detachable] ET_SYMBOL */
typedef struct S341 T341;

/* [detachable] DS_QUICK_SORTER [[attached] INTEGER_32] */
typedef struct S344 T344;

/* [detachable] PROCEDURE [[attached] TUPLE] */
typedef struct S346 T346;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_IDENTIFIER, [attached] ET_CURRENT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S347 T347;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_IDENTIFIER, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S348 T348;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S349 T349;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S350 T350;

/* [detachable] UT_INTEGER_FORMATTER */
typedef struct S351 T351;

/* detachable ET_COMPOUND */
typedef struct S355 T355;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S357 T357;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S358 T358;

/* [detachable] ET_EXTENDED_ATTRIBUTE */
typedef struct S359 T359;

/* [detachable] ET_ATTRIBUTE */
typedef struct S360 T360;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S361 T361;

/* detachable ET_ITERATION_COMPONENT_LIST */
typedef struct S365 T365;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef struct S367 T367;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, INTEGER_32, INTEGER_32] */
typedef struct S368 T368;

/* [detachable] ET_CONSTANT_ATTRIBUTE */
typedef struct S369 T369;

/* [detachable] ET_UNIQUE_ATTRIBUTE */
typedef struct S371 T371;

/* [detachable] ET_REGULAR_INTEGER_CONSTANT */
typedef struct S372 T372;

/* [detachable] ET_SYSTEM_MULTIPROCESSOR */
typedef struct S375 T375;

/* [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8] */
typedef struct S376 T376;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
typedef struct S377 T377;

/* [detachable] TYPE [[attached] STRING_8] */
#define T378 EIF_TYPE_OBJ

/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
typedef struct S379 T379;

/* [detachable] ET_FEATURE_CHECKER */
typedef struct S380 T380;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
typedef struct S381 T381;

/* detachable DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
typedef struct S382 T382;

/* [detachable] ET_TYPE_CHECKER */
typedef struct S384 T384;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_INLINE_AGENT] */
typedef struct S386 T386;

/* [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S387 T387;

/* [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S388 T388;

/* [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S389 T389;

/* [detachable] DS_ARRAYED_LIST [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
typedef struct S390 T390;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S391 T391;

/* [detachable] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
typedef struct S392 T392;

/* [detachable] ET_OBJECT_TEST_SCOPE */
typedef struct S393 T393;

/* [detachable] ET_OBJECT_TEST_SCOPE_BUILDER */
typedef struct S394 T394;

/* [detachable] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ITERATION_COMPONENT] */
typedef struct S395 T395;

/* [detachable] ET_ITERATION_ITEM_SCOPE */
typedef struct S396 T396;

/* [detachable] ET_ATTACHMENT_SCOPE */
typedef struct S397 T397;

/* [detachable] ET_ATTACHMENT_SCOPE_BUILDER */
typedef struct S398 T398;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S399 T399;

/* [detachable] DS_HASH_TABLE [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
typedef struct S400 T400;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_INDEXING_TERM] */
typedef struct S401 T401;

/* [detachable] ET_CLIENT_LIST */
typedef struct S402 T402;

/* [detachable] ET_CLIENT */
typedef struct S403 T403;

/* [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
typedef struct S404 T404;

/* [detachable] ET_ADAPTED_BASE_CLASS_CHECKER */
typedef struct S405 T405;

/* detachable ET_PRECONDITIONS */
typedef struct S406 T406;

/* detachable ET_FEATURE_LIST */
typedef struct S408 T408;

/* [detachable] ET_UNKNOWN_GROUP */
typedef struct S410 T410;

/* [detachable] ET_BASE_TYPE_LIST */
typedef struct S412 T412;

/* [detachable] ET_KEYWORD */
typedef struct S413 T413;

/* [detachable] ET_CLASS_CODES */
typedef struct S414 T414;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S415 T415;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S416 T416;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S417 T417;

/* [detachable] RX_BYTE_CODE */
typedef struct S422 T422;

/* [detachable] RX_CHARACTER_SET */
typedef struct S423 T423;

/* [detachable] RX_CASE_MAPPING */
typedef struct S424 T424;

/* [detachable] UC_UNICODE_ROUTINES */
typedef struct S426 T426;

/* [detachable] ARRAY [[attached] RX_CHARACTER_SET] */
typedef struct S427 T427;

/* [detachable] SPECIAL [[attached] RX_CHARACTER_SET] */
typedef struct S428 T428;

/* [detachable] UC_STRING */
typedef struct S429 T429;

/* [detachable] NATIVE_STRING */
typedef struct S430 T430;

/* [detachable] TUPLE [[attached] GECC, [attached] DS_ARRAYED_LIST [[attached] STRING_8], INTEGER_32, [attached] MUTEX] */
typedef struct S431 T431;

/* [detachable] WORKER_THREAD */
typedef struct S432 T432;

/* [detachable] UC_UTF8_ROUTINES */
typedef struct S434 T434;

/* [detachable] PATH */
typedef struct S435 T435;

/* [detachable] STRING_TO_INTEGER_CONVERTOR */
typedef struct S436 T436;

/* [detachable] ET_ECF_OPTIONS */
typedef struct S444 T444;

/* detachable DS_ARRAYED_LIST [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S445 T445;

/* [detachable] XM_NAMESPACE */
typedef struct S446 T446;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XM_ELEMENT_NODE] */
typedef struct S447 T447;

/* [detachable] ET_LIKE_CURRENT */
typedef struct S449 T449;

/* [detachable] ET_DYNAMIC_PROCEDURE_TYPE */
typedef struct S450 T450;

/* [detachable] ET_DYNAMIC_FUNCTION_TYPE */
typedef struct S451 T451;

/* [detachable] DT_SHARED_SYSTEM_CLOCK */
typedef struct S452 T452;

/* [detachable] DT_SYSTEM_CLOCK */
typedef struct S453 T453;

/* [detachable] ET_EIFFEL_PREPARSER */
typedef struct S454 T454;

/* [detachable] ET_MASTER_CLASS_CHECKER */
typedef struct S455 T455;

/* [detachable] ET_EIFFEL_PARSER */
typedef struct S456 T456;

/* [detachable] ET_PROVIDER_CHECKER */
typedef struct S457 T457;

/* [detachable] ET_ANCESTOR_BUILDER */
typedef struct S458 T458;

/* [detachable] ET_FEATURE_FLATTENER */
typedef struct S459 T459;

/* [detachable] ET_INTERFACE_CHECKER */
typedef struct S460 T460;

/* [detachable] ET_IMPLEMENTATION_CHECKER */
typedef struct S461 T461;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_CLASS] */
typedef struct S462 T462;

/* [detachable] TUPLE [[attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
typedef struct S463 T463;

/* [detachable] ET_AST_NULL_PROCESSOR */
typedef struct S464 T464;

/* [detachable] ET_AST_FACTORY */
typedef struct S465 T465;

/* [detachable] ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
typedef struct S467 T467;

/* [detachable] DT_DATE_TIME_DURATION */
typedef struct S468 T468;

/* [detachable] ET_DO_PROCEDURE */
typedef struct S469 T469;

/* [detachable] UT_CONFIG_PARSER */
typedef struct S471 T471;

/* [detachable] KL_COMPARABLE_COMPARATOR [[attached] INTEGER_32] */
typedef struct S472 T472;

/* [detachable] ET_DYNAMIC_PUSH_TYPE_SET */
typedef struct S475 T475;

/* [detachable] ET_DEFERRED_PROCEDURE */
typedef struct S476 T476;

/* [detachable] KL_PLATFORM */
typedef struct S477 T477;

/* UTF_CONVERTER */
typedef struct S478 T478;
extern T0* GE_boxed478(T478 a1);
extern T0* GE_boxed_pointer478(T478* a1);
typedef struct Sb478 Tb478;

/* [detachable] CELL [[attached] INTEGER_32] */
typedef struct S479 T479;

/* [detachable] HASH_TABLE [[attached] NATIVE_STRING, [attached] IMMUTABLE_STRING_32] */
typedef struct S480 T480;

/* [detachable] KL_ANY_ROUTINES */
typedef struct S481 T481;

/* [detachable] KL_PATHNAME */
typedef struct S483 T483;

/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32] */
typedef struct S484 T484;

/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32] */
typedef struct S485 T485;

/* [detachable] EXCEPTIONS */
typedef struct S486 T486;

/* [detachable] SPECIAL [[attached] NATURAL_8] */
typedef struct S487 T487;

/* TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
typedef struct S488 T488;
extern T0* GE_boxed488(T488 a1);
extern T0* GE_boxed_pointer488(T488* a1);
typedef struct Sb488 Tb488;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] AP_OPTION] */
typedef struct S489 T489;

/* [detachable] SPECIAL [[attached] AP_OPTION] */
typedef struct S491 T491;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] AP_OPTION] */
typedef struct S492 T492;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S493 T493;

/* [detachable] SPECIAL [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S494 T494;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S495 T495;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] STRING_8] */
typedef struct S496 T496;

/* [detachable] SPECIAL [detachable STRING_8] */
typedef struct S497 T497;

/* [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8] */
typedef struct S498 T498;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] STRING_8] */
typedef struct S499 T499;

/* detachable DS_LINKABLE [[attached] STRING_8] */
typedef struct S500 T500;

/* detachable DS_LINKABLE [[attached] BOOLEAN] */
typedef struct S501 T501;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] BOOLEAN] */
typedef struct S502 T502;

/* detachable DS_LINKABLE [[attached] INTEGER_32] */
typedef struct S503 T503;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] INTEGER_32] */
typedef struct S504 T504;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
typedef struct S506 T506;

/* [detachable] SPECIAL [[attached] ET_ECF_LIBRARY] */
typedef struct S507 T507;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_LIBRARY] */
typedef struct S510 T510;

/* [detachable] SPECIAL [[attached] ET_ECF_DOTNET_ASSEMBLY] */
typedef struct S511 T511;

/* detachable DS_HASH_TABLE_CURSOR [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
typedef struct S514 T514;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_DOTNET_ASSEMBLY] */
typedef struct S515 T515;

/* [detachable] XM_EIFFEL_SCANNER */
typedef struct S516 T516;

/* [detachable] XM_DEFAULT_POSITION */
typedef struct S517 T517;

/* [detachable] DS_BILINKED_LIST [[attached] XM_POSITION] */
typedef struct S519 T519;

/* [detachable] DS_LINKED_STACK [[attached] XM_EIFFEL_SCANNER] */
typedef struct S520 T520;

/* [detachable] XM_CALLBACKS_NULL */
typedef struct S521 T521;

/* [detachable] DS_HASH_TABLE [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
typedef struct S522 T522;

/* [detachable] XM_NULL_EXTERNAL_RESOLVER */
typedef struct S524 T524;

/* [detachable] XM_DTD_CALLBACKS_NULL */
typedef struct S526 T526;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ANY] */
typedef struct S527 T527;

/* [detachable] SPECIAL [detachable ANY] */
typedef struct S528 T528;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S529 T529;

/* [detachable] XM_EIFFEL_PARSER_NAME */
typedef struct S530 T530;

/* [detachable] SPECIAL [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S531 T531;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
typedef struct S532 T532;

/* [detachable] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S533 T533;

/* [detachable] SPECIAL [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
typedef struct S534 T534;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_EXTERNAL_ID] */
typedef struct S535 T535;

/* [detachable] XM_DTD_EXTERNAL_ID */
typedef struct S536 T536;

/* [detachable] SPECIAL [[attached] XM_DTD_EXTERNAL_ID] */
typedef struct S537 T537;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S538 T538;

/* [detachable] XM_DTD_ELEMENT_CONTENT */
typedef struct S539 T539;

/* [detachable] SPECIAL [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S540 T540;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
typedef struct S541 T541;

/* [detachable] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S542 T542;

/* [detachable] SPECIAL [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
typedef struct S543 T543;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S544 T544;

/* [detachable] XM_DTD_ATTRIBUTE_CONTENT */
typedef struct S545 T545;

/* [detachable] SPECIAL [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S546 T546;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
typedef struct S547 T547;

/* [detachable] DS_BILINKED_LIST [[attached] STRING_8] */
typedef struct S548 T548;

/* [detachable] SPECIAL [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
typedef struct S549 T549;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] BOOLEAN] */
typedef struct S550 T550;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_DECLARATION] */
typedef struct S551 T551;

/* [detachable] XM_EIFFEL_DECLARATION */
typedef struct S552 T552;

/* [detachable] SPECIAL [[attached] XM_EIFFEL_DECLARATION] */
typedef struct S553 T553;

/* [detachable] XM_EIFFEL_ENTITY_DEF */
typedef struct S556 T556;

/* [detachable] XM_EIFFEL_SCANNER_DTD */
typedef struct S557 T557;

/* [detachable] XM_EIFFEL_PE_ENTITY_DEF */
typedef struct S559 T559;

/* [detachable] XM_NAMESPACE_RESOLVER */
typedef struct S560 T560;

/* [detachable] ARRAY [[attached] XM_CALLBACKS_FILTER] */
typedef struct S561 T561;

/* [detachable] SPECIAL [[attached] XM_CALLBACKS_FILTER] */
typedef struct S562 T562;

/* [detachable] DS_HASH_SET [[attached] XM_NAMESPACE] */
typedef struct S563 T563;

/* [detachable] ET_CLUSTERS */
typedef struct S564 T564;

/* [detachable] ET_CLASS_NAME_TESTER */
typedef struct S565 T565;

/* [detachable] ET_PARENT */
typedef struct S567 T567;

/* [detachable] ET_PARENT_LIST */
typedef struct S568 T568;

/* [detachable] ET_BUILTIN_CONVERT_FEATURE */
typedef struct S569 T569;

/* [detachable] ET_MASTER_CLASS */
typedef struct S570 T570;

/* [detachable] ET_ATTACHMENT_MARK_SEPARATE_KEYWORD */
typedef struct S571 T571;

/* detachable ET_RENAME_LIST */
typedef struct S572 T572;

/* detachable ET_EXPORT_LIST */
typedef struct S573 T573;

/* detachable ET_KEYWORD_FEATURE_NAME_LIST */
typedef struct S574 T574;

/* [detachable] ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
typedef struct S577 T577;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S578 T578;

/* [detachable] XM_LINKED_LIST [[attached] XM_DOCUMENT_NODE] */
typedef struct S579 T579;

/* [detachable] SPECIAL [[attached] ET_ECF_SYSTEM_CONFIG] */
typedef struct S582 T582;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_SYSTEM_CONFIG] */
typedef struct S584 T584;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
typedef struct S585 T585;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_TARGET, [attached] STRING_8] */
typedef struct S586 T586;

/* [detachable] SPECIAL [[attached] ET_ECF_TARGET] */
typedef struct S587 T587;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET] */
typedef struct S589 T589;

/* detachable DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S591 T591;

/* [detachable] KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
typedef struct S592 T592;

/* [detachable] SPECIAL [detachable RX_REGULAR_EXPRESSION] */
typedef struct S593 T593;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S594 T594;

/* [detachable] ET_DOTNET_ASSEMBLIES */
typedef struct S595 T595;

/* [detachable] ET_DOTNET_ASSEMBLY */
typedef struct S596 T596;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_IDENTIFIER] */
typedef struct S599 T599;

/* [detachable] SPECIAL [[attached] ET_IDENTIFIER] */
typedef struct S600 T600;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_IDENTIFIER, [attached] STRING_8] */
typedef struct S601 T601;

/* detachable DS_HASH_TABLE [[attached] ET_DYNAMIC_FEATURE, [attached] INTEGER_32] */
typedef struct S602 T602;

/* detachable ET_FORMAL_PARAMETER_LIST */
typedef struct S603 T603;

/* detachable ET_FEATURE_IDS */
typedef struct S604 T604;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S605 T605;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S606 T606;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
typedef struct S609 T609;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_MASTER_CLASS] */
typedef struct S610 T610;

/* [detachable] SPECIAL [[attached] ET_MASTER_CLASS] */
typedef struct S611 T611;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLASS_NAME] */
typedef struct S612 T612;

/* [detachable] SPECIAL [[attached] ET_CLASS_NAME] */
typedef struct S613 T613;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] STRING_8] */
typedef struct S615 T615;

/* [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL, [attached] ET_CALL_NAME] */
typedef struct S616 T616;

/* [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, [attached] ET_CALL_NAME] */
typedef struct S617 T617;

/* [detachable] SPECIAL [[attached] ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S621 T621;

/* [detachable] ET_BRACKET_SYMBOL */
typedef struct S622 T622;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S624 T624;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE] */
typedef struct S625 T625;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_TYPE_SET] */
typedef struct S626 T626;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE_SET] */
typedef struct S627 T627;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_IDENTIFIER] */
typedef struct S628 T628;

/* [detachable] SPECIAL [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S629 T629;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S630 T630;

/* [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S631 T631;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
typedef struct S632 T632;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef struct S634 T634;

/* [detachable] ET_UNQUALIFIED_CALL_EXPRESSION */
typedef struct S635 T635;

/* [detachable] SPECIAL [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef struct S636 T636;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef struct S637 T637;

/* [detachable] SPECIAL [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S638 T638;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S639 T639;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S640 T640;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S641 T641;

/* [detachable] SPECIAL [[attached] ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S642 T642;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S643 T643;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RESULT] */
typedef struct S644 T644;

/* [detachable] SPECIAL [[attached] ET_RESULT] */
typedef struct S645 T645;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_RESULT] */
typedef struct S646 T646;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef struct S647 T647;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef struct S648 T648;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef struct S649 T649;

/* [detachable] SPECIAL [[attached] ET_EXPRESSION] */
typedef struct S650 T650;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION] */
typedef struct S651 T651;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_EXPRESSION] */
typedef struct S652 T652;

/* detachable KL_EQUALITY_TESTER [[attached] INTEGER_32] */
typedef struct S653 T653;

/* detachable DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] INTEGER_32] */
typedef struct S654 T654;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S655 T655;

/* [detachable] SPECIAL [[attached] ET_OBJECT_TEST] */
typedef struct S657 T657;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST] */
typedef struct S658 T658;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_TEST] */
typedef struct S659 T659;

/* [detachable] SPECIAL [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S660 T660;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S661 T661;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S662 T662;

/* [detachable] SPECIAL [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S663 T663;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S664 T664;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S665 T665;

/* [detachable] SPECIAL [[attached] ET_AGENT] */
typedef struct S666 T666;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT] */
typedef struct S667 T667;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_AGENT] */
typedef struct S668 T668;

/* [detachable] SPECIAL [[attached] ET_EXPRESSION_ITEM] */
typedef struct S669 T669;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION_ITEM] */
typedef struct S671 T671;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S672 T672;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S673 T673;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S674 T674;

/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FEATURE] */
typedef struct S676 T676;

/* [detachable] SPECIAL [[attached] ET_FEATURE] */
typedef struct S677 T677;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE] */
typedef struct S678 T678;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
typedef struct S679 T679;

/* [detachable] SPECIAL [[attached] ET_CONSTANT] */
typedef struct S680 T680;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONSTANT] */
typedef struct S682 T682;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] ET_INLINE_CONSTANT] */
typedef struct S683 T683;

/* [detachable] SPECIAL [[attached] ET_INLINE_CONSTANT] */
typedef struct S684 T684;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_CONSTANT] */
typedef struct S686 T686;

/* [detachable] SPECIAL [detachable ET_DYNAMIC_FEATURE] */
typedef struct S687 T687;

/* detachable DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S689 T689;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
typedef struct S690 T690;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S691 T691;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] STRING_8] */
typedef struct S692 T692;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_IDENTIFIER] */
typedef struct S695 T695;

/* [detachable] TYPE [[attached] ET_IDENTIFIER] */
#define T696 EIF_TYPE_OBJ

/* TYPED_POINTER [[attached] ANY] */
typedef struct S697 T697;
extern T0* GE_boxed697(T697 a1);
extern T0* GE_boxed_pointer697(T697* a1);
typedef struct Sb697 Tb697;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S698 T698;

/* [detachable] DS_ARRAYED_LIST [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S699 T699;

/* [detachable] ARRAY [[attached] BOOLEAN] */
typedef struct S700 T700;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S703 T703;

/* [detachable] TUPLE [[attached] ET_SYSTEM_PROCESSOR, [attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
typedef struct S704 T704;

/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
typedef struct S707 T707;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_TYPE] */
typedef struct S708 T708;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE] */
typedef struct S709 T709;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS] */
typedef struct S710 T710;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_AGENT] */
typedef struct S711 T711;

/* [detachable] SPECIAL [[attached] ET_INLINE_AGENT] */
typedef struct S712 T712;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_INLINE_AGENT] */
typedef struct S713 T713;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S714 T714;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_PROCEDURE] */
typedef struct S715 T715;

/* [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S716 T716;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S717 T717;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S718 T718;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_QUERY] */
typedef struct S719 T719;

/* [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S720 T720;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S721 T721;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S722 T722;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_FEATURE] */
typedef struct S723 T723;

/* [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S724 T724;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S725 T725;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
typedef struct S726 T726;

/* [detachable] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S727 T727;

/* [detachable] SPECIAL [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
typedef struct S728 T728;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
typedef struct S729 T729;

/* [detachable] SPECIAL [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S730 T730;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S731 T731;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S732 T732;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
typedef struct S733 T733;

/* [detachable] ET_NAMED_OBJECT_TEST */
typedef struct S736 T736;

/* [detachable] SPECIAL [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S737 T737;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S738 T738;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S739 T739;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ITERATION_COMPONENT] */
typedef struct S740 T740;

/* [detachable] SPECIAL [[attached] ET_ITERATION_COMPONENT] */
typedef struct S741 T741;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ITERATION_COMPONENT] */
typedef struct S743 T743;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ITERATION_COMPONENT] */
typedef struct S744 T744;

/* [detachable] DS_HASH_SET [[attached] INTEGER_32] */
typedef struct S745 T745;

/* [detachable] SPECIAL [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S747 T747;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S748 T748;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S749 T749;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
typedef struct S750 T750;

/* [detachable] SPECIAL [[attached] ET_ASSERTIONS] */
typedef struct S751 T751;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTIONS] */
typedef struct S753 T753;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM] */
typedef struct S754 T754;

/* [detachable] SPECIAL [[attached] ET_INDEXING_TERM] */
typedef struct S756 T756;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_INDEXING_TERM] */
typedef struct S757 T757;

/* [detachable] SPECIAL [[attached] ET_CLIENT_ITEM] */
typedef struct S758 T758;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_ITEM] */
typedef struct S759 T759;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
typedef struct S760 T760;

/* [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
typedef struct S761 T761;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
typedef struct S762 T762;

/* [detachable] DS_HASH_TABLE [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
typedef struct S763 T763;

/* [detachable] SPECIAL [[attached] NATURAL_32] */
typedef struct S764 T764;

/* [detachable] DS_ARRAYED_LIST [[attached] RX_CHARACTER_SET] */
typedef struct S765 T765;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_32] */
typedef struct S766 T766;

/* [detachable] SPECIAL [[attached] NATURAL_64] */
typedef struct S767 T767;

/* detachable DS_HASH_TABLE [[attached] NATURAL_64, [attached] NATURAL_32] */
typedef struct S768 T768;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] NATURAL_64, [attached] NATURAL_32] */
typedef struct S769 T769;

/* [detachable] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]] */
typedef struct S770 T770;

/* [detachable] SPECIAL [[attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]]] */
typedef struct S771 T771;

/* [detachable] KL_INTEGER_ROUTINES */
typedef struct S772 T772;

/* [detachable] UC_UTF8_STRING */
typedef struct S775 T775;

/* [detachable] THREAD_ATTRIBUTES */
typedef struct S776 T776;

/* [detachable] CELL [[attached] BOOLEAN] */
typedef struct S777 T777;

/* [detachable] CHARACTER_PROPERTY */
typedef struct S780 T780;

/* [detachable] STRING_8_SEARCHER */
typedef struct S781 T781;

/* detachable ARRAYED_LIST [[attached] INTEGER_32] */
typedef struct S782 T782;

/* [detachable] ET_SYSTEM_ERROR */
typedef struct S783 T783;

/* [detachable] ET_INTERNAL_ERROR */
typedef struct S784 T784;

/* [detachable] ET_VALIDITY_ERROR */
typedef struct S785 T785;

/* detachable ET_ECF_NOTE_ELEMENT */
typedef struct S786 T786;

/* detachable ET_ECF_CLUSTERS */
typedef struct S787 T787;

/* detachable ET_ECF_ADAPTED_LIBRARIES */
typedef struct S788 T788;

/* detachable ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
typedef struct S789 T789;

/* detachable ET_ECF_FILE_RULES */
typedef struct S790 T790;

/* detachable ET_ECF_EXTERNAL_CFLAGS */
typedef struct S791 T791;

/* detachable ET_ECF_EXTERNAL_INCLUDES */
typedef struct S792 T792;

/* detachable ET_ECF_EXTERNAL_LIBRARIES */
typedef struct S793 T793;

/* detachable ET_ECF_EXTERNAL_LINKER_FLAGS */
typedef struct S794 T794;

/* detachable ET_ECF_EXTERNAL_MAKES */
typedef struct S795 T795;

/* detachable ET_ECF_EXTERNAL_OBJECTS */
typedef struct S796 T796;

/* detachable ET_ECF_EXTERNAL_RESOURCES */
typedef struct S797 T797;

/* detachable DS_ARRAYED_LIST [[attached] ET_ECF_ACTION] */
typedef struct S798 T798;

/* detachable ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
typedef struct S799 T799;

/* detachable ET_ECF_VERSION */
typedef struct S801 T801;

/* detachable ET_ECF_CLUSTER */
typedef struct S802 T802;

/* detachable ET_ECF_EXTERNAL_CFLAG */
typedef struct S803 T803;

/* detachable ET_ECF_EXTERNAL_INCLUDE */
typedef struct S804 T804;

/* detachable ET_ECF_EXTERNAL_LIBRARY */
typedef struct S805 T805;

/* detachable ET_ECF_EXTERNAL_LINKER_FLAG */
typedef struct S806 T806;

/* detachable ET_ECF_EXTERNAL_MAKE */
typedef struct S807 T807;

/* detachable ET_ECF_EXTERNAL_OBJECT */
typedef struct S808 T808;

/* detachable ET_ECF_EXTERNAL_RESOURCE */
typedef struct S809 T809;

/* detachable ET_ECF_FILE_RULE */
typedef struct S810 T810;

/* detachable ET_ECF_ACTION */
typedef struct S811 T811;

/* detachable DS_HASH_TABLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
typedef struct S812 T812;

/* detachable DS_ARRAYED_LIST [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S813 T813;

/* [detachable] TUPLE [[attached] ET_ECF_OPTIONS] */
typedef struct S814 T814;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_OPTIONS]] */
typedef struct S815 T815;

/* detachable ET_ECF_ROOT_CLASS */
typedef struct S816 T816;

/* [detachable] ET_ECF_ROOT_ALL_CLASSES */
typedef struct S817 T817;

/* [detachable] ET_ECF_ORED_CONDITIONS */
typedef struct S818 T818;

/* detachable ET_ECF_ANDED_CONDITIONS */
typedef struct S819 T819;

/* detachable ET_ECF_VISIBLE_CLASS */
typedef struct S820 T820;

/* detachable ET_ECF_BUILD_CONDITION */
typedef struct S822 T822;

/* detachable ET_ECF_CONCURRENCY_CONDITION */
typedef struct S823 T823;

/* detachable ET_ECF_VOID_SAFETY_CONDITION */
typedef struct S824 T824;

/* detachable ET_ECF_CUSTOM_CONDITION */
typedef struct S825 T825;

/* detachable ET_ECF_DOTNET_CONDITION */
typedef struct S826 T826;

/* detachable ET_ECF_DYNAMIC_RUNTIME_CONDITION */
typedef struct S827 T827;

/* detachable ET_ECF_PLATFORM_CONDITION */
typedef struct S828 T828;

/* [detachable] ET_ECF_COMPILER_VERSION_CONDITION */
typedef struct S830 T830;

/* [detachable] ET_ECF_MSIL_CLR_VERSION_CONDITION */
typedef struct S831 T831;

/* [detachable] UT_COUNTER */
typedef struct S833 T833;

/* [detachable] KL_AGENT_ROUTINES [[attached] ET_CLASS] */
typedef struct S834 T834;

/* [detachable] TUPLE [[attached] UT_COUNTER] */
typedef struct S835 T835;

/* [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_CLASS], [attached] PROCEDURE [[attached] TUPLE]] */
typedef struct S836 T836;

/* [detachable] TUPLE [[attached] ET_UNIVERSE] */
typedef struct S837 T837;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
typedef struct S838 T838;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]]] */
typedef struct S839 T839;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
typedef struct S840 T840;

/* [detachable] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S842 T842;

/* [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S843 T843;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
typedef struct S844 T844;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
typedef struct S845 T845;

/* [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY]] */
typedef struct S846 T846;

/* [detachable] TUPLE [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S847 T847;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE]] */
typedef struct S848 T848;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]]] */
typedef struct S849 T849;

/* [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLIES] */
typedef struct S850 T850;

/* [detachable] TUPLE [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S851 T851;

/* [detachable] ET_LIBRARY */
typedef struct S852 T852;

/* [detachable] TUPLE [[attached] ET_LIBRARY] */
typedef struct S853 T853;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_LIBRARY]] */
typedef struct S854 T854;

/* [detachable] DS_HASH_SET [[attached] ET_UNIVERSE] */
typedef struct S855 T855;

/* [detachable] TUPLE [[attached] ET_CLUSTER] */
typedef struct S857 T857;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]] */
typedef struct S858 T858;

/* [detachable] TUPLE [[attached] ET_ECF_SYSTEM] */
typedef struct S859 T859;

/* [detachable] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S860 T860;

/* [detachable] TUPLE [[attached] ET_AST_PROCESSOR] */
typedef struct S861 T861;

/* [detachable] TUPLE [[attached] ET_MASTER_CLASS] */
typedef struct S862 T862;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
typedef struct S863 T863;

/* [detachable] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S864 T864;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY]] */
typedef struct S865 T865;

/* [detachable] ET_ANCESTORS_STATUS_CHECKER */
typedef struct S866 T866;

/* [detachable] ET_FLATTENING_STATUS_CHECKER */
typedef struct S867 T867;

/* [detachable] ET_INTERFACE_STATUS_CHECKER */
typedef struct S868 T868;

/* [detachable] ET_IMPLEMENTATION_STATUS_CHECKER */
typedef struct S869 T869;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]] */
typedef struct S870 T870;

/* [detachable] TUPLE [[attached] ET_ANCESTORS_STATUS_CHECKER] */
typedef struct S872 T872;

/* [detachable] TUPLE [[attached] ET_FLATTENING_STATUS_CHECKER] */
typedef struct S873 T873;

/* [detachable] TUPLE [[attached] ET_INTERFACE_STATUS_CHECKER] */
typedef struct S874 T874;

/* [detachable] TUPLE [[attached] ET_IMPLEMENTATION_STATUS_CHECKER] */
typedef struct S875 T875;

/* [detachable] TUPLE [BOOLEAN] */
typedef struct S876 T876;

/* [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_UNIVERSE]] */
typedef struct S877 T877;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]]] */
typedef struct S880 T880;

/* [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE]] */
typedef struct S884 T884;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]]] */
typedef struct S885 T885;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_CLASS], [attached] BOOLEAN]] */
typedef struct S886 T886;

/* [detachable] ET_NONE_GROUP */
typedef struct S887 T887;

/* detachable ET_PARENTHESIS_EXPRESSION */
typedef struct S888 T888;

/* detachable DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S890 T890;

/* [detachable] ET_FORMAL_PARAMETER */
typedef struct S891 T891;

/* [detachable] STD_FILES */
typedef struct S892 T892;

/* TYPED_POINTER [[attached] NATURAL_8] */
typedef struct S893 T893;
extern T0* GE_boxed893(T893 a1);
extern T0* GE_boxed_pointer893(T893* a1);
typedef struct Sb893 Tb893;

/* [detachable] XM_LINKED_LIST [[attached] XM_ELEMENT_NODE] */
typedef struct S894 T894;

/* [detachable] UC_STRING_EQUALITY_TESTER */
typedef struct S896 T896;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
typedef struct S898 T898;

/* [detachable] DS_LINKED_LIST [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
typedef struct S899 T899;

/* [detachable] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE] */
typedef struct S900 T900;

/* [detachable] XM_EIFFEL_INPUT_STREAM */
typedef struct S901 T901;

/* [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
typedef struct S902 T902;

/* [detachable] DS_HASH_SET [[attached] ET_LIBRARY] */
typedef struct S903 T903;

/* [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_LIBRARY]] */
typedef struct S904 T904;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XM_DOCUMENT_NODE] */
typedef struct S905 T905;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET] */
typedef struct S906 T906;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S907 T907;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S909 T909;

/* [detachable] SPECIAL [[attached] ET_QUERY] */
typedef struct S910 T910;

/* detachable ET_ALIAS_NAME_LIST */
typedef struct S911 T911;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_QUERY] */
typedef struct S912 T912;

/* [detachable] SPECIAL [[attached] ET_PROCEDURE] */
typedef struct S913 T913;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PROCEDURE] */
typedef struct S914 T914;

/* [detachable] TYPE [[attached] INTEGER_32] */
#define T919 EIF_TYPE_OBJ

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
typedef struct S921 T921;

/* [detachable] SPECIAL [[attached] ET_BASE_TYPE] */
typedef struct S923 T923;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE] */
typedef struct S924 T924;

/* [detachable] INTEGER_OVERFLOW_CHECKER */
typedef struct S925 T925;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S926 T926;

/* [detachable] SPECIAL [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S927 T927;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S928 T928;

/* detachable C_DATE */
typedef struct S929 T929;

/* [detachable] YY_FILE_BUFFER */
typedef struct S930 T930;

/* [detachable] KL_STDIN_FILE */
typedef struct S931 T931;

/* [detachable] YY_BUFFER */
typedef struct S932 T932;

/* [detachable] YY_UNICODE_FILE_BUFFER */
typedef struct S936 T936;

/* [detachable] DS_ARRAYED_STACK [[attached] INTEGER_32] */
typedef struct S937 T937;

/* [detachable] DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef struct S938 T938;

/* [detachable] DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
typedef struct S939 T939;

/* [detachable] DS_ARRAYED_STACK [detachable ET_KEYWORD] */
typedef struct S940 T940;

/* [detachable] DS_ARRAYED_STACK [detachable ET_SYMBOL] */
typedef struct S941 T941;

/* [detachable] DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
typedef struct S942 T942;

/* [detachable] DS_ARRAYED_STACK [[attached] ET_OBJECT_TEST_LIST] */
typedef struct S943 T943;

/* [detachable] DS_ARRAYED_STACK [detachable ET_ITERATION_COMPONENT_LIST] */
typedef struct S944 T944;

/* [detachable] DS_ARRAYED_STACK [[attached] ET_ITERATION_COMPONENT_LIST] */
typedef struct S945 T945;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ASSERTION_ITEM] */
typedef struct S946 T946;

/* [detachable] DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
typedef struct S947 T947;

/* [detachable] DS_HASH_SET [[attached] ET_NAMED_CLASS] */
typedef struct S948 T948;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
typedef struct S949 T949;

/* [detachable] SPECIAL [detachable ET_KEYWORD] */
typedef struct S950 T950;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
typedef struct S951 T951;

/* detachable ET_AGENT_KEYWORD */
typedef struct S952 T952;

/* [detachable] SPECIAL [detachable ET_AGENT_KEYWORD] */
typedef struct S953 T953;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
typedef struct S954 T954;

/* detachable ET_PRECURSOR_KEYWORD */
typedef struct S955 T955;

/* [detachable] SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
typedef struct S956 T956;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
typedef struct S957 T957;

/* [detachable] SPECIAL [detachable ET_SYMBOL] */
typedef struct S958 T958;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
typedef struct S959 T959;

/* [detachable] SPECIAL [detachable ET_POSITION] */
typedef struct S960 T960;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
typedef struct S961 T961;

/* [detachable] SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
typedef struct S963 T963;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
typedef struct S964 T964;

/* [detachable] SPECIAL [detachable ET_BREAK] */
typedef struct S966 T966;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
typedef struct S967 T967;

/* [detachable] SPECIAL [detachable ET_CHARACTER_CONSTANT] */
typedef struct S969 T969;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
typedef struct S970 T970;

/* [detachable] SPECIAL [detachable ET_CURRENT] */
typedef struct S971 T971;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
typedef struct S972 T972;

/* detachable ET_FREE_OPERATOR */
typedef struct S973 T973;

/* [detachable] SPECIAL [detachable ET_FREE_OPERATOR] */
typedef struct S974 T974;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
typedef struct S975 T975;

/* [detachable] SPECIAL [detachable ET_IDENTIFIER] */
typedef struct S976 T976;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
typedef struct S977 T977;

/* [detachable] SPECIAL [detachable ET_INTEGER_CONSTANT] */
typedef struct S979 T979;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
typedef struct S980 T980;

/* detachable ET_KEYWORD_OPERATOR */
typedef struct S981 T981;

/* [detachable] SPECIAL [detachable ET_KEYWORD_OPERATOR] */
typedef struct S982 T982;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
typedef struct S983 T983;

/* [detachable] SPECIAL [detachable ET_MANIFEST_STRING] */
typedef struct S984 T984;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
typedef struct S985 T985;

/* [detachable] SPECIAL [detachable ET_REAL_CONSTANT] */
typedef struct S987 T987;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
typedef struct S988 T988;

/* [detachable] SPECIAL [detachable ET_RESULT] */
typedef struct S989 T989;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
typedef struct S990 T990;

/* detachable ET_RETRY_INSTRUCTION */
typedef struct S991 T991;

/* [detachable] SPECIAL [detachable ET_RETRY_INSTRUCTION] */
typedef struct S992 T992;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
typedef struct S993 T993;

/* detachable ET_SYMBOL_OPERATOR */
typedef struct S994 T994;

/* [detachable] SPECIAL [detachable ET_SYMBOL_OPERATOR] */
typedef struct S995 T995;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_VOID] */
typedef struct S996 T996;

/* detachable ET_VOID */
typedef struct S997 T997;

/* [detachable] SPECIAL [detachable ET_VOID] */
typedef struct S998 T998;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
typedef struct S999 T999;

/* detachable ET_SEMICOLON_SYMBOL */
typedef struct S1000 T1000;

/* [detachable] SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
typedef struct S1001 T1001;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
typedef struct S1002 T1002;

/* [detachable] SPECIAL [detachable ET_BRACKET_SYMBOL] */
typedef struct S1003 T1003;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
typedef struct S1004 T1004;

/* detachable ET_QUESTION_MARK_SYMBOL */
typedef struct S1005 T1005;

/* [detachable] SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
typedef struct S1006 T1006;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
typedef struct S1007 T1007;

/* detachable ET_ACROSS_EXPRESSION */
typedef struct S1008 T1008;

/* [detachable] SPECIAL [detachable ET_ACROSS_EXPRESSION] */
typedef struct S1009 T1009;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
typedef struct S1010 T1010;

/* detachable ET_ACROSS_INSTRUCTION */
typedef struct S1011 T1011;

/* [detachable] SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
typedef struct S1012 T1012;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S1013 T1013;

/* [detachable] SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S1014 T1014;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S1015 T1015;

/* [detachable] SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S1016 T1016;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
typedef struct S1017 T1017;

/* [detachable] SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
typedef struct S1018 T1018;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
typedef struct S1019 T1019;

/* [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
typedef struct S1020 T1020;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S1021 T1021;

/* [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S1023 T1023;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
typedef struct S1024 T1024;

/* detachable ET_AGENT_ARGUMENT_OPERAND_LIST */
typedef struct S1025 T1025;

/* [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
typedef struct S1026 T1026;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
typedef struct S1027 T1027;

/* [detachable] SPECIAL [detachable ET_AGENT_TARGET] */
typedef struct S1028 T1028;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
typedef struct S1029 T1029;

/* detachable ET_ALIAS_NAME */
typedef struct S1030 T1030;

/* [detachable] SPECIAL [detachable ET_ALIAS_NAME] */
typedef struct S1031 T1031;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME_LIST] */
typedef struct S1032 T1032;

/* [detachable] SPECIAL [detachable ET_ALIAS_NAME_LIST] */
typedef struct S1033 T1033;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
typedef struct S1034 T1034;

/* [detachable] SPECIAL [detachable ET_ASSIGNER] */
typedef struct S1036 T1036;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
typedef struct S1037 T1037;

/* detachable ET_BRACKET_EXPRESSION */
typedef struct S1038 T1038;

/* [detachable] SPECIAL [detachable ET_BRACKET_EXPRESSION] */
typedef struct S1039 T1039;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
typedef struct S1040 T1040;

/* detachable ET_CALL_AGENT */
typedef struct S1041 T1041;

/* [detachable] SPECIAL [detachable ET_CALL_AGENT] */
typedef struct S1042 T1042;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S1043 T1043;

/* [detachable] SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S1044 T1044;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
typedef struct S1045 T1045;

/* detachable ET_CHECK_INSTRUCTION */
typedef struct S1046 T1046;

/* [detachable] SPECIAL [detachable ET_CHECK_INSTRUCTION] */
typedef struct S1047 T1047;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
typedef struct S1048 T1048;

/* [detachable] SPECIAL [detachable ET_CHOICE] */
typedef struct S1050 T1050;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
typedef struct S1051 T1051;

/* [detachable] SPECIAL [detachable ET_CHOICE_CONSTANT] */
typedef struct S1053 T1053;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
typedef struct S1054 T1054;

/* [detachable] SPECIAL [detachable ET_CHOICE_ITEM] */
typedef struct S1056 T1056;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
typedef struct S1057 T1057;

/* detachable ET_CHOICE_LIST */
typedef struct S1058 T1058;

/* [detachable] SPECIAL [detachable ET_CHOICE_LIST] */
typedef struct S1059 T1059;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
typedef struct S1060 T1060;

/* [detachable] SPECIAL [detachable ET_CLASS] */
typedef struct S1061 T1061;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
typedef struct S1062 T1062;

/* [detachable] SPECIAL [detachable ET_CLIENT_ITEM] */
typedef struct S1063 T1063;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
typedef struct S1064 T1064;

/* detachable ET_CLIENTS */
typedef struct S1065 T1065;

/* [detachable] SPECIAL [detachable ET_CLIENTS] */
typedef struct S1066 T1066;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
typedef struct S1067 T1067;

/* [detachable] SPECIAL [detachable ET_COMPOUND] */
typedef struct S1068 T1068;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
typedef struct S1069 T1069;

/* [detachable] SPECIAL [detachable ET_CONDITIONAL] */
typedef struct S1071 T1071;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
typedef struct S1072 T1072;

/* [detachable] SPECIAL [detachable ET_CONSTANT] */
typedef struct S1073 T1073;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S1074 T1074;

/* [detachable] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S1076 T1076;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
typedef struct S1077 T1077;

/* detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
typedef struct S1078 T1078;

/* [detachable] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
typedef struct S1079 T1079;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
typedef struct S1080 T1080;

/* detachable ET_CONSTRAINT_CREATOR */
typedef struct S1081 T1081;

/* [detachable] SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
typedef struct S1082 T1082;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_RENAME_LIST] */
typedef struct S1083 T1083;

/* detachable ET_CONSTRAINT_RENAME_LIST */
typedef struct S1084 T1084;

/* [detachable] SPECIAL [detachable ET_CONSTRAINT_RENAME_LIST] */
typedef struct S1085 T1085;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
typedef struct S1086 T1086;

/* [detachable] SPECIAL [detachable ET_CONSTRAINT_TYPE] */
typedef struct S1088 T1088;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
typedef struct S1089 T1089;

/* [detachable] SPECIAL [detachable ET_CONVERT_FEATURE] */
typedef struct S1091 T1091;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
typedef struct S1092 T1092;

/* [detachable] SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
typedef struct S1094 T1094;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
typedef struct S1095 T1095;

/* detachable ET_CONVERT_FEATURE_LIST */
typedef struct S1096 T1096;

/* [detachable] SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
typedef struct S1097 T1097;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
typedef struct S1098 T1098;

/* [detachable] SPECIAL [detachable ET_CREATE_EXPRESSION] */
typedef struct S1099 T1099;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
typedef struct S1100 T1100;

/* detachable ET_CREATION_REGION */
typedef struct S1101 T1101;

/* [detachable] SPECIAL [detachable ET_CREATION_REGION] */
typedef struct S1102 T1102;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
typedef struct S1103 T1103;

/* detachable ET_CREATOR */
typedef struct S1104 T1104;

/* [detachable] SPECIAL [detachable ET_CREATOR] */
typedef struct S1105 T1105;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
typedef struct S1106 T1106;

/* detachable ET_CREATOR_LIST */
typedef struct S1107 T1107;

/* [detachable] SPECIAL [detachable ET_CREATOR_LIST] */
typedef struct S1108 T1108;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
typedef struct S1109 T1109;

/* detachable ET_DEBUG_INSTRUCTION */
typedef struct S1110 T1110;

/* [detachable] SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
typedef struct S1111 T1111;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
typedef struct S1112 T1112;

/* detachable ET_ELSEIF_EXPRESSION */
typedef struct S1113 T1113;

/* [detachable] SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
typedef struct S1114 T1114;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
typedef struct S1115 T1115;

/* detachable ET_ELSEIF_EXPRESSION_LIST */
typedef struct S1116 T1116;

/* [detachable] SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
typedef struct S1117 T1117;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
typedef struct S1118 T1118;

/* detachable ET_ELSEIF_PART */
typedef struct S1119 T1119;

/* [detachable] SPECIAL [detachable ET_ELSEIF_PART] */
typedef struct S1120 T1120;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
typedef struct S1121 T1121;

/* detachable ET_ELSEIF_PART_LIST */
typedef struct S1122 T1122;

/* [detachable] SPECIAL [detachable ET_ELSEIF_PART_LIST] */
typedef struct S1123 T1123;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
typedef struct S1124 T1124;

/* [detachable] SPECIAL [detachable ET_EXPORT] */
typedef struct S1126 T1126;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
typedef struct S1127 T1127;

/* [detachable] SPECIAL [detachable ET_EXPORT_LIST] */
typedef struct S1128 T1128;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
typedef struct S1129 T1129;

/* [detachable] SPECIAL [detachable ET_EXPRESSION] */
typedef struct S1130 T1130;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
typedef struct S1131 T1131;

/* [detachable] SPECIAL [detachable ET_EXPRESSION_ITEM] */
typedef struct S1132 T1132;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
typedef struct S1133 T1133;

/* [detachable] SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
typedef struct S1134 T1134;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
typedef struct S1135 T1135;

/* [detachable] SPECIAL [detachable ET_EXTERNAL_ALIAS] */
typedef struct S1137 T1137;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
typedef struct S1138 T1138;

/* detachable ET_FEATURE_CLAUSE */
typedef struct S1139 T1139;

/* [detachable] SPECIAL [detachable ET_FEATURE_CLAUSE] */
typedef struct S1140 T1140;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
typedef struct S1141 T1141;

/* detachable ET_FEATURE_CLAUSE_LIST */
typedef struct S1142 T1142;

/* [detachable] SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
typedef struct S1143 T1143;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
typedef struct S1144 T1144;

/* detachable ET_FEATURE_EXPORT */
typedef struct S1145 T1145;

/* [detachable] SPECIAL [detachable ET_FEATURE_EXPORT] */
typedef struct S1146 T1146;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
typedef struct S1147 T1147;

/* [detachable] SPECIAL [detachable ET_FEATURE_NAME] */
typedef struct S1148 T1148;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
typedef struct S1149 T1149;

/* [detachable] SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
typedef struct S1151 T1151;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
typedef struct S1152 T1152;

/* [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT] */
typedef struct S1153 T1153;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S1154 T1154;

/* [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S1156 T1156;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef struct S1157 T1157;

/* [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef struct S1158 T1158;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
typedef struct S1159 T1159;

/* [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER] */
typedef struct S1160 T1160;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
typedef struct S1161 T1161;

/* [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
typedef struct S1163 T1163;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
typedef struct S1164 T1164;

/* [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
typedef struct S1165 T1165;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
typedef struct S1166 T1166;

/* detachable ET_IF_EXPRESSION */
typedef struct S1167 T1167;

/* [detachable] SPECIAL [detachable ET_IF_EXPRESSION] */
typedef struct S1168 T1168;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
typedef struct S1169 T1169;

/* detachable ET_IF_INSTRUCTION */
typedef struct S1170 T1170;

/* [detachable] SPECIAL [detachable ET_IF_INSTRUCTION] */
typedef struct S1171 T1171;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_LIST] */
typedef struct S1172 T1172;

/* detachable ET_INDEXING_LIST */
typedef struct S1173 T1173;

/* [detachable] SPECIAL [detachable ET_INDEXING_LIST] */
typedef struct S1174 T1174;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_ITEM] */
typedef struct S1175 T1175;

/* [detachable] SPECIAL [detachable ET_INDEXING_ITEM] */
typedef struct S1177 T1177;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM] */
typedef struct S1178 T1178;

/* [detachable] SPECIAL [detachable ET_INDEXING_TERM] */
typedef struct S1179 T1179;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_ITEM] */
typedef struct S1180 T1180;

/* [detachable] SPECIAL [detachable ET_INDEXING_TERM_ITEM] */
typedef struct S1182 T1182;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_LIST] */
typedef struct S1183 T1183;

/* detachable ET_INDEXING_TERM_LIST */
typedef struct S1184 T1184;

/* [detachable] SPECIAL [detachable ET_INDEXING_TERM_LIST] */
typedef struct S1185 T1185;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
typedef struct S1186 T1186;

/* [detachable] SPECIAL [detachable ET_INLINE_AGENT] */
typedef struct S1187 T1187;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_EXPRESSION] */
typedef struct S1188 T1188;

/* detachable ET_INSPECT_EXPRESSION */
typedef struct S1189 T1189;

/* [detachable] SPECIAL [detachable ET_INSPECT_EXPRESSION] */
typedef struct S1190 T1190;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
typedef struct S1191 T1191;

/* detachable ET_INSPECT_INSTRUCTION */
typedef struct S1192 T1192;

/* [detachable] SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
typedef struct S1193 T1193;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
typedef struct S1194 T1194;

/* [detachable] SPECIAL [detachable ET_INSTRUCTION] */
typedef struct S1196 T1196;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
typedef struct S1197 T1197;

/* detachable ET_INVARIANTS */
typedef struct S1198 T1198;

/* [detachable] SPECIAL [detachable ET_INVARIANTS] */
typedef struct S1199 T1199;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
typedef struct S1200 T1200;

/* [detachable] SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
typedef struct S1201 T1201;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
typedef struct S1202 T1202;

/* [detachable] SPECIAL [detachable ET_LIKE_TYPE] */
typedef struct S1204 T1204;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
typedef struct S1205 T1205;

/* detachable ET_LOCAL_VARIABLE */
typedef struct S1206 T1206;

/* [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE] */
typedef struct S1207 T1207;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
typedef struct S1208 T1208;

/* [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
typedef struct S1210 T1210;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
typedef struct S1211 T1211;

/* detachable ET_LOCAL_VARIABLE_LIST */
typedef struct S1212 T1212;

/* [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
typedef struct S1213 T1213;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
typedef struct S1214 T1214;

/* detachable ET_LOOP_INVARIANTS */
typedef struct S1215 T1215;

/* [detachable] SPECIAL [detachable ET_LOOP_INVARIANTS] */
typedef struct S1216 T1216;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
typedef struct S1217 T1217;

/* detachable ET_MANIFEST_ARRAY */
typedef struct S1218 T1218;

/* [detachable] SPECIAL [detachable ET_MANIFEST_ARRAY] */
typedef struct S1219 T1219;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
typedef struct S1220 T1220;

/* [detachable] SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
typedef struct S1222 T1222;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
typedef struct S1223 T1223;

/* detachable ET_MANIFEST_STRING_LIST */
typedef struct S1224 T1224;

/* [detachable] SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
typedef struct S1225 T1225;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
typedef struct S1226 T1226;

/* [detachable] SPECIAL [detachable ET_MANIFEST_TUPLE] */
typedef struct S1227 T1227;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
typedef struct S1228 T1228;

/* [detachable] SPECIAL [detachable ET_OBSOLETE] */
typedef struct S1230 T1230;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
typedef struct S1231 T1231;

/* detachable ET_PARENTHESIZED_EXPRESSION */
typedef struct S1232 T1232;

/* [detachable] SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
typedef struct S1233 T1233;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
typedef struct S1234 T1234;

/* [detachable] SPECIAL [detachable ET_PARENT] */
typedef struct S1235 T1235;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
typedef struct S1236 T1236;

/* detachable ET_PARENT_CLAUSE_LIST */
typedef struct S1237 T1237;

/* [detachable] SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
typedef struct S1238 T1238;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
typedef struct S1239 T1239;

/* [detachable] SPECIAL [detachable ET_PARENT_ITEM] */
typedef struct S1240 T1240;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
typedef struct S1241 T1241;

/* [detachable] SPECIAL [detachable ET_PARENT_LIST] */
typedef struct S1242 T1242;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
typedef struct S1243 T1243;

/* detachable ET_POSTCONDITIONS */
typedef struct S1244 T1244;

/* [detachable] SPECIAL [detachable ET_POSTCONDITIONS] */
typedef struct S1245 T1245;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
typedef struct S1246 T1246;

/* [detachable] SPECIAL [detachable ET_PRECONDITIONS] */
typedef struct S1247 T1247;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
typedef struct S1248 T1248;

/* [detachable] SPECIAL [detachable ET_PROCEDURE] */
typedef struct S1249 T1249;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
typedef struct S1250 T1250;

/* [detachable] SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
typedef struct S1252 T1252;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUANTIFIER_EXPRESSION] */
typedef struct S1253 T1253;

/* detachable ET_QUANTIFIER_EXPRESSION */
typedef struct S1254 T1254;

/* [detachable] SPECIAL [detachable ET_QUANTIFIER_EXPRESSION] */
typedef struct S1255 T1255;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
typedef struct S1256 T1256;

/* [detachable] SPECIAL [detachable ET_QUERY] */
typedef struct S1257 T1257;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
typedef struct S1258 T1258;

/* [detachable] SPECIAL [detachable ET_RENAME_ITEM] */
typedef struct S1260 T1260;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
typedef struct S1261 T1261;

/* [detachable] SPECIAL [detachable ET_RENAME_LIST] */
typedef struct S1262 T1262;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_REPEAT_INSTRUCTION] */
typedef struct S1263 T1263;

/* detachable ET_REPEAT_INSTRUCTION */
typedef struct S1264 T1264;

/* [detachable] SPECIAL [detachable ET_REPEAT_INSTRUCTION] */
typedef struct S1265 T1265;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
typedef struct S1266 T1266;

/* detachable ET_STATIC_CALL_EXPRESSION */
typedef struct S1267 T1267;

/* [detachable] SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
typedef struct S1268 T1268;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
typedef struct S1269 T1269;

/* detachable ET_STRIP_EXPRESSION */
typedef struct S1270 T1270;

/* [detachable] SPECIAL [detachable ET_STRIP_EXPRESSION] */
typedef struct S1271 T1271;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
typedef struct S1272 T1272;

/* [detachable] SPECIAL [detachable ET_TYPE] */
typedef struct S1273 T1273;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT] */
typedef struct S1274 T1274;

/* [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT] */
typedef struct S1276 T1276;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_ITEM] */
typedef struct S1277 T1277;

/* [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT_ITEM] */
typedef struct S1279 T1279;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_LIST] */
typedef struct S1280 T1280;

/* detachable ET_TYPE_CONSTRAINT_LIST */
typedef struct S1281 T1281;

/* [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT_LIST] */
typedef struct S1282 T1282;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
typedef struct S1283 T1283;

/* [detachable] SPECIAL [detachable ET_TYPE_ITEM] */
typedef struct S1285 T1285;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
typedef struct S1286 T1286;

/* [detachable] SPECIAL [detachable ET_TYPE_LIST] */
typedef struct S1288 T1288;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
typedef struct S1289 T1289;

/* detachable ET_VARIANT */
typedef struct S1290 T1290;

/* [detachable] SPECIAL [detachable ET_VARIANT] */
typedef struct S1291 T1291;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION] */
typedef struct S1292 T1292;

/* detachable ET_WHEN_EXPRESSION */
typedef struct S1293 T1293;

/* [detachable] SPECIAL [detachable ET_WHEN_EXPRESSION] */
typedef struct S1294 T1294;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION_LIST] */
typedef struct S1295 T1295;

/* detachable ET_WHEN_EXPRESSION_LIST */
typedef struct S1296 T1296;

/* [detachable] SPECIAL [detachable ET_WHEN_EXPRESSION_LIST] */
typedef struct S1297 T1297;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
typedef struct S1298 T1298;

/* detachable ET_WHEN_PART */
typedef struct S1299 T1299;

/* [detachable] SPECIAL [detachable ET_WHEN_PART] */
typedef struct S1300 T1300;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
typedef struct S1301 T1301;

/* detachable ET_WHEN_PART_LIST */
typedef struct S1302 T1302;

/* [detachable] SPECIAL [detachable ET_WHEN_PART_LIST] */
typedef struct S1303 T1303;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
typedef struct S1304 T1304;

/* [detachable] SPECIAL [detachable ET_WRITABLE] */
typedef struct S1305 T1305;

/* [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_CLASS] */
typedef struct S1306 T1306;

/* [detachable] DS_HASH_TABLE [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
typedef struct S1307 T1307;

/* [detachable] ET_PARENT_CHECKER1 */
typedef struct S1308 T1308;

/* [detachable] ET_FORMAL_PARAMETER_CHECKER1 */
typedef struct S1309 T1309;

/* [detachable] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
typedef struct S1310 T1310;

/* [detachable] ET_FEATURE_NAME_TESTER */
typedef struct S1311 T1311;

/* [detachable] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
typedef struct S1313 T1313;

/* [detachable] ET_ALIAS_NAME_TESTER */
typedef struct S1314 T1314;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_CLIENT_LIST] */
typedef struct S1316 T1316;

/* [detachable] DS_HASH_TABLE [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S1317 T1317;

/* [detachable] ET_FEATURE_ADAPTATION_RESOLVER */
typedef struct S1318 T1318;

/* [detachable] ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
typedef struct S1319 T1319;

/* [detachable] ET_IDENTIFIER_TYPE_RESOLVER */
typedef struct S1320 T1320;

/* [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
typedef struct S1321 T1321;

/* [detachable] ET_ANCHORED_TYPE_CHECKER */
typedef struct S1322 T1322;

/* [detachable] ET_SIGNATURE_CHECKER */
typedef struct S1323 T1323;

/* [detachable] ET_PARENT_CHECKER2 */
typedef struct S1324 T1324;

/* [detachable] ET_FORMAL_PARAMETER_CHECKER2 */
typedef struct S1325 T1325;

/* [detachable] ET_BUILTIN_FEATURE_CHECKER */
typedef struct S1326 T1326;

/* [detachable] ET_PRECURSOR_CHECKER */
typedef struct S1327 T1327;

/* [detachable] DS_HASH_TABLE [[attached] ET_FEATURE, [attached] INTEGER_32] */
typedef struct S1328 T1328;

/* [detachable] DS_HASH_SET [[attached] ET_CLASS] */
typedef struct S1329 T1329;

/* [detachable] ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
typedef struct S1330 T1330;

/* [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
typedef struct S1331 T1331;

/* [detachable] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_FEATURE_NAME] */
typedef struct S1332 T1332;

/* [detachable] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_ALIAS_NAME] */
typedef struct S1333 T1333;

/* [detachable] ET_RENAME */
typedef struct S1337 T1337;

/* [detachable] ET_PARENT_CHECKER3 */
typedef struct S1338 T1338;

/* [detachable] DS_HASH_SET [[attached] ET_PROCEDURE] */
typedef struct S1339 T1339;

/* [detachable] DS_HASH_SET [[attached] ET_QUERY] */
typedef struct S1340 T1340;

/* [detachable] ET_SUPPLIER_BUILDER */
typedef struct S1341 T1341;

/* detachable ET_PARENT_FEATURE */
typedef struct S1344 T1344;

/* [detachable] ET_INHERITED_FEATURE */
typedef struct S1345 T1345;

/* [detachable] SPECIAL [[attached] ET_CLASS] */
typedef struct S1346 T1346;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLASS] */
typedef struct S1347 T1347;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLASS] */
typedef struct S1348 T1348;

/* [detachable] DS_ARRAYED_STACK [[attached] YY_BUFFER] */
typedef struct S1349 T1349;

/* [detachable] UT_SYNTAX_ERROR */
typedef struct S1350 T1350;

/* [detachable] UT_TOO_MANY_INCLUDES_ERROR */
typedef struct S1351 T1351;

/* detachable ET_DYNAMIC_TARGET_LIST */
typedef struct S1353 T1353;

/* [detachable] SPECIAL [[attached] NATIVE_STRING] */
typedef struct S1354 T1354;

/* [detachable] ARRAY [detachable STRING_8] */
typedef struct S1355 T1355;

/* [detachable] XM_EIFFEL_CHARACTER_ENTITY */
typedef struct S1356 T1356;

/* detachable DS_BILINKABLE [[attached] XM_POSITION] */
typedef struct S1358 T1358;

/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_POSITION] */
typedef struct S1359 T1359;

/* detachable DS_LINKABLE [[attached] XM_EIFFEL_SCANNER] */
typedef struct S1360 T1360;

/* [detachable] SPECIAL [[attached] XM_EIFFEL_ENTITY_DEF] */
typedef struct S1361 T1361;

/* detachable DS_HASH_TABLE_CURSOR [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
typedef struct S1364 T1364;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_ENTITY_DEF] */
typedef struct S1365 T1365;

/* detachable KL_EQUALITY_TESTER [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S1368 T1368;

/* [detachable] DS_HASH_SET_CURSOR [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S1369 T1369;

/* [detachable] DS_BILINKED_LIST [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S1370 T1370;

/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S1371 T1371;

/* detachable DS_BILINKABLE [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S1372 T1372;

/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] STRING_8] */
typedef struct S1373 T1373;

/* detachable DS_BILINKABLE [[attached] STRING_8] */
typedef struct S1374 T1374;

/* [detachable] XM_NAMESPACE_RESOLVER_CONTEXT */
typedef struct S1375 T1375;

/* [detachable] DS_LINKED_QUEUE [detachable STRING_8] */
typedef struct S1378 T1378;

/* [detachable] DS_LINKED_QUEUE [[attached] STRING_8] */
typedef struct S1379 T1379;

/* detachable KL_EQUALITY_TESTER [[attached] XM_NAMESPACE] */
typedef struct S1380 T1380;

/* [detachable] DS_HASH_SET_CURSOR [[attached] XM_NAMESPACE] */
typedef struct S1381 T1381;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XM_NAMESPACE] */
typedef struct S1382 T1382;

/* [detachable] SPECIAL [[attached] XM_NAMESPACE] */
typedef struct S1383 T1383;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_CLUSTER] */
typedef struct S1384 T1384;

/* [detachable] SPECIAL [[attached] ET_PARENT_ITEM] */
typedef struct S1386 T1386;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_ITEM] */
typedef struct S1387 T1387;

/* [detachable] ET_BRACED_TYPE_LIST */
typedef struct S1388 T1388;

/* detachable DS_ARRAYED_LIST [[attached] ET_MASTER_CLASS] */
typedef struct S1389 T1389;

/* [detachable] SPECIAL [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S1390 T1390;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S1391 T1391;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S1392 T1392;

/* detachable DS_LINKABLE [[attached] XM_DOCUMENT_NODE] */
typedef struct S1393 T1393;

/* [detachable] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1395 T1395;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1396 T1396;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1397 T1397;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1398 T1398;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_FEATURE, [attached] INTEGER_32] */
typedef struct S1400 T1400;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
typedef struct S1402 T1402;

/* [detachable] SPECIAL [[attached] ET_CALL_NAME] */
typedef struct S1404 T1404;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
typedef struct S1406 T1406;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CALL_NAME] */
typedef struct S1407 T1407;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL, [attached] ET_CALL_NAME] */
typedef struct S1408 T1408;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, [attached] ET_CALL_NAME] */
typedef struct S1410 T1410;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
typedef struct S1411 T1411;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
typedef struct S1412 T1412;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S1413 T1413;

/* [detachable] SPECIAL [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1414 T1414;

/* [detachable] KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1415 T1415;

/* [detachable] DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1416 T1416;

/* [detachable] SPECIAL [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1418 T1418;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1419 T1419;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1420 T1420;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_CLASS] */
typedef struct S1421 T1421;

/* [detachable] SPECIAL [[attached] ET_ADAPTED_CLASS] */
typedef struct S1423 T1423;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_CLASS] */
typedef struct S1424 T1424;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_PROCEDURE] */
typedef struct S1425 T1425;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUERY] */
typedef struct S1426 T1426;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S1427 T1427;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ITERATION_COMPONENT] */
typedef struct S1428 T1428;

/* [detachable] DS_HASH_SET_CURSOR [[attached] INTEGER_32] */
typedef struct S1429 T1429;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_8] */
typedef struct S1433 T1433;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
typedef struct S1434 T1434;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] RX_CHARACTER_SET] */
typedef struct S1435 T1435;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] RX_CHARACTER_SET] */
typedef struct S1436 T1436;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_64] */
typedef struct S1443 T1443;

/* [detachable] TYPE [[attached] NATURAL_64] */
#define T1444 EIF_TYPE_OBJ

/* [detachable] TYPE [[attached] NATURAL_32] */
#define T1445 EIF_TYPE_OBJ

/* [detachable] KL_CHARACTER_BUFFER */
typedef struct S1446 T1446;

/* [detachable] UT_TRISTATE */
typedef struct S1448 T1448;

/* [detachable] KL_AGENT_ROUTINES [[attached] ANY] */
typedef struct S1449 T1449;

/* [detachable] TUPLE [[attached] UT_TRISTATE] */
typedef struct S1450 T1450;

/* [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ANY], [attached] PROCEDURE [[attached] TUPLE]] */
typedef struct S1451 T1451;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
typedef struct S1452 T1452;

/* [detachable] PREDICATE [[attached] TUPLE] */
typedef struct S1453 T1453;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
typedef struct S1455 T1455;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
typedef struct S1456 T1456;

/* [detachable] TUPLE [[attached] ET_ECF_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S1457 T1457;

/* [detachable] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE] */
typedef struct S1458 T1458;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
typedef struct S1459 T1459;

/* [detachable] TUPLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE]] */
typedef struct S1460 T1460;

/* [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE], [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]]] */
typedef struct S1461 T1461;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
typedef struct S1463 T1463;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN]] */
typedef struct S1464 T1464;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
typedef struct S1465 T1465;

/* [detachable] ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
typedef struct S1466 T1466;

/* TYPED_POINTER [[attached] NATURAL_16] */
typedef struct S1468 T1468;
extern T0* GE_boxed1468(T1468 a1);
extern T0* GE_boxed_pointer1468(T1468* a1);
typedef struct Sb1468 Tb1468;

/* [detachable] XM_COMMENT */
typedef struct S1469 T1469;

/* [detachable] XM_PROCESSING_INSTRUCTION */
typedef struct S1470 T1470;

/* [detachable] XM_CHARACTER_DATA */
typedef struct S1471 T1471;

/* [detachable] ET_LIKE_N */
typedef struct S1474 T1474;

/* [detachable] KL_STRING_INPUT_STREAM */
typedef struct S1477 T1477;

/* [detachable] KL_UNICODE_CHARACTER_BUFFER */
typedef struct S1479 T1479;

/* detachable ET_OBJECT_TEST_LIST */
typedef struct S1481 T1481;

/* detachable ET_C3_CHARACTER_CONSTANT */
typedef struct S1482 T1482;

/* detachable ET_REGULAR_MANIFEST_STRING */
typedef struct S1483 T1483;

/* detachable ET_SPECIAL_MANIFEST_STRING */
typedef struct S1484 T1484;

/* detachable ET_VERBATIM_STRING */
typedef struct S1485 T1485;

/* detachable ET_UNDERSCORED_INTEGER_CONSTANT */
typedef struct S1486 T1486;

/* detachable ET_HEXADECIMAL_INTEGER_CONSTANT */
typedef struct S1487 T1487;

/* detachable ET_OCTAL_INTEGER_CONSTANT */
typedef struct S1488 T1488;

/* detachable ET_BINARY_INTEGER_CONSTANT */
typedef struct S1489 T1489;

/* detachable ET_REGULAR_REAL_CONSTANT */
typedef struct S1490 T1490;

/* detachable ET_UNDERSCORED_REAL_CONSTANT */
typedef struct S1491 T1491;

/* detachable ET_TRUE_CONSTANT */
typedef struct S1493 T1493;

/* detachable ET_FALSE_CONSTANT */
typedef struct S1494 T1494;

/* detachable ET_C1_CHARACTER_CONSTANT */
typedef struct S1495 T1495;

/* detachable ET_C2_CHARACTER_CONSTANT */
typedef struct S1496 T1496;

/* detachable ET_TAGGED_INDEXING */
typedef struct S1498 T1498;

/* detachable ET_INDEXING */
typedef struct S1499 T1499;

/* detachable ET_CUSTOM_ATTRIBUTE */
typedef struct S1500 T1500;

/* detachable ET_CONSTRAINED_FORMAL_PARAMETER */
typedef struct S1501 T1501;

/* detachable ET_TYPE_RENAME_CONSTRAINT */
typedef struct S1502 T1502;

/* detachable ET_CONSTRAINT_NAMED_TYPE */
typedef struct S1503 T1503;

/* detachable ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
typedef struct S1504 T1504;

/* detachable ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
typedef struct S1505 T1505;

/* detachable ET_ALL_EXPORT */
typedef struct S1506 T1506;

/* detachable ET_CONVERT_FUNCTION */
typedef struct S1508 T1508;

/* detachable ET_CONVERT_PROCEDURE */
typedef struct S1509 T1509;

/* detachable ET_DO_FUNCTION */
typedef struct S1511 T1511;

/* detachable ET_ONCE_FUNCTION */
typedef struct S1512 T1512;

/* detachable ET_DEFERRED_FUNCTION */
typedef struct S1513 T1513;

/* detachable ET_ONCE_PROCEDURE */
typedef struct S1515 T1515;

/* detachable ET_EXTERNAL_PROCEDURE */
typedef struct S1516 T1516;

/* detachable ET_ALIASED_FEATURE_NAME */
typedef struct S1517 T1517;

/* detachable ET_ALIAS_FREE_NAME */
typedef struct S1518 T1518;

/* detachable ET_CLASS_ASSERTION */
typedef struct S1521 T1521;

/* detachable ET_LABELED_ACTUAL_PARAMETER */
typedef struct S1522 T1522;

/* detachable ET_LIKE_FEATURE */
typedef struct S1524 T1524;

/* detachable ET_QUALIFIED_LIKE_BRACED_TYPE */
typedef struct S1525 T1525;

/* detachable ET_QUALIFIED_LIKE_TYPE */
typedef struct S1526 T1526;

/* detachable ET_ASSIGNER_INSTRUCTION */
typedef struct S1527 T1527;

/* detachable ET_ASSIGNMENT */
typedef struct S1528 T1528;

/* detachable ET_ASSIGNMENT_ATTEMPT */
typedef struct S1529 T1529;

/* detachable ET_LOOP_INSTRUCTION */
typedef struct S1530 T1530;

/* detachable ET_BANG_INSTRUCTION */
typedef struct S1532 T1532;

/* detachable ET_CREATE_INSTRUCTION */
typedef struct S1533 T1533;

/* detachable ET_CHOICE_RANGE */
typedef struct S1534 T1534;

/* detachable ET_PRECURSOR_INSTRUCTION */
typedef struct S1535 T1535;

/* detachable ET_STATIC_CALL_INSTRUCTION */
typedef struct S1537 T1537;

/* detachable ET_PRECURSOR_EXPRESSION */
typedef struct S1538 T1538;

/* detachable ET_FEATURE_ADDRESS */
typedef struct S1539 T1539;

/* detachable ET_CURRENT_ADDRESS */
typedef struct S1540 T1540;

/* detachable ET_RESULT_ADDRESS */
typedef struct S1541 T1541;

/* detachable ET_EXPRESSION_ADDRESS */
typedef struct S1542 T1542;

/* detachable ET_INFIX_EXPRESSION */
typedef struct S1543 T1543;

/* detachable ET_INFIX_AND_THEN_OPERATOR */
typedef struct S1544 T1544;

/* detachable ET_INFIX_OR_ELSE_OPERATOR */
typedef struct S1545 T1545;

/* detachable ET_MANIFEST_TYPE */
typedef struct S1546 T1546;

/* detachable ET_PREFIX_EXPRESSION */
typedef struct S1547 T1547;

/* detachable ET_OLD_EXPRESSION */
typedef struct S1548 T1548;

/* detachable ET_OLD_OBJECT_TEST */
typedef struct S1549 T1549;

/* detachable ET_ONCE_MANIFEST_STRING */
typedef struct S1550 T1550;

/* detachable ET_ITERATION_CURSOR */
typedef struct S1551 T1551;

/* detachable ET_DO_FUNCTION_INLINE_AGENT */
typedef struct S1552 T1552;

/* detachable ET_ONCE_FUNCTION_INLINE_AGENT */
typedef struct S1553 T1553;

/* detachable ET_EXTERNAL_FUNCTION_INLINE_AGENT */
typedef struct S1554 T1554;

/* detachable ET_DO_PROCEDURE_INLINE_AGENT */
typedef struct S1555 T1555;

/* detachable ET_ONCE_PROCEDURE_INLINE_AGENT */
typedef struct S1556 T1556;

/* detachable ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
typedef struct S1557 T1557;

/* detachable ET_AGENT_OPEN_TARGET */
typedef struct S1558 T1558;

/* detachable ET_AGENT_TYPED_OPEN_ARGUMENT */
typedef struct S1559 T1559;

/* detachable ET_TAGGED_ASSERTION */
typedef struct S1562 T1562;

/* [detachable] ET_FILE_POSITION */
typedef struct S1564 T1564;

/* [detachable] DS_QUICK_SORTER [[attached] ET_QUERY] */
typedef struct S1566 T1566;

/* [detachable] DS_QUICK_SORTER [[attached] ET_PROCEDURE] */
typedef struct S1567 T1567;

/* [detachable] ET_REDECLARED_FEATURE */
typedef struct S1569 T1569;

/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
typedef struct S1571 T1571;

/* [detachable] TUPLE [[attached] ET_MASTER_CLASS, [attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1573 T1573;

/* [detachable] SPECIAL [[attached] NATURAL_16] */
typedef struct S1576 T1576;

/* [detachable] ARRAY [[attached] NATURAL_16] */
typedef struct S1577 T1577;

/* [detachable] ARRAY [[attached] NATURAL_32] */
typedef struct S1578 T1578;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_CLUSTER] */
typedef struct S1579 T1579;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S1580 T1580;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1581 T1581;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_FILE_RULE] */
typedef struct S1582 T1582;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S1583 T1583;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S1584 T1584;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S1585 T1585;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S1586 T1586;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S1587 T1587;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S1588 T1588;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S1589 T1589;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ACTION] */
typedef struct S1590 T1590;

/* [detachable] SPECIAL [[attached] ET_ECF_ACTION] */
typedef struct S1591 T1591;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ACTION] */
typedef struct S1592 T1592;

/* detachable DS_ARRAYED_LIST [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S1593 T1593;

/* [detachable] SPECIAL [[attached] ET_ECF_OPTIONS] */
typedef struct S1594 T1594;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_OPTIONS] */
typedef struct S1597 T1597;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
typedef struct S1598 T1598;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S1599 T1599;

/* [detachable] SPECIAL [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S1600 T1600;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S1601 T1601;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S1602 T1602;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S1603 T1603;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1604 T1604;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1605 T1605;

/* [detachable] SPECIAL [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1606 T1606;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_UNIVERSE] */
typedef struct S1608 T1608;

/* [detachable] SPECIAL [[attached] ET_UNIVERSE] */
typedef struct S1609 T1609;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_UNIVERSE] */
typedef struct S1611 T1611;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S1612 T1612;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S1614 T1614;

/* [detachable] SPECIAL [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S1615 T1615;

/* [detachable] ET_CLASS_TYPE_STATUS_CHECKER1 */
typedef struct S1616 T1616;

/* [detachable] ET_CLASS_TYPE_STATUS_CHECKER2 */
typedef struct S1617 T1617;

/* [detachable] ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
typedef struct S1618 T1618;

/* [detachable] ET_CLASS_TYPE_STATUS_CHECKER3 */
typedef struct S1619 T1619;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_NAMED_CLASS] */
typedef struct S1620 T1620;

/* [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S1621 T1621;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S1622 T1622;

/* [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S1623 T1623;

/* [detachable] CONSOLE */
typedef struct S1624 T1624;

/* [detachable] DS_LINKED_QUEUE [[attached] CHARACTER_8] */
typedef struct S1626 T1626;

/* [detachable] UC_UTF16_ROUTINES */
typedef struct S1627 T1627;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_LIBRARY] */
typedef struct S1628 T1628;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LIBRARY] */
typedef struct S1629 T1629;

/* [detachable] SPECIAL [[attached] ET_LIBRARY] */
typedef struct S1630 T1630;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET] */
typedef struct S1631 T1631;

/* [detachable] SPECIAL [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S1632 T1632;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S1633 T1633;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S1634 T1634;

/* [detachable] SPECIAL [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S1635 T1635;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S1636 T1636;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S1637 T1637;

/* [detachable] YY_UNICODE_BUFFER */
typedef struct S1638 T1638;

/* [detachable] SPECIAL [detachable ET_OBJECT_TEST_LIST] */
typedef struct S1639 T1639;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
typedef struct S1640 T1640;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST_LIST] */
typedef struct S1641 T1641;

/* [detachable] SPECIAL [[attached] ET_OBJECT_TEST_LIST] */
typedef struct S1642 T1642;

/* [detachable] SPECIAL [detachable ET_ITERATION_COMPONENT_LIST] */
typedef struct S1643 T1643;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ET_ITERATION_COMPONENT_LIST] */
typedef struct S1644 T1644;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ITERATION_COMPONENT_LIST] */
typedef struct S1645 T1645;

/* [detachable] SPECIAL [[attached] ET_ITERATION_COMPONENT_LIST] */
typedef struct S1646 T1646;

/* [detachable] SPECIAL [[attached] ET_ASSERTION_ITEM] */
typedef struct S1647 T1647;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTION_ITEM] */
typedef struct S1648 T1648;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ASSERTION_ITEM] */
typedef struct S1649 T1649;

/* [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
typedef struct S1650 T1650;

/* [detachable] SPECIAL [[attached] ET_NAMED_CLASS] */
typedef struct S1651 T1651;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_CLASS] */
typedef struct S1653 T1653;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_CLASS] */
typedef struct S1655 T1655;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
typedef struct S1656 T1656;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_BASE_TYPE_CONSTRAINT] */
typedef struct S1659 T1659;

/* [detachable] DS_ARRAYED_LIST [[attached] NATURAL_32] */
typedef struct S1660 T1660;

/* [detachable] ET_BASE_TYPE_CONSTRAINT_LIST */
typedef struct S1661 T1661;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
typedef struct S1662 T1662;

/* [detachable] SPECIAL [[attached] ET_FLATTENED_FEATURE] */
typedef struct S1663 T1663;

/* [detachable] SPECIAL [[attached] ET_FEATURE_NAME] */
typedef struct S1666 T1666;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FLATTENED_FEATURE] */
typedef struct S1667 T1667;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME] */
typedef struct S1668 T1668;

/* [detachable] SPECIAL [[attached] ET_ALIAS_NAME] */
typedef struct S1669 T1669;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ALIAS_NAME] */
typedef struct S1671 T1671;

/* detachable DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
typedef struct S1672 T1672;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_LIST] */
typedef struct S1673 T1673;

/* [detachable] SPECIAL [[attached] ET_CLIENT_LIST] */
typedef struct S1674 T1674;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLIENT_LIST] */
typedef struct S1675 T1675;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S1677 T1677;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT] */
typedef struct S1678 T1678;

/* [detachable] SPECIAL [[attached] ET_CLIENT] */
typedef struct S1679 T1679;

/* [detachable] DS_HASH_SET [[attached] ET_FEATURE_NAME] */
typedef struct S1680 T1680;

/* [detachable] DS_HASH_TABLE [[attached] BOOLEAN, [attached] ET_FEATURE_NAME] */
typedef struct S1681 T1681;

/* [detachable] DS_HASH_TABLE [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
typedef struct S1682 T1682;

/* [detachable] ET_REPLICATED_FEATURE */
typedef struct S1684 T1684;

/* [detachable] DS_LINKED_LIST [[attached] ET_ADAPTED_FEATURE] */
typedef struct S1685 T1685;

/* [detachable] DS_ARRAYED_LIST [[attached] ET_PARENT_FEATURE] */
typedef struct S1686 T1686;

/* [detachable] ET_DOTNET_SIGNATURE_TESTER */
typedef struct S1687 T1687;

/* [detachable] DS_HASH_SET [[attached] ET_DOTNET_FEATURE] */
typedef struct S1688 T1688;

/* [detachable] DS_HASH_TABLE [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
typedef struct S1690 T1690;

/* [detachable] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE] */
typedef struct S1692 T1692;

/* [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_LIKE_FEATURE] */
typedef struct S1693 T1693;

/* detachable DS_ARRAYED_LIST [[attached] ET_LIKE_FEATURE] */
typedef struct S1694 T1694;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
typedef struct S1695 T1695;

/* detachable DS_LINKED_LIST [[attached] ET_PARENT_FEATURE] */
typedef struct S1696 T1696;

/* [detachable] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
typedef struct S1697 T1697;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE, [attached] INTEGER_32] */
typedef struct S1698 T1698;

/* [detachable] TYPE [[attached] ET_FEATURE] */
#define T1699 EIF_TYPE_OBJ

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_CLASS] */
typedef struct S1701 T1701;

/* [detachable] TYPE [[attached] ET_CLASS] */
#define T1702 EIF_TYPE_OBJ

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_FEATURE_NAME] */
typedef struct S1703 T1703;

/* [detachable] SPECIAL [[attached] ET_RENAME] */
typedef struct S1704 T1704;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RENAME] */
typedef struct S1707 T1707;

/* [detachable] TYPE [[attached] ET_RENAME] */
#define T1708 EIF_TYPE_OBJ

/* [detachable] TYPE [[attached] ET_FEATURE_NAME] */
#define T1709 EIF_TYPE_OBJ

/* detachable DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_ALIAS_NAME] */
typedef struct S1711 T1711;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_PROCEDURE] */
typedef struct S1712 T1712;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_QUERY] */
typedef struct S1713 T1713;

/* [detachable] SPECIAL [[attached] YY_BUFFER] */
typedef struct S1714 T1714;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] YY_BUFFER] */
typedef struct S1715 T1715;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_TARGET] */
typedef struct S1716 T1716;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TARGET] */
typedef struct S1717 T1717;

/* [detachable] TYPE [detachable STRING_8] */
#define T1718 EIF_TYPE_OBJ

/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S1719 T1719;

/* detachable DS_BILINKABLE [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S1720 T1720;

/* [detachable] DS_BILINKED_LIST [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
typedef struct S1722 T1722;

/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
typedef struct S1723 T1723;

/* detachable DS_LINKABLE [detachable STRING_8] */
typedef struct S1724 T1724;

/* [detachable] SPECIAL [[attached] ET_CLUSTER] */
typedef struct S1725 T1725;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLUSTER] */
typedef struct S1726 T1726;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLUSTER] */
typedef struct S1727 T1727;

/* [detachable] SPECIAL [[attached] ET_TYPE_ITEM] */
typedef struct S1728 T1728;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_ITEM] */
typedef struct S1729 T1729;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_MASTER_CLASS] */
typedef struct S1730 T1730;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1731 T1731;

/* [detachable] TUPLE [[attached] ET_ECF_CLUSTER] */
typedef struct S1732 T1732;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_CLUSTER]] */
typedef struct S1733 T1733;

/* [detachable] TUPLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
typedef struct S1734 T1734;

/* [detachable] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S1736 T1736;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY]] */
typedef struct S1737 T1737;

/* [detachable] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1738 T1738;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
typedef struct S1739 T1739;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] STRING_8]] */
typedef struct S1742 T1742;

/* [detachable] TUPLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
typedef struct S1743 T1743;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8] */
typedef struct S1744 T1744;

/* [detachable] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
typedef struct S1745 T1745;

/* detachable DS_LINKABLE [[attached] XM_ELEMENT_NODE] */
typedef struct S1750 T1750;

/* detachable ET_CONSTRAINT_GENERIC_NAMED_TYPE */
typedef struct S1755 T1755;

/* detachable ET_UNQUALIFIED_CALL_INSTRUCTION */
typedef struct S1756 T1756;

/* detachable ET_CLUSTER_DEPENDENCE_CONSTRAINT */
typedef struct S1758 T1758;

/* [detachable] ET_SEEDED_QUERY_COMPARATOR */
typedef struct S1759 T1759;

/* [detachable] ET_SEEDED_PROCEDURE_COMPARATOR */
typedef struct S1762 T1762;

/* [detachable] TUPLE [[attached] ET_CLIENT_LIST] */
typedef struct S1766 T1766;

/* [detachable] ET_FORMAL_COMMA_ARGUMENT */
typedef struct S1767 T1767;

/* [detachable] ET_LOCAL_COMMA_VARIABLE */
typedef struct S1768 T1768;

/* [detachable] ET_LABELED_COMMA_ACTUAL_PARAMETER */
typedef struct S1769 T1769;

/* [detachable] ET_KEYWORD_EXPRESSION */
typedef struct S1771 T1771;

/* [detachable] SPECIAL [[attached] ET_RENAME_ITEM] */
typedef struct S1776 T1776;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RENAME_ITEM] */
typedef struct S1777 T1777;

/* [detachable] SPECIAL [[attached] ET_EXPORT] */
typedef struct S1779 T1779;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPORT] */
typedef struct S1780 T1780;

/* [detachable] SPECIAL [[attached] ET_FEATURE_NAME_ITEM] */
typedef struct S1782 T1782;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME_ITEM] */
typedef struct S1783 T1783;

/* [detachable] SPECIAL [[attached] ET_FORMAL_PARAMETER_ITEM] */
typedef struct S1784 T1784;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER_ITEM] */
typedef struct S1785 T1785;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S1788 T1788;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG]] */
typedef struct S1789 T1789;

/* [detachable] TUPLE [[attached] ET_ECF_TARGET, [attached] ET_ECF_STATE] */
typedef struct S1790 T1790;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S1791 T1791;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE]] */
typedef struct S1792 T1792;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S1793 T1793;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY]] */
typedef struct S1794 T1794;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S1795 T1795;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG]] */
typedef struct S1796 T1796;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S1797 T1797;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE]] */
typedef struct S1798 T1798;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S1799 T1799;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT]] */
typedef struct S1800 T1800;

/* [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S1801 T1801;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE]] */
typedef struct S1802 T1802;

/* detachable DS_LINKABLE [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
typedef struct S1806 T1806;

/* [detachable] SPECIAL [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S1809 T1809;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S1810 T1810;

/* [detachable] ET_AGENT_IMPLICIT_CURRENT_TARGET */
typedef struct S1811 T1811;

/* [detachable] SPECIAL [[attached] ET_CHOICE_ITEM] */
typedef struct S1813 T1813;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CHOICE_ITEM] */
typedef struct S1814 T1814;

/* [detachable] SPECIAL [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S1815 T1815;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S1817 T1817;

/* [detachable] SPECIAL [[attached] ET_CONVERT_FEATURE_ITEM] */
typedef struct S1819 T1819;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONVERT_FEATURE_ITEM] */
typedef struct S1820 T1820;

/* [detachable] SPECIAL [[attached] ET_CREATOR] */
typedef struct S1821 T1821;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CREATOR] */
typedef struct S1822 T1822;

/* [detachable] SPECIAL [[attached] ET_ELSEIF_EXPRESSION] */
typedef struct S1823 T1823;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_EXPRESSION] */
typedef struct S1824 T1824;

/* [detachable] SPECIAL [[attached] ET_ELSEIF_PART] */
typedef struct S1825 T1825;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_PART] */
typedef struct S1826 T1826;

/* [detachable] SPECIAL [[attached] ET_FEATURE_CLAUSE] */
typedef struct S1827 T1827;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_CLAUSE] */
typedef struct S1828 T1828;

/* [detachable] SPECIAL [[attached] ET_INDEXING_ITEM] */
typedef struct S1830 T1830;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_ITEM] */
typedef struct S1831 T1831;

/* [detachable] SPECIAL [[attached] ET_INDEXING_TERM_ITEM] */
typedef struct S1832 T1832;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM_ITEM] */
typedef struct S1833 T1833;

/* [detachable] SPECIAL [[attached] ET_LOCAL_VARIABLE_ITEM] */
typedef struct S1834 T1834;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LOCAL_VARIABLE_ITEM] */
typedef struct S1836 T1836;

/* [detachable] SPECIAL [[attached] ET_MANIFEST_STRING_ITEM] */
typedef struct S1838 T1838;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_MANIFEST_STRING_ITEM] */
typedef struct S1839 T1839;

/* [detachable] SPECIAL [[attached] ET_PARENT_LIST] */
typedef struct S1840 T1840;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_LIST] */
typedef struct S1841 T1841;

/* [detachable] SPECIAL [[attached] ET_TYPE_CONSTRAINT_ITEM] */
typedef struct S1843 T1843;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_CONSTRAINT_ITEM] */
typedef struct S1844 T1844;

/* [detachable] SPECIAL [[attached] ET_WHEN_EXPRESSION] */
typedef struct S1845 T1845;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_EXPRESSION] */
typedef struct S1846 T1846;

/* [detachable] SPECIAL [[attached] ET_WHEN_PART] */
typedef struct S1847 T1847;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_PART] */
typedef struct S1848 T1848;

/* [detachable] ET_BASE_TYPE_RENAME_CONSTRAINT */
typedef struct S1850 T1850;

/* detachable DS_SPARSE_TABLE_KEYS [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S1853 T1853;

/* [detachable] ET_STANDARD_ONCE_KEYS */
typedef struct S1857 T1857;

/* [detachable] SPECIAL [[attached] ET_ECF_CLUSTER] */
typedef struct S1859 T1859;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CLUSTER] */
typedef struct S1860 T1860;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CLUSTER] */
typedef struct S1861 T1861;

/* [detachable] SPECIAL [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S1862 T1862;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S1863 T1863;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S1864 T1864;

/* [detachable] SPECIAL [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1865 T1865;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1866 T1866;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1867 T1867;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_FILE_RULE] */
typedef struct S1868 T1868;

/* [detachable] SPECIAL [[attached] ET_ECF_FILE_RULE] */
typedef struct S1869 T1869;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_FILE_RULE] */
typedef struct S1870 T1870;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S1871 T1871;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S1872 T1872;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S1873 T1873;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S1874 T1874;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S1875 T1875;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S1876 T1876;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S1877 T1877;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S1878 T1878;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S1879 T1879;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S1880 T1880;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S1881 T1881;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S1882 T1882;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S1883 T1883;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S1884 T1884;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S1885 T1885;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S1886 T1886;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S1887 T1887;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S1888 T1888;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S1889 T1889;

/* [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S1890 T1890;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S1891 T1891;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S1892 T1892;

/* [detachable] SPECIAL [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S1893 T1893;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S1894 T1894;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S1895 T1895;

/* [detachable] SPECIAL [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S1896 T1896;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S1897 T1897;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S1898 T1898;

/* [detachable] SPECIAL [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S1899 T1899;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S1900 T1900;

/* detachable DS_LINKABLE [[attached] CHARACTER_8] */
typedef struct S1902 T1902;

/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_CLASS] */
typedef struct S1903 T1903;

/* [detachable] SPECIAL [[attached] ET_BASE_TYPE_CONSTRAINT] */
typedef struct S1904 T1904;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE_CONSTRAINT] */
typedef struct S1905 T1905;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_BASE_TYPE_CONSTRAINT] */
typedef struct S1906 T1906;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] NATURAL_32] */
typedef struct S1907 T1907;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_FEATURE_NAME] */
typedef struct S1908 T1908;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] ET_FEATURE_NAME] */
typedef struct S1909 T1909;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
typedef struct S1911 T1911;

/* [detachable] SPECIAL [[attached] ET_REPLICABLE_FEATURE] */
typedef struct S1912 T1912;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_REPLICABLE_FEATURE] */
typedef struct S1914 T1914;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_ADAPTED_FEATURE] */
typedef struct S1915 T1915;

/* detachable DS_LINKABLE [[attached] ET_ADAPTED_FEATURE] */
typedef struct S1916 T1916;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_FEATURE] */
typedef struct S1917 T1917;

/* [detachable] SPECIAL [[attached] ET_PARENT_FEATURE] */
typedef struct S1918 T1918;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
typedef struct S1919 T1919;

/* [detachable] SPECIAL [[attached] ET_DOTNET_FEATURE] */
typedef struct S1920 T1920;

/* [detachable] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_FEATURE] */
typedef struct S1921 T1921;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_FEATURE] */
typedef struct S1922 T1922;

/* [detachable] SPECIAL [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
typedef struct S1923 T1923;

/* detachable DS_HASH_TABLE_CURSOR [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
typedef struct S1926 T1926;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
typedef struct S1927 T1927;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_DOTNET_FEATURE] */
typedef struct S1928 T1928;

/* detachable DS_LINKABLE [[attached] ET_DOTNET_FEATURE] */
typedef struct S1929 T1929;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
typedef struct S1930 T1930;

/* [detachable] SPECIAL [[attached] ET_LIKE_FEATURE] */
typedef struct S1931 T1931;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LIKE_FEATURE] */
typedef struct S1932 T1932;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_LIKE_FEATURE] */
typedef struct S1933 T1933;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
typedef struct S1935 T1935;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
typedef struct S1936 T1936;

/* [detachable] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
typedef struct S1937 T1937;

/* [detachable] SPECIAL [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
typedef struct S1938 T1938;

/* detachable DS_BILINKABLE [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
typedef struct S1939 T1939;

/* [detachable] ET_CLUSTER_ERROR */
typedef struct S1940 T1940;

/* [detachable] ET_SYNTAX_ERROR */
typedef struct S1941 T1941;

/* [detachable] ET_DOTNET_ASSEMBLY_ERROR */
typedef struct S1942 T1942;

/* [detachable] AP_OPTION_COMPARATOR */
typedef struct S1943 T1943;

/* [detachable] DS_QUICK_SORTER [[attached] AP_OPTION] */
typedef struct S1944 T1944;

/* [detachable] ST_SPLITTER */
typedef struct S1953 T1953;

/* [detachable] KL_CHARACTER_ROUTINES */
typedef struct S1954 T1954;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_CURRENT, [attached] ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
typedef struct S1955 T1955;

/* [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
typedef struct S1956 T1956;

/* [detachable] ET_ACTUAL_PARAMETER_SUBLIST */
typedef struct S1957 T1957;

/* [detachable] TYPE [[detachable] DEVELOPER_EXCEPTION] */
#define T1958 EIF_TYPE_OBJ

/* [detachable] SPECIAL [[attached] ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S1960 T1960;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S1962 T1962;

/* [detachable] SPECIAL [[attached] ET_DYNAMIC_PRECURSOR] */
typedef struct S1964 T1964;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRECURSOR] */
typedef struct S1965 T1965;

/* [detachable] SPECIAL [[attached] ET_INSTRUCTION] */
typedef struct S1966 T1966;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INSTRUCTION] */
typedef struct S1968 T1968;

/* [detachable] ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
typedef struct S1977 T1977;

/* [detachable] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8] */
typedef struct S1978 T1978;

/* [detachable] ET_INFIX_CAST_EXPRESSION */
typedef struct S1980 T1980;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] STRING_8] */
typedef struct S1984 T1984;

/* [detachable] SPECIAL [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
typedef struct S1985 T1985;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
typedef struct S1988 T1988;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
typedef struct S1989 T1989;

/* [detachable] DS_SPARSE_TABLE_KEYS_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S1990 T1990;

/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
typedef struct S1993 T1993;

/* [detachable] SPECIAL [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
typedef struct S1994 T1994;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
typedef struct S1997 T1997;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
typedef struct S1998 T1998;

/* [detachable] ET_UNIVERSE_ERROR */
typedef struct S1999 T1999;

/* [detachable] TUPLE [[attached] ET_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2000 T2000;

/* detachable ET_PARENTHESIS_INSTRUCTION */
typedef struct S2002 T2002;

/* [detachable] TYPE [detachable RX_REGULAR_EXPRESSION] */
#define T2006 EIF_TYPE_OBJ

/* [detachable] ET_PARENTHESIS_SYMBOL */
typedef struct S2020 T2020;

/* [detachable] ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
typedef struct S2026 T2026;

/* [detachable] ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
typedef struct S2027 T2027;

/* [detachable] TYPE [[attached] ET_NESTED_TYPE_CONTEXT] */
#define T2030 EIF_TYPE_OBJ

/* [detachable] TYPE [[attached] ET_ITERATION_COMPONENT] */
#define T2031 EIF_TYPE_OBJ

/* [detachable] DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S2034 T2034;

/* [detachable] DS_HASH_TABLE [[attached] LX_WILDCARD, [attached] STRING_8] */
typedef struct S2040 T2040;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_WILDCARD, [attached] STRING_8] */
typedef struct S2041 T2041;

/* [detachable] ET_DYNAMIC_EQUALITY_EXPRESSION */
typedef struct S2042 T2042;

/* [detachable] ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
typedef struct S2043 T2043;

/* [detachable] TUPLE [[attached] ET_ECF_LIBRARY] */
typedef struct S2044 T2044;

/* [detachable] ET_CONVERT_BUILTIN_EXPRESSION */
typedef struct S2046 T2046;

/* [detachable] ET_CONVERT_FROM_EXPRESSION */
typedef struct S2047 T2047;

/* [detachable] ET_CONVERT_TO_EXPRESSION */
typedef struct S2048 T2048;

/* detachable DS_CELL [detachable ET_CLASS] */
typedef struct S2051 T2051;

/* [detachable] TUPLE [[attached] DS_CELL [detachable ET_CLASS]] */
typedef struct S2052 T2052;

/* [detachable] TUPLE [[attached] ET_EIFFEL_PREPARSER, [attached] DS_ARRAYED_LIST [[attached] STRING_8]] */
typedef struct S2054 T2054;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_FEATURE] */
typedef struct S2056 T2056;

/* [detachable] TUPLE [[attached] ET_ECF_FILE_RULE] */
typedef struct S2057 T2057;

/* [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
typedef struct S2058 T2058;

/* [detachable] TUPLE [[attached] ET_ECF_FILE_RULES] */
typedef struct S2059 T2059;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
typedef struct S2060 T2060;

/* [detachable] TUPLE [[attached] ET_ECF_STATE] */
typedef struct S2061 T2061;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] STRING_8]] */
typedef struct S2063 T2063;

/* [detachable] TUPLE [[attached] KL_STRING_ROUTINES, [attached] STRING_8] */
typedef struct S2064 T2064;

/* [detachable] TUPLE [[attached] DS_HASH_SET [[attached] STRING_8]] */
typedef struct S2066 T2066;

/* [detachable] LX_DFA_WILDCARD */
typedef struct S2068 T2068;

/* detachable DS_LINKABLE [[attached] ET_PARENT_FEATURE] */
typedef struct S2071 T2071;

/* [detachable] TUPLE [[attached] LX_WILDCARD] */
typedef struct S2073 T2073;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] LX_WILDCARD]] */
typedef struct S2074 T2074;

/* [detachable] TUPLE [[attached] ET_CLUSTER_DEPENDENCE_CONSTRAINT, [attached] ET_GROUP, [attached] STRING_8] */
typedef struct S2075 T2075;

/* [detachable] SPECIAL [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
typedef struct S2077 T2077;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
typedef struct S2078 T2078;

/* [detachable] SPECIAL [[attached] LX_WILDCARD] */
typedef struct S2079 T2079;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_WILDCARD] */
typedef struct S2082 T2082;

/* [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
typedef struct S2086 T2086;

/* [detachable] TUPLE [[attached] ET_ECF_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2089 T2089;

/* [detachable] TUPLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2090 T2090;

/* [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2091 T2091;

/* [detachable] TUPLE [[attached] ET_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2092 T2092;

/* [detachable] ET_AST_NULL_LEAF */
typedef struct S2094 T2094;

/* [detachable] LX_WILDCARD_PARSER */
typedef struct S2095 T2095;

/* [detachable] LX_DESCRIPTION */
typedef struct S2096 T2096;

/* [detachable] LX_FULL_DFA */
typedef struct S2097 T2097;

/* [detachable] TUPLE [[attached] ET_GROUP] */
typedef struct S2099 T2099;

/* [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]]] */
typedef struct S2100 T2100;

/* [detachable] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S2101 T2101;

/* [detachable] PREDICATE [[attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION]] */
typedef struct S2102 T2102;

/* [detachable] DS_ARRAYED_LIST [[attached] LX_RULE] */
typedef struct S2105 T2105;

/* [detachable] LX_START_CONDITIONS */
typedef struct S2106 T2106;

/* [detachable] LX_ACTION_FACTORY */
typedef struct S2107 T2107;

/* [detachable] DS_HASH_TABLE [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]], [attached] LX_SYMBOL_CLASS] */
typedef struct S2108 T2108;

/* [detachable] LX_SYMBOL_CLASS */
typedef struct S2109 T2109;

/* [detachable] DS_HASH_TABLE [[attached] STRING_32, [attached] STRING_8] */
typedef struct S2110 T2110;

/* [detachable] DS_HASH_SET [[attached] LX_SYMBOL_CLASS] */
typedef struct S2111 T2111;

/* [detachable] DS_HASH_TABLE [[attached] LX_SYMBOL_CLASS, [attached] STRING_32] */
typedef struct S2112 T2112;

/* [detachable] KL_EQUALITY_TESTER [[attached] STRING_32] */
typedef struct S2113 T2113;

/* [detachable] DS_ARRAYED_STACK [[attached] BOOLEAN] */
typedef struct S2114 T2114;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_32] */
typedef struct S2115 T2115;

/* [detachable] SPECIAL [[attached] STRING_32] */
typedef struct S2116 T2116;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_SYMBOL_CLASS] */
typedef struct S2117 T2117;

/* [detachable] SPECIAL [[attached] LX_SYMBOL_CLASS] */
typedef struct S2118 T2118;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_NFA] */
typedef struct S2119 T2119;

/* [detachable] LX_NFA */
typedef struct S2120 T2120;

/* [detachable] SPECIAL [[attached] LX_NFA] */
typedef struct S2121 T2121;

/* detachable LX_RULE */
typedef struct S2122 T2122;

/* [detachable] DS_HASH_SET_CURSOR [[attached] LX_SYMBOL_CLASS] */
typedef struct S2123 T2123;

/* [detachable] LX_EQUIVALENCE_CLASSES */
typedef struct S2124 T2124;

/* [detachable] LX_UNRECOGNIZED_RULE_ERROR */
typedef struct S2125 T2125;

/* [detachable] LX_INVALID_UNICODE_CHARACTER_ERROR */
typedef struct S2126 T2126;

/* [detachable] LX_MISSING_QUOTE_ERROR */
typedef struct S2127 T2127;

/* [detachable] LX_BAD_CHARACTER_CLASS_ERROR */
typedef struct S2128 T2128;

/* [detachable] LX_BAD_CHARACTER_ERROR */
typedef struct S2129 T2129;

/* [detachable] LX_FULL_AND_META_ERROR */
typedef struct S2130 T2130;

/* [detachable] LX_FULL_AND_REJECT_ERROR */
typedef struct S2131 T2131;

/* [detachable] LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
typedef struct S2132 T2132;

/* [detachable] LX_CHARACTER_OUT_OF_RANGE_ERROR */
typedef struct S2133 T2133;

/* [detachable] ARRAY [[attached] LX_RULE] */
typedef struct S2134 T2134;

/* [detachable] SPECIAL [[attached] LX_RULE] */
typedef struct S2135 T2135;

/* [detachable] LX_DFA_STATE */
typedef struct S2136 T2136;

/* [detachable] DS_ARRAYED_LIST [[attached] LX_NFA_STATE] */
typedef struct S2137 T2137;

/* [detachable] DS_ARRAYED_LIST [[attached] LX_DFA_STATE] */
typedef struct S2138 T2138;

/* detachable LX_SYMBOL_PARTITIONS */
typedef struct S2139 T2139;

/* [detachable] KL_ARRAY_ROUTINES [[attached] LX_RULE] */
typedef struct S2140 T2140;

/* [detachable] ARRAY [detachable LX_RULE] */
typedef struct S2141 T2141;

/* [detachable] SPECIAL [detachable LX_RULE] */
typedef struct S2142 T2142;

/* [detachable] LX_START_CONDITION */
typedef struct S2143 T2143;

/* [detachable] LX_TRANSITION_TABLE [[attached] LX_DFA_STATE] */
typedef struct S2144 T2144;

/* [detachable] DS_ARRAYED_LIST [[attached] LX_NFA] */
typedef struct S2145 T2145;

/* [detachable] DS_HASH_TABLE [[attached] LX_NFA, [attached] INTEGER_32] */
typedef struct S2146 T2146;

/* [detachable] LX_NFA_STATE */
typedef struct S2147 T2147;

/* detachable ARRAY [detachable LX_SYMBOL_CLASS] */
typedef struct S2148 T2148;

/* [detachable] SPECIAL [detachable LX_SYMBOL_CLASS] */
typedef struct S2149 T2149;

/* detachable ARRAY [detachable DS_HASH_SET [[attached] INTEGER_32]] */
typedef struct S2150 T2150;

/* [detachable] SPECIAL [detachable DS_HASH_SET [[attached] INTEGER_32]] */
typedef struct S2151 T2151;

/* detachable DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
typedef struct S2152 T2152;

/* [detachable] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS] */
typedef struct S2153 T2153;

/* [detachable] LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
typedef struct S2154 T2154;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_RULE] */
typedef struct S2155 T2155;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_RULE] */
typedef struct S2156 T2156;

/* [detachable] KL_ARRAY_ROUTINES [[attached] STRING_8] */
typedef struct S2157 T2157;

/* [detachable] SPECIAL [[attached] LX_START_CONDITION] */
typedef struct S2158 T2158;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_START_CONDITION] */
typedef struct S2159 T2159;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_START_CONDITION] */
typedef struct S2160 T2160;

/* [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]]] */
typedef struct S2161 T2161;

/* detachable KL_EQUALITY_TESTER [[attached] LX_SYMBOL_CLASS] */
typedef struct S2162 T2162;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]]] */
typedef struct S2164 T2164;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]], [attached] LX_SYMBOL_CLASS] */
typedef struct S2165 T2165;

/* detachable SPECIAL [detachable SPECIAL [[attached] NATURAL_64]] */
typedef struct S2166 T2166;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] STRING_32, [attached] STRING_8] */
typedef struct S2168 T2168;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_SYMBOL_CLASS, [attached] STRING_32] */
typedef struct S2170 T2170;

/* [detachable] LX_EPSILON_TRANSITION [[attached] LX_NFA_STATE] */
typedef struct S2171 T2171;

/* [detachable] LX_SYMBOL_CLASS_TRANSITION [[attached] LX_NFA_STATE] */
typedef struct S2174 T2174;

/* [detachable] LX_SYMBOL_TRANSITION [[attached] LX_NFA_STATE] */
typedef struct S2175 T2175;

/* [detachable] LX_ACTION */
typedef struct S2177 T2177;

/* [detachable] ARRAY [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
typedef struct S2178 T2178;

/* [detachable] DS_BILINKABLE [[attached] INTEGER_32] */
typedef struct S2179 T2179;

/* [detachable] SPECIAL [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
typedef struct S2180 T2180;

/* [detachable] DS_BUBBLE_SORTER [[attached] LX_NFA_STATE] */
typedef struct S2181 T2181;

/* [detachable] DS_BUBBLE_SORTER [[attached] LX_RULE] */
typedef struct S2183 T2183;

/* [detachable] SPECIAL [[attached] LX_NFA_STATE] */
typedef struct S2185 T2185;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_NFA_STATE] */
typedef struct S2187 T2187;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA_STATE] */
typedef struct S2188 T2188;

/* [detachable] SPECIAL [[attached] LX_DFA_STATE] */
typedef struct S2190 T2190;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] LX_DFA_STATE] */
typedef struct S2191 T2191;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_DFA_STATE] */
typedef struct S2192 T2192;

/* [detachable] KL_ARRAY [[attached] LX_RULE] */
typedef struct S2193 T2193;

/* [detachable] DS_HASH_TABLE [[attached] LX_DFA_STATE, [attached] INTEGER_32] */
typedef struct S2194 T2194;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA] */
typedef struct S2195 T2195;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_NFA, [attached] INTEGER_32] */
typedef struct S2197 T2197;

/* [detachable] KL_COMPARABLE_COMPARATOR [[attached] LX_RULE] */
typedef struct S2201 T2201;

/* [detachable] KL_COMPARABLE_COMPARATOR [[attached] LX_NFA_STATE] */
typedef struct S2204 T2204;

/* [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
typedef struct S2207 T2207;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
typedef struct S2208 T2208;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
typedef struct S2209 T2209;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_SYMBOL_CLASS] */
typedef struct S2210 T2210;

/* [detachable] KL_ARRAY [[attached] STRING_8] */
typedef struct S2211 T2211;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_DFA_STATE, [attached] INTEGER_32] */
typedef struct S2213 T2213;

/* Struct for boxed version of type BOOLEAN */
struct Sb1 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1* p1; /* pointer */
	T1 z1; /* item */
};

/* Struct for boxed version of type CHARACTER_8 */
struct Sb2 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2* p1; /* pointer */
	T2 z1; /* item */
};

/* Struct for boxed version of type CHARACTER_32 */
struct Sb3 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T3* p1; /* pointer */
	T3 z1; /* item */
};

/* Struct for boxed version of type INTEGER_8 */
struct Sb4 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T4* p1; /* pointer */
	T4 z1; /* item */
};

/* Struct for boxed version of type INTEGER_16 */
struct Sb5 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T5* p1; /* pointer */
	T5 z1; /* item */
};

/* Struct for boxed version of type INTEGER_32 */
struct Sb6 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6* p1; /* pointer */
	T6 z1; /* item */
};

/* Struct for boxed version of type INTEGER_64 */
struct Sb7 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T7* p1; /* pointer */
	T7 z1; /* item */
};

/* Struct for boxed version of type NATURAL_8 */
struct Sb8 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T8* p1; /* pointer */
	T8 z1; /* item */
};

/* Struct for boxed version of type NATURAL_16 */
struct Sb9 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T9* p1; /* pointer */
	T9 z1; /* item */
};

/* Struct for boxed version of type NATURAL_32 */
struct Sb10 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T10* p1; /* pointer */
	T10 z1; /* item */
};

/* Struct for boxed version of type NATURAL_64 */
struct Sb11 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T11* p1; /* pointer */
	T11 z1; /* item */
};

/* Struct for boxed version of type REAL_32 */
struct Sb12 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T12* p1; /* pointer */
	T12 z1; /* item */
};

/* Struct for boxed version of type REAL_64 */
struct Sb13 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T13* p1; /* pointer */
	T13 z1; /* item */
};

/* Struct for boxed version of type POINTER */
struct Sb14 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14* p1; /* pointer */
	T14 z1; /* item */
};

/* Struct for type UTF_CONVERTER */
struct S478 {
	char dummy;
};

/* Struct for boxed version of type UTF_CONVERTER */
struct Sb478 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T478* p1; /* pointer */
	T478 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] NATURAL_16] */
struct S1468 {
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] NATURAL_16] */
struct Sb1468 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1468* p1; /* pointer */
	T1468 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] NATURAL_8] */
struct S893 {
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] NATURAL_8] */
struct Sb893 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T893* p1; /* pointer */
	T893 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] ANY] */
struct S697 {
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] ANY] */
struct Sb697 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T697* p1; /* pointer */
	T697 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
struct S488 {
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
struct Sb488 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T488* p1; /* pointer */
	T488 z1; /* item */
};

/* Struct for type [detachable] SPECIAL [CHARACTER_8] */
struct S15 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T2 z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [CHARACTER_32] */
struct S16 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T3 z2[1]; /* item */
};

/* Struct for type [detachable] STRING_8 */
struct S17 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type [detachable] STRING_32 */
struct S18 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type [detachable] IMMUTABLE_STRING_32 */
struct S20 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* area_lower */
};

/* Struct for type [detachable] ISE_EXCEPTION_MANAGER */
struct S21 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] GEC */
struct S25 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* exit_code */
	T0* a2; /* error_handler */
	T0* a3; /* version_flag */
	T0* a4; /* ecf_filename */
	T0* a5; /* last_system */
	T0* a6; /* target_option */
	T0* a7; /* setting_option */
	T0* a8; /* capability_option */
	T0* a9; /* variable_option */
	T0* a10; /* finalize_flag */
	T0* a11; /* gelint_flag */
	T0* a12; /* ise_option */
	T0* a13; /* catcall_option */
	T0* a14; /* c_compile_option */
	T0* a15; /* split_option */
	T0* a16; /* split_size_option */
	T0* a17; /* new_instance_types_option */
	T0* a18; /* gc_option */
	T0* a19; /* thread_option */
	T0* a20; /* silent_flag */
	T0* a21; /* no_benchmark_flag */
	T0* a22; /* nested_benchmark_flag */
	T0* a23; /* metrics_flag */
	T0* a24; /* verbose_flag */
	T0* a25; /* ise_version */
	T0* a26; /* override_settings */
	T0* a27; /* override_capabilities */
	T0* a28; /* override_variables */
	T1 a29; /* ecf_gelint_option */
	T0* a30; /* new_instance_types */
	T6 a31; /* split_size */
};

/* Struct for type [detachable] CELL [detachable EXCEPTION] */
struct S26 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
struct S27 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T0* a2; /* content */
	T0* a3; /* keys */
	T0* a4; /* deleted_marks */
	T0* a5; /* indexes_map */
	T6 a6; /* iteration_position */
	T6 a7; /* count */
	T6 a8; /* deleted_item_position */
	T6 a9; /* control */
	T6 a10; /* found_item */
	T1 a11; /* has_default */
	T6 a12; /* item_position */
	T6 a13; /* ht_lowest_deleted_position */
	T6 a14; /* ht_deleted_item */
	T6 a15; /* ht_deleted_key */
	T1 a16; /* object_comparison */
};

/* Struct for type [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
struct S28 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] CELL [[attached] NO_MORE_MEMORY] */
struct S29 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type detachable C_STRING */
struct S30 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* managed_data */
	T6 a2; /* count */
};

/* Struct for type [detachable] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
struct S31 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1;
	T6 z2;
	T6 z3;
	T0* z4;
	T0* z5;
	T0* z6;
	T0* z7;
	T0* z8;
	T0* z9;
	T6 z10;
	T1 z11;
};

/* Struct for type [detachable] KL_ARGUMENTS */
struct S32 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* program_name */
};

/* Struct for type [detachable] ARRAY [[attached] STRING_8] */
struct S33 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] STRING_8] */
struct S34 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_EXCEPTIONS */
struct S35 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_ERROR_HANDLER */
struct S36 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* info_file */
	T0* a3; /* mutex */
	T0* a4; /* error_file */
	T0* a5; /* warning_file */
	T1 a6; /* is_ise */
	T1 a7; /* is_verbose */
	T1 a8; /* has_eiffel_error */
	T1 a9; /* has_internal_error */
	T1 a10; /* is_ge */
};

/* Struct for type [detachable] KL_TEXT_INPUT_FILE */
struct S37 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_file */
	T0* a2; /* last_string */
	T6 a3; /* mode */
	T0* a4; /* name */
	T0* a5; /* character_buffer */
	T2 a6; /* last_character */
	T14 a7; /* file_pointer */
	T1 a8; /* descriptor_available */
	T0* a9; /* last_string_32 */
	T0* a10; /* internal_name */
	T0* a11; /* internal_detachable_name_pointer */
};

/* Struct for type [detachable] ET_ISE_VARIABLES */
struct S38 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] AP_FLAG */
struct S39 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* occurrences */
	T0* a2; /* description */
	T0* a3; /* long_form */
	T2 a4; /* short_form */
	T1 a5; /* has_short_form */
	T1 a6; /* is_hidden */
	T1 a7; /* is_mandatory */
	T6 a8; /* maximum_occurrences */
};

/* Struct for type detachable ET_SYSTEM */
struct S40 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implicit_attachment_type_mark */
	T0* a2; /* boolean_type */
	T0* a3; /* character_type */
	T0* a4; /* character_32_type */
	T0* a5; /* integer_type */
	T0* a6; /* current_system */
	T0* a7; /* tuple_type */
	T0* a8; /* pointer_type */
	T0* a9; /* system_object_parents */
	T0* a10; /* any_parents */
	T0* a11; /* system_object_type */
	T0* a12; /* any_type */
	T0* a13; /* register_class_mutex */
	T0* a14; /* external_include_pathnames */
	T0* a15; /* external_object_pathnames */
	T0* a16; /* external_library_pathnames */
	T0* a17; /* external_make_pathnames */
	T0* a18; /* external_resource_pathnames */
	T0* a19; /* external_cflags */
	T0* a20; /* external_linker_flags */
	T1 a21; /* console_application_mode */
	T1 a22; /* total_order_on_reals_mode */
	T1 a23; /* attachment_type_conformance_mode */
	T1 a24; /* scoop_mode */
	T6 a25; /* default_create_seed */
	T0* a26; /* universe */
	T0* a27; /* name */
	T0* a28; /* libraries */
	T0* a29; /* clusters */
	T0* a30; /* dotnet_assemblies */
	T0* a31; /* master_classes */
	T0* a32; /* master_class_mutex */
	T0* a33; /* system_name */
	T0* a34; /* detachable_any_type */
	T0* a35; /* detachable_separate_any_type */
	T0* a36; /* any_parent */
	T0* a37; /* any_clients */
	T0* a38; /* detachable_tuple_type */
	T0* a39; /* tuple_identity_type */
	T0* a40; /* unfolded_empty_tuple_actual_parameters */
	T0* a41; /* array_any_type */
	T0* a42; /* array_detachable_any_type */
	T0* a43; /* array_none_type */
	T0* a44; /* array_identity_type */
	T0* a45; /* character_8_type */
	T0* a46; /* character_8_convert_feature */
	T0* a47; /* character_32_convert_feature */
	T0* a48; /* double_type */
	T0* a49; /* exception_type */
	T0* a50; /* detachable_exception_type */
	T0* a51; /* exception_manager_type */
	T0* a52; /* function_identity_any_type */
	T0* a53; /* immutable_string_8_type */
	T0* a54; /* immutable_string_32_type */
	T0* a55; /* integer_8_type */
	T0* a56; /* integer_8_convert_feature */
	T0* a57; /* integer_16_type */
	T0* a58; /* integer_16_convert_feature */
	T0* a59; /* integer_32_type */
	T0* a60; /* integer_32_convert_feature */
	T0* a61; /* integer_64_type */
	T0* a62; /* integer_64_convert_feature */
	T0* a63; /* ise_exception_manager_type */
	T0* a64; /* iterable_detachable_separate_any_type */
	T0* a65; /* natural_type */
	T0* a66; /* natural_8_type */
	T0* a67; /* natural_8_convert_feature */
	T0* a68; /* natural_16_type */
	T0* a69; /* natural_16_convert_feature */
	T0* a70; /* natural_32_type */
	T0* a71; /* natural_32_convert_feature */
	T0* a72; /* natural_64_type */
	T0* a73; /* natural_64_convert_feature */
	T0* a74; /* none_type */
	T0* a75; /* detachable_none_type */
	T0* a76; /* predicate_identity_type */
	T0* a77; /* procedure_identity_type */
	T0* a78; /* real_type */
	T0* a79; /* real_32_type */
	T0* a80; /* real_32_convert_feature */
	T0* a81; /* real_64_type */
	T0* a82; /* real_64_convert_feature */
	T0* a83; /* routine_identity_type */
	T0* a84; /* special_any_type */
	T0* a85; /* special_detachable_any_type */
	T0* a86; /* special_identity_type */
	T0* a87; /* string_type */
	T0* a88; /* detachable_string_type */
	T0* a89; /* string_8_type */
	T0* a90; /* detachable_string_8_type */
	T0* a91; /* string_8_convert_feature */
	T0* a92; /* string_32_type */
	T0* a93; /* string_32_convert_feature */
	T0* a94; /* system_string_type */
	T0* a95; /* type_detachable_any_type */
	T0* a96; /* detachable_type_exception_type */
	T0* a97; /* type_detachable_exception_type */
	T0* a98; /* type_detachable_like_current_type */
	T0* a99; /* type_identity_type */
	T0* a100; /* typed_pointer_identity_type */
	T0* a101; /* wide_character_type */
	T1 a102; /* is_read_only */
	T6 a103; /* registered_class_count */
	T1 a104; /* use_obsolete_syntax_mode */
	T1 a105; /* obsolete_iteration_mode */
	T6 a106; /* copy_seed */
	T6 a107; /* twin_seed */
	T6 a108; /* is_equal_seed */
	T6 a109; /* routine_call_seed */
	T6 a110; /* function_item_seed */
	T6 a111; /* dispose_seed */
	T6 a112; /* iterable_new_cursor_seed */
	T6 a113; /* iteration_cursor_item_seed */
	T6 a114; /* iteration_cursor_after_seed */
	T6 a115; /* iteration_cursor_forth_seed */
	T6 a116; /* special_item_seed */
	T1 a117; /* target_type_attachment_mode */
	T1 a118; /* obsolete_routine_type_mode */
	T1 a119; /* is_dotnet */
};

/* Struct for type [detachable] AP_PARSER */
struct S42 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* options */
	T0* a2; /* alternative_options_lists */
	T0* a3; /* parameters */
	T0* a4; /* parameters_description */
	T0* a5; /* help_option */
	T0* a6; /* error_handler */
	T0* a7; /* application_description */
	T0* a8; /* current_options */
	T1 a9; /* is_first_option */
	T0* a10; /* last_option_parameter */
};

/* Struct for type [detachable] AP_ALTERNATIVE_OPTIONS_LIST */
struct S43 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* introduction_option */
	T0* a2; /* parameters_description */
	T0* a3; /* internal_cursor */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] AP_STRING_OPTION */
struct S44 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* parameter_description */
	T1 a5; /* has_default_parameter */
	T1 a6; /* needs_parameter */
	T0* a7; /* long_form */
	T1 a8; /* is_hidden */
	T1 a9; /* is_mandatory */
	T6 a10; /* maximum_occurrences */
	T1 a11; /* has_short_form */
	T2 a12; /* short_form */
};

/* Struct for type [detachable] UT_VERSION */
struct S46 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_major */
	T6 a2; /* internal_minor */
	T6 a3; /* internal_revision */
	T6 a4; /* internal_build */
};

/* Struct for type [detachable] AP_ENUMERATION_OPTION */
struct S47 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* possible_values */
	T0* a5; /* parameter_description */
	T1 a6; /* needs_parameter */
	T0* a7; /* long_form */
	T1 a8; /* is_hidden */
	T1 a9; /* is_mandatory */
	T6 a10; /* maximum_occurrences */
	T1 a11; /* has_short_form */
	T2 a12; /* short_form */
};

/* Struct for type [detachable] AP_BOOLEAN_OPTION */
struct S48 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T1 a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* parameter_description */
	T1 a5; /* needs_parameter */
	T0* a6; /* long_form */
	T1 a7; /* is_hidden */
	T1 a8; /* is_mandatory */
	T6 a9; /* maximum_occurrences */
	T1 a10; /* has_short_form */
	T2 a11; /* short_form */
};

/* Struct for type [detachable] AP_INTEGER_OPTION */
struct S49 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T6 a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* parameter_description */
	T1 a5; /* needs_parameter */
	T0* a6; /* long_form */
	T2 a7; /* short_form */
	T1 a8; /* is_hidden */
	T1 a9; /* is_mandatory */
	T6 a10; /* maximum_occurrences */
	T1 a11; /* has_short_form */
};

/* Struct for type [detachable] ET_NULL_ERROR_HANDLER */
struct S53 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* info_file */
	T0* a3; /* mutex */
	T0* a4; /* error_file */
	T0* a5; /* warning_file */
	T1 a6; /* is_ise */
	T1 a7; /* is_verbose */
	T1 a8; /* has_eiffel_error */
	T1 a9; /* has_internal_error */
	T1 a10; /* is_ge */
};

/* Struct for type [detachable] ET_ECF_SYSTEM_PARSER */
struct S55 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_system */
	T0* a2; /* ast_factory */
	T0* a3; /* override_settings */
	T0* a4; /* override_capabilities */
	T0* a5; /* override_variables */
	T0* a6; /* parsed_libraries */
	T0* a7; /* parsed_dotnet_assemblies */
	T0* a8; /* redirected_locations */
	T0* a9; /* xml_parser */
	T0* a10; /* tree_pipe */
	T0* a11; /* error_handler */
	T0* a12; /* ise_version */
};

/* Struct for type [detachable] ET_ECF_ERROR_HANDLER */
struct S56 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* error_count */
	T0* a2; /* error_file */
	T0* a3; /* warning_file */
	T0* a4; /* info_file */
};

/* Struct for type detachable ET_ECF_SETTINGS */
struct S57 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_settings */
	T0* a2; /* secondary_settings */
};

/* Struct for type [detachable] ET_ECF_SYSTEM */
struct S58 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* filename */
	T0* a2; /* targets */
	T0* a3; /* library_target */
	T0* a4; /* libraries */
	T0* a5; /* selected_target */
	T0* a6; /* dotnet_assemblies */
	T0* a7; /* master_classes */
	T0* a8; /* root_type */
	T0* a9; /* none_type */
	T0* a10; /* any_type */
	T0* a11; /* root_creation */
	T6 a12; /* default_create_seed */
	T0* a13; /* boolean_type */
	T0* a14; /* character_8_type */
	T0* a15; /* character_32_type */
	T0* a16; /* integer_8_type */
	T0* a17; /* integer_16_type */
	T0* a18; /* integer_32_type */
	T0* a19; /* integer_64_type */
	T0* a20; /* natural_8_type */
	T0* a21; /* natural_16_type */
	T0* a22; /* natural_32_type */
	T0* a23; /* natural_64_type */
	T0* a24; /* real_32_type */
	T0* a25; /* real_64_type */
	T0* a26; /* pointer_type */
	T0* a27; /* special_any_type */
	T0* a28; /* universe */
	T0* a29; /* string_8_type */
	T0* a30; /* string_32_type */
	T0* a31; /* immutable_string_8_type */
	T0* a32; /* immutable_string_32_type */
	T0* a33; /* array_any_type */
	T0* a34; /* special_identity_type */
	T0* a35; /* typed_pointer_identity_type */
	T0* a36; /* routine_identity_type */
	T0* a37; /* detachable_tuple_type */
	T0* a38; /* ise_exception_manager_type */
	T0* a39; /* detachable_none_type */
	T1 a40; /* attachment_type_conformance_mode */
	T1 a41; /* use_boehm_gc */
	T0* a42; /* external_include_pathnames */
	T0* a43; /* external_cflags */
	T0* a44; /* external_linker_flags */
	T0* a45; /* external_library_pathnames */
	T0* a46; /* external_object_pathnames */
	T0* a47; /* character_type */
	T0* a48; /* wide_character_type */
	T0* a49; /* integer_type */
	T0* a50; /* natural_type */
	T0* a51; /* real_type */
	T0* a52; /* double_type */
	T0* a53; /* external_make_pathnames */
	T0* a54; /* external_resource_pathnames */
	T6 a55; /* is_equal_seed */
	T6 a56; /* routine_call_seed */
	T6 a57; /* function_item_seed */
	T6 a58; /* dispose_seed */
	T1 a59; /* trace_mode */
	T1 a60; /* exception_trace_mode */
	T1 a61; /* multithreaded_mode */
	T1 a62; /* total_order_on_reals_mode */
	T6 a63; /* twin_seed */
	T1 a64; /* console_application_mode */
	T0* a65; /* system_name */
	T6 a66; /* copy_seed */
	T0* a67; /* name */
	T0* a68; /* ecf_namespace */
	T0* a69; /* ecf_version */
	T0* a70; /* description */
	T0* a71; /* notes */
	T1 a72; /* is_read_only */
	T0* a73; /* uuid */
	T0* a74; /* register_class_mutex */
	T0* a75; /* current_system */
	T0* a76; /* tuple_type */
	T0* a77; /* detachable_exception_type */
	T0* a78; /* exception_type */
	T0* a79; /* string_type */
	T1 a80; /* is_preparsed */
	T0* a81; /* clusters */
	T0* a82; /* implicit_attachment_type_mark */
	T0* a83; /* master_class_mutex */
	T0* a84; /* detachable_any_type */
	T0* a85; /* detachable_separate_any_type */
	T0* a86; /* any_parent */
	T0* a87; /* any_parents */
	T0* a88; /* any_clients */
	T0* a89; /* tuple_identity_type */
	T0* a90; /* unfolded_empty_tuple_actual_parameters */
	T0* a91; /* array_detachable_any_type */
	T0* a92; /* array_none_type */
	T0* a93; /* array_identity_type */
	T0* a94; /* character_8_convert_feature */
	T0* a95; /* character_32_convert_feature */
	T0* a96; /* exception_manager_type */
	T0* a97; /* function_identity_any_type */
	T0* a98; /* integer_8_convert_feature */
	T0* a99; /* integer_16_convert_feature */
	T0* a100; /* integer_32_convert_feature */
	T0* a101; /* integer_64_convert_feature */
	T0* a102; /* iterable_detachable_separate_any_type */
	T0* a103; /* natural_8_convert_feature */
	T0* a104; /* natural_16_convert_feature */
	T0* a105; /* natural_32_convert_feature */
	T0* a106; /* natural_64_convert_feature */
	T0* a107; /* predicate_identity_type */
	T0* a108; /* procedure_identity_type */
	T0* a109; /* real_32_convert_feature */
	T0* a110; /* real_64_convert_feature */
	T0* a111; /* special_detachable_any_type */
	T0* a112; /* detachable_string_type */
	T0* a113; /* detachable_string_8_type */
	T0* a114; /* string_8_convert_feature */
	T0* a115; /* string_32_convert_feature */
	T0* a116; /* system_object_type */
	T0* a117; /* system_object_parents */
	T0* a118; /* system_string_type */
	T0* a119; /* type_detachable_any_type */
	T0* a120; /* detachable_type_exception_type */
	T0* a121; /* type_detachable_exception_type */
	T0* a122; /* type_detachable_like_current_type */
	T0* a123; /* type_identity_type */
	T0* a124; /* scm_read_mapping_builder */
	T0* a125; /* scm_write_mapping_builder */
	T6 a126; /* registered_class_count */
	T1 a127; /* scoop_mode */
	T1 a128; /* use_obsolete_syntax_mode */
	T1 a129; /* obsolete_iteration_mode */
	T1 a130; /* is_dotnet */
	T1 a131; /* obsolete_routine_type_mode */
	T1 a132; /* line_generation_mode */
	T1 a133; /* target_type_attachment_mode */
	T6 a134; /* iterable_new_cursor_seed */
	T6 a135; /* iteration_cursor_item_seed */
	T6 a136; /* iteration_cursor_after_seed */
	T6 a137; /* iteration_cursor_forth_seed */
	T6 a138; /* special_item_seed */
};

/* Struct for type [detachable] ET_ECF_TARGET */
struct S59 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* settings */
	T0* a2; /* capabilities */
	T0* a3; /* options */
	T0* a4; /* system_config */
	T0* a5; /* root */
	T0* a6; /* variables */
	T0* a7; /* name */
	T0* a8; /* parent */
	T0* a9; /* description */
	T0* a10; /* precompiled_library */
	T0* a11; /* version */
	T1 a12; /* is_abstract */
	T0* a13; /* clusters */
	T0* a14; /* libraries */
	T0* a15; /* dotnet_assemblies */
	T0* a16; /* file_rules */
	T0* a17; /* class_mappings */
	T0* a18; /* external_cflags */
	T0* a19; /* external_includes */
	T0* a20; /* external_libraries */
	T0* a21; /* external_linker_flags */
	T0* a22; /* external_makes */
	T0* a23; /* external_objects */
	T0* a24; /* external_resources */
	T0* a25; /* pre_compile_actions */
	T0* a26; /* post_compile_actions */
	T0* a27; /* notes */
};

/* Struct for type detachable ET_ECF_CAPABILITIES */
struct S61 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_use_capabilities */
	T0* a2; /* primary_support_capabilities */
	T0* a3; /* secondary_capabilities */
};

/* Struct for type detachable ET_ECF_VARIABLES */
struct S62 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* secondary_variables */
	T0* a2; /* primary_variables */
};

/* Struct for type [detachable] ET_DYNAMIC_SYSTEM */
struct S63 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* dynamic_types */
	T1 a3; /* catcall_error_mode */
	T0* a4; /* current_system */
	T0* a5; /* system_processor */
	T0* a6; /* dynamic_generic_types_by_name */
	T0* a7; /* null_dynamic_type_set_builder */
	T1 a8; /* full_class_checking */
	T1 a9; /* catcall_warning_mode */
	T0* a10; /* new_instance_types */
	T0* a11; /* dynamic_type_set_builder */
	T0* a12; /* unknown_type */
	T0* a13; /* boolean_type */
	T0* a14; /* character_8_type */
	T0* a15; /* character_32_type */
	T0* a16; /* immutable_string_8_type */
	T0* a17; /* immutable_string_32_type */
	T0* a18; /* integer_8_type */
	T0* a19; /* integer_16_type */
	T0* a20; /* integer_32_type */
	T0* a21; /* integer_64_type */
	T0* a22; /* natural_8_type */
	T0* a23; /* natural_16_type */
	T0* a24; /* natural_32_type */
	T0* a25; /* natural_64_type */
	T0* a26; /* real_32_type */
	T0* a27; /* real_64_type */
	T0* a28; /* pointer_type */
	T0* a29; /* string_8_type */
	T0* a30; /* string_32_type */
	T0* a31; /* special_character_8_type */
	T0* a32; /* special_character_32_type */
	T0* a33; /* ise_exception_manager_type */
	T0* a34; /* any_type */
	T0* a35; /* none_type */
	T0* a36; /* root_type */
	T0* a37; /* root_creation_procedure */
	T0* a38; /* special_count_feature */
	T0* a39; /* special_capacity_feature */
	T0* a40; /* special_item_feature */
	T0* a41; /* array_area_feature */
	T0* a42; /* array_lower_feature */
	T0* a43; /* array_upper_feature */
	T0* a44; /* typed_pointer_to_pointer_feature */
	T0* a45; /* routine_closed_operands_feature */
	T0* a46; /* routine_rout_disp_feature */
	T0* a47; /* routine_set_rout_disp_final_feature */
	T0* a48; /* ise_exception_manager_init_exception_manager_feature */
	T0* a49; /* ise_exception_manager_last_exception_feature */
	T0* a50; /* ise_exception_manager_once_raise_feature */
	T0* a51; /* ise_exception_manager_set_exception_data_feature */
	T0* a52; /* ise_runtime_type_conforms_to_feature */
	T1 a53; /* in_dynamic_primary_type */
	T0* a54; /* ise_runtime_new_type_instance_of_feature */
	T1 a55; /* in_create_meta_type */
	T1 a56; /* all_attributes_used */
	T0* a57; /* ise_runtime_reference_field_feature */
	T0* a58; /* ise_runtime_set_reference_field_feature */
};

/* Struct for type [detachable] ET_SYSTEM_PROCESSOR */
struct S64 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* interface_checker */
	T0* a2; /* stop_request */
	T0* a3; /* eiffel_parser */
	T0* a4; /* error_handler */
	T0* a5; /* implementation_checker */
	T1 a6; /* benchmark_shown */
	T0* a7; /* ise_version */
	T0* a8; /* eiffel_preparser */
	T0* a9; /* master_class_checker */
	T0* a10; /* provider_checker */
	T0* a11; /* ancestor_builder */
	T0* a12; /* feature_flattener */
	T0* a13; /* processed_class_count_stack */
	T0* a14; /* ast_factory */
	T0* a15; /* dotnet_assembly_consumer */
	T0* a16; /* ecma_version */
	T1 a17; /* nested_benchmark_shown */
	T1 a18; /* metrics_shown */
	T1 a19; /* flat_mode */
	T1 a20; /* flat_dbc_mode */
	T1 a21; /* preparse_shallow_mode */
	T1 a22; /* preparse_single_mode */
	T1 a23; /* preparse_multiple_mode */
	T1 a24; /* unknown_builtin_reported */
	T1 a25; /* qualified_anchored_types_cycle_detection_enabled */
	T1 a26; /* use_attribute_keyword */
	T1 a27; /* use_note_keyword */
	T1 a28; /* use_reference_keyword */
	T1 a29; /* use_attached_keyword */
	T1 a30; /* use_detachable_keyword */
	T1 a31; /* cluster_dependence_enabled */
	T6 a32; /* postponed_class_count */
	T6 a33; /* processed_class_count */
	T1 a34; /* suppliers_enabled */
	T1 a35; /* preparse_override_mode */
	T1 a36; /* preparse_readonly_mode */
	T1 a37; /* providers_enabled */
	T1 a38; /* use_cluster_dependence_pathnames */
};

/* Struct for type [detachable] ET_C_GENERATOR */
struct S67 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* never_void_target_count */
	T6 a2; /* can_be_void_target_count */
	T1 a3; /* has_fatal_error */
	T0* a4; /* system_processor */
	T0* a5; /* current_dynamic_system */
	T0* a6; /* dynamic_types */
	T1 a7; /* short_names */
	T1 a8; /* split_mode */
	T6 a9; /* split_threshold */
	T1 a10; /* use_boehm_gc */
	T0* a11; /* system_name */
	T0* a12; /* current_file */
	T0* a13; /* header_file */
	T0* a14; /* current_type */
	T0* a15; /* current_feature */
	T0* a16; /* extra_dynamic_type_sets */
	T0* a17; /* current_dynamic_type_sets */
	T0* a18; /* current_function_header_buffer */
	T0* a19; /* current_function_body_buffer */
	T0* a20; /* temp_variables */
	T0* a21; /* used_temp_variables */
	T0* a22; /* free_temp_variables */
	T0* a23; /* frozen_temp_variables */
	T0* a24; /* unused_unqualified_call_expressions */
	T0* a25; /* unused_qualified_call_expressions */
	T0* a26; /* unused_actual_argument_lists */
	T0* a27; /* unused_result_expressions */
	T0* a28; /* unused_identifiers */
	T0* a29; /* conforming_types */
	T0* a30; /* non_conforming_types */
	T0* a31; /* equality_type_set */
	T0* a32; /* equality_common_types */
	T0* a33; /* unused_standalone_type_sets */
	T0* a34; /* operand_stack */
	T0* a35; /* call_operands */
	T0* a36; /* attachment_dynamic_type_ids */
	T0* a37; /* target_dynamic_type_ids */
	T0* a38; /* target_dynamic_types */
	T0* a39; /* standalone_type_sets */
	T0* a40; /* deep_twin_types */
	T0* a41; /* deep_equal_types */
	T0* a42; /* deep_feature_target_type_sets */
	T0* a43; /* current_object_tests */
	T0* a44; /* current_object_equalities */
	T0* a45; /* current_equalities */
	T0* a46; /* current_agents */
	T0* a47; /* agent_open_operands */
	T0* a48; /* agent_closed_operands */
	T0* a49; /* agent_target */
	T0* a50; /* agent_arguments */
	T0* a51; /* agent_instruction */
	T0* a52; /* agent_expression */
	T0* a53; /* agent_closed_operands_type */
	T0* a54; /* agent_manifest_tuple */
	T0* a55; /* formal_arguments */
	T0* a56; /* manifest_array_types */
	T0* a57; /* big_manifest_array_types */
	T0* a58; /* manifest_tuple_types */
	T0* a59; /* once_features */
	T0* a60; /* once_per_process_counts */
	T0* a61; /* once_per_thread_counts */
	T0* a62; /* constant_features */
	T0* a63; /* inline_constants */
	T0* a64; /* dispose_procedures */
	T0* a65; /* dynamic_type_id_set_names */
	T0* a66; /* called_features */
	T0* a67; /* called_static_features */
	T0* a68; /* included_header_filenames */
	T0* a69; /* included_cpp_header_filenames */
	T0* a70; /* included_runtime_header_files */
	T0* a71; /* included_runtime_c_files */
	T0* a72; /* c_filenames */
	T1 a73; /* finalize_mode */
	T0* a74; /* locals_written_in_body */
	T0* a75; /* locals_written_in_rescue */
	T0* a76; /* locals_written */
	T0* a77; /* locals_read_in_body */
	T0* a78; /* locals_read_in_rescue */
	T0* a79; /* locals_read */
	T0* a80; /* external_c_regexp */
	T0* a81; /* external_c_macro_regexp */
	T0* a82; /* external_c_struct_regexp */
	T0* a83; /* external_c_inline_regexp */
	T0* a84; /* old_external_c_regexp */
	T0* a85; /* old_external_c_macro_regexp */
	T0* a86; /* old_external_c_struct_regexp */
	T0* a87; /* external_cpp_regexp */
	T0* a88; /* external_cpp_macro_regexp */
	T0* a89; /* external_cpp_struct_regexp */
	T0* a90; /* external_cpp_inline_regexp */
	T0* a91; /* external_dllwin_regexp */
	T0* a92; /* old_external_dllwin32_regexp */
	T1 a93; /* in_static_feature */
	T1 a94; /* type_info_ancestors_used */
	T1 a95; /* type_info_attributes_used */
	T1 a96; /* type_info_attribute_name_used */
	T1 a97; /* type_info_attribute_type_id_used */
	T1 a98; /* type_info_attribute_offset_used */
	T1 a99; /* type_info_generator_used */
	T1 a100; /* type_info_name_used */
	T1 a101; /* type_info_generic_parameters_used */
	T1 a102; /* type_info_object_size_used */
	T0* a103; /* c_file */
	T6 a104; /* c_file_size */
	T0* a105; /* cpp_file */
	T6 a106; /* cpp_file_size */
	T6 a107; /* indentation */
	T0* a108; /* call_target_type */
	T0* a109; /* assignment_target */
	T1 a110; /* in_operand */
	T1 a111; /* in_procedure_call_target */
	T0* a112; /* current_agent */
	T1 a113; /* call_target_check_void */
	T1 a114; /* has_rescue */
	T1 a115; /* result_written */
	T1 a116; /* result_read */
	T1 a117; /* has_retry */
};

/* Struct for type detachable DT_DATE_TIME */
struct S68 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* time_storage */
	T6 a2; /* date_storage */
};

/* Struct for type detachable DS_HASH_SET [[attached] STRING_8] */
struct S70 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* internal_cursor */
	T6 a3; /* count */
	T0* a4; /* item_storage */
	T0* a5; /* equality_tester */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* free_slot */
	T6 a9; /* last_position */
	T6 a10; /* modulus */
	T6 a11; /* clashes_previous_position */
	T6 a12; /* found_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
struct S71 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* dynamic_type_set_count */
	T1 a2; /* has_fatal_error */
	T0* a3; /* current_dynamic_system */
	T0* a4; /* feature_checker */
	T0* a5; /* current_dynamic_type */
	T0* a6; /* current_dynamic_feature */
	T0* a7; /* dynamic_type_sets */
	T0* a8; /* object_id_dynamic_type_set */
	T0* a9; /* constant_indexes */
	T1 a10; /* catcall_error_mode */
	T1 a11; /* catcall_warning_mode */
	T1 a12; /* no_debug */
	T1 a13; /* no_assertion */
	T1 a14; /* is_built */
	T0* a15; /* alive_conforming_descendants_per_type */
	T0* a16; /* type_checker */
	T0* a17; /* current_type */
	T0* a18; /* current_class */
	T0* a19; /* current_feature */
	T0* a20; /* current_feature_impl */
	T0* a21; /* current_class_impl */
	T0* a22; /* enclosing_inline_agents */
	T0* a23; /* unused_overloaded_procedures_list */
	T0* a24; /* unused_overloaded_queries_list */
	T0* a25; /* unused_overloaded_features_list */
	T0* a26; /* unused_call_infos */
	T0* a27; /* unused_contexts */
	T0* a28; /* current_context */
	T0* a29; /* current_target_type */
	T0* a30; /* current_object_test_types */
	T0* a31; /* current_object_test_scope */
	T0* a32; /* object_test_scope_builder */
	T0* a33; /* current_iteration_cursor_types */
	T0* a34; /* current_iteration_item_types */
	T0* a35; /* current_iteration_item_scope */
	T0* a36; /* current_initialization_scope */
	T0* a37; /* current_attachment_scope */
	T0* a38; /* attachment_scope_builder */
	T0* a39; /* unused_attachment_scopes */
	T0* a40; /* assertions_by_feature */
	T0* a41; /* common_ancestor_type_list */
	T0* a42; /* indexing_term_list */
	T0* a43; /* default_creation_call_name */
	T0* a44; /* default_creation_call */
	T0* a45; /* vape_non_descendant_clients */
	T0* a46; /* vape_creation_clients */
	T0* a47; /* vape_client */
	T0* a48; /* unused_adapted_base_classes */
	T0* a49; /* adapted_base_class_checker */
	T0* a50; /* system_processor */
	T6 a51; /* current_index */
	T6 a52; /* result_index */
	T6 a53; /* attached_result_index */
	T6 a54; /* void_index */
	T1 a55; /* in_precursor */
	T1 a56; /* in_static_feature */
	T1 a57; /* in_rescue */
	T0* a58; /* current_inline_agent */
	T0* a59; /* supplier_handler */
	T1 a60; /* in_invariant */
	T1 a61; /* in_precondition */
	T1 a62; /* in_postcondition */
	T1 a63; /* in_check_instruction */
	T0* a64; /* precursor_queries */
	T0* a65; /* precursor_procedures */
};

/* Struct for type [detachable] ET_CLASS_TYPE */
struct S72 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
	T0* a3; /* actual_parameters */
	T0* a4; /* named_base_class */
	T1 a5; /* tuple_actual_parameters_unfolded_1 */
	T1 a6; /* tuple_actual_parameters_unfolded_2 */
};

/* Struct for type [detachable] ET_TOKEN_CONSTANTS */
struct S73 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_CLASS */
struct S74 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* group */
	T0* a2; /* status_mutex */
	T1 a3; /* unprotected_is_parsed */
	T1 a4; /* unprotected_has_syntax_error */
	T1 a5; /* unprotected_ancestors_built */
	T1 a6; /* unprotected_has_ancestors_error */
	T1 a7; /* unprotected_features_flattened */
	T1 a8; /* unprotected_has_flattening_error */
	T1 a9; /* unprotected_interface_checked */
	T1 a10; /* unprotected_has_interface_error */
	T1 a11; /* unprotected_implementation_checked */
	T1 a12; /* unprotected_has_implementation_error */
	T0* a13; /* name */
	T6 a14; /* id */
	T0* a15; /* ancestors */
	T0* a16; /* conforming_ancestors */
	T0* a17; /* queries */
	T0* a18; /* procedures */
	T0* a19; /* class_keyword */
	T0* a20; /* end_keyword */
	T0* a21; /* processing_mutex */
	T0* a22; /* formal_parameter_types_mutex */
	T0* a23; /* named_base_class */
	T6 a24; /* time_stamp */
	T8 a25; /* class_code */
	T1 a26; /* tuple_actual_parameters_unfolded_1 */
	T1 a27; /* tuple_actual_parameters_unfolded_2 */
	T1 a28; /* is_checking_implementation */
	T0* a29; /* formal_parameters */
	T6 a30; /* index */
	T0* a31; /* class_mark */
	T1 a32; /* has_deferred_features */
	T0* a33; /* formal_parameter_types */
	T0* a34; /* parent_clauses */
	T0* a35; /* suppliers */
	T1 a36; /* redeclared_signatures_checked */
	T0* a37; /* invariants */
	T0* a38; /* filename */
	T0* a39; /* creators */
	T0* a40; /* convert_features */
	T1 a41; /* is_ignored */
	T0* a42; /* providers */
	T1 a43; /* is_interface */
	T1 a44; /* unprotected_is_marked */
	T1 a45; /* has_utf8_bom */
	T6 a46; /* registered_feature_count */
	T0* a47; /* external_keyword */
	T0* a48; /* frozen_keyword */
	T0* a49; /* feature_clauses */
	T0* a50; /* first_indexing */
	T0* a51; /* second_indexing */
	T6 a52; /* tuple_constraint_position */
	T0* a53; /* obsolete_message */
	T6 a54; /* registered_inline_constant_count */
	T0* a55; /* leading_break */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S76 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] UT_CANNOT_READ_FILE_ERROR */
struct S77 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] UT_VERSION_NUMBER */
struct S79 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] UT_MESSAGE */
struct S80 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] RX_PCRE_REGULAR_EXPRESSION */
struct S81 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* match_count */
	T0* a2; /* subject */
	T6 a3; /* subject_end */
	T6 a4; /* subject_start */
	T0* a5; /* offset_vector */
	T6 a6; /* offset_vector_count */
	T6 a7; /* brastart_capacity */
	T0* a8; /* brastart_vector */
	T6 a9; /* brastart_lower */
	T6 a10; /* brastart_count */
	T6 a11; /* eptr_capacity */
	T0* a12; /* eptr_vector */
	T6 a13; /* eptr_lower */
	T6 a14; /* eptr_upper */
	T0* a15; /* byte_code */
	T0* a16; /* internal_start_bits */
	T0* a17; /* pattern_buffer */
	T6 a18; /* subexpression_count */
	T0* a19; /* pattern */
	T6 a20; /* pattern_count */
	T6 a21; /* pattern_position */
	T6 a22; /* code_index */
	T6 a23; /* maxbackrefs */
	T10 a24; /* optchanged */
	T7 a25; /* first_character */
	T7 a26; /* required_character */
	T6 a27; /* regexp_countlits */
	T0* a28; /* start_bits */
	T0* a29; /* error_message */
	T1 a30; /* is_anchored */
	T0* a31; /* character_case_mapping */
	T0* a32; /* word_set */
	T6 a33; /* subject_next_start */
	T6 a34; /* error_code */
	T6 a35; /* error_position */
	T1 a36; /* is_startline */
	T1 a37; /* is_caseless */
	T1 a38; /* is_bol */
	T1 a39; /* is_eol */
	T1 a40; /* is_multiline */
	T1 a41; /* is_dotall */
	T1 a42; /* is_extended */
	T1 a43; /* is_empty_allowed */
	T1 a44; /* is_dollar_endonly */
	T1 a45; /* is_greedy */
	T1 a46; /* is_strict */
	T1 a47; /* is_ichanged */
	T6 a48; /* first_matched_index */
	T6 a49; /* eptr */
	T6 a50; /* offset_top */
	T1 a51; /* is_matching_caseless */
	T1 a52; /* is_matching_multiline */
	T1 a53; /* is_matching_dotall */
};

/* Struct for type [detachable] KL_STRING_ROUTINES */
struct S82 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_EXECUTION_ENVIRONMENT */
struct S86 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] AP_ERROR */
struct S87 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* default_template */
	T0* a3; /* code */
};

/* Struct for type [detachable] KL_SHELL_COMMAND */
struct S88 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* exit_code */
	T0* a2; /* command */
	T6 a3; /* return_code */
	T1 a4; /* is_system_code */
};

/* Struct for type [detachable] GECC */
struct S89 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* exit_code */
	T0* a2; /* error_handler */
	T0* a3; /* version_flag */
	T0* a4; /* script_filename */
	T0* a5; /* thread_option */
};

/* Struct for type [detachable] KL_OPERATING_SYSTEM */
struct S90 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] VOID_TARGET */
struct S93 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] ROUTINE_FAILURE */
struct S95 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* routine_name */
	T0* a2; /* class_name */
	T0* a3; /* throwing_exception */
	T0* a4; /* internal_trace */
	T0* a5; /* c_description */
	T0* a6; /* recipient_name */
	T0* a7; /* type_name */
};

/* Struct for type [detachable] OLD_VIOLATION */
struct S97 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] NO_MORE_MEMORY */
struct S99 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_trace */
	T6 a2; /* internal_code */
	T0* a3; /* throwing_exception */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] INVARIANT_VIOLATION */
struct S100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T1 a2; /* is_entry */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE */
struct S101 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* signal_code */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] IO_FAILURE */
struct S102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_code */
	T0* a2; /* throwing_exception */
	T6 a3; /* error_code */
	T0* a4; /* internal_trace */
	T0* a5; /* c_description */
	T0* a6; /* recipient_name */
	T0* a7; /* type_name */
};

/* Struct for type [detachable] OPERATING_SYSTEM_FAILURE */
struct S103 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* error_code */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] COM_FAILURE */
struct S104 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* hresult_code */
	T0* a3; /* exception_information */
	T6 a4; /* hresult */
	T0* a5; /* internal_trace */
	T0* a6; /* c_description */
	T0* a7; /* recipient_name */
	T0* a8; /* type_name */
};

/* Struct for type [detachable] EIFFEL_RUNTIME_PANIC */
struct S105 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_code */
	T0* a2; /* throwing_exception */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] PRECONDITION_VIOLATION */
struct S107 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] POSTCONDITION_VIOLATION */
struct S108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] FLOATING_POINT_FAILURE */
struct S109 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] CHECK_VIOLATION */
struct S110 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] BAD_INSPECT_VALUE */
struct S111 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] VARIANT_VIOLATION */
struct S112 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] LOOP_INVARIANT_VIOLATION */
struct S113 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] RESCUE_FAILURE */
struct S114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] RESUMPTION_FAILURE */
struct S115 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] CREATE_ON_DEFERRED */
struct S116 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] EXTERNAL_FAILURE */
struct S117 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] VOID_ASSIGNED_TO_EXPANDED */
struct S118 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
struct S119 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] MISMATCH_FAILURE */
struct S120 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] DEVELOPER_EXCEPTION */
struct S121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE */
struct S122 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] SERIALIZATION_FAILURE */
struct S123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] EXECUTION_ENVIRONMENT */
struct S124 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* return_code */
};

/* Struct for type [detachable] KL_WINDOWS_FILE_SYSTEM */
struct S125 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_UNIX_FILE_SYSTEM */
struct S126 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] PRIMES */
struct S127 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] INTEGER_32] */
struct S128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T6 z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] BOOLEAN] */
struct S129 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T1 z2[1]; /* item */
};

/* Struct for type [detachable] ARGUMENTS_32 */
struct S130 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] MUTEX */
struct S131 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* owner_thread_id */
	T14 a2; /* mutex_pointer */
};

/* Struct for type [detachable] UT_ERROR_HANDLER */
struct S132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* error_file */
	T0* a2; /* warning_file */
	T0* a3; /* info_file */
};

/* Struct for type [detachable] KL_STANDARD_FILES */
struct S133 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_STDERR_FILE */
struct S134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_STDOUT_FILE */
struct S135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable KL_LINKABLE [[attached] CHARACTER_8] */
struct S137 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] MANAGED_POINTER */
struct S140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* item */
	T1 a2; /* is_shared */
	T6 a3; /* count */
};

/* Struct for type [detachable] FILE_INFO */
struct S141 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* buffered_file_info */
	T1 a2; /* is_following_symlinks */
	T1 a3; /* exists */
	T0* a4; /* internal_file_name */
	T0* a5; /* internal_name_pointer */
};

/* Struct for type [detachable] ST_WORD_WRAPPER */
struct S142 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* new_line_indentation */
	T6 a2; /* broken_words */
	T6 a3; /* maximum_text_width */
};

/* Struct for type [detachable] AP_DISPLAY_HELP_FLAG */
struct S144 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* description */
	T2 a2; /* short_form */
	T1 a3; /* has_short_form */
	T0* a4; /* long_form */
	T6 a5; /* occurrences */
	T1 a6; /* is_hidden */
	T1 a7; /* is_mandatory */
	T6 a8; /* maximum_occurrences */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] AP_OPTION] */
struct S145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* count */
	T0* a3; /* equality_tester */
	T0* a4; /* storage */
	T0* a5; /* special_routines */
	T6 a6; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] STRING_8] */
struct S147 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* internal_cursor */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
	T0* a6; /* equality_tester */
};

/* Struct for type [detachable] AP_ERROR_HANDLER */
struct S148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* error_file */
	T0* a3; /* warning_file */
	T0* a4; /* info_file */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] AP_OPTION] */
struct S149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
	T1 a3; /* after */
	T0* a4; /* current_cell */
	T0* a5; /* next_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [detachable STRING_8] */
struct S150 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
struct S151 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] STRING_8] */
struct S152 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
	T0* a5; /* equality_tester */
};

/* Struct for type [detachable] KL_STRING_EQUALITY_TESTER */
struct S153 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] STRING_8] */
struct S154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] BOOLEAN] */
struct S157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* internal_cursor */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] INTEGER_32] */
struct S160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* internal_cursor */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] KL_NULL_TEXT_OUTPUT_STREAM */
struct S161 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
};

/* Struct for type [detachable] ET_ECF_AST_FACTORY */
struct S162 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_CELL [detachable ET_ECF_SYSTEM] */
struct S163 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_TARGET] */
struct S164 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]] */
struct S165 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type detachable TUPLE */
struct S166 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER] */
struct S167 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] XM_ELEMENT */
struct S168 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* namespace */
	T0* a3; /* parent */
	T0* a4; /* children */
};

/* Struct for type detachable XM_POSITION_TABLE */
struct S169 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table */
};

/* Struct for type [detachable] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8] */
struct S170 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8]] */
struct S171 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [detachable] STRING_8, [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]], [attached] DS_CELL [detachable ET_ECF_SYSTEM]] */
struct S172 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
struct S174 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* count */
	T6 a7; /* last_position */
	T6 a8; /* free_slot */
	T6 a9; /* position */
	T6 a10; /* capacity */
	T0* a11; /* equality_tester */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
struct S175 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
struct S176 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* count */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* capacity */
	T0* a10; /* equality_tester */
	T0* a11; /* internal_cursor */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] XM_EIFFEL_PARSER */
struct S178 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_last_error_description */
	T0* a2; /* scanner */
	T0* a3; /* error_positions */
	T0* a4; /* scanners */
	T1 a5; /* in_external_dtd */
	T0* a6; /* callbacks */
	T0* a7; /* entities */
	T0* a8; /* pe_entities */
	T0* a9; /* dtd_resolver */
	T0* a10; /* entity_resolver */
	T1 a11; /* use_namespaces */
	T6 a12; /* string_mode */
	T0* a13; /* dtd_callbacks */
	T0* a14; /* last_string_value */
	T0* a15; /* yyss */
	T0* a16; /* yyspecial_routines1 */
	T6 a17; /* yyvsc1 */
	T0* a18; /* yyvs1 */
	T0* a19; /* yyspecial_routines2 */
	T6 a20; /* yyvsc2 */
	T0* a21; /* yyvs2 */
	T0* a22; /* yyspecial_routines3 */
	T6 a23; /* yyvsc3 */
	T0* a24; /* yyvs3 */
	T0* a25; /* yyspecial_routines4 */
	T6 a26; /* yyvsc4 */
	T0* a27; /* yyvs4 */
	T0* a28; /* yyspecial_routines5 */
	T6 a29; /* yyvsc5 */
	T0* a30; /* yyvs5 */
	T0* a31; /* yyspecial_routines6 */
	T6 a32; /* yyvsc6 */
	T0* a33; /* yyvs6 */
	T0* a34; /* yyspecial_routines7 */
	T6 a35; /* yyvsc7 */
	T0* a36; /* yyvs7 */
	T0* a37; /* yyspecial_routines8 */
	T6 a38; /* yyvsc8 */
	T0* a39; /* yyvs8 */
	T0* a40; /* yyspecial_routines9 */
	T6 a41; /* yyvsc9 */
	T0* a42; /* yyvs9 */
	T0* a43; /* yyspecial_routines10 */
	T6 a44; /* yyvsc10 */
	T0* a45; /* yyvs10 */
	T0* a46; /* yyspecial_routines11 */
	T6 a47; /* yyvsc11 */
	T0* a48; /* yyvs11 */
	T0* a49; /* yytranslate */
	T0* a50; /* yyr1 */
	T0* a51; /* yytypes1 */
	T0* a52; /* yytypes2 */
	T0* a53; /* yydefact */
	T0* a54; /* yydefgoto */
	T0* a55; /* yypact */
	T0* a56; /* yypgoto */
	T0* a57; /* yytable */
	T0* a58; /* yycheck */
	T6 a59; /* yy_parsing_status */
	T6 a60; /* yy_suspended_yystacksize */
	T6 a61; /* yy_suspended_yystate */
	T6 a62; /* yy_suspended_yyn */
	T6 a63; /* yy_suspended_yychar1 */
	T6 a64; /* yy_suspended_index */
	T6 a65; /* yy_suspended_yyss_top */
	T6 a66; /* yy_suspended_yy_goto */
	T6 a67; /* yyssp */
	T6 a68; /* error_count */
	T1 a69; /* yy_lookahead_needed */
	T6 a70; /* yyerrstatus */
	T6 a71; /* last_token */
	T6 a72; /* yyvsp1 */
	T6 a73; /* yyvsp2 */
	T6 a74; /* yyvsp3 */
	T6 a75; /* yyvsp4 */
	T6 a76; /* yyvsp5 */
	T6 a77; /* yyvsp6 */
	T6 a78; /* yyvsp7 */
	T6 a79; /* yyvsp8 */
	T6 a80; /* yyvsp9 */
	T6 a81; /* yyvsp10 */
	T6 a82; /* yyvsp11 */
	T0* a83; /* last_detachable_any_value */
};

/* Struct for type [detachable] XM_TREE_CALLBACKS_PIPE */
struct S179 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* start */
	T0* a2; /* tree */
	T0* a3; /* error */
	T0* a4; /* last */
};

/* Struct for type [detachable] XM_CALLBACKS_TO_TREE_FILTER */
struct S182 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_position_table */
	T0* a2; /* document */
	T0* a3; /* source_parser */
	T0* a4; /* current_element */
	T0* a5; /* namespace_cache */
	T0* a6; /* next */
};

/* Struct for type [detachable] ET_ECF_STATE */
struct S183 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* ise_version */
};

/* Struct for type [detachable] ET_IDENTIFIER */
struct S184 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T6 a2; /* index */
	T0* a3; /* name */
	T2 a4; /* status_code */
	T6 a5; /* compressed_position */
	T6 a6; /* hash_code */
};

/* Struct for type [detachable] ET_ECF_SYSTEM_CONFIG */
struct S185 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* filename */
	T0* a3; /* universe */
	T0* a4; /* ecf_namespace */
	T0* a5; /* ecf_version */
	T0* a6; /* description */
	T0* a7; /* notes */
	T0* a8; /* targets */
	T1 a9; /* is_read_only */
	T0* a10; /* uuid */
	T0* a11; /* library_target */
};

/* Struct for type [detachable] ET_ECF_LIBRARY */
struct S187 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* filename */
	T0* a3; /* ecf_namespace */
	T0* a4; /* ecf_version */
	T0* a5; /* description */
	T0* a6; /* notes */
	T0* a7; /* targets */
	T1 a8; /* is_read_only */
	T0* a9; /* uuid */
	T0* a10; /* library_target */
	T0* a11; /* current_system */
	T0* a12; /* library */
	T0* a13; /* libraries */
	T0* a14; /* clusters */
	T0* a15; /* dotnet_assemblies */
	T0* a16; /* implicit_attachment_type_mark */
	T0* a17; /* master_classes */
	T0* a18; /* master_class_mutex */
	T0* a19; /* any_type */
	T0* a20; /* detachable_any_type */
	T0* a21; /* detachable_separate_any_type */
	T0* a22; /* any_parent */
	T0* a23; /* any_parents */
	T0* a24; /* any_clients */
	T0* a25; /* tuple_type */
	T0* a26; /* detachable_tuple_type */
	T0* a27; /* tuple_identity_type */
	T0* a28; /* unfolded_empty_tuple_actual_parameters */
	T0* a29; /* array_any_type */
	T0* a30; /* array_detachable_any_type */
	T0* a31; /* array_none_type */
	T0* a32; /* array_identity_type */
	T0* a33; /* boolean_type */
	T0* a34; /* character_type */
	T0* a35; /* character_8_type */
	T0* a36; /* character_8_convert_feature */
	T0* a37; /* character_32_type */
	T0* a38; /* character_32_convert_feature */
	T0* a39; /* double_type */
	T0* a40; /* exception_type */
	T0* a41; /* detachable_exception_type */
	T0* a42; /* exception_manager_type */
	T0* a43; /* function_identity_any_type */
	T0* a44; /* immutable_string_8_type */
	T0* a45; /* immutable_string_32_type */
	T0* a46; /* integer_type */
	T0* a47; /* integer_8_type */
	T0* a48; /* integer_8_convert_feature */
	T0* a49; /* integer_16_type */
	T0* a50; /* integer_16_convert_feature */
	T0* a51; /* integer_32_type */
	T0* a52; /* integer_32_convert_feature */
	T0* a53; /* integer_64_type */
	T0* a54; /* integer_64_convert_feature */
	T0* a55; /* ise_exception_manager_type */
	T0* a56; /* iterable_detachable_separate_any_type */
	T0* a57; /* natural_type */
	T0* a58; /* natural_8_type */
	T0* a59; /* natural_8_convert_feature */
	T0* a60; /* natural_16_type */
	T0* a61; /* natural_16_convert_feature */
	T0* a62; /* natural_32_type */
	T0* a63; /* natural_32_convert_feature */
	T0* a64; /* natural_64_type */
	T0* a65; /* natural_64_convert_feature */
	T0* a66; /* none_type */
	T0* a67; /* detachable_none_type */
	T0* a68; /* pointer_type */
	T0* a69; /* predicate_identity_type */
	T0* a70; /* procedure_identity_type */
	T0* a71; /* real_type */
	T0* a72; /* real_32_type */
	T0* a73; /* real_32_convert_feature */
	T0* a74; /* real_64_type */
	T0* a75; /* real_64_convert_feature */
	T0* a76; /* routine_identity_type */
	T0* a77; /* special_any_type */
	T0* a78; /* special_detachable_any_type */
	T0* a79; /* special_identity_type */
	T0* a80; /* string_type */
	T0* a81; /* detachable_string_type */
	T0* a82; /* string_8_type */
	T0* a83; /* detachable_string_8_type */
	T0* a84; /* string_8_convert_feature */
	T0* a85; /* string_32_type */
	T0* a86; /* string_32_convert_feature */
	T0* a87; /* system_object_type */
	T0* a88; /* system_object_parents */
	T0* a89; /* system_string_type */
	T0* a90; /* type_detachable_any_type */
	T0* a91; /* detachable_type_exception_type */
	T0* a92; /* type_detachable_exception_type */
	T0* a93; /* type_detachable_like_current_type */
	T0* a94; /* type_identity_type */
	T0* a95; /* typed_pointer_identity_type */
	T0* a96; /* wide_character_type */
	T0* a97; /* selected_target */
	T1 a98; /* obsolete_iteration_mode */
	T1 a99; /* obsolete_routine_type_mode */
	T1 a100; /* is_preparsed */
	T1 a101; /* use_obsolete_syntax_mode */
};

/* Struct for type [detachable] ET_ADAPTED_LIBRARIES */
struct S188 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* libraries */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ADAPTED_LIBRARY] */
struct S190 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_LIBRARY]] */
struct S191 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] XM_DOCUMENT */
struct S192 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* root_element */
	T0* a2; /* children */
};

/* Struct for type [detachable] ET_COMPRESSED_POSITION */
struct S193 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
};

/* Struct for type [detachable] XM_STOP_ON_ERROR_FILTER */
struct S195 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* last_error */
	T0* a3; /* next */
};

/* Struct for type detachable XM_ATTRIBUTE */
struct S196 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* namespace */
	T0* a3; /* value */
	T0* a4; /* parent */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
struct S198 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T6 a4; /* last_position */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* key_equality_tester */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T0* a14; /* special_key_routines */
	T0* a15; /* key_storage */
	T0* a16; /* equality_tester */
	T6 a17; /* free_slot */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
struct S199 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* position */
	T0* a3; /* internal_cursor */
	T0* a4; /* key_storage */
	T0* a5; /* item_storage */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* equality_tester */
	T6 a18; /* free_slot */
	T0* a19; /* hash_function */
};

/* Struct for type detachable ET_ECF_TARGET_PARENT */
struct S200 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* location_id */
	T0* a3; /* target */
};

/* Struct for type [detachable] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
struct S201 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [attached] ET_ECF_INTERNAL_UNIVERSE, [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
struct S202 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type detachable ET_ECF_TARGETS */
struct S203 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* targets */
};

/* Struct for type [detachable] DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S204 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T0* a3; /* hash_function */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* count */
	T0* a9; /* equality_tester */
	T6 a10; /* found_position */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* item_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S205 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] DS_CELL [detachable ET_ECF_LIBRARY] */
struct S206 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] ET_ECF_ADAPTED_LIBRARY */
struct S207 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* filename_id */
	T0* a3; /* universe */
	T0* a4; /* target */
	T0* a5; /* description */
	T0* a6; /* classname_prefix */
	T1 a7; /* is_read_only */
	T1 a8; /* use_application_options */
	T0* a9; /* options */
	T0* a10; /* class_options */
	T0* a11; /* class_renamings */
	T0* a12; /* visible_classes */
	T0* a13; /* conditions */
	T0* a14; /* notes */
	T0* a15; /* name */
	T0* a16; /* library */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [attached] ET_ECF_ADAPTED_LIBRARY, [attached] DS_CELL [detachable ET_ECF_LIBRARY]] */
struct S208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] ET_ADAPTED_DOTNET_ASSEMBLIES */
struct S210 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* dotnet_assemblies */
};

/* Struct for type [detachable] ET_ECF_DOTNET_ASSEMBLY */
struct S211 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* referenced_assemblies */
	T0* a3; /* current_system */
	T0* a4; /* dotnet_assembly */
	T0* a5; /* name */
	T0* a6; /* implicit_attachment_type_mark */
	T0* a7; /* master_classes */
	T0* a8; /* master_class_mutex */
	T0* a9; /* any_type */
	T0* a10; /* detachable_any_type */
	T0* a11; /* detachable_separate_any_type */
	T0* a12; /* any_parent */
	T0* a13; /* any_parents */
	T0* a14; /* any_clients */
	T0* a15; /* tuple_type */
	T0* a16; /* detachable_tuple_type */
	T0* a17; /* tuple_identity_type */
	T0* a18; /* unfolded_empty_tuple_actual_parameters */
	T0* a19; /* array_any_type */
	T0* a20; /* array_detachable_any_type */
	T0* a21; /* array_none_type */
	T0* a22; /* array_identity_type */
	T0* a23; /* boolean_type */
	T0* a24; /* character_type */
	T0* a25; /* character_8_type */
	T0* a26; /* character_8_convert_feature */
	T0* a27; /* character_32_type */
	T0* a28; /* character_32_convert_feature */
	T0* a29; /* double_type */
	T0* a30; /* exception_type */
	T0* a31; /* detachable_exception_type */
	T0* a32; /* exception_manager_type */
	T0* a33; /* function_identity_any_type */
	T0* a34; /* immutable_string_8_type */
	T0* a35; /* immutable_string_32_type */
	T0* a36; /* integer_type */
	T0* a37; /* integer_8_type */
	T0* a38; /* integer_8_convert_feature */
	T0* a39; /* integer_16_type */
	T0* a40; /* integer_16_convert_feature */
	T0* a41; /* integer_32_type */
	T0* a42; /* integer_32_convert_feature */
	T0* a43; /* integer_64_type */
	T0* a44; /* integer_64_convert_feature */
	T0* a45; /* ise_exception_manager_type */
	T0* a46; /* iterable_detachable_separate_any_type */
	T0* a47; /* natural_type */
	T0* a48; /* natural_8_type */
	T0* a49; /* natural_8_convert_feature */
	T0* a50; /* natural_16_type */
	T0* a51; /* natural_16_convert_feature */
	T0* a52; /* natural_32_type */
	T0* a53; /* natural_32_convert_feature */
	T0* a54; /* natural_64_type */
	T0* a55; /* natural_64_convert_feature */
	T0* a56; /* none_type */
	T0* a57; /* detachable_none_type */
	T0* a58; /* pointer_type */
	T0* a59; /* predicate_identity_type */
	T0* a60; /* procedure_identity_type */
	T0* a61; /* real_type */
	T0* a62; /* real_32_type */
	T0* a63; /* real_32_convert_feature */
	T0* a64; /* real_64_type */
	T0* a65; /* real_64_convert_feature */
	T0* a66; /* routine_identity_type */
	T0* a67; /* special_any_type */
	T0* a68; /* special_detachable_any_type */
	T0* a69; /* special_identity_type */
	T0* a70; /* string_type */
	T0* a71; /* detachable_string_type */
	T0* a72; /* string_8_type */
	T0* a73; /* detachable_string_8_type */
	T0* a74; /* string_8_convert_feature */
	T0* a75; /* string_32_type */
	T0* a76; /* string_32_convert_feature */
	T0* a77; /* system_object_type */
	T0* a78; /* system_object_parents */
	T0* a79; /* system_string_type */
	T0* a80; /* type_detachable_any_type */
	T0* a81; /* detachable_type_exception_type */
	T0* a82; /* type_detachable_exception_type */
	T0* a83; /* type_detachable_like_current_type */
	T0* a84; /* type_identity_type */
	T0* a85; /* typed_pointer_identity_type */
	T0* a86; /* wide_character_type */
	T1 a87; /* is_read_only */
	T1 a88; /* is_preparsed */
};

/* Struct for type [detachable] ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
struct S213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* filename_id */
	T0* a3; /* universe */
	T0* a4; /* target */
	T0* a5; /* description */
	T0* a6; /* assembly_culture */
	T0* a7; /* assembly_key */
	T0* a8; /* assembly_name */
	T0* a9; /* assembly_version */
	T0* a10; /* classname_prefix */
	T1 a11; /* is_read_only */
	T0* a12; /* options */
	T0* a13; /* class_options */
	T0* a14; /* class_renamings */
	T0* a15; /* conditions */
	T0* a16; /* notes */
	T0* a17; /* name */
	T0* a18; /* dotnet_assembly */
};

/* Struct for type [detachable] ET_ECF_ERROR */
struct S214 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* system_config */
	T0* a4; /* position */
	T0* a5; /* code */
};

/* Struct for type [detachable] KL_AGENT_HASH_FUNCTION [[attached] STRING_8] */
struct S215 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* hash_agent */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8] */
struct S216 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T0* a4; /* item_storage */
	T0* a5; /* key_storage */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_keys */
	T0* a8; /* hash_function */
	T6 a9; /* position */
	T6 a10; /* last_position */
	T6 a11; /* capacity */
	T6 a12; /* slots_position */
	T6 a13; /* free_slot */
	T0* a14; /* equality_tester */
	T6 a15; /* modulus */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* clashes */
	T0* a19; /* slots */
	T0* a20; /* special_key_routines */
};

/* Struct for type [detachable] TUPLE [[attached] STRING_8] */
struct S217 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] FUNCTION [[attached] TUPLE [[attached] STRING_8], [attached] INTEGER_32] */
struct S218 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] KL_STRING_ROUTINES] */
struct S219 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_IDENTIFIER, [attached] STRING_8] */
struct S221 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T0* a3; /* hash_function */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* count */
	T0* a9; /* equality_tester */
	T6 a10; /* found_position */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* item_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type [detachable] ET_DYNAMIC_PRIMARY_TYPE */
struct S223 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* attribute_count */
	T0* a2; /* queries */
	T1 a3; /* is_alive */
	T6 a4; /* id */
	T0* a5; /* base_type */
	T0* a6; /* attached_type */
	T0* a7; /* base_class */
	T0* a8; /* meta_type */
	T0* a9; /* conforming_ancestors */
	T0* a10; /* queries_by_seed */
	T0* a11; /* procedures_by_seed */
	T0* a12; /* procedures */
	T1 a13; /* has_redefined_copy_routine */
	T1 a14; /* has_redefined_is_equal_routine */
	T1 a15; /* has_once_per_object_routines */
	T6 a16; /* hash_code */
	T0* a17; /* next_type */
	T1 a18; /* has_reference_attributes */
	T0* a19; /* query_calls */
	T1 a20; /* has_static */
	T0* a21; /* procedure_calls */
	T0* a22; /* equality_expressions */
	T0* a23; /* object_equality_expressions */
};

/* Struct for type [detachable] ET_DYNAMIC_FEATURE_LIST */
struct S224 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
struct S225 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T0* a3; /* equality_tester */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T0* a9; /* special_item_routines */
	T0* a10; /* item_storage */
	T0* a11; /* special_key_routines */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T6 a15; /* found_position */
	T0* a16; /* internal_cursor */
	T6 a17; /* count */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] STRING_8] */
struct S226 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* count */
	T0* a9; /* equality_tester */
	T6 a10; /* modulus */
	T6 a11; /* clashes_previous_position */
	T0* a12; /* special_item_routines */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_key_routines */
	T0* a16; /* key_storage */
	T6 a17; /* free_slot */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] TUPLE [[attached] ET_CLASS] */
struct S227 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]] */
struct S228 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [INTEGER_32] */
struct S229 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1;
};

/* Struct for type [detachable] ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
struct S230 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_dynamic_system */
	T1 a3; /* catcall_error_mode */
	T1 a4; /* catcall_warning_mode */
	T1 a5; /* no_debug */
	T1 a6; /* no_assertion */
	T0* a7; /* alive_conforming_descendants_per_type */
};

/* Struct for type [detachable] TUPLE [[attached] ET_DYNAMIC_SYSTEM] */
struct S231 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type detachable ET_DYNAMIC_FEATURE */
struct S236 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_feature */
	T0* a2; /* dynamic_type_sets */
	T6 a3; /* id */
	T0* a4; /* target_type */
	T1 a5; /* is_generated */
	T0* a6; /* result_type_set */
	T8 a7; /* builtin_class_code */
	T8 a8; /* builtin_feature_code */
	T1 a9; /* is_is_equal_routine */
	T1 a10; /* is_copy_routine */
	T0* a11; /* query_calls */
	T0* a12; /* procedure_calls */
	T1 a13; /* is_regular */
	T0* a14; /* first_precursor */
	T1 a15; /* is_creation */
	T0* a16; /* other_precursors */
	T1 a17; /* is_built */
	T1 a18; /* is_address */
	T1 a19; /* is_static_generated */
};

/* Struct for type [detachable] ET_ACTUAL_PARAMETER_LIST */
struct S240 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_bracket */
	T0* a4; /* right_bracket */
};

/* Struct for type [detachable] ET_NESTED_TYPE_CONTEXT */
struct S241 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* root_context */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] ET_FORMAL_PARAMETER_TYPE */
struct S242 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
	T6 a3; /* index */
	T0* a4; /* implementation_class */
};

/* Struct for type [detachable] ARRAY [[attached] ET_TYPE] */
struct S244 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_TYPE] */
struct S245 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] ET_EXTERNAL_FUNCTION */
struct S253 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T0* a8; /* preconditions */
	T6 a9; /* hash_code */
	T8 a10; /* builtin_class_code */
	T8 a11; /* builtin_feature_code */
	T0* a12; /* arguments */
	T0* a13; /* postconditions */
	T0* a14; /* assigner */
	T0* a15; /* extended_name */
	T0* a16; /* declared_type */
	T0* a17; /* frozen_keyword */
	T0* a18; /* language */
	T0* a19; /* obsolete_message */
	T0* a20; /* alias_clause */
	T0* a21; /* clients */
	T0* a22; /* is_keyword */
	T0* a23; /* end_keyword */
	T0* a24; /* semicolon */
	T0* a25; /* feature_clause */
	T0* a26; /* first_indexing */
	T0* a27; /* object_tests */
	T1 a28; /* validity_checked */
	T1 a29; /* has_validity_error */
	T6 a30; /* id */
	T6 a31; /* version */
	T0* a32; /* synonym */
};

/* Struct for type [detachable] ET_TUPLE_TYPE */
struct S254 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* actual_parameters */
	T0* a2; /* type_mark */
	T0* a3; /* named_base_class */
	T0* a4; /* tuple_keyword */
};

/* Struct for type detachable ET_FORMAL_ARGUMENT_LIST */
struct S255 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_parenthesis */
	T0* a4; /* right_parenthesis */
};

/* Struct for type [detachable] ET_FORMAL_ARGUMENT */
struct S256 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T6 a3; /* index */
	T6 a4; /* attached_index */
	T1 a5; /* is_used */
};

/* Struct for type [detachable] ET_QUERY_LIST */
struct S257 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* declared_count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] ET_PROCEDURE_LIST */
struct S258 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* declared_count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] ET_DYNAMIC_TYPE_SET_LIST */
struct S259 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] KL_STRING_OUTPUT_STREAM */
struct S260 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* string */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_IDENTIFIER] */
struct S261 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S262 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32] */
struct S263 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S264 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
struct S265 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ACTUAL_ARGUMENT_LIST] */
struct S266 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_RESULT] */
struct S267 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST */
struct S268 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* clashes */
	T0* a4; /* slots */
};

/* Struct for type [detachable] ET_DYNAMIC_STANDALONE_TYPE_SET */
struct S269 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T6 a2; /* count */
	T0* a3; /* dynamic_types */
	T1 a4; /* is_never_void */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S270 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [[attached] ET_EXPRESSION] */
struct S271 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_EXPRESSION] */
struct S272 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* count */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] INTEGER_32] */
struct S273 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* item_storage */
	T6 a3; /* last_position */
	T6 a4; /* capacity */
	T6 a5; /* count */
	T6 a6; /* free_slot */
	T6 a7; /* slots_position */
	T6 a8; /* modulus */
	T6 a9; /* clashes_previous_position */
	T0* a10; /* key_equality_tester */
	T0* a11; /* equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
struct S274 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S275 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* count */
	T6 a3; /* position */
	T0* a4; /* item_storage */
	T6 a5; /* capacity */
	T6 a6; /* modulus */
	T6 a7; /* last_position */
	T6 a8; /* free_slot */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S276 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T0* a4; /* item_storage */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_OBJECT_TEST] */
struct S277 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S278 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_EQUALITY_EXPRESSION] */
struct S279 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_AGENT] */
struct S280 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] ET_CURRENT */
struct S282 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T2 a2; /* code */
	T0* a3; /* text */
	T6 a4; /* compressed_position */
};

/* Struct for type [detachable] ET_ACTUAL_ARGUMENT_LIST */
struct S283 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_symbol */
	T0* a4; /* right_symbol */
};

/* Struct for type [detachable] ET_QUALIFIED_CALL_INSTRUCTION */
struct S284 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* qualified_name */
	T0* a3; /* arguments */
	T0* a4; /* parenthesis_call */
};

/* Struct for type [detachable] ET_QUALIFIED_CALL_EXPRESSION */
struct S287 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T6 a2; /* index */
	T0* a3; /* arguments */
	T0* a4; /* qualified_name */
	T0* a5; /* parenthesis_call */
};

/* Struct for type [detachable] ET_MANIFEST_TUPLE */
struct S288 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_symbol */
	T0* a4; /* right_symbol */
	T6 a5; /* index */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S289 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* count */
	T0* a9; /* special_item_routines */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FEATURE] */
struct S290 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T0* a12; /* equality_tester */
	T0* a13; /* internal_cursor */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] ARRAY [[attached] INTEGER_32] */
struct S291 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
struct S292 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* key_storage */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T0* a10; /* equality_tester */
	T0* a11; /* key_equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] ET_INLINE_CONSTANT] */
struct S293 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* key_storage */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* equality_tester */
	T0* a9; /* key_equality_tester */
	T6 a10; /* found_position */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S294 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T6 a7; /* capacity */
	T0* a8; /* equality_tester */
	T0* a9; /* key_equality_tester */
	T0* a10; /* internal_cursor */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_FEATURE] */
struct S295 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] BOOLEAN, [attached] STRING_8] */
struct S296 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T0* a4; /* key_storage */
	T0* a5; /* key_equality_tester */
	T0* a6; /* internal_keys */
	T6 a7; /* count */
	T6 a8; /* last_position */
	T6 a9; /* free_slot */
	T6 a10; /* capacity */
	T6 a11; /* slots_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* equality_tester */
	T6 a15; /* found_position */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_IDENTIFIER] */
struct S297 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* capacity */
	T6 a3; /* modulus */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T0* a6; /* equality_tester */
	T6 a7; /* slots_position */
	T6 a8; /* count */
	T6 a9; /* clashes_previous_position */
	T0* a10; /* special_item_routines */
	T0* a11; /* item_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] ET_IDENTIFIER_TESTER */
struct S298 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_TEXT_OUTPUT_FILE */
struct S300 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* mode */
	T14 a3; /* file_pointer */
	T1 a4; /* descriptor_available */
	T0* a5; /* last_string_32 */
	T0* a6; /* last_string */
	T0* a7; /* internal_name */
	T0* a8; /* internal_detachable_name_pointer */
};

/* Struct for type [detachable] ET_IMPLICIT_TYPE_MARK */
struct S301 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_expanded_mark */
	T1 a2; /* is_attached_mark */
	T1 a3; /* is_reference_mark */
	T1 a4; /* is_separate_mark */
	T1 a5; /* is_detachable_mark */
};

/* Struct for type [detachable] ET_DYNAMIC_TUPLE_TYPE */
struct S302 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T0* a5; /* attached_type */
	T6 a6; /* attribute_count */
	T0* a7; /* queries */
	T0* a8; /* meta_type */
	T0* a9; /* conforming_ancestors */
	T0* a10; /* queries_by_seed */
	T0* a11; /* procedures_by_seed */
	T0* a12; /* procedures */
	T1 a13; /* has_redefined_copy_routine */
	T0* a14; /* item_type_sets */
	T1 a15; /* has_redefined_is_equal_routine */
	T1 a16; /* has_once_per_object_routines */
	T1 a17; /* has_reference_attributes */
	T6 a18; /* hash_code */
	T0* a19; /* next_type */
	T0* a20; /* query_calls */
	T1 a21; /* has_static */
	T0* a22; /* procedure_calls */
	T0* a23; /* equality_expressions */
	T0* a24; /* object_equality_expressions */
};

/* Struct for type [detachable] DS_STRING_HASH_TABLE */
struct S303 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T6 a4; /* modulus */
	T0* a5; /* key_equality_tester */
	T0* a6; /* slots */
	T0* a7; /* hash_function */
	T0* a8; /* key_storage */
	T0* a9; /* clashes */
	T6 a10; /* count */
	T6 a11; /* capacity */
	T6 a12; /* slots_position */
	T6 a13; /* free_slot */
	T6 a14; /* last_position */
	T0* a15; /* internal_keys */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* equality_tester */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] STRING_8, [attached] STRING_8] */
struct S304 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] UT_CANNOT_WRITE_TO_FILE_ERROR */
struct S305 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type detachable ET_DYNAMIC_PRECURSOR */
struct S306 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_static_generated */
	T0* a2; /* target_type */
	T0* a3; /* static_feature */
	T0* a4; /* dynamic_type_sets */
	T0* a5; /* current_feature */
	T6 a6; /* id */
	T0* a7; /* result_type_set */
	T1 a8; /* is_regular */
	T1 a9; /* is_creation */
	T1 a10; /* is_address */
	T8 a11; /* builtin_class_code */
	T0* a12; /* first_precursor */
	T0* a13; /* other_precursors */
	T1 a14; /* is_built */
	T8 a15; /* builtin_feature_code */
	T1 a16; /* is_is_equal_routine */
	T1 a17; /* is_generated */
	T1 a18; /* is_copy_routine */
	T0* a19; /* parent_type */
	T0* a20; /* query_calls */
	T0* a21; /* procedure_calls */
};

/* Struct for type detachable ET_DYNAMIC_PRECURSOR_LIST */
struct S307 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S308 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sorted_items */
	T0* a2; /* cycle */
	T0* a3; /* indexes */
	T0* a4; /* items */
	T0* a5; /* counts */
	T0* a6; /* successors */
};

/* Struct for type [detachable] ET_RESULT */
struct S309 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T2 a2; /* code */
	T0* a3; /* text */
	T6 a4; /* compressed_position */
};

/* Struct for type [detachable] ET_OBJECT_TEST */
struct S311 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* attached_keyword */
	T0* a3; /* declared_type */
	T0* a4; /* expression */
};

/* Struct for type [detachable] ET_OBJECT_EQUALITY_EXPRESSION */
struct S312 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left */
	T0* a3; /* operator */
	T0* a4; /* right */
};

/* Struct for type [detachable] ET_EQUALITY_EXPRESSION */
struct S313 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left */
	T0* a3; /* operator */
	T0* a4; /* right */
};

/* Struct for type detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL */
struct S314 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_feature */
	T0* a2; /* static_call */
	T0* a3; /* target_type_set */
	T0* a4; /* result_type_set */
	T0* a5; /* current_type */
	T0* a6; /* next */
	T1 a7; /* is_generated */
	T1 a8; /* force_result_boxing */
};

/* Struct for type detachable ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
struct S318 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target_type_set */
	T1 a2; /* is_generated */
	T0* a3; /* next */
	T0* a4; /* static_call */
	T0* a5; /* current_feature */
	T0* a6; /* current_type */
};

/* Struct for type [detachable] ET_DYNAMIC_SPECIAL_TYPE */
struct S319 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T0* a5; /* attached_type */
	T6 a6; /* attribute_count */
	T0* a7; /* queries */
	T0* a8; /* meta_type */
	T0* a9; /* conforming_ancestors */
	T0* a10; /* queries_by_seed */
	T0* a11; /* procedures_by_seed */
	T0* a12; /* procedures */
	T0* a13; /* item_type_set */
	T1 a14; /* has_redefined_copy_routine */
	T1 a15; /* has_redefined_is_equal_routine */
	T1 a16; /* has_once_per_object_routines */
	T1 a17; /* has_reference_attributes */
	T6 a18; /* hash_code */
	T0* a19; /* next_type */
	T0* a20; /* query_calls */
	T1 a21; /* has_static */
	T0* a22; /* procedure_calls */
	T0* a23; /* equality_expressions */
	T0* a24; /* object_equality_expressions */
};

/* Struct for type [detachable] DS_QUICK_SORTER [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S322 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type detachable ET_DYNAMIC_SECONDARY_TYPE */
struct S323 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_type */
	T0* a2; /* meta_type */
	T0* a3; /* type_mark */
	T0* a4; /* internal_base_type */
};

/* Struct for type [detachable] ET_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID */
struct S324 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_CREATE_EXPRESSION */
struct S328 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* creation_call */
	T0* a3; /* creation_type */
	T0* a4; /* create_keyword */
	T0* a5; /* creation_region */
};

/* Struct for type [detachable] ET_QUALIFIED_CALL */
struct S329 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* arguments */
	T0* a2; /* qualified_name */
};

/* Struct for type [detachable] KL_DIRECTORY */
struct S333 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_input */
	T0* a2; /* last_entry */
	T6 a3; /* mode */
	T0* a4; /* name */
	T0* a5; /* string_name */
	T0* a6; /* entry_buffer */
	T1 a7; /* old_end_of_input */
	T14 a8; /* directory_pointer */
	T14 a9; /* last_entry_pointer */
	T0* a10; /* lastentry */
	T0* a11; /* internal_name */
	T0* a12; /* internal_detachable_name_pointer */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S334 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] ET_SYMBOL */
struct S341 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* break */
};

/* Struct for type [detachable] DS_QUICK_SORTER [[attached] INTEGER_32] */
struct S344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE] */
struct S346 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_IDENTIFIER, [attached] ET_CURRENT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S347 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_IDENTIFIER, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S348 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S349 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S350 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T6 z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] UT_INTEGER_FORMATTER */
struct S351 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_COMPOUND */
struct S355 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* keyword */
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S357 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T6 z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S358 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] ET_EXTENDED_ATTRIBUTE */
struct S359 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T0* a8; /* preconditions */
	T6 a9; /* hash_code */
	T0* a10; /* postconditions */
	T0* a11; /* assigner */
	T0* a12; /* locals */
	T0* a13; /* rescue_clause */
	T0* a14; /* compound */
	T0* a15; /* extended_name */
	T0* a16; /* declared_type */
	T0* a17; /* frozen_keyword */
	T0* a18; /* obsolete_message */
	T0* a19; /* clients */
	T0* a20; /* end_keyword */
	T0* a21; /* semicolon */
	T0* a22; /* feature_clause */
	T0* a23; /* first_indexing */
	T0* a24; /* object_tests */
	T1 a25; /* validity_checked */
	T1 a26; /* has_validity_error */
	T6 a27; /* id */
	T6 a28; /* version */
	T0* a29; /* synonym */
};

/* Struct for type [detachable] ET_ATTRIBUTE */
struct S360 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T6 a8; /* hash_code */
	T0* a9; /* assigner */
	T0* a10; /* extended_name */
	T0* a11; /* declared_type */
	T0* a12; /* frozen_keyword */
	T0* a13; /* clients */
	T0* a14; /* semicolon */
	T0* a15; /* feature_clause */
	T0* a16; /* first_indexing */
	T0* a17; /* object_tests */
	T1 a18; /* validity_checked */
	T1 a19; /* has_validity_error */
	T6 a20; /* id */
	T6 a21; /* version */
	T0* a22; /* synonym */
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S361 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T1 z4;
};

/* Struct for type detachable ET_ITERATION_COMPONENT_LIST */
struct S365 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S367 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, INTEGER_32, INTEGER_32] */
struct S368 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
	T6 z4;
};

/* Struct for type [detachable] ET_CONSTANT_ATTRIBUTE */
struct S369 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* constant */
	T0* a5; /* implementation_feature */
	T0* a6; /* other_seeds */
	T0* a7; /* first_precursor */
	T0* a8; /* other_precursors */
	T6 a9; /* hash_code */
	T0* a10; /* assigner */
	T0* a11; /* extended_name */
	T0* a12; /* declared_type */
	T0* a13; /* frozen_keyword */
	T0* a14; /* clients */
	T0* a15; /* is_keyword */
	T0* a16; /* semicolon */
	T0* a17; /* feature_clause */
	T0* a18; /* first_indexing */
	T0* a19; /* object_tests */
	T1 a20; /* validity_checked */
	T1 a21; /* has_validity_error */
	T6 a22; /* id */
	T6 a23; /* version */
	T0* a24; /* synonym */
};

/* Struct for type [detachable] ET_UNIQUE_ATTRIBUTE */
struct S371 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constant */
	T6 a2; /* id */
	T6 a3; /* first_seed */
	T0* a4; /* iteration_components */
	T0* a5; /* implementation_class */
	T0* a6; /* implementation_feature */
	T0* a7; /* other_seeds */
	T0* a8; /* first_precursor */
	T0* a9; /* other_precursors */
	T6 a10; /* hash_code */
	T0* a11; /* assigner */
	T0* a12; /* extended_name */
	T0* a13; /* declared_type */
	T0* a14; /* frozen_keyword */
	T0* a15; /* clients */
	T0* a16; /* is_keyword */
	T0* a17; /* unique_keyword */
	T0* a18; /* semicolon */
	T0* a19; /* feature_clause */
	T0* a20; /* first_indexing */
	T0* a21; /* object_tests */
	T1 a22; /* validity_checked */
	T1 a23; /* has_validity_error */
	T6 a24; /* version */
	T0* a25; /* synonym */
};

/* Struct for type [detachable] ET_REGULAR_INTEGER_CONSTANT */
struct S372 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T11 a1; /* value */
	T0* a2; /* sign */
	T0* a3; /* literal */
	T1 a4; /* has_overflow */
	T6 a5; /* compressed_position */
	T6 a6; /* index */
	T0* a7; /* cast_type */
	T0* a8; /* type */
};

/* Struct for type [detachable] ET_SYSTEM_MULTIPROCESSOR */
struct S375 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* interface_checker */
	T0* a2; /* stop_request */
	T0* a3; /* eiffel_parser */
	T0* a4; /* error_handler */
	T0* a5; /* implementation_checker */
	T1 a6; /* benchmark_shown */
	T0* a7; /* ise_version */
	T0* a8; /* other_processors */
	T0* a9; /* eiffel_preparser */
	T0* a10; /* master_class_checker */
	T0* a11; /* provider_checker */
	T0* a12; /* ancestor_builder */
	T0* a13; /* feature_flattener */
	T0* a14; /* ecma_version */
	T1 a15; /* nested_benchmark_shown */
	T1 a16; /* metrics_shown */
	T1 a17; /* flat_mode */
	T1 a18; /* flat_dbc_mode */
	T0* a19; /* processed_class_count_stack */
	T0* a20; /* ast_factory */
	T0* a21; /* dotnet_assembly_consumer */
	T1 a22; /* use_attribute_keyword */
	T1 a23; /* use_note_keyword */
	T1 a24; /* use_reference_keyword */
	T1 a25; /* use_attached_keyword */
	T1 a26; /* use_detachable_keyword */
	T1 a27; /* preparse_shallow_mode */
	T1 a28; /* preparse_single_mode */
	T1 a29; /* preparse_multiple_mode */
	T1 a30; /* unknown_builtin_reported */
	T1 a31; /* qualified_anchored_types_cycle_detection_enabled */
	T1 a32; /* cluster_dependence_enabled */
	T6 a33; /* postponed_class_count */
	T6 a34; /* processed_class_count */
	T1 a35; /* suppliers_enabled */
	T1 a36; /* preparse_override_mode */
	T1 a37; /* preparse_readonly_mode */
	T1 a38; /* providers_enabled */
	T1 a39; /* use_cluster_dependence_pathnames */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8] */
struct S376 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
struct S377 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
struct S379 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_FEATURE_CHECKER */
struct S380 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_checker */
	T0* a2; /* current_type */
	T0* a3; /* current_class */
	T0* a4; /* current_feature */
	T0* a5; /* current_feature_impl */
	T0* a6; /* current_class_impl */
	T0* a7; /* enclosing_inline_agents */
	T0* a8; /* unused_overloaded_procedures_list */
	T0* a9; /* unused_overloaded_queries_list */
	T0* a10; /* unused_overloaded_features_list */
	T0* a11; /* unused_call_infos */
	T0* a12; /* unused_contexts */
	T0* a13; /* current_context */
	T0* a14; /* current_target_type */
	T0* a15; /* current_object_test_types */
	T0* a16; /* current_object_test_scope */
	T0* a17; /* object_test_scope_builder */
	T0* a18; /* current_iteration_cursor_types */
	T0* a19; /* current_iteration_item_types */
	T0* a20; /* current_iteration_item_scope */
	T0* a21; /* current_initialization_scope */
	T0* a22; /* current_attachment_scope */
	T0* a23; /* attachment_scope_builder */
	T0* a24; /* unused_attachment_scopes */
	T0* a25; /* assertions_by_feature */
	T0* a26; /* common_ancestor_type_list */
	T0* a27; /* indexing_term_list */
	T0* a28; /* default_creation_call_name */
	T0* a29; /* default_creation_call */
	T0* a30; /* vape_non_descendant_clients */
	T0* a31; /* vape_creation_clients */
	T0* a32; /* vape_client */
	T0* a33; /* unused_adapted_base_classes */
	T0* a34; /* adapted_base_class_checker */
	T0* a35; /* system_processor */
	T1 a36; /* has_fatal_error */
	T1 a37; /* in_static_feature */
	T1 a38; /* in_precondition */
	T0* a39; /* supplier_handler */
	T1 a40; /* in_invariant */
	T0* a41; /* precursor_queries */
	T0* a42; /* precursor_procedures */
	T1 a43; /* in_postcondition */
	T1 a44; /* in_precursor */
	T1 a45; /* in_rescue */
	T0* a46; /* current_inline_agent */
	T1 a47; /* in_check_instruction */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
struct S381 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T6 a3; /* free_slot */
	T6 a4; /* position */
	T0* a5; /* equality_tester */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_cursor */
	T6 a8; /* found_position */
	T0* a9; /* item_storage */
	T0* a10; /* key_storage */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type detachable DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
struct S382 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* count */
	T6 a8; /* modulus */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T0* a14; /* special_item_routines */
	T0* a15; /* special_key_routines */
	T0* a16; /* key_storage */
	T0* a17; /* equality_tester */
	T6 a18; /* free_slot */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] ET_TYPE_CHECKER */
struct S384 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_context */
	T0* a2; /* current_class_impl */
	T0* a3; /* current_feature_impl */
	T0* a4; /* constraint_context */
	T0* a5; /* current_class */
	T0* a6; /* target_context */
	T0* a7; /* other_context */
	T0* a8; /* adapted_base_class_checker */
	T0* a9; /* adapted_base_classes */
	T0* a10; /* system_processor */
	T0* a11; /* supplier_handler */
	T1 a12; /* has_fatal_error */
	T1 a13; /* class_interface_error_only */
	T1 a14; /* in_qualified_anchored_type */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_INLINE_AGENT] */
struct S386 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S387 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S388 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S389 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
struct S390 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S391 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
struct S392 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T0* a7; /* equality_tester */
	T0* a8; /* key_equality_tester */
	T6 a9; /* found_position */
	T0* a10; /* key_storage */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] ET_OBJECT_TEST_SCOPE */
struct S393 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* object_tests */
	T6 a2; /* hidden_count */
};

/* Struct for type [detachable] ET_OBJECT_TEST_SCOPE_BUILDER */
struct S394 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T0* a4; /* scope */
	T1 a5; /* is_negated */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ITERATION_COMPONENT] */
struct S395 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T0* a3; /* key_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* equality_tester */
	T0* a9; /* key_equality_tester */
	T6 a10; /* found_position */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* internal_keys */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] ET_ITERATION_ITEM_SCOPE */
struct S396 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* hidden_count */
	T0* a2; /* iteration_components */
};

/* Struct for type [detachable] ET_ATTACHMENT_SCOPE */
struct S397 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* locals_attached */
	T0* a2; /* arguments_attached */
	T0* a3; /* attributes_attached */
	T1 a4; /* result_attached */
	T1 a5; /* is_code_unreachable */
};

/* Struct for type [detachable] ET_ATTACHMENT_SCOPE_BUILDER */
struct S398 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* scope */
	T1 a2; /* is_negated */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ATTACHMENT_SCOPE] */
struct S399 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
struct S400 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* clashes */
	T0* a4; /* item_storage */
	T6 a5; /* last_position */
	T6 a6; /* position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* free_slot */
	T0* a10; /* equality_tester */
	T0* a11; /* key_equality_tester */
	T6 a12; /* found_position */
	T6 a13; /* modulus */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* special_item_routines */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_INDEXING_TERM] */
struct S401 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] ET_CLIENT_LIST */
struct S402 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] ET_CLIENT */
struct S403 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* named_base_class */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
struct S404 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] ET_ADAPTED_BASE_CLASS_CHECKER */
struct S405 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class_impl */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T1 a4; /* feature_flattening_error_only */
	T1 a5; /* has_fatal_error */
	T1 a6; /* class_interface_error_only */
};

/* Struct for type detachable ET_PRECONDITIONS */
struct S406 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* validity_checked */
	T1 a2; /* has_validity_error */
	T6 a3; /* count */
	T0* a4; /* storage */
	T0* a5; /* require_keyword */
	T0* a6; /* else_keyword */
};

/* Struct for type detachable ET_FEATURE_LIST */
struct S408 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T6 a2; /* count */
};

/* Struct for type [detachable] ET_UNKNOWN_GROUP */
struct S410 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T1 a2; /* use_obsolete_syntax */
	T0* a3; /* cached_absolute_pathname */
};

/* Struct for type [detachable] ET_BASE_TYPE_LIST */
struct S412 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] ET_KEYWORD */
struct S413 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T0* a2; /* text */
	T6 a3; /* compressed_position */
};

/* Struct for type [detachable] ET_CLASS_CODES */
struct S414 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S415 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S416 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S417 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] RX_BYTE_CODE */
struct S422 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* byte_code */
	T0* a3; /* character_sets */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] RX_CHARACTER_SET */
struct S423 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_empty */
	T1 a2; /* is_negated */
	T11 a3; /* first_set */
	T11 a4; /* second_set */
	T11 a5; /* third_set */
	T11 a6; /* fourth_set */
	T0* a7; /* other_sets */
	T1 a8; /* is_reverted */
};

/* Struct for type [detachable] RX_CASE_MAPPING */
struct S424 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* lower_table */
	T0* a2; /* flip_table */
};

/* Struct for type [detachable] UC_UNICODE_ROUTINES */
struct S426 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ARRAY [[attached] RX_CHARACTER_SET] */
struct S427 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] RX_CHARACTER_SET] */
struct S428 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] UC_STRING */
struct S429 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* area */
	T6 a3; /* byte_count */
	T6 a4; /* internal_hash_code */
	T6 a5; /* last_byte_index_input */
	T6 a6; /* last_byte_index_result */
	T6 a7; /* internal_case_insensitive_hash_code */
};

/* Struct for type [detachable] NATIVE_STRING */
struct S430 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* managed_data */
	T6 a2; /* unit_count */
};

/* Struct for type [detachable] TUPLE [[attached] GECC, [attached] DS_ARRAYED_LIST [[attached] STRING_8], INTEGER_32, [attached] MUTEX] */
struct S431 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
	T0* z4;
};

/* Struct for type [detachable] WORKER_THREAD */
struct S432 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* thread_procedure */
	T0* a2; /* launch_mutex */
	T14 a3; /* thread_id */
	T1 a4; /* terminated */
};

/* Struct for type [detachable] UC_UTF8_ROUTINES */
struct S434 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] PATH */
struct S435 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_name */
	T0* a2; /* storage */
	T1 a3; /* is_normalized */
};

/* Struct for type [detachable] STRING_TO_INTEGER_CONVERTOR */
struct S436 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* sign */
	T11 a2; /* part1 */
	T11 a3; /* part2 */
	T6 a4; /* last_state */
	T1 a5; /* internal_overflowed */
	T0* a6; /* leading_separators */
	T0* a7; /* trailing_separators */
	T1 a8; /* leading_separators_acceptable */
	T1 a9; /* trailing_separators_acceptable */
	T6 a10; /* conversion_type */
};

/* Struct for type [detachable] ET_ECF_OPTIONS */
struct S444 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_assertions */
	T0* a2; /* primary_debugs */
	T0* a3; /* primary_warnings */
	T0* a4; /* primary_options */
	T0* a5; /* description */
	T0* a6; /* secondary_options */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] ET_ECF_NOTE_ELEMENT] */
struct S445 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] XM_NAMESPACE */
struct S446 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* uri */
	T0* a2; /* ns_prefix */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XM_ELEMENT_NODE] */
struct S447 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type [detachable] ET_LIKE_CURRENT */
struct S449 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* like_keyword */
	T0* a3; /* current_keyword */
};

/* Struct for type [detachable] ET_DYNAMIC_PROCEDURE_TYPE */
struct S450 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T0* a5; /* attached_type */
	T6 a6; /* attribute_count */
	T0* a7; /* queries */
	T0* a8; /* meta_type */
	T0* a9; /* conforming_ancestors */
	T1 a10; /* has_redefined_is_equal_routine */
	T1 a11; /* has_redefined_copy_routine */
	T0* a12; /* next_type */
	T1 a13; /* has_once_per_object_routines */
	T0* a14; /* procedures */
	T1 a15; /* has_static */
	T0* a16; /* query_calls */
	T0* a17; /* procedure_calls */
	T0* a18; /* open_operand_type_sets */
	T6 a19; /* hash_code */
	T0* a20; /* queries_by_seed */
	T1 a21; /* has_reference_attributes */
	T0* a22; /* open_operand_tuple_type_set */
	T0* a23; /* set_rout_disp_final_feature */
	T0* a24; /* procedures_by_seed */
	T0* a25; /* equality_expressions */
	T0* a26; /* object_equality_expressions */
};

/* Struct for type [detachable] ET_DYNAMIC_FUNCTION_TYPE */
struct S451 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T0* a5; /* attached_type */
	T6 a6; /* attribute_count */
	T0* a7; /* queries */
	T0* a8; /* meta_type */
	T0* a9; /* conforming_ancestors */
	T1 a10; /* has_redefined_is_equal_routine */
	T1 a11; /* has_redefined_copy_routine */
	T0* a12; /* next_type */
	T1 a13; /* has_once_per_object_routines */
	T0* a14; /* procedures */
	T1 a15; /* has_static */
	T0* a16; /* query_calls */
	T0* a17; /* procedure_calls */
	T0* a18; /* open_operand_type_sets */
	T0* a19; /* result_type_set */
	T6 a20; /* hash_code */
	T0* a21; /* queries_by_seed */
	T1 a22; /* has_reference_attributes */
	T0* a23; /* procedures_by_seed */
	T0* a24; /* open_operand_tuple_type_set */
	T0* a25; /* set_rout_disp_final_feature */
	T0* a26; /* equality_expressions */
	T0* a27; /* object_equality_expressions */
};

/* Struct for type [detachable] DT_SHARED_SYSTEM_CLOCK */
struct S452 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DT_SYSTEM_CLOCK */
struct S453 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* second */
	T6 a2; /* year */
	T6 a3; /* month */
	T6 a4; /* day */
	T6 a5; /* hour */
	T6 a6; /* minute */
	T6 a7; /* millisecond */
	T0* a8; /* local_clock */
};

/* Struct for type [detachable] ET_EIFFEL_PREPARSER */
struct S454 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* eiffel_buffer */
	T0* a2; /* system_processor */
	T6 a3; /* last_text_count */
	T6 a4; /* last_literal_start */
	T0* a5; /* filename */
	T0* a6; /* group */
	T0* a7; /* verbatim_marker */
	T6 a8; /* verbatim_marker_count */
	T0* a9; /* verbatim_open_white_characters */
	T0* a10; /* verbatim_close_white_characters */
	T0* a11; /* yy_pushed_start_conditions */
	T0* a12; /* input_buffer */
	T0* a13; /* yy_state_stack */
	T6 a14; /* yy_start_state */
	T6 a15; /* yy_line */
	T6 a16; /* yy_column */
	T6 a17; /* yy_position */
	T6 a18; /* yy_start */
	T6 a19; /* yy_end */
	T6 a20; /* line */
	T6 a21; /* column */
	T6 a22; /* position */
	T0* a23; /* yy_nxt */
	T0* a24; /* yy_chk */
	T0* a25; /* yy_base */
	T0* a26; /* yy_def */
	T0* a27; /* yy_ec */
	T0* a28; /* yy_meta */
	T0* a29; /* yy_accept */
	T0* a30; /* yy_acclist */
	T0* a31; /* yy_content */
	T0* a32; /* yy_content_area */
	T0* a33; /* yy_unicode_content_area */
	T0* a34; /* last_classname */
	T1 a35; /* class_keyword_found */
	T6 a36; /* last_token */
	T1 a37; /* yy_more_flag */
	T6 a38; /* yy_more_len */
	T6 a39; /* yy_state_count */
	T6 a40; /* yy_last_accepting_state */
	T6 a41; /* yy_last_accepting_cpos */
	T6 a42; /* yy_lp */
	T6 a43; /* yy_looking_for_trail_begin */
	T6 a44; /* yy_full_match */
	T6 a45; /* yy_full_state */
	T6 a46; /* yy_full_lp */
	T1 a47; /* yy_rejected */
	T6 a48; /* last_literal_end */
	T6 a49; /* pushed_start_condition_count */
	T6 a50; /* ms_column */
	T6 a51; /* ms_line */
	T6 a52; /* last_break_end */
	T6 a53; /* last_comment_end */
	T3 a54; /* verbatim_opening_character */
};

/* Struct for type [detachable] ET_MASTER_CLASS_CHECKER */
struct S455 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
	T1 a3; /* has_fatal_error */
};

/* Struct for type [detachable] ET_EIFFEL_PARSER */
struct S456 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* eiffel_buffer */
	T0* a2; /* counters */
	T0* a3; /* last_formal_arguments_stack */
	T0* a4; /* last_local_variables_stack */
	T0* a5; /* last_keywords */
	T0* a6; /* last_symbols */
	T0* a7; /* last_object_tests_stack */
	T0* a8; /* last_object_tests_pool */
	T0* a9; /* last_iteration_components_stack */
	T0* a10; /* last_iteration_components_pool */
	T0* a11; /* assertions */
	T0* a12; /* assertion_counters */
	T0* a13; /* assertion_kinds */
	T6 a14; /* assertion_kind */
	T0* a15; /* queries */
	T0* a16; /* procedures */
	T0* a17; /* constraints */
	T0* a18; /* providers */
	T0* a19; /* current_class */
	T0* a20; /* system_processor */
	T6 a21; /* last_text_count */
	T6 a22; /* last_literal_start */
	T0* a23; /* filename */
	T0* a24; /* group */
	T0* a25; /* verbatim_marker */
	T6 a26; /* verbatim_marker_count */
	T0* a27; /* verbatim_open_white_characters */
	T0* a28; /* verbatim_close_white_characters */
	T0* a29; /* yyss */
	T0* a30; /* yy_pushed_start_conditions */
	T0* a31; /* input_buffer */
	T0* a32; /* yyspecial_routines1 */
	T6 a33; /* yyvsc1 */
	T0* a34; /* yyvs1 */
	T0* a35; /* yyspecial_routines2 */
	T6 a36; /* yyvsc2 */
	T0* a37; /* yyvs2 */
	T0* a38; /* yyspecial_routines3 */
	T6 a39; /* yyvsc3 */
	T0* a40; /* yyvs3 */
	T0* a41; /* yyspecial_routines4 */
	T6 a42; /* yyvsc4 */
	T0* a43; /* yyvs4 */
	T0* a44; /* yyspecial_routines5 */
	T6 a45; /* yyvsc5 */
	T0* a46; /* yyvs5 */
	T0* a47; /* yyspecial_routines6 */
	T6 a48; /* yyvsc6 */
	T0* a49; /* yyvs6 */
	T0* a50; /* yyspecial_routines7 */
	T6 a51; /* yyvsc7 */
	T0* a52; /* yyvs7 */
	T0* a53; /* yyspecial_routines8 */
	T6 a54; /* yyvsc8 */
	T0* a55; /* yyvs8 */
	T0* a56; /* yyspecial_routines9 */
	T6 a57; /* yyvsc9 */
	T0* a58; /* yyvs9 */
	T0* a59; /* yyspecial_routines10 */
	T6 a60; /* yyvsc10 */
	T0* a61; /* yyvs10 */
	T0* a62; /* yyspecial_routines11 */
	T6 a63; /* yyvsc11 */
	T0* a64; /* yyvs11 */
	T0* a65; /* yyspecial_routines12 */
	T6 a66; /* yyvsc12 */
	T0* a67; /* yyvs12 */
	T0* a68; /* yyspecial_routines13 */
	T6 a69; /* yyvsc13 */
	T0* a70; /* yyvs13 */
	T0* a71; /* yyspecial_routines14 */
	T6 a72; /* yyvsc14 */
	T0* a73; /* yyvs14 */
	T0* a74; /* yyspecial_routines15 */
	T6 a75; /* yyvsc15 */
	T0* a76; /* yyvs15 */
	T0* a77; /* yyspecial_routines16 */
	T6 a78; /* yyvsc16 */
	T0* a79; /* yyvs16 */
	T0* a80; /* yyspecial_routines17 */
	T6 a81; /* yyvsc17 */
	T0* a82; /* yyvs17 */
	T0* a83; /* yyspecial_routines18 */
	T6 a84; /* yyvsc18 */
	T0* a85; /* yyvs18 */
	T0* a86; /* yyspecial_routines19 */
	T6 a87; /* yyvsc19 */
	T0* a88; /* yyvs19 */
	T0* a89; /* yyspecial_routines20 */
	T6 a90; /* yyvsc20 */
	T0* a91; /* yyvs20 */
	T0* a92; /* yyspecial_routines21 */
	T6 a93; /* yyvsc21 */
	T0* a94; /* yyvs21 */
	T0* a95; /* yyspecial_routines22 */
	T6 a96; /* yyvsc22 */
	T0* a97; /* yyvs22 */
	T0* a98; /* yyspecial_routines23 */
	T6 a99; /* yyvsc23 */
	T0* a100; /* yyvs23 */
	T0* a101; /* yyspecial_routines24 */
	T6 a102; /* yyvsc24 */
	T0* a103; /* yyvs24 */
	T0* a104; /* yyspecial_routines25 */
	T6 a105; /* yyvsc25 */
	T0* a106; /* yyvs25 */
	T0* a107; /* yyspecial_routines26 */
	T6 a108; /* yyvsc26 */
	T0* a109; /* yyvs26 */
	T0* a110; /* yyspecial_routines27 */
	T6 a111; /* yyvsc27 */
	T0* a112; /* yyvs27 */
	T0* a113; /* yyspecial_routines28 */
	T6 a114; /* yyvsc28 */
	T0* a115; /* yyvs28 */
	T0* a116; /* yyspecial_routines29 */
	T6 a117; /* yyvsc29 */
	T0* a118; /* yyvs29 */
	T0* a119; /* yyspecial_routines30 */
	T6 a120; /* yyvsc30 */
	T0* a121; /* yyvs30 */
	T0* a122; /* yyspecial_routines31 */
	T6 a123; /* yyvsc31 */
	T0* a124; /* yyvs31 */
	T0* a125; /* yyspecial_routines32 */
	T6 a126; /* yyvsc32 */
	T0* a127; /* yyvs32 */
	T0* a128; /* yyspecial_routines33 */
	T6 a129; /* yyvsc33 */
	T0* a130; /* yyvs33 */
	T0* a131; /* yyspecial_routines34 */
	T6 a132; /* yyvsc34 */
	T0* a133; /* yyvs34 */
	T0* a134; /* yyspecial_routines35 */
	T6 a135; /* yyvsc35 */
	T0* a136; /* yyvs35 */
	T0* a137; /* yyspecial_routines36 */
	T6 a138; /* yyvsc36 */
	T0* a139; /* yyvs36 */
	T0* a140; /* yyspecial_routines37 */
	T6 a141; /* yyvsc37 */
	T0* a142; /* yyvs37 */
	T0* a143; /* yyspecial_routines38 */
	T6 a144; /* yyvsc38 */
	T0* a145; /* yyvs38 */
	T0* a146; /* yyspecial_routines39 */
	T6 a147; /* yyvsc39 */
	T0* a148; /* yyvs39 */
	T0* a149; /* yyspecial_routines40 */
	T6 a150; /* yyvsc40 */
	T0* a151; /* yyvs40 */
	T0* a152; /* yyspecial_routines41 */
	T6 a153; /* yyvsc41 */
	T0* a154; /* yyvs41 */
	T0* a155; /* yyspecial_routines42 */
	T6 a156; /* yyvsc42 */
	T0* a157; /* yyvs42 */
	T0* a158; /* yyspecial_routines43 */
	T6 a159; /* yyvsc43 */
	T0* a160; /* yyvs43 */
	T0* a161; /* yyspecial_routines44 */
	T6 a162; /* yyvsc44 */
	T0* a163; /* yyvs44 */
	T0* a164; /* yyspecial_routines45 */
	T6 a165; /* yyvsc45 */
	T0* a166; /* yyvs45 */
	T0* a167; /* yyspecial_routines46 */
	T6 a168; /* yyvsc46 */
	T0* a169; /* yyvs46 */
	T0* a170; /* yyspecial_routines47 */
	T6 a171; /* yyvsc47 */
	T0* a172; /* yyvs47 */
	T0* a173; /* yyspecial_routines48 */
	T6 a174; /* yyvsc48 */
	T0* a175; /* yyvs48 */
	T0* a176; /* yyspecial_routines49 */
	T6 a177; /* yyvsc49 */
	T0* a178; /* yyvs49 */
	T0* a179; /* yyspecial_routines50 */
	T6 a180; /* yyvsc50 */
	T0* a181; /* yyvs50 */
	T0* a182; /* yyspecial_routines51 */
	T6 a183; /* yyvsc51 */
	T0* a184; /* yyvs51 */
	T0* a185; /* yyspecial_routines52 */
	T6 a186; /* yyvsc52 */
	T0* a187; /* yyvs52 */
	T0* a188; /* yyspecial_routines53 */
	T6 a189; /* yyvsc53 */
	T0* a190; /* yyvs53 */
	T0* a191; /* yyspecial_routines54 */
	T6 a192; /* yyvsc54 */
	T0* a193; /* yyvs54 */
	T0* a194; /* yyspecial_routines55 */
	T6 a195; /* yyvsc55 */
	T0* a196; /* yyvs55 */
	T0* a197; /* yyspecial_routines56 */
	T6 a198; /* yyvsc56 */
	T0* a199; /* yyvs56 */
	T0* a200; /* yyspecial_routines57 */
	T6 a201; /* yyvsc57 */
	T0* a202; /* yyvs57 */
	T0* a203; /* yyspecial_routines58 */
	T6 a204; /* yyvsc58 */
	T0* a205; /* yyvs58 */
	T0* a206; /* yyspecial_routines59 */
	T6 a207; /* yyvsc59 */
	T0* a208; /* yyvs59 */
	T0* a209; /* yyspecial_routines60 */
	T6 a210; /* yyvsc60 */
	T0* a211; /* yyvs60 */
	T0* a212; /* yyspecial_routines61 */
	T6 a213; /* yyvsc61 */
	T0* a214; /* yyvs61 */
	T0* a215; /* yyspecial_routines62 */
	T6 a216; /* yyvsc62 */
	T0* a217; /* yyvs62 */
	T0* a218; /* yyspecial_routines63 */
	T6 a219; /* yyvsc63 */
	T0* a220; /* yyvs63 */
	T0* a221; /* yyspecial_routines64 */
	T6 a222; /* yyvsc64 */
	T0* a223; /* yyvs64 */
	T0* a224; /* yyspecial_routines65 */
	T6 a225; /* yyvsc65 */
	T0* a226; /* yyvs65 */
	T0* a227; /* yyspecial_routines66 */
	T6 a228; /* yyvsc66 */
	T0* a229; /* yyvs66 */
	T0* a230; /* yyspecial_routines67 */
	T6 a231; /* yyvsc67 */
	T0* a232; /* yyvs67 */
	T0* a233; /* yyspecial_routines68 */
	T6 a234; /* yyvsc68 */
	T0* a235; /* yyvs68 */
	T0* a236; /* yyspecial_routines69 */
	T6 a237; /* yyvsc69 */
	T0* a238; /* yyvs69 */
	T0* a239; /* yyspecial_routines70 */
	T6 a240; /* yyvsc70 */
	T0* a241; /* yyvs70 */
	T0* a242; /* yyspecial_routines71 */
	T6 a243; /* yyvsc71 */
	T0* a244; /* yyvs71 */
	T0* a245; /* yyspecial_routines72 */
	T6 a246; /* yyvsc72 */
	T0* a247; /* yyvs72 */
	T0* a248; /* yyspecial_routines73 */
	T6 a249; /* yyvsc73 */
	T0* a250; /* yyvs73 */
	T0* a251; /* yyspecial_routines74 */
	T6 a252; /* yyvsc74 */
	T0* a253; /* yyvs74 */
	T0* a254; /* yyspecial_routines75 */
	T6 a255; /* yyvsc75 */
	T0* a256; /* yyvs75 */
	T0* a257; /* yyspecial_routines76 */
	T6 a258; /* yyvsc76 */
	T0* a259; /* yyvs76 */
	T0* a260; /* yyspecial_routines77 */
	T6 a261; /* yyvsc77 */
	T0* a262; /* yyvs77 */
	T0* a263; /* yyspecial_routines78 */
	T6 a264; /* yyvsc78 */
	T0* a265; /* yyvs78 */
	T0* a266; /* yyspecial_routines79 */
	T6 a267; /* yyvsc79 */
	T0* a268; /* yyvs79 */
	T0* a269; /* yyspecial_routines80 */
	T6 a270; /* yyvsc80 */
	T0* a271; /* yyvs80 */
	T0* a272; /* yyspecial_routines81 */
	T6 a273; /* yyvsc81 */
	T0* a274; /* yyvs81 */
	T0* a275; /* yyspecial_routines82 */
	T6 a276; /* yyvsc82 */
	T0* a277; /* yyvs82 */
	T0* a278; /* yyspecial_routines83 */
	T6 a279; /* yyvsc83 */
	T0* a280; /* yyvs83 */
	T0* a281; /* yyspecial_routines84 */
	T6 a282; /* yyvsc84 */
	T0* a283; /* yyvs84 */
	T0* a284; /* yyspecial_routines85 */
	T6 a285; /* yyvsc85 */
	T0* a286; /* yyvs85 */
	T0* a287; /* yyspecial_routines86 */
	T6 a288; /* yyvsc86 */
	T0* a289; /* yyvs86 */
	T0* a290; /* yyspecial_routines87 */
	T6 a291; /* yyvsc87 */
	T0* a292; /* yyvs87 */
	T0* a293; /* yyspecial_routines88 */
	T6 a294; /* yyvsc88 */
	T0* a295; /* yyvs88 */
	T0* a296; /* yyspecial_routines89 */
	T6 a297; /* yyvsc89 */
	T0* a298; /* yyvs89 */
	T0* a299; /* yyspecial_routines90 */
	T6 a300; /* yyvsc90 */
	T0* a301; /* yyvs90 */
	T0* a302; /* yyspecial_routines91 */
	T6 a303; /* yyvsc91 */
	T0* a304; /* yyvs91 */
	T0* a305; /* yyspecial_routines92 */
	T6 a306; /* yyvsc92 */
	T0* a307; /* yyvs92 */
	T0* a308; /* yyspecial_routines93 */
	T6 a309; /* yyvsc93 */
	T0* a310; /* yyvs93 */
	T0* a311; /* yyspecial_routines94 */
	T6 a312; /* yyvsc94 */
	T0* a313; /* yyvs94 */
	T0* a314; /* yyspecial_routines95 */
	T6 a315; /* yyvsc95 */
	T0* a316; /* yyvs95 */
	T0* a317; /* yyspecial_routines96 */
	T6 a318; /* yyvsc96 */
	T0* a319; /* yyvs96 */
	T0* a320; /* yyspecial_routines97 */
	T6 a321; /* yyvsc97 */
	T0* a322; /* yyvs97 */
	T0* a323; /* yyspecial_routines98 */
	T6 a324; /* yyvsc98 */
	T0* a325; /* yyvs98 */
	T0* a326; /* yyspecial_routines99 */
	T6 a327; /* yyvsc99 */
	T0* a328; /* yyvs99 */
	T0* a329; /* yyspecial_routines100 */
	T6 a330; /* yyvsc100 */
	T0* a331; /* yyvs100 */
	T0* a332; /* yyspecial_routines101 */
	T6 a333; /* yyvsc101 */
	T0* a334; /* yyvs101 */
	T0* a335; /* yyspecial_routines102 */
	T6 a336; /* yyvsc102 */
	T0* a337; /* yyvs102 */
	T0* a338; /* yyspecial_routines103 */
	T6 a339; /* yyvsc103 */
	T0* a340; /* yyvs103 */
	T0* a341; /* yyspecial_routines104 */
	T6 a342; /* yyvsc104 */
	T0* a343; /* yyvs104 */
	T0* a344; /* yyspecial_routines105 */
	T6 a345; /* yyvsc105 */
	T0* a346; /* yyvs105 */
	T0* a347; /* yyspecial_routines106 */
	T6 a348; /* yyvsc106 */
	T0* a349; /* yyvs106 */
	T0* a350; /* yyspecial_routines107 */
	T6 a351; /* yyvsc107 */
	T0* a352; /* yyvs107 */
	T0* a353; /* yyspecial_routines108 */
	T6 a354; /* yyvsc108 */
	T0* a355; /* yyvs108 */
	T0* a356; /* yyspecial_routines109 */
	T6 a357; /* yyvsc109 */
	T0* a358; /* yyvs109 */
	T0* a359; /* yyspecial_routines110 */
	T6 a360; /* yyvsc110 */
	T0* a361; /* yyvs110 */
	T0* a362; /* yyspecial_routines111 */
	T6 a363; /* yyvsc111 */
	T0* a364; /* yyvs111 */
	T0* a365; /* yyspecial_routines112 */
	T6 a366; /* yyvsc112 */
	T0* a367; /* yyvs112 */
	T0* a368; /* yyspecial_routines113 */
	T6 a369; /* yyvsc113 */
	T0* a370; /* yyvs113 */
	T0* a371; /* yyspecial_routines114 */
	T6 a372; /* yyvsc114 */
	T0* a373; /* yyvs114 */
	T0* a374; /* yyspecial_routines115 */
	T6 a375; /* yyvsc115 */
	T0* a376; /* yyvs115 */
	T0* a377; /* yyspecial_routines116 */
	T6 a378; /* yyvsc116 */
	T0* a379; /* yyvs116 */
	T0* a380; /* yyspecial_routines117 */
	T6 a381; /* yyvsc117 */
	T0* a382; /* yyvs117 */
	T0* a383; /* yyspecial_routines118 */
	T6 a384; /* yyvsc118 */
	T0* a385; /* yyvs118 */
	T0* a386; /* yyspecial_routines119 */
	T6 a387; /* yyvsc119 */
	T0* a388; /* yyvs119 */
	T0* a389; /* yyspecial_routines120 */
	T6 a390; /* yyvsc120 */
	T0* a391; /* yyvs120 */
	T0* a392; /* yyspecial_routines121 */
	T6 a393; /* yyvsc121 */
	T0* a394; /* yyvs121 */
	T0* a395; /* yyspecial_routines122 */
	T6 a396; /* yyvsc122 */
	T0* a397; /* yyvs122 */
	T0* a398; /* yyspecial_routines123 */
	T6 a399; /* yyvsc123 */
	T0* a400; /* yyvs123 */
	T0* a401; /* yyspecial_routines124 */
	T6 a402; /* yyvsc124 */
	T0* a403; /* yyvs124 */
	T0* a404; /* yyspecial_routines125 */
	T6 a405; /* yyvsc125 */
	T0* a406; /* yyvs125 */
	T0* a407; /* yyspecial_routines126 */
	T6 a408; /* yyvsc126 */
	T0* a409; /* yyvs126 */
	T0* a410; /* yyspecial_routines127 */
	T6 a411; /* yyvsc127 */
	T0* a412; /* yyvs127 */
	T0* a413; /* yyspecial_routines128 */
	T6 a414; /* yyvsc128 */
	T0* a415; /* yyvs128 */
	T0* a416; /* yyspecial_routines129 */
	T6 a417; /* yyvsc129 */
	T0* a418; /* yyvs129 */
	T0* a419; /* yyspecial_routines130 */
	T6 a420; /* yyvsc130 */
	T0* a421; /* yyvs130 */
	T0* a422; /* yyspecial_routines131 */
	T6 a423; /* yyvsc131 */
	T0* a424; /* yyvs131 */
	T0* a425; /* yyspecial_routines132 */
	T6 a426; /* yyvsc132 */
	T0* a427; /* yyvs132 */
	T0* a428; /* yyspecial_routines133 */
	T6 a429; /* yyvsc133 */
	T0* a430; /* yyvs133 */
	T0* a431; /* yyspecial_routines134 */
	T6 a432; /* yyvsc134 */
	T0* a433; /* yyvs134 */
	T0* a434; /* yytranslate */
	T0* a435; /* yyr1 */
	T0* a436; /* yytypes1 */
	T0* a437; /* yytypes2 */
	T0* a438; /* yydefact */
	T0* a439; /* yydefgoto */
	T0* a440; /* yypact */
	T0* a441; /* yypgoto */
	T0* a442; /* yytable */
	T0* a443; /* yycheck */
	T0* a444; /* yy_state_stack */
	T6 a445; /* yy_start_state */
	T6 a446; /* yy_line */
	T6 a447; /* yy_column */
	T6 a448; /* yy_position */
	T6 a449; /* yy_start */
	T6 a450; /* yy_end */
	T6 a451; /* line */
	T6 a452; /* column */
	T6 a453; /* position */
	T0* a454; /* yy_nxt */
	T0* a455; /* yy_chk */
	T0* a456; /* yy_base */
	T0* a457; /* yy_def */
	T0* a458; /* yy_ec */
	T0* a459; /* yy_meta */
	T0* a460; /* yy_accept */
	T0* a461; /* yy_acclist */
	T0* a462; /* yy_content */
	T0* a463; /* yy_content_area */
	T0* a464; /* yy_unicode_content_area */
	T6 a465; /* time_stamp */
	T0* a466; /* last_class */
	T6 a467; /* yy_parsing_status */
	T6 a468; /* yy_suspended_yystacksize */
	T6 a469; /* yy_suspended_yystate */
	T6 a470; /* yy_suspended_yyn */
	T6 a471; /* yy_suspended_yychar1 */
	T6 a472; /* yy_suspended_index */
	T6 a473; /* yy_suspended_yyss_top */
	T6 a474; /* yy_suspended_yy_goto */
	T6 a475; /* yyssp */
	T6 a476; /* error_count */
	T1 a477; /* yy_lookahead_needed */
	T6 a478; /* yyerrstatus */
	T6 a479; /* last_token */
	T0* a480; /* last_clients */
	T0* a481; /* last_export_clients */
	T0* a482; /* last_feature_clause */
	T6 a483; /* last_literal_end */
	T1 a484; /* yy_rejected */
	T6 a485; /* yy_state_count */
	T6 a486; /* yy_full_match */
	T6 a487; /* yy_lp */
	T6 a488; /* yy_looking_for_trail_begin */
	T6 a489; /* yy_full_lp */
	T6 a490; /* yy_full_state */
	T6 a491; /* pushed_start_condition_count */
	T1 a492; /* yy_more_flag */
	T6 a493; /* yy_more_len */
	T6 a494; /* yy_last_accepting_state */
	T6 a495; /* yy_last_accepting_cpos */
	T6 a496; /* yyvsp1 */
	T6 a497; /* yyvsp2 */
	T6 a498; /* yyvsp3 */
	T6 a499; /* yyvsp4 */
	T6 a500; /* yyvsp5 */
	T6 a501; /* yyvsp6 */
	T6 a502; /* yyvsp7 */
	T6 a503; /* yyvsp8 */
	T6 a504; /* yyvsp9 */
	T6 a505; /* yyvsp10 */
	T6 a506; /* yyvsp11 */
	T6 a507; /* yyvsp12 */
	T6 a508; /* yyvsp13 */
	T6 a509; /* yyvsp14 */
	T6 a510; /* yyvsp15 */
	T6 a511; /* yyvsp16 */
	T6 a512; /* yyvsp17 */
	T6 a513; /* yyvsp18 */
	T6 a514; /* yyvsp19 */
	T6 a515; /* yyvsp20 */
	T6 a516; /* yyvsp21 */
	T6 a517; /* yyvsp22 */
	T6 a518; /* yyvsp23 */
	T6 a519; /* yyvsp24 */
	T6 a520; /* yyvsp25 */
	T6 a521; /* yyvsp26 */
	T6 a522; /* yyvsp27 */
	T6 a523; /* yyvsp28 */
	T6 a524; /* yyvsp29 */
	T6 a525; /* yyvsp30 */
	T6 a526; /* yyvsp31 */
	T6 a527; /* yyvsp32 */
	T6 a528; /* yyvsp33 */
	T6 a529; /* yyvsp34 */
	T6 a530; /* yyvsp35 */
	T6 a531; /* yyvsp36 */
	T6 a532; /* yyvsp37 */
	T6 a533; /* yyvsp38 */
	T6 a534; /* yyvsp39 */
	T6 a535; /* yyvsp40 */
	T6 a536; /* yyvsp41 */
	T6 a537; /* yyvsp42 */
	T6 a538; /* yyvsp43 */
	T6 a539; /* yyvsp44 */
	T6 a540; /* yyvsp45 */
	T6 a541; /* yyvsp46 */
	T6 a542; /* yyvsp47 */
	T6 a543; /* yyvsp48 */
	T6 a544; /* yyvsp49 */
	T6 a545; /* yyvsp50 */
	T6 a546; /* yyvsp51 */
	T6 a547; /* yyvsp52 */
	T6 a548; /* yyvsp53 */
	T6 a549; /* yyvsp54 */
	T6 a550; /* yyvsp55 */
	T6 a551; /* yyvsp56 */
	T6 a552; /* yyvsp57 */
	T6 a553; /* yyvsp58 */
	T6 a554; /* yyvsp59 */
	T6 a555; /* yyvsp60 */
	T6 a556; /* yyvsp61 */
	T6 a557; /* yyvsp62 */
	T6 a558; /* yyvsp63 */
	T6 a559; /* yyvsp64 */
	T6 a560; /* yyvsp65 */
	T6 a561; /* yyvsp66 */
	T6 a562; /* yyvsp67 */
	T6 a563; /* yyvsp68 */
	T6 a564; /* yyvsp69 */
	T6 a565; /* yyvsp70 */
	T6 a566; /* yyvsp71 */
	T6 a567; /* yyvsp72 */
	T6 a568; /* yyvsp73 */
	T6 a569; /* yyvsp74 */
	T6 a570; /* yyvsp75 */
	T6 a571; /* yyvsp76 */
	T6 a572; /* yyvsp77 */
	T6 a573; /* yyvsp78 */
	T6 a574; /* yyvsp79 */
	T6 a575; /* yyvsp80 */
	T6 a576; /* yyvsp81 */
	T6 a577; /* yyvsp82 */
	T6 a578; /* yyvsp83 */
	T6 a579; /* yyvsp84 */
	T6 a580; /* yyvsp85 */
	T6 a581; /* yyvsp86 */
	T6 a582; /* yyvsp87 */
	T6 a583; /* yyvsp88 */
	T6 a584; /* yyvsp89 */
	T6 a585; /* yyvsp90 */
	T6 a586; /* yyvsp91 */
	T6 a587; /* yyvsp92 */
	T6 a588; /* yyvsp93 */
	T6 a589; /* yyvsp94 */
	T6 a590; /* yyvsp95 */
	T6 a591; /* yyvsp96 */
	T6 a592; /* yyvsp97 */
	T6 a593; /* yyvsp98 */
	T6 a594; /* yyvsp99 */
	T6 a595; /* yyvsp100 */
	T6 a596; /* yyvsp101 */
	T6 a597; /* yyvsp102 */
	T6 a598; /* yyvsp103 */
	T6 a599; /* yyvsp104 */
	T6 a600; /* yyvsp105 */
	T6 a601; /* yyvsp106 */
	T6 a602; /* yyvsp107 */
	T6 a603; /* yyvsp108 */
	T6 a604; /* yyvsp109 */
	T6 a605; /* yyvsp110 */
	T6 a606; /* yyvsp111 */
	T6 a607; /* yyvsp112 */
	T6 a608; /* yyvsp113 */
	T6 a609; /* yyvsp114 */
	T6 a610; /* yyvsp115 */
	T6 a611; /* yyvsp116 */
	T6 a612; /* yyvsp117 */
	T6 a613; /* yyvsp118 */
	T6 a614; /* yyvsp119 */
	T6 a615; /* yyvsp120 */
	T6 a616; /* yyvsp121 */
	T6 a617; /* yyvsp122 */
	T6 a618; /* yyvsp123 */
	T6 a619; /* yyvsp124 */
	T6 a620; /* yyvsp125 */
	T6 a621; /* yyvsp126 */
	T6 a622; /* yyvsp127 */
	T6 a623; /* yyvsp128 */
	T6 a624; /* yyvsp129 */
	T6 a625; /* yyvsp130 */
	T6 a626; /* yyvsp131 */
	T6 a627; /* yyvsp132 */
	T6 a628; /* yyvsp133 */
	T6 a629; /* yyvsp134 */
	T0* a630; /* last_detachable_any_value */
	T0* a631; /* last_detachable_et_keyword_value */
	T0* a632; /* last_detachable_et_agent_keyword_value */
	T0* a633; /* last_detachable_et_precursor_keyword_value */
	T0* a634; /* last_detachable_et_symbol_value */
	T0* a635; /* last_detachable_et_position_value */
	T0* a636; /* last_detachable_et_boolean_constant_value */
	T0* a637; /* last_detachable_et_break_value */
	T0* a638; /* last_detachable_et_character_constant_value */
	T0* a639; /* last_detachable_et_current_value */
	T0* a640; /* last_detachable_et_free_operator_value */
	T0* a641; /* last_detachable_et_identifier_value */
	T0* a642; /* last_detachable_et_integer_constant_value */
	T0* a643; /* last_detachable_et_keyword_operator_value */
	T0* a644; /* last_detachable_et_manifest_string_value */
	T0* a645; /* last_detachable_et_real_constant_value */
	T0* a646; /* last_detachable_et_result_value */
	T0* a647; /* last_detachable_et_retry_instruction_value */
	T0* a648; /* last_detachable_et_symbol_operator_value */
	T0* a649; /* last_detachable_et_void_value */
	T0* a650; /* last_detachable_et_semicolon_symbol_value */
	T0* a651; /* last_detachable_et_bracket_symbol_value */
	T0* a652; /* last_detachable_et_question_mark_symbol_value */
	T0* a653; /* last_formal_arguments */
	T0* a654; /* last_local_variables */
	T0* a655; /* last_object_tests */
	T0* a656; /* last_iteration_components */
	T6 a657; /* last_break_end */
	T6 a658; /* last_comment_end */
	T6 a659; /* ms_column */
	T6 a660; /* ms_line */
	T6 a661; /* break_kind */
	T3 a662; /* verbatim_opening_character */
	T1 a663; /* has_natural_64_overflow */
	T11 a664; /* last_natural_64 */
};

/* Struct for type [detachable] ET_PROVIDER_CHECKER */
struct S457 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
};

/* Struct for type [detachable] ET_ANCESTOR_BUILDER */
struct S458 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_sorter */
	T0* a2; /* ancestors */
	T0* a3; /* conforming_ancestors */
	T0* a4; /* parent_checker */
	T0* a5; /* formal_parameter_checker */
	T0* a6; /* parent_context */
	T0* a7; /* current_class */
	T0* a8; /* system_processor */
};

/* Struct for type [detachable] ET_FEATURE_FLATTENER */
struct S459 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* named_features */
	T0* a2; /* queries */
	T0* a3; /* procedures */
	T0* a4; /* aliased_features */
	T0* a5; /* clients_list */
	T0* a6; /* client_classes */
	T0* a7; /* feature_adaptation_resolver */
	T0* a8; /* dotnet_feature_adaptation_resolver */
	T0* a9; /* identifier_type_resolver */
	T0* a10; /* unfolded_tuple_actual_parameters_resolver */
	T0* a11; /* anchored_type_checker */
	T0* a12; /* signature_checker */
	T0* a13; /* parent_checker */
	T0* a14; /* formal_parameter_checker */
	T0* a15; /* builtin_feature_checker */
	T0* a16; /* precursor_checker */
	T0* a17; /* precursors */
	T0* a18; /* current_class */
	T0* a19; /* system_processor */
	T1 a20; /* has_signature_error */
};

/* Struct for type [detachable] ET_INTERFACE_CHECKER */
struct S460 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* classes_to_be_processed */
	T0* a2; /* qualified_anchored_type_checker */
	T0* a3; /* unfolded_tuple_actual_parameters_resolver */
	T0* a4; /* old_name_rename_table */
	T0* a5; /* new_name_rename_table */
	T0* a6; /* new_alias_name_rename_table */
	T0* a7; /* current_class */
	T0* a8; /* system_processor */
};

/* Struct for type [detachable] ET_IMPLEMENTATION_CHECKER */
struct S461 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
	T0* a3; /* feature_checker */
	T0* a4; /* named_features */
	T0* a5; /* feature_adaptation_resolver */
	T0* a6; /* dotnet_feature_adaptation_resolver */
	T0* a7; /* signature_checker */
	T0* a8; /* parent_checker3 */
	T0* a9; /* precursor_procedures */
	T0* a10; /* precursor_queries */
	T0* a11; /* supplier_builder */
	T0* a12; /* no_suppliers */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_CLASS] */
struct S462 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
	T0* a6; /* equality_tester */
};

/* Struct for type [detachable] TUPLE [[attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
struct S463 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] ET_AST_NULL_PROCESSOR */
struct S464 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_AST_FACTORY */
struct S465 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
struct S467 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* system_processor */
};

/* Struct for type [detachable] DT_DATE_TIME_DURATION */
struct S468 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* year */
	T6 a2; /* month */
	T6 a3; /* day */
	T6 a4; /* hour */
	T6 a5; /* minute */
	T6 a6; /* second */
	T6 a7; /* millisecond */
};

/* Struct for type [detachable] ET_DO_PROCEDURE */
struct S469 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* iteration_components */
	T0* a2; /* implementation_class */
	T0* a3; /* implementation_feature */
	T6 a4; /* first_seed */
	T0* a5; /* other_seeds */
	T0* a6; /* arguments */
	T0* a7; /* compound */
	T0* a8; /* extended_name */
	T6 a9; /* hash_code */
	T0* a10; /* end_keyword */
	T0* a11; /* clients */
	T1 a12; /* validity_checked */
	T1 a13; /* has_validity_error */
	T0* a14; /* preconditions */
	T0* a15; /* first_precursor */
	T0* a16; /* other_precursors */
	T0* a17; /* postconditions */
	T0* a18; /* rescue_clause */
	T0* a19; /* locals */
	T0* a20; /* object_tests */
	T0* a21; /* frozen_keyword */
	T0* a22; /* obsolete_message */
	T0* a23; /* is_keyword */
	T0* a24; /* semicolon */
	T0* a25; /* feature_clause */
	T0* a26; /* first_indexing */
	T6 a27; /* id */
	T6 a28; /* version */
	T0* a29; /* synonym */
};

/* Struct for type [detachable] UT_CONFIG_PARSER */
struct S471 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* config_values */
	T0* a3; /* error_handler */
	T0* a4; /* defined_values */
	T0* a5; /* include_stack */
	T0* a6; /* line_nb_stack */
	T0* a7; /* last_string_value */
	T6 a8; /* line_nb */
	T0* a9; /* yyss */
	T0* a10; /* input_buffer */
	T6 a11; /* yy_end */
	T6 a12; /* yy_position */
	T6 a13; /* yy_line */
	T6 a14; /* yy_column */
	T6 a15; /* yy_parsing_status */
	T6 a16; /* yy_suspended_yystacksize */
	T6 a17; /* yy_suspended_yystate */
	T6 a18; /* yy_suspended_yyn */
	T6 a19; /* yy_suspended_yychar1 */
	T6 a20; /* yy_suspended_index */
	T6 a21; /* yy_suspended_yyss_top */
	T6 a22; /* yy_suspended_yy_goto */
	T0* a23; /* yyr1 */
	T6 a24; /* yyssp */
	T0* a25; /* yypgoto */
	T0* a26; /* yycheck */
	T0* a27; /* yytable */
	T0* a28; /* yydefgoto */
	T6 a29; /* error_count */
	T1 a30; /* yy_lookahead_needed */
	T6 a31; /* yyerrstatus */
	T0* a32; /* yypact */
	T6 a33; /* last_token */
	T0* a34; /* yytranslate */
	T0* a35; /* yydefact */
	T0* a36; /* yy_pushed_start_conditions */
	T0* a37; /* yyspecial_routines1 */
	T6 a38; /* yyvsc1 */
	T0* a39; /* yyvs1 */
	T0* a40; /* yyspecial_routines2 */
	T6 a41; /* yyvsc2 */
	T0* a42; /* yyvs2 */
	T0* a43; /* yyspecial_routines3 */
	T6 a44; /* yyvsc3 */
	T0* a45; /* yyvs3 */
	T0* a46; /* yytypes1 */
	T0* a47; /* yytypes2 */
	T6 a48; /* yy_start */
	T6 a49; /* yyvsp1 */
	T6 a50; /* yyvsp2 */
	T6 a51; /* yyvsp3 */
	T1 a52; /* yy_more_flag */
	T6 a53; /* yy_more_len */
	T6 a54; /* line */
	T6 a55; /* column */
	T6 a56; /* position */
	T6 a57; /* yy_start_state */
	T0* a58; /* yy_state_stack */
	T6 a59; /* yy_state_count */
	T0* a60; /* yy_content_area */
	T0* a61; /* yy_unicode_content_area */
	T0* a62; /* yy_accept */
	T6 a63; /* yy_last_accepting_state */
	T6 a64; /* yy_last_accepting_cpos */
	T0* a65; /* yy_content */
	T0* a66; /* yy_ec */
	T0* a67; /* yy_chk */
	T0* a68; /* yy_base */
	T0* a69; /* yy_def */
	T0* a70; /* yy_meta */
	T0* a71; /* yy_nxt */
	T6 a72; /* yy_lp */
	T0* a73; /* yy_acclist */
	T6 a74; /* yy_looking_for_trail_begin */
	T6 a75; /* yy_full_match */
	T6 a76; /* yy_full_state */
	T6 a77; /* yy_full_lp */
	T1 a78; /* yy_rejected */
	T0* a79; /* last_detachable_any_value */
	T6 a80; /* if_level */
	T6 a81; /* ignored_level */
};

/* Struct for type [detachable] KL_COMPARABLE_COMPARATOR [[attached] INTEGER_32] */
struct S472 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_DYNAMIC_PUSH_TYPE_SET */
struct S475 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T6 a2; /* count */
	T1 a3; /* is_never_void */
	T0* a4; /* dynamic_types */
	T0* a5; /* targets */
	T1 a6; /* is_dynamic_types_readonly */
};

/* Struct for type [detachable] ET_DEFERRED_PROCEDURE */
struct S476 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T6 a4; /* first_seed */
	T0* a5; /* other_seeds */
	T0* a6; /* arguments */
	T0* a7; /* extended_name */
	T0* a8; /* deferred_keyword */
	T6 a9; /* hash_code */
	T0* a10; /* end_keyword */
	T0* a11; /* clients */
	T1 a12; /* validity_checked */
	T1 a13; /* has_validity_error */
	T0* a14; /* preconditions */
	T0* a15; /* first_precursor */
	T0* a16; /* other_precursors */
	T0* a17; /* postconditions */
	T0* a18; /* object_tests */
	T0* a19; /* frozen_keyword */
	T0* a20; /* obsolete_message */
	T0* a21; /* is_keyword */
	T0* a22; /* semicolon */
	T0* a23; /* feature_clause */
	T0* a24; /* first_indexing */
	T6 a25; /* id */
	T6 a26; /* version */
	T0* a27; /* synonym */
};

/* Struct for type [detachable] KL_PLATFORM */
struct S477 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] CELL [[attached] INTEGER_32] */
struct S479 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [detachable] HASH_TABLE [[attached] NATIVE_STRING, [attached] IMMUTABLE_STRING_32] */
struct S480 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T0* a2; /* content */
	T0* a3; /* keys */
	T0* a4; /* deleted_marks */
	T0* a5; /* indexes_map */
	T6 a6; /* iteration_position */
	T6 a7; /* count */
	T6 a8; /* deleted_item_position */
	T6 a9; /* control */
	T0* a10; /* found_item */
	T1 a11; /* has_default */
	T6 a12; /* item_position */
	T6 a13; /* ht_lowest_deleted_position */
	T0* a14; /* ht_deleted_item */
	T0* a15; /* ht_deleted_key */
	T1 a16; /* object_comparison */
};

/* Struct for type [detachable] KL_ANY_ROUTINES */
struct S481 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_PATHNAME */
struct S483 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* drive */
	T0* a3; /* hostname */
	T0* a4; /* sharename */
	T1 a5; /* is_relative */
	T0* a6; /* components */
};

/* Struct for type [detachable] ARRAY [[attached] IMMUTABLE_STRING_32] */
struct S484 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
	T1 a4; /* object_comparison */
};

/* Struct for type [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32] */
struct S485 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] EXCEPTIONS */
struct S486 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] NATURAL_8] */
struct S487 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T8 z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] AP_OPTION] */
struct S489 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] AP_OPTION] */
struct S491 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] AP_OPTION] */
struct S492 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S493 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S494 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S495 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] STRING_8] */
struct S496 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [detachable STRING_8] */
struct S497 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8] */
struct S498 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] STRING_8] */
struct S499 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type detachable DS_LINKABLE [[attached] STRING_8] */
struct S500 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type detachable DS_LINKABLE [[attached] BOOLEAN] */
struct S501 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] BOOLEAN] */
struct S502 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type detachable DS_LINKABLE [[attached] INTEGER_32] */
struct S503 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] INTEGER_32] */
struct S504 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
struct S506 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_LIBRARY] */
struct S507 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_LIBRARY] */
struct S510 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_DOTNET_ASSEMBLY] */
struct S511 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
struct S514 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_DOTNET_ASSEMBLY] */
struct S515 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_EIFFEL_SCANNER */
struct S516 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_token */
	T0* a2; /* last_value */
	T0* a3; /* input_name */
	T6 a4; /* position */
	T6 a5; /* column */
	T6 a6; /* line */
	T0* a7; /* input_filter */
	T6 a8; /* yy_start_state */
	T0* a9; /* character_entity */
	T0* a10; /* input_stream */
	T0* a11; /* input_resolver */
	T1 a12; /* yy_more_flag */
	T6 a13; /* yy_more_len */
	T6 a14; /* yy_end */
	T6 a15; /* yy_start */
	T6 a16; /* yy_line */
	T6 a17; /* yy_column */
	T6 a18; /* yy_position */
	T0* a19; /* yy_state_stack */
	T6 a20; /* yy_state_count */
	T0* a21; /* yy_content_area */
	T0* a22; /* yy_unicode_content_area */
	T0* a23; /* yy_accept */
	T6 a24; /* yy_last_accepting_state */
	T6 a25; /* yy_last_accepting_cpos */
	T0* a26; /* yy_content */
	T0* a27; /* yy_ec */
	T0* a28; /* yy_chk */
	T0* a29; /* yy_base */
	T0* a30; /* yy_def */
	T0* a31; /* yy_meta */
	T0* a32; /* yy_nxt */
	T6 a33; /* yy_lp */
	T0* a34; /* yy_acclist */
	T6 a35; /* yy_looking_for_trail_begin */
	T6 a36; /* yy_full_match */
	T6 a37; /* yy_full_state */
	T6 a38; /* yy_full_lp */
	T0* a39; /* input_buffer */
	T1 a40; /* yy_rejected */
	T0* a41; /* yy_pushed_start_conditions */
	T0* a42; /* last_string_value */
	T0* a43; /* last_error */
	T6 a44; /* pushed_start_condition_count */
};

/* Struct for type [detachable] XM_DEFAULT_POSITION */
struct S517 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* row */
	T6 a2; /* column */
	T0* a3; /* source_name */
	T6 a4; /* byte_index */
};

/* Struct for type [detachable] DS_BILINKED_LIST [[attached] XM_POSITION] */
struct S519 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] DS_LINKED_STACK [[attached] XM_EIFFEL_SCANNER] */
struct S520 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* first_cell */
};

/* Struct for type [detachable] XM_CALLBACKS_NULL */
struct S521 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
struct S522 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* capacity */
	T0* a7; /* key_equality_tester */
	T0* a8; /* internal_keys */
	T6 a9; /* modulus */
	T6 a10; /* slots_position */
	T6 a11; /* clashes_previous_position */
	T0* a12; /* internal_cursor */
	T6 a13; /* found_position */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* equality_tester */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] XM_NULL_EXTERNAL_RESOLVER */
struct S524 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_DTD_CALLBACKS_NULL */
struct S526 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ANY] */
struct S527 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ANY] */
struct S528 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_PARSER_NAME] */
struct S529 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_EIFFEL_PARSER_NAME */
struct S530 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* use_namespaces */
	T6 a2; /* count */
	T0* a3; /* first */
	T0* a4; /* second */
	T0* a5; /* tail */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_EIFFEL_PARSER_NAME] */
struct S531 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
struct S532 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME] */
struct S533 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* equality_tester */
	T6 a3; /* count */
	T6 a4; /* capacity */
	T6 a5; /* free_slot */
	T6 a6; /* last_position */
	T6 a7; /* modulus */
	T6 a8; /* slots_position */
	T6 a9; /* clashes_previous_position */
	T6 a10; /* found_position */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T0* a13; /* special_item_routines */
	T0* a14; /* item_storage */
	T0* a15; /* internal_cursor */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
struct S534 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_EXTERNAL_ID] */
struct S535 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_DTD_EXTERNAL_ID */
struct S536 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* system_id */
	T0* a2; /* public_id */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_DTD_EXTERNAL_ID] */
struct S537 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S538 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_DTD_ELEMENT_CONTENT */
struct S539 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* items */
	T2 a2; /* repetition */
	T0* a3; /* name */
	T2 a4; /* type */
	T1 a5; /* is_character_data_allowed */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S540 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
struct S541 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S542 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
struct S543 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S544 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_DTD_ATTRIBUTE_CONTENT */
struct S545 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* default_value */
	T2 a3; /* type */
	T1 a4; /* is_list_type */
	T0* a5; /* enumeration_list */
	T2 a6; /* value */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S546 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
struct S547 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_BILINKED_LIST [[attached] STRING_8] */
struct S548 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* equality_tester */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
struct S549 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] BOOLEAN] */
struct S550 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_DECLARATION] */
struct S551 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XM_EIFFEL_DECLARATION */
struct S552 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* encoding */
	T0* a2; /* version */
	T1 a3; /* stand_alone */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_EIFFEL_DECLARATION] */
struct S553 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XM_EIFFEL_ENTITY_DEF */
struct S556 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* literal_name */
	T0* a2; /* value */
	T0* a3; /* resolver */
	T0* a4; /* external_id */
	T0* a5; /* character_entity */
	T0* a6; /* yy_pushed_start_conditions */
	T0* a7; /* input_buffer */
	T1 a8; /* in_use */
	T0* a9; /* input_name */
	T0* a10; /* last_value */
	T0* a11; /* last_string_value */
	T0* a12; /* last_error */
	T1 a13; /* yy_rejected */
	T6 a14; /* yy_state_count */
	T6 a15; /* yy_full_match */
	T6 a16; /* yy_lp */
	T6 a17; /* yy_looking_for_trail_begin */
	T6 a18; /* yy_full_lp */
	T6 a19; /* yy_full_state */
	T6 a20; /* pushed_start_condition_count */
	T6 a21; /* yy_start_state */
	T6 a22; /* yy_line */
	T6 a23; /* yy_column */
	T6 a24; /* yy_position */
	T6 a25; /* yy_start */
	T6 a26; /* yy_end */
	T6 a27; /* line */
	T6 a28; /* column */
	T6 a29; /* position */
	T1 a30; /* yy_more_flag */
	T6 a31; /* yy_more_len */
	T6 a32; /* yy_last_accepting_state */
	T6 a33; /* yy_last_accepting_cpos */
	T0* a34; /* yy_state_stack */
	T0* a35; /* yy_nxt */
	T0* a36; /* yy_chk */
	T0* a37; /* yy_base */
	T0* a38; /* yy_def */
	T0* a39; /* yy_ec */
	T0* a40; /* yy_meta */
	T0* a41; /* yy_accept */
	T0* a42; /* yy_content */
	T0* a43; /* yy_content_area */
	T0* a44; /* yy_unicode_content_area */
	T6 a45; /* last_token */
	T0* a46; /* input_filter */
	T0* a47; /* input_stream */
	T0* a48; /* input_resolver */
	T0* a49; /* yy_acclist */
};

/* Struct for type [detachable] XM_EIFFEL_SCANNER_DTD */
struct S557 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_token */
	T0* a2; /* last_value */
	T0* a3; /* input_name */
	T6 a4; /* position */
	T6 a5; /* column */
	T6 a6; /* line */
	T0* a7; /* input_filter */
	T6 a8; /* yy_start_state */
	T0* a9; /* character_entity */
	T0* a10; /* input_stream */
	T0* a11; /* input_resolver */
	T1 a12; /* decl_start_sent */
	T1 a13; /* decl_end_sent */
	T1 a14; /* yy_more_flag */
	T6 a15; /* yy_more_len */
	T6 a16; /* yy_end */
	T6 a17; /* yy_start */
	T6 a18; /* yy_line */
	T6 a19; /* yy_column */
	T6 a20; /* yy_position */
	T0* a21; /* yy_state_stack */
	T6 a22; /* yy_state_count */
	T0* a23; /* yy_content_area */
	T0* a24; /* yy_unicode_content_area */
	T0* a25; /* yy_accept */
	T6 a26; /* yy_last_accepting_state */
	T6 a27; /* yy_last_accepting_cpos */
	T0* a28; /* yy_content */
	T0* a29; /* yy_ec */
	T0* a30; /* yy_chk */
	T0* a31; /* yy_base */
	T0* a32; /* yy_def */
	T0* a33; /* yy_meta */
	T0* a34; /* yy_nxt */
	T6 a35; /* yy_lp */
	T0* a36; /* yy_acclist */
	T6 a37; /* yy_looking_for_trail_begin */
	T6 a38; /* yy_full_match */
	T6 a39; /* yy_full_state */
	T6 a40; /* yy_full_lp */
	T0* a41; /* input_buffer */
	T1 a42; /* yy_rejected */
	T0* a43; /* yy_pushed_start_conditions */
	T0* a44; /* last_string_value */
	T0* a45; /* last_error */
	T6 a46; /* pushed_start_condition_count */
};

/* Struct for type [detachable] XM_EIFFEL_PE_ENTITY_DEF */
struct S559 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* resolver */
	T0* a2; /* external_id */
	T0* a3; /* literal_name */
	T0* a4; /* value */
	T0* a5; /* character_entity */
	T0* a6; /* yy_pushed_start_conditions */
	T0* a7; /* input_buffer */
	T1 a8; /* in_use */
	T0* a9; /* input_name */
	T0* a10; /* last_value */
	T0* a11; /* last_string_value */
	T0* a12; /* last_error */
	T1 a13; /* yy_rejected */
	T6 a14; /* yy_state_count */
	T6 a15; /* yy_full_match */
	T6 a16; /* yy_lp */
	T6 a17; /* yy_looking_for_trail_begin */
	T6 a18; /* yy_full_lp */
	T6 a19; /* yy_full_state */
	T6 a20; /* pushed_start_condition_count */
	T6 a21; /* yy_start_state */
	T6 a22; /* yy_line */
	T6 a23; /* yy_column */
	T6 a24; /* yy_position */
	T6 a25; /* yy_start */
	T6 a26; /* yy_end */
	T6 a27; /* line */
	T6 a28; /* column */
	T6 a29; /* position */
	T1 a30; /* yy_more_flag */
	T6 a31; /* yy_more_len */
	T6 a32; /* yy_last_accepting_state */
	T6 a33; /* yy_last_accepting_cpos */
	T0* a34; /* yy_state_stack */
	T1 a35; /* pre_sent */
	T1 a36; /* post_sent */
	T0* a37; /* yy_nxt */
	T0* a38; /* yy_chk */
	T0* a39; /* yy_base */
	T0* a40; /* yy_def */
	T0* a41; /* yy_ec */
	T0* a42; /* yy_meta */
	T0* a43; /* yy_accept */
	T0* a44; /* yy_content */
	T0* a45; /* yy_content_area */
	T0* a46; /* yy_unicode_content_area */
	T6 a47; /* last_token */
	T0* a48; /* input_filter */
	T0* a49; /* input_stream */
	T0* a50; /* input_resolver */
	T0* a51; /* yy_acclist */
};

/* Struct for type [detachable] XM_NAMESPACE_RESOLVER */
struct S560 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* element_local_part */
	T0* a3; /* element_prefix */
	T0* a4; /* context */
	T1 a5; /* forward_xmlns */
	T0* a6; /* attributes_prefix */
	T0* a7; /* attributes_local_part */
	T0* a8; /* attributes_value */
};

/* Struct for type [detachable] ARRAY [[attached] XM_CALLBACKS_FILTER] */
struct S561 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_CALLBACKS_FILTER] */
struct S562 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] XM_NAMESPACE] */
struct S563 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T6 a2; /* capacity */
	T6 a3; /* modulus */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T0* a7; /* special_item_routines */
	T0* a8; /* item_storage */
	T0* a9; /* clashes */
	T0* a10; /* slots */
	T6 a11; /* found_position */
	T0* a12; /* internal_cursor */
	T6 a13; /* slots_position */
	T6 a14; /* count */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] ET_CLUSTERS */
struct S564 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clusters */
};

/* Struct for type [detachable] ET_CLASS_NAME_TESTER */
struct S565 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_PARENT */
struct S567 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* renames */
	T0* a3; /* exports */
	T0* a4; /* undefines */
	T0* a5; /* redefines */
	T0* a6; /* selects */
	T0* a7; /* end_keyword */
};

/* Struct for type [detachable] ET_PARENT_LIST */
struct S568 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* inherit_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
	T0* a4; /* clients_clause */
};

/* Struct for type [detachable] ET_BUILTIN_CONVERT_FEATURE */
struct S569 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* types */
	T0* a3; /* type */
};

/* Struct for type [detachable] ET_MASTER_CLASS */
struct S570 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* mapped_class */
	T0* a2; /* first_overriding_class */
	T0* a3; /* intrinsic_class */
	T0* a4; /* name */
	T0* a5; /* universe */
	T0* a6; /* status_mutex */
	T0* a7; /* processing_mutex */
	T1 a8; /* unprotected_is_marked */
	T0* a9; /* first_imported_class */
	T0* a10; /* other_imported_classes */
	T0* a11; /* first_local_override_class */
	T0* a12; /* first_local_non_override_class */
	T0* a13; /* first_local_ignored_class */
	T0* a14; /* first_local_hidden_class */
	T1 a15; /* is_modified */
	T0* a16; /* other_local_override_classes */
	T0* a17; /* other_local_non_override_classes */
	T0* a18; /* other_local_ignored_classes */
	T0* a19; /* other_overriding_classes */
	T0* a20; /* other_local_hidden_classes */
};

/* Struct for type [detachable] ET_ATTACHMENT_MARK_SEPARATE_KEYWORD */
struct S571 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* attachment_mark */
	T0* a2; /* separateness_keyword */
};

/* Struct for type detachable ET_RENAME_LIST */
struct S572 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* rename_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type detachable ET_EXPORT_LIST */
struct S573 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* export_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type detachable ET_KEYWORD_FEATURE_NAME_LIST */
struct S574 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
struct S577 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* tuple_type */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_LIBRARY] */
struct S578 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] XM_LINKED_LIST [[attached] XM_DOCUMENT_NODE] */
struct S579 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_SYSTEM_CONFIG] */
struct S582 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_SYSTEM_CONFIG] */
struct S584 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
struct S585 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_TARGET, [attached] STRING_8] */
struct S586 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_TARGET] */
struct S587 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET] */
struct S589 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S591 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table */
	T0* a2; /* equality_tester */
	T0* a3; /* internal_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
struct S592 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable RX_REGULAR_EXPRESSION] */
struct S593 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S594 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] ET_DOTNET_ASSEMBLIES */
struct S595 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* dotnet_assemblies */
};

/* Struct for type [detachable] ET_DOTNET_ASSEMBLY */
struct S596 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* referenced_assemblies */
	T0* a3; /* current_system */
	T0* a4; /* dotnet_assembly */
	T0* a5; /* name */
	T0* a6; /* implicit_attachment_type_mark */
	T0* a7; /* master_classes */
	T0* a8; /* master_class_mutex */
	T0* a9; /* any_type */
	T0* a10; /* detachable_any_type */
	T0* a11; /* detachable_separate_any_type */
	T0* a12; /* any_parent */
	T0* a13; /* any_parents */
	T0* a14; /* any_clients */
	T0* a15; /* tuple_type */
	T0* a16; /* detachable_tuple_type */
	T0* a17; /* tuple_identity_type */
	T0* a18; /* unfolded_empty_tuple_actual_parameters */
	T0* a19; /* array_any_type */
	T0* a20; /* array_detachable_any_type */
	T0* a21; /* array_none_type */
	T0* a22; /* array_identity_type */
	T0* a23; /* boolean_type */
	T0* a24; /* character_type */
	T0* a25; /* character_8_type */
	T0* a26; /* character_8_convert_feature */
	T0* a27; /* character_32_type */
	T0* a28; /* character_32_convert_feature */
	T0* a29; /* double_type */
	T0* a30; /* exception_type */
	T0* a31; /* detachable_exception_type */
	T0* a32; /* exception_manager_type */
	T0* a33; /* function_identity_any_type */
	T0* a34; /* immutable_string_8_type */
	T0* a35; /* immutable_string_32_type */
	T0* a36; /* integer_type */
	T0* a37; /* integer_8_type */
	T0* a38; /* integer_8_convert_feature */
	T0* a39; /* integer_16_type */
	T0* a40; /* integer_16_convert_feature */
	T0* a41; /* integer_32_type */
	T0* a42; /* integer_32_convert_feature */
	T0* a43; /* integer_64_type */
	T0* a44; /* integer_64_convert_feature */
	T0* a45; /* ise_exception_manager_type */
	T0* a46; /* iterable_detachable_separate_any_type */
	T0* a47; /* natural_type */
	T0* a48; /* natural_8_type */
	T0* a49; /* natural_8_convert_feature */
	T0* a50; /* natural_16_type */
	T0* a51; /* natural_16_convert_feature */
	T0* a52; /* natural_32_type */
	T0* a53; /* natural_32_convert_feature */
	T0* a54; /* natural_64_type */
	T0* a55; /* natural_64_convert_feature */
	T0* a56; /* none_type */
	T0* a57; /* detachable_none_type */
	T0* a58; /* pointer_type */
	T0* a59; /* predicate_identity_type */
	T0* a60; /* procedure_identity_type */
	T0* a61; /* real_type */
	T0* a62; /* real_32_type */
	T0* a63; /* real_32_convert_feature */
	T0* a64; /* real_64_type */
	T0* a65; /* real_64_convert_feature */
	T0* a66; /* routine_identity_type */
	T0* a67; /* special_any_type */
	T0* a68; /* special_detachable_any_type */
	T0* a69; /* special_identity_type */
	T0* a70; /* string_type */
	T0* a71; /* detachable_string_type */
	T0* a72; /* string_8_type */
	T0* a73; /* detachable_string_8_type */
	T0* a74; /* string_8_convert_feature */
	T0* a75; /* string_32_type */
	T0* a76; /* string_32_convert_feature */
	T0* a77; /* system_object_type */
	T0* a78; /* system_object_parents */
	T0* a79; /* system_string_type */
	T0* a80; /* type_detachable_any_type */
	T0* a81; /* detachable_type_exception_type */
	T0* a82; /* type_detachable_exception_type */
	T0* a83; /* type_detachable_like_current_type */
	T0* a84; /* type_identity_type */
	T0* a85; /* typed_pointer_identity_type */
	T0* a86; /* wide_character_type */
	T1 a87; /* is_read_only */
	T1 a88; /* is_preparsed */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_IDENTIFIER] */
struct S599 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_IDENTIFIER] */
struct S600 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_IDENTIFIER, [attached] STRING_8] */
struct S601 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type detachable DS_HASH_TABLE [[attached] ET_DYNAMIC_FEATURE, [attached] INTEGER_32] */
struct S602 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* modulus */
	T0* a2; /* key_equality_tester */
	T0* a3; /* item_storage */
	T0* a4; /* slots */
	T0* a5; /* hash_function */
	T0* a6; /* key_storage */
	T0* a7; /* clashes */
	T6 a8; /* position */
	T6 a9; /* last_position */
	T6 a10; /* capacity */
	T6 a11; /* slots_position */
	T6 a12; /* count */
	T0* a13; /* equality_tester */
	T6 a14; /* found_position */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* special_item_routines */
	T0* a17; /* special_key_routines */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
};

/* Struct for type detachable ET_FORMAL_PARAMETER_LIST */
struct S603 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_bracket */
	T0* a4; /* right_bracket */
};

/* Struct for type detachable ET_FEATURE_IDS */
struct S604 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* feature_ids */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_FEATURE] */
struct S605 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_FEATURE] */
struct S606 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
struct S609 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_MASTER_CLASS] */
struct S610 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_MASTER_CLASS] */
struct S611 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLASS_NAME] */
struct S612 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CLASS_NAME] */
struct S613 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] STRING_8] */
struct S615 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL, [attached] ET_CALL_NAME] */
struct S616 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* modulus */
	T0* a2; /* key_equality_tester */
	T0* a3; /* item_storage */
	T0* a4; /* slots */
	T0* a5; /* hash_function */
	T0* a6; /* key_storage */
	T0* a7; /* clashes */
	T6 a8; /* position */
	T6 a9; /* count */
	T6 a10; /* capacity */
	T6 a11; /* slots_position */
	T6 a12; /* free_slot */
	T6 a13; /* last_position */
	T0* a14; /* equality_tester */
	T6 a15; /* found_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, [attached] ET_CALL_NAME] */
struct S617 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* special_key_routines */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T0* a16; /* hash_function */
	T6 a17; /* count */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ACTUAL_PARAMETER_ITEM] */
struct S621 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] ET_BRACKET_SYMBOL */
struct S622 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
	T2 a2; /* code */
	T0* a3; /* break */
	T6 a4; /* seed */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_PARAMETER_ITEM] */
struct S624 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE] */
struct S625 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_TYPE_SET] */
struct S626 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE_SET] */
struct S627 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_IDENTIFIER] */
struct S628 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S629 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S630 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S631 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
struct S632 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S634 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_UNQUALIFIED_CALL_EXPRESSION */
struct S635 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* arguments */
	T0* a4; /* parenthesis_call */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S636 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S637 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
struct S638 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
struct S639 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
struct S640 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_ARGUMENT_LIST] */
struct S641 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ACTUAL_ARGUMENT_LIST] */
struct S642 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ACTUAL_ARGUMENT_LIST] */
struct S643 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RESULT] */
struct S644 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_RESULT] */
struct S645 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_RESULT] */
struct S646 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S647 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S648 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S649 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_EXPRESSION] */
struct S650 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION] */
struct S651 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_EXPRESSION] */
struct S652 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] INTEGER_32] */
struct S653 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] INTEGER_32] */
struct S654 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S655 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_OBJECT_TEST] */
struct S657 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST] */
struct S658 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_TEST] */
struct S659 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S660 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S661 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S662 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_EQUALITY_EXPRESSION] */
struct S663 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EQUALITY_EXPRESSION] */
struct S664 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_EQUALITY_EXPRESSION] */
struct S665 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_AGENT] */
struct S666 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT] */
struct S667 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_AGENT] */
struct S668 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_EXPRESSION_ITEM] */
struct S669 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION_ITEM] */
struct S671 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S672 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S673 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S674 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FEATURE] */
struct S676 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FEATURE] */
struct S677 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE] */
struct S678 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
struct S679 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CONSTANT] */
struct S680 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONSTANT] */
struct S682 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] ET_INLINE_CONSTANT] */
struct S683 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INLINE_CONSTANT] */
struct S684 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_CONSTANT] */
struct S686 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_DYNAMIC_FEATURE] */
struct S687 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S689 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
struct S690 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_FEATURE] */
struct S691 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] STRING_8] */
struct S692 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_IDENTIFIER] */
struct S695 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S698 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* free_slot */
	T6 a8; /* count */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T0* a12; /* equality_tester */
	T0* a13; /* slots */
	T0* a14; /* clashes */
	T0* a15; /* special_item_routines */
	T0* a16; /* key_storage */
	T0* a17; /* special_key_routines */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S699 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T0* a2; /* special_routines */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] ARRAY [[attached] BOOLEAN] */
struct S700 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_SYSTEM_PROCESSOR] */
struct S703 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] TUPLE [[attached] ET_SYSTEM_PROCESSOR, [attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
struct S704 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
struct S707 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_TYPE] */
struct S708 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE] */
struct S709 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS] */
struct S710 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_AGENT] */
struct S711 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INLINE_AGENT] */
struct S712 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_INLINE_AGENT] */
struct S713 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S714 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_PROCEDURE] */
struct S715 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S716 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S717 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S718 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_QUERY] */
struct S719 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S720 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S721 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S722 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_FEATURE] */
struct S723 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* equality_tester */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
	T0* a6; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S724 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S725 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
struct S726 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT] */
struct S727 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] SPECIAL [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
struct S728 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
struct S729 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S730 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S731 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S732 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
struct S733 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] ET_NAMED_OBJECT_TEST */
struct S736 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* attached_keyword */
	T0* a4; /* declared_type */
	T0* a5; /* expression */
	T0* a6; /* as_keyword */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_NAMED_OBJECT_TEST] */
struct S737 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_OBJECT_TEST] */
struct S738 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_NAMED_OBJECT_TEST] */
struct S739 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* special_routines */
	T0* a3; /* storage */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ITERATION_COMPONENT] */
struct S740 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ITERATION_COMPONENT] */
struct S741 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ITERATION_COMPONENT] */
struct S743 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ITERATION_COMPONENT] */
struct S744 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] INTEGER_32] */
struct S745 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* count */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ATTACHMENT_SCOPE] */
struct S747 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ATTACHMENT_SCOPE] */
struct S748 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ATTACHMENT_SCOPE] */
struct S749 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
struct S750 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ASSERTIONS] */
struct S751 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTIONS] */
struct S753 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM] */
struct S754 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INDEXING_TERM] */
struct S756 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_INDEXING_TERM] */
struct S757 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CLIENT_ITEM] */
struct S758 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_ITEM] */
struct S759 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
struct S760 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
struct S761 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
struct S762 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
struct S763 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* count */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] SPECIAL [[attached] NATURAL_32] */
struct S764 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T10 z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] RX_CHARACTER_SET] */
struct S765 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_32] */
struct S766 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] NATURAL_64] */
struct S767 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T11 z2[1]; /* item */
};

/* Struct for type detachable DS_HASH_TABLE [[attached] NATURAL_64, [attached] NATURAL_32] */
struct S768 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* position */
	T0* a3; /* internal_cursor */
	T6 a4; /* found_position */
	T0* a5; /* item_storage */
	T6 a6; /* modulus */
	T0* a7; /* key_equality_tester */
	T0* a8; /* key_storage */
	T0* a9; /* slots */
	T0* a10; /* hash_function */
	T0* a11; /* clashes */
	T0* a12; /* internal_keys */
	T6 a13; /* capacity */
	T6 a14; /* free_slot */
	T6 a15; /* last_position */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* equality_tester */
	T0* a19; /* special_item_routines */
	T0* a20; /* special_key_routines */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] NATURAL_64, [attached] NATURAL_32] */
struct S769 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]] */
struct S770 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]]] */
struct S771 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_INTEGER_ROUTINES */
struct S772 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] UC_UTF8_STRING */
struct S775 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* area */
	T6 a3; /* byte_count */
	T6 a4; /* internal_hash_code */
	T6 a5; /* last_byte_index_input */
	T6 a6; /* last_byte_index_result */
	T6 a7; /* internal_case_insensitive_hash_code */
};

/* Struct for type [detachable] THREAD_ATTRIBUTES */
struct S776 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* shared */
	T14 a2; /* internal_item */
	T0* a3; /* managed_pointer */
};

/* Struct for type [detachable] CELL [[attached] BOOLEAN] */
struct S777 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* item */
};

/* Struct for type [detachable] CHARACTER_PROPERTY */
struct S780 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] STRING_8_SEARCHER */
struct S781 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* deltas */
};

/* Struct for type detachable ARRAYED_LIST [[attached] INTEGER_32] */
struct S782 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area_v2 */
	T6 a2; /* index */
};

/* Struct for type [detachable] ET_SYSTEM_ERROR */
struct S783 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
};

/* Struct for type [detachable] ET_INTERNAL_ERROR */
struct S784 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
};

/* Struct for type [detachable] ET_VALIDITY_ERROR */
struct S785 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* ise_reported */
	T1 a2; /* ge_reported */
	T0* a3; /* default_template */
	T0* a4; /* parameters */
	T0* a5; /* current_class */
	T0* a6; /* class_impl */
	T0* a7; /* position */
	T0* a8; /* code */
	T0* a9; /* etl_code */
	T1 a10; /* ise_fatal */
	T1 a11; /* ge_fatal */
};

/* Struct for type detachable ET_ECF_NOTE_ELEMENT */
struct S786 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* attributes */
	T0* a3; /* elements */
	T0* a4; /* content */
};

/* Struct for type detachable ET_ECF_CLUSTERS */
struct S787 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clusters */
};

/* Struct for type detachable ET_ECF_ADAPTED_LIBRARIES */
struct S788 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* libraries */
};

/* Struct for type detachable ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
struct S789 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* dotnet_assemblies */
};

/* Struct for type detachable ET_ECF_FILE_RULES */
struct S790 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* file_rules */
};

/* Struct for type detachable ET_ECF_EXTERNAL_CFLAGS */
struct S791 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable ET_ECF_EXTERNAL_INCLUDES */
struct S792 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable ET_ECF_EXTERNAL_LIBRARIES */
struct S793 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable ET_ECF_EXTERNAL_LINKER_FLAGS */
struct S794 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable ET_ECF_EXTERNAL_MAKES */
struct S795 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable ET_ECF_EXTERNAL_OBJECTS */
struct S796 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable ET_ECF_EXTERNAL_RESOURCES */
struct S797 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] ET_ECF_ACTION] */
struct S798 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type detachable ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
struct S799 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* filename_id */
	T0* a3; /* universe */
	T0* a4; /* target */
	T0* a5; /* description */
	T0* a6; /* eifgens_location */
	T0* a7; /* classname_prefix */
	T1 a8; /* is_read_only */
	T1 a9; /* use_application_options */
	T0* a10; /* options */
	T0* a11; /* class_options */
	T0* a12; /* class_renamings */
	T0* a13; /* visible_classes */
	T0* a14; /* conditions */
	T0* a15; /* name */
	T0* a16; /* library */
};

/* Struct for type detachable ET_ECF_VERSION */
struct S801 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_major */
	T6 a2; /* internal_minor */
	T6 a3; /* internal_release */
	T6 a4; /* internal_build */
	T0* a5; /* product */
	T0* a6; /* company */
	T0* a7; /* copyright */
};

/* Struct for type detachable ET_ECF_CLUSTER */
struct S802 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* pathname */
	T1 a3; /* is_relative */
	T0* a4; /* universe */
	T0* a5; /* target */
	T0* a6; /* description */
	T1 a7; /* is_hidden */
	T0* a8; /* classname_prefix */
	T1 a9; /* is_read_only */
	T1 a10; /* is_recursive */
	T0* a11; /* conditioned_subclusters */
	T0* a12; /* conditioned_file_rules */
	T0* a13; /* options */
	T0* a14; /* class_options */
	T0* a15; /* class_renamings */
	T0* a16; /* class_mappings */
	T0* a17; /* visible_classes */
	T0* a18; /* provider_groups */
	T0* a19; /* conditions */
	T0* a20; /* notes */
	T1 a21; /* is_override */
	T0* a22; /* overridden_group */
	T1 a23; /* overridden_constraint_enabled */
	T1 a24; /* scm_mapping_constraint_enabled */
	T0* a25; /* subclusters */
	T1 a26; /* use_obsolete_syntax */
	T0* a27; /* scm_read_mapping */
	T0* a28; /* scm_write_mapping */
	T1 a29; /* is_implicit */
	T0* a30; /* file_rules */
	T0* a31; /* parent */
	T0* a32; /* provider_constraint */
	T0* a33; /* dependant_constraint */
	T1 a34; /* is_preparsed */
	T1 a35; /* is_abstract */
	T0* a36; /* cached_absolute_pathname */
};

/* Struct for type detachable ET_ECF_EXTERNAL_CFLAG */
struct S803 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* flag */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_EXTERNAL_INCLUDE */
struct S804 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_EXTERNAL_LIBRARY */
struct S805 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_EXTERNAL_LINKER_FLAG */
struct S806 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* flag */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_EXTERNAL_MAKE */
struct S807 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_EXTERNAL_OBJECT */
struct S808 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_EXTERNAL_RESOURCE */
struct S809 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type detachable ET_ECF_FILE_RULE */
struct S810 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* exclude */
	T0* a2; /* include */
	T0* a3; /* exclude_regexp */
	T0* a4; /* include_regexp */
	T0* a5; /* conditions */
	T0* a6; /* description */
};

/* Struct for type detachable ET_ECF_ACTION */
struct S811 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* command_name */
	T0* a2; /* description */
	T1 a3; /* must_succeed */
	T0* a4; /* working_directory */
	T0* a5; /* conditions */
};

/* Struct for type detachable DS_HASH_TABLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
struct S812 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* last_position */
	T6 a4; /* position */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* key_equality_tester */
	T0* a9; /* internal_keys */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] ET_ECF_VISIBLE_CLASS] */
struct S813 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_OPTIONS] */
struct S814 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_OPTIONS]] */
struct S815 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type detachable ET_ECF_ROOT_CLASS */
struct S816 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_name */
	T0* a2; /* creation_procedure_name */
	T0* a3; /* cluster_name */
};

/* Struct for type [detachable] ET_ECF_ROOT_ALL_CLASSES */
struct S817 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_ECF_ORED_CONDITIONS */
struct S818 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditions */
};

/* Struct for type detachable ET_ECF_ANDED_CONDITIONS */
struct S819 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditions */
};

/* Struct for type detachable ET_ECF_VISIBLE_CLASS */
struct S820 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_name */
	T0* a2; /* feature_name */
	T0* a3; /* new_class_name */
	T0* a4; /* new_feature_name */
};

/* Struct for type detachable ET_ECF_BUILD_CONDITION */
struct S822 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type detachable ET_ECF_CONCURRENCY_CONDITION */
struct S823 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type detachable ET_ECF_VOID_SAFETY_CONDITION */
struct S824 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type detachable ET_ECF_CUSTOM_CONDITION */
struct S825 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* value */
	T0* a3; /* match */
	T1 a4; /* is_excluded */
};

/* Struct for type detachable ET_ECF_DOTNET_CONDITION */
struct S826 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* value */
};

/* Struct for type detachable ET_ECF_DYNAMIC_RUNTIME_CONDITION */
struct S827 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* value */
};

/* Struct for type detachable ET_ECF_PLATFORM_CONDITION */
struct S828 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type [detachable] ET_ECF_COMPILER_VERSION_CONDITION */
struct S830 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* min_value */
	T0* a2; /* max_value */
};

/* Struct for type [detachable] ET_ECF_MSIL_CLR_VERSION_CONDITION */
struct S831 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* min_value */
	T0* a2; /* max_value */
};

/* Struct for type [detachable] UT_COUNTER */
struct S833 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [detachable] KL_AGENT_ROUTINES [[attached] ET_CLASS] */
struct S834 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] UT_COUNTER] */
struct S835 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_CLASS], [attached] PROCEDURE [[attached] TUPLE]] */
struct S836 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] ET_UNIVERSE] */
struct S837 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
struct S838 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]]] */
struct S839 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
struct S840 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY] */
struct S842 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* modulus */
	T6 a3; /* last_position */
	T6 a4; /* free_slot */
	T6 a5; /* position */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T0* a13; /* internal_cursor */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLY] */
struct S843 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
struct S844 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
struct S845 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY]] */
struct S846 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_INTERNAL_UNIVERSE] */
struct S847 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE]] */
struct S848 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]]] */
struct S849 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLIES] */
struct S850 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_SYSTEM_PROCESSOR] */
struct S851 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] ET_LIBRARY */
struct S852 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_system */
	T0* a2; /* library */
	T0* a3; /* name */
	T0* a4; /* libraries */
	T0* a5; /* clusters */
	T0* a6; /* dotnet_assemblies */
	T0* a7; /* implicit_attachment_type_mark */
	T0* a8; /* master_classes */
	T0* a9; /* master_class_mutex */
	T0* a10; /* any_type */
	T0* a11; /* detachable_any_type */
	T0* a12; /* detachable_separate_any_type */
	T0* a13; /* any_parent */
	T0* a14; /* any_parents */
	T0* a15; /* any_clients */
	T0* a16; /* tuple_type */
	T0* a17; /* detachable_tuple_type */
	T0* a18; /* tuple_identity_type */
	T0* a19; /* unfolded_empty_tuple_actual_parameters */
	T0* a20; /* array_any_type */
	T0* a21; /* array_detachable_any_type */
	T0* a22; /* array_none_type */
	T0* a23; /* array_identity_type */
	T0* a24; /* boolean_type */
	T0* a25; /* character_type */
	T0* a26; /* character_8_type */
	T0* a27; /* character_8_convert_feature */
	T0* a28; /* character_32_type */
	T0* a29; /* character_32_convert_feature */
	T0* a30; /* double_type */
	T0* a31; /* exception_type */
	T0* a32; /* detachable_exception_type */
	T0* a33; /* exception_manager_type */
	T0* a34; /* function_identity_any_type */
	T0* a35; /* immutable_string_8_type */
	T0* a36; /* immutable_string_32_type */
	T0* a37; /* integer_type */
	T0* a38; /* integer_8_type */
	T0* a39; /* integer_8_convert_feature */
	T0* a40; /* integer_16_type */
	T0* a41; /* integer_16_convert_feature */
	T0* a42; /* integer_32_type */
	T0* a43; /* integer_32_convert_feature */
	T0* a44; /* integer_64_type */
	T0* a45; /* integer_64_convert_feature */
	T0* a46; /* ise_exception_manager_type */
	T0* a47; /* iterable_detachable_separate_any_type */
	T0* a48; /* natural_type */
	T0* a49; /* natural_8_type */
	T0* a50; /* natural_8_convert_feature */
	T0* a51; /* natural_16_type */
	T0* a52; /* natural_16_convert_feature */
	T0* a53; /* natural_32_type */
	T0* a54; /* natural_32_convert_feature */
	T0* a55; /* natural_64_type */
	T0* a56; /* natural_64_convert_feature */
	T0* a57; /* none_type */
	T0* a58; /* detachable_none_type */
	T0* a59; /* pointer_type */
	T0* a60; /* predicate_identity_type */
	T0* a61; /* procedure_identity_type */
	T0* a62; /* real_type */
	T0* a63; /* real_32_type */
	T0* a64; /* real_32_convert_feature */
	T0* a65; /* real_64_type */
	T0* a66; /* real_64_convert_feature */
	T0* a67; /* routine_identity_type */
	T0* a68; /* special_any_type */
	T0* a69; /* special_detachable_any_type */
	T0* a70; /* special_identity_type */
	T0* a71; /* string_type */
	T0* a72; /* detachable_string_type */
	T0* a73; /* string_8_type */
	T0* a74; /* detachable_string_8_type */
	T0* a75; /* string_8_convert_feature */
	T0* a76; /* string_32_type */
	T0* a77; /* string_32_convert_feature */
	T0* a78; /* system_object_type */
	T0* a79; /* system_object_parents */
	T0* a80; /* system_string_type */
	T0* a81; /* type_detachable_any_type */
	T0* a82; /* detachable_type_exception_type */
	T0* a83; /* type_detachable_exception_type */
	T0* a84; /* type_detachable_like_current_type */
	T0* a85; /* type_identity_type */
	T0* a86; /* typed_pointer_identity_type */
	T0* a87; /* wide_character_type */
	T1 a88; /* is_read_only */
	T1 a89; /* is_preparsed */
};

/* Struct for type [detachable] TUPLE [[attached] ET_LIBRARY] */
struct S853 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_LIBRARY]] */
struct S854 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_UNIVERSE] */
struct S855 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* slots_position */
	T6 a9; /* count */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] TUPLE [[attached] ET_CLUSTER] */
struct S857 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]] */
struct S858 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_SYSTEM] */
struct S859 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE] */
struct S860 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* capacity */
	T6 a3; /* modulus */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T6 a8; /* clashes_previous_position */
	T0* a9; /* special_item_routines */
	T0* a10; /* item_storage */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T6 a13; /* found_position */
	T0* a14; /* internal_cursor */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] TUPLE [[attached] ET_AST_PROCESSOR] */
struct S861 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_MASTER_CLASS] */
struct S862 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
struct S863 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S864 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY]] */
struct S865 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] ET_ANCESTORS_STATUS_CHECKER */
struct S866 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_type_checker */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type [detachable] ET_FLATTENING_STATUS_CHECKER */
struct S867 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_type_checker */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type [detachable] ET_INTERFACE_STATUS_CHECKER */
struct S868 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* qualified_anchored_type_checker */
	T0* a2; /* class_type_checker */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type [detachable] ET_IMPLEMENTATION_STATUS_CHECKER */
struct S869 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]] */
struct S870 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ANCESTORS_STATUS_CHECKER] */
struct S872 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_FLATTENING_STATUS_CHECKER] */
struct S873 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_INTERFACE_STATUS_CHECKER] */
struct S874 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_IMPLEMENTATION_STATUS_CHECKER] */
struct S875 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [BOOLEAN] */
struct S876 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 z1;
};

/* Struct for type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_UNIVERSE]] */
struct S877 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]]] */
struct S880 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE]] */
struct S884 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]]] */
struct S885 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_CLASS], [attached] BOOLEAN]] */
struct S886 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] ET_NONE_GROUP */
struct S887 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* universe */
	T0* a2; /* name */
	T1 a3; /* use_obsolete_syntax */
	T0* a4; /* cached_absolute_pathname */
};

/* Struct for type detachable ET_PARENTHESIS_EXPRESSION */
struct S888 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* name */
	T0* a3; /* arguments */
	T6 a4; /* index */
};

/* Struct for type detachable DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S890 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] ET_FORMAL_PARAMETER */
struct S891 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* index */
	T0* a3; /* implementation_class */
	T0* a4; /* type_mark */
};

/* Struct for type [detachable] STD_FILES */
struct S892 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_output */
};

/* Struct for type [detachable] XM_LINKED_LIST [[attached] XM_ELEMENT_NODE] */
struct S894 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] UC_STRING_EQUALITY_TESTER */
struct S896 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
struct S898 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
struct S899 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE] */
struct S900 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first */
	T0* a2; /* second */
};

/* Struct for type [detachable] XM_EIFFEL_INPUT_STREAM */
struct S901 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_string */
	T6 a2; /* encoding */
	T0* a3; /* impl */
	T0* a4; /* utf_queue */
};

/* Struct for type [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
struct S902 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_LIBRARY] */
struct S903 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* modulus */
	T6 a3; /* last_position */
	T6 a4; /* free_slot */
	T6 a5; /* position */
	T0* a6; /* special_item_routines */
	T0* a7; /* item_storage */
	T0* a8; /* clashes */
	T0* a9; /* slots */
	T6 a10; /* found_position */
	T0* a11; /* internal_cursor */
	T6 a12; /* slots_position */
	T6 a13; /* count */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_LIBRARY]] */
struct S904 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XM_DOCUMENT_NODE] */
struct S905 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET] */
struct S906 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_UNIVERSE] */
struct S907 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET_PARENT] */
struct S909 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_QUERY] */
struct S910 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable ET_ALIAS_NAME_LIST */
struct S911 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_QUERY] */
struct S912 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_PROCEDURE] */
struct S913 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PROCEDURE] */
struct S914 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
struct S921 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_BASE_TYPE] */
struct S923 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE] */
struct S924 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] INTEGER_OVERFLOW_CHECKER */
struct S925 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_overflow_state1 */
	T0* a2; /* integer_overflow_state2 */
	T0* a3; /* natural_overflow_state1 */
	T0* a4; /* natural_overflow_state2 */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_NOTE_ELEMENT] */
struct S926 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_NOTE_ELEMENT] */
struct S927 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_NOTE_ELEMENT] */
struct S928 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type detachable C_DATE */
struct S929 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* millisecond_now */
	T0* a2; /* internal_item */
	T1 a3; /* is_utc */
};

/* Struct for type [detachable] YY_FILE_BUFFER */
struct S930 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T6 a2; /* index */
	T6 a3; /* line */
	T6 a4; /* column */
	T6 a5; /* position */
	T6 a6; /* count */
	T1 a7; /* filled */
	T0* a8; /* file */
	T1 a9; /* beginning_of_line */
	T6 a10; /* capacity */
	T1 a11; /* end_of_file */
	T1 a12; /* interactive */
};

/* Struct for type [detachable] KL_STDIN_FILE */
struct S931 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_file */
	T2 a2; /* last_character */
	T0* a3; /* last_string */
	T0* a4; /* character_buffer */
};

/* Struct for type [detachable] YY_BUFFER */
struct S932 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T6 a2; /* index */
	T6 a3; /* line */
	T6 a4; /* column */
	T6 a5; /* position */
	T6 a6; /* count */
	T1 a7; /* filled */
	T1 a8; /* beginning_of_line */
	T6 a9; /* capacity */
};

/* Struct for type [detachable] YY_UNICODE_FILE_BUFFER */
struct S936 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T6 a2; /* index */
	T6 a3; /* line */
	T6 a4; /* column */
	T6 a5; /* position */
	T1 a6; /* has_utf8_bom */
	T6 a7; /* count */
	T1 a8; /* filled */
	T6 a9; /* capacity */
	T1 a10; /* end_of_file */
	T1 a11; /* interactive */
	T6 a12; /* encoding */
	T0* a13; /* file */
	T6 a14; /* default_encoding */
	T1 a15; /* beginning_of_line */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [[attached] INTEGER_32] */
struct S937 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T6 a2; /* count */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S938 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
struct S939 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [detachable ET_KEYWORD] */
struct S940 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [detachable ET_SYMBOL] */
struct S941 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
struct S942 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [[attached] ET_OBJECT_TEST_LIST] */
struct S943 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [detachable ET_ITERATION_COMPONENT_LIST] */
struct S944 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [[attached] ET_ITERATION_COMPONENT_LIST] */
struct S945 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ASSERTION_ITEM] */
struct S946 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
struct S947 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_NAMED_CLASS] */
struct S948 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* slots_position */
	T0* a10; /* special_item_routines */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T6 a13; /* found_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
struct S949 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_KEYWORD] */
struct S950 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
struct S951 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_AGENT_KEYWORD */
struct S952 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* name */
};

/* Struct for type [detachable] SPECIAL [detachable ET_AGENT_KEYWORD] */
struct S953 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
struct S954 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_PRECURSOR_KEYWORD */
struct S955 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* name */
	T6 a4; /* seed */
};

/* Struct for type [detachable] SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
struct S956 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
struct S957 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_SYMBOL] */
struct S958 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
struct S959 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_POSITION] */
struct S960 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
struct S961 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
struct S963 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
struct S964 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_BREAK] */
struct S966 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
struct S967 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CHARACTER_CONSTANT] */
struct S969 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
struct S970 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CURRENT] */
struct S971 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
struct S972 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_FREE_OPERATOR */
struct S973 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* hash_code */
	T6 a3; /* compressed_position */
	T0* a4; /* break */
	T0* a5; /* operator_name */
	T6 a6; /* seed */
};

/* Struct for type [detachable] SPECIAL [detachable ET_FREE_OPERATOR] */
struct S974 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
struct S975 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_IDENTIFIER] */
struct S976 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
struct S977 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_INTEGER_CONSTANT] */
struct S979 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
struct S980 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_KEYWORD_OPERATOR */
struct S981 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
	T2 a2; /* code */
	T0* a3; /* text */
	T6 a4; /* seed */
};

/* Struct for type [detachable] SPECIAL [detachable ET_KEYWORD_OPERATOR] */
struct S982 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
struct S983 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_MANIFEST_STRING] */
struct S984 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
struct S985 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_REAL_CONSTANT] */
struct S987 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
struct S988 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_RESULT] */
struct S989 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
struct S990 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_RETRY_INSTRUCTION */
struct S991 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* text */
};

/* Struct for type [detachable] SPECIAL [detachable ET_RETRY_INSTRUCTION] */
struct S992 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
struct S993 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_SYMBOL_OPERATOR */
struct S994 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* break */
	T2 a2; /* code */
	T6 a3; /* compressed_position */
	T6 a4; /* seed */
};

/* Struct for type [detachable] SPECIAL [detachable ET_SYMBOL_OPERATOR] */
struct S995 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_VOID] */
struct S996 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_VOID */
struct S997 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
	T0* a4; /* text */
};

/* Struct for type [detachable] SPECIAL [detachable ET_VOID] */
struct S998 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
struct S999 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_SEMICOLON_SYMBOL */
struct S1000 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
};

/* Struct for type [detachable] SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
struct S1001 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
struct S1002 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_BRACKET_SYMBOL] */
struct S1003 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
struct S1004 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_QUESTION_MARK_SYMBOL */
struct S1005 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T6 a3; /* index */
};

/* Struct for type [detachable] SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
struct S1006 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
struct S1007 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ACROSS_EXPRESSION */
struct S1008 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* item_name */
	T0* a3; /* as_keyword */
	T0* a4; /* unfolded_cursor_name */
	T0* a5; /* across_keyword */
	T0* a6; /* iterable_expression */
	T0* a7; /* until_conditional */
	T0* a8; /* iteration_conditional */
	T0* a9; /* end_keyword */
	T1 a10; /* is_all */
	T1 a11; /* has_cursor_name */
	T0* a12; /* invariant_part */
	T0* a13; /* variant_part */
	T0* a14; /* new_cursor_expression */
	T0* a15; /* cursor_item_expression */
	T0* a16; /* cursor_after_expression */
	T0* a17; /* cursor_forth_instruction */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ACROSS_EXPRESSION] */
struct S1009 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
struct S1010 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ACROSS_INSTRUCTION */
struct S1011 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item_name */
	T0* a2; /* as_keyword */
	T0* a3; /* unfolded_cursor_name */
	T0* a4; /* across_keyword */
	T0* a5; /* iterable_expression */
	T0* a6; /* from_compound */
	T0* a7; /* until_conditional */
	T0* a8; /* loop_compound */
	T0* a9; /* end_keyword */
	T1 a10; /* has_cursor_name */
	T0* a11; /* invariant_part */
	T0* a12; /* variant_part */
	T0* a13; /* new_cursor_expression */
	T0* a14; /* cursor_item_expression */
	T0* a15; /* cursor_after_expression */
	T0* a16; /* cursor_forth_instruction */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
struct S1012 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
struct S1013 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
struct S1014 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
struct S1015 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
struct S1016 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
struct S1017 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
struct S1018 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
struct S1019 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
struct S1020 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S1021 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S1023 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
struct S1024 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_AGENT_ARGUMENT_OPERAND_LIST */
struct S1025 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_parenthesis */
	T0* a2; /* right_parenthesis */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
struct S1026 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
struct S1027 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_AGENT_TARGET] */
struct S1028 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
struct S1029 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ALIAS_NAME */
struct S1030 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* alias_keyword */
	T0* a2; /* alias_string */
	T2 a3; /* code */
	T0* a4; /* convert_keyword */
	T6 a5; /* seed */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ALIAS_NAME] */
struct S1031 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME_LIST] */
struct S1032 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_ALIAS_NAME_LIST] */
struct S1033 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
struct S1034 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_ASSIGNER] */
struct S1036 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
struct S1037 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_BRACKET_EXPRESSION */
struct S1038 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* target */
	T0* a3; /* name */
	T0* a4; /* arguments */
};

/* Struct for type [detachable] SPECIAL [detachable ET_BRACKET_EXPRESSION] */
struct S1039 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
struct S1040 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CALL_AGENT */
struct S1041 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* target */
	T0* a4; /* qualified_name */
	T0* a5; /* arguments */
	T0* a6; /* implicit_result */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CALL_AGENT] */
struct S1042 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S1043 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S1044 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
struct S1045 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CHECK_INSTRUCTION */
struct S1046 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* check_keyword */
	T0* a2; /* end_keyword */
	T6 a3; /* count */
	T0* a4; /* storage */
	T0* a5; /* then_compound */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CHECK_INSTRUCTION] */
struct S1047 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
struct S1048 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CHOICE] */
struct S1050 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
struct S1051 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CHOICE_CONSTANT] */
struct S1053 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
struct S1054 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CHOICE_ITEM] */
struct S1056 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
struct S1057 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CHOICE_LIST */
struct S1058 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* when_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CHOICE_LIST] */
struct S1059 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
struct S1060 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CLASS] */
struct S1061 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
struct S1062 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CLIENT_ITEM] */
struct S1063 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
struct S1064 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CLIENTS */
struct S1065 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* right_brace */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CLIENTS] */
struct S1066 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
struct S1067 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_COMPOUND] */
struct S1068 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
struct S1069 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONDITIONAL] */
struct S1071 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
struct S1072 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONSTANT] */
struct S1073 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1074 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1076 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
struct S1077 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
struct S1078 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_bracket */
	T0* a4; /* right_bracket */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
struct S1079 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
struct S1080 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CONSTRAINT_CREATOR */
struct S1081 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* create_keyword */
	T0* a2; /* end_keyword */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
struct S1082 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_RENAME_LIST] */
struct S1083 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CONSTRAINT_RENAME_LIST */
struct S1084 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* end_keyword */
	T0* a2; /* rename_keyword */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONSTRAINT_RENAME_LIST] */
struct S1085 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
struct S1086 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONSTRAINT_TYPE] */
struct S1088 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
struct S1089 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONVERT_FEATURE] */
struct S1091 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
struct S1092 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
struct S1094 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
struct S1095 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CONVERT_FEATURE_LIST */
struct S1096 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* convert_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
struct S1097 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
struct S1098 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_CREATE_EXPRESSION] */
struct S1099 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
struct S1100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CREATION_REGION */
struct S1101 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_name */
	T0* a2; /* left_symbol */
	T0* a3; /* right_symbol */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CREATION_REGION] */
struct S1102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
struct S1103 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CREATOR */
struct S1104 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* create_keyword */
	T0* a2; /* clients */
	T0* a3; /* storage */
	T6 a4; /* count */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CREATOR] */
struct S1105 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
struct S1106 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_CREATOR_LIST */
struct S1107 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_CREATOR_LIST] */
struct S1108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
struct S1109 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_DEBUG_INSTRUCTION */
struct S1110 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* keys */
	T0* a2; /* compound */
	T0* a3; /* end_keyword */
};

/* Struct for type [detachable] SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
struct S1111 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
struct S1112 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ELSEIF_EXPRESSION */
struct S1113 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* then_keyword */
	T0* a3; /* then_expression */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
struct S1114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
struct S1115 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ELSEIF_EXPRESSION_LIST */
struct S1116 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
struct S1117 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
struct S1118 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ELSEIF_PART */
struct S1119 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* then_compound */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ELSEIF_PART] */
struct S1120 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
struct S1121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_ELSEIF_PART_LIST */
struct S1122 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ELSEIF_PART_LIST] */
struct S1123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
struct S1124 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_EXPORT] */
struct S1126 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
struct S1127 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_EXPORT_LIST] */
struct S1128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
struct S1129 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_EXPRESSION] */
struct S1130 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
struct S1131 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_EXPRESSION_ITEM] */
struct S1132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
struct S1133 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
struct S1134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
struct S1135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_EXTERNAL_ALIAS] */
struct S1137 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
struct S1138 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_FEATURE_CLAUSE */
struct S1139 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_keyword */
	T0* a2; /* clients */
};

/* Struct for type [detachable] SPECIAL [detachable ET_FEATURE_CLAUSE] */
struct S1140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
struct S1141 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_FEATURE_CLAUSE_LIST */
struct S1142 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
struct S1143 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
struct S1144 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_FEATURE_EXPORT */
struct S1145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clients_clause */
	T0* a2; /* storage */
	T6 a3; /* count */
};

/* Struct for type [detachable] SPECIAL [detachable ET_FEATURE_EXPORT] */
struct S1146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
struct S1147 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FEATURE_NAME] */
struct S1148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
struct S1149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
struct S1151 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
struct S1152 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT] */
struct S1153 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
struct S1154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
struct S1156 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S1157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S1158 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
struct S1159 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER] */
struct S1160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
struct S1161 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
struct S1163 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
struct S1164 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
struct S1165 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
struct S1166 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_IF_EXPRESSION */
struct S1167 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* conditional */
	T0* a3; /* then_keyword */
	T0* a4; /* then_expression */
	T0* a5; /* else_keyword */
	T0* a6; /* else_expression */
	T0* a7; /* end_keyword */
	T0* a8; /* elseif_parts */
};

/* Struct for type [detachable] SPECIAL [detachable ET_IF_EXPRESSION] */
struct S1168 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
struct S1169 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_IF_INSTRUCTION */
struct S1170 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* then_compound */
	T0* a3; /* end_keyword */
	T0* a4; /* elseif_parts */
	T0* a5; /* else_compound */
};

/* Struct for type [detachable] SPECIAL [detachable ET_IF_INSTRUCTION] */
struct S1171 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_LIST] */
struct S1172 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_INDEXING_LIST */
struct S1173 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* indexing_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_INDEXING_LIST] */
struct S1174 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_ITEM] */
struct S1175 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_INDEXING_ITEM] */
struct S1177 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM] */
struct S1178 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_INDEXING_TERM] */
struct S1179 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_ITEM] */
struct S1180 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_INDEXING_TERM_ITEM] */
struct S1182 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_LIST] */
struct S1183 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_INDEXING_TERM_LIST */
struct S1184 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_INDEXING_TERM_LIST] */
struct S1185 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
struct S1186 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_INLINE_AGENT] */
struct S1187 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_EXPRESSION] */
struct S1188 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_INSPECT_EXPRESSION */
struct S1189 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* conditional */
	T0* a3; /* when_parts */
	T0* a4; /* end_keyword */
	T0* a5; /* else_part */
};

/* Struct for type [detachable] SPECIAL [detachable ET_INSPECT_EXPRESSION] */
struct S1190 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
struct S1191 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_INSPECT_INSTRUCTION */
struct S1192 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* when_parts */
	T0* a3; /* end_keyword */
	T0* a4; /* else_compound */
};

/* Struct for type [detachable] SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
struct S1193 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
struct S1194 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_INSTRUCTION] */
struct S1196 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
struct S1197 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_INVARIANTS */
struct S1198 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_class */
	T0* a2; /* invariant_keyword */
	T0* a3; /* storage */
	T6 a4; /* count */
	T0* a5; /* object_tests */
	T0* a6; /* iteration_components */
	T1 a7; /* validity_checked */
	T1 a8; /* has_validity_error */
};

/* Struct for type [detachable] SPECIAL [detachable ET_INVARIANTS] */
struct S1199 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
struct S1200 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
struct S1201 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
struct S1202 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_LIKE_TYPE] */
struct S1204 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
struct S1205 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_LOCAL_VARIABLE */
struct S1206 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T6 a3; /* index */
	T6 a4; /* attached_index */
	T1 a5; /* is_used */
};

/* Struct for type [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE] */
struct S1207 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
struct S1208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
struct S1210 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
struct S1211 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_LOCAL_VARIABLE_LIST */
struct S1212 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* local_keyword */
};

/* Struct for type [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
struct S1213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
struct S1214 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_LOOP_INVARIANTS */
struct S1215 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* invariant_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_LOOP_INVARIANTS] */
struct S1216 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
struct S1217 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_MANIFEST_ARRAY */
struct S1218 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T0* a3; /* left_symbol */
	T6 a4; /* count */
	T0* a5; /* storage */
	T0* a6; /* right_symbol */
};

/* Struct for type [detachable] SPECIAL [detachable ET_MANIFEST_ARRAY] */
struct S1219 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
struct S1220 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
struct S1222 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
struct S1223 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_MANIFEST_STRING_LIST */
struct S1224 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_parenthesis */
	T0* a2; /* right_parenthesis */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
struct S1225 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
struct S1226 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_MANIFEST_TUPLE] */
struct S1227 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
struct S1228 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_OBSOLETE] */
struct S1230 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
struct S1231 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_PARENTHESIZED_EXPRESSION */
struct S1232 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* left_parenthesis */
	T0* a4; /* right_parenthesis */
};

/* Struct for type [detachable] SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
struct S1233 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
struct S1234 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_PARENT] */
struct S1235 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
struct S1236 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_PARENT_CLAUSE_LIST */
struct S1237 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
struct S1238 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
struct S1239 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_PARENT_ITEM] */
struct S1240 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
struct S1241 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_PARENT_LIST] */
struct S1242 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
struct S1243 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_POSTCONDITIONS */
struct S1244 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* validity_checked */
	T1 a2; /* has_validity_error */
	T6 a3; /* count */
	T0* a4; /* storage */
	T0* a5; /* ensure_keyword */
	T0* a6; /* then_keyword */
};

/* Struct for type [detachable] SPECIAL [detachable ET_POSTCONDITIONS] */
struct S1245 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
struct S1246 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_PRECONDITIONS] */
struct S1247 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
struct S1248 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_PROCEDURE] */
struct S1249 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
struct S1250 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
struct S1252 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUANTIFIER_EXPRESSION] */
struct S1253 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_QUANTIFIER_EXPRESSION */
struct S1254 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* item_name */
	T0* a3; /* unfolded_cursor_name */
	T0* a4; /* quantifier_symbol */
	T0* a5; /* colon_symbol */
	T0* a6; /* iterable_expression */
	T0* a7; /* bar_symbol */
	T0* a8; /* iteration_expression */
	T1 a9; /* is_all */
	T0* a10; /* new_cursor_expression */
	T0* a11; /* cursor_item_expression */
	T0* a12; /* cursor_after_expression */
	T0* a13; /* cursor_forth_instruction */
};

/* Struct for type [detachable] SPECIAL [detachable ET_QUANTIFIER_EXPRESSION] */
struct S1255 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
struct S1256 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_QUERY] */
struct S1257 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
struct S1258 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_RENAME_ITEM] */
struct S1260 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
struct S1261 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_RENAME_LIST] */
struct S1262 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_REPEAT_INSTRUCTION] */
struct S1263 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_REPEAT_INSTRUCTION */
struct S1264 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item_name */
	T0* a2; /* unfolded_cursor_name */
	T0* a3; /* open_repeat_symbol */
	T0* a4; /* colon_symbol */
	T0* a5; /* iterable_expression */
	T0* a6; /* bar_symbol */
	T0* a7; /* loop_compound */
	T0* a8; /* close_repeat_symbol */
	T0* a9; /* new_cursor_expression */
	T0* a10; /* cursor_item_expression */
	T0* a11; /* cursor_after_expression */
	T0* a12; /* cursor_forth_instruction */
};

/* Struct for type [detachable] SPECIAL [detachable ET_REPEAT_INSTRUCTION] */
struct S1265 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
struct S1266 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_STATIC_CALL_EXPRESSION */
struct S1267 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* feature_keyword */
	T0* a3; /* static_type */
	T0* a4; /* qualified_name */
	T0* a5; /* arguments */
	T0* a6; /* parenthesis_call */
};

/* Struct for type [detachable] SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
struct S1268 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
struct S1269 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_STRIP_EXPRESSION */
struct S1270 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* strip_keyword */
	T0* a3; /* left_parenthesis */
	T6 a4; /* count */
	T0* a5; /* storage */
	T0* a6; /* right_parenthesis */
};

/* Struct for type [detachable] SPECIAL [detachable ET_STRIP_EXPRESSION] */
struct S1271 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
struct S1272 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_TYPE] */
struct S1273 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT] */
struct S1274 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT] */
struct S1276 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_ITEM] */
struct S1277 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT_ITEM] */
struct S1279 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_LIST] */
struct S1280 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_TYPE_CONSTRAINT_LIST */
struct S1281 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* right_brace */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT_LIST] */
struct S1282 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
struct S1283 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_TYPE_ITEM] */
struct S1285 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
struct S1286 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_TYPE_LIST] */
struct S1288 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
struct S1289 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_VARIANT */
struct S1290 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* variant_keyword */
	T0* a2; /* tag */
	T0* a3; /* expression */
};

/* Struct for type [detachable] SPECIAL [detachable ET_VARIANT] */
struct S1291 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION] */
struct S1292 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_WHEN_EXPRESSION */
struct S1293 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* choices */
	T0* a2; /* then_keyword */
	T0* a3; /* then_expression */
};

/* Struct for type [detachable] SPECIAL [detachable ET_WHEN_EXPRESSION] */
struct S1294 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION_LIST] */
struct S1295 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_WHEN_EXPRESSION_LIST */
struct S1296 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_WHEN_EXPRESSION_LIST] */
struct S1297 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
struct S1298 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_WHEN_PART */
struct S1299 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* choices */
	T0* a2; /* then_compound */
};

/* Struct for type [detachable] SPECIAL [detachable ET_WHEN_PART] */
struct S1300 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
struct S1301 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable ET_WHEN_PART_LIST */
struct S1302 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [detachable ET_WHEN_PART_LIST] */
struct S1303 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
struct S1304 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable ET_WRITABLE] */
struct S1305 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_CLASS] */
struct S1306 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sorted_items */
	T0* a2; /* cycle */
	T0* a3; /* indexes */
	T0* a4; /* counts */
	T0* a5; /* successors */
	T0* a6; /* items */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
struct S1307 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T0* a4; /* item_storage */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] ET_PARENT_CHECKER1 */
struct S1308 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T0* a4; /* current_parent */
};

/* Struct for type [detachable] ET_FORMAL_PARAMETER_CHECKER1 */
struct S1309 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* all_base_types */
	T0* a3; /* base_types */
	T0* a4; /* formal_dependencies */
	T0* a5; /* base_type_dependencies */
	T0* a6; /* recursive_formal_constraints */
	T0* a7; /* recursive_formal_constraints_to_be_processed */
	T0* a8; /* current_class */
	T0* a9; /* system_processor */
	T0* a10; /* current_formal */
	T0* a11; /* current_type_constraint */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
struct S1310 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T6 a4; /* position */
	T0* a5; /* item_storage */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_keys */
	T6 a8; /* last_position */
	T6 a9; /* free_slot */
	T6 a10; /* modulus */
	T6 a11; /* slots_position */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* equality_tester */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T6 a17; /* capacity */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] ET_FEATURE_NAME_TESTER */
struct S1311 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
struct S1313 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_storage */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* position */
	T6 a7; /* last_position */
	T6 a8; /* capacity */
	T6 a9; /* count */
	T6 a10; /* free_slot */
	T0* a11; /* equality_tester */
	T6 a12; /* modulus */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] ET_ALIAS_NAME_TESTER */
struct S1314 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_CLIENT_LIST] */
struct S1316 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S1317 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* special_key_routines */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T6 a16; /* count */
	T0* a17; /* internal_keys */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] ET_FEATURE_ADAPTATION_RESOLVER */
struct S1318 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* rename_table */
	T0* a3; /* export_table */
	T0* a4; /* undefine_table */
	T0* a5; /* redefine_table */
	T0* a6; /* select_table */
	T0* a7; /* replicable_features */
	T0* a8; /* current_class */
	T0* a9; /* system_processor */
	T0* a10; /* free_parent_feature */
	T0* a11; /* parent_feature_list */
	T0* a12; /* free_inherited_feature */
	T0* a13; /* inherited_feature_list */
	T0* a14; /* free_redeclared_feature */
	T0* a15; /* redeclared_feature_list */
};

/* Struct for type [detachable] ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
struct S1319 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* dotnet_features */
	T0* a3; /* other_dotnet_features */
	T0* a4; /* current_class */
	T0* a5; /* system_processor */
	T0* a6; /* free_parent_feature */
	T0* a7; /* parent_feature_list */
	T0* a8; /* free_inherited_feature */
	T0* a9; /* inherited_feature_list */
	T0* a10; /* free_redeclared_feature */
	T0* a11; /* redeclared_feature_list */
};

/* Struct for type [detachable] ET_IDENTIFIER_TYPE_RESOLVER */
struct S1320 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T0* a4; /* current_feature */
};

/* Struct for type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
struct S1321 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constraint_context */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type [detachable] ET_ANCHORED_TYPE_CHECKER */
struct S1322 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* anchored_type_sorter */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
	T0* a5; /* current_anchored_type */
};

/* Struct for type [detachable] ET_SIGNATURE_CHECKER */
struct S1323 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* parent_context */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
	T6 a5; /* processing_mode */
};

/* Struct for type [detachable] ET_PARENT_CHECKER2 */
struct S1324 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* constraint_context */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type [detachable] ET_FORMAL_PARAMETER_CHECKER2 */
struct S1325 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* constraint_context */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type [detachable] ET_BUILTIN_FEATURE_CHECKER */
struct S1326 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* builtin_features */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type [detachable] ET_PRECURSOR_CHECKER */
struct S1327 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_feature */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_FEATURE, [attached] INTEGER_32] */
struct S1328 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* item_storage */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* free_slot */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T6 a11; /* found_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T0* a19; /* internal_keys */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_CLASS] */
struct S1329 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T0* a3; /* clashes */
	T0* a4; /* item_storage */
	T6 a5; /* capacity */
	T6 a6; /* modulus */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T0* a9; /* special_item_routines */
	T0* a10; /* slots */
	T6 a11; /* found_position */
	T0* a12; /* internal_cursor */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
struct S1330 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class_impl */
	T0* a3; /* current_class */
	T0* a4; /* adapted_base_class_checker */
	T0* a5; /* adapted_base_classes */
	T0* a6; /* target_context */
	T0* a7; /* other_context */
	T0* a8; /* classes_to_be_processed */
	T0* a9; /* system_processor */
	T1 a10; /* in_qualified_anchored_type */
};

/* Struct for type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
struct S1331 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constraint_context */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_FEATURE_NAME] */
struct S1332 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* found_position */
	T6 a3; /* position */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
	T0* a6; /* item_storage */
	T0* a7; /* key_equality_tester */
	T0* a8; /* internal_keys */
	T6 a9; /* last_position */
	T6 a10; /* free_slot */
	T6 a11; /* modulus */
	T6 a12; /* slots_position */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* equality_tester */
	T0* a15; /* key_storage */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_ALIAS_NAME] */
struct S1333 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* position */
	T6 a3; /* found_position */
	T0* a4; /* item_storage */
	T0* a5; /* key_storage */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_keys */
	T6 a8; /* count */
	T6 a9; /* last_position */
	T6 a10; /* free_slot */
	T6 a11; /* modulus */
	T6 a12; /* slots_position */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* equality_tester */
	T0* a15; /* internal_cursor */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] ET_RENAME */
struct S1337 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* old_name */
	T0* a2; /* as_keyword */
	T0* a3; /* new_name */
};

/* Struct for type [detachable] ET_PARENT_CHECKER3 */
struct S1338 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_PROCEDURE] */
struct S1339 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* count */
	T0* a9; /* special_item_routines */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_QUERY] */
struct S1340 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* count */
	T0* a9; /* special_item_routines */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [detachable] ET_SUPPLIER_BUILDER */
struct S1341 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* supplier_classes */
	T0* a2; /* current_class */
};

/* Struct for type detachable ET_PARENT_FEATURE */
struct S1344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* parent */
	T0* a3; /* precursor_feature */
	T0* a4; /* new_name */
	T0* a5; /* undefine_name */
	T0* a6; /* redefine_name */
	T0* a7; /* select_name */
	T0* a8; /* merged_feature */
	T1 a9; /* has_other_deferred */
	T1 a10; /* has_other_effective */
};

/* Struct for type [detachable] ET_INHERITED_FEATURE */
struct S1345 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* parent_feature */
	T0* a3; /* flattened_feature */
	T0* a4; /* flattened_parent */
	T6 a5; /* first_seed */
	T0* a6; /* other_seeds */
	T1 a7; /* is_selected */
	T0* a8; /* replicated_features */
	T0* a9; /* replicated_seeds */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CLASS] */
struct S1346 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLASS] */
struct S1347 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLASS] */
struct S1348 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_STACK [[attached] YY_BUFFER] */
struct S1349 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
};

/* Struct for type [detachable] UT_SYNTAX_ERROR */
struct S1350 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] UT_TOO_MANY_INCLUDES_ERROR */
struct S1351 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type detachable ET_DYNAMIC_TARGET_LIST */
struct S1353 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] SPECIAL [[attached] NATIVE_STRING] */
struct S1354 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] ARRAY [detachable STRING_8] */
struct S1355 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] XM_EIFFEL_CHARACTER_ENTITY */
struct S1356 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* code */
};

/* Struct for type detachable DS_BILINKABLE [[attached] XM_POSITION] */
struct S1358 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_POSITION] */
struct S1359 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type detachable DS_LINKABLE [[attached] XM_EIFFEL_SCANNER] */
struct S1360 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] SPECIAL [[attached] XM_EIFFEL_ENTITY_DEF] */
struct S1361 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
struct S1364 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_ENTITY_DEF] */
struct S1365 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] XM_EIFFEL_PARSER_NAME] */
struct S1368 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] XM_EIFFEL_PARSER_NAME] */
struct S1369 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_BILINKED_LIST [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S1370 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T6 a2; /* count */
	T0* a3; /* last_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S1371 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type detachable DS_BILINKABLE [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S1372 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [detachable] DS_BILINKED_LIST_CURSOR [[attached] STRING_8] */
struct S1373 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type detachable DS_BILINKABLE [[attached] STRING_8] */
struct S1374 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [detachable] XM_NAMESPACE_RESOLVER_CONTEXT */
struct S1375 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* context */
};

/* Struct for type [detachable] DS_LINKED_QUEUE [detachable STRING_8] */
struct S1378 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T6 a2; /* count */
	T0* a3; /* last_cell */
};

/* Struct for type [detachable] DS_LINKED_QUEUE [[attached] STRING_8] */
struct S1379 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T0* a2; /* last_cell */
	T6 a3; /* count */
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] XM_NAMESPACE] */
struct S1380 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] XM_NAMESPACE] */
struct S1381 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_NAMESPACE] */
struct S1382 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XM_NAMESPACE] */
struct S1383 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_CLUSTER] */
struct S1384 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_PARENT_ITEM] */
struct S1386 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_ITEM] */
struct S1387 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_BRACED_TYPE_LIST */
struct S1388 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* right_brace */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] ET_MASTER_CLASS] */
struct S1389 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* equality_tester */
	T0* a6; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ADAPTED_LIBRARY] */
struct S1390 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_LIBRARY] */
struct S1391 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_LIBRARY] */
struct S1392 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type detachable DS_LINKABLE [[attached] XM_DOCUMENT_NODE] */
struct S1393 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1395 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1396 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1397 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_DOTNET_ASSEMBLY] */
struct S1398 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
	T0* a6; /* equality_tester */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_FEATURE, [attached] INTEGER_32] */
struct S1400 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
struct S1402 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CALL_NAME] */
struct S1404 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
struct S1406 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CALL_NAME] */
struct S1407 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL, [attached] ET_CALL_NAME] */
struct S1408 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, [attached] ET_CALL_NAME] */
struct S1410 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
struct S1411 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
struct S1412 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S1413 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] SPECIAL [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1414 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1415 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1416 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_SYSTEM_PROCESSOR] */
struct S1418 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_SYSTEM_PROCESSOR] */
struct S1419 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_SYSTEM_PROCESSOR] */
struct S1420 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_CLASS] */
struct S1421 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ADAPTED_CLASS] */
struct S1423 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_CLASS] */
struct S1424 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_PROCEDURE] */
struct S1425 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUERY] */
struct S1426 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_NAMED_OBJECT_TEST] */
struct S1427 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ITERATION_COMPONENT] */
struct S1428 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] INTEGER_32] */
struct S1429 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_8] */
struct S1433 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
struct S1434 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] RX_CHARACTER_SET] */
struct S1435 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] RX_CHARACTER_SET] */
struct S1436 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_64] */
struct S1443 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_CHARACTER_BUFFER */
struct S1446 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* as_special */
	T0* a2; /* area */
};

/* Struct for type [detachable] UT_TRISTATE */
struct S1448 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* value */
};

/* Struct for type [detachable] KL_AGENT_ROUTINES [[attached] ANY] */
struct S1449 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] UT_TRISTATE] */
struct S1450 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ANY], [attached] PROCEDURE [[attached] TUPLE]] */
struct S1451 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
struct S1452 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE] */
struct S1453 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
struct S1455 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* last_position */
	T6 a3; /* capacity */
	T6 a4; /* slots_position */
	T6 a5; /* count */
	T6 a6; /* modulus */
	T6 a7; /* clashes_previous_position */
	T0* a8; /* key_equality_tester */
	T0* a9; /* equality_tester */
	T6 a10; /* found_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* item_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_key_routines */
	T0* a16; /* key_storage */
	T6 a17; /* free_slot */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
struct S1456 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S1457 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [detachable] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE] */
struct S1458 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
struct S1459 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE]] */
struct S1460 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE], [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]]] */
struct S1461 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
struct S1463 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN]] */
struct S1464 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
struct S1465 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
struct S1466 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T6 a2; /* count */
	T1 a3; /* is_never_void */
	T0* a4; /* dynamic_types */
	T0* a5; /* agent_type */
	T1 a6; /* is_dynamic_types_readonly */
};

/* Struct for type [detachable] XM_COMMENT */
struct S1469 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* data */
	T0* a2; /* parent */
};

/* Struct for type [detachable] XM_PROCESSING_INSTRUCTION */
struct S1470 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* data */
	T0* a3; /* parent */
};

/* Struct for type [detachable] XM_CHARACTER_DATA */
struct S1471 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T0* a2; /* parent */
};

/* Struct for type [detachable] ET_LIKE_N */
struct S1474 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* previous */
	T0* a3; /* type_mark */
	T0* a4; /* like_keyword */
};

/* Struct for type [detachable] KL_STRING_INPUT_STREAM */
struct S1477 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_input */
	T2 a2; /* last_character */
	T0* a3; /* last_string */
	T0* a4; /* string */
	T6 a5; /* location */
};

/* Struct for type [detachable] KL_UNICODE_CHARACTER_BUFFER */
struct S1479 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* as_unicode_special */
	T2 a2; /* invalid_character */
	T0* a3; /* unicode_area */
	T0* a4; /* area */
	T3 a5; /* invalid_unicode_character */
};

/* Struct for type detachable ET_OBJECT_TEST_LIST */
struct S1481 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type detachable ET_C3_CHARACTER_CONSTANT */
struct S1482 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T1 a3; /* has_invalid_code */
	T3 a4; /* value */
	T6 a5; /* compressed_position */
	T0* a6; /* literal */
	T0* a7; /* type */
};

/* Struct for type detachable ET_REGULAR_MANIFEST_STRING */
struct S1483 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* type */
};

/* Struct for type detachable ET_SPECIAL_MANIFEST_STRING */
struct S1484 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T1 a4; /* has_invalid_code */
	T6 a5; /* compressed_position */
	T0* a6; /* literal */
	T0* a7; /* type */
};

/* Struct for type detachable ET_VERBATIM_STRING */
struct S1485 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* marker */
	T0* a7; /* open_white_characters */
	T0* a8; /* close_white_characters */
	T1 a9; /* is_left_aligned */
	T0* a10; /* type */
};

/* Struct for type detachable ET_UNDERSCORED_INTEGER_CONSTANT */
struct S1486 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* sign */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type detachable ET_HEXADECIMAL_INTEGER_CONSTANT */
struct S1487 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* sign */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type detachable ET_OCTAL_INTEGER_CONSTANT */
struct S1488 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* sign */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type detachable ET_BINARY_INTEGER_CONSTANT */
struct S1489 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* sign */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type detachable ET_REGULAR_REAL_CONSTANT */
struct S1490 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* type */
};

/* Struct for type detachable ET_UNDERSCORED_REAL_CONSTANT */
struct S1491 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* type */
};

/* Struct for type detachable ET_TRUE_CONSTANT */
struct S1493 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
	T0* a4; /* text */
};

/* Struct for type detachable ET_FALSE_CONSTANT */
struct S1494 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
	T0* a4; /* text */
};

/* Struct for type detachable ET_C1_CHARACTER_CONSTANT */
struct S1495 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T6 a3; /* compressed_position */
	T3 a4; /* value */
	T0* a5; /* type */
};

/* Struct for type detachable ET_C2_CHARACTER_CONSTANT */
struct S1496 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T6 a3; /* compressed_position */
	T3 a4; /* value */
	T0* a5; /* type */
};

/* Struct for type detachable ET_TAGGED_INDEXING */
struct S1498 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* tag */
	T0* a2; /* terms */
};

/* Struct for type detachable ET_INDEXING */
struct S1499 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* terms */
};

/* Struct for type detachable ET_CUSTOM_ATTRIBUTE */
struct S1500 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* creation_expression */
	T0* a2; /* settings */
	T0* a3; /* end_keyword */
};

/* Struct for type detachable ET_CONSTRAINED_FORMAL_PARAMETER */
struct S1501 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* arrow_symbol */
	T0* a2; /* constraint */
	T0* a3; /* creation_procedures */
	T0* a4; /* constraint_base_types */
	T0* a5; /* implementation_class */
	T0* a6; /* type_mark */
	T6 a7; /* index */
	T0* a8; /* name */
	T0* a9; /* recursive_formal_constraints */
};

/* Struct for type detachable ET_TYPE_RENAME_CONSTRAINT */
struct S1502 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* renames */
};

/* Struct for type detachable ET_CONSTRAINT_NAMED_TYPE */
struct S1503 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
};

/* Struct for type detachable ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
struct S1504 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* label */
	T0* a2; /* colon */
	T0* a3; /* type */
};

/* Struct for type detachable ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
struct S1505 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* label */
	T0* a2; /* comma */
	T0* a3; /* type */
};

/* Struct for type detachable ET_ALL_EXPORT */
struct S1506 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clients_clause */
	T0* a2; /* all_keyword */
};

/* Struct for type detachable ET_CONVERT_FUNCTION */
struct S1508 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* types */
	T0* a3; /* colon */
};

/* Struct for type detachable ET_CONVERT_PROCEDURE */
struct S1509 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* types */
	T0* a3; /* left_parenthesis */
	T0* a4; /* right_parenthesis */
};

/* Struct for type detachable ET_DO_FUNCTION */
struct S1511 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T0* a8; /* preconditions */
	T6 a9; /* hash_code */
	T0* a10; /* arguments */
	T0* a11; /* postconditions */
	T0* a12; /* assigner */
	T0* a13; /* extended_name */
	T0* a14; /* declared_type */
	T0* a15; /* frozen_keyword */
	T0* a16; /* obsolete_message */
	T0* a17; /* locals */
	T0* a18; /* compound */
	T0* a19; /* rescue_clause */
	T0* a20; /* clients */
	T0* a21; /* is_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* first_indexing */
	T0* a26; /* object_tests */
	T1 a27; /* validity_checked */
	T1 a28; /* has_validity_error */
	T6 a29; /* id */
	T6 a30; /* version */
	T0* a31; /* synonym */
};

/* Struct for type detachable ET_ONCE_FUNCTION */
struct S1512 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T0* a8; /* preconditions */
	T6 a9; /* hash_code */
	T0* a10; /* arguments */
	T0* a11; /* postconditions */
	T0* a12; /* assigner */
	T0* a13; /* keys */
	T0* a14; /* extended_name */
	T0* a15; /* declared_type */
	T0* a16; /* first_indexing */
	T0* a17; /* frozen_keyword */
	T0* a18; /* obsolete_message */
	T0* a19; /* locals */
	T0* a20; /* compound */
	T0* a21; /* rescue_clause */
	T0* a22; /* clients */
	T0* a23; /* is_keyword */
	T0* a24; /* end_keyword */
	T0* a25; /* semicolon */
	T0* a26; /* feature_clause */
	T0* a27; /* object_tests */
	T1 a28; /* validity_checked */
	T1 a29; /* has_validity_error */
	T6 a30; /* id */
	T6 a31; /* version */
	T0* a32; /* synonym */
};

/* Struct for type detachable ET_DEFERRED_FUNCTION */
struct S1513 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T0* a8; /* preconditions */
	T6 a9; /* hash_code */
	T0* a10; /* arguments */
	T0* a11; /* postconditions */
	T0* a12; /* assigner */
	T0* a13; /* extended_name */
	T0* a14; /* declared_type */
	T0* a15; /* frozen_keyword */
	T0* a16; /* obsolete_message */
	T0* a17; /* clients */
	T0* a18; /* is_keyword */
	T0* a19; /* deferred_keyword */
	T0* a20; /* end_keyword */
	T0* a21; /* semicolon */
	T0* a22; /* feature_clause */
	T0* a23; /* first_indexing */
	T0* a24; /* object_tests */
	T1 a25; /* validity_checked */
	T1 a26; /* has_validity_error */
	T6 a27; /* id */
	T6 a28; /* version */
	T0* a29; /* synonym */
};

/* Struct for type detachable ET_ONCE_PROCEDURE */
struct S1515 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* compound */
	T0* a7; /* first_precursor */
	T0* a8; /* other_precursors */
	T0* a9; /* preconditions */
	T6 a10; /* hash_code */
	T0* a11; /* arguments */
	T0* a12; /* postconditions */
	T0* a13; /* keys */
	T0* a14; /* extended_name */
	T0* a15; /* first_indexing */
	T0* a16; /* frozen_keyword */
	T0* a17; /* obsolete_message */
	T0* a18; /* locals */
	T0* a19; /* rescue_clause */
	T0* a20; /* clients */
	T0* a21; /* is_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* object_tests */
	T1 a26; /* validity_checked */
	T1 a27; /* has_validity_error */
	T6 a28; /* id */
	T6 a29; /* version */
	T0* a30; /* synonym */
};

/* Struct for type detachable ET_EXTERNAL_PROCEDURE */
struct S1516 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* first_seed */
	T0* a2; /* iteration_components */
	T0* a3; /* implementation_class */
	T0* a4; /* implementation_feature */
	T0* a5; /* other_seeds */
	T0* a6; /* first_precursor */
	T0* a7; /* other_precursors */
	T0* a8; /* preconditions */
	T6 a9; /* hash_code */
	T8 a10; /* builtin_class_code */
	T8 a11; /* builtin_feature_code */
	T0* a12; /* arguments */
	T0* a13; /* postconditions */
	T0* a14; /* extended_name */
	T0* a15; /* frozen_keyword */
	T0* a16; /* language */
	T0* a17; /* obsolete_message */
	T0* a18; /* alias_clause */
	T0* a19; /* clients */
	T0* a20; /* is_keyword */
	T0* a21; /* end_keyword */
	T0* a22; /* semicolon */
	T0* a23; /* feature_clause */
	T0* a24; /* first_indexing */
	T0* a25; /* object_tests */
	T1 a26; /* validity_checked */
	T1 a27; /* has_validity_error */
	T6 a28; /* id */
	T6 a29; /* version */
	T0* a30; /* synonym */
};

/* Struct for type detachable ET_ALIASED_FEATURE_NAME */
struct S1517 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_name */
	T0* a2; /* alias_names */
};

/* Struct for type detachable ET_ALIAS_FREE_NAME */
struct S1518 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* alias_keyword */
	T0* a2; /* alias_string */
	T2 a3; /* code */
	T6 a4; /* hash_code */
	T0* a5; /* convert_keyword */
	T6 a6; /* seed */
};

/* Struct for type detachable ET_CLASS_ASSERTION */
struct S1521 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_keyword */
};

/* Struct for type detachable ET_LABELED_ACTUAL_PARAMETER */
struct S1522 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* label_item */
};

/* Struct for type detachable ET_LIKE_FEATURE */
struct S1524 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* like_keyword */
	T0* a3; /* name */
	T6 a4; /* seed */
	T1 a5; /* is_procedure */
};

/* Struct for type detachable ET_QUALIFIED_LIKE_BRACED_TYPE */
struct S1525 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* like_keyword */
	T0* a3; /* target_type */
	T0* a4; /* implementation_class */
	T0* a5; /* qualified_name */
	T0* a6; /* left_brace */
	T0* a7; /* right_brace */
};

/* Struct for type detachable ET_QUALIFIED_LIKE_TYPE */
struct S1526 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* target_type */
	T0* a3; /* implementation_class */
	T0* a4; /* qualified_name */
};

/* Struct for type detachable ET_ASSIGNER_INSTRUCTION */
struct S1527 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* call */
	T0* a2; /* source */
	T0* a3; /* assign_symbol */
	T0* a4; /* name */
};

/* Struct for type detachable ET_ASSIGNMENT */
struct S1528 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* source */
	T0* a3; /* assign_symbol */
};

/* Struct for type detachable ET_ASSIGNMENT_ATTEMPT */
struct S1529 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* source */
	T0* a3; /* assign_attempt_symbol */
};

/* Struct for type detachable ET_LOOP_INSTRUCTION */
struct S1530 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* from_compound */
	T0* a2; /* until_conditional */
	T0* a3; /* loop_compound */
	T0* a4; /* end_keyword */
	T0* a5; /* invariant_part */
	T0* a6; /* variant_part */
	T1 a7; /* has_old_variant_syntax */
};

/* Struct for type detachable ET_BANG_INSTRUCTION */
struct S1532 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* target */
	T0* a3; /* creation_call */
	T0* a4; /* left_bang */
	T0* a5; /* right_bang */
};

/* Struct for type detachable ET_CREATE_INSTRUCTION */
struct S1533 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* creation_type */
	T0* a2; /* target */
	T0* a3; /* creation_call */
	T0* a4; /* create_keyword */
	T0* a5; /* creation_region */
};

/* Struct for type detachable ET_CHOICE_RANGE */
struct S1534 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* lower */
	T0* a2; /* upper */
	T0* a3; /* dotdot */
};

/* Struct for type detachable ET_PRECURSOR_INSTRUCTION */
struct S1535 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parent_name */
	T0* a2; /* arguments */
	T0* a3; /* precursor_keyword */
	T1 a4; /* is_parent_prefixed */
	T0* a5; /* parenthesis_call */
	T0* a6; /* parent_type */
};

/* Struct for type detachable ET_STATIC_CALL_INSTRUCTION */
struct S1537 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T0* a2; /* feature_keyword */
	T0* a3; /* qualified_name */
	T0* a4; /* arguments */
	T0* a5; /* parenthesis_call */
};

/* Struct for type detachable ET_PRECURSOR_EXPRESSION */
struct S1538 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T1 a2; /* is_parent_prefixed */
	T0* a3; /* parent_name */
	T0* a4; /* precursor_keyword */
	T0* a5; /* arguments */
	T0* a6; /* parenthesis_call */
	T0* a7; /* parent_type */
};

/* Struct for type detachable ET_FEATURE_ADDRESS */
struct S1539 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* name */
};

/* Struct for type detachable ET_CURRENT_ADDRESS */
struct S1540 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* current_keyword */
};

/* Struct for type detachable ET_RESULT_ADDRESS */
struct S1541 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* result_keyword */
};

/* Struct for type detachable ET_EXPRESSION_ADDRESS */
struct S1542 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* expression */
};

/* Struct for type detachable ET_INFIX_EXPRESSION */
struct S1543 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left */
	T0* a3; /* name */
	T0* a4; /* right */
	T1 a5; /* is_boolean_operator */
};

/* Struct for type detachable ET_INFIX_AND_THEN_OPERATOR */
struct S1544 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* and_keyword */
	T0* a2; /* then_keyword */
	T6 a3; /* seed */
};

/* Struct for type detachable ET_INFIX_OR_ELSE_OPERATOR */
struct S1545 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* or_keyword */
	T0* a2; /* else_keyword */
	T6 a3; /* seed */
};

/* Struct for type detachable ET_MANIFEST_TYPE */
struct S1546 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left_brace */
	T0* a3; /* type */
	T0* a4; /* right_brace */
};

/* Struct for type detachable ET_PREFIX_EXPRESSION */
struct S1547 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* expression */
	T1 a4; /* is_boolean_operator */
};

/* Struct for type detachable ET_OLD_EXPRESSION */
struct S1548 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* old_keyword */
	T0* a3; /* expression */
};

/* Struct for type detachable ET_OLD_OBJECT_TEST */
struct S1549 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* left_brace */
	T0* a4; /* attached_keyword */
	T0* a5; /* as_keyword */
	T0* a6; /* colon */
	T0* a7; /* declared_type */
	T0* a8; /* right_brace */
	T0* a9; /* expression */
};

/* Struct for type detachable ET_ONCE_MANIFEST_STRING */
struct S1550 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* once_keyword */
	T0* a3; /* manifest_string */
	T6 a4; /* id */
};

/* Struct for type detachable ET_ITERATION_CURSOR */
struct S1551 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* at_symbol */
	T0* a3; /* item_name */
};

/* Struct for type detachable ET_DO_FUNCTION_INLINE_AGENT */
struct S1552 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* declared_type */
	T0* a5; /* end_keyword */
	T0* a6; /* implicit_result */
	T0* a7; /* object_tests */
	T0* a8; /* iteration_components */
	T0* a9; /* actual_arguments */
	T0* a10; /* preconditions */
	T0* a11; /* locals */
	T0* a12; /* compound */
	T0* a13; /* postconditions */
	T0* a14; /* rescue_clause */
	T0* a15; /* target */
	T6 a16; /* attached_result_index */
	T6 a17; /* result_index */
};

/* Struct for type detachable ET_ONCE_FUNCTION_INLINE_AGENT */
struct S1553 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* declared_type */
	T0* a5; /* end_keyword */
	T0* a6; /* implicit_result */
	T0* a7; /* object_tests */
	T0* a8; /* iteration_components */
	T0* a9; /* actual_arguments */
	T0* a10; /* preconditions */
	T0* a11; /* locals */
	T0* a12; /* keys */
	T0* a13; /* compound */
	T0* a14; /* postconditions */
	T0* a15; /* rescue_clause */
	T0* a16; /* target */
	T6 a17; /* attached_result_index */
	T6 a18; /* result_index */
};

/* Struct for type detachable ET_EXTERNAL_FUNCTION_INLINE_AGENT */
struct S1554 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* language */
	T0* a4; /* object_tests */
	T0* a5; /* iteration_components */
	T0* a6; /* actual_arguments */
	T0* a7; /* preconditions */
	T0* a8; /* alias_clause */
	T0* a9; /* postconditions */
	T0* a10; /* formal_arguments */
	T0* a11; /* declared_type */
	T0* a12; /* end_keyword */
	T0* a13; /* implicit_result */
	T0* a14; /* target */
	T6 a15; /* attached_result_index */
	T6 a16; /* result_index */
};

/* Struct for type detachable ET_DO_PROCEDURE_INLINE_AGENT */
struct S1555 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* end_keyword */
	T0* a5; /* object_tests */
	T0* a6; /* iteration_components */
	T0* a7; /* actual_arguments */
	T0* a8; /* preconditions */
	T0* a9; /* locals */
	T0* a10; /* compound */
	T0* a11; /* postconditions */
	T0* a12; /* rescue_clause */
	T0* a13; /* target */
	T6 a14; /* attached_result_index */
	T6 a15; /* result_index */
};

/* Struct for type detachable ET_ONCE_PROCEDURE_INLINE_AGENT */
struct S1556 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* end_keyword */
	T0* a5; /* object_tests */
	T0* a6; /* iteration_components */
	T0* a7; /* actual_arguments */
	T0* a8; /* preconditions */
	T0* a9; /* locals */
	T0* a10; /* keys */
	T0* a11; /* compound */
	T0* a12; /* postconditions */
	T0* a13; /* rescue_clause */
	T0* a14; /* target */
	T6 a15; /* attached_result_index */
	T6 a16; /* result_index */
};

/* Struct for type detachable ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
struct S1557 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* language */
	T0* a4; /* object_tests */
	T0* a5; /* iteration_components */
	T0* a6; /* actual_arguments */
	T0* a7; /* preconditions */
	T0* a8; /* alias_clause */
	T0* a9; /* postconditions */
	T0* a10; /* formal_arguments */
	T0* a11; /* end_keyword */
	T0* a12; /* target */
	T6 a13; /* attached_result_index */
	T6 a14; /* result_index */
};

/* Struct for type detachable ET_AGENT_OPEN_TARGET */
struct S1558 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* type */
	T0* a3; /* right_brace */
	T6 a4; /* index */
};

/* Struct for type detachable ET_AGENT_TYPED_OPEN_ARGUMENT */
struct S1559 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* left_brace */
	T0* a3; /* right_brace */
	T0* a4; /* question_mark */
	T6 a5; /* index */
};

/* Struct for type detachable ET_TAGGED_ASSERTION */
struct S1562 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* tag */
	T0* a2; /* untagged_assertion */
};

/* Struct for type [detachable] ET_FILE_POSITION */
struct S1564 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
	T0* a2; /* filename */
};

/* Struct for type [detachable] DS_QUICK_SORTER [[attached] ET_QUERY] */
struct S1566 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [detachable] DS_QUICK_SORTER [[attached] ET_PROCEDURE] */
struct S1567 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [detachable] ET_REDECLARED_FEATURE */
struct S1569 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* flattened_feature */
	T0* a3; /* parent_feature */
	T6 a4; /* first_seed */
	T0* a5; /* other_seeds */
	T1 a6; /* is_selected */
	T0* a7; /* replicated_features */
	T0* a8; /* replicated_seeds */
};

/* Struct for type [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
struct S1571 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_state */
	T1 a2; /* internal_overflowed */
	T11 a3; /* part1 */
	T11 a4; /* part2 */
	T0* a5; /* leading_separators */
	T0* a6; /* trailing_separators */
	T6 a7; /* sign */
	T6 a8; /* conversion_type */
	T2 a9; /* internal_lookahead */
	T1 a10; /* leading_separators_acceptable */
	T1 a11; /* trailing_separators_acceptable */
};

/* Struct for type [detachable] TUPLE [[attached] ET_MASTER_CLASS, [attached] ET_SYSTEM_PROCESSOR] */
struct S1573 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] SPECIAL [[attached] NATURAL_16] */
struct S1576 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T9 z2[1]; /* item */
};

/* Struct for type [detachable] ARRAY [[attached] NATURAL_16] */
struct S1577 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] ARRAY [[attached] NATURAL_32] */
struct S1578 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_CLUSTER] */
struct S1579 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S1580 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S1581 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_FILE_RULE] */
struct S1582 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S1583 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S1584 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S1585 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S1586 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S1587 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S1588 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S1589 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ACTION] */
struct S1590 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_ACTION] */
struct S1591 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ACTION] */
struct S1592 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S1593 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_OPTIONS] */
struct S1594 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_OPTIONS] */
struct S1597 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
struct S1598 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_VISIBLE_CLASS] */
struct S1599 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_VISIBLE_CLASS] */
struct S1600 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_VISIBLE_CLASS] */
struct S1601 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S1602 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_CONDITION_ITEM] */
struct S1603 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
struct S1604 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_ASSEMBLY] */
struct S1605 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DOTNET_ASSEMBLY] */
struct S1606 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_UNIVERSE] */
struct S1608 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_UNIVERSE] */
struct S1609 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_UNIVERSE] */
struct S1611 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_INTERNAL_UNIVERSE] */
struct S1612 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INTERNAL_UNIVERSE] */
struct S1614 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INTERNAL_UNIVERSE] */
struct S1615 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] ET_CLASS_TYPE_STATUS_CHECKER1 */
struct S1616 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
};

/* Struct for type [detachable] ET_CLASS_TYPE_STATUS_CHECKER2 */
struct S1617 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
};

/* Struct for type [detachable] ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
struct S1618 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type [detachable] ET_CLASS_TYPE_STATUS_CHECKER3 */
struct S1619 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_NAMED_CLASS] */
struct S1620 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1621 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1622 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1623 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] CONSOLE */
struct S1624 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* file_pointer */
	T0* a2; /* last_string_32 */
	T6 a3; /* mode */
	T0* a4; /* last_string */
	T0* a5; /* internal_name */
	T0* a6; /* internal_detachable_name_pointer */
	T2 a7; /* last_character */
};

/* Struct for type [detachable] DS_LINKED_QUEUE [[attached] CHARACTER_8] */
struct S1626 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* first_cell */
	T0* a3; /* last_cell */
};

/* Struct for type [detachable] UC_UTF16_ROUTINES */
struct S1627 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_LIBRARY] */
struct S1628 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LIBRARY] */
struct S1629 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_LIBRARY] */
struct S1630 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET] */
struct S1631 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ADAPTED_UNIVERSE] */
struct S1632 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_UNIVERSE] */
struct S1633 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_UNIVERSE] */
struct S1634 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_TARGET_PARENT] */
struct S1635 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET_PARENT] */
struct S1636 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET_PARENT] */
struct S1637 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] YY_UNICODE_BUFFER */
struct S1638 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T6 a2; /* index */
	T6 a3; /* line */
	T6 a4; /* column */
	T6 a5; /* position */
	T6 a6; /* count */
	T1 a7; /* filled */
	T1 a8; /* beginning_of_line */
	T6 a9; /* capacity */
};

/* Struct for type [detachable] SPECIAL [detachable ET_OBJECT_TEST_LIST] */
struct S1639 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
struct S1640 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST_LIST] */
struct S1641 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_OBJECT_TEST_LIST] */
struct S1642 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable ET_ITERATION_COMPONENT_LIST] */
struct S1643 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ITERATION_COMPONENT_LIST] */
struct S1644 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ITERATION_COMPONENT_LIST] */
struct S1645 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ITERATION_COMPONENT_LIST] */
struct S1646 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ASSERTION_ITEM] */
struct S1647 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTION_ITEM] */
struct S1648 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ASSERTION_ITEM] */
struct S1649 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
struct S1650 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_NAMED_CLASS] */
struct S1651 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_CLASS] */
struct S1653 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_CLASS] */
struct S1655 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* modulus */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T0* a12; /* equality_tester */
	T0* a13; /* internal_cursor */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
struct S1656 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_BASE_TYPE_CONSTRAINT] */
struct S1659 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] NATURAL_32] */
struct S1660 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T0* a2; /* special_routines */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] ET_BASE_TYPE_CONSTRAINT_LIST */
struct S1661 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* right_brace */
	T0* a3; /* storage */
	T6 a4; /* count */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
struct S1662 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FLATTENED_FEATURE] */
struct S1663 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FEATURE_NAME] */
struct S1666 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FLATTENED_FEATURE] */
struct S1667 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME] */
struct S1668 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ALIAS_NAME] */
struct S1669 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ALIAS_NAME] */
struct S1671 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
struct S1672 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_LIST] */
struct S1673 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CLIENT_LIST] */
struct S1674 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLIENT_LIST] */
struct S1675 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S1677 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT] */
struct S1678 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CLIENT] */
struct S1679 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_FEATURE_NAME] */
struct S1680 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* found_position */
	T0* a3; /* internal_cursor */
	T6 a4; /* count */
	T0* a5; /* item_storage */
	T6 a6; /* modulus */
	T6 a7; /* last_position */
	T6 a8; /* free_slot */
	T6 a9; /* position */
	T0* a10; /* equality_tester */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* slots_position */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] BOOLEAN, [attached] ET_FEATURE_NAME] */
struct S1681 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T6 a3; /* found_position */
	T0* a4; /* internal_cursor */
	T0* a5; /* key_storage */
	T0* a6; /* item_storage */
	T0* a7; /* key_equality_tester */
	T0* a8; /* internal_keys */
	T6 a9; /* position */
	T6 a10; /* last_position */
	T6 a11; /* free_slot */
	T6 a12; /* modulus */
	T0* a13; /* equality_tester */
	T6 a14; /* slots_position */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* special_item_routines */
	T0* a17; /* clashes */
	T0* a18; /* slots */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
struct S1682 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_storage */
	T6 a5; /* count */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* capacity */
	T0* a10; /* equality_tester */
	T0* a11; /* key_equality_tester */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] ET_REPLICATED_FEATURE */
struct S1684 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* features */
	T6 a2; /* selected_count */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] ET_ADAPTED_FEATURE] */
struct S1685 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] ET_PARENT_FEATURE] */
struct S1686 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] ET_DOTNET_SIGNATURE_TESTER */
struct S1687 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_SET [[attached] ET_DOTNET_FEATURE] */
struct S1688 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* equality_tester */
	T6 a9; /* count */
	T6 a10; /* slots_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T0* a14; /* internal_cursor */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
struct S1690 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* count */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* capacity */
	T0* a10; /* equality_tester */
	T0* a11; /* internal_cursor */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE] */
struct S1692 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_LIKE_FEATURE] */
struct S1693 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* cycle */
	T0* a2; /* counts */
	T0* a3; /* successors */
	T0* a4; /* items */
	T0* a5; /* sorted_items */
	T0* a6; /* indexes */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] ET_LIKE_FEATURE] */
struct S1694 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
struct S1695 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type detachable DS_LINKED_LIST [[attached] ET_PARENT_FEATURE] */
struct S1696 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
struct S1697 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* special_key_routines */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T6 a16; /* slots_position */
	T6 a17; /* count */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE, [attached] INTEGER_32] */
struct S1698 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_CLASS] */
struct S1701 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_FEATURE_NAME] */
struct S1703 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_RENAME] */
struct S1704 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RENAME] */
struct S1707 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_ALIAS_NAME] */
struct S1711 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_PROCEDURE] */
struct S1712 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_QUERY] */
struct S1713 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] YY_BUFFER] */
struct S1714 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] YY_BUFFER] */
struct S1715 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_TARGET] */
struct S1716 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TARGET] */
struct S1717 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S1719 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type detachable DS_BILINKABLE [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S1720 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [detachable] DS_BILINKED_LIST [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
struct S1722 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] DS_BILINKED_LIST_CURSOR [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
struct S1723 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* before */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* after */
};

/* Struct for type detachable DS_LINKABLE [detachable STRING_8] */
struct S1724 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CLUSTER] */
struct S1725 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLUSTER] */
struct S1726 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLUSTER] */
struct S1727 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_TYPE_ITEM] */
struct S1728 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_ITEM] */
struct S1729 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_MASTER_CLASS] */
struct S1730 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
struct S1731 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_CLUSTER] */
struct S1732 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_CLUSTER]] */
struct S1733 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
struct S1734 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S1736 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY]] */
struct S1737 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S1738 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
struct S1739 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] STRING_8]] */
struct S1742 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
struct S1743 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8] */
struct S1744 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* key_storage */
	T0* a3; /* item_storage */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T0* a6; /* hash_function */
	T6 a7; /* position */
	T6 a8; /* last_position */
	T6 a9; /* capacity */
	T6 a10; /* slots_position */
	T6 a11; /* count */
	T6 a12; /* free_slot */
	T0* a13; /* equality_tester */
	T6 a14; /* modulus */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* special_item_routines */
	T0* a17; /* clashes */
	T0* a18; /* slots */
	T0* a19; /* special_key_routines */
	T0* a20; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
struct S1745 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T0* a6; /* hash_function */
	T6 a7; /* last_position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T6 a10; /* count */
	T0* a11; /* equality_tester */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type detachable DS_LINKABLE [[attached] XM_ELEMENT_NODE] */
struct S1750 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* right */
	T0* a2; /* item */
};

/* Struct for type detachable ET_CONSTRAINT_GENERIC_NAMED_TYPE */
struct S1755 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
	T0* a3; /* actual_parameters */
};

/* Struct for type detachable ET_UNQUALIFIED_CALL_INSTRUCTION */
struct S1756 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* arguments */
	T0* a3; /* parenthesis_call */
};

/* Struct for type detachable ET_CLUSTER_DEPENDENCE_CONSTRAINT */
struct S1758 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_cluster */
	T0* a2; /* group_names */
	T0* a3; /* group_pathnames */
	T0* a4; /* pathname_buffer */
};

/* Struct for type [detachable] ET_SEEDED_QUERY_COMPARATOR */
struct S1759 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_SEEDED_PROCEDURE_COMPARATOR */
struct S1762 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] ET_CLIENT_LIST] */
struct S1766 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] ET_FORMAL_COMMA_ARGUMENT */
struct S1767 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T6 a3; /* index */
	T6 a4; /* attached_index */
	T1 a5; /* is_used */
};

/* Struct for type [detachable] ET_LOCAL_COMMA_VARIABLE */
struct S1768 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T6 a3; /* index */
	T6 a4; /* attached_index */
	T1 a5; /* is_used */
};

/* Struct for type [detachable] ET_LABELED_COMMA_ACTUAL_PARAMETER */
struct S1769 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* label_item */
};

/* Struct for type [detachable] ET_KEYWORD_EXPRESSION */
struct S1771 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
	T0* a2; /* keyword */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_RENAME_ITEM] */
struct S1776 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RENAME_ITEM] */
struct S1777 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_EXPORT] */
struct S1779 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPORT] */
struct S1780 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FEATURE_NAME_ITEM] */
struct S1782 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME_ITEM] */
struct S1783 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FORMAL_PARAMETER_ITEM] */
struct S1784 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER_ITEM] */
struct S1785 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S1788 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG]] */
struct S1789 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_TARGET, [attached] ET_ECF_STATE] */
struct S1790 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S1791 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE]] */
struct S1792 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S1793 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY]] */
struct S1794 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S1795 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG]] */
struct S1796 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S1797 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE]] */
struct S1798 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S1799 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT]] */
struct S1800 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S1801 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE]] */
struct S1802 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type detachable DS_LINKABLE [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
struct S1806 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S1809 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S1810 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_AGENT_IMPLICIT_CURRENT_TARGET */
struct S1811 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_expression */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CHOICE_ITEM] */
struct S1813 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CHOICE_ITEM] */
struct S1814 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1815 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1817 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CONVERT_FEATURE_ITEM] */
struct S1819 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONVERT_FEATURE_ITEM] */
struct S1820 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_CREATOR] */
struct S1821 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CREATOR] */
struct S1822 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ELSEIF_EXPRESSION] */
struct S1823 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_EXPRESSION] */
struct S1824 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ELSEIF_PART] */
struct S1825 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_PART] */
struct S1826 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FEATURE_CLAUSE] */
struct S1827 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_CLAUSE] */
struct S1828 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INDEXING_ITEM] */
struct S1830 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_ITEM] */
struct S1831 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INDEXING_TERM_ITEM] */
struct S1832 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM_ITEM] */
struct S1833 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_LOCAL_VARIABLE_ITEM] */
struct S1834 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LOCAL_VARIABLE_ITEM] */
struct S1836 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_MANIFEST_STRING_ITEM] */
struct S1838 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_MANIFEST_STRING_ITEM] */
struct S1839 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_PARENT_LIST] */
struct S1840 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_LIST] */
struct S1841 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_TYPE_CONSTRAINT_ITEM] */
struct S1843 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_CONSTRAINT_ITEM] */
struct S1844 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_WHEN_EXPRESSION] */
struct S1845 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_EXPRESSION] */
struct S1846 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_WHEN_PART] */
struct S1847 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_PART] */
struct S1848 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_BASE_TYPE_RENAME_CONSTRAINT */
struct S1850 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* renames */
};

/* Struct for type detachable DS_SPARSE_TABLE_KEYS [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S1853 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table */
	T0* a2; /* equality_tester */
	T0* a3; /* internal_cursor */
};

/* Struct for type [detachable] ET_STANDARD_ONCE_KEYS */
struct S1857 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_CLUSTER] */
struct S1859 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CLUSTER] */
struct S1860 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CLUSTER] */
struct S1861 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S1862 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S1863 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S1864 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S1865 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S1866 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S1867 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_FILE_RULE] */
struct S1868 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_FILE_RULE] */
struct S1869 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_FILE_RULE] */
struct S1870 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S1871 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S1872 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S1873 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S1874 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S1875 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S1876 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S1877 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S1878 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S1879 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S1880 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S1881 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S1882 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S1883 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S1884 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S1885 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S1886 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S1887 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S1888 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S1889 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S1890 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S1891 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S1892 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S1893 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S1894 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S1895 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S1896 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S1897 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CONDITION_ITEM] */
struct S1898 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_ECF_CONDITION_ITEM] */
struct S1899 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CONDITION_ITEM] */
struct S1900 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type detachable DS_LINKABLE [[attached] CHARACTER_8] */
struct S1902 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_CLASS] */
struct S1903 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_BASE_TYPE_CONSTRAINT] */
struct S1904 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE_CONSTRAINT] */
struct S1905 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_BASE_TYPE_CONSTRAINT] */
struct S1906 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] NATURAL_32] */
struct S1907 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_FEATURE_NAME] */
struct S1908 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] ET_FEATURE_NAME] */
struct S1909 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
struct S1911 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_REPLICABLE_FEATURE] */
struct S1912 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_REPLICABLE_FEATURE] */
struct S1914 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_ADAPTED_FEATURE] */
struct S1915 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
	T1 a3; /* after */
	T0* a4; /* current_cell */
	T0* a5; /* next_cursor */
};

/* Struct for type detachable DS_LINKABLE [[attached] ET_ADAPTED_FEATURE] */
struct S1916 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_FEATURE] */
struct S1917 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_PARENT_FEATURE] */
struct S1918 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
struct S1919 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DOTNET_FEATURE] */
struct S1920 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_FEATURE] */
struct S1921 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_FEATURE] */
struct S1922 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
struct S1923 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
struct S1926 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
struct S1927 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_DOTNET_FEATURE] */
struct S1928 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
	T1 a3; /* after */
	T0* a4; /* current_cell */
	T0* a5; /* next_cursor */
};

/* Struct for type detachable DS_LINKABLE [[attached] ET_DOTNET_FEATURE] */
struct S1929 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
struct S1930 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T6 a3; /* free_slot */
	T6 a4; /* position */
	T6 a5; /* modulus */
	T6 a6; /* capacity */
	T0* a7; /* equality_tester */
	T0* a8; /* key_equality_tester */
	T0* a9; /* internal_cursor */
	T6 a10; /* found_position */
	T0* a11; /* item_storage */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T0* a17; /* hash_function */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_LIKE_FEATURE] */
struct S1931 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LIKE_FEATURE] */
struct S1932 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_LIKE_FEATURE] */
struct S1933 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
struct S1935 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
struct S1936 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
struct S1937 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* position */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* count */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
struct S1938 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable DS_BILINKABLE [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
struct S1939 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* left */
	T0* a3; /* right */
};

/* Struct for type [detachable] ET_CLUSTER_ERROR */
struct S1940 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* cluster */
};

/* Struct for type [detachable] ET_SYNTAX_ERROR */
struct S1941 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* filename */
	T0* a6; /* position */
	T1 a7; /* ise_reported */
	T1 a8; /* ise_fatal */
	T1 a9; /* ge_reported */
	T1 a10; /* ge_fatal */
};

/* Struct for type [detachable] ET_DOTNET_ASSEMBLY_ERROR */
struct S1942 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* assembly */
};

/* Struct for type [detachable] AP_OPTION_COMPARATOR */
struct S1943 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_QUICK_SORTER [[attached] AP_OPTION] */
struct S1944 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [detachable] ST_SPLITTER */
struct S1953 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_escape_character */
	T2 a2; /* escape_character */
	T0* a3; /* separator_codes */
	T0* a4; /* separators */
};

/* Struct for type [detachable] KL_CHARACTER_ROUTINES */
struct S1954 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_CURRENT, [attached] ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
struct S1955 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T6 z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
struct S1956 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* actual_parameters */
	T6 a3; /* tuple_position */
	T0* a4; /* tuple_type */
};

/* Struct for type [detachable] ET_ACTUAL_PARAMETER_SUBLIST */
struct S1957 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* actual_parameters */
	T6 a2; /* upper */
	T6 a3; /* lower */
};

/* Struct for type [detachable] SPECIAL [[attached] ET_FORMAL_ARGUMENT_ITEM] */
struct S1960 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_ARGUMENT_ITEM] */
struct S1962 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_DYNAMIC_PRECURSOR] */
struct S1964 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRECURSOR] */
struct S1965 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_INSTRUCTION] */
struct S1966 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INSTRUCTION] */
struct S1968 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
struct S1977 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* actual_arguments */
	T0* a3; /* left_symbol */
	T0* a4; /* storage */
	T0* a5; /* right_symbol */
};

/* Struct for type [detachable] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8] */
struct S1978 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T8 z3;
};

/* Struct for type [detachable] ET_INFIX_CAST_EXPRESSION */
struct S1980 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* type */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] STRING_8] */
struct S1984 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
struct S1985 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
struct S1988 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
struct S1989 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] DS_SPARSE_TABLE_KEYS_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S1990 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table_cursor */
	T0* a2; /* container */
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
struct S1993 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
struct S1994 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
struct S1997 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
struct S1998 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] ET_UNIVERSE_ERROR */
struct S1999 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* universe */
};

/* Struct for type [detachable] TUPLE [[attached] ET_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2000 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type detachable ET_PARENTHESIS_INSTRUCTION */
struct S2002 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* name */
	T0* a3; /* arguments */
};

/* Struct for type [detachable] ET_PARENTHESIS_SYMBOL */
struct S2020 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
};

/* Struct for type [detachable] ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
struct S2026 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [detachable] ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
struct S2027 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* agent_expression */
	T6 a2; /* argument_index */
	T6 a3; /* index */
};

/* Struct for type [detachable] DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S2034 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table_cursor */
	T0* a2; /* container */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] LX_WILDCARD, [attached] STRING_8] */
struct S2040 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T0* a3; /* clashes */
	T0* a4; /* item_storage */
	T0* a5; /* key_equality_tester */
	T0* a6; /* internal_keys */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T0* a10; /* equality_tester */
	T6 a11; /* found_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_WILDCARD, [attached] STRING_8] */
struct S2041 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] ET_DYNAMIC_EQUALITY_EXPRESSION */
struct S2042 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target_type_set */
	T0* a2; /* static_equality */
	T0* a3; /* current_feature */
	T0* a4; /* current_type */
	T0* a5; /* next */
};

/* Struct for type [detachable] ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
struct S2043 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target_type_set */
	T0* a2; /* static_equality */
	T0* a3; /* current_feature */
	T0* a4; /* current_type */
	T0* a5; /* next */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_LIBRARY] */
struct S2044 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] ET_CONVERT_BUILTIN_EXPRESSION */
struct S2046 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* convert_feature */
	T0* a4; /* type */
};

/* Struct for type [detachable] ET_CONVERT_FROM_EXPRESSION */
struct S2047 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* type */
	T6 a3; /* index */
	T0* a4; /* expression */
	T0* a5; /* convert_feature */
};

/* Struct for type [detachable] ET_CONVERT_TO_EXPRESSION */
struct S2048 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* index */
	T0* a3; /* expression */
	T0* a4; /* convert_feature */
};

/* Struct for type detachable DS_CELL [detachable ET_CLASS] */
struct S2051 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] TUPLE [[attached] DS_CELL [detachable ET_CLASS]] */
struct S2052 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] ET_EIFFEL_PREPARSER, [attached] DS_ARRAYED_LIST [[attached] STRING_8]] */
struct S2054 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_FEATURE] */
struct S2056 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_FILE_RULE] */
struct S2057 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
struct S2058 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_FILE_RULES] */
struct S2059 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
struct S2060 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_STATE] */
struct S2061 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] STRING_8]] */
struct S2063 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] KL_STRING_ROUTINES, [attached] STRING_8] */
struct S2064 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] STRING_8]] */
struct S2066 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] LX_DFA_WILDCARD */
struct S2068 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* yy_nxt */
	T0* a2; /* yy_accept */
	T6 a3; /* subject_start */
	T6 a4; /* subject_end */
	T6 a5; /* match_count */
	T6 a6; /* matched_start */
	T6 a7; /* matched_end */
	T0* a8; /* yy_ec */
	T6 a9; /* yyNull_equiv_class */
	T6 a10; /* yyMax_symbol_equiv_class */
	T6 a11; /* yyNb_rows */
	T0* a12; /* pattern */
	T1 a13; /* is_case_insensitive */
	T0* a14; /* subject */
};

/* Struct for type detachable DS_LINKABLE [[attached] ET_PARENT_FEATURE] */
struct S2071 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] TUPLE [[attached] LX_WILDCARD] */
struct S2073 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] LX_WILDCARD]] */
struct S2074 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] TUPLE [[attached] ET_CLUSTER_DEPENDENCE_CONSTRAINT, [attached] ET_GROUP, [attached] STRING_8] */
struct S2075 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] SPECIAL [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
struct S2077 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
struct S2078 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] LX_WILDCARD] */
struct S2079 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_WILDCARD] */
struct S2082 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
struct S2086 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2089 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [detachable] TUPLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2090 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2091 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [detachable] TUPLE [[attached] ET_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2092 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [detachable] ET_AST_NULL_LEAF */
struct S2094 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
};

/* Struct for type [detachable] LX_WILDCARD_PARSER */
struct S2095 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* successful */
	T0* a2; /* pending_rules */
	T0* a3; /* start_condition_stack */
	T0* a4; /* action_factory */
	T0* a5; /* old_singleton_lines */
	T0* a6; /* old_singleton_columns */
	T0* a7; /* old_singleton_counts */
	T0* a8; /* old_regexp_lines */
	T0* a9; /* old_regexp_columns */
	T0* a10; /* old_regexp_counts */
	T0* a11; /* unions_of_concatenations_of_symbol_classes_by_utf8_character_class */
	T0* a12; /* buffer */
	T6 a13; /* maximum_used_symbol */
	T0* a14; /* description */
	T0* a15; /* last_string_value */
	T0* a16; /* last_string_32_value */
	T0* a17; /* last_lx_symbol_class_value */
	T0* a18; /* error_handler */
	T0* a19; /* name_definitions */
	T0* a20; /* character_classes */
	T0* a21; /* character_classes_by_name */
	T0* a22; /* equiv_character_classes */
	T0* a23; /* utf8_character_classes_by_name */
	T0* a24; /* utf8_mode */
	T6 a25; /* line_nb */
	T0* a26; /* eiffel_verbatim_marker */
	T0* a27; /* yyss */
	T0* a28; /* input_buffer */
	T6 a29; /* yy_end */
	T6 a30; /* yy_position */
	T6 a31; /* yy_line */
	T6 a32; /* yy_column */
	T6 a33; /* yy_parsing_status */
	T6 a34; /* yy_suspended_yystacksize */
	T6 a35; /* yy_suspended_yystate */
	T6 a36; /* yy_suspended_yyn */
	T6 a37; /* yy_suspended_yychar1 */
	T6 a38; /* yy_suspended_index */
	T6 a39; /* yy_suspended_yyss_top */
	T6 a40; /* yy_suspended_yy_goto */
	T0* a41; /* yyr1 */
	T6 a42; /* yyssp */
	T0* a43; /* yypgoto */
	T0* a44; /* yycheck */
	T0* a45; /* yytable */
	T0* a46; /* yydefgoto */
	T6 a47; /* error_count */
	T1 a48; /* yy_lookahead_needed */
	T6 a49; /* yyerrstatus */
	T0* a50; /* yypact */
	T6 a51; /* last_token */
	T0* a52; /* yytranslate */
	T0* a53; /* yydefact */
	T0* a54; /* yy_pushed_start_conditions */
	T0* a55; /* yyspecial_routines1 */
	T6 a56; /* yyvsc1 */
	T0* a57; /* yyvs1 */
	T0* a58; /* yyspecial_routines2 */
	T6 a59; /* yyvsc2 */
	T0* a60; /* yyvs2 */
	T0* a61; /* yyspecial_routines3 */
	T6 a62; /* yyvsc3 */
	T0* a63; /* yyvs3 */
	T0* a64; /* yyspecial_routines4 */
	T6 a65; /* yyvsc4 */
	T0* a66; /* yyvs4 */
	T0* a67; /* yyspecial_routines5 */
	T6 a68; /* yyvsc5 */
	T0* a69; /* yyvs5 */
	T0* a70; /* yytypes1 */
	T0* a71; /* yytypes2 */
	T6 a72; /* yy_start */
	T6 a73; /* yyvsp1 */
	T6 a74; /* yyvsp2 */
	T6 a75; /* yyvsp3 */
	T6 a76; /* yyvsp4 */
	T6 a77; /* yyvsp5 */
	T1 a78; /* yy_more_flag */
	T6 a79; /* yy_more_len */
	T6 a80; /* line */
	T6 a81; /* column */
	T6 a82; /* position */
	T6 a83; /* yy_start_state */
	T0* a84; /* yy_state_stack */
	T6 a85; /* yy_state_count */
	T0* a86; /* yy_content_area */
	T0* a87; /* yy_unicode_content_area */
	T0* a88; /* yy_accept */
	T6 a89; /* yy_last_accepting_state */
	T6 a90; /* yy_last_accepting_cpos */
	T0* a91; /* yy_content */
	T0* a92; /* yy_ec */
	T0* a93; /* yy_chk */
	T0* a94; /* yy_base */
	T0* a95; /* yy_def */
	T0* a96; /* yy_meta */
	T0* a97; /* yy_nxt */
	T6 a98; /* yy_lp */
	T0* a99; /* yy_acclist */
	T6 a100; /* yy_looking_for_trail_begin */
	T6 a101; /* yy_full_match */
	T6 a102; /* yy_full_state */
	T6 a103; /* yy_full_lp */
	T1 a104; /* yy_rejected */
	T0* a105; /* last_detachable_any_value */
	T6 a106; /* last_integer_value */
	T1 a107; /* in_trail_context */
	T0* a108; /* rule */
	T1 a109; /* has_trail_context */
	T6 a110; /* head_count */
	T6 a111; /* head_line */
	T6 a112; /* head_column */
	T6 a113; /* trail_count */
	T6 a114; /* i_ */
	T6 a115; /* rule_line_nb */
	T1 a116; /* has_bol_context */
};

/* Struct for type [detachable] LX_DESCRIPTION */
struct S2096 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* maximum_symbol */
	T1 a2; /* utf8_mode */
	T1 a3; /* equiv_classes_used */
	T1 a4; /* full_table */
	T1 a5; /* meta_equiv_classes_used */
	T1 a6; /* reject_used */
	T1 a7; /* variable_trail_context */
	T0* a8; /* rules */
	T0* a9; /* start_conditions */
	T6 a10; /* array_size */
	T1 a11; /* line_pragma */
	T1 a12; /* inspect_used */
	T0* a13; /* eof_rules */
	T0* a14; /* eiffel_header */
	T1 a15; /* case_insensitive */
	T0* a16; /* equiv_classes */
	T1 a17; /* bol_needed */
	T0* a18; /* input_filename */
	T1 a19; /* has_utf8_enconding */
	T1 a20; /* actions_separated */
	T0* a21; /* eiffel_code */
	T1 a22; /* pre_action_used */
	T1 a23; /* post_action_used */
	T1 a24; /* pre_eof_action_used */
	T1 a25; /* post_eof_action_used */
	T1 a26; /* line_used */
	T1 a27; /* position_used */
};

/* Struct for type [detachable] LX_FULL_DFA */
struct S2097 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* yy_nxt */
	T0* a2; /* yy_accept */
	T0* a3; /* yy_ec */
	T6 a4; /* yyNb_rows */
	T6 a5; /* yyNull_equiv_class */
	T6 a6; /* yyMax_symbol_equiv_class */
	T0* a7; /* input_filename */
	T1 a8; /* has_utf8_enconding */
	T6 a9; /* array_size */
	T1 a10; /* inspect_used */
	T1 a11; /* actions_separated */
	T0* a12; /* eiffel_code */
	T0* a13; /* eiffel_header */
	T1 a14; /* bol_needed */
	T1 a15; /* pre_action_used */
	T1 a16; /* post_action_used */
	T1 a17; /* pre_eof_action_used */
	T1 a18; /* post_eof_action_used */
	T1 a19; /* line_pragma */
	T0* a20; /* yy_start_conditions */
	T6 a21; /* yyNb_rules */
	T0* a22; /* yy_rules */
	T6 a23; /* yyEnd_of_buffer */
	T1 a24; /* yyLine_used */
	T1 a25; /* yyPosition_used */
	T6 a26; /* minimum_symbol */
	T6 a27; /* maximum_symbol */
	T0* a28; /* states */
	T6 a29; /* backing_up_count */
	T0* a30; /* partitions */
	T6 a31; /* start_states_count */
	T1 a32; /* yyBacking_up */
	T0* a33; /* yy_eof_rules */
};

/* Struct for type [detachable] TUPLE [[attached] ET_GROUP] */
struct S2099 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]]] */
struct S2100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [detachable] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S2101 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [detachable] PREDICATE [[attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION]] */
struct S2102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] LX_RULE] */
struct S2105 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] LX_START_CONDITIONS */
struct S2106 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] LX_ACTION_FACTORY */
struct S2107 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]], [attached] LX_SYMBOL_CLASS] */
struct S2108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T6 a4; /* position */
	T6 a5; /* count */
	T6 a6; /* capacity */
	T6 a7; /* free_slot */
	T6 a8; /* last_position */
	T0* a9; /* equality_tester */
	T6 a10; /* modulus */
	T6 a11; /* slots_position */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] LX_SYMBOL_CLASS */
struct S2109 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* upper */
	T1 a2; /* is_empty */
	T1 a3; /* is_negated */
	T11 a4; /* first_set */
	T11 a5; /* second_set */
	T11 a6; /* third_set */
	T11 a7; /* fourth_set */
	T0* a8; /* other_sets */
	T6 a9; /* lower */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] STRING_32, [attached] STRING_8] */
struct S2110 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T0* a3; /* equality_tester */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T0* a9; /* special_item_routines */
	T0* a10; /* item_storage */
	T0* a11; /* special_key_routines */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T6 a15; /* found_position */
	T0* a16; /* internal_cursor */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] LX_SYMBOL_CLASS] */
struct S2111 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* equality_tester */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* count */
	T6 a10; /* slots_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T0* a14; /* internal_cursor */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] LX_SYMBOL_CLASS, [attached] STRING_32] */
struct S2112 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* position */
	T6 a6; /* count */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* free_slot */
	T6 a10; /* last_position */
	T0* a11; /* equality_tester */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] KL_EQUALITY_TESTER [[attached] STRING_32] */
struct S2113 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_STACK [[attached] BOOLEAN] */
struct S2114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T6 a2; /* count */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_32] */
struct S2115 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] STRING_32] */
struct S2116 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_SYMBOL_CLASS] */
struct S2117 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] LX_SYMBOL_CLASS] */
struct S2118 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_NFA] */
struct S2119 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] LX_NFA */
struct S2120 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* in_trail_context */
	T0* a2; /* states */
};

/* Struct for type [detachable] SPECIAL [[attached] LX_NFA] */
struct S2121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable LX_RULE */
struct S2122 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* pattern */
	T0* a3; /* action */
	T6 a4; /* head_count */
	T6 a5; /* trail_count */
	T6 a6; /* line_count */
	T6 a7; /* column_count */
	T6 a8; /* line_nb */
	T1 a9; /* has_trail_context */
	T1 a10; /* is_useful */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] LX_SYMBOL_CLASS] */
struct S2123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [detachable] LX_EQUIVALENCE_CLASSES */
struct S2124 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* new_upper */
	T0* a2; /* storage */
};

/* Struct for type [detachable] LX_UNRECOGNIZED_RULE_ERROR */
struct S2125 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_INVALID_UNICODE_CHARACTER_ERROR */
struct S2126 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_MISSING_QUOTE_ERROR */
struct S2127 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_BAD_CHARACTER_CLASS_ERROR */
struct S2128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_BAD_CHARACTER_ERROR */
struct S2129 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_FULL_AND_META_ERROR */
struct S2130 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_FULL_AND_REJECT_ERROR */
struct S2131 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
struct S2132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] LX_CHARACTER_OUT_OF_RANGE_ERROR */
struct S2133 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] ARRAY [[attached] LX_RULE] */
struct S2134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] LX_RULE] */
struct S2135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] LX_DFA_STATE */
struct S2136 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* accepted_rules */
	T0* a2; /* states */
	T0* a3; /* transitions */
	T0* a4; /* accepted_head_rules */
	T6 a5; /* code */
	T6 a6; /* id */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] LX_NFA_STATE] */
struct S2137 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* equality_tester */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
	T0* a6; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] LX_DFA_STATE] */
struct S2138 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type detachable LX_SYMBOL_PARTITIONS */
struct S2139 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* symbols */
	T0* a2; /* storage */
	T6 a3; /* new_upper */
};

/* Struct for type [detachable] KL_ARRAY_ROUTINES [[attached] LX_RULE] */
struct S2140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ARRAY [detachable LX_RULE] */
struct S2141 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [detachable LX_RULE] */
struct S2142 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] LX_START_CONDITION */
struct S2143 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* id */
	T1 a3; /* is_exclusive */
	T0* a4; /* patterns */
	T0* a5; /* bol_patterns */
};

/* Struct for type [detachable] LX_TRANSITION_TABLE [[attached] LX_DFA_STATE] */
struct S2144 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* lower */
	T6 a2; /* upper */
	T6 a3; /* minimum_label */
	T6 a4; /* second_minimum_label */
	T6 a5; /* maximum_label */
	T0* a6; /* transitions */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] LX_NFA] */
struct S2145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] LX_NFA, [attached] INTEGER_32] */
struct S2146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T6 a4; /* count */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* free_slot */
	T6 a8; /* last_position */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [detachable] LX_NFA_STATE */
struct S2147 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* in_trail_context */
	T0* a2; /* transition */
	T0* a3; /* epsilon_transition */
	T6 a4; /* id */
	T0* a5; /* accepted_rule */
};

/* Struct for type detachable ARRAY [detachable LX_SYMBOL_CLASS] */
struct S2148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [detachable LX_SYMBOL_CLASS] */
struct S2149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable ARRAY [detachable DS_HASH_SET [[attached] INTEGER_32]] */
struct S2150 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [detachable DS_HASH_SET [[attached] INTEGER_32]] */
struct S2151 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
struct S2152 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS] */
struct S2153 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
struct S2154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_RULE] */
struct S2155 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_RULE] */
struct S2156 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] KL_ARRAY_ROUTINES [[attached] STRING_8] */
struct S2157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] LX_START_CONDITION] */
struct S2158 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_START_CONDITION] */
struct S2159 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_START_CONDITION] */
struct S2160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]]] */
struct S2161 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] LX_SYMBOL_CLASS] */
struct S2162 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]]] */
struct S2164 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]], [attached] LX_SYMBOL_CLASS] */
struct S2165 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type detachable SPECIAL [detachable SPECIAL [[attached] NATURAL_64]] */
struct S2166 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] STRING_32, [attached] STRING_8] */
struct S2168 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_SYMBOL_CLASS, [attached] STRING_32] */
struct S2170 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] LX_EPSILON_TRANSITION [[attached] LX_NFA_STATE] */
struct S2171 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
};

/* Struct for type [detachable] LX_SYMBOL_CLASS_TRANSITION [[attached] LX_NFA_STATE] */
struct S2174 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* label */
	T0* a2; /* target */
};

/* Struct for type [detachable] LX_SYMBOL_TRANSITION [[attached] LX_NFA_STATE] */
struct S2175 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* label */
	T0* a2; /* target */
};

/* Struct for type [detachable] LX_ACTION */
struct S2177 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* text */
};

/* Struct for type [detachable] ARRAY [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
struct S2178 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_BILINKABLE [[attached] INTEGER_32] */
struct S2179 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
	T0* a2; /* left */
	T0* a3; /* right */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
struct S2180 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_BUBBLE_SORTER [[attached] LX_NFA_STATE] */
struct S2181 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [detachable] DS_BUBBLE_SORTER [[attached] LX_RULE] */
struct S2183 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [detachable] SPECIAL [[attached] LX_NFA_STATE] */
struct S2185 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_NFA_STATE] */
struct S2187 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA_STATE] */
struct S2188 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [detachable] SPECIAL [[attached] LX_DFA_STATE] */
struct S2190 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_DFA_STATE] */
struct S2191 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_DFA_STATE] */
struct S2192 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_ARRAY [[attached] LX_RULE] */
struct S2193 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] LX_DFA_STATE, [attached] INTEGER_32] */
struct S2194 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* special_key_routines */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T0* a16; /* hash_function */
	T6 a17; /* count */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA] */
struct S2195 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_NFA, [attached] INTEGER_32] */
struct S2197 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_COMPARABLE_COMPARATOR [[attached] LX_RULE] */
struct S2201 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_COMPARABLE_COMPARATOR [[attached] LX_NFA_STATE] */
struct S2204 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
struct S2207 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
struct S2208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
struct S2209 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_SYMBOL_CLASS] */
struct S2210 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] KL_ARRAY [[attached] STRING_8] */
struct S2211 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_DFA_STATE, [attached] INTEGER_32] */
struct S2213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};


extern T1 GE_default1;
extern T2 GE_default2;
extern T3 GE_default3;
extern T4 GE_default4;
extern T5 GE_default5;
extern T6 GE_default6;
extern T7 GE_default7;
extern T8 GE_default8;
extern T9 GE_default9;
extern T10 GE_default10;
extern T11 GE_default11;
extern T12 GE_default12;
extern T13 GE_default13;
extern T14 GE_default14;
extern T15 GE_default15;
extern T16 GE_default16;
extern T17 GE_default17;
extern T18 GE_default18;
extern T20 GE_default20;
extern T21 GE_default21;
extern T25 GE_default25;
extern T26 GE_default26;
extern T27 GE_default27;
extern T28 GE_default28;
extern T29 GE_default29;
extern T30 GE_default30;
extern T31 GE_default31;
extern T32 GE_default32;
extern T33 GE_default33;
extern T34 GE_default34;
extern T35 GE_default35;
extern T36 GE_default36;
extern T37 GE_default37;
extern T38 GE_default38;
extern T39 GE_default39;
extern T40 GE_default40;
extern T42 GE_default42;
extern T43 GE_default43;
extern T44 GE_default44;
extern T46 GE_default46;
extern T47 GE_default47;
extern T48 GE_default48;
extern T49 GE_default49;
extern T53 GE_default53;
extern T55 GE_default55;
extern T56 GE_default56;
extern T57 GE_default57;
extern T58 GE_default58;
extern T59 GE_default59;
extern T61 GE_default61;
extern T62 GE_default62;
extern T63 GE_default63;
extern T64 GE_default64;
extern T67 GE_default67;
extern T68 GE_default68;
extern T70 GE_default70;
extern T71 GE_default71;
extern T72 GE_default72;
extern T73 GE_default73;
extern T74 GE_default74;
extern T76 GE_default76;
extern T77 GE_default77;
extern T79 GE_default79;
extern T80 GE_default80;
extern T81 GE_default81;
extern T82 GE_default82;
extern T86 GE_default86;
extern T87 GE_default87;
extern T88 GE_default88;
extern T89 GE_default89;
extern T90 GE_default90;
extern T93 GE_default93;
extern T95 GE_default95;
extern T97 GE_default97;
extern T99 GE_default99;
extern T100 GE_default100;
extern T101 GE_default101;
extern T102 GE_default102;
extern T103 GE_default103;
extern T104 GE_default104;
extern T105 GE_default105;
extern T107 GE_default107;
extern T108 GE_default108;
extern T109 GE_default109;
extern T110 GE_default110;
extern T111 GE_default111;
extern T112 GE_default112;
extern T113 GE_default113;
extern T114 GE_default114;
extern T115 GE_default115;
extern T116 GE_default116;
extern T117 GE_default117;
extern T118 GE_default118;
extern T119 GE_default119;
extern T120 GE_default120;
extern T121 GE_default121;
extern T122 GE_default122;
extern T123 GE_default123;
extern T124 GE_default124;
extern T125 GE_default125;
extern T126 GE_default126;
extern T127 GE_default127;
extern T128 GE_default128;
extern T129 GE_default129;
extern T130 GE_default130;
extern T131 GE_default131;
extern T132 GE_default132;
extern T133 GE_default133;
extern T134 GE_default134;
extern T135 GE_default135;
extern T137 GE_default137;
extern T140 GE_default140;
extern T141 GE_default141;
extern T142 GE_default142;
extern T144 GE_default144;
extern T145 GE_default145;
extern T146 GE_default146;
extern T147 GE_default147;
extern T148 GE_default148;
extern T149 GE_default149;
extern T150 GE_default150;
extern T151 GE_default151;
extern T152 GE_default152;
extern T153 GE_default153;
extern T154 GE_default154;
extern T157 GE_default157;
extern T160 GE_default160;
extern T161 GE_default161;
extern T162 GE_default162;
extern T163 GE_default163;
extern T164 GE_default164;
extern T165 GE_default165;
extern T166 GE_default166;
extern T167 GE_default167;
extern T168 GE_default168;
extern T169 GE_default169;
extern T170 GE_default170;
extern T171 GE_default171;
extern T172 GE_default172;
extern T174 GE_default174;
extern T175 GE_default175;
extern T176 GE_default176;
extern T178 GE_default178;
extern T179 GE_default179;
extern T182 GE_default182;
extern T183 GE_default183;
extern T184 GE_default184;
extern T185 GE_default185;
extern T187 GE_default187;
extern T188 GE_default188;
extern T190 GE_default190;
extern T191 GE_default191;
extern T192 GE_default192;
extern T193 GE_default193;
extern T195 GE_default195;
extern T196 GE_default196;
extern T198 GE_default198;
extern T199 GE_default199;
extern T200 GE_default200;
extern T201 GE_default201;
extern T202 GE_default202;
extern T203 GE_default203;
extern T204 GE_default204;
extern T205 GE_default205;
extern T206 GE_default206;
extern T207 GE_default207;
extern T208 GE_default208;
extern T210 GE_default210;
extern T211 GE_default211;
extern T213 GE_default213;
extern T214 GE_default214;
extern T215 GE_default215;
extern T216 GE_default216;
extern T217 GE_default217;
extern T218 GE_default218;
extern T219 GE_default219;
extern T221 GE_default221;
extern T223 GE_default223;
extern T224 GE_default224;
extern T225 GE_default225;
extern T226 GE_default226;
extern T227 GE_default227;
extern T228 GE_default228;
extern T229 GE_default229;
extern T230 GE_default230;
extern T231 GE_default231;
extern T236 GE_default236;
extern T240 GE_default240;
extern T241 GE_default241;
extern T242 GE_default242;
extern T244 GE_default244;
extern T245 GE_default245;
extern T253 GE_default253;
extern T254 GE_default254;
extern T255 GE_default255;
extern T256 GE_default256;
extern T257 GE_default257;
extern T258 GE_default258;
extern T259 GE_default259;
extern T260 GE_default260;
extern T261 GE_default261;
extern T262 GE_default262;
extern T263 GE_default263;
extern T264 GE_default264;
extern T265 GE_default265;
extern T266 GE_default266;
extern T267 GE_default267;
extern T268 GE_default268;
extern T269 GE_default269;
extern T270 GE_default270;
extern T271 GE_default271;
extern T272 GE_default272;
extern T273 GE_default273;
extern T274 GE_default274;
extern T275 GE_default275;
extern T276 GE_default276;
extern T277 GE_default277;
extern T278 GE_default278;
extern T279 GE_default279;
extern T280 GE_default280;
extern T282 GE_default282;
extern T283 GE_default283;
extern T284 GE_default284;
extern T287 GE_default287;
extern T288 GE_default288;
extern T289 GE_default289;
extern T290 GE_default290;
extern T291 GE_default291;
extern T292 GE_default292;
extern T293 GE_default293;
extern T294 GE_default294;
extern T295 GE_default295;
extern T296 GE_default296;
extern T297 GE_default297;
extern T298 GE_default298;
extern T300 GE_default300;
extern T301 GE_default301;
extern T302 GE_default302;
extern T303 GE_default303;
extern T304 GE_default304;
extern T305 GE_default305;
extern T306 GE_default306;
extern T307 GE_default307;
extern T308 GE_default308;
extern T309 GE_default309;
extern T311 GE_default311;
extern T312 GE_default312;
extern T313 GE_default313;
extern T314 GE_default314;
extern T318 GE_default318;
extern T319 GE_default319;
extern T322 GE_default322;
extern T323 GE_default323;
extern T324 GE_default324;
extern T328 GE_default328;
extern T329 GE_default329;
extern T333 GE_default333;
extern T334 GE_default334;
extern T341 GE_default341;
extern T344 GE_default344;
extern T346 GE_default346;
extern T347 GE_default347;
extern T348 GE_default348;
extern T349 GE_default349;
extern T350 GE_default350;
extern T351 GE_default351;
extern T355 GE_default355;
extern T357 GE_default357;
extern T358 GE_default358;
extern T359 GE_default359;
extern T360 GE_default360;
extern T361 GE_default361;
extern T365 GE_default365;
extern T367 GE_default367;
extern T368 GE_default368;
extern T369 GE_default369;
extern T371 GE_default371;
extern T372 GE_default372;
extern T375 GE_default375;
extern T376 GE_default376;
extern T377 GE_default377;
extern T379 GE_default379;
extern T380 GE_default380;
extern T381 GE_default381;
extern T382 GE_default382;
extern T384 GE_default384;
extern T386 GE_default386;
extern T387 GE_default387;
extern T388 GE_default388;
extern T389 GE_default389;
extern T390 GE_default390;
extern T391 GE_default391;
extern T392 GE_default392;
extern T393 GE_default393;
extern T394 GE_default394;
extern T395 GE_default395;
extern T396 GE_default396;
extern T397 GE_default397;
extern T398 GE_default398;
extern T399 GE_default399;
extern T400 GE_default400;
extern T401 GE_default401;
extern T402 GE_default402;
extern T403 GE_default403;
extern T404 GE_default404;
extern T405 GE_default405;
extern T406 GE_default406;
extern T408 GE_default408;
extern T410 GE_default410;
extern T412 GE_default412;
extern T413 GE_default413;
extern T414 GE_default414;
extern T415 GE_default415;
extern T416 GE_default416;
extern T417 GE_default417;
extern T422 GE_default422;
extern T423 GE_default423;
extern T424 GE_default424;
extern T426 GE_default426;
extern T427 GE_default427;
extern T428 GE_default428;
extern T429 GE_default429;
extern T430 GE_default430;
extern T431 GE_default431;
extern T432 GE_default432;
extern T434 GE_default434;
extern T435 GE_default435;
extern T436 GE_default436;
extern T444 GE_default444;
extern T445 GE_default445;
extern T446 GE_default446;
extern T447 GE_default447;
extern T449 GE_default449;
extern T450 GE_default450;
extern T451 GE_default451;
extern T452 GE_default452;
extern T453 GE_default453;
extern T454 GE_default454;
extern T455 GE_default455;
extern T456 GE_default456;
extern T457 GE_default457;
extern T458 GE_default458;
extern T459 GE_default459;
extern T460 GE_default460;
extern T461 GE_default461;
extern T462 GE_default462;
extern T463 GE_default463;
extern T464 GE_default464;
extern T465 GE_default465;
extern T467 GE_default467;
extern T468 GE_default468;
extern T469 GE_default469;
extern T471 GE_default471;
extern T472 GE_default472;
extern T475 GE_default475;
extern T476 GE_default476;
extern T477 GE_default477;
extern T478 GE_default478;
extern T479 GE_default479;
extern T480 GE_default480;
extern T481 GE_default481;
extern T483 GE_default483;
extern T484 GE_default484;
extern T485 GE_default485;
extern T486 GE_default486;
extern T487 GE_default487;
extern T488 GE_default488;
extern T489 GE_default489;
extern T491 GE_default491;
extern T492 GE_default492;
extern T493 GE_default493;
extern T494 GE_default494;
extern T495 GE_default495;
extern T496 GE_default496;
extern T497 GE_default497;
extern T498 GE_default498;
extern T499 GE_default499;
extern T500 GE_default500;
extern T501 GE_default501;
extern T502 GE_default502;
extern T503 GE_default503;
extern T504 GE_default504;
extern T506 GE_default506;
extern T507 GE_default507;
extern T510 GE_default510;
extern T511 GE_default511;
extern T514 GE_default514;
extern T515 GE_default515;
extern T516 GE_default516;
extern T517 GE_default517;
extern T519 GE_default519;
extern T520 GE_default520;
extern T521 GE_default521;
extern T522 GE_default522;
extern T524 GE_default524;
extern T526 GE_default526;
extern T527 GE_default527;
extern T528 GE_default528;
extern T529 GE_default529;
extern T530 GE_default530;
extern T531 GE_default531;
extern T532 GE_default532;
extern T533 GE_default533;
extern T534 GE_default534;
extern T535 GE_default535;
extern T536 GE_default536;
extern T537 GE_default537;
extern T538 GE_default538;
extern T539 GE_default539;
extern T540 GE_default540;
extern T541 GE_default541;
extern T542 GE_default542;
extern T543 GE_default543;
extern T544 GE_default544;
extern T545 GE_default545;
extern T546 GE_default546;
extern T547 GE_default547;
extern T548 GE_default548;
extern T549 GE_default549;
extern T550 GE_default550;
extern T551 GE_default551;
extern T552 GE_default552;
extern T553 GE_default553;
extern T556 GE_default556;
extern T557 GE_default557;
extern T559 GE_default559;
extern T560 GE_default560;
extern T561 GE_default561;
extern T562 GE_default562;
extern T563 GE_default563;
extern T564 GE_default564;
extern T565 GE_default565;
extern T567 GE_default567;
extern T568 GE_default568;
extern T569 GE_default569;
extern T570 GE_default570;
extern T571 GE_default571;
extern T572 GE_default572;
extern T573 GE_default573;
extern T574 GE_default574;
extern T577 GE_default577;
extern T578 GE_default578;
extern T579 GE_default579;
extern T582 GE_default582;
extern T584 GE_default584;
extern T585 GE_default585;
extern T586 GE_default586;
extern T587 GE_default587;
extern T589 GE_default589;
extern T591 GE_default591;
extern T592 GE_default592;
extern T593 GE_default593;
extern T594 GE_default594;
extern T595 GE_default595;
extern T596 GE_default596;
extern T599 GE_default599;
extern T600 GE_default600;
extern T601 GE_default601;
extern T602 GE_default602;
extern T603 GE_default603;
extern T604 GE_default604;
extern T605 GE_default605;
extern T606 GE_default606;
extern T609 GE_default609;
extern T610 GE_default610;
extern T611 GE_default611;
extern T612 GE_default612;
extern T613 GE_default613;
extern T615 GE_default615;
extern T616 GE_default616;
extern T617 GE_default617;
extern T621 GE_default621;
extern T622 GE_default622;
extern T624 GE_default624;
extern T625 GE_default625;
extern T626 GE_default626;
extern T627 GE_default627;
extern T628 GE_default628;
extern T629 GE_default629;
extern T630 GE_default630;
extern T631 GE_default631;
extern T632 GE_default632;
extern T634 GE_default634;
extern T635 GE_default635;
extern T636 GE_default636;
extern T637 GE_default637;
extern T638 GE_default638;
extern T639 GE_default639;
extern T640 GE_default640;
extern T641 GE_default641;
extern T642 GE_default642;
extern T643 GE_default643;
extern T644 GE_default644;
extern T645 GE_default645;
extern T646 GE_default646;
extern T647 GE_default647;
extern T648 GE_default648;
extern T649 GE_default649;
extern T650 GE_default650;
extern T651 GE_default651;
extern T652 GE_default652;
extern T653 GE_default653;
extern T654 GE_default654;
extern T655 GE_default655;
extern T657 GE_default657;
extern T658 GE_default658;
extern T659 GE_default659;
extern T660 GE_default660;
extern T661 GE_default661;
extern T662 GE_default662;
extern T663 GE_default663;
extern T664 GE_default664;
extern T665 GE_default665;
extern T666 GE_default666;
extern T667 GE_default667;
extern T668 GE_default668;
extern T669 GE_default669;
extern T671 GE_default671;
extern T672 GE_default672;
extern T673 GE_default673;
extern T674 GE_default674;
extern T676 GE_default676;
extern T677 GE_default677;
extern T678 GE_default678;
extern T679 GE_default679;
extern T680 GE_default680;
extern T682 GE_default682;
extern T683 GE_default683;
extern T684 GE_default684;
extern T686 GE_default686;
extern T687 GE_default687;
extern T689 GE_default689;
extern T690 GE_default690;
extern T691 GE_default691;
extern T692 GE_default692;
extern T695 GE_default695;
extern T697 GE_default697;
extern T698 GE_default698;
extern T699 GE_default699;
extern T700 GE_default700;
extern T703 GE_default703;
extern T704 GE_default704;
extern T707 GE_default707;
extern T708 GE_default708;
extern T709 GE_default709;
extern T710 GE_default710;
extern T711 GE_default711;
extern T712 GE_default712;
extern T713 GE_default713;
extern T714 GE_default714;
extern T715 GE_default715;
extern T716 GE_default716;
extern T717 GE_default717;
extern T718 GE_default718;
extern T719 GE_default719;
extern T720 GE_default720;
extern T721 GE_default721;
extern T722 GE_default722;
extern T723 GE_default723;
extern T724 GE_default724;
extern T725 GE_default725;
extern T726 GE_default726;
extern T727 GE_default727;
extern T728 GE_default728;
extern T729 GE_default729;
extern T730 GE_default730;
extern T731 GE_default731;
extern T732 GE_default732;
extern T733 GE_default733;
extern T736 GE_default736;
extern T737 GE_default737;
extern T738 GE_default738;
extern T739 GE_default739;
extern T740 GE_default740;
extern T741 GE_default741;
extern T743 GE_default743;
extern T744 GE_default744;
extern T745 GE_default745;
extern T747 GE_default747;
extern T748 GE_default748;
extern T749 GE_default749;
extern T750 GE_default750;
extern T751 GE_default751;
extern T753 GE_default753;
extern T754 GE_default754;
extern T756 GE_default756;
extern T757 GE_default757;
extern T758 GE_default758;
extern T759 GE_default759;
extern T760 GE_default760;
extern T761 GE_default761;
extern T762 GE_default762;
extern T763 GE_default763;
extern T764 GE_default764;
extern T765 GE_default765;
extern T766 GE_default766;
extern T767 GE_default767;
extern T768 GE_default768;
extern T769 GE_default769;
extern T770 GE_default770;
extern T771 GE_default771;
extern T772 GE_default772;
extern T775 GE_default775;
extern T776 GE_default776;
extern T777 GE_default777;
extern T780 GE_default780;
extern T781 GE_default781;
extern T782 GE_default782;
extern T783 GE_default783;
extern T784 GE_default784;
extern T785 GE_default785;
extern T786 GE_default786;
extern T787 GE_default787;
extern T788 GE_default788;
extern T789 GE_default789;
extern T790 GE_default790;
extern T791 GE_default791;
extern T792 GE_default792;
extern T793 GE_default793;
extern T794 GE_default794;
extern T795 GE_default795;
extern T796 GE_default796;
extern T797 GE_default797;
extern T798 GE_default798;
extern T799 GE_default799;
extern T801 GE_default801;
extern T802 GE_default802;
extern T803 GE_default803;
extern T804 GE_default804;
extern T805 GE_default805;
extern T806 GE_default806;
extern T807 GE_default807;
extern T808 GE_default808;
extern T809 GE_default809;
extern T810 GE_default810;
extern T811 GE_default811;
extern T812 GE_default812;
extern T813 GE_default813;
extern T814 GE_default814;
extern T815 GE_default815;
extern T816 GE_default816;
extern T817 GE_default817;
extern T818 GE_default818;
extern T819 GE_default819;
extern T820 GE_default820;
extern T822 GE_default822;
extern T823 GE_default823;
extern T824 GE_default824;
extern T825 GE_default825;
extern T826 GE_default826;
extern T827 GE_default827;
extern T828 GE_default828;
extern T830 GE_default830;
extern T831 GE_default831;
extern T833 GE_default833;
extern T834 GE_default834;
extern T835 GE_default835;
extern T836 GE_default836;
extern T837 GE_default837;
extern T838 GE_default838;
extern T839 GE_default839;
extern T840 GE_default840;
extern T842 GE_default842;
extern T843 GE_default843;
extern T844 GE_default844;
extern T845 GE_default845;
extern T846 GE_default846;
extern T847 GE_default847;
extern T848 GE_default848;
extern T849 GE_default849;
extern T850 GE_default850;
extern T851 GE_default851;
extern T852 GE_default852;
extern T853 GE_default853;
extern T854 GE_default854;
extern T855 GE_default855;
extern T857 GE_default857;
extern T858 GE_default858;
extern T859 GE_default859;
extern T860 GE_default860;
extern T861 GE_default861;
extern T862 GE_default862;
extern T863 GE_default863;
extern T864 GE_default864;
extern T865 GE_default865;
extern T866 GE_default866;
extern T867 GE_default867;
extern T868 GE_default868;
extern T869 GE_default869;
extern T870 GE_default870;
extern T872 GE_default872;
extern T873 GE_default873;
extern T874 GE_default874;
extern T875 GE_default875;
extern T876 GE_default876;
extern T877 GE_default877;
extern T880 GE_default880;
extern T884 GE_default884;
extern T885 GE_default885;
extern T886 GE_default886;
extern T887 GE_default887;
extern T888 GE_default888;
extern T890 GE_default890;
extern T891 GE_default891;
extern T892 GE_default892;
extern T893 GE_default893;
extern T894 GE_default894;
extern T896 GE_default896;
extern T898 GE_default898;
extern T899 GE_default899;
extern T900 GE_default900;
extern T901 GE_default901;
extern T902 GE_default902;
extern T903 GE_default903;
extern T904 GE_default904;
extern T905 GE_default905;
extern T906 GE_default906;
extern T907 GE_default907;
extern T909 GE_default909;
extern T910 GE_default910;
extern T911 GE_default911;
extern T912 GE_default912;
extern T913 GE_default913;
extern T914 GE_default914;
extern T921 GE_default921;
extern T923 GE_default923;
extern T924 GE_default924;
extern T925 GE_default925;
extern T926 GE_default926;
extern T927 GE_default927;
extern T928 GE_default928;
extern T929 GE_default929;
extern T930 GE_default930;
extern T931 GE_default931;
extern T932 GE_default932;
extern T936 GE_default936;
extern T937 GE_default937;
extern T938 GE_default938;
extern T939 GE_default939;
extern T940 GE_default940;
extern T941 GE_default941;
extern T942 GE_default942;
extern T943 GE_default943;
extern T944 GE_default944;
extern T945 GE_default945;
extern T946 GE_default946;
extern T947 GE_default947;
extern T948 GE_default948;
extern T949 GE_default949;
extern T950 GE_default950;
extern T951 GE_default951;
extern T952 GE_default952;
extern T953 GE_default953;
extern T954 GE_default954;
extern T955 GE_default955;
extern T956 GE_default956;
extern T957 GE_default957;
extern T958 GE_default958;
extern T959 GE_default959;
extern T960 GE_default960;
extern T961 GE_default961;
extern T963 GE_default963;
extern T964 GE_default964;
extern T966 GE_default966;
extern T967 GE_default967;
extern T969 GE_default969;
extern T970 GE_default970;
extern T971 GE_default971;
extern T972 GE_default972;
extern T973 GE_default973;
extern T974 GE_default974;
extern T975 GE_default975;
extern T976 GE_default976;
extern T977 GE_default977;
extern T979 GE_default979;
extern T980 GE_default980;
extern T981 GE_default981;
extern T982 GE_default982;
extern T983 GE_default983;
extern T984 GE_default984;
extern T985 GE_default985;
extern T987 GE_default987;
extern T988 GE_default988;
extern T989 GE_default989;
extern T990 GE_default990;
extern T991 GE_default991;
extern T992 GE_default992;
extern T993 GE_default993;
extern T994 GE_default994;
extern T995 GE_default995;
extern T996 GE_default996;
extern T997 GE_default997;
extern T998 GE_default998;
extern T999 GE_default999;
extern T1000 GE_default1000;
extern T1001 GE_default1001;
extern T1002 GE_default1002;
extern T1003 GE_default1003;
extern T1004 GE_default1004;
extern T1005 GE_default1005;
extern T1006 GE_default1006;
extern T1007 GE_default1007;
extern T1008 GE_default1008;
extern T1009 GE_default1009;
extern T1010 GE_default1010;
extern T1011 GE_default1011;
extern T1012 GE_default1012;
extern T1013 GE_default1013;
extern T1014 GE_default1014;
extern T1015 GE_default1015;
extern T1016 GE_default1016;
extern T1017 GE_default1017;
extern T1018 GE_default1018;
extern T1019 GE_default1019;
extern T1020 GE_default1020;
extern T1021 GE_default1021;
extern T1023 GE_default1023;
extern T1024 GE_default1024;
extern T1025 GE_default1025;
extern T1026 GE_default1026;
extern T1027 GE_default1027;
extern T1028 GE_default1028;
extern T1029 GE_default1029;
extern T1030 GE_default1030;
extern T1031 GE_default1031;
extern T1032 GE_default1032;
extern T1033 GE_default1033;
extern T1034 GE_default1034;
extern T1036 GE_default1036;
extern T1037 GE_default1037;
extern T1038 GE_default1038;
extern T1039 GE_default1039;
extern T1040 GE_default1040;
extern T1041 GE_default1041;
extern T1042 GE_default1042;
extern T1043 GE_default1043;
extern T1044 GE_default1044;
extern T1045 GE_default1045;
extern T1046 GE_default1046;
extern T1047 GE_default1047;
extern T1048 GE_default1048;
extern T1050 GE_default1050;
extern T1051 GE_default1051;
extern T1053 GE_default1053;
extern T1054 GE_default1054;
extern T1056 GE_default1056;
extern T1057 GE_default1057;
extern T1058 GE_default1058;
extern T1059 GE_default1059;
extern T1060 GE_default1060;
extern T1061 GE_default1061;
extern T1062 GE_default1062;
extern T1063 GE_default1063;
extern T1064 GE_default1064;
extern T1065 GE_default1065;
extern T1066 GE_default1066;
extern T1067 GE_default1067;
extern T1068 GE_default1068;
extern T1069 GE_default1069;
extern T1071 GE_default1071;
extern T1072 GE_default1072;
extern T1073 GE_default1073;
extern T1074 GE_default1074;
extern T1076 GE_default1076;
extern T1077 GE_default1077;
extern T1078 GE_default1078;
extern T1079 GE_default1079;
extern T1080 GE_default1080;
extern T1081 GE_default1081;
extern T1082 GE_default1082;
extern T1083 GE_default1083;
extern T1084 GE_default1084;
extern T1085 GE_default1085;
extern T1086 GE_default1086;
extern T1088 GE_default1088;
extern T1089 GE_default1089;
extern T1091 GE_default1091;
extern T1092 GE_default1092;
extern T1094 GE_default1094;
extern T1095 GE_default1095;
extern T1096 GE_default1096;
extern T1097 GE_default1097;
extern T1098 GE_default1098;
extern T1099 GE_default1099;
extern T1100 GE_default1100;
extern T1101 GE_default1101;
extern T1102 GE_default1102;
extern T1103 GE_default1103;
extern T1104 GE_default1104;
extern T1105 GE_default1105;
extern T1106 GE_default1106;
extern T1107 GE_default1107;
extern T1108 GE_default1108;
extern T1109 GE_default1109;
extern T1110 GE_default1110;
extern T1111 GE_default1111;
extern T1112 GE_default1112;
extern T1113 GE_default1113;
extern T1114 GE_default1114;
extern T1115 GE_default1115;
extern T1116 GE_default1116;
extern T1117 GE_default1117;
extern T1118 GE_default1118;
extern T1119 GE_default1119;
extern T1120 GE_default1120;
extern T1121 GE_default1121;
extern T1122 GE_default1122;
extern T1123 GE_default1123;
extern T1124 GE_default1124;
extern T1126 GE_default1126;
extern T1127 GE_default1127;
extern T1128 GE_default1128;
extern T1129 GE_default1129;
extern T1130 GE_default1130;
extern T1131 GE_default1131;
extern T1132 GE_default1132;
extern T1133 GE_default1133;
extern T1134 GE_default1134;
extern T1135 GE_default1135;
extern T1137 GE_default1137;
extern T1138 GE_default1138;
extern T1139 GE_default1139;
extern T1140 GE_default1140;
extern T1141 GE_default1141;
extern T1142 GE_default1142;
extern T1143 GE_default1143;
extern T1144 GE_default1144;
extern T1145 GE_default1145;
extern T1146 GE_default1146;
extern T1147 GE_default1147;
extern T1148 GE_default1148;
extern T1149 GE_default1149;
extern T1151 GE_default1151;
extern T1152 GE_default1152;
extern T1153 GE_default1153;
extern T1154 GE_default1154;
extern T1156 GE_default1156;
extern T1157 GE_default1157;
extern T1158 GE_default1158;
extern T1159 GE_default1159;
extern T1160 GE_default1160;
extern T1161 GE_default1161;
extern T1163 GE_default1163;
extern T1164 GE_default1164;
extern T1165 GE_default1165;
extern T1166 GE_default1166;
extern T1167 GE_default1167;
extern T1168 GE_default1168;
extern T1169 GE_default1169;
extern T1170 GE_default1170;
extern T1171 GE_default1171;
extern T1172 GE_default1172;
extern T1173 GE_default1173;
extern T1174 GE_default1174;
extern T1175 GE_default1175;
extern T1177 GE_default1177;
extern T1178 GE_default1178;
extern T1179 GE_default1179;
extern T1180 GE_default1180;
extern T1182 GE_default1182;
extern T1183 GE_default1183;
extern T1184 GE_default1184;
extern T1185 GE_default1185;
extern T1186 GE_default1186;
extern T1187 GE_default1187;
extern T1188 GE_default1188;
extern T1189 GE_default1189;
extern T1190 GE_default1190;
extern T1191 GE_default1191;
extern T1192 GE_default1192;
extern T1193 GE_default1193;
extern T1194 GE_default1194;
extern T1196 GE_default1196;
extern T1197 GE_default1197;
extern T1198 GE_default1198;
extern T1199 GE_default1199;
extern T1200 GE_default1200;
extern T1201 GE_default1201;
extern T1202 GE_default1202;
extern T1204 GE_default1204;
extern T1205 GE_default1205;
extern T1206 GE_default1206;
extern T1207 GE_default1207;
extern T1208 GE_default1208;
extern T1210 GE_default1210;
extern T1211 GE_default1211;
extern T1212 GE_default1212;
extern T1213 GE_default1213;
extern T1214 GE_default1214;
extern T1215 GE_default1215;
extern T1216 GE_default1216;
extern T1217 GE_default1217;
extern T1218 GE_default1218;
extern T1219 GE_default1219;
extern T1220 GE_default1220;
extern T1222 GE_default1222;
extern T1223 GE_default1223;
extern T1224 GE_default1224;
extern T1225 GE_default1225;
extern T1226 GE_default1226;
extern T1227 GE_default1227;
extern T1228 GE_default1228;
extern T1230 GE_default1230;
extern T1231 GE_default1231;
extern T1232 GE_default1232;
extern T1233 GE_default1233;
extern T1234 GE_default1234;
extern T1235 GE_default1235;
extern T1236 GE_default1236;
extern T1237 GE_default1237;
extern T1238 GE_default1238;
extern T1239 GE_default1239;
extern T1240 GE_default1240;
extern T1241 GE_default1241;
extern T1242 GE_default1242;
extern T1243 GE_default1243;
extern T1244 GE_default1244;
extern T1245 GE_default1245;
extern T1246 GE_default1246;
extern T1247 GE_default1247;
extern T1248 GE_default1248;
extern T1249 GE_default1249;
extern T1250 GE_default1250;
extern T1252 GE_default1252;
extern T1253 GE_default1253;
extern T1254 GE_default1254;
extern T1255 GE_default1255;
extern T1256 GE_default1256;
extern T1257 GE_default1257;
extern T1258 GE_default1258;
extern T1260 GE_default1260;
extern T1261 GE_default1261;
extern T1262 GE_default1262;
extern T1263 GE_default1263;
extern T1264 GE_default1264;
extern T1265 GE_default1265;
extern T1266 GE_default1266;
extern T1267 GE_default1267;
extern T1268 GE_default1268;
extern T1269 GE_default1269;
extern T1270 GE_default1270;
extern T1271 GE_default1271;
extern T1272 GE_default1272;
extern T1273 GE_default1273;
extern T1274 GE_default1274;
extern T1276 GE_default1276;
extern T1277 GE_default1277;
extern T1279 GE_default1279;
extern T1280 GE_default1280;
extern T1281 GE_default1281;
extern T1282 GE_default1282;
extern T1283 GE_default1283;
extern T1285 GE_default1285;
extern T1286 GE_default1286;
extern T1288 GE_default1288;
extern T1289 GE_default1289;
extern T1290 GE_default1290;
extern T1291 GE_default1291;
extern T1292 GE_default1292;
extern T1293 GE_default1293;
extern T1294 GE_default1294;
extern T1295 GE_default1295;
extern T1296 GE_default1296;
extern T1297 GE_default1297;
extern T1298 GE_default1298;
extern T1299 GE_default1299;
extern T1300 GE_default1300;
extern T1301 GE_default1301;
extern T1302 GE_default1302;
extern T1303 GE_default1303;
extern T1304 GE_default1304;
extern T1305 GE_default1305;
extern T1306 GE_default1306;
extern T1307 GE_default1307;
extern T1308 GE_default1308;
extern T1309 GE_default1309;
extern T1310 GE_default1310;
extern T1311 GE_default1311;
extern T1313 GE_default1313;
extern T1314 GE_default1314;
extern T1316 GE_default1316;
extern T1317 GE_default1317;
extern T1318 GE_default1318;
extern T1319 GE_default1319;
extern T1320 GE_default1320;
extern T1321 GE_default1321;
extern T1322 GE_default1322;
extern T1323 GE_default1323;
extern T1324 GE_default1324;
extern T1325 GE_default1325;
extern T1326 GE_default1326;
extern T1327 GE_default1327;
extern T1328 GE_default1328;
extern T1329 GE_default1329;
extern T1330 GE_default1330;
extern T1331 GE_default1331;
extern T1332 GE_default1332;
extern T1333 GE_default1333;
extern T1337 GE_default1337;
extern T1338 GE_default1338;
extern T1339 GE_default1339;
extern T1340 GE_default1340;
extern T1341 GE_default1341;
extern T1344 GE_default1344;
extern T1345 GE_default1345;
extern T1346 GE_default1346;
extern T1347 GE_default1347;
extern T1348 GE_default1348;
extern T1349 GE_default1349;
extern T1350 GE_default1350;
extern T1351 GE_default1351;
extern T1353 GE_default1353;
extern T1354 GE_default1354;
extern T1355 GE_default1355;
extern T1356 GE_default1356;
extern T1358 GE_default1358;
extern T1359 GE_default1359;
extern T1360 GE_default1360;
extern T1361 GE_default1361;
extern T1364 GE_default1364;
extern T1365 GE_default1365;
extern T1368 GE_default1368;
extern T1369 GE_default1369;
extern T1370 GE_default1370;
extern T1371 GE_default1371;
extern T1372 GE_default1372;
extern T1373 GE_default1373;
extern T1374 GE_default1374;
extern T1375 GE_default1375;
extern T1378 GE_default1378;
extern T1379 GE_default1379;
extern T1380 GE_default1380;
extern T1381 GE_default1381;
extern T1382 GE_default1382;
extern T1383 GE_default1383;
extern T1384 GE_default1384;
extern T1386 GE_default1386;
extern T1387 GE_default1387;
extern T1388 GE_default1388;
extern T1389 GE_default1389;
extern T1390 GE_default1390;
extern T1391 GE_default1391;
extern T1392 GE_default1392;
extern T1393 GE_default1393;
extern T1395 GE_default1395;
extern T1396 GE_default1396;
extern T1397 GE_default1397;
extern T1398 GE_default1398;
extern T1400 GE_default1400;
extern T1402 GE_default1402;
extern T1404 GE_default1404;
extern T1406 GE_default1406;
extern T1407 GE_default1407;
extern T1408 GE_default1408;
extern T1410 GE_default1410;
extern T1411 GE_default1411;
extern T1412 GE_default1412;
extern T1413 GE_default1413;
extern T1414 GE_default1414;
extern T1415 GE_default1415;
extern T1416 GE_default1416;
extern T1418 GE_default1418;
extern T1419 GE_default1419;
extern T1420 GE_default1420;
extern T1421 GE_default1421;
extern T1423 GE_default1423;
extern T1424 GE_default1424;
extern T1425 GE_default1425;
extern T1426 GE_default1426;
extern T1427 GE_default1427;
extern T1428 GE_default1428;
extern T1429 GE_default1429;
extern T1433 GE_default1433;
extern T1434 GE_default1434;
extern T1435 GE_default1435;
extern T1436 GE_default1436;
extern T1443 GE_default1443;
extern T1446 GE_default1446;
extern T1448 GE_default1448;
extern T1449 GE_default1449;
extern T1450 GE_default1450;
extern T1451 GE_default1451;
extern T1452 GE_default1452;
extern T1453 GE_default1453;
extern T1455 GE_default1455;
extern T1456 GE_default1456;
extern T1457 GE_default1457;
extern T1458 GE_default1458;
extern T1459 GE_default1459;
extern T1460 GE_default1460;
extern T1461 GE_default1461;
extern T1463 GE_default1463;
extern T1464 GE_default1464;
extern T1465 GE_default1465;
extern T1466 GE_default1466;
extern T1468 GE_default1468;
extern T1469 GE_default1469;
extern T1470 GE_default1470;
extern T1471 GE_default1471;
extern T1474 GE_default1474;
extern T1477 GE_default1477;
extern T1479 GE_default1479;
extern T1481 GE_default1481;
extern T1482 GE_default1482;
extern T1483 GE_default1483;
extern T1484 GE_default1484;
extern T1485 GE_default1485;
extern T1486 GE_default1486;
extern T1487 GE_default1487;
extern T1488 GE_default1488;
extern T1489 GE_default1489;
extern T1490 GE_default1490;
extern T1491 GE_default1491;
extern T1493 GE_default1493;
extern T1494 GE_default1494;
extern T1495 GE_default1495;
extern T1496 GE_default1496;
extern T1498 GE_default1498;
extern T1499 GE_default1499;
extern T1500 GE_default1500;
extern T1501 GE_default1501;
extern T1502 GE_default1502;
extern T1503 GE_default1503;
extern T1504 GE_default1504;
extern T1505 GE_default1505;
extern T1506 GE_default1506;
extern T1508 GE_default1508;
extern T1509 GE_default1509;
extern T1511 GE_default1511;
extern T1512 GE_default1512;
extern T1513 GE_default1513;
extern T1515 GE_default1515;
extern T1516 GE_default1516;
extern T1517 GE_default1517;
extern T1518 GE_default1518;
extern T1521 GE_default1521;
extern T1522 GE_default1522;
extern T1524 GE_default1524;
extern T1525 GE_default1525;
extern T1526 GE_default1526;
extern T1527 GE_default1527;
extern T1528 GE_default1528;
extern T1529 GE_default1529;
extern T1530 GE_default1530;
extern T1532 GE_default1532;
extern T1533 GE_default1533;
extern T1534 GE_default1534;
extern T1535 GE_default1535;
extern T1537 GE_default1537;
extern T1538 GE_default1538;
extern T1539 GE_default1539;
extern T1540 GE_default1540;
extern T1541 GE_default1541;
extern T1542 GE_default1542;
extern T1543 GE_default1543;
extern T1544 GE_default1544;
extern T1545 GE_default1545;
extern T1546 GE_default1546;
extern T1547 GE_default1547;
extern T1548 GE_default1548;
extern T1549 GE_default1549;
extern T1550 GE_default1550;
extern T1551 GE_default1551;
extern T1552 GE_default1552;
extern T1553 GE_default1553;
extern T1554 GE_default1554;
extern T1555 GE_default1555;
extern T1556 GE_default1556;
extern T1557 GE_default1557;
extern T1558 GE_default1558;
extern T1559 GE_default1559;
extern T1562 GE_default1562;
extern T1564 GE_default1564;
extern T1566 GE_default1566;
extern T1567 GE_default1567;
extern T1569 GE_default1569;
extern T1571 GE_default1571;
extern T1573 GE_default1573;
extern T1576 GE_default1576;
extern T1577 GE_default1577;
extern T1578 GE_default1578;
extern T1579 GE_default1579;
extern T1580 GE_default1580;
extern T1581 GE_default1581;
extern T1582 GE_default1582;
extern T1583 GE_default1583;
extern T1584 GE_default1584;
extern T1585 GE_default1585;
extern T1586 GE_default1586;
extern T1587 GE_default1587;
extern T1588 GE_default1588;
extern T1589 GE_default1589;
extern T1590 GE_default1590;
extern T1591 GE_default1591;
extern T1592 GE_default1592;
extern T1593 GE_default1593;
extern T1594 GE_default1594;
extern T1597 GE_default1597;
extern T1598 GE_default1598;
extern T1599 GE_default1599;
extern T1600 GE_default1600;
extern T1601 GE_default1601;
extern T1602 GE_default1602;
extern T1603 GE_default1603;
extern T1604 GE_default1604;
extern T1605 GE_default1605;
extern T1606 GE_default1606;
extern T1608 GE_default1608;
extern T1609 GE_default1609;
extern T1611 GE_default1611;
extern T1612 GE_default1612;
extern T1614 GE_default1614;
extern T1615 GE_default1615;
extern T1616 GE_default1616;
extern T1617 GE_default1617;
extern T1618 GE_default1618;
extern T1619 GE_default1619;
extern T1620 GE_default1620;
extern T1621 GE_default1621;
extern T1622 GE_default1622;
extern T1623 GE_default1623;
extern T1624 GE_default1624;
extern T1626 GE_default1626;
extern T1627 GE_default1627;
extern T1628 GE_default1628;
extern T1629 GE_default1629;
extern T1630 GE_default1630;
extern T1631 GE_default1631;
extern T1632 GE_default1632;
extern T1633 GE_default1633;
extern T1634 GE_default1634;
extern T1635 GE_default1635;
extern T1636 GE_default1636;
extern T1637 GE_default1637;
extern T1638 GE_default1638;
extern T1639 GE_default1639;
extern T1640 GE_default1640;
extern T1641 GE_default1641;
extern T1642 GE_default1642;
extern T1643 GE_default1643;
extern T1644 GE_default1644;
extern T1645 GE_default1645;
extern T1646 GE_default1646;
extern T1647 GE_default1647;
extern T1648 GE_default1648;
extern T1649 GE_default1649;
extern T1650 GE_default1650;
extern T1651 GE_default1651;
extern T1653 GE_default1653;
extern T1655 GE_default1655;
extern T1656 GE_default1656;
extern T1659 GE_default1659;
extern T1660 GE_default1660;
extern T1661 GE_default1661;
extern T1662 GE_default1662;
extern T1663 GE_default1663;
extern T1666 GE_default1666;
extern T1667 GE_default1667;
extern T1668 GE_default1668;
extern T1669 GE_default1669;
extern T1671 GE_default1671;
extern T1672 GE_default1672;
extern T1673 GE_default1673;
extern T1674 GE_default1674;
extern T1675 GE_default1675;
extern T1677 GE_default1677;
extern T1678 GE_default1678;
extern T1679 GE_default1679;
extern T1680 GE_default1680;
extern T1681 GE_default1681;
extern T1682 GE_default1682;
extern T1684 GE_default1684;
extern T1685 GE_default1685;
extern T1686 GE_default1686;
extern T1687 GE_default1687;
extern T1688 GE_default1688;
extern T1690 GE_default1690;
extern T1692 GE_default1692;
extern T1693 GE_default1693;
extern T1694 GE_default1694;
extern T1695 GE_default1695;
extern T1696 GE_default1696;
extern T1697 GE_default1697;
extern T1698 GE_default1698;
extern T1701 GE_default1701;
extern T1703 GE_default1703;
extern T1704 GE_default1704;
extern T1707 GE_default1707;
extern T1711 GE_default1711;
extern T1712 GE_default1712;
extern T1713 GE_default1713;
extern T1714 GE_default1714;
extern T1715 GE_default1715;
extern T1716 GE_default1716;
extern T1717 GE_default1717;
extern T1719 GE_default1719;
extern T1720 GE_default1720;
extern T1722 GE_default1722;
extern T1723 GE_default1723;
extern T1724 GE_default1724;
extern T1725 GE_default1725;
extern T1726 GE_default1726;
extern T1727 GE_default1727;
extern T1728 GE_default1728;
extern T1729 GE_default1729;
extern T1730 GE_default1730;
extern T1731 GE_default1731;
extern T1732 GE_default1732;
extern T1733 GE_default1733;
extern T1734 GE_default1734;
extern T1736 GE_default1736;
extern T1737 GE_default1737;
extern T1738 GE_default1738;
extern T1739 GE_default1739;
extern T1742 GE_default1742;
extern T1743 GE_default1743;
extern T1744 GE_default1744;
extern T1745 GE_default1745;
extern T1750 GE_default1750;
extern T1755 GE_default1755;
extern T1756 GE_default1756;
extern T1758 GE_default1758;
extern T1759 GE_default1759;
extern T1762 GE_default1762;
extern T1766 GE_default1766;
extern T1767 GE_default1767;
extern T1768 GE_default1768;
extern T1769 GE_default1769;
extern T1771 GE_default1771;
extern T1776 GE_default1776;
extern T1777 GE_default1777;
extern T1779 GE_default1779;
extern T1780 GE_default1780;
extern T1782 GE_default1782;
extern T1783 GE_default1783;
extern T1784 GE_default1784;
extern T1785 GE_default1785;
extern T1788 GE_default1788;
extern T1789 GE_default1789;
extern T1790 GE_default1790;
extern T1791 GE_default1791;
extern T1792 GE_default1792;
extern T1793 GE_default1793;
extern T1794 GE_default1794;
extern T1795 GE_default1795;
extern T1796 GE_default1796;
extern T1797 GE_default1797;
extern T1798 GE_default1798;
extern T1799 GE_default1799;
extern T1800 GE_default1800;
extern T1801 GE_default1801;
extern T1802 GE_default1802;
extern T1806 GE_default1806;
extern T1809 GE_default1809;
extern T1810 GE_default1810;
extern T1811 GE_default1811;
extern T1813 GE_default1813;
extern T1814 GE_default1814;
extern T1815 GE_default1815;
extern T1817 GE_default1817;
extern T1819 GE_default1819;
extern T1820 GE_default1820;
extern T1821 GE_default1821;
extern T1822 GE_default1822;
extern T1823 GE_default1823;
extern T1824 GE_default1824;
extern T1825 GE_default1825;
extern T1826 GE_default1826;
extern T1827 GE_default1827;
extern T1828 GE_default1828;
extern T1830 GE_default1830;
extern T1831 GE_default1831;
extern T1832 GE_default1832;
extern T1833 GE_default1833;
extern T1834 GE_default1834;
extern T1836 GE_default1836;
extern T1838 GE_default1838;
extern T1839 GE_default1839;
extern T1840 GE_default1840;
extern T1841 GE_default1841;
extern T1843 GE_default1843;
extern T1844 GE_default1844;
extern T1845 GE_default1845;
extern T1846 GE_default1846;
extern T1847 GE_default1847;
extern T1848 GE_default1848;
extern T1850 GE_default1850;
extern T1853 GE_default1853;
extern T1857 GE_default1857;
extern T1859 GE_default1859;
extern T1860 GE_default1860;
extern T1861 GE_default1861;
extern T1862 GE_default1862;
extern T1863 GE_default1863;
extern T1864 GE_default1864;
extern T1865 GE_default1865;
extern T1866 GE_default1866;
extern T1867 GE_default1867;
extern T1868 GE_default1868;
extern T1869 GE_default1869;
extern T1870 GE_default1870;
extern T1871 GE_default1871;
extern T1872 GE_default1872;
extern T1873 GE_default1873;
extern T1874 GE_default1874;
extern T1875 GE_default1875;
extern T1876 GE_default1876;
extern T1877 GE_default1877;
extern T1878 GE_default1878;
extern T1879 GE_default1879;
extern T1880 GE_default1880;
extern T1881 GE_default1881;
extern T1882 GE_default1882;
extern T1883 GE_default1883;
extern T1884 GE_default1884;
extern T1885 GE_default1885;
extern T1886 GE_default1886;
extern T1887 GE_default1887;
extern T1888 GE_default1888;
extern T1889 GE_default1889;
extern T1890 GE_default1890;
extern T1891 GE_default1891;
extern T1892 GE_default1892;
extern T1893 GE_default1893;
extern T1894 GE_default1894;
extern T1895 GE_default1895;
extern T1896 GE_default1896;
extern T1897 GE_default1897;
extern T1898 GE_default1898;
extern T1899 GE_default1899;
extern T1900 GE_default1900;
extern T1902 GE_default1902;
extern T1903 GE_default1903;
extern T1904 GE_default1904;
extern T1905 GE_default1905;
extern T1906 GE_default1906;
extern T1907 GE_default1907;
extern T1908 GE_default1908;
extern T1909 GE_default1909;
extern T1911 GE_default1911;
extern T1912 GE_default1912;
extern T1914 GE_default1914;
extern T1915 GE_default1915;
extern T1916 GE_default1916;
extern T1917 GE_default1917;
extern T1918 GE_default1918;
extern T1919 GE_default1919;
extern T1920 GE_default1920;
extern T1921 GE_default1921;
extern T1922 GE_default1922;
extern T1923 GE_default1923;
extern T1926 GE_default1926;
extern T1927 GE_default1927;
extern T1928 GE_default1928;
extern T1929 GE_default1929;
extern T1930 GE_default1930;
extern T1931 GE_default1931;
extern T1932 GE_default1932;
extern T1933 GE_default1933;
extern T1935 GE_default1935;
extern T1936 GE_default1936;
extern T1937 GE_default1937;
extern T1938 GE_default1938;
extern T1939 GE_default1939;
extern T1940 GE_default1940;
extern T1941 GE_default1941;
extern T1942 GE_default1942;
extern T1943 GE_default1943;
extern T1944 GE_default1944;
extern T1953 GE_default1953;
extern T1954 GE_default1954;
extern T1955 GE_default1955;
extern T1956 GE_default1956;
extern T1957 GE_default1957;
extern T1960 GE_default1960;
extern T1962 GE_default1962;
extern T1964 GE_default1964;
extern T1965 GE_default1965;
extern T1966 GE_default1966;
extern T1968 GE_default1968;
extern T1977 GE_default1977;
extern T1978 GE_default1978;
extern T1980 GE_default1980;
extern T1984 GE_default1984;
extern T1985 GE_default1985;
extern T1988 GE_default1988;
extern T1989 GE_default1989;
extern T1990 GE_default1990;
extern T1993 GE_default1993;
extern T1994 GE_default1994;
extern T1997 GE_default1997;
extern T1998 GE_default1998;
extern T1999 GE_default1999;
extern T2000 GE_default2000;
extern T2002 GE_default2002;
extern T2020 GE_default2020;
extern T2026 GE_default2026;
extern T2027 GE_default2027;
extern T2034 GE_default2034;
extern T2040 GE_default2040;
extern T2041 GE_default2041;
extern T2042 GE_default2042;
extern T2043 GE_default2043;
extern T2044 GE_default2044;
extern T2046 GE_default2046;
extern T2047 GE_default2047;
extern T2048 GE_default2048;
extern T2051 GE_default2051;
extern T2052 GE_default2052;
extern T2054 GE_default2054;
extern T2056 GE_default2056;
extern T2057 GE_default2057;
extern T2058 GE_default2058;
extern T2059 GE_default2059;
extern T2060 GE_default2060;
extern T2061 GE_default2061;
extern T2063 GE_default2063;
extern T2064 GE_default2064;
extern T2066 GE_default2066;
extern T2068 GE_default2068;
extern T2071 GE_default2071;
extern T2073 GE_default2073;
extern T2074 GE_default2074;
extern T2075 GE_default2075;
extern T2077 GE_default2077;
extern T2078 GE_default2078;
extern T2079 GE_default2079;
extern T2082 GE_default2082;
extern T2086 GE_default2086;
extern T2089 GE_default2089;
extern T2090 GE_default2090;
extern T2091 GE_default2091;
extern T2092 GE_default2092;
extern T2094 GE_default2094;
extern T2095 GE_default2095;
extern T2096 GE_default2096;
extern T2097 GE_default2097;
extern T2099 GE_default2099;
extern T2100 GE_default2100;
extern T2101 GE_default2101;
extern T2102 GE_default2102;
extern T2105 GE_default2105;
extern T2106 GE_default2106;
extern T2107 GE_default2107;
extern T2108 GE_default2108;
extern T2109 GE_default2109;
extern T2110 GE_default2110;
extern T2111 GE_default2111;
extern T2112 GE_default2112;
extern T2113 GE_default2113;
extern T2114 GE_default2114;
extern T2115 GE_default2115;
extern T2116 GE_default2116;
extern T2117 GE_default2117;
extern T2118 GE_default2118;
extern T2119 GE_default2119;
extern T2120 GE_default2120;
extern T2121 GE_default2121;
extern T2122 GE_default2122;
extern T2123 GE_default2123;
extern T2124 GE_default2124;
extern T2125 GE_default2125;
extern T2126 GE_default2126;
extern T2127 GE_default2127;
extern T2128 GE_default2128;
extern T2129 GE_default2129;
extern T2130 GE_default2130;
extern T2131 GE_default2131;
extern T2132 GE_default2132;
extern T2133 GE_default2133;
extern T2134 GE_default2134;
extern T2135 GE_default2135;
extern T2136 GE_default2136;
extern T2137 GE_default2137;
extern T2138 GE_default2138;
extern T2139 GE_default2139;
extern T2140 GE_default2140;
extern T2141 GE_default2141;
extern T2142 GE_default2142;
extern T2143 GE_default2143;
extern T2144 GE_default2144;
extern T2145 GE_default2145;
extern T2146 GE_default2146;
extern T2147 GE_default2147;
extern T2148 GE_default2148;
extern T2149 GE_default2149;
extern T2150 GE_default2150;
extern T2151 GE_default2151;
extern T2152 GE_default2152;
extern T2153 GE_default2153;
extern T2154 GE_default2154;
extern T2155 GE_default2155;
extern T2156 GE_default2156;
extern T2157 GE_default2157;
extern T2158 GE_default2158;
extern T2159 GE_default2159;
extern T2160 GE_default2160;
extern T2161 GE_default2161;
extern T2162 GE_default2162;
extern T2164 GE_default2164;
extern T2165 GE_default2165;
extern T2166 GE_default2166;
extern T2168 GE_default2168;
extern T2170 GE_default2170;
extern T2171 GE_default2171;
extern T2174 GE_default2174;
extern T2175 GE_default2175;
extern T2177 GE_default2177;
extern T2178 GE_default2178;
extern T2179 GE_default2179;
extern T2180 GE_default2180;
extern T2181 GE_default2181;
extern T2183 GE_default2183;
extern T2185 GE_default2185;
extern T2187 GE_default2187;
extern T2188 GE_default2188;
extern T2190 GE_default2190;
extern T2191 GE_default2191;
extern T2192 GE_default2192;
extern T2193 GE_default2193;
extern T2194 GE_default2194;
extern T2195 GE_default2195;
extern T2197 GE_default2197;
extern T2201 GE_default2201;
extern T2204 GE_default2204;
extern T2207 GE_default2207;
extern T2208 GE_default2208;
extern T2209 GE_default2209;
extern T2210 GE_default2210;
extern T2211 GE_default2211;
extern T2213 GE_default2213;

extern T0* GE_new_str8(T6 c);
extern T0* GE_new_str32(T6 c);
extern T0* GE_new_istr8(T6 c);
extern T0* GE_new_istr32(T6 c);
/* New instance of type [detachable] SPECIAL [CHARACTER_8] */
extern T0* GE_new15(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [CHARACTER_32] */
extern T0* GE_new16(T6 a1, T1 initialize);
/* New instance of type [detachable] STRING_8 */
extern T0* GE_new17(T1 initialize);
/* New instance of type [detachable] STRING_32 */
extern T0* GE_new18(T1 initialize);
/* New instance of type [detachable] IMMUTABLE_STRING_32 */
extern T0* GE_new20(T1 initialize);
/* New instance of type [detachable] ISE_EXCEPTION_MANAGER */
extern T0* GE_new21(T1 initialize);
/* New instance of type [detachable] GEC */
extern T0* GE_new25(T1 initialize);
/* New instance of type [detachable] CELL [detachable EXCEPTION] */
extern T0* GE_new26(T1 initialize);
/* New instance of type [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
extern T0* GE_new27(T1 initialize);
/* New instance of type [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
extern T0* GE_new28(T1 initialize);
/* New instance of type [detachable] CELL [[attached] NO_MORE_MEMORY] */
extern T0* GE_new29(T1 initialize);
/* New instance of type detachable C_STRING */
extern T0* GE_new30(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
extern T0* GE_new31(T1 initialize);
/* New instance of type [detachable] KL_ARGUMENTS */
extern T0* GE_new32(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] STRING_8] */
extern T0* GE_new33(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] STRING_8] */
extern T0* GE_new34(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_EXCEPTIONS */
extern T0* GE_new35(T1 initialize);
/* New instance of type [detachable] ET_ERROR_HANDLER */
extern T0* GE_new36(T1 initialize);
/* New instance of type [detachable] KL_TEXT_INPUT_FILE */
extern T0* GE_new37(T1 initialize);
/* New instance of type [detachable] ET_ISE_VARIABLES */
extern T0* GE_new38(T1 initialize);
/* New instance of type [detachable] AP_FLAG */
extern T0* GE_new39(T1 initialize);
/* New instance of type detachable ET_SYSTEM */
extern T0* GE_new40(T1 initialize);
/* New instance of type [detachable] AP_PARSER */
extern T0* GE_new42(T1 initialize);
/* New instance of type [detachable] AP_ALTERNATIVE_OPTIONS_LIST */
extern T0* GE_new43(T1 initialize);
/* New instance of type [detachable] AP_STRING_OPTION */
extern T0* GE_new44(T1 initialize);
/* New instance of type [detachable] UT_VERSION */
extern T0* GE_new46(T1 initialize);
/* New instance of type [detachable] AP_ENUMERATION_OPTION */
extern T0* GE_new47(T1 initialize);
/* New instance of type [detachable] AP_BOOLEAN_OPTION */
extern T0* GE_new48(T1 initialize);
/* New instance of type [detachable] AP_INTEGER_OPTION */
extern T0* GE_new49(T1 initialize);
/* New instance of type [detachable] ET_NULL_ERROR_HANDLER */
extern T0* GE_new53(T1 initialize);
/* New instance of type [detachable] ET_ECF_SYSTEM_PARSER */
extern T0* GE_new55(T1 initialize);
/* New instance of type [detachable] ET_ECF_ERROR_HANDLER */
extern T0* GE_new56(T1 initialize);
/* New instance of type detachable ET_ECF_SETTINGS */
extern T0* GE_new57(T1 initialize);
/* New instance of type [detachable] ET_ECF_SYSTEM */
extern T0* GE_new58(T1 initialize);
/* New instance of type [detachable] ET_ECF_TARGET */
extern T0* GE_new59(T1 initialize);
/* New instance of type detachable ET_ECF_CAPABILITIES */
extern T0* GE_new61(T1 initialize);
/* New instance of type detachable ET_ECF_VARIABLES */
extern T0* GE_new62(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_SYSTEM */
extern T0* GE_new63(T1 initialize);
/* New instance of type [detachable] ET_SYSTEM_PROCESSOR */
extern T0* GE_new64(T1 initialize);
/* New instance of type [detachable] ET_C_GENERATOR */
extern T0* GE_new67(T1 initialize);
/* New instance of type detachable DT_DATE_TIME */
extern T0* GE_new68(T1 initialize);
/* New instance of type detachable DS_HASH_SET [[attached] STRING_8] */
extern T0* GE_new70(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
extern T0* GE_new71(T1 initialize);
/* New instance of type [detachable] ET_CLASS_TYPE */
extern T0* GE_new72(T1 initialize);
/* New instance of type [detachable] ET_TOKEN_CONSTANTS */
extern T0* GE_new73(T1 initialize);
/* New instance of type [detachable] ET_CLASS */
extern T0* GE_new74(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new76(T1 initialize);
/* New instance of type [detachable] UT_CANNOT_READ_FILE_ERROR */
extern T0* GE_new77(T1 initialize);
/* New instance of type [detachable] UT_VERSION_NUMBER */
extern T0* GE_new79(T1 initialize);
/* New instance of type [detachable] UT_MESSAGE */
extern T0* GE_new80(T1 initialize);
/* New instance of type [detachable] RX_PCRE_REGULAR_EXPRESSION */
extern T0* GE_new81(T1 initialize);
/* New instance of type [detachable] KL_STRING_ROUTINES */
extern T0* GE_new82(T1 initialize);
/* New instance of type [detachable] KL_EXECUTION_ENVIRONMENT */
extern T0* GE_new86(T1 initialize);
/* New instance of type [detachable] AP_ERROR */
extern T0* GE_new87(T1 initialize);
/* New instance of type [detachable] KL_SHELL_COMMAND */
extern T0* GE_new88(T1 initialize);
/* New instance of type [detachable] GECC */
extern T0* GE_new89(T1 initialize);
/* New instance of type [detachable] KL_OPERATING_SYSTEM */
extern T0* GE_new90(T1 initialize);
/* New instance of type [detachable] VOID_TARGET */
extern T0* GE_new93(T1 initialize);
/* New instance of type [detachable] ROUTINE_FAILURE */
extern T0* GE_new95(T1 initialize);
/* New instance of type [detachable] OLD_VIOLATION */
extern T0* GE_new97(T1 initialize);
/* New instance of type [detachable] NO_MORE_MEMORY */
extern T0* GE_new99(T1 initialize);
/* New instance of type [detachable] INVARIANT_VIOLATION */
extern T0* GE_new100(T1 initialize);
/* New instance of type [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE */
extern T0* GE_new101(T1 initialize);
/* New instance of type [detachable] IO_FAILURE */
extern T0* GE_new102(T1 initialize);
/* New instance of type [detachable] OPERATING_SYSTEM_FAILURE */
extern T0* GE_new103(T1 initialize);
/* New instance of type [detachable] COM_FAILURE */
extern T0* GE_new104(T1 initialize);
/* New instance of type [detachable] EIFFEL_RUNTIME_PANIC */
extern T0* GE_new105(T1 initialize);
/* New instance of type [detachable] PRECONDITION_VIOLATION */
extern T0* GE_new107(T1 initialize);
/* New instance of type [detachable] POSTCONDITION_VIOLATION */
extern T0* GE_new108(T1 initialize);
/* New instance of type [detachable] FLOATING_POINT_FAILURE */
extern T0* GE_new109(T1 initialize);
/* New instance of type [detachable] CHECK_VIOLATION */
extern T0* GE_new110(T1 initialize);
/* New instance of type [detachable] BAD_INSPECT_VALUE */
extern T0* GE_new111(T1 initialize);
/* New instance of type [detachable] VARIANT_VIOLATION */
extern T0* GE_new112(T1 initialize);
/* New instance of type [detachable] LOOP_INVARIANT_VIOLATION */
extern T0* GE_new113(T1 initialize);
/* New instance of type [detachable] RESCUE_FAILURE */
extern T0* GE_new114(T1 initialize);
/* New instance of type [detachable] RESUMPTION_FAILURE */
extern T0* GE_new115(T1 initialize);
/* New instance of type [detachable] CREATE_ON_DEFERRED */
extern T0* GE_new116(T1 initialize);
/* New instance of type [detachable] EXTERNAL_FAILURE */
extern T0* GE_new117(T1 initialize);
/* New instance of type [detachable] VOID_ASSIGNED_TO_EXPANDED */
extern T0* GE_new118(T1 initialize);
/* New instance of type [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
extern T0* GE_new119(T1 initialize);
/* New instance of type [detachable] MISMATCH_FAILURE */
extern T0* GE_new120(T1 initialize);
/* New instance of type [detachable] DEVELOPER_EXCEPTION */
extern T0* GE_new121(T1 initialize);
/* New instance of type [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE */
extern T0* GE_new122(T1 initialize);
/* New instance of type [detachable] SERIALIZATION_FAILURE */
extern T0* GE_new123(T1 initialize);
/* New instance of type [detachable] EXECUTION_ENVIRONMENT */
extern T0* GE_new124(T1 initialize);
/* New instance of type [detachable] KL_WINDOWS_FILE_SYSTEM */
extern T0* GE_new125(T1 initialize);
/* New instance of type [detachable] KL_UNIX_FILE_SYSTEM */
extern T0* GE_new126(T1 initialize);
/* New instance of type [detachable] PRIMES */
extern T0* GE_new127(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] INTEGER_32] */
extern T0* GE_new128(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] BOOLEAN] */
extern T0* GE_new129(T6 a1, T1 initialize);
/* New instance of type [detachable] ARGUMENTS_32 */
extern T0* GE_new130(T1 initialize);
/* New instance of type [detachable] MUTEX */
extern T0* GE_new131(T1 initialize);
/* New instance of type [detachable] UT_ERROR_HANDLER */
extern T0* GE_new132(T1 initialize);
/* New instance of type [detachable] KL_STANDARD_FILES */
extern T0* GE_new133(T1 initialize);
/* New instance of type [detachable] KL_STDERR_FILE */
extern T0* GE_new134(T1 initialize);
/* New instance of type [detachable] KL_STDOUT_FILE */
extern T0* GE_new135(T1 initialize);
/* New instance of type detachable KL_LINKABLE [[attached] CHARACTER_8] */
extern T0* GE_new137(T1 initialize);
/* New instance of type [detachable] MANAGED_POINTER */
extern T0* GE_new140(T1 initialize);
/* New instance of type [detachable] FILE_INFO */
extern T0* GE_new141(T1 initialize);
/* New instance of type [detachable] ST_WORD_WRAPPER */
extern T0* GE_new142(T1 initialize);
/* New instance of type [detachable] AP_DISPLAY_HELP_FLAG */
extern T0* GE_new144(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] AP_OPTION] */
extern T0* GE_new145(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new146(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] STRING_8] */
extern T0* GE_new147(T1 initialize);
/* New instance of type [detachable] AP_ERROR_HANDLER */
extern T0* GE_new148(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] AP_OPTION] */
extern T0* GE_new149(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [detachable STRING_8] */
extern T0* GE_new150(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
extern T0* GE_new151(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] STRING_8] */
extern T0* GE_new152(T1 initialize);
/* New instance of type [detachable] KL_STRING_EQUALITY_TESTER */
extern T0* GE_new153(T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] STRING_8] */
extern T0* GE_new154(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] BOOLEAN] */
extern T0* GE_new157(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] INTEGER_32] */
extern T0* GE_new160(T1 initialize);
/* New instance of type [detachable] KL_NULL_TEXT_OUTPUT_STREAM */
extern T0* GE_new161(T1 initialize);
/* New instance of type [detachable] ET_ECF_AST_FACTORY */
extern T0* GE_new162(T1 initialize);
/* New instance of type [detachable] DS_CELL [detachable ET_ECF_SYSTEM] */
extern T0* GE_new163(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_TARGET] */
extern T0* GE_new164(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]] */
extern T0* GE_new165(T1 initialize);
/* New instance of type detachable TUPLE */
extern T0* GE_new166(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER] */
extern T0* GE_new167(T1 initialize);
/* New instance of type [detachable] XM_ELEMENT */
extern T0* GE_new168(T1 initialize);
/* New instance of type detachable XM_POSITION_TABLE */
extern T0* GE_new169(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8] */
extern T0* GE_new170(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8]] */
extern T0* GE_new171(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [detachable] STRING_8, [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]], [attached] DS_CELL [detachable ET_ECF_SYSTEM]] */
extern T0* GE_new172(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
extern T0* GE_new174(T1 initialize);
/* New instance of type [detachable] KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
extern T0* GE_new175(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
extern T0* GE_new176(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_PARSER */
extern T0* GE_new178(T1 initialize);
/* New instance of type [detachable] XM_TREE_CALLBACKS_PIPE */
extern T0* GE_new179(T1 initialize);
/* New instance of type [detachable] XM_CALLBACKS_TO_TREE_FILTER */
extern T0* GE_new182(T1 initialize);
/* New instance of type [detachable] ET_ECF_STATE */
extern T0* GE_new183(T1 initialize);
/* New instance of type [detachable] ET_IDENTIFIER */
extern T0* GE_new184(T1 initialize);
/* New instance of type [detachable] ET_ECF_SYSTEM_CONFIG */
extern T0* GE_new185(T1 initialize);
/* New instance of type [detachable] ET_ECF_LIBRARY */
extern T0* GE_new187(T1 initialize);
/* New instance of type [detachable] ET_ADAPTED_LIBRARIES */
extern T0* GE_new188(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new190(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_LIBRARY]] */
extern T0* GE_new191(T1 initialize);
/* New instance of type [detachable] XM_DOCUMENT */
extern T0* GE_new192(T1 initialize);
/* New instance of type [detachable] ET_COMPRESSED_POSITION */
extern T0* GE_new193(T1 initialize);
/* New instance of type [detachable] XM_STOP_ON_ERROR_FILTER */
extern T0* GE_new195(T1 initialize);
/* New instance of type detachable XM_ATTRIBUTE */
extern T0* GE_new196(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
extern T0* GE_new198(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
extern T0* GE_new199(T1 initialize);
/* New instance of type detachable ET_ECF_TARGET_PARENT */
extern T0* GE_new200(T1 initialize);
/* New instance of type [detachable] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new201(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [attached] ET_ECF_INTERNAL_UNIVERSE, [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
extern T0* GE_new202(T1 initialize);
/* New instance of type detachable ET_ECF_TARGETS */
extern T0* GE_new203(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new204(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new205(T1 initialize);
/* New instance of type [detachable] DS_CELL [detachable ET_ECF_LIBRARY] */
extern T0* GE_new206(T1 initialize);
/* New instance of type [detachable] ET_ECF_ADAPTED_LIBRARY */
extern T0* GE_new207(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_SYSTEM_PARSER, [attached] ET_ECF_ADAPTED_LIBRARY, [attached] DS_CELL [detachable ET_ECF_LIBRARY]] */
extern T0* GE_new208(T1 initialize);
/* New instance of type [detachable] ET_ADAPTED_DOTNET_ASSEMBLIES */
extern T0* GE_new210(T1 initialize);
/* New instance of type [detachable] ET_ECF_DOTNET_ASSEMBLY */
extern T0* GE_new211(T1 initialize);
/* New instance of type [detachable] ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
extern T0* GE_new213(T1 initialize);
/* New instance of type [detachable] ET_ECF_ERROR */
extern T0* GE_new214(T1 initialize);
/* New instance of type [detachable] KL_AGENT_HASH_FUNCTION [[attached] STRING_8] */
extern T0* GE_new215(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8] */
extern T0* GE_new216(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] STRING_8] */
extern T0* GE_new217(T1 initialize);
/* New instance of type [detachable] FUNCTION [[attached] TUPLE [[attached] STRING_8], [attached] INTEGER_32] */
extern T0* GE_new218(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] KL_STRING_ROUTINES] */
extern T0* GE_new219(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_IDENTIFIER, [attached] STRING_8] */
extern T0* GE_new221(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_PRIMARY_TYPE */
extern T0* GE_new223(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_FEATURE_LIST */
extern T0* GE_new224(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
extern T0* GE_new225(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] STRING_8] */
extern T0* GE_new226(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_CLASS] */
extern T0* GE_new227(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]] */
extern T0* GE_new228(T1 initialize);
/* New instance of type [detachable] TUPLE [INTEGER_32] */
extern T0* GE_new229(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
extern T0* GE_new230(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_DYNAMIC_SYSTEM] */
extern T0* GE_new231(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_FEATURE */
extern T0* GE_new236(T1 initialize);
/* New instance of type [detachable] ET_ACTUAL_PARAMETER_LIST */
extern T0* GE_new240(T1 initialize);
/* New instance of type [detachable] ET_NESTED_TYPE_CONTEXT */
extern T0* GE_new241(T1 initialize);
/* New instance of type [detachable] ET_FORMAL_PARAMETER_TYPE */
extern T0* GE_new242(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] ET_TYPE] */
extern T0* GE_new244(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_TYPE] */
extern T0* GE_new245(T6 a1, T1 initialize);
/* New instance of type [detachable] ET_EXTERNAL_FUNCTION */
extern T0* GE_new253(T1 initialize);
/* New instance of type [detachable] ET_TUPLE_TYPE */
extern T0* GE_new254(T1 initialize);
/* New instance of type detachable ET_FORMAL_ARGUMENT_LIST */
extern T0* GE_new255(T1 initialize);
/* New instance of type [detachable] ET_FORMAL_ARGUMENT */
extern T0* GE_new256(T1 initialize);
/* New instance of type [detachable] ET_QUERY_LIST */
extern T0* GE_new257(T1 initialize);
/* New instance of type [detachable] ET_PROCEDURE_LIST */
extern T0* GE_new258(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_TYPE_SET_LIST */
extern T0* GE_new259(T1 initialize);
/* New instance of type [detachable] KL_STRING_OUTPUT_STREAM */
extern T0* GE_new260(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_IDENTIFIER] */
extern T0* GE_new261(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new262(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32] */
extern T0* GE_new263(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new264(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new265(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new266(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_RESULT] */
extern T0* GE_new267(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST */
extern T0* GE_new268(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_STANDALONE_TYPE_SET */
extern T0* GE_new269(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new270(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [[attached] ET_EXPRESSION] */
extern T0* GE_new271(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_EXPRESSION] */
extern T0* GE_new272(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] INTEGER_32] */
extern T0* GE_new273(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
extern T0* GE_new274(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new275(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new276(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_OBJECT_TEST] */
extern T0* GE_new277(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new278(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new279(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_AGENT] */
extern T0* GE_new280(T1 initialize);
/* New instance of type [detachable] ET_CURRENT */
extern T0* GE_new282(T1 initialize);
/* New instance of type [detachable] ET_ACTUAL_ARGUMENT_LIST */
extern T0* GE_new283(T1 initialize);
/* New instance of type [detachable] ET_QUALIFIED_CALL_INSTRUCTION */
extern T0* GE_new284(T1 initialize);
/* New instance of type [detachable] ET_QUALIFIED_CALL_EXPRESSION */
extern T0* GE_new287(T1 initialize);
/* New instance of type [detachable] ET_MANIFEST_TUPLE */
extern T0* GE_new288(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new289(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FEATURE] */
extern T0* GE_new290(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] INTEGER_32] */
extern T0* GE_new291(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
extern T0* GE_new292(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] ET_INLINE_CONSTANT] */
extern T0* GE_new293(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new294(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new295(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] BOOLEAN, [attached] STRING_8] */
extern T0* GE_new296(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_IDENTIFIER] */
extern T0* GE_new297(T1 initialize);
/* New instance of type [detachable] ET_IDENTIFIER_TESTER */
extern T0* GE_new298(T1 initialize);
/* New instance of type [detachable] KL_TEXT_OUTPUT_FILE */
extern T0* GE_new300(T1 initialize);
/* New instance of type [detachable] ET_IMPLICIT_TYPE_MARK */
extern T0* GE_new301(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_TUPLE_TYPE */
extern T0* GE_new302(T1 initialize);
/* New instance of type [detachable] DS_STRING_HASH_TABLE */
extern T0* GE_new303(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] STRING_8, [attached] STRING_8] */
extern T0* GE_new304(T1 initialize);
/* New instance of type [detachable] UT_CANNOT_WRITE_TO_FILE_ERROR */
extern T0* GE_new305(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_PRECURSOR */
extern T0* GE_new306(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_PRECURSOR_LIST */
extern T0* GE_new307(T1 initialize);
/* New instance of type [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new308(T1 initialize);
/* New instance of type [detachable] ET_RESULT */
extern T0* GE_new309(T1 initialize);
/* New instance of type [detachable] ET_OBJECT_TEST */
extern T0* GE_new311(T1 initialize);
/* New instance of type [detachable] ET_OBJECT_EQUALITY_EXPRESSION */
extern T0* GE_new312(T1 initialize);
/* New instance of type [detachable] ET_EQUALITY_EXPRESSION */
extern T0* GE_new313(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL */
extern T0* GE_new314(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
extern T0* GE_new318(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_SPECIAL_TYPE */
extern T0* GE_new319(T1 initialize);
/* New instance of type [detachable] DS_QUICK_SORTER [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new322(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_SECONDARY_TYPE */
extern T0* GE_new323(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID */
extern T0* GE_new324(T1 initialize);
/* New instance of type [detachable] ET_CREATE_EXPRESSION */
extern T0* GE_new328(T1 initialize);
/* New instance of type [detachable] ET_QUALIFIED_CALL */
extern T0* GE_new329(T1 initialize);
/* New instance of type [detachable] KL_DIRECTORY */
extern T0* GE_new333(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new334(T1 initialize);
/* New instance of type [detachable] ET_SYMBOL */
extern T0* GE_new341(T1 initialize);
/* New instance of type [detachable] DS_QUICK_SORTER [[attached] INTEGER_32] */
extern T0* GE_new344(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE] */
extern T0* GE_new346(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_IDENTIFIER, [attached] ET_CURRENT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new347(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_IDENTIFIER, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new348(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new349(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_RESULT, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new350(T1 initialize);
/* New instance of type [detachable] UT_INTEGER_FORMATTER */
extern T0* GE_new351(T1 initialize);
/* New instance of type detachable ET_COMPOUND */
extern T0* GE_new355(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new357(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new358(T1 initialize);
/* New instance of type [detachable] ET_EXTENDED_ATTRIBUTE */
extern T0* GE_new359(T1 initialize);
/* New instance of type [detachable] ET_ATTRIBUTE */
extern T0* GE_new360(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new361(T1 initialize);
/* New instance of type detachable ET_ITERATION_COMPONENT_LIST */
extern T0* GE_new365(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_EXPRESSION, [attached] ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new367(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, INTEGER_32, INTEGER_32] */
extern T0* GE_new368(T1 initialize);
/* New instance of type [detachable] ET_CONSTANT_ATTRIBUTE */
extern T0* GE_new369(T1 initialize);
/* New instance of type [detachable] ET_UNIQUE_ATTRIBUTE */
extern T0* GE_new371(T1 initialize);
/* New instance of type [detachable] ET_REGULAR_INTEGER_CONSTANT */
extern T0* GE_new372(T1 initialize);
/* New instance of type [detachable] ET_SYSTEM_MULTIPROCESSOR */
extern T0* GE_new375(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8] */
extern T0* GE_new376(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
extern T0* GE_new377(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
extern T0* GE_new379(T1 initialize);
/* New instance of type [detachable] ET_FEATURE_CHECKER */
extern T0* GE_new380(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new381(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new382(T1 initialize);
/* New instance of type [detachable] ET_TYPE_CHECKER */
extern T0* GE_new384(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_INLINE_AGENT] */
extern T0* GE_new386(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new387(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new388(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new389(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new390(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new391(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new392(T1 initialize);
/* New instance of type [detachable] ET_OBJECT_TEST_SCOPE */
extern T0* GE_new393(T1 initialize);
/* New instance of type [detachable] ET_OBJECT_TEST_SCOPE_BUILDER */
extern T0* GE_new394(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ITERATION_COMPONENT] */
extern T0* GE_new395(T1 initialize);
/* New instance of type [detachable] ET_ITERATION_ITEM_SCOPE */
extern T0* GE_new396(T1 initialize);
/* New instance of type [detachable] ET_ATTACHMENT_SCOPE */
extern T0* GE_new397(T1 initialize);
/* New instance of type [detachable] ET_ATTACHMENT_SCOPE_BUILDER */
extern T0* GE_new398(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new399(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
extern T0* GE_new400(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_INDEXING_TERM] */
extern T0* GE_new401(T1 initialize);
/* New instance of type [detachable] ET_CLIENT_LIST */
extern T0* GE_new402(T1 initialize);
/* New instance of type [detachable] ET_CLIENT */
extern T0* GE_new403(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
extern T0* GE_new404(T1 initialize);
/* New instance of type [detachable] ET_ADAPTED_BASE_CLASS_CHECKER */
extern T0* GE_new405(T1 initialize);
/* New instance of type detachable ET_PRECONDITIONS */
extern T0* GE_new406(T1 initialize);
/* New instance of type detachable ET_FEATURE_LIST */
extern T0* GE_new408(T1 initialize);
/* New instance of type [detachable] ET_UNKNOWN_GROUP */
extern T0* GE_new410(T1 initialize);
/* New instance of type [detachable] ET_BASE_TYPE_LIST */
extern T0* GE_new412(T1 initialize);
/* New instance of type [detachable] ET_KEYWORD */
extern T0* GE_new413(T1 initialize);
/* New instance of type [detachable] ET_CLASS_CODES */
extern T0* GE_new414(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new415(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new416(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new417(T1 initialize);
/* New instance of type [detachable] RX_BYTE_CODE */
extern T0* GE_new422(T1 initialize);
/* New instance of type [detachable] RX_CHARACTER_SET */
extern T0* GE_new423(T1 initialize);
/* New instance of type [detachable] RX_CASE_MAPPING */
extern T0* GE_new424(T1 initialize);
/* New instance of type [detachable] UC_UNICODE_ROUTINES */
extern T0* GE_new426(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] RX_CHARACTER_SET] */
extern T0* GE_new427(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] RX_CHARACTER_SET] */
extern T0* GE_new428(T6 a1, T1 initialize);
/* New instance of type [detachable] UC_STRING */
extern T0* GE_new429(T1 initialize);
/* New instance of type [detachable] NATIVE_STRING */
extern T0* GE_new430(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] GECC, [attached] DS_ARRAYED_LIST [[attached] STRING_8], INTEGER_32, [attached] MUTEX] */
extern T0* GE_new431(T1 initialize);
/* New instance of type [detachable] WORKER_THREAD */
extern T0* GE_new432(T1 initialize);
/* New instance of type [detachable] UC_UTF8_ROUTINES */
extern T0* GE_new434(T1 initialize);
/* New instance of type [detachable] PATH */
extern T0* GE_new435(T1 initialize);
/* New instance of type [detachable] STRING_TO_INTEGER_CONVERTOR */
extern T0* GE_new436(T1 initialize);
/* New instance of type [detachable] ET_ECF_OPTIONS */
extern T0* GE_new444(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new445(T1 initialize);
/* New instance of type [detachable] XM_NAMESPACE */
extern T0* GE_new446(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XM_ELEMENT_NODE] */
extern T0* GE_new447(T1 initialize);
/* New instance of type [detachable] ET_LIKE_CURRENT */
extern T0* GE_new449(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_PROCEDURE_TYPE */
extern T0* GE_new450(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_FUNCTION_TYPE */
extern T0* GE_new451(T1 initialize);
/* New instance of type [detachable] DT_SHARED_SYSTEM_CLOCK */
extern T0* GE_new452(T1 initialize);
/* New instance of type [detachable] DT_SYSTEM_CLOCK */
extern T0* GE_new453(T1 initialize);
/* New instance of type [detachable] ET_EIFFEL_PREPARSER */
extern T0* GE_new454(T1 initialize);
/* New instance of type [detachable] ET_MASTER_CLASS_CHECKER */
extern T0* GE_new455(T1 initialize);
/* New instance of type [detachable] ET_EIFFEL_PARSER */
extern T0* GE_new456(T1 initialize);
/* New instance of type [detachable] ET_PROVIDER_CHECKER */
extern T0* GE_new457(T1 initialize);
/* New instance of type [detachable] ET_ANCESTOR_BUILDER */
extern T0* GE_new458(T1 initialize);
/* New instance of type [detachable] ET_FEATURE_FLATTENER */
extern T0* GE_new459(T1 initialize);
/* New instance of type [detachable] ET_INTERFACE_CHECKER */
extern T0* GE_new460(T1 initialize);
/* New instance of type [detachable] ET_IMPLEMENTATION_CHECKER */
extern T0* GE_new461(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_CLASS] */
extern T0* GE_new462(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
extern T0* GE_new463(T1 initialize);
/* New instance of type [detachable] ET_AST_NULL_PROCESSOR */
extern T0* GE_new464(T1 initialize);
/* New instance of type [detachable] ET_AST_FACTORY */
extern T0* GE_new465(T1 initialize);
/* New instance of type [detachable] ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
extern T0* GE_new467(T1 initialize);
/* New instance of type [detachable] DT_DATE_TIME_DURATION */
extern T0* GE_new468(T1 initialize);
/* New instance of type [detachable] ET_DO_PROCEDURE */
extern T0* GE_new469(T1 initialize);
/* New instance of type [detachable] UT_CONFIG_PARSER */
extern T0* GE_new471(T1 initialize);
/* New instance of type [detachable] KL_COMPARABLE_COMPARATOR [[attached] INTEGER_32] */
extern T0* GE_new472(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_PUSH_TYPE_SET */
extern T0* GE_new475(T1 initialize);
/* New instance of type [detachable] ET_DEFERRED_PROCEDURE */
extern T0* GE_new476(T1 initialize);
/* New instance of type [detachable] KL_PLATFORM */
extern T0* GE_new477(T1 initialize);
/* New instance of type [detachable] CELL [[attached] INTEGER_32] */
extern T0* GE_new479(T1 initialize);
/* New instance of type [detachable] HASH_TABLE [[attached] NATIVE_STRING, [attached] IMMUTABLE_STRING_32] */
extern T0* GE_new480(T1 initialize);
/* New instance of type [detachable] KL_ANY_ROUTINES */
extern T0* GE_new481(T1 initialize);
/* New instance of type [detachable] KL_PATHNAME */
extern T0* GE_new483(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] IMMUTABLE_STRING_32] */
extern T0* GE_new484(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32] */
extern T0* GE_new485(T6 a1, T1 initialize);
/* New instance of type [detachable] EXCEPTIONS */
extern T0* GE_new486(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATURAL_8] */
extern T0* GE_new487(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] AP_OPTION] */
extern T0* GE_new489(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] AP_OPTION] */
extern T0* GE_new491(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] AP_OPTION] */
extern T0* GE_new492(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new493(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new494(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new495(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] STRING_8] */
extern T0* GE_new496(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable STRING_8] */
extern T0* GE_new497(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8] */
extern T0* GE_new498(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] STRING_8] */
extern T0* GE_new499(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] STRING_8] */
extern T0* GE_new500(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] BOOLEAN] */
extern T0* GE_new501(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] BOOLEAN] */
extern T0* GE_new502(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] INTEGER_32] */
extern T0* GE_new503(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] INTEGER_32] */
extern T0* GE_new504(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
extern T0* GE_new506(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_LIBRARY] */
extern T0* GE_new507(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_LIBRARY] */
extern T0* GE_new510(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_DOTNET_ASSEMBLY] */
extern T0* GE_new511(T6 a1, T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
extern T0* GE_new514(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_DOTNET_ASSEMBLY] */
extern T0* GE_new515(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_SCANNER */
extern T0* GE_new516(T1 initialize);
/* New instance of type [detachable] XM_DEFAULT_POSITION */
extern T0* GE_new517(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST [[attached] XM_POSITION] */
extern T0* GE_new519(T1 initialize);
/* New instance of type [detachable] DS_LINKED_STACK [[attached] XM_EIFFEL_SCANNER] */
extern T0* GE_new520(T1 initialize);
/* New instance of type [detachable] XM_CALLBACKS_NULL */
extern T0* GE_new521(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
extern T0* GE_new522(T1 initialize);
/* New instance of type [detachable] XM_NULL_EXTERNAL_RESOLVER */
extern T0* GE_new524(T1 initialize);
/* New instance of type [detachable] XM_DTD_CALLBACKS_NULL */
extern T0* GE_new526(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ANY] */
extern T0* GE_new527(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ANY] */
extern T0* GE_new528(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new529(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_PARSER_NAME */
extern T0* GE_new530(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new531(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
extern T0* GE_new532(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new533(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
extern T0* GE_new534(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_EXTERNAL_ID] */
extern T0* GE_new535(T1 initialize);
/* New instance of type [detachable] XM_DTD_EXTERNAL_ID */
extern T0* GE_new536(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_DTD_EXTERNAL_ID] */
extern T0* GE_new537(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new538(T1 initialize);
/* New instance of type [detachable] XM_DTD_ELEMENT_CONTENT */
extern T0* GE_new539(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new540(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
extern T0* GE_new541(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new542(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
extern T0* GE_new543(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new544(T1 initialize);
/* New instance of type [detachable] XM_DTD_ATTRIBUTE_CONTENT */
extern T0* GE_new545(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new546(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
extern T0* GE_new547(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST [[attached] STRING_8] */
extern T0* GE_new548(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
extern T0* GE_new549(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] BOOLEAN] */
extern T0* GE_new550(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_DECLARATION] */
extern T0* GE_new551(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_DECLARATION */
extern T0* GE_new552(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_EIFFEL_DECLARATION] */
extern T0* GE_new553(T6 a1, T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_ENTITY_DEF */
extern T0* GE_new556(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_SCANNER_DTD */
extern T0* GE_new557(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_PE_ENTITY_DEF */
extern T0* GE_new559(T1 initialize);
/* New instance of type [detachable] XM_NAMESPACE_RESOLVER */
extern T0* GE_new560(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] XM_CALLBACKS_FILTER] */
extern T0* GE_new561(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_CALLBACKS_FILTER] */
extern T0* GE_new562(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] XM_NAMESPACE] */
extern T0* GE_new563(T1 initialize);
/* New instance of type [detachable] ET_CLUSTERS */
extern T0* GE_new564(T1 initialize);
/* New instance of type [detachable] ET_CLASS_NAME_TESTER */
extern T0* GE_new565(T1 initialize);
/* New instance of type [detachable] ET_PARENT */
extern T0* GE_new567(T1 initialize);
/* New instance of type [detachable] ET_PARENT_LIST */
extern T0* GE_new568(T1 initialize);
/* New instance of type [detachable] ET_BUILTIN_CONVERT_FEATURE */
extern T0* GE_new569(T1 initialize);
/* New instance of type [detachable] ET_MASTER_CLASS */
extern T0* GE_new570(T1 initialize);
/* New instance of type [detachable] ET_ATTACHMENT_MARK_SEPARATE_KEYWORD */
extern T0* GE_new571(T1 initialize);
/* New instance of type detachable ET_RENAME_LIST */
extern T0* GE_new572(T1 initialize);
/* New instance of type detachable ET_EXPORT_LIST */
extern T0* GE_new573(T1 initialize);
/* New instance of type detachable ET_KEYWORD_FEATURE_NAME_LIST */
extern T0* GE_new574(T1 initialize);
/* New instance of type [detachable] ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
extern T0* GE_new577(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new578(T1 initialize);
/* New instance of type [detachable] XM_LINKED_LIST [[attached] XM_DOCUMENT_NODE] */
extern T0* GE_new579(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new582(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new584(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
extern T0* GE_new585(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_TARGET, [attached] STRING_8] */
extern T0* GE_new586(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_TARGET] */
extern T0* GE_new587(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET] */
extern T0* GE_new589(T1 initialize);
/* New instance of type detachable DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new591(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
extern T0* GE_new592(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable RX_REGULAR_EXPRESSION] */
extern T0* GE_new593(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new594(T1 initialize);
/* New instance of type [detachable] ET_DOTNET_ASSEMBLIES */
extern T0* GE_new595(T1 initialize);
/* New instance of type [detachable] ET_DOTNET_ASSEMBLY */
extern T0* GE_new596(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_IDENTIFIER] */
extern T0* GE_new599(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_IDENTIFIER] */
extern T0* GE_new600(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_IDENTIFIER, [attached] STRING_8] */
extern T0* GE_new601(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE [[attached] ET_DYNAMIC_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new602(T1 initialize);
/* New instance of type detachable ET_FORMAL_PARAMETER_LIST */
extern T0* GE_new603(T1 initialize);
/* New instance of type detachable ET_FEATURE_IDS */
extern T0* GE_new604(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new605(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new606(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
extern T0* GE_new609(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_MASTER_CLASS] */
extern T0* GE_new610(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_MASTER_CLASS] */
extern T0* GE_new611(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLASS_NAME] */
extern T0* GE_new612(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CLASS_NAME] */
extern T0* GE_new613(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] STRING_8] */
extern T0* GE_new615(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL, [attached] ET_CALL_NAME] */
extern T0* GE_new616(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, [attached] ET_CALL_NAME] */
extern T0* GE_new617(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new621(T6 a1, T1 initialize);
/* New instance of type [detachable] ET_BRACKET_SYMBOL */
extern T0* GE_new622(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new624(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE] */
extern T0* GE_new625(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_TYPE_SET] */
extern T0* GE_new626(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE_SET] */
extern T0* GE_new627(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_IDENTIFIER] */
extern T0* GE_new628(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new629(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new630(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new631(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
extern T0* GE_new632(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new634(T1 initialize);
/* New instance of type [detachable] ET_UNQUALIFIED_CALL_EXPRESSION */
extern T0* GE_new635(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new636(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new637(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new638(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new639(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new640(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new641(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new642(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new643(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RESULT] */
extern T0* GE_new644(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_RESULT] */
extern T0* GE_new645(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_RESULT] */
extern T0* GE_new646(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new647(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new648(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new649(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_EXPRESSION] */
extern T0* GE_new650(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION] */
extern T0* GE_new651(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_EXPRESSION] */
extern T0* GE_new652(T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] INTEGER_32] */
extern T0* GE_new653(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] INTEGER_32] */
extern T0* GE_new654(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new655(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_OBJECT_TEST] */
extern T0* GE_new657(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST] */
extern T0* GE_new658(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_TEST] */
extern T0* GE_new659(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new660(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new661(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new662(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new663(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new664(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new665(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_AGENT] */
extern T0* GE_new666(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT] */
extern T0* GE_new667(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_AGENT] */
extern T0* GE_new668(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_EXPRESSION_ITEM] */
extern T0* GE_new669(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION_ITEM] */
extern T0* GE_new671(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new672(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new673(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new674(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FEATURE] */
extern T0* GE_new676(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FEATURE] */
extern T0* GE_new677(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE] */
extern T0* GE_new678(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
extern T0* GE_new679(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CONSTANT] */
extern T0* GE_new680(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONSTANT] */
extern T0* GE_new682(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_PRIMARY_TYPE, [attached] ET_INLINE_CONSTANT] */
extern T0* GE_new683(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INLINE_CONSTANT] */
extern T0* GE_new684(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_CONSTANT] */
extern T0* GE_new686(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_DYNAMIC_FEATURE] */
extern T0* GE_new687(T6 a1, T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new689(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
extern T0* GE_new690(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new691(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] STRING_8] */
extern T0* GE_new692(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_IDENTIFIER] */
extern T0* GE_new695(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new698(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [detachable DS_LINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new699(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] BOOLEAN] */
extern T0* GE_new700(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new703(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_SYSTEM_PROCESSOR, [attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
extern T0* GE_new704(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new707(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new708(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new709(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS] */
extern T0* GE_new710(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_AGENT] */
extern T0* GE_new711(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INLINE_AGENT] */
extern T0* GE_new712(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_INLINE_AGENT] */
extern T0* GE_new713(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new714(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_PROCEDURE] */
extern T0* GE_new715(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new716(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new717(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new718(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_QUERY] */
extern T0* GE_new719(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new720(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new721(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new722(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_FEATURE] */
extern T0* GE_new723(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new724(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new725(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new726(T1 initialize);
/* New instance of type [detachable] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new727(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new728(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] TUPLE [detachable ET_QUERY, [attached] ET_CLASS, [attached] ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new729(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new730(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new731(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new732(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new733(T1 initialize);
/* New instance of type [detachable] ET_NAMED_OBJECT_TEST */
extern T0* GE_new736(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new737(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new738(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new739(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ITERATION_COMPONENT] */
extern T0* GE_new740(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ITERATION_COMPONENT] */
extern T0* GE_new741(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ITERATION_COMPONENT] */
extern T0* GE_new743(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ITERATION_COMPONENT] */
extern T0* GE_new744(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] INTEGER_32] */
extern T0* GE_new745(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new747(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new748(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new749(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
extern T0* GE_new750(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ASSERTIONS] */
extern T0* GE_new751(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTIONS] */
extern T0* GE_new753(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM] */
extern T0* GE_new754(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INDEXING_TERM] */
extern T0* GE_new756(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_INDEXING_TERM] */
extern T0* GE_new757(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CLIENT_ITEM] */
extern T0* GE_new758(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_ITEM] */
extern T0* GE_new759(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
extern T0* GE_new760(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
extern T0* GE_new761(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_CLASS]] */
extern T0* GE_new762(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
extern T0* GE_new763(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATURAL_32] */
extern T0* GE_new764(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] RX_CHARACTER_SET] */
extern T0* GE_new765(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_32] */
extern T0* GE_new766(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATURAL_64] */
extern T0* GE_new767(T6 a1, T1 initialize);
/* New instance of type detachable DS_HASH_TABLE [[attached] NATURAL_64, [attached] NATURAL_32] */
extern T0* GE_new768(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] NATURAL_64, [attached] NATURAL_32] */
extern T0* GE_new769(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]] */
extern T0* GE_new770(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]]] */
extern T0* GE_new771(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_INTEGER_ROUTINES */
extern T0* GE_new772(T1 initialize);
/* New instance of type [detachable] UC_UTF8_STRING */
extern T0* GE_new775(T1 initialize);
/* New instance of type [detachable] THREAD_ATTRIBUTES */
extern T0* GE_new776(T1 initialize);
/* New instance of type [detachable] CELL [[attached] BOOLEAN] */
extern T0* GE_new777(T1 initialize);
/* New instance of type [detachable] CHARACTER_PROPERTY */
extern T0* GE_new780(T1 initialize);
/* New instance of type [detachable] STRING_8_SEARCHER */
extern T0* GE_new781(T1 initialize);
/* New instance of type detachable ARRAYED_LIST [[attached] INTEGER_32] */
extern T0* GE_new782(T1 initialize);
/* New instance of type [detachable] ET_SYSTEM_ERROR */
extern T0* GE_new783(T1 initialize);
/* New instance of type [detachable] ET_INTERNAL_ERROR */
extern T0* GE_new784(T1 initialize);
/* New instance of type [detachable] ET_VALIDITY_ERROR */
extern T0* GE_new785(T1 initialize);
/* New instance of type detachable ET_ECF_NOTE_ELEMENT */
extern T0* GE_new786(T1 initialize);
/* New instance of type detachable ET_ECF_CLUSTERS */
extern T0* GE_new787(T1 initialize);
/* New instance of type detachable ET_ECF_ADAPTED_LIBRARIES */
extern T0* GE_new788(T1 initialize);
/* New instance of type detachable ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
extern T0* GE_new789(T1 initialize);
/* New instance of type detachable ET_ECF_FILE_RULES */
extern T0* GE_new790(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_CFLAGS */
extern T0* GE_new791(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_INCLUDES */
extern T0* GE_new792(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_LIBRARIES */
extern T0* GE_new793(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_LINKER_FLAGS */
extern T0* GE_new794(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_MAKES */
extern T0* GE_new795(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_OBJECTS */
extern T0* GE_new796(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_RESOURCES */
extern T0* GE_new797(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] ET_ECF_ACTION] */
extern T0* GE_new798(T1 initialize);
/* New instance of type detachable ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
extern T0* GE_new799(T1 initialize);
/* New instance of type detachable ET_ECF_VERSION */
extern T0* GE_new801(T1 initialize);
/* New instance of type detachable ET_ECF_CLUSTER */
extern T0* GE_new802(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_CFLAG */
extern T0* GE_new803(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_INCLUDE */
extern T0* GE_new804(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_LIBRARY */
extern T0* GE_new805(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_LINKER_FLAG */
extern T0* GE_new806(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_MAKE */
extern T0* GE_new807(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_OBJECT */
extern T0* GE_new808(T1 initialize);
/* New instance of type detachable ET_ECF_EXTERNAL_RESOURCE */
extern T0* GE_new809(T1 initialize);
/* New instance of type detachable ET_ECF_FILE_RULE */
extern T0* GE_new810(T1 initialize);
/* New instance of type detachable ET_ECF_ACTION */
extern T0* GE_new811(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
extern T0* GE_new812(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new813(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_OPTIONS] */
extern T0* GE_new814(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_OPTIONS]] */
extern T0* GE_new815(T1 initialize);
/* New instance of type detachable ET_ECF_ROOT_CLASS */
extern T0* GE_new816(T1 initialize);
/* New instance of type [detachable] ET_ECF_ROOT_ALL_CLASSES */
extern T0* GE_new817(T1 initialize);
/* New instance of type [detachable] ET_ECF_ORED_CONDITIONS */
extern T0* GE_new818(T1 initialize);
/* New instance of type detachable ET_ECF_ANDED_CONDITIONS */
extern T0* GE_new819(T1 initialize);
/* New instance of type detachable ET_ECF_VISIBLE_CLASS */
extern T0* GE_new820(T1 initialize);
/* New instance of type detachable ET_ECF_BUILD_CONDITION */
extern T0* GE_new822(T1 initialize);
/* New instance of type detachable ET_ECF_CONCURRENCY_CONDITION */
extern T0* GE_new823(T1 initialize);
/* New instance of type detachable ET_ECF_VOID_SAFETY_CONDITION */
extern T0* GE_new824(T1 initialize);
/* New instance of type detachable ET_ECF_CUSTOM_CONDITION */
extern T0* GE_new825(T1 initialize);
/* New instance of type detachable ET_ECF_DOTNET_CONDITION */
extern T0* GE_new826(T1 initialize);
/* New instance of type detachable ET_ECF_DYNAMIC_RUNTIME_CONDITION */
extern T0* GE_new827(T1 initialize);
/* New instance of type detachable ET_ECF_PLATFORM_CONDITION */
extern T0* GE_new828(T1 initialize);
/* New instance of type [detachable] ET_ECF_COMPILER_VERSION_CONDITION */
extern T0* GE_new830(T1 initialize);
/* New instance of type [detachable] ET_ECF_MSIL_CLR_VERSION_CONDITION */
extern T0* GE_new831(T1 initialize);
/* New instance of type [detachable] UT_COUNTER */
extern T0* GE_new833(T1 initialize);
/* New instance of type [detachable] KL_AGENT_ROUTINES [[attached] ET_CLASS] */
extern T0* GE_new834(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] UT_COUNTER] */
extern T0* GE_new835(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_CLASS], [attached] PROCEDURE [[attached] TUPLE]] */
extern T0* GE_new836(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_UNIVERSE] */
extern T0* GE_new837(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
extern T0* GE_new838(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]]] */
extern T0* GE_new839(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
extern T0* GE_new840(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new842(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new843(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new844(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new845(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new846(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new847(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE]] */
extern T0* GE_new848(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]]] */
extern T0* GE_new849(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLIES] */
extern T0* GE_new850(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new851(T1 initialize);
/* New instance of type [detachable] ET_LIBRARY */
extern T0* GE_new852(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_LIBRARY] */
extern T0* GE_new853(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_LIBRARY]] */
extern T0* GE_new854(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_UNIVERSE] */
extern T0* GE_new855(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_CLUSTER] */
extern T0* GE_new857(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]] */
extern T0* GE_new858(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_SYSTEM] */
extern T0* GE_new859(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new860(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_AST_PROCESSOR] */
extern T0* GE_new861(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_MASTER_CLASS] */
extern T0* GE_new862(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
extern T0* GE_new863(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new864(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY]] */
extern T0* GE_new865(T1 initialize);
/* New instance of type [detachable] ET_ANCESTORS_STATUS_CHECKER */
extern T0* GE_new866(T1 initialize);
/* New instance of type [detachable] ET_FLATTENING_STATUS_CHECKER */
extern T0* GE_new867(T1 initialize);
/* New instance of type [detachable] ET_INTERFACE_STATUS_CHECKER */
extern T0* GE_new868(T1 initialize);
/* New instance of type [detachable] ET_IMPLEMENTATION_STATUS_CHECKER */
extern T0* GE_new869(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]] */
extern T0* GE_new870(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ANCESTORS_STATUS_CHECKER] */
extern T0* GE_new872(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_FLATTENING_STATUS_CHECKER] */
extern T0* GE_new873(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_INTERFACE_STATUS_CHECKER] */
extern T0* GE_new874(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_IMPLEMENTATION_STATUS_CHECKER] */
extern T0* GE_new875(T1 initialize);
/* New instance of type [detachable] TUPLE [BOOLEAN] */
extern T0* GE_new876(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_UNIVERSE]] */
extern T0* GE_new877(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]]] */
extern T0* GE_new880(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE]] */
extern T0* GE_new884(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]]] */
extern T0* GE_new885(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_CLASS], [attached] BOOLEAN]] */
extern T0* GE_new886(T1 initialize);
/* New instance of type [detachable] ET_NONE_GROUP */
extern T0* GE_new887(T1 initialize);
/* New instance of type detachable ET_PARENTHESIS_EXPRESSION */
extern T0* GE_new888(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new890(T1 initialize);
/* New instance of type [detachable] ET_FORMAL_PARAMETER */
extern T0* GE_new891(T1 initialize);
/* New instance of type [detachable] STD_FILES */
extern T0* GE_new892(T1 initialize);
/* New instance of type [detachable] XM_LINKED_LIST [[attached] XM_ELEMENT_NODE] */
extern T0* GE_new894(T1 initialize);
/* New instance of type [detachable] UC_STRING_EQUALITY_TESTER */
extern T0* GE_new896(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
extern T0* GE_new898(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
extern T0* GE_new899(T1 initialize);
/* New instance of type [detachable] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE] */
extern T0* GE_new900(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_INPUT_STREAM */
extern T0* GE_new901(T1 initialize);
/* New instance of type [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
extern T0* GE_new902(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_LIBRARY] */
extern T0* GE_new903(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] ET_LIBRARY]] */
extern T0* GE_new904(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XM_DOCUMENT_NODE] */
extern T0* GE_new905(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET] */
extern T0* GE_new906(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ADAPTED_UNIVERSE] */
extern T0* GE_new907(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET_PARENT] */
extern T0* GE_new909(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_QUERY] */
extern T0* GE_new910(T6 a1, T1 initialize);
/* New instance of type detachable ET_ALIAS_NAME_LIST */
extern T0* GE_new911(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_QUERY] */
extern T0* GE_new912(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_PROCEDURE] */
extern T0* GE_new913(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PROCEDURE] */
extern T0* GE_new914(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new921(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_BASE_TYPE] */
extern T0* GE_new923(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE] */
extern T0* GE_new924(T1 initialize);
/* New instance of type [detachable] INTEGER_OVERFLOW_CHECKER */
extern T0* GE_new925(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new926(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new927(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new928(T1 initialize);
/* New instance of type detachable C_DATE */
extern T0* GE_new929(T1 initialize);
/* New instance of type [detachable] YY_FILE_BUFFER */
extern T0* GE_new930(T1 initialize);
/* New instance of type [detachable] KL_STDIN_FILE */
extern T0* GE_new931(T1 initialize);
/* New instance of type [detachable] YY_BUFFER */
extern T0* GE_new932(T1 initialize);
/* New instance of type [detachable] YY_UNICODE_FILE_BUFFER */
extern T0* GE_new936(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [[attached] INTEGER_32] */
extern T0* GE_new937(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new938(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new939(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [detachable ET_KEYWORD] */
extern T0* GE_new940(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [detachable ET_SYMBOL] */
extern T0* GE_new941(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new942(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [[attached] ET_OBJECT_TEST_LIST] */
extern T0* GE_new943(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [detachable ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new944(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [[attached] ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new945(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ASSERTION_ITEM] */
extern T0* GE_new946(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new947(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_NAMED_CLASS] */
extern T0* GE_new948(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
extern T0* GE_new949(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_KEYWORD] */
extern T0* GE_new950(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
extern T0* GE_new951(T1 initialize);
/* New instance of type detachable ET_AGENT_KEYWORD */
extern T0* GE_new952(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_AGENT_KEYWORD] */
extern T0* GE_new953(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
extern T0* GE_new954(T1 initialize);
/* New instance of type detachable ET_PRECURSOR_KEYWORD */
extern T0* GE_new955(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
extern T0* GE_new956(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
extern T0* GE_new957(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_SYMBOL] */
extern T0* GE_new958(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
extern T0* GE_new959(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_POSITION] */
extern T0* GE_new960(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
extern T0* GE_new961(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
extern T0* GE_new963(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
extern T0* GE_new964(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_BREAK] */
extern T0* GE_new966(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
extern T0* GE_new967(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CHARACTER_CONSTANT] */
extern T0* GE_new969(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
extern T0* GE_new970(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CURRENT] */
extern T0* GE_new971(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
extern T0* GE_new972(T1 initialize);
/* New instance of type detachable ET_FREE_OPERATOR */
extern T0* GE_new973(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FREE_OPERATOR] */
extern T0* GE_new974(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
extern T0* GE_new975(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_IDENTIFIER] */
extern T0* GE_new976(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
extern T0* GE_new977(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INTEGER_CONSTANT] */
extern T0* GE_new979(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
extern T0* GE_new980(T1 initialize);
/* New instance of type detachable ET_KEYWORD_OPERATOR */
extern T0* GE_new981(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_KEYWORD_OPERATOR] */
extern T0* GE_new982(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
extern T0* GE_new983(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_MANIFEST_STRING] */
extern T0* GE_new984(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
extern T0* GE_new985(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_REAL_CONSTANT] */
extern T0* GE_new987(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
extern T0* GE_new988(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_RESULT] */
extern T0* GE_new989(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
extern T0* GE_new990(T1 initialize);
/* New instance of type detachable ET_RETRY_INSTRUCTION */
extern T0* GE_new991(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_RETRY_INSTRUCTION] */
extern T0* GE_new992(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
extern T0* GE_new993(T1 initialize);
/* New instance of type detachable ET_SYMBOL_OPERATOR */
extern T0* GE_new994(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_SYMBOL_OPERATOR] */
extern T0* GE_new995(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_VOID] */
extern T0* GE_new996(T1 initialize);
/* New instance of type detachable ET_VOID */
extern T0* GE_new997(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_VOID] */
extern T0* GE_new998(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
extern T0* GE_new999(T1 initialize);
/* New instance of type detachable ET_SEMICOLON_SYMBOL */
extern T0* GE_new1000(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
extern T0* GE_new1001(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
extern T0* GE_new1002(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_BRACKET_SYMBOL] */
extern T0* GE_new1003(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
extern T0* GE_new1004(T1 initialize);
/* New instance of type detachable ET_QUESTION_MARK_SYMBOL */
extern T0* GE_new1005(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
extern T0* GE_new1006(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
extern T0* GE_new1007(T1 initialize);
/* New instance of type detachable ET_ACROSS_EXPRESSION */
extern T0* GE_new1008(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ACROSS_EXPRESSION] */
extern T0* GE_new1009(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
extern T0* GE_new1010(T1 initialize);
/* New instance of type detachable ET_ACROSS_INSTRUCTION */
extern T0* GE_new1011(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
extern T0* GE_new1012(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new1013(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new1014(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1015(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1016(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new1017(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new1018(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
extern T0* GE_new1019(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
extern T0* GE_new1020(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new1021(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new1023(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
extern T0* GE_new1024(T1 initialize);
/* New instance of type detachable ET_AGENT_ARGUMENT_OPERAND_LIST */
extern T0* GE_new1025(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
extern T0* GE_new1026(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
extern T0* GE_new1027(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_AGENT_TARGET] */
extern T0* GE_new1028(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
extern T0* GE_new1029(T1 initialize);
/* New instance of type detachable ET_ALIAS_NAME */
extern T0* GE_new1030(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ALIAS_NAME] */
extern T0* GE_new1031(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME_LIST] */
extern T0* GE_new1032(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ALIAS_NAME_LIST] */
extern T0* GE_new1033(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
extern T0* GE_new1034(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ASSIGNER] */
extern T0* GE_new1036(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
extern T0* GE_new1037(T1 initialize);
/* New instance of type detachable ET_BRACKET_EXPRESSION */
extern T0* GE_new1038(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_BRACKET_EXPRESSION] */
extern T0* GE_new1039(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
extern T0* GE_new1040(T1 initialize);
/* New instance of type detachable ET_CALL_AGENT */
extern T0* GE_new1041(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CALL_AGENT] */
extern T0* GE_new1042(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1043(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1044(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
extern T0* GE_new1045(T1 initialize);
/* New instance of type detachable ET_CHECK_INSTRUCTION */
extern T0* GE_new1046(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CHECK_INSTRUCTION] */
extern T0* GE_new1047(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
extern T0* GE_new1048(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CHOICE] */
extern T0* GE_new1050(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
extern T0* GE_new1051(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CHOICE_CONSTANT] */
extern T0* GE_new1053(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
extern T0* GE_new1054(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CHOICE_ITEM] */
extern T0* GE_new1056(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
extern T0* GE_new1057(T1 initialize);
/* New instance of type detachable ET_CHOICE_LIST */
extern T0* GE_new1058(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CHOICE_LIST] */
extern T0* GE_new1059(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
extern T0* GE_new1060(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CLASS] */
extern T0* GE_new1061(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
extern T0* GE_new1062(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CLIENT_ITEM] */
extern T0* GE_new1063(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
extern T0* GE_new1064(T1 initialize);
/* New instance of type detachable ET_CLIENTS */
extern T0* GE_new1065(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CLIENTS] */
extern T0* GE_new1066(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
extern T0* GE_new1067(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_COMPOUND] */
extern T0* GE_new1068(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
extern T0* GE_new1069(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONDITIONAL] */
extern T0* GE_new1071(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
extern T0* GE_new1072(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONSTANT] */
extern T0* GE_new1073(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1074(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1076(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new1077(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
extern T0* GE_new1078(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new1079(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
extern T0* GE_new1080(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_CREATOR */
extern T0* GE_new1081(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
extern T0* GE_new1082(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_RENAME_LIST] */
extern T0* GE_new1083(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_RENAME_LIST */
extern T0* GE_new1084(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONSTRAINT_RENAME_LIST] */
extern T0* GE_new1085(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new1086(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new1088(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
extern T0* GE_new1089(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONVERT_FEATURE] */
extern T0* GE_new1091(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1092(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1094(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
extern T0* GE_new1095(T1 initialize);
/* New instance of type detachable ET_CONVERT_FEATURE_LIST */
extern T0* GE_new1096(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
extern T0* GE_new1097(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
extern T0* GE_new1098(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CREATE_EXPRESSION] */
extern T0* GE_new1099(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
extern T0* GE_new1100(T1 initialize);
/* New instance of type detachable ET_CREATION_REGION */
extern T0* GE_new1101(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CREATION_REGION] */
extern T0* GE_new1102(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
extern T0* GE_new1103(T1 initialize);
/* New instance of type detachable ET_CREATOR */
extern T0* GE_new1104(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CREATOR] */
extern T0* GE_new1105(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
extern T0* GE_new1106(T1 initialize);
/* New instance of type detachable ET_CREATOR_LIST */
extern T0* GE_new1107(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_CREATOR_LIST] */
extern T0* GE_new1108(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
extern T0* GE_new1109(T1 initialize);
/* New instance of type detachable ET_DEBUG_INSTRUCTION */
extern T0* GE_new1110(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
extern T0* GE_new1111(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1112(T1 initialize);
/* New instance of type detachable ET_ELSEIF_EXPRESSION */
extern T0* GE_new1113(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1114(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
extern T0* GE_new1115(T1 initialize);
/* New instance of type detachable ET_ELSEIF_EXPRESSION_LIST */
extern T0* GE_new1116(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
extern T0* GE_new1117(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
extern T0* GE_new1118(T1 initialize);
/* New instance of type detachable ET_ELSEIF_PART */
extern T0* GE_new1119(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ELSEIF_PART] */
extern T0* GE_new1120(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
extern T0* GE_new1121(T1 initialize);
/* New instance of type detachable ET_ELSEIF_PART_LIST */
extern T0* GE_new1122(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ELSEIF_PART_LIST] */
extern T0* GE_new1123(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
extern T0* GE_new1124(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_EXPORT] */
extern T0* GE_new1126(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
extern T0* GE_new1127(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_EXPORT_LIST] */
extern T0* GE_new1128(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
extern T0* GE_new1129(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_EXPRESSION] */
extern T0* GE_new1130(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
extern T0* GE_new1131(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_EXPRESSION_ITEM] */
extern T0* GE_new1132(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
extern T0* GE_new1133(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
extern T0* GE_new1134(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
extern T0* GE_new1135(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_EXTERNAL_ALIAS] */
extern T0* GE_new1137(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
extern T0* GE_new1138(T1 initialize);
/* New instance of type detachable ET_FEATURE_CLAUSE */
extern T0* GE_new1139(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FEATURE_CLAUSE] */
extern T0* GE_new1140(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
extern T0* GE_new1141(T1 initialize);
/* New instance of type detachable ET_FEATURE_CLAUSE_LIST */
extern T0* GE_new1142(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
extern T0* GE_new1143(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
extern T0* GE_new1144(T1 initialize);
/* New instance of type detachable ET_FEATURE_EXPORT */
extern T0* GE_new1145(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FEATURE_EXPORT] */
extern T0* GE_new1146(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
extern T0* GE_new1147(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FEATURE_NAME] */
extern T0* GE_new1148(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1149(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1151(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
extern T0* GE_new1152(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT] */
extern T0* GE_new1153(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1154(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1156(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new1157(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new1158(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
extern T0* GE_new1159(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER] */
extern T0* GE_new1160(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1161(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1163(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
extern T0* GE_new1164(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
extern T0* GE_new1165(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
extern T0* GE_new1166(T1 initialize);
/* New instance of type detachable ET_IF_EXPRESSION */
extern T0* GE_new1167(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_IF_EXPRESSION] */
extern T0* GE_new1168(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
extern T0* GE_new1169(T1 initialize);
/* New instance of type detachable ET_IF_INSTRUCTION */
extern T0* GE_new1170(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_IF_INSTRUCTION] */
extern T0* GE_new1171(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_LIST] */
extern T0* GE_new1172(T1 initialize);
/* New instance of type detachable ET_INDEXING_LIST */
extern T0* GE_new1173(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INDEXING_LIST] */
extern T0* GE_new1174(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_ITEM] */
extern T0* GE_new1175(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INDEXING_ITEM] */
extern T0* GE_new1177(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM] */
extern T0* GE_new1178(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INDEXING_TERM] */
extern T0* GE_new1179(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_ITEM] */
extern T0* GE_new1180(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INDEXING_TERM_ITEM] */
extern T0* GE_new1182(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_LIST] */
extern T0* GE_new1183(T1 initialize);
/* New instance of type detachable ET_INDEXING_TERM_LIST */
extern T0* GE_new1184(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INDEXING_TERM_LIST] */
extern T0* GE_new1185(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
extern T0* GE_new1186(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INLINE_AGENT] */
extern T0* GE_new1187(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_EXPRESSION] */
extern T0* GE_new1188(T1 initialize);
/* New instance of type detachable ET_INSPECT_EXPRESSION */
extern T0* GE_new1189(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INSPECT_EXPRESSION] */
extern T0* GE_new1190(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
extern T0* GE_new1191(T1 initialize);
/* New instance of type detachable ET_INSPECT_INSTRUCTION */
extern T0* GE_new1192(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
extern T0* GE_new1193(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
extern T0* GE_new1194(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INSTRUCTION] */
extern T0* GE_new1196(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
extern T0* GE_new1197(T1 initialize);
/* New instance of type detachable ET_INVARIANTS */
extern T0* GE_new1198(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_INVARIANTS] */
extern T0* GE_new1199(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
extern T0* GE_new1200(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
extern T0* GE_new1201(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
extern T0* GE_new1202(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_LIKE_TYPE] */
extern T0* GE_new1204(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
extern T0* GE_new1205(T1 initialize);
/* New instance of type detachable ET_LOCAL_VARIABLE */
extern T0* GE_new1206(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE] */
extern T0* GE_new1207(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1208(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1210(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new1211(T1 initialize);
/* New instance of type detachable ET_LOCAL_VARIABLE_LIST */
extern T0* GE_new1212(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new1213(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
extern T0* GE_new1214(T1 initialize);
/* New instance of type detachable ET_LOOP_INVARIANTS */
extern T0* GE_new1215(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_LOOP_INVARIANTS] */
extern T0* GE_new1216(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
extern T0* GE_new1217(T1 initialize);
/* New instance of type detachable ET_MANIFEST_ARRAY */
extern T0* GE_new1218(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_MANIFEST_ARRAY] */
extern T0* GE_new1219(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1220(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1222(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
extern T0* GE_new1223(T1 initialize);
/* New instance of type detachable ET_MANIFEST_STRING_LIST */
extern T0* GE_new1224(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
extern T0* GE_new1225(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
extern T0* GE_new1226(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_MANIFEST_TUPLE] */
extern T0* GE_new1227(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
extern T0* GE_new1228(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_OBSOLETE] */
extern T0* GE_new1230(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
extern T0* GE_new1231(T1 initialize);
/* New instance of type detachable ET_PARENTHESIZED_EXPRESSION */
extern T0* GE_new1232(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
extern T0* GE_new1233(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
extern T0* GE_new1234(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PARENT] */
extern T0* GE_new1235(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
extern T0* GE_new1236(T1 initialize);
/* New instance of type detachable ET_PARENT_CLAUSE_LIST */
extern T0* GE_new1237(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
extern T0* GE_new1238(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
extern T0* GE_new1239(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PARENT_ITEM] */
extern T0* GE_new1240(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
extern T0* GE_new1241(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PARENT_LIST] */
extern T0* GE_new1242(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
extern T0* GE_new1243(T1 initialize);
/* New instance of type detachable ET_POSTCONDITIONS */
extern T0* GE_new1244(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_POSTCONDITIONS] */
extern T0* GE_new1245(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
extern T0* GE_new1246(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PRECONDITIONS] */
extern T0* GE_new1247(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
extern T0* GE_new1248(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_PROCEDURE] */
extern T0* GE_new1249(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
extern T0* GE_new1250(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
extern T0* GE_new1252(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUANTIFIER_EXPRESSION] */
extern T0* GE_new1253(T1 initialize);
/* New instance of type detachable ET_QUANTIFIER_EXPRESSION */
extern T0* GE_new1254(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_QUANTIFIER_EXPRESSION] */
extern T0* GE_new1255(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
extern T0* GE_new1256(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_QUERY] */
extern T0* GE_new1257(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
extern T0* GE_new1258(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_RENAME_ITEM] */
extern T0* GE_new1260(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
extern T0* GE_new1261(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_RENAME_LIST] */
extern T0* GE_new1262(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_REPEAT_INSTRUCTION] */
extern T0* GE_new1263(T1 initialize);
/* New instance of type detachable ET_REPEAT_INSTRUCTION */
extern T0* GE_new1264(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_REPEAT_INSTRUCTION] */
extern T0* GE_new1265(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
extern T0* GE_new1266(T1 initialize);
/* New instance of type detachable ET_STATIC_CALL_EXPRESSION */
extern T0* GE_new1267(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
extern T0* GE_new1268(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
extern T0* GE_new1269(T1 initialize);
/* New instance of type detachable ET_STRIP_EXPRESSION */
extern T0* GE_new1270(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_STRIP_EXPRESSION] */
extern T0* GE_new1271(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
extern T0* GE_new1272(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_TYPE] */
extern T0* GE_new1273(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT] */
extern T0* GE_new1274(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT] */
extern T0* GE_new1276(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new1277(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new1279(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_LIST] */
extern T0* GE_new1280(T1 initialize);
/* New instance of type detachable ET_TYPE_CONSTRAINT_LIST */
extern T0* GE_new1281(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_TYPE_CONSTRAINT_LIST] */
extern T0* GE_new1282(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
extern T0* GE_new1283(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_TYPE_ITEM] */
extern T0* GE_new1285(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
extern T0* GE_new1286(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_TYPE_LIST] */
extern T0* GE_new1288(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
extern T0* GE_new1289(T1 initialize);
/* New instance of type detachable ET_VARIANT */
extern T0* GE_new1290(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_VARIANT] */
extern T0* GE_new1291(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION] */
extern T0* GE_new1292(T1 initialize);
/* New instance of type detachable ET_WHEN_EXPRESSION */
extern T0* GE_new1293(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_WHEN_EXPRESSION] */
extern T0* GE_new1294(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION_LIST] */
extern T0* GE_new1295(T1 initialize);
/* New instance of type detachable ET_WHEN_EXPRESSION_LIST */
extern T0* GE_new1296(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_WHEN_EXPRESSION_LIST] */
extern T0* GE_new1297(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
extern T0* GE_new1298(T1 initialize);
/* New instance of type detachable ET_WHEN_PART */
extern T0* GE_new1299(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_WHEN_PART] */
extern T0* GE_new1300(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
extern T0* GE_new1301(T1 initialize);
/* New instance of type detachable ET_WHEN_PART_LIST */
extern T0* GE_new1302(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_WHEN_PART_LIST] */
extern T0* GE_new1303(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
extern T0* GE_new1304(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_WRITABLE] */
extern T0* GE_new1305(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_CLASS] */
extern T0* GE_new1306(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
extern T0* GE_new1307(T1 initialize);
/* New instance of type [detachable] ET_PARENT_CHECKER1 */
extern T0* GE_new1308(T1 initialize);
/* New instance of type [detachable] ET_FORMAL_PARAMETER_CHECKER1 */
extern T0* GE_new1309(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1310(T1 initialize);
/* New instance of type [detachable] ET_FEATURE_NAME_TESTER */
extern T0* GE_new1311(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
extern T0* GE_new1313(T1 initialize);
/* New instance of type [detachable] ET_ALIAS_NAME_TESTER */
extern T0* GE_new1314(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_CLIENT_LIST] */
extern T0* GE_new1316(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_CLIENT, [attached] ET_CLASS] */
extern T0* GE_new1317(T1 initialize);
/* New instance of type [detachable] ET_FEATURE_ADAPTATION_RESOLVER */
extern T0* GE_new1318(T1 initialize);
/* New instance of type [detachable] ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
extern T0* GE_new1319(T1 initialize);
/* New instance of type [detachable] ET_IDENTIFIER_TYPE_RESOLVER */
extern T0* GE_new1320(T1 initialize);
/* New instance of type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
extern T0* GE_new1321(T1 initialize);
/* New instance of type [detachable] ET_ANCHORED_TYPE_CHECKER */
extern T0* GE_new1322(T1 initialize);
/* New instance of type [detachable] ET_SIGNATURE_CHECKER */
extern T0* GE_new1323(T1 initialize);
/* New instance of type [detachable] ET_PARENT_CHECKER2 */
extern T0* GE_new1324(T1 initialize);
/* New instance of type [detachable] ET_FORMAL_PARAMETER_CHECKER2 */
extern T0* GE_new1325(T1 initialize);
/* New instance of type [detachable] ET_BUILTIN_FEATURE_CHECKER */
extern T0* GE_new1326(T1 initialize);
/* New instance of type [detachable] ET_PRECURSOR_CHECKER */
extern T0* GE_new1327(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new1328(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_CLASS] */
extern T0* GE_new1329(T1 initialize);
/* New instance of type [detachable] ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
extern T0* GE_new1330(T1 initialize);
/* New instance of type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
extern T0* GE_new1331(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1332(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_ALIAS_NAME] */
extern T0* GE_new1333(T1 initialize);
/* New instance of type [detachable] ET_RENAME */
extern T0* GE_new1337(T1 initialize);
/* New instance of type [detachable] ET_PARENT_CHECKER3 */
extern T0* GE_new1338(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_PROCEDURE] */
extern T0* GE_new1339(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_QUERY] */
extern T0* GE_new1340(T1 initialize);
/* New instance of type [detachable] ET_SUPPLIER_BUILDER */
extern T0* GE_new1341(T1 initialize);
/* New instance of type detachable ET_PARENT_FEATURE */
extern T0* GE_new1344(T1 initialize);
/* New instance of type [detachable] ET_INHERITED_FEATURE */
extern T0* GE_new1345(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CLASS] */
extern T0* GE_new1346(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLASS] */
extern T0* GE_new1347(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLASS] */
extern T0* GE_new1348(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [[attached] YY_BUFFER] */
extern T0* GE_new1349(T1 initialize);
/* New instance of type [detachable] UT_SYNTAX_ERROR */
extern T0* GE_new1350(T1 initialize);
/* New instance of type [detachable] UT_TOO_MANY_INCLUDES_ERROR */
extern T0* GE_new1351(T1 initialize);
/* New instance of type detachable ET_DYNAMIC_TARGET_LIST */
extern T0* GE_new1353(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATIVE_STRING] */
extern T0* GE_new1354(T6 a1, T1 initialize);
/* New instance of type [detachable] ARRAY [detachable STRING_8] */
extern T0* GE_new1355(T1 initialize);
/* New instance of type [detachable] XM_EIFFEL_CHARACTER_ENTITY */
extern T0* GE_new1356(T1 initialize);
/* New instance of type detachable DS_BILINKABLE [[attached] XM_POSITION] */
extern T0* GE_new1358(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_POSITION] */
extern T0* GE_new1359(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XM_EIFFEL_SCANNER] */
extern T0* GE_new1360(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_EIFFEL_ENTITY_DEF] */
extern T0* GE_new1361(T6 a1, T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
extern T0* GE_new1364(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_ENTITY_DEF] */
extern T0* GE_new1365(T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1368(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1369(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new1370(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1371(T1 initialize);
/* New instance of type detachable DS_BILINKABLE [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1372(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST_CURSOR [[attached] STRING_8] */
extern T0* GE_new1373(T1 initialize);
/* New instance of type detachable DS_BILINKABLE [[attached] STRING_8] */
extern T0* GE_new1374(T1 initialize);
/* New instance of type [detachable] XM_NAMESPACE_RESOLVER_CONTEXT */
extern T0* GE_new1375(T1 initialize);
/* New instance of type [detachable] DS_LINKED_QUEUE [detachable STRING_8] */
extern T0* GE_new1378(T1 initialize);
/* New instance of type [detachable] DS_LINKED_QUEUE [[attached] STRING_8] */
extern T0* GE_new1379(T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] XM_NAMESPACE] */
extern T0* GE_new1380(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] XM_NAMESPACE] */
extern T0* GE_new1381(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XM_NAMESPACE] */
extern T0* GE_new1382(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XM_NAMESPACE] */
extern T0* GE_new1383(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_CLUSTER] */
extern T0* GE_new1384(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_PARENT_ITEM] */
extern T0* GE_new1386(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_ITEM] */
extern T0* GE_new1387(T1 initialize);
/* New instance of type [detachable] ET_BRACED_TYPE_LIST */
extern T0* GE_new1388(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] ET_MASTER_CLASS] */
extern T0* GE_new1389(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new1390(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new1391(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new1392(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XM_DOCUMENT_NODE] */
extern T0* GE_new1393(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1395(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1396(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1397(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1398(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new1400(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
extern T0* GE_new1402(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CALL_NAME] */
extern T0* GE_new1404(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
extern T0* GE_new1406(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CALL_NAME] */
extern T0* GE_new1407(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_QUALIFIED_QUERY_CALL, [attached] ET_CALL_NAME] */
extern T0* GE_new1408(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, [attached] ET_CALL_NAME] */
extern T0* GE_new1410(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
extern T0* GE_new1411(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
extern T0* GE_new1412(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1413(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable DS_LINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new1414(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new1415(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new1416(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1418(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1419(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1420(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_CLASS] */
extern T0* GE_new1421(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ADAPTED_CLASS] */
extern T0* GE_new1423(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_CLASS] */
extern T0* GE_new1424(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_PROCEDURE] */
extern T0* GE_new1425(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUERY] */
extern T0* GE_new1426(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1427(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ITERATION_COMPONENT] */
extern T0* GE_new1428(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] INTEGER_32] */
extern T0* GE_new1429(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_8] */
extern T0* GE_new1433(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
extern T0* GE_new1434(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] RX_CHARACTER_SET] */
extern T0* GE_new1435(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] RX_CHARACTER_SET] */
extern T0* GE_new1436(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] NATURAL_64] */
extern T0* GE_new1443(T1 initialize);
/* New instance of type [detachable] KL_CHARACTER_BUFFER */
extern T0* GE_new1446(T1 initialize);
/* New instance of type [detachable] UT_TRISTATE */
extern T0* GE_new1448(T1 initialize);
/* New instance of type [detachable] KL_AGENT_ROUTINES [[attached] ANY] */
extern T0* GE_new1449(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] UT_TRISTATE] */
extern T0* GE_new1450(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ANY], [attached] PROCEDURE [[attached] TUPLE]] */
extern T0* GE_new1451(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
extern T0* GE_new1452(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE] */
extern T0* GE_new1453(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
extern T0* GE_new1455(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
extern T0* GE_new1456(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new1457(T1 initialize);
/* New instance of type [detachable] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE] */
extern T0* GE_new1458(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
extern T0* GE_new1459(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE]] */
extern T0* GE_new1460(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE], [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]]] */
extern T0* GE_new1461(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
extern T0* GE_new1463(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN]] */
extern T0* GE_new1464(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN], [attached] FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
extern T0* GE_new1465(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
extern T0* GE_new1466(T1 initialize);
/* New instance of type [detachable] XM_COMMENT */
extern T0* GE_new1469(T1 initialize);
/* New instance of type [detachable] XM_PROCESSING_INSTRUCTION */
extern T0* GE_new1470(T1 initialize);
/* New instance of type [detachable] XM_CHARACTER_DATA */
extern T0* GE_new1471(T1 initialize);
/* New instance of type [detachable] ET_LIKE_N */
extern T0* GE_new1474(T1 initialize);
/* New instance of type [detachable] KL_STRING_INPUT_STREAM */
extern T0* GE_new1477(T1 initialize);
/* New instance of type [detachable] KL_UNICODE_CHARACTER_BUFFER */
extern T0* GE_new1479(T1 initialize);
/* New instance of type detachable ET_OBJECT_TEST_LIST */
extern T0* GE_new1481(T1 initialize);
/* New instance of type detachable ET_C3_CHARACTER_CONSTANT */
extern T0* GE_new1482(T1 initialize);
/* New instance of type detachable ET_REGULAR_MANIFEST_STRING */
extern T0* GE_new1483(T1 initialize);
/* New instance of type detachable ET_SPECIAL_MANIFEST_STRING */
extern T0* GE_new1484(T1 initialize);
/* New instance of type detachable ET_VERBATIM_STRING */
extern T0* GE_new1485(T1 initialize);
/* New instance of type detachable ET_UNDERSCORED_INTEGER_CONSTANT */
extern T0* GE_new1486(T1 initialize);
/* New instance of type detachable ET_HEXADECIMAL_INTEGER_CONSTANT */
extern T0* GE_new1487(T1 initialize);
/* New instance of type detachable ET_OCTAL_INTEGER_CONSTANT */
extern T0* GE_new1488(T1 initialize);
/* New instance of type detachable ET_BINARY_INTEGER_CONSTANT */
extern T0* GE_new1489(T1 initialize);
/* New instance of type detachable ET_REGULAR_REAL_CONSTANT */
extern T0* GE_new1490(T1 initialize);
/* New instance of type detachable ET_UNDERSCORED_REAL_CONSTANT */
extern T0* GE_new1491(T1 initialize);
/* New instance of type detachable ET_TRUE_CONSTANT */
extern T0* GE_new1493(T1 initialize);
/* New instance of type detachable ET_FALSE_CONSTANT */
extern T0* GE_new1494(T1 initialize);
/* New instance of type detachable ET_C1_CHARACTER_CONSTANT */
extern T0* GE_new1495(T1 initialize);
/* New instance of type detachable ET_C2_CHARACTER_CONSTANT */
extern T0* GE_new1496(T1 initialize);
/* New instance of type detachable ET_TAGGED_INDEXING */
extern T0* GE_new1498(T1 initialize);
/* New instance of type detachable ET_INDEXING */
extern T0* GE_new1499(T1 initialize);
/* New instance of type detachable ET_CUSTOM_ATTRIBUTE */
extern T0* GE_new1500(T1 initialize);
/* New instance of type detachable ET_CONSTRAINED_FORMAL_PARAMETER */
extern T0* GE_new1501(T1 initialize);
/* New instance of type detachable ET_TYPE_RENAME_CONSTRAINT */
extern T0* GE_new1502(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_NAMED_TYPE */
extern T0* GE_new1503(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
extern T0* GE_new1504(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
extern T0* GE_new1505(T1 initialize);
/* New instance of type detachable ET_ALL_EXPORT */
extern T0* GE_new1506(T1 initialize);
/* New instance of type detachable ET_CONVERT_FUNCTION */
extern T0* GE_new1508(T1 initialize);
/* New instance of type detachable ET_CONVERT_PROCEDURE */
extern T0* GE_new1509(T1 initialize);
/* New instance of type detachable ET_DO_FUNCTION */
extern T0* GE_new1511(T1 initialize);
/* New instance of type detachable ET_ONCE_FUNCTION */
extern T0* GE_new1512(T1 initialize);
/* New instance of type detachable ET_DEFERRED_FUNCTION */
extern T0* GE_new1513(T1 initialize);
/* New instance of type detachable ET_ONCE_PROCEDURE */
extern T0* GE_new1515(T1 initialize);
/* New instance of type detachable ET_EXTERNAL_PROCEDURE */
extern T0* GE_new1516(T1 initialize);
/* New instance of type detachable ET_ALIASED_FEATURE_NAME */
extern T0* GE_new1517(T1 initialize);
/* New instance of type detachable ET_ALIAS_FREE_NAME */
extern T0* GE_new1518(T1 initialize);
/* New instance of type detachable ET_CLASS_ASSERTION */
extern T0* GE_new1521(T1 initialize);
/* New instance of type detachable ET_LABELED_ACTUAL_PARAMETER */
extern T0* GE_new1522(T1 initialize);
/* New instance of type detachable ET_LIKE_FEATURE */
extern T0* GE_new1524(T1 initialize);
/* New instance of type detachable ET_QUALIFIED_LIKE_BRACED_TYPE */
extern T0* GE_new1525(T1 initialize);
/* New instance of type detachable ET_QUALIFIED_LIKE_TYPE */
extern T0* GE_new1526(T1 initialize);
/* New instance of type detachable ET_ASSIGNER_INSTRUCTION */
extern T0* GE_new1527(T1 initialize);
/* New instance of type detachable ET_ASSIGNMENT */
extern T0* GE_new1528(T1 initialize);
/* New instance of type detachable ET_ASSIGNMENT_ATTEMPT */
extern T0* GE_new1529(T1 initialize);
/* New instance of type detachable ET_LOOP_INSTRUCTION */
extern T0* GE_new1530(T1 initialize);
/* New instance of type detachable ET_BANG_INSTRUCTION */
extern T0* GE_new1532(T1 initialize);
/* New instance of type detachable ET_CREATE_INSTRUCTION */
extern T0* GE_new1533(T1 initialize);
/* New instance of type detachable ET_CHOICE_RANGE */
extern T0* GE_new1534(T1 initialize);
/* New instance of type detachable ET_PRECURSOR_INSTRUCTION */
extern T0* GE_new1535(T1 initialize);
/* New instance of type detachable ET_STATIC_CALL_INSTRUCTION */
extern T0* GE_new1537(T1 initialize);
/* New instance of type detachable ET_PRECURSOR_EXPRESSION */
extern T0* GE_new1538(T1 initialize);
/* New instance of type detachable ET_FEATURE_ADDRESS */
extern T0* GE_new1539(T1 initialize);
/* New instance of type detachable ET_CURRENT_ADDRESS */
extern T0* GE_new1540(T1 initialize);
/* New instance of type detachable ET_RESULT_ADDRESS */
extern T0* GE_new1541(T1 initialize);
/* New instance of type detachable ET_EXPRESSION_ADDRESS */
extern T0* GE_new1542(T1 initialize);
/* New instance of type detachable ET_INFIX_EXPRESSION */
extern T0* GE_new1543(T1 initialize);
/* New instance of type detachable ET_INFIX_AND_THEN_OPERATOR */
extern T0* GE_new1544(T1 initialize);
/* New instance of type detachable ET_INFIX_OR_ELSE_OPERATOR */
extern T0* GE_new1545(T1 initialize);
/* New instance of type detachable ET_MANIFEST_TYPE */
extern T0* GE_new1546(T1 initialize);
/* New instance of type detachable ET_PREFIX_EXPRESSION */
extern T0* GE_new1547(T1 initialize);
/* New instance of type detachable ET_OLD_EXPRESSION */
extern T0* GE_new1548(T1 initialize);
/* New instance of type detachable ET_OLD_OBJECT_TEST */
extern T0* GE_new1549(T1 initialize);
/* New instance of type detachable ET_ONCE_MANIFEST_STRING */
extern T0* GE_new1550(T1 initialize);
/* New instance of type detachable ET_ITERATION_CURSOR */
extern T0* GE_new1551(T1 initialize);
/* New instance of type detachable ET_DO_FUNCTION_INLINE_AGENT */
extern T0* GE_new1552(T1 initialize);
/* New instance of type detachable ET_ONCE_FUNCTION_INLINE_AGENT */
extern T0* GE_new1553(T1 initialize);
/* New instance of type detachable ET_EXTERNAL_FUNCTION_INLINE_AGENT */
extern T0* GE_new1554(T1 initialize);
/* New instance of type detachable ET_DO_PROCEDURE_INLINE_AGENT */
extern T0* GE_new1555(T1 initialize);
/* New instance of type detachable ET_ONCE_PROCEDURE_INLINE_AGENT */
extern T0* GE_new1556(T1 initialize);
/* New instance of type detachable ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
extern T0* GE_new1557(T1 initialize);
/* New instance of type detachable ET_AGENT_OPEN_TARGET */
extern T0* GE_new1558(T1 initialize);
/* New instance of type detachable ET_AGENT_TYPED_OPEN_ARGUMENT */
extern T0* GE_new1559(T1 initialize);
/* New instance of type detachable ET_TAGGED_ASSERTION */
extern T0* GE_new1562(T1 initialize);
/* New instance of type [detachable] ET_FILE_POSITION */
extern T0* GE_new1564(T1 initialize);
/* New instance of type [detachable] DS_QUICK_SORTER [[attached] ET_QUERY] */
extern T0* GE_new1566(T1 initialize);
/* New instance of type [detachable] DS_QUICK_SORTER [[attached] ET_PROCEDURE] */
extern T0* GE_new1567(T1 initialize);
/* New instance of type [detachable] ET_REDECLARED_FEATURE */
extern T0* GE_new1569(T1 initialize);
/* New instance of type [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
extern T0* GE_new1571(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_MASTER_CLASS, [attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1573(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATURAL_16] */
extern T0* GE_new1576(T6 a1, T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] NATURAL_16] */
extern T0* GE_new1577(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] NATURAL_32] */
extern T0* GE_new1578(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_CLUSTER] */
extern T0* GE_new1579(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_LIBRARY] */
extern T0* GE_new1580(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1581(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_FILE_RULE] */
extern T0* GE_new1582(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_CFLAG] */
extern T0* GE_new1583(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_INCLUDE] */
extern T0* GE_new1584(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LIBRARY] */
extern T0* GE_new1585(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
extern T0* GE_new1586(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_MAKE] */
extern T0* GE_new1587(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_OBJECT] */
extern T0* GE_new1588(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_RESOURCE] */
extern T0* GE_new1589(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ACTION] */
extern T0* GE_new1590(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_ACTION] */
extern T0* GE_new1591(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ACTION] */
extern T0* GE_new1592(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] RX_PCRE_REGULAR_EXPRESSION] */
extern T0* GE_new1593(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_OPTIONS] */
extern T0* GE_new1594(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_OPTIONS] */
extern T0* GE_new1597(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
extern T0* GE_new1598(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1599(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1600(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1601(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_ANDED_CONDITIONS] */
extern T0* GE_new1602(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_ECF_CONDITION_ITEM] */
extern T0* GE_new1603(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1604(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1605(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1606(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_UNIVERSE] */
extern T0* GE_new1608(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_UNIVERSE] */
extern T0* GE_new1609(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_UNIVERSE] */
extern T0* GE_new1611(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1612(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1614(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1615(T6 a1, T1 initialize);
/* New instance of type [detachable] ET_CLASS_TYPE_STATUS_CHECKER1 */
extern T0* GE_new1616(T1 initialize);
/* New instance of type [detachable] ET_CLASS_TYPE_STATUS_CHECKER2 */
extern T0* GE_new1617(T1 initialize);
/* New instance of type [detachable] ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
extern T0* GE_new1618(T1 initialize);
/* New instance of type [detachable] ET_CLASS_TYPE_STATUS_CHECKER3 */
extern T0* GE_new1619(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_NAMED_CLASS] */
extern T0* GE_new1620(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1621(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1622(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1623(T1 initialize);
/* New instance of type [detachable] CONSOLE */
extern T0* GE_new1624(T1 initialize);
/* New instance of type [detachable] DS_LINKED_QUEUE [[attached] CHARACTER_8] */
extern T0* GE_new1626(T1 initialize);
/* New instance of type [detachable] UC_UTF16_ROUTINES */
extern T0* GE_new1627(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_LIBRARY] */
extern T0* GE_new1628(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LIBRARY] */
extern T0* GE_new1629(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_LIBRARY] */
extern T0* GE_new1630(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET] */
extern T0* GE_new1631(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1632(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1633(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1634(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_TARGET_PARENT] */
extern T0* GE_new1635(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET_PARENT] */
extern T0* GE_new1636(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET_PARENT] */
extern T0* GE_new1637(T1 initialize);
/* New instance of type [detachable] YY_UNICODE_BUFFER */
extern T0* GE_new1638(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new1639(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new1640(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST_LIST] */
extern T0* GE_new1641(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_OBJECT_TEST_LIST] */
extern T0* GE_new1642(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1643(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1644(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1645(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1646(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ASSERTION_ITEM] */
extern T0* GE_new1647(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTION_ITEM] */
extern T0* GE_new1648(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ASSERTION_ITEM] */
extern T0* GE_new1649(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new1650(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_NAMED_CLASS] */
extern T0* GE_new1651(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_CLASS] */
extern T0* GE_new1653(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_CLASS] */
extern T0* GE_new1655(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
extern T0* GE_new1656(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1659(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] NATURAL_32] */
extern T0* GE_new1660(T1 initialize);
/* New instance of type [detachable] ET_BASE_TYPE_CONSTRAINT_LIST */
extern T0* GE_new1661(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1662(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FLATTENED_FEATURE] */
extern T0* GE_new1663(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FEATURE_NAME] */
extern T0* GE_new1666(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FLATTENED_FEATURE] */
extern T0* GE_new1667(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME] */
extern T0* GE_new1668(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ALIAS_NAME] */
extern T0* GE_new1669(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ALIAS_NAME] */
extern T0* GE_new1671(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
extern T0* GE_new1672(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_LIST] */
extern T0* GE_new1673(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CLIENT_LIST] */
extern T0* GE_new1674(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLIENT_LIST] */
extern T0* GE_new1675(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
extern T0* GE_new1677(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT] */
extern T0* GE_new1678(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CLIENT] */
extern T0* GE_new1679(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_FEATURE_NAME] */
extern T0* GE_new1680(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] BOOLEAN, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1681(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new1682(T1 initialize);
/* New instance of type [detachable] ET_REPLICATED_FEATURE */
extern T0* GE_new1684(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] ET_ADAPTED_FEATURE] */
extern T0* GE_new1685(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new1686(T1 initialize);
/* New instance of type [detachable] ET_DOTNET_SIGNATURE_TESTER */
extern T0* GE_new1687(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1688(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1690(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1692(T1 initialize);
/* New instance of type [detachable] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_LIKE_FEATURE] */
extern T0* GE_new1693(T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] ET_LIKE_FEATURE] */
extern T0* GE_new1694(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new1695(T1 initialize);
/* New instance of type detachable DS_LINKED_LIST [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new1696(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
extern T0* GE_new1697(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new1698(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_CLASS] */
extern T0* GE_new1701(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1703(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_RENAME] */
extern T0* GE_new1704(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RENAME] */
extern T0* GE_new1707(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_ALIAS_NAME] */
extern T0* GE_new1711(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_PROCEDURE] */
extern T0* GE_new1712(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_QUERY] */
extern T0* GE_new1713(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] YY_BUFFER] */
extern T0* GE_new1714(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] YY_BUFFER] */
extern T0* GE_new1715(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_TARGET] */
extern T0* GE_new1716(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TARGET] */
extern T0* GE_new1717(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new1719(T1 initialize);
/* New instance of type detachable DS_BILINKABLE [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new1720(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
extern T0* GE_new1722(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST_CURSOR [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
extern T0* GE_new1723(T1 initialize);
/* New instance of type detachable DS_LINKABLE [detachable STRING_8] */
extern T0* GE_new1724(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CLUSTER] */
extern T0* GE_new1725(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CLUSTER] */
extern T0* GE_new1726(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLUSTER] */
extern T0* GE_new1727(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_TYPE_ITEM] */
extern T0* GE_new1728(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_ITEM] */
extern T0* GE_new1729(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_MASTER_CLASS] */
extern T0* GE_new1730(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1731(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_CLUSTER] */
extern T0* GE_new1732(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_CLUSTER]] */
extern T0* GE_new1733(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
extern T0* GE_new1734(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY] */
extern T0* GE_new1736(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY]] */
extern T0* GE_new1737(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1738(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
extern T0* GE_new1739(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] STRING_8]] */
extern T0* GE_new1742(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
extern T0* GE_new1743(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8] */
extern T0* GE_new1744(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
extern T0* GE_new1745(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XM_ELEMENT_NODE] */
extern T0* GE_new1750(T1 initialize);
/* New instance of type detachable ET_CONSTRAINT_GENERIC_NAMED_TYPE */
extern T0* GE_new1755(T1 initialize);
/* New instance of type detachable ET_UNQUALIFIED_CALL_INSTRUCTION */
extern T0* GE_new1756(T1 initialize);
/* New instance of type detachable ET_CLUSTER_DEPENDENCE_CONSTRAINT */
extern T0* GE_new1758(T1 initialize);
/* New instance of type [detachable] ET_SEEDED_QUERY_COMPARATOR */
extern T0* GE_new1759(T1 initialize);
/* New instance of type [detachable] ET_SEEDED_PROCEDURE_COMPARATOR */
extern T0* GE_new1762(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_CLIENT_LIST] */
extern T0* GE_new1766(T1 initialize);
/* New instance of type [detachable] ET_FORMAL_COMMA_ARGUMENT */
extern T0* GE_new1767(T1 initialize);
/* New instance of type [detachable] ET_LOCAL_COMMA_VARIABLE */
extern T0* GE_new1768(T1 initialize);
/* New instance of type [detachable] ET_LABELED_COMMA_ACTUAL_PARAMETER */
extern T0* GE_new1769(T1 initialize);
/* New instance of type [detachable] ET_KEYWORD_EXPRESSION */
extern T0* GE_new1771(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_RENAME_ITEM] */
extern T0* GE_new1776(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_RENAME_ITEM] */
extern T0* GE_new1777(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_EXPORT] */
extern T0* GE_new1779(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_EXPORT] */
extern T0* GE_new1780(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1782(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1783(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1784(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1785(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG] */
extern T0* GE_new1788(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG]] */
extern T0* GE_new1789(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_TARGET, [attached] ET_ECF_STATE] */
extern T0* GE_new1790(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE] */
extern T0* GE_new1791(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE]] */
extern T0* GE_new1792(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY] */
extern T0* GE_new1793(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY]] */
extern T0* GE_new1794(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
extern T0* GE_new1795(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG]] */
extern T0* GE_new1796(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE] */
extern T0* GE_new1797(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE]] */
extern T0* GE_new1798(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT] */
extern T0* GE_new1799(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT]] */
extern T0* GE_new1800(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE] */
extern T0* GE_new1801(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE]] */
extern T0* GE_new1802(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
extern T0* GE_new1806(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new1809(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new1810(T1 initialize);
/* New instance of type [detachable] ET_AGENT_IMPLICIT_CURRENT_TARGET */
extern T0* GE_new1811(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CHOICE_ITEM] */
extern T0* GE_new1813(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CHOICE_ITEM] */
extern T0* GE_new1814(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1815(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1817(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1819(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1820(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_CREATOR] */
extern T0* GE_new1821(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_CREATOR] */
extern T0* GE_new1822(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1823(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1824(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ELSEIF_PART] */
extern T0* GE_new1825(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_PART] */
extern T0* GE_new1826(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FEATURE_CLAUSE] */
extern T0* GE_new1827(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_CLAUSE] */
extern T0* GE_new1828(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INDEXING_ITEM] */
extern T0* GE_new1830(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_ITEM] */
extern T0* GE_new1831(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INDEXING_TERM_ITEM] */
extern T0* GE_new1832(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM_ITEM] */
extern T0* GE_new1833(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1834(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1836(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1838(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1839(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_PARENT_LIST] */
extern T0* GE_new1840(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_LIST] */
extern T0* GE_new1841(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new1843(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new1844(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_WHEN_EXPRESSION] */
extern T0* GE_new1845(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_EXPRESSION] */
extern T0* GE_new1846(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_WHEN_PART] */
extern T0* GE_new1847(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_PART] */
extern T0* GE_new1848(T1 initialize);
/* New instance of type [detachable] ET_BASE_TYPE_RENAME_CONSTRAINT */
extern T0* GE_new1850(T1 initialize);
/* New instance of type detachable DS_SPARSE_TABLE_KEYS [[attached] ET_CLIENT, [attached] ET_CLASS] */
extern T0* GE_new1853(T1 initialize);
/* New instance of type [detachable] ET_STANDARD_ONCE_KEYS */
extern T0* GE_new1857(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_CLUSTER] */
extern T0* GE_new1859(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CLUSTER] */
extern T0* GE_new1860(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CLUSTER] */
extern T0* GE_new1861(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_ADAPTED_LIBRARY] */
extern T0* GE_new1862(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_LIBRARY] */
extern T0* GE_new1863(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_LIBRARY] */
extern T0* GE_new1864(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1865(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1866(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1867(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_FILE_RULE] */
extern T0* GE_new1868(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_FILE_RULE] */
extern T0* GE_new1869(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_FILE_RULE] */
extern T0* GE_new1870(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_CFLAG] */
extern T0* GE_new1871(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_CFLAG] */
extern T0* GE_new1872(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_CFLAG] */
extern T0* GE_new1873(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_INCLUDE] */
extern T0* GE_new1874(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_INCLUDE] */
extern T0* GE_new1875(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_INCLUDE] */
extern T0* GE_new1876(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LIBRARY] */
extern T0* GE_new1877(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_LIBRARY] */
extern T0* GE_new1878(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LIBRARY] */
extern T0* GE_new1879(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
extern T0* GE_new1880(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
extern T0* GE_new1881(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
extern T0* GE_new1882(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_MAKE] */
extern T0* GE_new1883(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_MAKE] */
extern T0* GE_new1884(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_MAKE] */
extern T0* GE_new1885(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_OBJECT] */
extern T0* GE_new1886(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_OBJECT] */
extern T0* GE_new1887(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_OBJECT] */
extern T0* GE_new1888(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_RESOURCE] */
extern T0* GE_new1889(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_EXTERNAL_RESOURCE] */
extern T0* GE_new1890(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_RESOURCE] */
extern T0* GE_new1891(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] RX_PCRE_REGULAR_EXPRESSION] */
extern T0* GE_new1892(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] RX_PCRE_REGULAR_EXPRESSION] */
extern T0* GE_new1893(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] RX_PCRE_REGULAR_EXPRESSION] */
extern T0* GE_new1894(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ANDED_CONDITIONS] */
extern T0* GE_new1895(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_ANDED_CONDITIONS] */
extern T0* GE_new1896(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ANDED_CONDITIONS] */
extern T0* GE_new1897(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CONDITION_ITEM] */
extern T0* GE_new1898(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_ECF_CONDITION_ITEM] */
extern T0* GE_new1899(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CONDITION_ITEM] */
extern T0* GE_new1900(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] CHARACTER_8] */
extern T0* GE_new1902(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_CLASS] */
extern T0* GE_new1903(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1904(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1905(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1906(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] NATURAL_32] */
extern T0* GE_new1907(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_FEATURE_NAME] */
extern T0* GE_new1908(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1909(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new1911(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_REPLICABLE_FEATURE] */
extern T0* GE_new1912(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_REPLICABLE_FEATURE] */
extern T0* GE_new1914(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_ADAPTED_FEATURE] */
extern T0* GE_new1915(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] ET_ADAPTED_FEATURE] */
extern T0* GE_new1916(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new1917(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new1918(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new1919(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1920(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1921(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1922(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
extern T0* GE_new1923(T6 a1, T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1926(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
extern T0* GE_new1927(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1928(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] ET_DOTNET_FEATURE] */
extern T0* GE_new1929(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
extern T0* GE_new1930(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_LIKE_FEATURE] */
extern T0* GE_new1931(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_LIKE_FEATURE] */
extern T0* GE_new1932(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_LIKE_FEATURE] */
extern T0* GE_new1933(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
extern T0* GE_new1935(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
extern T0* GE_new1936(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
extern T0* GE_new1937(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
extern T0* GE_new1938(T6 a1, T1 initialize);
/* New instance of type detachable DS_BILINKABLE [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
extern T0* GE_new1939(T1 initialize);
/* New instance of type [detachable] ET_CLUSTER_ERROR */
extern T0* GE_new1940(T1 initialize);
/* New instance of type [detachable] ET_SYNTAX_ERROR */
extern T0* GE_new1941(T1 initialize);
/* New instance of type [detachable] ET_DOTNET_ASSEMBLY_ERROR */
extern T0* GE_new1942(T1 initialize);
/* New instance of type [detachable] AP_OPTION_COMPARATOR */
extern T0* GE_new1943(T1 initialize);
/* New instance of type [detachable] DS_QUICK_SORTER [[attached] AP_OPTION] */
extern T0* GE_new1944(T1 initialize);
/* New instance of type [detachable] ST_SPLITTER */
extern T0* GE_new1953(T1 initialize);
/* New instance of type [detachable] KL_CHARACTER_ROUTINES */
extern T0* GE_new1954(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, INTEGER_32, [attached] ET_CURRENT, [attached] ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
extern T0* GE_new1955(T1 initialize);
/* New instance of type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
extern T0* GE_new1956(T1 initialize);
/* New instance of type [detachable] ET_ACTUAL_PARAMETER_SUBLIST */
extern T0* GE_new1957(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1960(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1962(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_DYNAMIC_PRECURSOR] */
extern T0* GE_new1964(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRECURSOR] */
extern T0* GE_new1965(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_INSTRUCTION] */
extern T0* GE_new1966(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_INSTRUCTION] */
extern T0* GE_new1968(T1 initialize);
/* New instance of type [detachable] ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
extern T0* GE_new1977(T1 initialize);
/* New instance of type [detachable] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8] */
extern T0* GE_new1978(T1 initialize);
/* New instance of type [detachable] ET_INFIX_CAST_EXPRESSION */
extern T0* GE_new1980(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] STRING_8] */
extern T0* GE_new1984(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
extern T0* GE_new1985(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
extern T0* GE_new1988(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
extern T0* GE_new1989(T1 initialize);
/* New instance of type [detachable] DS_SPARSE_TABLE_KEYS_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
extern T0* GE_new1990(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
extern T0* GE_new1993(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
extern T0* GE_new1994(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
extern T0* GE_new1997(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
extern T0* GE_new1998(T1 initialize);
/* New instance of type [detachable] ET_UNIVERSE_ERROR */
extern T0* GE_new1999(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new2000(T1 initialize);
/* New instance of type detachable ET_PARENTHESIS_INSTRUCTION */
extern T0* GE_new2002(T1 initialize);
/* New instance of type [detachable] ET_PARENTHESIS_SYMBOL */
extern T0* GE_new2020(T1 initialize);
/* New instance of type [detachable] ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
extern T0* GE_new2026(T1 initialize);
/* New instance of type [detachable] ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
extern T0* GE_new2027(T1 initialize);
/* New instance of type [detachable] DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new2034(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] LX_WILDCARD, [attached] STRING_8] */
extern T0* GE_new2040(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_WILDCARD, [attached] STRING_8] */
extern T0* GE_new2041(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_EQUALITY_EXPRESSION */
extern T0* GE_new2042(T1 initialize);
/* New instance of type [detachable] ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
extern T0* GE_new2043(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_LIBRARY] */
extern T0* GE_new2044(T1 initialize);
/* New instance of type [detachable] ET_CONVERT_BUILTIN_EXPRESSION */
extern T0* GE_new2046(T1 initialize);
/* New instance of type [detachable] ET_CONVERT_FROM_EXPRESSION */
extern T0* GE_new2047(T1 initialize);
/* New instance of type [detachable] ET_CONVERT_TO_EXPRESSION */
extern T0* GE_new2048(T1 initialize);
/* New instance of type detachable DS_CELL [detachable ET_CLASS] */
extern T0* GE_new2051(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_CELL [detachable ET_CLASS]] */
extern T0* GE_new2052(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_EIFFEL_PREPARSER, [attached] DS_ARRAYED_LIST [[attached] STRING_8]] */
extern T0* GE_new2054(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] ET_FEATURE] */
extern T0* GE_new2056(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_FILE_RULE] */
extern T0* GE_new2057(T1 initialize);
/* New instance of type [detachable] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
extern T0* GE_new2058(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_FILE_RULES] */
extern T0* GE_new2059(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
extern T0* GE_new2060(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_STATE] */
extern T0* GE_new2061(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] STRING_8]] */
extern T0* GE_new2063(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] KL_STRING_ROUTINES, [attached] STRING_8] */
extern T0* GE_new2064(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] DS_HASH_SET [[attached] STRING_8]] */
extern T0* GE_new2066(T1 initialize);
/* New instance of type [detachable] LX_DFA_WILDCARD */
extern T0* GE_new2068(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] ET_PARENT_FEATURE] */
extern T0* GE_new2071(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] LX_WILDCARD] */
extern T0* GE_new2073(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] LX_WILDCARD]] */
extern T0* GE_new2074(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_CLUSTER_DEPENDENCE_CONSTRAINT, [attached] ET_GROUP, [attached] STRING_8] */
extern T0* GE_new2075(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
extern T0* GE_new2077(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
extern T0* GE_new2078(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_WILDCARD] */
extern T0* GE_new2079(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_WILDCARD] */
extern T0* GE_new2082(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_C_GENERATOR, [attached] ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new2086(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new2089(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new2090(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new2091(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new2092(T1 initialize);
/* New instance of type [detachable] ET_AST_NULL_LEAF */
extern T0* GE_new2094(T1 initialize);
/* New instance of type [detachable] LX_WILDCARD_PARSER */
extern T0* GE_new2095(T1 initialize);
/* New instance of type [detachable] LX_DESCRIPTION */
extern T0* GE_new2096(T1 initialize);
/* New instance of type [detachable] LX_FULL_DFA */
extern T0* GE_new2097(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] ET_GROUP] */
extern T0* GE_new2099(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]]] */
extern T0* GE_new2100(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION] */
extern T0* GE_new2101(T1 initialize);
/* New instance of type [detachable] PREDICATE [[attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION]] */
extern T0* GE_new2102(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] LX_RULE] */
extern T0* GE_new2105(T1 initialize);
/* New instance of type [detachable] LX_START_CONDITIONS */
extern T0* GE_new2106(T1 initialize);
/* New instance of type [detachable] LX_ACTION_FACTORY */
extern T0* GE_new2107(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]], [attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2108(T1 initialize);
/* New instance of type [detachable] LX_SYMBOL_CLASS */
extern T0* GE_new2109(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] STRING_32, [attached] STRING_8] */
extern T0* GE_new2110(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2111(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] LX_SYMBOL_CLASS, [attached] STRING_32] */
extern T0* GE_new2112(T1 initialize);
/* New instance of type [detachable] KL_EQUALITY_TESTER [[attached] STRING_32] */
extern T0* GE_new2113(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_STACK [[attached] BOOLEAN] */
extern T0* GE_new2114(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_32] */
extern T0* GE_new2115(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] STRING_32] */
extern T0* GE_new2116(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2117(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2118(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_NFA] */
extern T0* GE_new2119(T1 initialize);
/* New instance of type [detachable] LX_NFA */
extern T0* GE_new2120(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_NFA] */
extern T0* GE_new2121(T6 a1, T1 initialize);
/* New instance of type detachable LX_RULE */
extern T0* GE_new2122(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2123(T1 initialize);
/* New instance of type [detachable] LX_EQUIVALENCE_CLASSES */
extern T0* GE_new2124(T1 initialize);
/* New instance of type [detachable] LX_UNRECOGNIZED_RULE_ERROR */
extern T0* GE_new2125(T1 initialize);
/* New instance of type [detachable] LX_INVALID_UNICODE_CHARACTER_ERROR */
extern T0* GE_new2126(T1 initialize);
/* New instance of type [detachable] LX_MISSING_QUOTE_ERROR */
extern T0* GE_new2127(T1 initialize);
/* New instance of type [detachable] LX_BAD_CHARACTER_CLASS_ERROR */
extern T0* GE_new2128(T1 initialize);
/* New instance of type [detachable] LX_BAD_CHARACTER_ERROR */
extern T0* GE_new2129(T1 initialize);
/* New instance of type [detachable] LX_FULL_AND_META_ERROR */
extern T0* GE_new2130(T1 initialize);
/* New instance of type [detachable] LX_FULL_AND_REJECT_ERROR */
extern T0* GE_new2131(T1 initialize);
/* New instance of type [detachable] LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
extern T0* GE_new2132(T1 initialize);
/* New instance of type [detachable] LX_CHARACTER_OUT_OF_RANGE_ERROR */
extern T0* GE_new2133(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] LX_RULE] */
extern T0* GE_new2134(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_RULE] */
extern T0* GE_new2135(T6 a1, T1 initialize);
/* New instance of type [detachable] LX_DFA_STATE */
extern T0* GE_new2136(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] LX_NFA_STATE] */
extern T0* GE_new2137(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] LX_DFA_STATE] */
extern T0* GE_new2138(T1 initialize);
/* New instance of type detachable LX_SYMBOL_PARTITIONS */
extern T0* GE_new2139(T1 initialize);
/* New instance of type [detachable] KL_ARRAY_ROUTINES [[attached] LX_RULE] */
extern T0* GE_new2140(T1 initialize);
/* New instance of type [detachable] ARRAY [detachable LX_RULE] */
extern T0* GE_new2141(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable LX_RULE] */
extern T0* GE_new2142(T6 a1, T1 initialize);
/* New instance of type [detachable] LX_START_CONDITION */
extern T0* GE_new2143(T1 initialize);
/* New instance of type [detachable] LX_TRANSITION_TABLE [[attached] LX_DFA_STATE] */
extern T0* GE_new2144(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] LX_NFA] */
extern T0* GE_new2145(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] LX_NFA, [attached] INTEGER_32] */
extern T0* GE_new2146(T1 initialize);
/* New instance of type [detachable] LX_NFA_STATE */
extern T0* GE_new2147(T1 initialize);
/* New instance of type detachable ARRAY [detachable LX_SYMBOL_CLASS] */
extern T0* GE_new2148(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable LX_SYMBOL_CLASS] */
extern T0* GE_new2149(T6 a1, T1 initialize);
/* New instance of type detachable ARRAY [detachable DS_HASH_SET [[attached] INTEGER_32]] */
extern T0* GE_new2150(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable DS_HASH_SET [[attached] INTEGER_32]] */
extern T0* GE_new2151(T6 a1, T1 initialize);
/* New instance of type detachable DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
extern T0* GE_new2152(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2153(T1 initialize);
/* New instance of type [detachable] LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
extern T0* GE_new2154(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_RULE] */
extern T0* GE_new2155(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_RULE] */
extern T0* GE_new2156(T1 initialize);
/* New instance of type [detachable] KL_ARRAY_ROUTINES [[attached] STRING_8] */
extern T0* GE_new2157(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_START_CONDITION] */
extern T0* GE_new2158(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_START_CONDITION] */
extern T0* GE_new2159(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_START_CONDITION] */
extern T0* GE_new2160(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]]] */
extern T0* GE_new2161(T6 a1, T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2162(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]]] */
extern T0* GE_new2164(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]], [attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2165(T1 initialize);
/* New instance of type detachable SPECIAL [detachable SPECIAL [[attached] NATURAL_64]] */
extern T0* GE_new2166(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] STRING_32, [attached] STRING_8] */
extern T0* GE_new2168(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_SYMBOL_CLASS, [attached] STRING_32] */
extern T0* GE_new2170(T1 initialize);
/* New instance of type [detachable] LX_EPSILON_TRANSITION [[attached] LX_NFA_STATE] */
extern T0* GE_new2171(T1 initialize);
/* New instance of type [detachable] LX_SYMBOL_CLASS_TRANSITION [[attached] LX_NFA_STATE] */
extern T0* GE_new2174(T1 initialize);
/* New instance of type [detachable] LX_SYMBOL_TRANSITION [[attached] LX_NFA_STATE] */
extern T0* GE_new2175(T1 initialize);
/* New instance of type [detachable] LX_ACTION */
extern T0* GE_new2177(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new2178(T1 initialize);
/* New instance of type [detachable] DS_BILINKABLE [[attached] INTEGER_32] */
extern T0* GE_new2179(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new2180(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_BUBBLE_SORTER [[attached] LX_NFA_STATE] */
extern T0* GE_new2181(T1 initialize);
/* New instance of type [detachable] DS_BUBBLE_SORTER [[attached] LX_RULE] */
extern T0* GE_new2183(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_NFA_STATE] */
extern T0* GE_new2185(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_NFA_STATE] */
extern T0* GE_new2187(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA_STATE] */
extern T0* GE_new2188(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] LX_DFA_STATE] */
extern T0* GE_new2190(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] LX_DFA_STATE] */
extern T0* GE_new2191(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_DFA_STATE] */
extern T0* GE_new2192(T1 initialize);
/* New instance of type [detachable] KL_ARRAY [[attached] LX_RULE] */
extern T0* GE_new2193(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] LX_DFA_STATE, [attached] INTEGER_32] */
extern T0* GE_new2194(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA] */
extern T0* GE_new2195(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_NFA, [attached] INTEGER_32] */
extern T0* GE_new2197(T1 initialize);
/* New instance of type [detachable] KL_COMPARABLE_COMPARATOR [[attached] LX_RULE] */
extern T0* GE_new2201(T1 initialize);
/* New instance of type [detachable] KL_COMPARABLE_COMPARATOR [[attached] LX_NFA_STATE] */
extern T0* GE_new2204(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
extern T0* GE_new2207(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
extern T0* GE_new2208(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] LX_SYMBOL_CLASS]] */
extern T0* GE_new2209(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] LX_SYMBOL_CLASS] */
extern T0* GE_new2210(T1 initialize);
/* New instance of type [detachable] KL_ARRAY [[attached] STRING_8] */
extern T0* GE_new2211(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] LX_DFA_STATE, [attached] INTEGER_32] */
extern T0* GE_new2213(T1 initialize);
/* [detachable] GEC.execute */
extern T0* T25c61(GE_context* ac);
/* [detachable] GEC.execute_with_arguments */
extern void T25f62(GE_context* ac, T0* C, T0* a1);
/* [detachable] GEC.execute_with_arguments_and_error_handler */
extern void T25f63(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] GEC.report_cannot_read_error */
extern void T25f67(GE_context* ac, T0* C, T0* a1);
/* [detachable] ET_NULL_ERROR_HANDLER.report_error */
extern void T53f183(GE_context* ac, T0* C, T0* a1);
/* [detachable] ET_NULL_ERROR_HANDLER.report_error_message */
extern void T53f185(GE_context* ac, T0* C, T0* a1);
/* [detachable] MUTEX.unlock */
extern void T131f12(GE_context* ac, T0* C);
/* [detachable] MUTEX.eif_thr_mutex_unlock */
extern void T131f15(GE_context* ac, T0* C, T14 a1);
/* [detachable] MUTEX.default_pointer */
extern T14 T131f3(GE_context* ac, T0* C);
/* [detachable] ET_NULL_ERROR_HANDLER.set_has_error */
extern void T53f187(GE_context* ac, T0* C, T1 a1);
/* [detachable] ET_NULL_ERROR_HANDLER.report_error_message */
extern void T53f185p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.put_line */
extern void T161f4(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.put_new_line */
extern void T161f7(GE_context* ac, T0* C);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.put_string */
extern void T161f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] MUTEX.lock */
extern void T131f11(GE_context* ac, T0* C);
/* [detachable] MUTEX.current_thread_id */
extern T14 T131f4(GE_context* ac, T0* C);
/* [detachable] MUTEX.eif_thr_mutex_lock */
extern void T131f14(GE_context* ac, T0* C, T14 a1);
/* [detachable] ET_NULL_ERROR_HANDLER.message */
extern T0* T53f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] ET_UNIVERSE_ERROR.default_message */
extern T0* T1999f12(GE_context* ac, T0* C);
/* [detachable] ET_UNIVERSE_ERROR.message */
extern T0* T1999f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] ET_UNIVERSE_ERROR.arguments */
extern T0* T1999f6(GE_context* ac, T0* C);
/* [detachable] KL_ARGUMENTS.make */
extern T0* T32c6(GE_context* ac);
/* [detachable] KL_ARGUMENTS.argument */
extern T0* T32f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] IMMUTABLE_STRING_32.as_string_8 */
extern T0* T20f9(GE_context* ac, T0* C);
/* [detachable] STRING_8.put_code */
extern void T17f65(GE_context* ac, T0* C, T10 a1, T6 a2);
/* [detachable] STRING_8.reset_hash_codes */
extern void T17f53(GE_context* ac, T0* C);
/* [detachable] STRING_8.valid_code */
extern T1 T17f39(GE_context* ac, T0* C, T10 a1);
/* NATURAL_32.is_less_equal */
extern T1 T10f18(GE_context* ac, T10* C, T10 a1);
/* INTEGER_32.to_natural_32 */
extern T10 T6f21(GE_context* ac, T6* C);
/* [detachable] IMMUTABLE_STRING_32.code */
extern T10 T20f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.set_count */
extern void T17f62(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.make */
extern void T17f48(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.make */
extern T0* T17c48(GE_context* ac, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].make_filled */
extern T0* T15c12(GE_context* ac, T2 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_8].fill_with */
extern void T15f15(GE_context* ac, T0* C, T2 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_8].extend */
extern void T15f19(GE_context* ac, T0* C, T2 a1);
/* INTEGER_32.min */
extern T6 T6f17(GE_context* ac, T6* C, T6 a1);
/* INTEGER_32.is_less_equal */
extern T1 T6f15(GE_context* ac, T6* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].make_empty */
extern void T15f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].make_empty */
extern T0* T15c16(GE_context* ac, T6 a1);
/* [detachable] ARGUMENTS_32.argument */
extern T0* T130f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].item */
extern T0* T484f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] ARGUMENTS_32.internal_argument_array */
extern T0* T130f3(GE_context* ac, T0* C);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].put */
extern void T484f8(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].compare_objects */
extern void T484f7(GE_context* ac, T0* C);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].make_filled */
extern T0* T484c6(GE_context* ac, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].make_filled_area */
extern void T484f9(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].make_filled */
extern T0* T485c6(GE_context* ac, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].fill_with */
extern void T485f9(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].extend */
extern void T485f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].make_empty */
extern void T485f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].make_empty */
extern T0* T485c4(GE_context* ac, T6 a1);
/* [detachable] IMMUTABLE_STRING_32.make_empty */
extern T0* T20c14(GE_context* ac);
/* [detachable] IMMUTABLE_STRING_32.make */
extern void T20f18(GE_context* ac, T0* C, T6 a1);
/* [detachable] IMMUTABLE_STRING_32.make */
extern T0* T20c18(GE_context* ac, T6 a1);
/* [detachable] SPECIAL [CHARACTER_32].make_filled */
extern T0* T16c8(GE_context* ac, T3 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_32].fill_with */
extern void T16f11(GE_context* ac, T0* C, T3 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_32].extend */
extern void T16f15(GE_context* ac, T0* C, T3 a1);
/* [detachable] SPECIAL [CHARACTER_32].make_empty */
extern void T16f12(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_32].make_empty */
extern T0* T16c12(GE_context* ac, T6 a1);
/* [detachable] KL_ARGUMENTS.internal_arguments */
extern T0* T32f4(GE_context* ac, T0* C);
/* [detachable] ARGUMENTS_32.default_create */
extern T0* T130c5(GE_context* ac);
/* [detachable] KL_STRING_ROUTINES.appended_string */
extern T0* T82f10(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] UC_UTF8_STRING.to_string_32 */
extern T0* T775f20(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.next_byte_index */
extern T6 T775f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.encoded_byte_count */
extern T6 T434f4(GE_context* ac, T0* C, T2 a1);
/* CHARACTER_8.is_less_equal */
extern T1 T2f9(GE_context* ac, T2* C, T2 a1);
/* CHARACTER_8.is_less */
extern T1 T2f14(GE_context* ac, T2* C, T2 a1);
/* CHARACTER_8.natural_32_code */
extern T10 T2f10(GE_context* ac, T2* C);
/* [detachable] UC_UTF8_STRING.byte_item */
extern T2 T775f14(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.old_item */
extern T2 T775f48(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.set_count */
extern void T775f72(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.reset_byte_index_cache */
extern void T775f93(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.utf8 */
extern T0* T775f42(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_ROUTINES.default_create */
extern T0* T434c53(GE_context* ac);
/* [detachable] STRING_32.append_code */
extern void T18f25(GE_context* ac, T0* C, T10 a1);
/* [detachable] STRING_32.put_code */
extern void T18f31(GE_context* ac, T0* C, T10 a1, T6 a2);
/* [detachable] STRING_32.reset_hash_codes */
extern void T18f30(GE_context* ac, T0* C);
/* [detachable] STRING_32.set_count */
extern void T18f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_32.resize */
extern void T18f28(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_32].aliased_resized_area_with_default */
extern T0* T16f4(GE_context* ac, T0* C, T3 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_32].aliased_resized_area */
extern T0* T16f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_32.capacity */
extern T6 T18f5(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.item_code_at_byte_index */
extern T6 T775f15(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.encoded_next_value */
extern T6 T434f5(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_UTF8_ROUTINES.encoded_first_value */
extern T6 T434f3(GE_context* ac, T0* C, T2 a1);
/* [detachable] STRING_32.make */
extern void T18f23(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_32.make */
extern T0* T18c23(GE_context* ac, T6 a1);
/* [detachable] UC_STRING.to_string_32 */
extern T0* T429f55(GE_context* ac, T0* C);
/* [detachable] UC_STRING.next_byte_index */
extern T6 T429f34(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.byte_item */
extern T2 T429f37(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.old_item */
extern T2 T429f53(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.set_count */
extern void T429f86(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.reset_byte_index_cache */
extern void T429f99(GE_context* ac, T0* C);
/* [detachable] UC_STRING.utf8 */
extern T0* T429f40(GE_context* ac, T0* C);
/* [detachable] UC_STRING.item_code_at_byte_index */
extern T6 T429f33(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.to_string_32 */
extern T0* T17f30(GE_context* ac, T0* C);
/* [detachable] STRING_8.code */
extern T10 T17f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.append_string_general */
extern void T775f73(GE_context* ac, T0* C, T0* a1);
extern T1 T775f73ot1(T0* a1, T0** a2);
/* [detachable] UC_UTF8_STRING.append_string_general */
extern void T775f73p1(GE_context* ac, T0* C, T0* a1);
extern T1 T775f73p1ot1(T0* a1, T0** a2);
/* [detachable] UC_UTF8_STRING.append */
extern void T775f73p0(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_UTF8_STRING.reset_hash_codes */
extern void T775f96(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.old_set_count */
extern void T775f89(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.append_code */
extern void T775f78(GE_context* ac, T0* C, T10 a1);
/* [detachable] UC_UTF8_STRING.append_item_code */
extern void T775f102(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.put_code_at_byte_index */
extern void T775f109(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] UC_UTF8_STRING.put_byte */
extern void T775f110(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] UC_UTF8_STRING.old_put */
extern void T775f98(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] KL_INTEGER_ROUTINES.to_character */
extern T2 T772f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.integer_ */
extern T0* T775f56(GE_context* ac, T0* C);
/* [detachable] KL_INTEGER_ROUTINES.default_create */
extern T0* T772c5(GE_context* ac);
/* [detachable] UC_UTF8_STRING.resize_byte_storage */
extern void T775f97(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.code_byte_count */
extern T6 T434f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.natural_32_code_byte_count */
extern T6 T434f18(GE_context* ac, T0* C, T10 a1);
/* [detachable] STRING_32.code */
extern T10 T18f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.code */
extern T10 T775f17(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.item_code */
extern T6 T775f46(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.byte_index */
extern T6 T775f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.code */
extern T10 T429f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.item_code */
extern T6 T429f23(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.byte_index */
extern T6 T429f38(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.resize */
extern void T775f95(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].aliased_resized_area_with_default */
extern T0* T15f4(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_8].aliased_resized_area */
extern T0* T15f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.byte_capacity */
extern T6 T775f59(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.capacity */
extern T6 T775f59p1(GE_context* ac, T0* C);
/* [detachable] STRING_32.count */
extern T6 T18f2(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.count */
extern T6 T775f1(GE_context* ac, T0* C);
/* [detachable] UC_STRING.count */
extern T6 T429f1(GE_context* ac, T0* C);
/* [detachable] STRING_8.count */
extern T6 T17f2(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.append */
extern void T775f94(GE_context* ac, T0* C, T0* a1);
extern T1 T775f94ot1(T0* a1, T0** a2);
extern T1 T775f94ot2(T0* a1);
/* [detachable] UC_UTF8_STRING.dummy_uc_string */
extern T0* T775f63(GE_context* ac, T0* C);
/* [detachable] UC_STRING.make_empty */
extern T0* T429c106(GE_context* ac);
/* [detachable] UC_STRING.make */
extern void T429f80(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.make */
extern T0* T429c80(GE_context* ac, T6 a1);
/* [detachable] UC_STRING.old_set_count */
extern void T429f83(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.reset_hash_codes */
extern void T429f90(GE_context* ac, T0* C);
/* [detachable] UC_STRING.byte_capacity */
extern T6 T429f57(GE_context* ac, T0* C);
/* [detachable] UC_STRING.capacity */
extern T6 T429f57p1(GE_context* ac, T0* C);
/* [detachable] UC_STRING.make */
extern void T429f80p1(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.gobo_append_substring */
extern void T775f75(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] UC_UTF8_STRING.put_substring_at_byte_index */
extern void T775f100(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4, T6 a5);
extern T1 T775f100ot1(T0* a1, T0** a2);
extern T1 T775f100ot2(T0* a1, T0** a2);
extern T1 T775f100ot3(T0* a1, T0** a2);
/* NATURAL_32.to_integer_32 */
extern T6 T10f8(GE_context* ac, T10* C);
/* [detachable] UC_UTF8_STRING.put_character_at_byte_index */
extern void T775f99(GE_context* ac, T0* C, T2 a1, T6 a2, T6 a3);
/* [detachable] UC_UTF8_ROUTINES.character_byte_count */
extern T6 T434f6(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_UTF8_ROUTINES.character_8_byte_count */
extern T6 T434f11(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_STRING.item */
extern T2 T429f21(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.character_item_at_byte_index */
extern T2 T429f47(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.integer_ */
extern T0* T429f56(GE_context* ac, T0* C);
/* [detachable] KL_PLATFORM.maximum_character_code */
extern T6 T477f2(GE_context* ac, T0* C);
/* [detachable] UC_STRING.platform */
extern T0* T429f43(GE_context* ac, T0* C);
/* [detachable] KL_PLATFORM.default_create */
extern T0* T477c3(GE_context* ac);
/* [detachable] UC_UTF8_STRING.item */
extern T2 T775f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.character_item_at_byte_index */
extern T2 T775f43(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.platform */
extern T0* T775f45(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.cloned_string */
extern T0* T775f32(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.twin */
extern T0* T775f25(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.copy */
extern void T775f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_UTF8_STRING.copy */
extern void T775f88p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [CHARACTER_8].copy_data */
extern void T15f14(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] SPECIAL [CHARACTER_8].force */
extern void T15f18(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_8].move_data */
extern void T15f17(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_8].overlapping_move */
extern void T15f21(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* INTEGER_32.is_greater_equal */
extern T1 T6f14(GE_context* ac, T6* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].non_overlapping_move */
extern void T15f20(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_8].resized_area */
extern T0* T15f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.append */
extern void T775f94p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_UTF8_STRING.area_lower */
extern T6 T775f47(GE_context* ac, T0* C);
/* [detachable] UC_STRING.area_lower */
extern T6 T429f17(GE_context* ac, T0* C);
/* [detachable] STRING_8.area_lower */
extern T6 T17f9(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.area */
extern T0* T775f2(GE_context* ac, T0* C);
/* [detachable] UC_STRING.area */
extern T0* T429f2(GE_context* ac, T0* C);
/* [detachable] STRING_8.area */
extern T0* T17f1(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_STRING.additional_space */
extern T6 T775f60(GE_context* ac, T0* C);
/* INTEGER_32.max */
extern T6 T6f16(GE_context* ac, T6* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.substring_byte_count */
extern T6 T434f2(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
extern T1 T434f2ot1(T0* a1, T0** a2);
extern T1 T434f2ot2(T0* a1, T0** a2);
extern T1 T434f2ot3(T0* a1, T0** a2);
/* [detachable] UC_UTF8_ROUTINES.character_32_byte_count */
extern T6 T434f13(GE_context* ac, T0* C, T3 a1);
/* [detachable] STRING_32.item */
extern T3 T18f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.character_32_item */
extern T3 T775f13(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.character_32_item */
extern T3 T429f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.character_32_item */
extern T3 T17f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_STRING.shifted_byte_index */
extern T6 T775f12(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] UC_STRING.shifted_byte_index */
extern T6 T429f62(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] UC_UTF8_ROUTINES.dummy_uc_string */
extern T0* T434f12(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_ROUTINES.any_ */
extern T0* T434f9(GE_context* ac, T0* C);
/* [detachable] KL_ANY_ROUTINES.default_create */
extern T0* T481c2(GE_context* ac);
/* [detachable] KL_ANY_ROUTINES.same_types */
extern T1 T481f1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_ARRAYED_LIST [[attached] LX_NFA_STATE].same_type */
extern T1 T2137f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] LX_DFA_STATE.same_type */
extern T1 T2136f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_SET [[attached] INTEGER_32].same_type */
extern T1 T745f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_32.same_type */
extern T1 T18f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_UTF8_STRING.same_type */
extern T1 T775f23(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.same_type */
extern T1 T17f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_STRING.same_type */
extern T1 T429f32(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_UNIX_FILE_SYSTEM.same_type */
extern T1 T126f26(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_WINDOWS_FILE_SYSTEM.same_type */
extern T1 T125f29(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_UTF8_STRING.any_ */
extern T0* T775f41(GE_context* ac, T0* C);
/* [detachable] UC_STRING.append_string_general */
extern void T429f74(GE_