/*
 * Decompiled with CFR 0.152.
 */
package ch.cyberduck.core;

import ch.cyberduck.core.Host;
import ch.cyberduck.core.Protocol;
import ch.cyberduck.core.ProtocolFactory;
import ch.cyberduck.core.exception.HostParserException;
import ch.cyberduck.core.preferences.Preferences;
import ch.cyberduck.core.preferences.PreferencesFactory;
import java.util.Optional;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.conn.util.InetAddressUtils;
import org.apache.log4j.Logger;

public final class HostParser {
    private static final Logger log = Logger.getLogger(HostParser.class);
    private static final Preferences preferences = PreferencesFactory.get();
    private final Protocol defaultScheme;
    private final ProtocolFactory factory;
    static final String URI_SCHEME = "+-.";
    static final String URI_UNRESERVED = "-._~";
    static final String URI_SUBDELIMS = "!$&'()*+,;=";
    static final String URI_PCHAR = ":@";
    private static final Pattern IPV6_STD_PATTERN = Pattern.compile("(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))");

    public HostParser() {
        this(ProtocolFactory.get());
    }

    public HostParser(ProtocolFactory factory) {
        this.factory = factory;
        this.defaultScheme = factory.forName(preferences.getProperty("connection.protocol.default"));
    }

    public HostParser(ProtocolFactory factory, Protocol defaultScheme) {
        this.factory = factory;
        this.defaultScheme = defaultScheme;
    }

    public Host get(String url) throws HostParserException {
        Host parsed = HostParser.parse(this.factory, this.defaultScheme, url);
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Parsed %s as %s", url, parsed));
        }
        return parsed;
    }

    public static Host parse(String url) throws HostParserException {
        Host parsed = new HostParser(ProtocolFactory.get(), ProtocolFactory.get().forName(preferences.getProperty("connection.protocol.default"))).get(url);
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Parsed %s as %s", url, parsed));
        }
        return parsed;
    }

    private static Host parse(ProtocolFactory factory, Protocol defaultScheme, String url) throws HostParserException {
        Protocol protocol;
        StringReader reader = new StringReader(url);
        Value<String> schemeValue = new Value<String>();
        if (!HostParser.parseScheme(reader, schemeValue) || null == schemeValue.getValue() || (protocol = factory.forName(schemeValue.getValue())) == null) {
            protocol = defaultScheme;
        }
        Host host = new Host(protocol);
        URITypes uriType = HostParser.findURIType(reader);
        if (uriType == URITypes.Undefined) {
            if (StringUtils.isBlank((CharSequence)protocol.getDefaultHostname())) {
                throw new HostParserException(String.format("Missing hostname in URI %s", url));
            }
            return host;
        }
        if (uriType == URITypes.Authority) {
            if (host.getProtocol().isHostnameConfigurable()) {
                HostParser.parseAuthority(reader, host);
            } else {
                HostParser.parseRootless(reader, host);
            }
        } else if (uriType == URITypes.Rootless) {
            HostParser.parseRootless(reader, host);
        } else if (uriType == URITypes.Absolute) {
            HostParser.parseAbsolute(reader, host);
        }
        return host;
    }

    static boolean parseScheme(StringReader reader, Value<String> scheme) {
        StringBuilder stringBuilder = new StringBuilder();
        int tracker = reader.position;
        while (!reader.endOfString()) {
            char c = (char)reader.read();
            if (Character.isAlphabetic(c) || Character.isDigit(c) || URI_SCHEME.indexOf(c) != -1) {
                if (c == '.') {
                    reader.skip(tracker - reader.position);
                    return false;
                }
                stringBuilder.append(c);
                continue;
            }
            if (c == ':') {
                tracker = reader.position;
                break;
            }
            if (c == ' ' && stringBuilder.length() == 0) continue;
            reader.skip(tracker - reader.position);
            return false;
        }
        reader.skip(tracker - reader.position);
        scheme.setValue(stringBuilder.toString());
        return true;
    }

    static void parseAuthority(StringReader reader, Host host) throws HostParserException {
        HostParser.parseUserInfo(reader, host);
        HostParser.parseHostname(reader, host);
        HostParser.parsePath(reader, host, false);
    }

    static void parseHostname(StringReader reader, Host host) throws HostParserException {
        char c;
        StringBuilder buffer = new StringBuilder();
        boolean isPort = false;
        boolean bracketFlag = false;
        while (!reader.endOfString() && '/' != (c = (char)reader.peek())) {
            if (bracketFlag) {
                reader.skip(1);
                if (c == '[') {
                    throw new HostParserException("Illegal character '[' inside IPv6 address");
                }
                if (c == ']') {
                    bracketFlag = false;
                    continue;
                }
                if (Character.isLetterOrDigit(c) || c == ':' || c == '%') {
                    buffer.append(c);
                    continue;
                }
                throw new HostParserException(String.format("Illegal character '%s' at %d inside IPv6 address", Character.valueOf(c), reader.position));
            }
            if (HostParser.readPercentEscapedSequence(reader, buffer)) continue;
            reader.skip(1);
            if (c == ']') {
                throw new HostParserException("Illegal character ']' outside IPv6 address");
            }
            if (c == '[') {
                bracketFlag = true;
                continue;
            }
            if (c == ':') {
                isPort = true;
                break;
            }
            buffer.append(c);
        }
        if (bracketFlag) {
            throw new HostParserException("IPv6 bracket not closed in URI");
        }
        if (buffer.length() == 0) {
            if (StringUtils.isEmpty((CharSequence)host.getHostname())) {
                throw new HostParserException("Missing hostname in URI");
            }
        } else {
            host.setHostname(buffer.toString());
        }
        if (isPort) {
            HostParser.parsePort(reader, host);
        }
    }

    static void parsePort(StringReader reader, Host host) throws HostParserException {
        Integer port = null;
        int tracker = reader.position;
        while (!reader.endOfString()) {
            char c = (char)reader.read();
            if (Character.isDigit(c)) {
                port = Optional.ofNullable(port).orElse(0) * 10 + Character.getNumericValue(c);
                continue;
            }
            if (c != '/') {
                port = null;
                log.warn((Object)String.format("Got %s in port. This is unsupported. Continuing with default port", Character.valueOf(c)));
                reader.skip(tracker - reader.position);
                break;
            }
            reader.skip(-1);
            break;
        }
        if (port != null && host.getProtocol().isPortConfigurable()) {
            if (port <= 0 || port >= 65536) {
                throw new HostParserException(String.format("Port %d is outside allowed range 0-65536", port));
            }
            host.setPort(port);
        }
    }

    static void parseAbsolute(StringReader reader, Host host) {
        HostParser.parsePath(reader, host, true);
    }

    static void parseRootless(StringReader reader, Host host) throws HostParserException {
        boolean userInfoResult = HostParser.parseUserInfo(reader, host);
        if (host.getProtocol().isHostnameConfigurable() && StringUtils.isWhitespace((CharSequence)host.getHostname())) {
            HostParser.parseHostname(reader, host);
        }
        HostParser.parsePath(reader, host, false);
    }

    static boolean parseUserInfo(StringReader reader, Host host) throws HostParserException {
        int tracker = reader.position;
        StringBuilder buffer = new StringBuilder();
        StringBuilder userBuilder = new StringBuilder();
        StringBuilder passwordBuilder = null;
        boolean atSignFlag = false;
        while (!reader.endOfString()) {
            if (HostParser.readPercentEscapedSequence(reader, buffer)) continue;
            char c = (char)reader.read();
            if ('@' == c) {
                if (atSignFlag) {
                    buffer.insert(0, c);
                }
                atSignFlag = true;
                int length = buffer.length();
                for (int i = 0; i < length; ++i) {
                    char t = buffer.charAt(i);
                    if (t == ' ') {
                        throw new HostParserException(String.format("Space character in user info part of URL at %d", reader.position));
                    }
                    if (t == ':' && passwordBuilder == null) {
                        passwordBuilder = new StringBuilder();
                        continue;
                    }
                    if (passwordBuilder != null) {
                        passwordBuilder.append(t);
                        continue;
                    }
                    userBuilder.append(t);
                }
                tracker = reader.position;
                buffer.setLength(0);
                continue;
            }
            if (c == '/') break;
            buffer.append(c);
        }
        reader.skip(tracker - reader.position);
        if (host.getProtocol().isAnonymousConfigurable()) {
            host.getCredentials().setUsername(preferences.getProperty("connection.login.anon.name"));
        } else {
            host.getCredentials().setUsername(preferences.getProperty("connection.login.name"));
        }
        host.getCredentials().setPassword(null);
        if (atSignFlag) {
            if (userBuilder.length() > 0) {
                if (host.getProtocol().isUsernameConfigurable()) {
                    host.getCredentials().setUsername(userBuilder.toString());
                } else {
                    log.warn((Object)"Username specified on protocol which does not support user credentials. Username will be ignored.");
                }
            }
            userBuilder.setLength(0);
            if (passwordBuilder != null) {
                if (passwordBuilder.length() > 0) {
                    if (host.getProtocol().isPasswordConfigurable()) {
                        host.getCredentials().setPassword(passwordBuilder.toString());
                    } else {
                        log.warn((Object)"Password specified on protocol which does not support user credentials. Password will be ignored.");
                    }
                }
                passwordBuilder.setLength(0);
            }
            return true;
        }
        return false;
    }

    static void parsePath(StringReader reader, Host host, boolean assumeRoot) {
        StringBuilder pathBuilder = new StringBuilder();
        if (assumeRoot) {
            if (reader.peek() == 47) {
                pathBuilder.append((char)reader.read());
            } else {
                pathBuilder.append('/');
            }
        }
        while (!reader.endOfString()) {
            if (HostParser.readPercentEscapedSequence(reader, pathBuilder)) continue;
            pathBuilder.append((char)reader.read());
        }
        if (pathBuilder.length() > 0) {
            if (host.getProtocol().isPathConfigurable()) {
                host.setDefaultPath(pathBuilder.toString());
            } else if (StringUtils.isNotBlank((CharSequence)host.getDefaultPath())) {
                if (pathBuilder.indexOf(host.getDefaultPath()) != -1) {
                    host.setDefaultPath(pathBuilder.toString());
                } else {
                    host.setDefaultPath(String.format("%s%s", host.getDefaultPath(), pathBuilder));
                }
            } else {
                host.setDefaultPath(pathBuilder.toString());
            }
        }
    }

    static URITypes findURIType(StringReader reader) {
        StringReader copy = reader.copy();
        if (!copy.endOfString()) {
            char c = (char)copy.read();
            if (c == '/') {
                reader.skip(1);
                if (!copy.endOfString()) {
                    c = (char)copy.read();
                    if (c == '/') {
                        reader.skip(1);
                        return URITypes.Authority;
                    }
                    reader.skip(-1);
                }
                return URITypes.Absolute;
            }
            return URITypes.Rootless;
        }
        return URITypes.Undefined;
    }

    private static boolean isUnreservedCharacter(char c) {
        return Character.isAlphabetic(c) || Character.isDigit(c) || URI_UNRESERVED.indexOf(c) != -1;
    }

    private static boolean isSubDelimsCharacter(char c) {
        return URI_SUBDELIMS.indexOf(c) != -1;
    }

    private static boolean isPChar(char c) {
        return HostParser.isUnreservedCharacter(c) || HostParser.isSubDelimsCharacter(c) || URI_PCHAR.indexOf(c) != -1;
    }

    private static boolean isValidUserInfo(char c) {
        return HostParser.isUnreservedCharacter(c) || HostParser.isSubDelimsCharacter(c) || c == ':';
    }

    private static boolean readPercentEscapedSequence(StringReader reader, StringBuilder builder) {
        char append;
        Integer b = HostParser.readPercentCharacter(reader);
        if (b == null) {
            return false;
        }
        if (b < 128) {
            append = (char)b.byteValue();
        } else if ((b & 0xE0) == 192) {
            Character c = HostParser.handleTwoByteSequence(b.byteValue(), reader);
            if (c == null) {
                return false;
            }
            append = c.charValue();
        } else if ((b & 0xF0) == 224) {
            Character c = HostParser.handleThreeByteSequence(b.byteValue(), reader);
            if (c == null) {
                return false;
            }
            append = c.charValue();
        } else if ((b & 0xF8) == 240 && b <= 244) {
            Character c = HostParser.handleFourByteSequence(b.byteValue(), reader);
            if (c == null) {
                return false;
            }
            append = c.charValue();
        } else {
            return false;
        }
        if (append >= '\ud800' && append <= '\udfff') {
            return false;
        }
        builder.append(append);
        return true;
    }

    private static Character handleTwoByteSequence(byte b, StringReader reader) {
        Integer b2 = HostParser.readPercentCharacter(reader);
        if (b2 == null) {
            return null;
        }
        return Character.valueOf((char)((b & 0x1F) << 6 | b2.byteValue() & 0x3F));
    }

    private static Character handleThreeByteSequence(byte b, StringReader reader) {
        Integer b2 = HostParser.readPercentCharacter(reader);
        if (b2 == null) {
            return null;
        }
        Integer b3 = HostParser.readPercentCharacter(reader);
        if (b3 == null) {
            return null;
        }
        return Character.valueOf((char)((b & 0xF) << 12 | (b2.byteValue() & 0x3F) << 6 | b3.byteValue() & 0x3F));
    }

    private static Character handleFourByteSequence(byte b, StringReader reader) {
        Integer b2 = HostParser.readPercentCharacter(reader);
        if (b2 == null) {
            return null;
        }
        Integer b3 = HostParser.readPercentCharacter(reader);
        if (b3 == null) {
            return null;
        }
        Integer b4 = HostParser.readPercentCharacter(reader);
        if (b4 == null) {
            return null;
        }
        return Character.valueOf((char)((b & 7) << 18 | (b2.byteValue() & 0x3F) << 12 | (b3.byteValue() & 0x3F) << 6 | b4.byteValue() & 0x4F));
    }

    private static Integer readPercentCharacter(StringReader reader) {
        int start = reader.position;
        if (reader.read() != 37) {
            reader.skip(start - reader.position);
            return null;
        }
        int value = 0;
        for (int i = 0; i < 2 && !reader.endOfString(); ++i) {
            int cv = Character.digit((char)reader.read(), 16);
            if (cv < 0) {
                reader.skip(start - reader.position);
                return null;
            }
            value = value * 16 + cv;
        }
        return value;
    }

    private static boolean isv6Address(String address) {
        if (IPV6_STD_PATTERN.matcher(address).matches()) {
            return true;
        }
        return InetAddressUtils.isIPv6Address((String)address);
    }

    static enum URITypes {
        Authority,
        Absolute,
        Rootless,
        Undefined;

    }

    static final class Value<T> {
        private T value;

        Value() {
        }

        public T getValue() {
            return this.value;
        }

        public T setValue(T value) {
            this.value = value;
            return value;
        }
    }

    static final class StringReader {
        private final int eof;
        private final CharSequence text;
        private int position = 0;

        public StringReader(CharSequence text) {
            this.text = text;
            this.eof = this.text.length();
        }

        private StringReader(StringReader reader) {
            this.text = reader.text;
            this.position = reader.position;
            this.eof = reader.eof;
        }

        public boolean endOfString() {
            return this.position >= this.eof;
        }

        public int read() {
            if (this.position >= this.eof) {
                return -1;
            }
            char c = this.text.charAt(this.position);
            ++this.position;
            return c;
        }

        public int peek() {
            if (this.position >= this.eof) {
                return -1;
            }
            return this.text.charAt(this.position);
        }

        public void skip(int chars) {
            this.position += chars;
            if (chars > 0 && this.position > this.eof) {
                this.position = this.eof;
            }
            if (chars < 0 && this.position < 0) {
                this.position = 0;
            }
        }

        public StringReader copy() {
            return new StringReader(this);
        }
    }
}

