/*
 * Build a media database from filesystem content.
 *
 * Copyright (C) 2009 W. Michael Petullo <mike@flyn.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "db-builder.h"
#include "db-builder-gdir.h"

#include <libdmapsharing/dmap.h>

struct DbBuilderGDirPrivate {
	GSList          *monitor_list;
	DmapDb          *db;
	DmapContainerDb *container_db; // NULL if we don't want directory containers.
};

enum {
	PROP_0,
	PROP_DB,
	PROP_CONTAINER_DB
};

static void
_set_property (GObject *object,
               guint prop_id,
               const GValue *value,
               GParamSpec *pspec)
{
	DbBuilderGDir *builder = DB_BUILDER_GDIR (object);

        switch (prop_id) {
	case PROP_DB:
		if (builder->priv->db) {
			g_object_unref(builder->priv->db);
		}
		builder->priv->db = g_value_dup_object (value);
		break;
	case PROP_CONTAINER_DB:
		if (builder->priv->container_db) {
			g_object_unref(builder->priv->container_db);
		}
		builder->priv->container_db = g_value_dup_object (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	DbBuilderGDir *builder = DB_BUILDER_GDIR (object);

	switch (prop_id) {
	case PROP_DB:
                g_value_set_object (value, builder->priv->db);
                break;
        case PROP_CONTAINER_DB:
                g_value_set_object (value, builder->priv->container_db);
                break;
	default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
	}
}

static gboolean
_build_db_starting_at(DbBuilder *builder,
                      const char *dir,
                      DmapContainerRecord *container_record,
                      GError **error);

static void
_monitor_directory (DbBuilderGDir *builder, const char *path);

static gboolean
_handle_path(DbBuilderGDir *builder,
             const char *path,
             DmapContainerRecord *container_record,
             GError **error)
{
	gboolean ok = FALSE;
	gchar *filename = g_path_get_basename (path);

	if (g_file_test (path, G_FILE_TEST_IS_DIR)) {
		DmapContainerRecord *record = DMAP_CONTAINER_RECORD (g_object_new (TYPE_DMAPD_DMAP_CONTAINER_RECORD,
		                                                                  "name", filename,
		                                                                  "full-db", builder->priv->db,
		                                                                   NULL));
		ok = _build_db_starting_at(DB_BUILDER (builder),
		                           path,
		                           record,
		                           error);
		if (!ok) {
			goto done;
		}

		if (NULL != builder->priv->container_db) {
			if (dmap_container_record_get_entry_count (record) > 0) {
				dmap_container_db_add (builder->priv->container_db, record, NULL);
			} else {
				g_warning ("Container %s is empty, skipping", path);
			}
		}
		g_object_unref (record);

		_monitor_directory (DB_BUILDER_GDIR (builder), path);
	} else {
		gchar *location;
		guint id = 0;

		location = g_filename_to_uri (path, NULL, NULL);
		// FIXME: very expensive for BDB module:
		id = dmap_db_lookup_id_by_location (builder->priv->db, location);
		g_free (location);

		if (! id) {
			id = dmap_db_add_path (builder->priv->db, path, error);
			if (!id) {
				goto done;
			}
			g_debug ("Done processing %s with id. %u (record #%lu).", path, id, dmap_db_count (builder->priv->db));
		} else {
			g_debug ("Done processing (cached) %s with id. %u (record #%lu).", path, id, dmap_db_count (builder->priv->db));
		}

		if (container_record) {
			dmap_container_record_add_entry (container_record, NULL, id, NULL);
		}
	}

done:
	g_free (filename);

	return ok;
}

static void
_file_changed_cb (GFileMonitor *monitor,
                  GFile *file,
                  GFile *other_file,
                  GFileMonitorEvent event_type,
                  DbBuilderGDir *builder)
{
	g_assert (G_IS_FILE_MONITOR (monitor));
	g_assert (G_IS_FILE (file));
	g_assert (NULL != builder);

	// FIXME: Need to lock database?

	gboolean ok;
	GError *error = NULL;

	gchar *path = g_file_get_path (file);
	if (NULL == path) {
		g_warning ("Could not determine path for changed file\n");
		goto done;
	}

	switch (event_type) {
		case G_FILE_MONITOR_EVENT_CREATED:
			g_debug ("New file: %s\n", path);

			ok = _handle_path (builder, path, NULL, &error);
			if (!ok) {
				g_warning("Error handling new file: %s", error->message);
				goto done;
			}

			break;
		case G_FILE_MONITOR_EVENT_DELETED:
			// FIXME: Not implemented, need remove function in database.
			g_debug ("Removed file: %s\n", path);
			g_warning ("Removing file %s not implemented", path);
			break;
		default:
			g_warning ("Unhandled change to %s: %d", path, event_type);
			break;
	}

done:
	if (NULL != path) {
		g_free (path);
	}

	return;
}

static void
_monitor_directory (DbBuilderGDir *builder, const char *path)
{
	g_assert (IS_DB_BUILDER_GDIR (builder));
	g_assert (NULL != path);

	GFile *dir = g_file_new_for_path (path);
	if (NULL == dir) {
		g_warning ("Unable to open %s for monitoring", path);
		goto done;
	}

	GError *error = NULL;
        GFileMonitor *monitor = g_file_monitor_directory (dir, 0, NULL, &error);
        if (NULL == monitor) {
                g_warning ("Unable to monitor %s: %s\n", path, error->message);
		goto done;
        }

	gchar *file = g_file_get_basename (dir);
	g_debug ("Monitoring %s\n", file);
	g_free (file);

        g_signal_connect (monitor, "changed", (GCallback) _file_changed_cb, builder);	

        builder->priv->monitor_list = g_slist_append (builder->priv->monitor_list, monitor);

done:
	if (NULL != dir) {
		g_object_unref (dir);
	}

	return;
}

static gboolean
_build_db_starting_at(DbBuilder *builder,
                      const char *dir,
                      DmapContainerRecord *container_record,
                      GError **error)
{
	g_assert (IS_DB_BUILDER_GDIR (builder));

	GDir *d;
	gboolean ok = TRUE;
	const gchar *entry;

	d = g_dir_open (dir, 0, error);
	if (NULL == d) {
		ok = FALSE;
		goto done;
	}

	while ((entry = g_dir_read_name (d))) {
		gchar *path = g_strdup_printf ("%s/%s", dir, entry);

		ok = _handle_path (DB_BUILDER_GDIR (builder),
		                   path,
		                   container_record,
		                   error);

		g_free (path);

		if (!ok) {
			goto done;
		}
	}

	_monitor_directory (DB_BUILDER_GDIR (builder), dir);

done:
	if (NULL != d) {
		g_dir_close (d);
	}

	return ok;
}

static void
db_builder_gdir_init (DbBuilderGDir *builder)
{
        builder->priv = DB_BUILDER_GDIR_GET_PRIVATE (builder);

	builder->priv->monitor_list = NULL;
	builder->priv->db           = NULL;
	builder->priv->container_db = NULL;
}

static void
db_builder_gdir_class_finalize (DbBuilderGDirClass *class)
{
}

static void
_finalize (GObject *object)
{
	g_debug ("Finalizing DbBuilderGDir");

	DbBuilderGDir *builder = DB_BUILDER_GDIR (object);

	if (NULL != builder->priv->monitor_list) {
		g_slist_free_full (builder->priv->monitor_list, g_object_unref);
	}
}

static void db_builder_gdir_class_init (DbBuilderGDirClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	DbBuilderClass *db_builder_class = DB_BUILDER_CLASS (klass);

        object_class->set_property = _set_property;
        object_class->get_property = _get_property;
        object_class->finalize     = _finalize;

	db_builder_class->build_db_starting_at = _build_db_starting_at;

	g_object_class_install_property (object_class,
                                         PROP_DB,
                                         g_param_spec_object ("db",
                                                              "DB",
                                                              "DB object",
	                                                       DMAP_TYPE_DB,
                                                               G_PARAM_READWRITE));

        g_object_class_install_property (object_class,
                                         PROP_CONTAINER_DB,
                                         g_param_spec_object ("container-db",
                                                              "Container DB",
                                                              "Container DB object",
	                                                       DMAP_TYPE_CONTAINER_DB,
                                                               G_PARAM_READWRITE));

	g_type_class_add_private (klass, sizeof (DbBuilderGDir));
}

static void db_builder_gdir_register_type (GTypeModule *module);

G_MODULE_EXPORT gboolean
dmapd_module_load (GTypeModule *module)
{
        db_builder_gdir_register_type (module);
	return TRUE;
}

G_MODULE_EXPORT void
dmapd_module_unload (GTypeModule *module)
{
}

G_DEFINE_DYNAMIC_TYPE (DbBuilderGDir, db_builder_gdir, TYPE_DB_BUILDER)
