/*
 * desproxy-test.c: test & config tool for desproxy (UNFINISHED)
 *
 * Copyright (C) 2003 Miguelanxo Otero Salgueiro
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the tems of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include "desproxy.h"

#define PROGRAM_NAME "desproxy-test"
#define PROGRAM_VERSION VERSION

#define ADDRESS_NOT_OK 1
#define ADDRESS_OK 2
#define PORT_NOT_OK 3
#define PORT_OK 4
#define GET_OK 5

#define AUTH_NONE 0
#define AUTH_BASIC 1
#define AUTH_DIGEST 2
#define AUTH_NOT_OK 3

int proxy_auth = AUTH_NONE;
int GET_count = 0, CONNECT_count = 0;

/*
 * Function : display_gethostbyname_error(void)
 * Purpose  : display error reported by gethostbyname()
 * Params   : none
 */
void
display_gethostbyname_error (void)
{
  printf (gettext ("\n*** An error ocurred connecting to "));
  printf ("%s: ", proxy_host);
  switch (h_errno)
    {
    case TRY_AGAIN:
      {
	printf (gettext ("temporary error in name resolution\n"));
	break;
      }
    case HOST_NOT_FOUND:
      {
	printf (gettext ("unknown host\n"));
	break;
      }
    default:
      {
	printf (gettext ("non-recoverable name server error\n"));
      }
    }
  printf ("*** Please, double-check the address and try again\n\n");
}

/*
 * Function : void connect_to_proxy(void)
 * Purpose  : connects to proxy proxy_host:proxy_port
 * Params   : none
 * Note     : proxy_hostent must be initialized before:
 *          : proxy_hostent=gethostbyname(proxy_host)
 */
void
connect_to_proxy (void)
{
  memset (&proxy, 0, sizeof proxy);
  proxy.sin_family = AF_INET;
  memcpy (&proxy.sin_addr, proxy_hostent->h_addr, proxy_hostent->h_length);
  proxy.sin_port = htons (atoi (proxy_port));
  if ((proxy_socket[0] = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      printf (gettext ("\n*** An error ocurred connecting to "));
      printf ("(%s:%s)\n", proxy_host, proxy_port);
      printf ("*** socket: %s\n", strerror (errno));
      printf (gettext ("*** FATAL: unrecoverable error."));
      printf (gettext (" Exiting...\n"));
      exit (1);
    }
  if (connect (proxy_socket[0], (struct sockaddr *) &proxy, sizeof proxy) < 0)
    {
      printf (gettext ("\n*** An error ocurred connecting to "));
      printf ("(%s:%s)\n", proxy_host, proxy_port);
      printf ("*** connect: %s\n", strerror (errno));
      printf (gettext ("*** Please, double-check the port number"));
      printf (gettext (" and try again.\n\n"));
      status = PORT_NOT_OK;
      return;
    }
  status = PORT_OK;
}

/*
 * Function : void HTTP_authenticate(char *site)
 * Purpose  : requests user & password & does authentication
 *          : after finding that proxy auth is needed
 * Params   : char *site - sitename (for example "www.sun.com")
 */
void
HTTP_authenticate (char *site)
{
  char HTTP_header[256];
  int count;

  if (proxy_auth == AUTH_NONE)
    {
      printf (gettext ("\n\n*** Error when accessing page:\n"));
      printf (gettext ("*** PROXY AUTHENTICATION REQUIRED\n"));
    }
  else
    {
      /*
       * HTTP_authenticate called with proxy_auth != AUTH_NONE
       * it's only posible if a previous HTTP_authenticate set
       * proxy_auth != AUTH_NONE, so if we need authentication
       * again, that's only because authentication failed...
       */
      printf (gettext ("\n\n*** PROXY AUTHENTICATION FAILED\n"));
      printf (gettext ("*** Double-check username and password"));
      printf (gettext (" and try again\n\n"));
      printf (gettext ("Press [ENTER] to continue\n"));
      get_console_line ();

      /*
       * force new authentification
       */
      proxy_auth = AUTH_NONE;
      return;
    }
  while (memcpy (buffer, "\r\n", 2))
    {
      /*
       * try to find "Proxy-Authenticate" HTTP header
       */
      wait_for_crlf (proxy_socket[0]);
      strncpy (HTTP_header, buffer, 256);
      strtolower (HTTP_header);
      if (!memcmp (HTTP_header, "proxy-authenticate:", 19))
	{
	  /*
	   * ok, found it
	   */
	  printf (gettext ("*** Authentication Method: "));
	  printf ("%s", &buffer[19]);

	  /*
	   * parse header to find authentication method
	   */
	  for (count = 19; buffer[count] == ' '; count++);
	  strcpy (HTTP_header, &buffer[count]);
	  for (count = 0; HTTP_header[count] != ' '; count++);
	  HTTP_header[count] = 0;
	  strtolower (HTTP_header);

	  if (!memcmp (HTTP_header, "basic", 5))
	    {
	      proxy_auth = AUTH_BASIC;
	      break;
	    }
	}
    }
  if (proxy_auth == AUTH_BASIC)
    {
      printf (gettext ("\n\nUsing \"Basic\" Authentication"));
      printf (gettext (" method.\n"));
      printf (gettext ("You must supply an username"));
      printf (gettext (" and a password.\n\n"));
      get_username_and_password ();
      printf ("\n");
      return;
    }
  printf (gettext ("*** Only \"Basic\" authentication method"));
  printf (gettext (" is supported.\n\n"));
  printf (gettext ("*** FATAL: None of the authentication methods"));
  printf (gettext (" is supported at this time.\n"));
  printf (gettext ("*** Exiting...\n"));
  exit (1);
}

/*
 * Function : void test_GET(char *site)
 * Purpose  : does a HTTP GET for testing
 * Params   : char *site - sitename (for example "www.sun.com")
 */
void
test_GET (char *site)
{
  char basic_auth_string[256];

  printf (gettext ("Testing GET method ("));
  printf ("%s) ", site);

  /*
   * this is for ease in screen reading
   */
  fflush (stdout);
  sleep (2);

  /*
   * try to connect to proxy
   */
  connect_to_proxy ();
  if (status != PORT_OK)
    {
      printf (gettext ("\n\n*** Connect failed,"));
      printf (gettext (" when connecting to proxy.\n"));
      printf (gettext ("*** Proxy server may be down."));
      printf (gettext (" Try again later.\n\n"));
      return;
    }

  /*
   * send HTTP GET request header
   */
  strsend (proxy_socket[0], "GET HTTP://");
  strsend (proxy_socket[0], site);
  strsend (proxy_socket[0], "/ HTTP/1.1\r\nHost: ");
  strsend (proxy_socket[0], site);

  /*
   * if proxy requests "basic" authentication, send basic header.
   * At that point, proxy_auth == AUTH_BASIC and user has
   * already been requested to type username and password,
   * and they are stored in global variables username and password
   */
  if (proxy_auth == AUTH_BASIC)
    {
      strcpy (basic_auth_string, "");
      strcat (basic_auth_string, username);
      strcat (basic_auth_string, ":");
      strcat (basic_auth_string, password);

      printf ("%s", basic_auth_string);
      exit (1);
      strsend (proxy_socket[0], "\r\nProxy-Authorization:");
      strsend (proxy_socket[0], " Basic bWlndWVsOllrNjkhOik=");
    }

  /*
   * User-Agent: desproxy "borrowed" the identity of
   * MS Internet Explorer because it's much "quieter" this way
   */
  strsend (proxy_socket[0], "\r\nUser-Agent: ");
  strsend (proxy_socket[0],
	   "Mozilla/4.0 (compatible; MSIE 5.5; Windows 98)\r\n\r\n");

  wait_for_crlf (proxy_socket[0]);
  parse_HTTP_return_code ();

  /*
   * look at HTTP return codes to determine success, fail or other...
   */
  if (!memcmp (HTTP_return_code, "200", 3))
    {
      /*
       * got HTTP 200 OK nice!
       *
       * GET_count++ -> takes acount of sites accessed OK with
       * HTTP GET method
       */
      printf ("OK\n");
      GET_count++;

      close (proxy_socket[0]);
      return;
    }
  if (!memcmp (HTTP_return_code, "302", 3))
    {
      /*
       * got HTTP 302 Redirected nice!
       * it's not 200, but that isn't important
       */
      printf (gettext ("OK (redirected)\n"));
      GET_count++;

      close (proxy_socket[0]);
      return;
    }
  if (!memcmp (HTTP_return_code, "407", 3))
    {
      /*
       * got HTTP 407 Proxy Authentication Required
       *
       * authenticate via HTTP_authenticate and then
       * RECURSIVELY (:P) call test_GET to try to not
       * miss this test.
       */
      HTTP_authenticate (site);
      close (proxy_socket[0]);
      test_GET (site);
      return;
    }
  /*
   * not OK, not failed due to authentication
   * so... display and error for this test
   */
  printf (gettext ("FAILED ("));
  printf ("%s)\n", HTTP_return_code);

  close (proxy_socket[0]);
}

int
main (int argc, char **argv)
{
  /*
   * as the other desproxy tools take those from
   * command line, they are defined as pointers, so
   * we must reserve memomy for them here.
   */
  proxy_host = malloc (256);
  proxy_port = malloc (256);

  initialize_gettext ();
  print_program_version (PROGRAM_NAME, PROGRAM_VERSION);

  printf ("-----------------------------------\n");
  printf (gettext ("This is the desproxy test tool. It\n"));
  printf (gettext ("will configure your desproxy setup,\n"));
  printf (gettext ("and show your proxy features: Type,\n"));
  printf (gettext ("Supported connections...\n"));
  printf ("-----------------------------------\n\n");
  printf (gettext ("Press [ENTER] to continue\n"));
  get_console_line ();

  /*
   * request user for proxy adress
   * 
   * address is stored in proxy_address
   */
  printf (gettext ("What's your proxy address?\n"));
  printf (gettext ("You can give its network name or its IP.\n"));
  printf (gettext ("Just look at the setings in your web browser.\n\n"));
  printf (gettext ("Examples: HTTP_proxy\n"));
  printf (gettext ("          proxy.domain.biz\n"));
  printf (gettext ("          192.0.0.1\n\n"));

  for (status = ADDRESS_NOT_OK; status != ADDRESS_OK;)
    {
      strcpy (proxy_host, "");
      while (!strcmp (proxy_host, ""))
	{
	  printf (gettext ("Proxy address: "));
	  get_console_line ();
	  strcpy (proxy_host, console_line);
	}

      if ((proxy_hostent = gethostbyname (proxy_host)) == NULL)
	display_gethostbyname_error ();
      else
	status = ADDRESS_OK;
    }

  /*
   * request user for proxy port
   *
   * port is stored in proxy_port
   */
  printf (gettext ("\n\n\n\nWhat's your proxy port?\n"));
  printf (gettext ("Look at the setings in your web browser.\n\n"));
  printf (gettext ("Typical values: 80\n"));
  printf (gettext ("                4480\n"));
  printf (gettext ("                8080\n\n"));

  for (status = PORT_NOT_OK; status != PORT_OK;)
    {
      strcpy (proxy_port, "");
      while (!strcmp (proxy_port, ""))
	{
	  printf (gettext ("Proxy port: "));
	  get_console_line ();

	  strcpy (proxy_port, console_line);
	}
      connect_to_proxy ();
    }
  close (proxy_socket[0]);
  printf (gettext ("\nConnection to "));
  printf ("(%s:%s) OK.\n", proxy_host, proxy_port);

  /*
   * proxy address and port are ok, so begin the tests
   */
  printf (gettext ("Now, we are going to test"));
  printf (gettext (" HTTP proxy compatibility.\n\n"));
  printf (gettext ("Press [ENTER] to continue\n"));
  get_console_line ();

  /*
   * first test HTTP GET methods, to see if we can use
   * the proxy
   */
  test_GET ("desproxy.sf.net");
  test_GET ("www.linux.org");
  test_GET ("www.linux.com");
  test_GET ("www.sun.com");
  test_GET ("www.microsoft.com");
  test_GET ("www.hotmail.com");

  /*
   * display HTTP GET test results
   */
  printf (gettext ("\nTotal number of sites accessed OK"));
  printf (gettext (" via GET method: "));
  printf ("%d\n\n", GET_count);

  exit (0);
}
