// This file is part of Deark.
// Copyright (C) 2018 Jason Summers
// See the file COPYING for terms of use.

// Apple II disk image formats, etc.

#include <deark-private.h>
#include <deark-fmtutil.h>
DE_DECLARE_MODULE(de_module_woz);

#define CODE_INFO 0x494e464fU
#define CODE_META 0x4d455441U
#define CODE_TMAP 0x544d4150U
#define CODE_TRKS 0x54524b53U
#define CODE_WRIT 0x57524954U

typedef struct localctx_struct {
	u8 wozver;
} lctx;

static const char *get_woz_disk_type_name(u8 t)
{
	switch(t) {
	case 1: return "5.25";
	case 2: return "3.5";
	}
	return "?";
}

static void do_woz_INFO(deark *c, struct de_iffctx *ictx,
	const struct de_iffchunkctx *chunkctx)
{
	u8 b;
	i64 n;
	i64 pos = chunkctx->dpos;
	lctx *d = ictx->userdata;
	de_ucstring *s = NULL;

	if(chunkctx->dlen<37) return;
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "INFO chunk version: %d", (int)b);
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "disk type: %d (%s)", (int)b, get_woz_disk_type_name(b));
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "write protected: %d", (int)b);
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "synchronized: %d", (int)b);
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "cleaned: %d", (int)b);

	s = ucstring_create(c);
	dbuf_read_to_ucstring(ictx->f, pos, 32, s, 0, DE_ENCODING_UTF8);
	ucstring_strip_trailing_spaces(s);
	de_dbg(c, "creator: \"%s\"", ucstring_getpsz(s));
	pos += 32;

	if(d->wozver<'2') goto done;

	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "disk sides: %d", (int)b);
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "boot sector format: %d", (int)b);
	b = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "optimal bit timing: %d", (int)b);
	n = dbuf_getu16le_p(ictx->f, &pos);
	de_dbg(c, "compatible hardware: %d", (int)n);
	n = dbuf_getu16le_p(ictx->f, &pos);
	de_dbg(c, "required RAM: %dK", (int)n);
	n = dbuf_getu16le_p(ictx->f, &pos);
	de_dbg(c, "largest track: %d blocks", (int)n);

done:
	ucstring_destroy(s);
}

static void do_woz_print_metadata_item(deark *c, de_ucstring *name, de_ucstring *val)
{
	if(name->len==0 && val->len==0) return;
	de_dbg(c, "item: \"%s\" = \"%s\"",
		ucstring_getpsz_d(name),
		ucstring_getpsz_d(val));
}

static void do_woz_META(deark *c, struct de_iffctx *ictx,
	const struct de_iffchunkctx *chunkctx)
{
	i64 k;
	int reading_val;
	de_ucstring *s = NULL;
	de_ucstring *name = NULL;
	de_ucstring *val = NULL;

	// Read the entire metadata string.
	s = ucstring_create(c);
	dbuf_read_to_ucstring_n(ictx->f, chunkctx->dpos, chunkctx->dlen,
		65536, s, 0, DE_ENCODING_UTF8);

	// Parse out the individual metadata items
	name = ucstring_create(c);
	val = ucstring_create(c);
	reading_val = 0;

	for(k=0; k<s->len; k++) {
		i32 ch = s->str[k];

		if(ch==0x0a) { // End of item
			do_woz_print_metadata_item(c, name, val);
			ucstring_empty(name);
			ucstring_empty(val);
			reading_val = 0;
		}
		else if(ch==0x09 && !reading_val) { // Name/value separator
			reading_val = 1;
		}
		else { // A non-special character
			if(reading_val) {
				ucstring_append_char(val, ch);
			}
			else {
				ucstring_append_char(name, ch);
			}
		}
	}
	do_woz_print_metadata_item(c, name, val);

	ucstring_destroy(s);
	ucstring_destroy(name);
	ucstring_destroy(val);
}

static int my_preprocess_woz_chunk_fn(struct de_iffctx *ictx)
{
	const char *name = NULL;

	switch(ictx->chunkctx->chunk4cc.id) {
	case CODE_TMAP: name = "track map"; break;
	case CODE_TRKS: name = "data for tracks"; break;
	case CODE_META: name = "metadata"; break;
	case CODE_WRIT: name = "disk writing instructions"; break;
	}

	if(name) {
		ictx->chunkctx->chunk_name = name;
	}
	return 1;
}

static int my_woz_chunk_handler(struct de_iffctx *ictx)
{
	deark *c = ictx->c;

	switch(ictx->chunkctx->chunk4cc.id) {
	case CODE_INFO:
		ictx->handled = 1;
		do_woz_INFO(c, ictx, ictx->chunkctx);
		break;
	case CODE_META:
		ictx->handled = 1;
		do_woz_META(c, ictx, ictx->chunkctx);
		break;
	}

	return 1;
}

static void de_run_woz(deark *c, de_module_params *mparams)
{
	lctx *d = NULL;
	struct de_iffctx *ictx = NULL;
	u32 crc;
	i64 pos = 0;

	// WOZ has a 12-byte header, then sequence of chunks that are basically the
	// same format as RIFF.
	d = de_malloc(c, sizeof(lctx));
	ictx = fmtutil_create_iff_decoder(c);

	ictx->userdata = (void*)d;
	ictx->preprocess_chunk_fn = my_preprocess_woz_chunk_fn;
	ictx->handle_chunk_fn = my_woz_chunk_handler;
	ictx->f = c->infile;
	ictx->is_le = 1;
	ictx->reversed_4cc = 0;

	if(ictx->f->len<12) goto done;
	de_dbg(c, "header at %d", (int)pos);
	de_dbg_indent(c, 1);
	pos += 3; // "WOZ" part of signature
	d->wozver = dbuf_getbyte_p(ictx->f, &pos);
	de_dbg(c, "format version: '%c'", de_byte_to_printable_char(d->wozver));
	if(d->wozver<'1' || d->wozver>'2') {
		de_err(c, "Unsupported WOZ format version");
		goto done;
	}
	pos += 4; // rest of signature
	crc = (u32)dbuf_getu32le_p(ictx->f, &pos);
	de_dbg(c, "crc: 0x%08x", (unsigned int)crc);
	de_dbg_indent(c, -1);

	fmtutil_read_iff_format(ictx, pos, ictx->f->len-pos);

done:
	fmtutil_destroy_iff_decoder(ictx);
	de_free(c, d);
}

static int de_identify_woz(deark *c)
{
	if(dbuf_memcmp(c->infile, 0, "WOZ", 3))
		return 0;
	if(dbuf_memcmp(c->infile, 4, "\xff\x0a\x0d\x0a", 4))
		return 0;
	return 100;
}

void de_module_woz(deark *c, struct deark_module_info *mi)
{
	mi->id = "woz";
	mi->desc = "WOZ floppy disk image";
	mi->desc2 = "metadata only";
	mi->run_fn = de_run_woz;
	mi->identify_fn = de_identify_woz;
}
