# Copyright (C) 2004 Scott W. Dunlop <sdunlop at users.sourceforge.net>
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from database import Database, DatabaseError
from server import Server
from authenticator import NoneAuthenticator, authentication_schemes
from interface import collate_interfaces

import sys
from types import StringType

class Wiki( object ):
    __slots__ = (
        'server', 'database', 'logFile', 'siteName', 'stylesheetPath',
        'flagCacheStylesheets', 'authenticator', 'baseUrl', 'httpPort',
        'method_interfaces', 'gateway_interfaces', 'lexicon', 'stylesheet'
    )
    
    def __init__( self, path, log=None, port=None, init = False ):
        self.server = None
        self.siteName = None
        self.stylesheetPath = None
        self.flagCacheStylesheets = None
        self.authenticator = None
        self.baseUrl = None
        self.httpPort = port
        self.stylesheet = None

        self.logFile = None

        self.database = Database( wiki = self )
        if not self.database.connect( path, init ):
            raise DatabaseError( "Could not connect to database at '%s'" % (path,) )

        if log is None:
            log = self.database.getConfig( 'log-path' )
        
            if log is None:
                log = path + ".log"

        if isinstance( log, StringType ):
            try:
                self.logFile = open( log, "a" )
            except IOError, exc:
                self.logFile = sys.stderr
                self.logLine( 
                    "Could not access %s for logging. Using stderr.",
                    log
                )
        else:
            self.logFile = log

        self.lexicon = None
        self.gateway_interfaces = None
        self.method_interfaces = None
                    
    def runForever( self ):
        self.lexicon, self.gateway_interfaces, self.method_interfaces = (
            collate_interfaces( self )
        )
        
        self.getServer().runForever( )

    def getServer( self ):
        if self.server is None: self.genServer()
        
        return self.server
                
    def getDatabase( self ):
        return self.database
                
    def genServer( self ):    
        self.server = Server( wiki = self, port = self.getHttpPort() )
        
    def getStylesheetPath( self ):
        if self.stylesheetPath is None: self.genStylesheetPath()
        
        return self.stylesheetPath

    def genStylesheetPath( self ):
        self.stylesheetPath = self.getDatabase().getConfig( 
            "stylesheet-path", 'style.css' 
        )

    def shouldCacheStylesheets( self ):
        if self.flagCacheStylesheets is None: self.checkCacheStylesheets()
        return self.flagCacheStylesheets

    def checkCacheStylesheets( self ):
        self.flagCacheStylesheets = self.getDatabase().getConfig( 
            'cache-stylesheets', 'no'
        ).lower() in [
            'true', 't', 'yes', 'y'
        ]

    def getStylesheet( self ):
        if self.stylesheet is None:
            if self.shouldCacheStylesheets():
                self.stylesheet = "\n".join(
                    open( self.getStylesheetPath(), "r" ).readlines()
                )
            else:
                return "\n".join(
                    open( self.getStylesheetPath(), "r" ).readlines()
                )

        return self.stylesheet

    def getLogFile( self ):
        return self.logFile
        
    def logLine( self, message, *args ):
        self.logData( message + "\n", *args )

    def logData( self, message, *args ):
        lf = self.getLogFile()
        
        if lf is None: return

        if args:
            lf.write( message % args )
        else:
            lf.write( message )

    def getHttpPort( self ):
        if self.httpPort is None: self.genHttpPort()
        return self.httpPort
            
    def genHttpPort( self ):
        port = self.getDatabase().getConfig( 'http-port', None )
        
        if port is not None: 
            self.httpPort = int( port )
        else:
            self.httpPort = 8080

    def getSitename( self ):
        if self.siteName is None: self.genSitename()
        return self.siteName

    def genSitename( self ):
        self.siteName = self.getDatabase().getConfig( 
            "site-title", "AnonyWiki" 
        ) 
    
    def getBaseUrl( self ):
        if self.baseUrl is None:
            self.genBaseUrl()
        return self.baseUrl

    def genBaseUrl( self ):
        self.baseUrl = self.getDatabase().getConfig( "base-url", "" )

    def shouldMentionUsers( self ):
        return self.getAuthenticator().shouldMentionUsers()
    
    def getAuthenticator( self ):
        if self.authenticator is None:
            self.genAuthenticator( )

        return self.authenticator

    def genAuthenticator( self ):
        key = self.getDatabase().getConfig( 'site-auth' )
        scheme = authentication_schemes.get( key )

        if scheme is None:
            scheme = NoneAuthenticator

        self.authenticator = scheme( self )

    def getGatewayFor( self, key ):
        return self.gateway_interfaces.get( key )

    def getMethodFor( self, key ):
        return self.method_interfaces.get( key )

    def getLexicon( self ):
        return self.lexicon

    def getVersion( self ):
        return 1,1

