//
//  JCF Java Class File Manipulation Package (package lti.java.jcf)
//  Copyright (c) 1997 Matt T. Yourst. All rights reserved.
//
//  Version:  1.00.177
//  Date:     January through May 1997
//  Author:   Matt Yourst [yourst@laserstars.com]
//  Language: Java 1.1+
//

package lti.java.jcf;

import java.io.*;
import java.util.*;
import lti.java.jcf.RuntimeConstants;       

/**
 * JcfConstantPool is a dynamic, in-memory class file constant pool.
 * The pool can be read and written, and can be extended to support
 * new constant pool tags.
 */
public class JcfConstantPool extends Vector implements RuntimeConstants
{
  protected JcfClassFile parent;
    public final JcfClassFile getClassFile() { return parent; }

  /** Factory method to read in and add a new CptGeneric object (or subclass)
   * to the pool.
   * @param st The JcfClassInput to read the constant from
   * @param tag the tag value of the constant to be read
   * @returns The number of extra pool slots the constant takes up */
  public int readConstant (JcfClassInput st, byte tag)
    throws IOException, ClassFormatError
  {
    int extra = 0;
    switch (tag)
    {
      case CONSTANT_UTF8:
        addConstant(new CptUTF(st, this, size())); break;
      case CONSTANT_NAMEANDTYPE:
        addConstant(new CptNameType(st, this, size())); break;
      case CONSTANT_CLASS:
        addConstant(new CptClass(st, this, size())); break;
      case CONSTANT_METHOD:
        addConstant(new CptMethod(st, this, size())); break;
      case CONSTANT_FIELD:
        addConstant(new CptField(st, this, size())); break;
      case CONSTANT_STRING:
        addConstant(new CptString(st, this, size())); break;
      case CONSTANT_INTEGER:
        addConstant(new CptInteger(st, this, size())); break;
      case CONSTANT_FLOAT:
        addConstant(new CptFloat(st, this, size())); break;
      case CONSTANT_LONG:
        addConstant(new CptLong(st, this, size())); addElement(null); extra++; break;
      case CONSTANT_DOUBLE:
        addConstant(new CptDouble(st, this, size())); addElement(null); extra++; break;
      case CONSTANT_UNICODE:
        addConstant(new CptUnicode(st, this, size())); break;
      case CONSTANT_INTERFACEMETHOD:
        addConstant(new CptInterface(st, this, size())); break;
      default:
        // in subclasses, make this "super.readConstant(...)"
        throw new ClassFormatError("Unrecognized constant pool tag: " + tag);
    }
    return extra;
  }

  /** Reads the entire constant pool from a JcfClassInput, calling readConstant()
   * to read individual constants.
   * @param ist The JcfClassInput to read the pool from
   * @param p The parent JcfClassFile that owns this constant pool */
  public void read (JcfClassInput ist)
    throws ClassFormatError, IOException
  {
    int constCount = ist.readShort();
    ensureCapacity(constCount);
    addElement(null); // First slot (0) is always unused.
    for (int i = 1; i < constCount; i++)
    {
      byte tag = ist.readByte();
      i += readConstant(ist, tag);
    }
  }

  /** Instantiates the constant pool from a JcfClassInput, calling read()
   * to perform the actual read from the JcfClassInput.
   * @param ist The JcfClassInput to read the pool from
   * @param p The parent JcfClassFile that owns this constant pool */
  public JcfConstantPool (JcfClassInput ist, JcfClassFile p)
    throws ClassFormatError, IOException
  {
    super(256, 256);
    parent = p;
    read(ist);
  }

  /** Creates an empty constant pool.
   * @param p The parent JcfClassFile that owns this constant pool */
  public JcfConstantPool (JcfClassFile p,
    int initialCapacity, int capacityIncrement)
  {
    super(initialCapacity, capacityIncrement);
    parent = p;
    addElement(null); // First slot (0) is always unused.
  }

  /** Instantiates with a default increment of 256. */
  public JcfConstantPool (JcfClassFile p, int initialCapacity)
    { this(p, initialCapacity, 256); }

  /** Instantiates with a default capacity of 256 and increment of 256. */
  public JcfConstantPool (JcfClassFile p)
    { this(p, 256, 256); }

  /** Adds a new constant to the end of the constant pool.
   * @param constant The constant to be added. */
  public void addConstant (CptGeneric constant)
    { addElement(constant); }

  /** Returns a specific constant in the constant pool, by index. */
  public CptGeneric constantAt (int index)
    throws ArrayIndexOutOfBoundsException
  {
    if (index == 0)
      throw new ArrayIndexOutOfBoundsException(0);
    else return (CptGeneric)elementAt(index);
  }

  /** Inserts a constant in the constant pool at a specific index. */
  public void insertConstantAt (CptGeneric constant, int index)
  {
    insertElementAt (constant, index);
  }

  /** Writes out the constant pool to a JcfClassOutput
   * @param ost The JcfClassOutput to write the pool to. */
  public void write (JcfClassOutput ost) throws IOException
  {
    int count = size();
    ost.writeShort(count);
    for (int i = 1; i < count; i++)
    {
      CptGeneric constant = constantAt(i);
      if (constant != null) writeConstant(ost, constant, i);
    }
  }

  /** Writes out a single tag and constant. Override to implement tag
   * altering.
   * @param ost The JcfClassOutput to write to
   * @param constant The CptGeneric constant object to be written
   * @param index The index of the specified constant in the pool */
  public void writeConstant (JcfClassOutput ost, CptGeneric constant, int index)
    throws IOException
  {
    ost.writeByte(constant.getTag());
    constant.write(ost);
  }

  /**
   * @param index the index of a CptUTF
   * @returns actual string at that index
   */
  public final String utfAt(int index)
    { return ((CptUTF)elementAt(index)).value; }

  /**
   * @param index the index of a CptClass
   * @returns name of the class referred to by that CptClass
   */
  public final String classNameAt(int index)
    { return utfAt(((CptClass)elementAt(index)).nameIndex); }
    
  /**
   * @param index the index of a CptString
   * @returns the string of the CptUTF pointed to by the CptString
   */
  public final String stringTextAt(int index)
    { return utfAt(((CptString)elementAt(index)).StringIndex); }

  /**
   * @param index the index of a CptMember
   * @returns the class name of the member
   */
  public final String memberClassNameAt(int index)
    { return classNameAt(((CptMemberOrInterface)elementAt(index)).classIndex); }

  /**
   * @param index the index of a CptMember
   * @returns the name of the member
   */
  public final String memberNameAt(int index)
  {
    return utfAt(((CptNameType)elementAt(((CptMemberOrInterface)elementAt(index)).
      nameTypeIndex)).nameIndex);
  }

  /**
   * @param index the index of a CptMember
   * @returns the signature of the member
   */
  public final String memberSignatureAt(int index)
  {
    return utfAt(((CptNameType)elementAt(((CptMemberOrInterface)elementAt(index)).
      nameTypeIndex)).signatureIndex);
  }
}
