/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.xml;

import ccl.util.FileUtil;
import ccl.util.Util;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

/**
 * This class provides generic helper methods to deal
 * with xml or xslt processing.<p>
 * 
 * Credit: code to escape a string so it can be used as xml
 *         text content or an attribute value is taken from 
 *         the Apache Xerces project.
 *
 * @author    Chr. Clemens Lee (mailto:clemens@kclee.com)
 * @version   $Id: XMLUtil.java,v 1.11 2003/10/18 07:53:26 clemens Exp clemens $
 */
public class XMLUtil
{
    /**
     * Identifies the last printable character in the Unicode range
     * that is supported by the encoding used with this serializer.
     * For 8-bit encodings this will be either 0x7E or 0xFF.
     * For 16-bit encodings this will be 0xFFFF. Characters that are
     * not printable will be escaped using character references.
     */
    static private int _lastPrintable = 0x7E;

    /**
     * Encode special XML characters into the equivalent character references.
     * These five are defined by default for all XML documents.
     * Converts '<', '>', '"'. and '\'' to "lt", "gt", "quot", or "apos".
     */
    static private String getEntityRef( char ch )
    {
        switch ( ch ) 
        {
        case '<':
            return "lt";
        case '>':
            return "gt";
        case '"':
            return "quot";
        case '\'':
            return "apos";
        case '&':
            return "amp";
        }

        return null;
    }

    /**
     * This is an utility class which should never be 
     * instantiated.
     */
    private XMLUtil()
    {
        super();
    }

    /**
     * If there is a suitable entity reference for this
     * character, return it. The list of available entity
     * references is almost but not identical between
     * XML and HTML.
     */
    static public String escape( char ch )
    {
        String charRef;

        charRef = getEntityRef( ch );
        if ( charRef != null ) 
        {
            return "&" + charRef + ";";
        }
        else if ( ( ch >= ' ' && ch <= _lastPrintable && ch != 0xF7 ) ||
                  ch == '\n' || ch == '\r' || ch == '\t' ) 
        {
            // If the character is not printable, print as character reference.
            // Non printables are below ASCII space but not tab or line
            // terminator, ASCII delete, or above a certain Unicode threshold.
            return "" + ch;
        }
        else 
        {
            return "&#"
                   + Integer.toString( ch )
                   + ";";
        }
    }

    /**
     * Escapes a string so it may be returned as text content or attribute
     * value. Non printable characters are escaped using character references.
     * Where the format specifies a deault entity reference, that reference
     * is used (e.g. <tt>&amp;lt;</tt>).
     *
     * @param   source   the string to escape or "" for null.
     */
    static public String escape( String source )
    {
        if ( source == null ) 
        {
            return "";
        }

        StringBuffer pBuffer = new StringBuffer();

        for ( int i = 0 ; i < source.length() ; ++i ) 
        {
            pBuffer.append( escape( source.charAt( i ) ) );
        }

        return pBuffer.toString();
    }

    /**
     * Transforms a given xml content with a given xslt stylesheet file.
     *
     * @exception   IOException            if the xml file can't be red.
     * @exception   TransformerException   if parsing the xml or xslt data fails.
     */
    static public String getXML( File xmlFile
                                 , String xsltFile )
        throws IOException
               , TransformerException
    {
        try
        {
            return getXML( FileUtil.readFile( xmlFile.getPath() ), xsltFile );
        }
        catch( TransformerException transformerException )
        {
            //throw transformerException;
            throw new TransformerException( "XML file '" + xmlFile + "': " + Util.getStackTrace( transformerException ) );
        }
    }

    /**
     * Transforms a given xml content with a given xslt stylesheet file.
     *
     * @exception   IOException            if the xml or xslt file can't be red.
     * @exception   TransformerException   if parsing the xml or xslt data fails.
     */
    static public String getXML( File xmlFile
                                 , File xsltFile )
        throws IOException
               , TransformerException
    {
        try
        {
            return getXML( FileUtil.readFile( xmlFile.getPath() ), FileUtil.readFile( xsltFile.getPath() ) );
        }
        catch( TransformerException transformerException )
        {
            throw new TransformerException( "XML file '" + xmlFile + "' or XSLT file '" + xsltFile + "': " + Util.getStackTrace( transformerException ) );
        }
    }

    /**
     * Transforms a given xml content with a given xslt stylesheet file.
     *
     * @exception   IOException            if the xslt file can't be red.
     * @exception   TransformerException   if parsing the xml or xslt data fails.
     */
    static public String getXML( String xmlContent
                                 , File xsltFile )
        throws IOException
               , TransformerException
    {
        try
        {
            return getXML( xmlContent, FileUtil.readFile( xsltFile.getPath() ) );
        }
        catch( TransformerException transformerException )
        {
            throw new TransformerException( "XSLT file '" + xsltFile + "': " + Util.getStackTrace( transformerException ) );
        }
    }

    /**
     * Transforms a given xml content using the given xslt stylesheet content.
     *
     * @exception   TransformerException   if parsing the xml or xslt data fails.
     */
    static public String getXML( String xmlContent
                                 , String xsltContent )
        throws TransformerException
    {
        StreamSource xmlSource = new StreamSource( new StringReader( xmlContent ) );
        StreamSource styleSource = new StreamSource( new StringReader( xsltContent ) );

        TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer transformer = transformerFactory.newTransformer( styleSource );

        ByteArrayOutputStream output = new ByteArrayOutputStream( 1024 );
        StreamResult result = new StreamResult( output );

        transformer.transform( xmlSource, result );

        String retVal = output.toString();
        // Closing a byte array output stream has no effect anyway, so we avoid
        // handling the io exception which would never come anyway.
        // output.close();

        return retVal;
    }

    /* *
     * Transforms a given xml content with a given xslt stylesheet file
     * into the specified output file.
     *
     * @exception   Exception   if anything goes wrong.
     */
    /*
      static public void transformXML( String xmlContent
      , String xsltContent
      , String outputFile )
      throws Exception
      {
      XSLTResultTarget result       = new XSLTResultTarget( outputFile );

      transformXML( xmlContent
      , xsltContent
      , result   );
      }
      // */

    /* *
     * Transforms a given xml content with a given xslt stylesheet file
     * into the given XSLTResultTarget object.
     *
     * @exception   Exception   if anything goes wrong.
     */
    /*
      static public void transformXML( String xmlContent
      , File xsltFile
      , XSLTResultTarget result )
      throws Exception
      {
      InputStream     isStylesheet = new FileInputStream( xsltFile.getPath() );

      XSLTInputSource  inputXSL     = new XSLTInputSource( isStylesheet );
      XSLTProcessor    processor    = XSLTProcessorFactory.getProcessor();
      StylesheetRoot   stylesheet   = processor.processStylesheet( inputXSL );

      XSLTInputSource  inputXML     = new XSLTInputSource( new StringReader( xmlContent ) );

      stylesheet.process( inputXML, result );

      isStylesheet.close();
      }
      // */

    /* *
     * Transforms a given xml content with the given xslt stylesheet content
     * into the given XSLTResultTarget object.
     *
     * @exception   Exception   if anything goes wrong.
     */
    /*
      static public void transformXML( String xmlContent
      , String xsltContent
      , XSLTResultTarget result )
      throws Exception
      {
      XSLTInputSource  inputXSL     = new XSLTInputSource( new StringReader( xsltContent ) );
      XSLTProcessor    processor    = XSLTProcessorFactory.getProcessor();
      StylesheetRoot   stylesheet   = processor.processStylesheet( inputXSL );

      XSLTInputSource  inputXML     = new XSLTInputSource( new StringReader( xmlContent ) );

      stylesheet.process( inputXML, result );
      }
      // */

    /**
     * Return the xml value in the given xml string described
     * by the XPath value.
     *                        
     * @param       xmlContent   a string with xml data.
     * @param       xpath        an XPath string describing the 
     *                           location of an xml element.
     *
     * @exception   Exception    if any parsing error happens.
     *                        
     * @return                   the xml value specified by the 
     *                           XPath parameter.
     */
    static public String getValueOf( String xmlContent
                                     , String xpath    )
        throws Exception
    {
        String stylesheet = "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n" +
               "<xsl:stylesheet\n" +
               "   version=\"1.0\"\n" +
               "   xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\">\n" +
               "<xsl:output method=\"text\"/>\n" +
               "\n" +
               "<xsl:template match=\"/\"><xsl:value-of select=\"" + xpath + "\"/></xsl:template>\n" +
               "\n" +
               "</xsl:stylesheet>\n";

        return getXML( xmlContent, stylesheet );
    }

    /**
     * Return the xml value in the given xml file described
     * by the XPath value.
     *                        
     * @param       xmlFile      a file with xml data.
     * @param       xpath        an XPath string describing the 
     *                           location of an xml element.
     *
     * @exception   Exception    if any parsing error happens.
     *                        
     * @return                   the xml value specified by the 
     *                           XPath parameter.
     */
    static public String getValueOf( File xmlFile
                                     , String xpath    )
        throws Exception
    {
        String stylesheet = "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n" +
               "<xsl:stylesheet\n" +
               "   version=\"1.0\"\n" +
               "   xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\">\n" +
               "<xsl:output method=\"text\"/>\n" +
               "\n" +
               "<xsl:template match=\"/\"><xsl:value-of select=\"" + xpath + "\"/></xsl:template>\n" +
               "\n" +
               "</xsl:stylesheet>\n";

        return getXML( xmlFile, stylesheet );
    }
}
