/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util.test;

import ccl.util.FileUtil;
import ccl.util.Init;
import ccl.util.Test;
import ccl.util.Util;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;

/**
 * Test cases for ccl.util.Util class.
 * 
 * @version $Id: UtilTest.java,v 1.32 2003/05/01 16:44:55 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class UtilTest extends Test 
{
    private Init _init = null;

    /**
     * This method tests the Util.formatBlock method.
     */
    private void _checkFormatCenter()
    {
        _enterSubTest( "text center formatting" );

        //Util.setDebug( true );

        String sInput = "hallo";
        String sExpectedResult = "hallo\n";
        int lineLength = 3;
        Assert( Util.formatCenter( sInput, lineLength ).equals( sExpectedResult )
                , "Formatting should be '" 
                + sExpectedResult 
                + "' but is: '" 
                + Util.formatCenter( sInput, lineLength )
                + "'" );

        sExpectedResult = " hallo\n";
        lineLength = 7;
        Assert( Util.centerLine( sInput, lineLength ).equals( sExpectedResult.substring( 0, sExpectedResult.length() - 1 ) ) 
                , "Formatting should be '" 
                + sExpectedResult 
                + "' but is: '" 
                + Util.formatCenter( sInput, lineLength )
                + "'" );
        Assert( Util.formatCenter( sInput, lineLength ).equals( sExpectedResult ) );

        sExpectedResult = " hallo\n";
        lineLength = 8;
        Assert( Util.formatCenter( sInput, lineLength ).equals( sExpectedResult ) );

        sExpectedResult = "hallo\n";
        lineLength = 6;
        Assert( Util.formatCenter( sInput, lineLength ).equals( sExpectedResult ) );

        sInput = "hal  lo";
        sExpectedResult = "hal lo\n";
        lineLength = 7;
        Assert( Util.formatCenter( sInput, lineLength ).equals( sExpectedResult ) );

        _exitSubTest();
    }

    /**
     * This method tests the Util.formatBlock method.
     */
    private void _checkFormatBlock()
    {
        _enterSubTest( "text block formatting" );

        //Util.setDebug( true );

        String sInput = "hallo";
        String sExpectedResult = "hallo\n";
        Assert( Util.formatBlock( sInput ).equals( sExpectedResult ) );

        int lineLength = 20;
        sInput = "1234 6789 1234 6789 1234 6789";
        sExpectedResult = "1234 6789 1234 6789\n1234 6789\n";
        String sResult = Util.formatLeft( sInput, lineLength );
        Assert( Util.formatLeft( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        sInput = "1234 6789 1234 6789_ 1234 6789";
        sExpectedResult = "1234 6789 1234 6789_\n1234 6789\n";
        sResult = Util.formatLeft( sInput, lineLength );
        Assert( Util.formatLeft( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        sInput = "1234 6789 1234 6789_1234 6789 67890";
        sExpectedResult = "1234 6789 1234\n6789_1234 6789 67890\n";
        sResult = Util.formatLeft( sInput, lineLength );
        Assert( Util.formatLeft( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        sInput = "1234 6789 1234 6789_1234 6789 67890_";
        sExpectedResult = "1234 6789 1234\n6789_1234 6789\n67890_\n";
        sResult = Util.formatLeft( sInput, lineLength );
        Assert( Util.formatLeft( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        sInput = "";
        sExpectedResult = "\n";
        sResult = Util.formatLeft( sInput, lineLength );
        Assert( Util.formatLeft( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        lineLength = 72;
        sInput = "Contract work as a Java programmer in Zurich or tele-commuting via Internet.";
        sExpectedResult = "Contract work as a Java  programmer  in  Zurich  or  tele-commuting  via\nInternet.\n";
        sResult = Util.formatBlock( sInput, lineLength );
        Assert( Util.formatBlock( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        lineLength = 64;
        sInput = "I  am  familiar  with  object  oriented  methodologies  (Coad,\n"
               + "Jacobson, Rumbaugh, Booch, UML) and tools (Rational Rose).";
        sExpectedResult = "I  am  familiar  with  object  oriented   methodologies   (Coad,\n"
               + "Jacobson, Rumbaugh, Booch, UML) and tools (Rational Rose).\n";
        sResult = Util.formatBlock( sInput, lineLength );
        Assert( Util.formatBlock( sInput, lineLength ).equals( sExpectedResult )
                , "Result: '" + sResult + "'" );

        _exitSubTest();
    }

    /**
     * This is the default constructor.
     */
    public UtilTest()
    {
        super();
    }

    /**
     * This constructor is needed for this test to be a
     * sub test.
     */
    public UtilTest(Test pTest_) 
    {
        super(pTest_);
    }

    /**
     * @throws   Exception   Whatever can go wrong.
     */
    protected void _doIt() 
        throws Exception
    {
        String sLocalPath = FileUtil.getClassPath(this);

        testGetNextDay     ();
        testGetDiffPosition();
        _checkFormatCenter ();
        _checkFormatBlock  ();

        // Nr. 6    fRnd
        boolean bRndBug = false;
        for(int anz = 0; anz < 100; anz++) 
        {
            float fR = Util.rnd((float) 5.0);
            if (fR <= 0.0 || 5.0 <= fR) 
            {
                bRndBug = true;
            }
        }
        bugIf(bRndBug, "Fehler in Util.fRnd");
        // Nr. 7    fRnd double
        bRndBug = false;
        for(int anz = 0; anz < 100; anz++) 
        {
            float fR = Util.rnd((float) 5.0);
            if (fR <= 0.0 || 5.0 <= fR) 
            {
                bRndBug = true;
            }
        }
        bugIf(bRndBug, "Fehler in Util.fRnd double");
        
        // Nr. 8    firstCharToUpperCase
        bugIf(!Util.firstCharToUpperCase("jamacs").equals("Jamacs"));
        bugIf(!Util.firstCharToUpperCase("Jamacs").equals("Jamacs"));
        bugIf(!Util.firstCharToUpperCase((String) null).equals(""));
        bugIf(!Util.firstCharToUpperCase(new String()).equals(""));
        bugIf(!Util.firstCharToUpperCase(new String()).equals(new String()));
        bugIf(!Util.firstCharToUpperCase("++*").equals("++*"));
        
        // Nr. 14
        boolean bZero = false;
        boolean bOne = false;
        for(int i = 0; i < 20; i++) 
        {
            int r = Util.rnd(1);
            if (r == 1) 
            {
                bOne = true;
            }
            else if (r == 0) 
            {
                bZero = true;
            }
            else 
            {
                bugIf(true, "Andere Zahl als 0 oder 1");
            }
        }
        bugIf(bZero == false || bOne == false);
        
        // Nr. 15
        // replace
        String sTest31 = "eh Allo\t du Arsch!\t\tWie geht's?\n";
        String sErg31 = Util.replace(sTest31, "\t", "   ");
        bugIf(!sErg31.equals("eh Allo    du Arsch!      Wie geht's?\n"), sErg31);
        sErg31 = Util.replace(sTest31, "eh", "boh");
        bugIf(!sErg31.equals("boh Allo\t du Arsch!\t\tWie gboht's?\n"));
        
        // Nr. 17
        // Allgemeiner Test von getClass eigenschaften bei Inheritance
        Object o = (Object) (new String("Hello World!"));
        Util.debug("Test: Nr. 33: o.getClassName: " + o.getClass().getName());
        bugIf(!o.getClass().getName().equals("java.lang.String"));
        
        // Nr. 18
        String sTest18 = "-g   -deprecation";
        sTest18 = Util.replace( sTest18, "  ", " " );
        bugIf( !sTest18.equals( "-g  -deprecation" ) );

        String sReplaceString = "/home/clemens/bunin/jacob/RCS/Jacob.java";
        String sReplaceResult = "\\home\\clemens\\bunin\\jacob\\RCS\\Jacob.java";
        bugIf( !Util.replace( sReplaceString, "/", "\\" ).equals( sReplaceResult ) );

        InputStream inputStream = FileUtil.getResourceAsStream
               ( this, "sWindowsMakefile.txt" );
        sReplaceString = FileUtil.readStream( inputStream );
        sReplaceResult = Util.replace( sReplaceString, "/", "\\" );
        bugIf( sReplaceString.equals( sReplaceResult ) );
        bugIf( sReplaceString.length() != sReplaceResult.length() );

        _checkAtoi();

        _checkIntAndBytes();

        _checkDateMethods();

        bugIf( !Util.isDigit( '0' ) );
        bugIf( !Util.isDigit( '1' ) );
        bugIf( !Util.isDigit( '8' ) );
        bugIf( !Util.isDigit( '9' ) );
        bugIf( Util.isDigit( 'O' ) );
        bugIf( Util.isDigit( 'A' ) );

        _checkDateValidation();

        // rightPaddWithSpace test
        {
            bugIf( Util.rightPaddWithSpace( "123", 12 ).equals( "123000000000" ) );
            bugIf( !Util.rightPaddWithSpace( "123", 12 ).equals( "123         " ) );
        }

        _checkNumberOfDays();

        // shrinkString test
        String sShrinked = Util.shrinkString( "/home/clemens/src/java/ccl/src/ccl/util/test/UtilTest.java", 20 );
        Assert( sShrinked.equals( "/home/cl[...]st.java" ), sShrinked );
        sShrinked = Util.shrinkString( "/home/clemens/src/java/ccl/src/ccl/util/test/UtilTest.java", 21 );
        Assert( sShrinked.equals( "/home/cl[...]est.java" ), sShrinked );
        sShrinked = Util.shrinkString( "/home/clemens/src/java/ccl/src/ccl/util/test/UtilTest.java", 19 );
        Assert( sShrinked.equals( "/home/c[...]st.java" ), sShrinked );
    }

    /**
     * Tests Util.atoi(..).
     */
    private void _checkAtoi()
    {
        _enterSubTest( "atoi" );

        // null in   0 out
        bugIf( Util.atoi( null ) != 0 );
        // "" in     0 out
        bugIf( Util.atoi( "" ) != 0 );
            
        bugIf( Util.atoi( " " ) != 0 );
        bugIf( Util.atoi( "-+\"$324" ) != 0 );
        bugIf( Util.atoi( "-1" ) != -1 );
        bugIf( Util.atoi( "- 1" ) != 0 );
        bugIf( Util.atoi( "   -27   " ) != -27 );
        bugIf( Util.atoi( "   -27343l   " ) != -27343 );

        // MAX_INT in   MAX_INT out
        bugIf( Util.atoi( "2147483646" ) != (Integer.MAX_VALUE - 1) );
        bugIf( Util.atoi( "2147483647" ) != Integer.MAX_VALUE );
        bugIf( Util.atoi( "2147483648" ) != 2147483640, "" + Util.atoi( "2147483648" ) );
        bugIf( Util.atoi( "-2147483648" ) != Integer.MIN_VALUE );
        bugIf( Util.atoi( "-2147483647" ) != (Integer.MIN_VALUE + 1) );
        bugIf( Util.atoi( "-2147483649" ) != -2147483640, "" + Util.atoi( "-2147483649" ) );
        
        // number + alpha in   number out
        bugIf( Util.atoi( "234234wjsldfkj" ) != 234234 );
        // alpha + number in   0 out
        bugIf( Util.atoi( "aaaaasdf34234" ) != 0 );
        // hex in   
        bugIf( Util.atoi( "0x1a" ) != 0 );
        
        _exitSubTest();
    }

    /**
     * Tests intToBytes and bytesToInt.
     */
    private void _checkIntAndBytes()
    {
        _enterSubTest( "Int/Bytes Conversion" );

        byte[] ab = Util.intToBytes( Integer.MAX_VALUE );
        bugIf( ab[ 0 ] != 0x7f, "1. ab[ 0 ] should be 0x7f but is: " + ab[ 0 ] );
        bugIf( ab[ 1 ] != -1, "1. ab[ 1 ] should be -1 but is: " + ab[ 1 ] );
        bugIf( ab[ 2 ] != -1, "1. ab[ 2 ] should be -1 but is: " + ab[ 2 ] );
        bugIf( ab[ 3 ] != -1, "1. ab[ 3 ] should be -1 but is: " + ab[ 3 ] );
        ab = Util.intToBytes( Integer.MIN_VALUE );
        bugIf( ab[ 0 ] != -128, "2. ab[ 0 ] should be -128 but is: " + ab[ 0 ] );
        bugIf( ab[ 1 ] != 0, "2. ab[ 1 ] should be 0 but is: " + ab[ 1 ] );
        bugIf( ab[ 2 ] != 0, "2. ab[ 2 ] should be 0 but is: " + ab[ 2 ] );
        bugIf( ab[ 3 ] != 0, "2. ab[ 3 ] should be 0 but is: " + ab[ 3 ] );
        ab = Util.intToBytes( Integer.MIN_VALUE + 1 );
        bugIf( ab[ 0 ] != -128, "3. ab[ 0 ] should be -128 but is: " + ab[ 0 ] );
        bugIf( ab[ 1 ] != 0, "3. ab[ 1 ] should be 0 but is: " + ab[ 1 ] );
        bugIf( ab[ 2 ] != 0, "3. ab[ 2 ] should be 0 but is: " + ab[ 2 ] );
        bugIf( ab[ 3 ] != 1, "3. ab[ 3 ] should be 1 but is: " + ab[ 3 ] );
        /*Util.println( "" + (((Integer.MIN_VALUE + 1) >> 24) & 255)  );*/
        bugIf( (Integer.MIN_VALUE + 1) != 0x80000001, "Peng!" );
        bugIf( Util.bytesToInt( ab ) != (Integer.MIN_VALUE + 1) );
        ab[ 0 ] = -128;
        ab[ 1 ] = 0;
        ab[ 2 ] = 1;
        ab[ 3 ] = 3;
        bugIf( Util.bytesToInt( ab ) != (Integer.MIN_VALUE + 256 + 3) );
        ab = Util.intToBytes( 0 );
        bugIf( ab[ 0 ] != 0, "4." );
        bugIf( ab[ 1 ] != 0, "4." );
        bugIf( ab[ 2 ] != 0, "4." );
        bugIf( ab[ 3 ] != 0, "4." );
        ab = Util.intToBytes( 1 );
        bugIf( ab[ 0 ] != 0, "5." );
        bugIf( ab[ 1 ] != 0, "5." );
        bugIf( ab[ 2 ] != 0, "5." );
        bugIf( ab[ 3 ] != 1, "5." );
        ab = Util.intToBytes( 255 );
        bugIf( ab[ 0 ] != 0, "6. ab[ 0 ]: " + ab[ 0 ] );
        bugIf( ab[ 1 ] != 0, "6." );
        bugIf( ab[ 2 ] != 0, "6." );
        bugIf( ab[ 3 ] != -1, "6. ab[ 3 ]: " + ab[ 3 ] );
        ab = Util.intToBytes( -1 );
        bugIf( ab[ 0 ] != -1, "7. ab[ 0 ]: " + ab[ 0 ] );
        bugIf( ab[ 1 ] != -1, "7." );
        bugIf( ab[ 2 ] != -1, "7." );
        bugIf( ab[ 3 ] != -1, "7. ab[ 3 ]: " + ab[ 3 ] );
        ab = Util.intToBytes( 0x01020304 );
        bugIf( ab[ 0 ] != 1, "8. ab[ 0 ] should be 1 but is: " + ab[ 0 ] );
        bugIf( ab[ 1 ] != 2, "8. ab[ 1 ] should be 2 but is: " + ab[ 1 ] );
        bugIf( ab[ 2 ] != 3, "8. ab[ 2 ] should be 3 but is: " + ab[ 2 ] );
        bugIf( ab[ 3 ] != 4, "8. ab[ 3 ] should be 4 but is: " + ab[ 3 ] );
        bugIf( Util.bytesToInt( Util.intToBytes( -1 ) ) != -1 );
        bugIf( Util.bytesToInt( Util.intToBytes( -543542453 ) ) != -543542453 );

        _exitSubTest();
    }

    /**
     * Tests date methods.
     */
    private void _checkDateMethods()
    {
        //Util.setDebug( true );
        Calendar pCalendar = Util.getCalendar();
        String sTestTime = "00:00:00";
        Util.setTime( pCalendar, sTestTime );
        String sResultTime = Util.getTime( pCalendar );
        bugIf( !sResultTime.equals( sTestTime ),
               "Time is not " + sTestTime + " but " +
               sResultTime + "!" );
        
        pCalendar = Util.getCalendar();
        sTestTime = "00:00:01";
        Util.setTime( pCalendar, sTestTime );
        sResultTime = Util.getTime( pCalendar );
        bugIf( !sResultTime.equals( sTestTime ),
               "Time is not " + sTestTime + " but " +
               sResultTime + "!" );
        
        pCalendar = Util.getCalendar();
        sTestTime = "23:59:59";
        Util.setTime( pCalendar, sTestTime );
        sResultTime = Util.getTime( pCalendar );
        bugIf( !sResultTime.equals( sTestTime ),
               "Time is not " + sTestTime + " but " +
               sResultTime + "!" );
        
        pCalendar = Util.getDate( "1999-11-20" );
        sTestTime = "15:22:38";
        Util.setTime( pCalendar, sTestTime );
        sResultTime = Util.getTime( pCalendar );
        bugIf( !sResultTime.equals( sTestTime ),
               "Time is not " + sTestTime + " but " +
               sResultTime + "!" );
        bugIf( !Util.getDate( pCalendar ).equals( "1999-11-20" ),
               "Date changed!" );
        
        // millis should be 0
        long lMillis = pCalendar.getTime().getTime() % 
               1000L;
        bugIf( lMillis != 0, "Millis are not 0 but " +
               lMillis );
        
        pCalendar = Util.getDate( "1996-12-09" );
        bugIf( !Util.getDate( pCalendar ).equals( "1996-12-09" ),
               "Date conversion bug!" );
        
        // test getLastDayOfMonth(..)
        bugIf( !Util.getLastDayOfMonth( "2000-05-01" )
               .equals( "2000-05-31" ) );
        bugIf( !Util.getLastDayOfMonth( "2000-06-03" )
               .equals( "2000-06-30" ) );
        bugIf( !Util.getLastDayOfMonth( "2000-07-06" )
               .equals( "2000-07-31" ) );
    }

    /**
     * Test getNumberOfDays(..).
     */
    private void _checkNumberOfDays()
    {
        bugIf( Util.getNumberOfDays( "2000-06-13",
                                     "2000-06-14" ) != 2,
               "Number of days between 2000-06-13 and 2000-06-14 is falsely "
               + Util.getNumberOfDays( "2000-06-13",
                                       "2000-06-14" ) );
        bugIf( Util.getNumberOfDays( "2000-01-10",
                                     "2000-01-21" ) != 12,
               "Number of days between 2000-01-10 and 2000-01-21 is falsely "
               + Util.getNumberOfDays( "2000-01-10",
                                       "2000-01-21" ) );
    }

    /**
     * Tests date validation.
     */
    private void _checkDateValidation()
    {
        _enterSubTest( "Date Validation" );

        // stringToDate white box test
        // offset used for '-' char
        String sDate = "20000525";
        int offset = 1;
        if ( sDate.length() == 8 ) 
        {
            offset = 0;
        }
        bugIf( offset != 0,
               "Assuming wrong date length." );
        
        Date dtRetVal = new Date();
        try 
        {
            String sYear = sDate.substring( 0, 4 );
            bugIf( !sYear.equals( "2000" ) );
            String sMonth = sDate.substring( 4 + offset, 
                                             6 + offset );
            bugIf( !sMonth.equals( "05" ) );
            String sDay = sDate.substring( 6 + 2 * offset,
                                           8 + 2 * offset );
            bugIf( !sDay.equals( "25" ) );
            int year = Util.atoi( sYear );
            bugIf( year != 2000 );
            int month = Util.atoi( sMonth ) - 1;
            bugIf( month != 4 );
            int day = Util.atoi( sDay );
            bugIf( day != 25 );
            bugIf( false );
        }
        catch( Exception pException ) 
        {
            //dtRetVal = null;
            bugIf( true );
        }

        bugIf( !Util.isDateValid( "2000-05-25" ),
               "Date 2000-05-25 should be valid but it isn't." );
        bugIf( Util.stringToDate( "20000525" ) == null,
               "Date string \"20000525\" could not be converted to a date." );
        Date dateShort  = Util.stringToDate( "20000525"   );
        Date dateNormal = Util.stringToDate( "2000-05-25" );
        bugIf ( !Util.getStandardDate( dateShort ).equals
                ( Util.getStandardDate( dateNormal ) )
                , "Short and normal dates differ!" );
        bugIf ( !Util.getStandardDate( dateShort ).equals
                ( "2000-05-25" )
                , "Short and normal dates differ!" );
        
        bugIf( sDate == null 
               || (sDate.length() != 8
                   && sDate.length() != 10) );
        
        Date pDate = Util.stringToDate( sDate );
        bugIf( !Util.getStandardDate( pDate ).equals
               ( "2000-05-25" ) );
        Date dtValidation = new Date( pDate.getTime() );
        bugIf( !Util.getStandardDate( dtValidation ).equals
               ( "2000-05-25" ) );
        String sValidation = Util.getStandardDate( dtValidation );
        bugIf( !sValidation.equals( "2000-05-25" ),
               "sValidation: --->" 
               + sValidation 
               + "<---" );
        
        bugIf( !Util.isDateValid( "20000525" ),
               "Date 20000525 should be valid but it isn't." );
        bugIf( Util.isDateValid( "20000231" ),
               "Date 20000231 should not be valid but it is." );

        _exitSubTest();
    }

    /**
     * Checks Util.getDiffPosition(..).
     */
    public void testGetDiffPosition()
    {
        _enterSubTest( "getDiffPosition" );

        try
        {
            int position = Util.getDiffPosition( null, null );
            bugIf( true, "Should not reach this point" );
        }
        catch( IllegalArgumentException argumentException )
        {
            Assert( true, "Should reach this point" );
        }

        try
        {
            int position = Util.getDiffPosition( "something", null );
            bugIf( true, "Should not reach this point" );
        }
        catch( IllegalArgumentException argumentException )
        {
            Assert( true, "Should reach this point" );
        }

        try
        {
            int position = Util.getDiffPosition( null, "happy day" );
            bugIf( true, "Should not reach this point" );
        }
        catch( IllegalArgumentException argumentException )
        {
            Assert( true, "Should reach this point" );
        }

        int position = Util.getDiffPosition( "something", "happy day" );
        Assert( position == 0 );

        position = Util.getDiffPosition( "something", "Shappy day" );
        Assert( position == 0 );

        position = Util.getDiffPosition( "something", "" );
        Assert( position == 0 );

        position = Util.getDiffPosition( "", "Shappy day" );
        Assert( position == 0 );

        position = Util.getDiffPosition( "something", "someShappy day" );
        Assert( position == 4 );

        position = Util.getDiffPosition( "something", "something happy day" );
        Assert( position == 9 );

        _exitSubTest ();
    }

    /**
     * Tests 'getNextDay' method.
     */
    public void testGetNextDay()
    {
        Assert( Util.getNextDay( "2002-11-03" ).equals( "2002-11-04" ) );
        Assert( Util.getNextDay( "2002-10-30" ).equals( "2002-10-31" ) );
        Assert( Util.getNextDay( "2002-10-31" ).equals( "2002-11-01" ) );
    }

    /**
     * The InitTest uses the ccl init object itself. The
     * init object is forwarded from outer test to the
     * next test.
     */
    public void setInit( Init init_ ) 
    {
        _init = init_;
    }

    /**
     * The main method of this class in case you want to execute the tests of it
     * stand alone without the other test classes.
     *
     * @param   args   string array with command line arguments are totally unused.
     */
    public static void main(String[] args) 
    {
        Test pTest = (Test) (new UtilTest());
        pTest.setVerbose( true );
        pTest.setTiming( true );
        pTest.run();
        pTest.printResult();

        System.exit( 0 );
    }
}
