#!/usr/bin/env python3
#
# Copyright (c) 2021-2023 Paul Mattes.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the names of Paul Mattes nor the names of his contributors
#       may be used to endorse or promote products derived from this software
#       without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY PAUL MATTES "AS IS" AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL PAUL MATTES BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
# OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# c3270 ibm_hosts tests

import os
import pty
import requests
from subprocess import Popen, PIPE, DEVNULL
import sys
import tempfile
import unittest
import Common.Test.playback as playback
import Common.Test.cti as cti

@unittest.skipIf(sys.platform.startswith('win'), "Windows uses different c3270 graphic tests")
class TestC3270IbmHosts(cti.cti):

    # c3270 ibm_hosts case sensitivity test
    def test_c3270_ibm_hosts_ci(self):

        # Start 'playback' to read c3270's output.
        playback_port, ts = cti.unused_port()
        with playback.playback(self, 's3270/Test/ibmlink.trc', port=playback_port) as p:
            ts.close()

            # Create an ibm_hosts file that points to playback.
            (handle, hostsfile_name) = tempfile.mkstemp()
            os.close(handle)
            with open(hostsfile_name, 'w') as f:
                f.write(f"fooey primary a:c:t:127.0.0.1:{playback_port}\n")

            # Fork a child process with a PTY between this process and it.
            c3270_port, ts = cti.unused_port()
            os.environ['TERM'] = 'xterm-256color'
            (pid, fd) = pty.fork()
            if pid == 0:
                # Child process
                ts.close()
                os.execvp(cti.vgwrap_ecmd('c3270'),
                    cti.vgwrap_eargs(['c3270', '-model', '2',
                        '-httpd', f'127.0.0.1:{c3270_port}',
                        '-hostsfile', hostsfile_name,
                        'fOOey']))
                self.assertTrue(False, 'c3270 did not start')

            # Parent process.

            # Make sure c3270 started.
            self.check_listen(c3270_port)
            ts.close()

            # Make sure c3270 is connected.
            r = requests.get(f'http://127.0.0.1:{c3270_port}/3270/rest/json/Wait(2,inputField)')
            self.assertTrue(r.ok)
            cs = requests.get(f'http://127.0.0.1:{c3270_port}/3270/rest/json/Query(connectionState)').json()['result'][0]
            self.assertEqual('connected-nvt', cs)
            requests.get(f'http://127.0.0.1:{c3270_port}/3270/rest/json/Quit()')

        # Wait for the processes to exit.
        self.vgwait_pid(pid)
        os.unlink(hostsfile_name)

if __name__ == '__main__':
    unittest.main()
