/*
 * Copyright (c) 2012-2017 The ANTLR Project. All rights reserved.
 * Use of this file is governed by the BSD 3-clause license that
 * can be found in the LICENSE.txt file in the project root.
 */

import 'dart:developer';

import 'package:logging/logging.dart';

import '../../dfa/dfa.dart';
import '../../error/error.dart';
import '../../input_stream.dart';
import '../../interval_set.dart';
import '../../lexer.dart';
import '../../prediction_context.dart';
import '../../token.dart';
import 'atn.dart';
import 'atn_config.dart';
import 'atn_config_set.dart';
import 'atn_simulator.dart';
import 'atn_state.dart';
import 'lexer_action_executor.dart';
import 'transition.dart';

/// When we hit an accept state in either the DFA or the ATN, we
///  have to notify the character stream to start buffering characters
///  via {@link IntStream#mark} and record the current state. The current sim state
///  includes the current index into the input, the current line,
///  and current character position in that line. Note that the Lexer is
///  tracking the starting line and characterization of the token. These
///  variables track the "state" of the simulator when it hits an accept state.
///
///  <p>We track these variables separately for the DFA and ATN simulation
///  because the DFA simulation often has to fail over to the ATN
///  simulation. If the ATN simulation fails, we need the DFA to fall
///  back to its previously accepted state, if any. If the ATN succeeds,
///  then the ATN does the accept and the DFA simulator that invoked it
///  can simply return the predicted token type.</p>
class SimState {
  int index = -1;
  int line = 0;
  int charPos = -1;

  DFAState dfaState;

  void reset() {
    index = -1;
    line = 0;
    charPos = -1;
    dfaState = null;
  }
}

/// "dup" of ParserInterpreter */
class LexerATNSimulator extends ATNSimulator {
  static final bool debug = true;
  static final bool dfa_debug = true;

  static final int MIN_DFA_EDGE = 0;
  static final int MAX_DFA_EDGE = 127; // forces unicode to stay in ATN

  final Lexer recog;

  /// The current token's starting index into the character stream.
  ///  Shared across DFA to ATN simulation in case the ATN fails and the
  ///  DFA did not have a previous accept state. In this case, we use the
  ///  ATN-generated exception object.
  int startIndex = -1;

  /// line number 1..n within the input */
  int line = 1;

  /// The index of the character relative to the beginning of the line 0..n-1 */
  int charPositionInLine = 0;

  List<DFA> decisionToDFA;
  int mode = Lexer.DEFAULT_MODE;

  /// Used during DFA/ATN exec to record the most recent accept configuration info */

  final SimState prevAccept = SimState();

  LexerATNSimulator(ATN atn, this.decisionToDFA,
      PredictionContextCache sharedContextCache,
      {this.recog})
      : super(atn, sharedContextCache);

  void copyState(LexerATNSimulator simulator) {
    charPositionInLine = simulator.charPositionInLine;
    line = simulator.line;
    mode = simulator.mode;
    startIndex = simulator.startIndex;
  }

  int match(CharStream input, int mode) {
    this.mode = mode;
    final mark = input.mark();
    try {
      startIndex = input.index;
      prevAccept.reset();
      final dfa = decisionToDFA[mode];
      if (dfa.s0 == null) {
        return matchATN(input);
      } else {
        return execATN(input, dfa.s0);
      }
    } finally {
      input.release(mark);
    }
  }

  @override
  void reset() {
    prevAccept.reset();
    startIndex = -1;
    line = 1;
    charPositionInLine = 0;
    mode = Lexer.DEFAULT_MODE;
  }

  @override
  void clearDFA() {
    for (var d = 0; d < decisionToDFA.length; d++) {
      decisionToDFA[d] = DFA(atn.getDecisionState(d), d);
    }
  }

  int matchATN(CharStream input) {
    ATNState startState = atn.modeToStartState[mode];

    if (debug) {
      log('matchATN mode $mode start: $startState\n', level: Level.FINE.value);
    }

    final old_mode = mode;

    final s0_closure = computeStartState(input, startState);
    final suppressEdge = s0_closure.hasSemanticContext;
    s0_closure.hasSemanticContext = false;

    final next = addDFAState(s0_closure);
    if (!suppressEdge) {
      decisionToDFA[mode].s0 = next;
    }

    final predict = execATN(input, next);

    if (debug) {
      log('DFA after matchATN: ${decisionToDFA[old_mode].toLexerString()}\n',
          level: Level.FINE.value);
    }

    return predict;
  }

  int execATN(CharStream input, DFAState ds0) {
    //log("enter exec index "+input.index()+" from "+ds0.configs, level: Level.FINE.value);
    if (debug) {
      log('start state closure=${ds0.configs}\n', level: Level.FINE.value);
    }

    if (ds0.isAcceptState) {
      // allow zero-length tokens
      captureSimState(prevAccept, input, ds0);
    }

    var t = input.LA(1);

    var s = ds0; // s is current/from DFA state

    while (true) {
      // while more work
      if (debug) {
        log('execATN loop starting closure: ${s.configs}\n',
            level: Level.FINE.value);
      }

      // As we move src->trg, src->trg, we keep track of the previous trg to
      // avoid looking up the DFA state again, which is expensive.
      // If the previous target was already part of the DFA, we might
      // be able to avoid doing a reach operation upon t. If s!=null,
      // it means that semantic predicates didn't prevent us from
      // creating a DFA state. Once we know s!=null, we check to see if
      // the DFA state has an edge already for t. If so, we can just reuse
      // it's configuration set; there's no point in re-computing it.
      // This is kind of like doing DFA simulation within the ATN
      // simulation because DFA simulation is really just a way to avoid
      // computing reach/closure sets. Technically, once we know that
      // we have a previously added DFA state, we could jump over to
      // the DFA simulator. But, that would mean popping back and forth
      // a lot and making things more complicated algorithmically.
      // This optimization makes a lot of sense for loops within DFA.
      // A character will take us back to an existing DFA state
      // that already has lots of edges out of it. e.g., .* in comments.
      var target = getExistingTargetState(s, t);
      target ??= computeTargetState(input, s, t);

      if (target == ATNSimulator.ERROR) {
        break;
      }

      // If this is a consumable input element, make sure to consume before
      // capturing the accept state so the input index, line, and char
      // position accurately reflect the state of the interpreter at the
      // end of the token.
      if (t != IntStream.EOF) {
        consume(input);
      }

      if (target.isAcceptState) {
        captureSimState(prevAccept, input, target);
        if (t == IntStream.EOF) {
          break;
        }
      }

      t = input.LA(1);
      s = target; // flip; current DFA target becomes new src/from state
    }

    return failOrAccept(prevAccept, input, s.configs, t);
  }

  /// Get an existing target state for an edge in the DFA. If the target state
  /// for the edge has not yet been computed or is otherwise not available,
  /// this method returns null.
  ///
  /// @param s The current DFA state
  /// @param t The next input symbol
  /// @return The existing target DFA state for the given input symbol
  /// [t], or null if the target state for this edge is not
  /// already cached

  DFAState getExistingTargetState(DFAState s, int t) {
    if (s.edges == null || t < MIN_DFA_EDGE || t > MAX_DFA_EDGE) {
      return null;
    }

    final target = s.edges[t - MIN_DFA_EDGE];
    if (debug && target != null) {
      log('reuse state ${s.stateNumber} edge to ${target.stateNumber}',
          level: Level.FINE.value);
    }

    return target;
  }

  /// Compute a target state for an edge in the DFA, and attempt to add the
  /// computed state and corresponding edge to the DFA.
  ///
  /// @param input The input stream
  /// @param s The current DFA state
  /// @param t The next input symbol
  ///
  /// @return The computed target DFA state for the given input symbol
  /// [t]. If [t] does not lead to a valid DFA state, this method
  /// returns {@link #ERROR}.

  DFAState computeTargetState(CharStream input, DFAState s, int t) {
    ATNConfigSet reach = OrderedATNConfigSet();

    // if we don't find an existing DFA state
    // Fill reach starting from closure, following t transitions
    getReachableConfigSet(input, s.configs, reach, t);

    if (reach.isEmpty) {
      // we got nowhere on t from s
      if (!reach.hasSemanticContext) {
        // we got nowhere on t, don't throw out this knowledge; it'd
        // cause a failover from DFA later.
        addDFAEdge(s, t, ATNSimulator.ERROR);
      }

      // stop when we can't match any more char
      return ATNSimulator.ERROR;
    }

    // Add an edge from s to target DFA found/created for reach
    return addDFAEdgeByConfig(s, t, reach);
  }

  int failOrAccept(
      SimState prevAccept, CharStream input, ATNConfigSet reach, int t) {
    if (prevAccept.dfaState != null) {
      final lexerActionExecutor =
          prevAccept.dfaState.lexerActionExecutor;
      accept(input, lexerActionExecutor, startIndex, prevAccept.index,
          prevAccept.line, prevAccept.charPos);
      return prevAccept.dfaState.prediction;
    } else {
      // if no accept and EOF is first char, return EOF
      if (t == IntStream.EOF && input.index == startIndex) {
        return Token.EOF;
      }

      throw LexerNoViableAltException(recog, input, startIndex, reach);
    }
  }

  /// Given a starting configuration set, figure out all ATN configurations
  ///  we can reach upon input [t]. Parameter [reach] is a return
  ///  parameter.
  void getReachableConfigSet(
      CharStream input, ATNConfigSet configs, ATNConfigSet reach, int t) {
    // this is used to skip processing for configs which have a lower priority
    // than a config that already reached an accept state for the same rule
    var skipAlt = ATN.INVALID_ALT_NUMBER;
    for (var c in configs) {
      final currentAltReachedAcceptState = c.alt == skipAlt;
      if (currentAltReachedAcceptState &&
          (c as LexerATNConfig).hasPassedThroughNonGreedyDecision()) {
        continue;
      }

      if (debug) {
        log('testing ${getTokenName(t)} at ${c.toString(recog, true)}\n',
            level: Level.FINE.value);
      }

      final n = c.state.numberOfTransitions;
      for (var ti = 0; ti < n; ti++) {
        // for each transition
        final trans = c.state.transition(ti);
        final target = getReachableTarget(trans, t);
        if (target != null) {
          var lexerActionExecutor =
              (c as LexerATNConfig).lexerActionExecutor;
          if (lexerActionExecutor != null) {
            lexerActionExecutor = lexerActionExecutor
                .fixOffsetBeforeMatch(input.index - startIndex);
          }

          final treatEofAsEpsilon = t == IntStream.EOF;
          if (closure(
              input,
              LexerATNConfig.dup(c, target,
                  lexerActionExecutor: lexerActionExecutor),
              reach,
              currentAltReachedAcceptState,
              true,
              treatEofAsEpsilon)) {
            // any remaining configs for this alt have a lower priority than
            // the one that just reached an accept state.
            skipAlt = c.alt;
            break;
          }
        }
      }
    }
  }

  void accept(CharStream input, LexerActionExecutor lexerActionExecutor,
      int startIndex, int index, int line, int charPos) {
    if (debug) {
      log('ACTION $lexerActionExecutor\n', level: Level.FINE.value);
    }

    // seek to after last char in token
    input.seek(index);
    this.line = line;
    charPositionInLine = charPos;

    if (lexerActionExecutor != null && recog != null) {
      lexerActionExecutor.execute(recog, input, startIndex);
    }
  }

  ATNState getReachableTarget(Transition trans, int t) {
    if (trans.matches(t, Lexer.MIN_CHAR_VALUE, Lexer.MAX_CHAR_VALUE)) {
      return trans.target;
    }

    return null;
  }

  ATNConfigSet computeStartState(CharStream input, ATNState p) {
    PredictionContext initialContext = PredictionContext.EMPTY;
    ATNConfigSet configs = OrderedATNConfigSet();
    for (var i = 0; i < p.numberOfTransitions; i++) {
      final target = p.transition(i).target;
      final c = LexerATNConfig(target, i + 1, initialContext);
      closure(input, c, configs, false, false, false);
    }
    return configs;
  }

  /// Since the alternatives within any lexer decision are ordered by
  /// preference, this method stops pursuing the closure as soon as an accept
  /// state is reached. After the first accept state is reached by depth-first
  /// search from [config], all other (potentially reachable) states for
  /// this rule would have a lower priority.
  ///
  /// @return [true] if an accept state is reached, otherwise
  /// [false].
  bool closure(
      CharStream input,
      LexerATNConfig config,
      ATNConfigSet configs,
      bool currentAltReachedAcceptState,
      bool speculative,
      bool treatEofAsEpsilon) {
    if (debug) {
      log('closure(' + config.toString(recog, true) + ')',
          level: Level.FINE.value);
    }

    if (config.state is RuleStopState) {
      if (debug) {
        if (recog != null) {
          log('closure at ${recog.ruleNames[config.state.ruleIndex]} rule stop $config\n',
              level: Level.FINE.value);
        } else {
          log('closure at rule stop $config\n', level: Level.FINE.value);
        }
      }

      if (config.context == null || config.context.hasEmptyPath()) {
        if (config.context == null || config.context.isEmpty) {
          configs.add(config);
          return true;
        } else {
          configs.add(LexerATNConfig.dup(config, config.state,
              context: PredictionContext.EMPTY));
          currentAltReachedAcceptState = true;
        }
      }

      if (config.context != null && !config.context.isEmpty) {
        for (var i = 0; i < config.context.length; i++) {
          if (config.context.getReturnState(i) !=
              PredictionContext.EMPTY_RETURN_STATE) {
            final newContext =
                config.context.getParent(i); // "pop" return state
            final returnState = atn.states[config.context.getReturnState(i)];
            final c = LexerATNConfig.dup(config, returnState,
                context: newContext);
            currentAltReachedAcceptState = closure(input, c, configs,
                currentAltReachedAcceptState, speculative, treatEofAsEpsilon);
          }
        }
      }

      return currentAltReachedAcceptState;
    }

    // optimization
    if (!config.state.onlyHasEpsilonTransitions()) {
      if (!currentAltReachedAcceptState ||
          !config.hasPassedThroughNonGreedyDecision()) {
        configs.add(config);
      }
    }

    final p = config.state;
    for (var i = 0; i < p.numberOfTransitions; i++) {
      final t = p.transition(i);
      final c = getEpsilonTarget(
          input, config, t, configs, speculative, treatEofAsEpsilon);
      if (c != null) {
        currentAltReachedAcceptState = closure(input, c, configs,
            currentAltReachedAcceptState, speculative, treatEofAsEpsilon);
      }
    }

    return currentAltReachedAcceptState;
  }

  // side-effect: can alter configs.hasSemanticContext

  LexerATNConfig getEpsilonTarget(
      CharStream input,
      LexerATNConfig config,
      Transition t,
      ATNConfigSet configs,
      bool speculative,
      bool treatEofAsEpsilon) {
    LexerATNConfig c;
    switch (t.type) {
      case TransitionType.RULE:
        RuleTransition ruleTransition = t;
        PredictionContext newContext = SingletonPredictionContext.create(
            config.context, ruleTransition.followState.stateNumber);
        c = LexerATNConfig.dup(config, t.target, context: newContext);
        break;

      case TransitionType.PRECEDENCE:
        throw UnsupportedError(
            'Precedence predicates are not supported in lexers.');
      case TransitionType.PREDICATE:
        /*  Track traversing semantic predicates. If we traverse,
				 we cannot add a DFA state for this "reach" computation
				 because the DFA would not test the predicate again in the
				 future. Rather than creating collections of semantic predicates
				 like v3 and testing them on prediction, v4 will test them on the
				 fly all the time using the ATN not the DFA. This is slower but
				 semantically it's not used that often. One of the key elements to
				 this predicate mechanism is not adding DFA states that see
				 predicates immediately afterwards in the ATN. For example,

				 a : ID {p1}? | ID {p2}? ;

				 should create the start state for rule 'a' (to save start state
				 competition), but should not create target of ID state. The
				 collection of ATN states the following ID references includes
				 states reached by traversing predicates. Since this is when we
				 test them, we cannot cash the DFA state target of ID.
			 */
        PredicateTransition pt = t;
        if (debug) {
          log('EVAL rule ${pt.ruleIndex}:${pt.predIndex}',
              level: Level.FINE.value);
        }
        configs.hasSemanticContext = true;
        if (evaluatePredicate(input, pt.ruleIndex, pt.predIndex, speculative)) {
          c = LexerATNConfig.dup(config, t.target);
        }
        break;
      case TransitionType.ACTION:
        if (config.context == null || config.context.hasEmptyPath()) {
          // execute actions anywhere in the start rule for a token.
          //
          // TODO: if the entry rule is invoked recursively, some
          // actions may be executed during the recursive call. The
          // problem can appear when hasEmptyPath() is true but
          // isEmpty is false. In this case, the config needs to be
          // split into two contexts - one with just the empty path
          // and another with everything but the empty path.
          // Unfortunately, the current algorithm does not allow
          // getEpsilonTarget to return two configurations, so
          // additional modifications are needed before we can support
          // the split operation.
          final lexerActionExecutor = LexerActionExecutor.append(
              config.lexerActionExecutor,
              atn.lexerActions[(t as ActionTransition).actionIndex]);
          c = LexerATNConfig.dup(config, t.target,
              lexerActionExecutor: lexerActionExecutor);
        } else {
          // ignore actions in referenced rules
          c = LexerATNConfig.dup(config, t.target);
        }
        break;

      case TransitionType.EPSILON:
        c = LexerATNConfig.dup(config, t.target);
        break;

      case TransitionType.ATOM:
      case TransitionType.RANGE:
      case TransitionType.SET:
        if (treatEofAsEpsilon) {
          if (t.matches(
              IntStream.EOF, Lexer.MIN_CHAR_VALUE, Lexer.MAX_CHAR_VALUE)) {
            c = LexerATNConfig.dup(config, t.target);
            break;
          }
        }
        break;
      case TransitionType.NOT_SET:
        break;
      case TransitionType.WILDCARD:
        break;
      case TransitionType.INVALID:
        throw ArgumentError.value(t.type, 'TransitionType');
        break;
    }

    return c;
  }

  /// Evaluate a predicate specified in the lexer.
  ///
  /// <p>If [speculative] is [true], this method was called before
  /// {@link #consume} for the matched character. This method should call
  /// {@link #consume} before evaluating the predicate to ensure position
  /// sensitive values, including {@link Lexer#getText}, {@link Lexer#getLine},
  /// and {@link Lexer#getCharPositionInLine}, properly reflect the current
  /// lexer state. This method should restore [input] and the simulator
  /// to the original state before returning (i.e. undo the actions made by the
  /// call to {@link #consume}.</p>
  ///
  /// @param input The input stream.
  /// @param ruleIndex The rule containing the predicate.
  /// @param predIndex The index of the predicate within the rule.
  /// @param speculative [true] if the current index in [input] is
  /// one character before the predicate's location.
  ///
  /// @return [true] if the specified predicate evaluates to
  /// [true].
  bool evaluatePredicate(
      CharStream input, int ruleIndex, int predIndex, bool speculative) {
    // assume true if no recognizer was provided
    if (recog == null) {
      return true;
    }

    if (!speculative) {
      return recog.sempred(null, ruleIndex, predIndex);
    }

    final savedCharPositionInLine = charPositionInLine;
    final savedLine = line;
    final index = input.index;
    final marker = input.mark();
    try {
      consume(input);
      return recog.sempred(null, ruleIndex, predIndex);
    } finally {
      charPositionInLine = savedCharPositionInLine;
      line = savedLine;
      input.seek(index);
      input.release(marker);
    }
  }

  void captureSimState(SimState settings, CharStream input, DFAState dfaState) {
    settings.index = input.index;
    settings.line = line;
    settings.charPos = charPositionInLine;
    settings.dfaState = dfaState;
  }

  DFAState addDFAEdgeByConfig(DFAState from, int t, ATNConfigSet q) {
    /* leading to this call, ATNConfigSet.hasSemanticContext is used as a
		 * marker indicating dynamic predicate evaluation makes this edge
		 * dependent on the specific input sequence, so the static edge in the
		 * DFA should be omitted. The target DFAState is still created since
		 * execATN has the ability to resynchronize with the DFA state cache
		 * following the predicate evaluation step.
		 *
		 * TJP notes: next time through the DFA, we see a pred again and eval.
		 * If that gets us to a previously created (but dangling) DFA
		 * state, we can continue in pure DFA mode from there.
		 */
    final suppressEdge = q.hasSemanticContext;
    q.hasSemanticContext = false;

    final to = addDFAState(q);

    if (suppressEdge) {
      return to;
    }

    addDFAEdge(from, t, to);
    return to;
  }

  void addDFAEdge(DFAState p, int t, DFAState q) {
    if (t < MIN_DFA_EDGE || t > MAX_DFA_EDGE) {
      // Only track edges within the DFA bounds
      return;
    }

    if (debug) {
      log('EDGE $p -> $q upon ${String.fromCharCode(t)}',
          level: Level.FINE.value);
    }

    p.edges ??= List<DFAState>(MAX_DFA_EDGE - MIN_DFA_EDGE + 1);
    p.edges[t - MIN_DFA_EDGE] = q; // connect
  }

  /// Add a new DFA state if there isn't one with this set of
  /// configurations already. This method also detects the first
  /// configuration containing an ATN rule stop state. Later, when
  /// traversing the DFA, we will know which rule to accept.
  DFAState addDFAState(ATNConfigSet configs) {
    /* the lexer evaluates predicates on-the-fly; by this point configs
		 * should not contain any configurations with unevaluated predicates.
		 */
    assert(!configs.hasSemanticContext);

    final proposed = DFAState(configs: configs);
    ATNConfig firstConfigWithRuleStopState;
    for (var c in configs) {
      if (c.state is RuleStopState) {
        firstConfigWithRuleStopState = c;
        break;
      }
    }

    if (firstConfigWithRuleStopState != null) {
      proposed.isAcceptState = true;
      proposed.lexerActionExecutor =
          (firstConfigWithRuleStopState as LexerATNConfig)
              .lexerActionExecutor;
      proposed.prediction =
          atn.ruleToTokenType[firstConfigWithRuleStopState.state.ruleIndex];
    }

    final dfa = decisionToDFA[mode];
    final existing = dfa.states[proposed];
    if (existing != null) return existing;

    final newState = proposed;

    newState.stateNumber = dfa.states.length;
    configs.readOnly = true;
    newState.configs = configs;
    dfa.states[newState] = newState;
    return newState;
  }

  DFA getDFA(int mode) {
    return decisionToDFA[mode];
  }

  /// Get the text matched so far for the current token.

  String getText(CharStream input) {
    // index is first lookahead char, don't include.
    return input.getText(Interval.of(startIndex, input.index - 1));
  }

  void consume(CharStream input) {
    final curChar = input.LA(1);
    if (curChar == 10) { // Is new line
      line++;
      charPositionInLine = 0;
    } else {
      charPositionInLine++;
    }
    input.consume();
  }

  String getTokenName(int t) {
    if (t == -1) return 'EOF';
    //if ( atn.g!=null ) return atn.g.getTokenDisplayName(t);
    return "'${String.fromCharCode(t)}'";
  }
}
