/*****************************************************************************
*
* Tutorial: How to ALPSize your applications
*
* Copyright (C) 2005-2010 by Synge Todo <wistaria@comp-phys.org>
*
* Permission is hereby granted, free of charge, to any person or organization
* obtaining a copy of the software and accompanying documentation covered by
* this license (the "Software") to use, reproduce, display, distribute,
* execute, and transmit the Software, and to prepare derivative works of the
* Software, and to permit third-parties to whom the Software is furnished to
* do so, all subject to the following:
*
* The copyright notices in the Software and this entire statement, including
* the above license grant, this restriction and the following disclaimer,
* must be included in all copies of the Software, in whole or in part, and
* all derivative works of the Software, unless such copies or derivative
* works are solely in the form of machine-executable object code generated by
* a source language processor.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT 
* SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE 
* FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, 
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
* DEALINGS IN THE SOFTWARE.
*
*****************************************************************************/

#include <alps/parameter.h>
#include <alps/lattice.h>
#include <boost/foreach.hpp>
#include <iostream>

int main() {

  alps::Parameters params(std::cin);
  alps::graph_helper<> graph(params);

  std::cout << "lattice name = " << params["LATTICE"] << std::endl;

  //
  // number of sites
  //

  std::cout << "number of sites = " << graph.num_sites() << std::endl;

  //
  // number of bonds
  //

  std::cout << "number of bonds = " << graph.num_bonds() << std::endl;

  //
  // iterate over all the sites and print out indices
  //

  // (1)
  for (int i = 0; i < graph.num_sites(); ++i)
    std::cout << "site index = " << i << std::endl;

  // (2)
  alps::graph_helper<>::site_iterator itr0, itr0_end;
  for (boost::tie(itr0, itr0_end) = graph.sites(); itr0 != itr0_end; ++itr0) {
    std::cout << "site index = " << graph.index(*itr0) << std::endl;
    // this is equivalent to
    alps::graph_helper<>::site_descriptor desc = *itr0;
    std::cout << "site index = " << graph.index(desc) << std::endl;
    // or simply
    std::cout << "site index = " << desc << std::endl;
    // or just
    std::cout << "site index = " << *itr0 << std::endl;
  }

  // (3) (using BOOST_FOREACH)
  BOOST_FOREACH(const alps::graph_helper<>::site_descriptor& s, graph.sites()) {
    std::cout << "site index = " << graph.index(s) << std::endl;
    // or just
    std::cout << "site index = " << s << std::endl;
  }

  // (4) since site_descriptor is equivalent to int the following code also works
  BOOST_FOREACH(int s, graph.sites())
    std::cout << "site index = " << s << std::endl;

  //
  // iterate over all the bonds and print out indices
  //

  // (1)
  for (int i = 0; i < graph.num_bonds(); ++i)
    std::cout << "bond index = " << i << std::endl;

  // (2)
  alps::graph_helper<>::bond_iterator itr1, itr1_end;
  for (boost::tie(itr1, itr1_end) = graph.bonds(); itr1 != itr1_end; ++itr1) {
    std::cout << "bond index = " << graph.index(*itr1) << std::endl;
    // this is equivalent to
    alps::graph_helper<>::bond_descriptor desc = *itr1;
    std::cout << "bond index = " << graph.index(desc) << std::endl;
    // Warning: the following codes do not output the bond index, but pair of its source and target
    std::cout << "bond index != " << desc << std::endl;
    std::cout << "bond index != " << *itr1 << std::endl;
  }

  // (3) (using BOOST_FOREACH)
  BOOST_FOREACH(const alps::graph_helper<>::bond_descriptor& b, graph.bonds()) {
    std::cout << "bond index = " << graph.index(b) << std::endl;
    // Warning: the following code does not output the bond index, but pair of its source and target
    std::cout << "bond index != " << b << std::endl;
  }

  return 0;
}
