#!/usr/bin/perl -w
#ROC.pl

use strict;

use vars qw ($opt_a $opt_c $opt_C $opt_d $opt_i $opt_I $opt_j $opt_J $opt_G $opt_H $opt_n $opt_r $opt_s $opt_u);
use Getopt::Std;
use constant GNUPLOT => '/usr/bin/gnuplot';   

getopts ('a:c:Cd:i:I:j:J:G:H:n:rs:u:'); 

if (!@ARGV) { 
    print "usage: ROC.pl [options] file1 filesh1 file2 filesh2\n";
    print "options:\n";   
    print "-a <max_n_ali>          : max number of alignments                 [default all]\n";
    print "-c <min_cutoff>         : min cutoff                               [default min_cutoff = -200]\n";
    print "-C                      : ROC curves for COD logsigmoidalscores    [default RNA sigmoidal scores]\n";
    print "-d <max_cutoff>         : max cutoff                               [default max_cutoff =  200]\n";
    print "-i <inc_cutoff>         : inc cutoff                               [default inc_cutoff =  1]\n";
    print "-I <min_id>             : min ID for analysis                      [default min_id = 0]\n";
    print "-j <max_sen>            : max sensitivity for plotting                      []\n";
    print "-J <max_id>             : max ID for analysis                      [default max_id = 100]\n";
    print "-G <min_gc>             : min GC for analysis                      [default min_gc = 0]\n";
    print "-H <max_gc>             : max GC for analysis                      [default max_gc = 100]\n";
    print "-n <num>                : number of additional files               [default none\n";
    print "-r                      : second file is an rnaz file              [default none\n";
    print "-s <spe_cutoff>         : specifity cutoff for plotting            [default none\n";
    print "-u <score_cutoff>       : score cutoff for plotting                [default none\n";
}

my $file1   = shift;
my $filesh1 = shift;
my $file2   = shift;
my $filesh2 = shift;

my $file3;
my $filesh3;
my $n_files;
if (!$opt_n) { $n_files = 2;      }
else         { $n_files = $opt_n; }

if ($n_files == 3) { $file3 = shift; $filesh3 = shift; }

#
#options
#
 my $roc_type = "RNA";
if ($opt_C) { $roc_type = "COD"; }

my $min_cutoff;
if ($opt_c) { $min_cutoff = $opt_c; }
else        { $min_cutoff = -500;    }
my $max_cutoff;
if ($opt_d) { $max_cutoff = $opt_d; }
else        { $max_cutoff = 700;    }

my $inc_cutoff;
if ($opt_i) { $inc_cutoff = $opt_i; }
else        { $inc_cutoff = 5;      }

my $max_n_ali;
if ($opt_a) { $max_n_ali = $opt_a; }
else        { $max_n_ali = 1000000; }
my $min_id;
if ($opt_I) { $min_id = $opt_I; }
else        { $min_id = 0;      }
my $max_id;
if ($opt_J) { $max_id = $opt_J; }
else        { $max_id = 100;    }
my $min_gc;
if ($opt_G) { $min_gc = $opt_G; }
else        { $min_gc = 0;      }
my $max_gc;
if ($opt_H) { $max_gc = $opt_H; }
else        { $max_gc = 100;    }

my $sc_limit = $min_cutoff;
if ($opt_u) { $sc_limit = $opt_u; }

my $spec = "ID[$max_id:$min_id].GC[$max_gc:$min_gc]";


my $sp_limit = 0.0;
if ($opt_s) { $sp_limit = $opt_s; }


my $rocfile1;
if ($opt_u) { $rocfile1 = "$file1.SCcutoff$sc_limit.$roc_type.roc"; }
else        { $rocfile1 = "$file1.$roc_type.roc";                }

my $total1;
my $total1_sh;

roc_file($roc_type, "$file1", "$filesh1", "$rocfile1", "qrna", 
	 $min_cutoff, $max_cutoff, $inc_cutoff, 
	 $min_id, $max_id, $min_gc, $max_gc, $max_n_ali, \$total1, \$total1_sh);

my $rocfile2;
if ($opt_u) { $rocfile2 = "$file2.SCcutoff$sc_limit.$roc_type.roc"; }
else        { $rocfile2 = "$file2.$roc_type.roc";                }

my $type;
if ($opt_r) { $type = "RNAz"; }
else        { $type = "qrna"; }

my $total2;
my $total2_sh;

roc_file($roc_type, "$file2", "$filesh2", "$rocfile2", $type, 
	 $min_cutoff, $max_cutoff, $inc_cutoff, 
	 $min_id, $max_id, $min_gc, $max_gc, $max_n_ali, \$total2, \$total2_sh);
 
if ($total1 != $total2) {
    print "\nWarning: different number of alignments total1 = $total1, total2 = $total2\n"; 
}
if ($total1_sh != $total2_sh) {
    print "\nWarning: different number of shuffled alignments total1_sh = $total1_sh, total2_sh = $total2_sh\n"; 
}

plot_roc_together ($roc_type, $rocfile1, $rocfile2);

if ($n_files == 3) {
    my $rocfile3;
    if ($opt_u) { $rocfile3 = "$file3.SCcutoff$sc_limit.$roc_type.roc"; }
    else        { $rocfile3 = "$file3.$roc_type.roc";                }
    
    my $total3;
    my $total3_sh;

    roc_file($roc_type, "$file3", "$filesh3", "$rocfile3", "ddbrna", 
	     $min_cutoff, $max_cutoff, $inc_cutoff, 
	     $min_id, $max_id, $min_gc, $max_gc, $max_n_ali, \$total3, \$total3_sh);

    if ($total1 != $total2 || $total1 != $total3) {
	print "\nWarning: different number of alignments total1 = $total1, total2 = $total2, total3 = $total3\n"; 
    }
    if ($total1_sh != $total2_sh || $total1_sh != $total3_sh) {
	print "\nWarning: different number of shuffled alignments total1_sh = $total1_sh, total2_sh = $total2_sh, total3_sh = $total3_sh\n"; 
    }

    plot_roc_together3 ($roc_type, "$rocfile1", "$rocfile2", "$rocfile3");
}


#
#
# subrutines
#
#

sub cal_error_area {

    my ($his_ref, $his_sh_ref, $dtotal, $dtotalsh, $min_cutoff, $max_cutoff, $inc_cutoff) = @_; 

    my $total   = 0;
    my $totalsh = 0;

    my $error_area = 0;
    my $cut = $inc_cutoff*($max_cutoff-$min_cutoff);

    my @his;
    my @his_sh;

    my $cutoff;

    if (0) {
	print "TOTAL $dtotal\n";
	for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
	    print "CUTOFF ", $cutoff/$inc_cutoff+$min_cutoff, " $his_ref->[$cutoff] \n";
	}
	print "TOTAL_SH $dtotalsh\n";
	for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
	    print "CUTOFF ", $cutoff/$inc_cutoff+$min_cutoff, " $his_sh_ref->[$cutoff] \n";
	}
    }


    for ($cutoff = 0;  $cutoff < $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) { 

	$his   [$cutoff] = $his_ref->   [$cutoff] - $his_ref->   [$cutoff + 1];
	$his_sh[$cutoff] = $his_sh_ref->[$cutoff] - $his_sh_ref->[$cutoff + 1];

	if ($his[$cutoff] < 0 || $his_sh[$cutoff] < 0) { 
	    print "You did something wrong calculating the error area: cutoff=$cutoff his=$his[$cutoff] his_sh=$his_sh[$cutoff]\n";
	    die;
	}
	
    }
    $his   [$inc_cutoff*($max_cutoff-$min_cutoff)] = $his_ref->   [$cutoff];
    $his_sh[$inc_cutoff*($max_cutoff-$min_cutoff)] = $his_sh_ref->[$cutoff];

    for ($cutoff = $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff >= 0; $cutoff --) { 	
	$total   += $his[$cutoff];
	$totalsh += $his_sh[$cutoff];

	#print $cutoff+$min_cutoff, "  $his_ref->[$cutoff] $his[$cutoff] $total\n";
	#print $cutoff+$min_cutoff, "  $his_sh_ref->[$cutoff] $his_sh[$cutoff] $totalsh\n";
    }

    if ($total   != $dtotal)   { print "counting error total $total $dtotal\n"; die; }
    if ($totalsh != $dtotalsh) { print "counting error total_sh $totalsh $dtotalsh\n"; die; }

    for ($cutoff = $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff >= 0; $cutoff --) { 	

	if ($his[$cutoff] > 0 && $his_sh[$cutoff] > 0 && $his_sh[$cutoff] > $his[$cutoff]) { 
	    $cut = $cutoff; 
	    last; 
	}
    }

    if (0) {
	print "total = ",    $total, " \n";
	print "totalsh = ", $totalsh, " \n";
	print "CUT = ", $cut/$inc_cutoff+$min_cutoff, " \n";
    }
    
    for ($cutoff = $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff > $cut; $cutoff --) {	
	$error_area += $his[$cutoff]-$his_sh[$cutoff]; 
    }

    if ($total > 0) { $error_area /= $total; }
    else            { print "No points in this histogram \n"; die; }

    $error_area = 1.0 - $error_area;

    $error_area = int($error_area*10000)/100;

    return $error_area;

}

sub is_above_cutoff {
    my ($cutoff, $lodsigm_sc) = @_;

    my $above = 0;
  
    if ($lodsigm_sc >= $cutoff) { $above = 1; }

    return $above;
}



sub parse_qrna {

    my ($file, $rna_ref, $cod_ref, $total_ref, $min_cutoff, $max_cutoff, $inc_cutoff, $min_id, $max_id, $min_gc, $max_gc, $max_n_ali) = @_;

    my $othsc;
    my $codsc;
    my $rnasc;

    my $othlodsigm;
    my $codlodsigm;
    my $rnalodsigm;

    my $verbose = 0;

    my @rna;
    my @cod;

    my $len;
    my $id;
    my $gc;
    my $n_ali = 0;

    my $version;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#\-/) { next;
	}
	elsif (/^\#\s+qrna\s(\S+)/) {
	    $version = $1;
	}
	elsif (/^length alignment:\s+(\d+)\s+\(id=(\d+\.\d+)\)/) 
	{ 
	    $len = $1+1;
	    $id  = $2;
	} elsif (/^posX:.+\(\S+\s(\S+)\s(\S+)\s\S+\)/) 
	{ 
	    $gc = $1 + $2; 
	    
	}  elsif (/^posY:.+\(\S+\s(\S+)\s(\S+)\s\S+\)/) 
	{ 
	    $gc += $1 + $2; $gc *= 50;
	    
	}
	elsif ($version =~ /^1.2b$/ && /^\s+OTH =\s+(\S+)\s+COD =\s+(\S+)\s+RNA =\s+(\S+)\s+$/) 
	{ 
	    $n_ali ++;

	    $othsc = $1;
	    $codsc = $2;
	    $rnasc = $3;	
		    
	    if ($codsc-$othsc < -1000 && $rnasc-$othsc < -1000) {
		$othlodsigm = 1000;
	    }
	    else {
		$othlodsigm = - log(exp(log(2.0)*($codsc-$othsc)) + exp(log(2.0)*($rnasc-$othsc)))/log(2.0);
	    }
	    
	    if ($othsc-$codsc < -1000 && $rnasc-$codsc < -1000) {
		$othlodsigm = 1000;
	    }
	    else {
			$codlodsigm = - log(exp(log(2.0)*($othsc-$codsc)) + exp(log(2.0)*($rnasc-$codsc)))/log(2.0);
		    }
	    
	    if ($othsc-$rnasc < -1000 && $codsc-$rnasc < -1000) {
		$othlodsigm = 1000;
	    }
	    else {
		$rnalodsigm = - log(exp(log(2.0)*($othsc-$rnasc)) + exp(log(2.0)*($codsc-$rnasc)))/log(2.0);
	    }
	    
	    
	    if ($othlodsigm < -1000) { $othlodsigm = -1000; next; }
	    if ($codlodsigm < -1000) { $codlodsigm = -1000; next; }
	    if ($rnalodsigm < -1000) { $rnalodsigm = -1000; next; }

	    if ($id >= $min_id && $id <= $max_id && $gc >= $min_gc && $gc <= $max_gc && $n_ali <= $max_n_ali) {
		
		#print "SSS $othlodsigm $codlodsigm $rnalodsigm\n";
		
		for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
		    
		    if(is_above_cutoff($cutoff/$inc_cutoff+$min_cutoff, $rnalodsigm)) { $rna_ref->[$cutoff] ++; }
		    if(is_above_cutoff($cutoff/$inc_cutoff+$min_cutoff, $codlodsigm)) { $cod_ref->[$cutoff] ++; }
		    
		}
		
		$$total_ref ++;
	    }
	    elsif ($n_ali > $max_n_ali) { last; }
	    
	}	    
	elsif (/^\s+sigmoidalOTH =\s+(\S+)\s+sigmoidalCOD =\s+(\S+)\s+sigmoidalRNA =\s+(\S+)\s+$/) 
	{ 	    
	    $n_ali ++;

	    $othlodsigm = $1;
	    $codlodsigm = $2;
	    $rnalodsigm = $3;

	    if ($id >= $min_id && $id <= $max_id && $gc >= $min_gc && $gc <= $max_gc && $n_ali <= $max_n_ali) {
		
		#print "SSS $othlodsigm $codlodsigm $rnalodsigm\n";
		
		for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
		    
		    if(is_above_cutoff($cutoff/$inc_cutoff+$min_cutoff, $rnalodsigm)) { $rna_ref->[$cutoff] ++; }
		    if(is_above_cutoff($cutoff/$inc_cutoff+$min_cutoff, $codlodsigm)) { $cod_ref->[$cutoff] ++; }
		    
		}
		
		$$total_ref ++;
	    }	   
	    elsif ($n_ali > $max_n_ali) { last; }

	}
    }
    close (FILE);

    if (0) {
	for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
	    print "CUTOFF ", $cutoff/$inc_cutoff+$min_cutoff, " $rna_ref->[$cutoff] $cod_ref->[$cutoff]\n";
	}
    }

}

sub parse_ddbrna {

    my ($file, $rna_ref, $total_ref, $min_cutoff, $max_cutoff, $inc_cutoff, $max_n_ali) = @_;

    my $n_ali = 0;

    my $val;
    my $mean;
    my $std;

    my $scale = 50;
    my $zsc;

    open (FILE,"$file") || die;
    while (<FILE>) {
	
	if (/^\S+\s+\S+\s+(\S+)\s+(\S+)\s+\(\+\/\-\s+(\S+)\)/) {
	    $n_ali ++;

	    $val  = $1;
	    $mean = $2;
	    $std  = $3;

	    $zsc = 0.0; 
	    
	    if ($std > 0) { $zsc = ($val - $mean) / $std; }

	    if ($n_ali <= $max_n_ali) {
		
		for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
		    
		    if ($zsc*$scale >= $cutoff/$inc_cutoff+$min_cutoff) {  $rna_ref->[$cutoff] ++; }
		}
		
		$$total_ref ++;
	    }
	}
    }
    close (FILE);

}

sub parse_RNAz {

    my ($file, $rna_ref, $total_ref, $min_cutoff, $max_cutoff, $inc_cutoff, $min_id, $max_id, $max_n_ali) = @_;

    my $n_ali = 0;

    my $p;
    my $scale = 1000;
    my $logp;

    my $id;
    my $max_p = 0.0;

    open (FILE,"$file") || die;
    while (<FILE>) {
	
	if  (/Mean pairwise identity:\s+(\S+)/)
	{
	    $id  = $1;
	}
	elsif (/SVM RNA-class probability:\s+(\S+)/) {
	    $n_ali ++;

	    $p = $1;
	    if ($p > $max_p) { $max_p = $p; }
	    
	    if ($p > 0.0) { 
		$logp = $scale*log($p/0.5); 			    
		if ($logp < $min_cutoff) { $logp = $min_cutoff; } 
		if ($logp > $max_cutoff) { $logp = $max_cutoff; } 
	    }
	    else          { $logp = $min_cutoff; }

	    if ($id >= $min_id && $id <= $max_id && $n_ali <= $max_n_ali) {

		for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
		    if ($logp >= $cutoff/$inc_cutoff+$min_cutoff) {  
			$rna_ref->[$cutoff] ++; 
		    }
		}
		
		$$total_ref ++;
	    }
	}
    }
    close (FILE);

    print "\nmax_p $max_p max_logp ", $scale*log($max_p/0.5), "\n";

}


sub plot_roc {
    
    my ($roc_type, $file) = @_;
    
    my $tag1;
    my $tag2;
    my $tag3;
    if    ($roc_type =~ /^RNA$/) { $tag1 = "8";  $tag2 = "9";  $tag3 = "10";  }
    elsif ($roc_type =~ /^COD$/) { $tag1 = "11"; $tag2 = "12"; $tag3 = "13"; }
    else                         { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

    my $title;
    my $xlabel = "SENSITIVITY";
    my $ylabel = "SPECIFICITY";

    my $key = read_head("$file", \$title);
    my $ea  = read_error_area("$file");

    $key .= " ** EA=$ea";

    if ($opt_u) { $title .= "Score_cutoff = $sc_limit"; }
    
    my $max_sen1 = read_max_sen (1, $roc_type, "$file");
    my $max_sen2 = read_max_sen (2, $roc_type, "$file");
    my $min_spe1 = read_min_spe (1, $roc_type, "$file");
    my $min_spe2 = read_min_spe (2, $roc_type, "$file");

    if ($opt_j) { $max_sen1 = $opt_j; $max_sen2 = $opt_j; }

    my $outplot = "$file.ps";
    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";
    
    print GP "set terminal postscript  default color solid 14\n";
    print GP "set linestyle 1 lt 1 lw 4\n";
    print GP "set linestyle 2 lt 2 lw 4\n";
    print GP "set linestyle 3  lt 3 lw 4\n";
    print GP "set linestyle 4 lt 4 lw 4\n";
    print GP "set linestyle 5 lt 5 lw 4\n";
    print GP "set linestyle 6 lt 6 lw 4\n";
    print GP "set linestyle 7 lt 7 lw 4\n";
    print GP "set linestyle 8 lt 8 lw 4\n";
    print GP "set linestyle 9 lt 9 lw 4\n";
    print GP "set linestyle 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set linestyle 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set linestyle 33 lt 3 lw 2 pt 3 ps 1.2\n";
    print GP "set linestyle 44 lt 4 lw 2 pt 4 ps 1.2\n";
    print GP "set linestyle 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set linestyle 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set linestyle 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set linestyle 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set linestyle 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set title '$title'\n"; 
    print GP "set ylabel '$ylabel'\n";
    print GP "set xlabel '$xlabel'\n";

    print GP "plot [0:$max_sen1] [$min_spe1:1.0+0.1*(1.0-$min_spe1)/0.99]  [0:1.18] '$file' using $tag1:$tag2 title '$key' with lines ls 3\n";
    print GP "plot [0:$max_sen2] [$min_spe2:1.0+0.1*(1.0-$min_spe2)/0.99]  [0:1.18] '$file' using $tag1:$tag3 title '$key' with lines ls 3\n";
    close(GP);

    #system ("gv -landscape -magstep -2 $outplot&\n");
}

sub plot_roc_together {
    
    my ($roc_type, $file_a, $file_b) = @_;
    
    my $tag1;
    my $tag2;
    my $tag3;
    if    ($roc_type =~ /^RNA$/) { $tag1 = "8";  $tag2 = "9";  $tag3 = "10";  }
    elsif ($roc_type =~ /^COD$/) { $tag1 = "11"; $tag2 = "12"; $tag3 = "13"; }
    else                         { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

    my $title;
    my $xlabel = "SENSITIVITY";
    my $ylabel = "SPECIFICITY";
    
    my $key_a = read_head("$file_a", \$title);
    my $key_b = read_head("$file_b", \$title);
    #print "$file_a ++ $key_a1\n$file_b ++ $key_b1\n";

    $title .= " $spec";
    if ($opt_u) { $title .= "Score_cutoff = $sc_limit"; }

    my $ea_a = read_error_area("$file_a");
    my $ea_b = read_error_area("$file_b");

    my $max_sen_a1 = read_max_sen (1, $roc_type, "$file_a");
    my $max_sen_a2 = read_max_sen (2, $roc_type, "$file_a");
    my $max_sen_b1 = read_max_sen (1, $roc_type, "$file_b");
    my $max_sen_b2 = read_max_sen (2, $roc_type, "$file_b");

    my $cutoff_a = 4.0;
    my $rna_sen_zero_a1;
    my $rna_sen_zero_a2;
    my $rna_spe_zero_a1;
    my $rna_spe_zero_a2;
    sens_and_spec_at_cutoff (1, "$file_a", $roc_type, $cutoff_a, \$rna_sen_zero_a1, \$rna_spe_zero_a1);
    sens_and_spec_at_cutoff (2, "$file_a", $roc_type, $cutoff_a, \$rna_sen_zero_a2, \$rna_spe_zero_a2);

    my $cutoff_b;
    my $cutoff_b_prob;
    if ($opt_r) { $cutoff_b_prob = 0.99; $cutoff_b = 1000*log($cutoff_b_prob/0.5); }
    else        { $cutoff_b_prob = $cutoff_a; $cutoff_b = $cutoff_a; }
    my $rna_sen_zero_b1;
    my $rna_sen_zero_b2;
    my $rna_spe_zero_b1;
    my $rna_spe_zero_b2;
    sens_and_spec_at_cutoff (1, "$file_b", $roc_type, $cutoff_b, \$rna_sen_zero_b1, \$rna_spe_zero_b1);
    sens_and_spec_at_cutoff (2, "$file_b", $roc_type, $cutoff_b, \$rna_sen_zero_b2, \$rna_spe_zero_b2);

    my $cutoff_file_a = "$file_a.$cutoff_a";
    open (CUTFILE,">$cutoff_file_a") || die;
    print CUTFILE "$rna_sen_zero_a1\t$rna_spe_zero_a1\t$rna_sen_zero_a2\t$rna_spe_zero_a2\n";
    close (CUTFILE); 

    $rna_sen_zero_a1 = int($rna_sen_zero_a1*10000)/100;
    $rna_sen_zero_a2 = int($rna_sen_zero_a2*10000)/100;
    $rna_spe_zero_a1 = int($rna_spe_zero_a1*10000)/100;
    $rna_spe_zero_a2 = int($rna_spe_zero_a2*10000)/100;
    my $key_a1_c = "cutoff= $cutoff_a sen=$rna_sen_zero_a1, spe=$rna_spe_zero_a1";
    my $key_a2_c = "cutoff= $cutoff_a sen=$rna_sen_zero_a2, spe=$rna_spe_zero_a2";

    my $cutoff_file_b = "$file_b.$cutoff_b";
    open (CUTFILE,">$cutoff_file_b") || die;
    print CUTFILE "$rna_sen_zero_b1\t$rna_spe_zero_b1\t$rna_sen_zero_b2\t$rna_spe_zero_b2\n";
    close (CUTFILE);

    $rna_sen_zero_b1 = int($rna_sen_zero_b1*10000)/100;
    $rna_sen_zero_b2 = int($rna_sen_zero_b2*10000)/100;
    $rna_spe_zero_b1 = int($rna_spe_zero_b1*10000)/100;
    $rna_spe_zero_b2 = int($rna_spe_zero_b2*10000)/100;
    my $key_b1_c = "cutoff= $cutoff_b_prob sen=$rna_sen_zero_b1, spe=$rna_spe_zero_b1";
    my $key_b2_c = "cutoff= $cutoff_b_prob sen=$rna_sen_zero_b2, spe=$rna_spe_zero_b2";

    my $max_sen1 = $max_sen_a1;
    my $max_sen2 = $max_sen_a2;
    if ($max_sen_b1 > $max_sen1) { $max_sen1 = $max_sen_b1; }
    if ($max_sen_b2 > $max_sen2) { $max_sen2 = $max_sen_b2; }

    if ($opt_j) { $max_sen1 = $opt_j; $max_sen2 = $opt_j; }

    my $min_spe_a1 = read_min_spe (1, $roc_type, "$file_a");
    my $min_spe_a2 = read_min_spe (2, $roc_type, "$file_a");
    my $min_spe_b1 = read_min_spe (1, $roc_type, "$file_b");
    my $min_spe_b2 = read_min_spe (2, $roc_type, "$file_b");

    my $min_spe1 = $min_spe_a1;
    my $min_spe2 = $min_spe_a2;
    if ($min_spe_b1 < $min_spe1) { $min_spe1 = $min_spe_b1; }
    if ($min_spe_b2 < $min_spe2) { $min_spe2 = $min_spe_b2; }

    $key_a .= " ** Error Area = $ea_a";
    $key_b .= " ** Error Area = $ea_b";

    my $outplot = "$file_a\_$file_b.ps";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";
    
   # print GP "set terminal postscript  default color solid 14\n";
    print GP "set terminal postscript  default color  14\n";
    print GP "set linestyle 1 lt 1 lw 4\n";
    print GP "set linestyle 2 lt 2 lw 4\n";
    print GP "set linestyle 3  lt 3 lw 4\n";
    print GP "set linestyle 4 lt 4 lw 4\n";
    print GP "set linestyle 5 lt 5 lw 4\n";
    print GP "set linestyle 6 lt 6 lw 4\n";
    print GP "set linestyle 7 lt 7 lw 4\n";
    print GP "set linestyle 8 lt 8 lw 4\n";
    print GP "set linestyle 9 lt 9 lw 4\n";
    print GP "set linestyle 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set linestyle 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set linestyle 33 lt 3 lw 2 pt 2  ps 2.0\n";
    print GP "set linestyle 44 lt 4 lw 2 pt 12 ps 2.0\n";
    print GP "set linestyle 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set linestyle 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set linestyle 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set linestyle 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set linestyle 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set title '$title'\n"; 
    print GP "set ylabel '$ylabel'\n";
    print GP "set xlabel '$xlabel'\n";

    #print GP "plot [0:$max_sen] [$min_spe:1.0+0.1*(1.0-$min_spe)/0.99] '$file_a' using $tag1:$tag2 title '$key1' with lines ls 3, '$file_b' using $tag1:$tag2 title '$key2' with lines ls 4\n";

    print GP "plot [0:$max_sen1] [$min_spe1:1.0+0.2*(1.0-$min_spe1)/0.99]'$file_a' using $tag1:$tag2 title '$key_a' with lines ls 3, '$cutoff_file_a' using 1:2 title '$key_a1_c' ls 33, '$file_b' using $tag1:$tag2 title '$key_b' with lines ls 4, '$cutoff_file_b' using 1:2 title '$key_b1_c' ls 44\n";

    #print GP "plot [0:$max_sen2] [$min_spe2:1.0+0.2*(1.0-$min_spe2)/0.99]'$file_a' using $tag1:$tag3 title '$key_a' with lines ls 3, '$cutoff_file_a' using 3:4 title '$key_a2_c' ls 33, '$file_b' using $tag1:$tag3 title '$key_b' with lines ls 4, '$cutoff_file_b' using 3:4 title '$key_b2_c' ls 44\n";
    close(GP);

    system ("gv -landscape -magstep -2 $outplot&\n");

    system("rm $cutoff_file_a\n");
    system("rm $cutoff_file_b\n");

}

sub plot_roc_together3 {
    
    my ($roc_type, $file_a, $file_b, $file_c) = @_;
    
    my $tag1;
    my $tag2;
    my $tag3;
    if    ($roc_type =~ /^RNA$/) { $tag1 = "8";  $tag2 = "9";  $tag3 = "10";  }
    elsif ($roc_type =~ /^COD$/) { $tag1 = "11"; $tag2 = "12"; $tag3 = "13"; }
    else                         { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

    my $title;
    my $xlabel = "SENSITIVITY";
    my $ylabel = "SPECIFICITY";
    
    my $key1 = read_head("$file_a", \$title);
    my $key2 = read_head("$file_b", \$title);
    my $key3 = read_head("$file_c", \$title);
    #print "$file_a ++ $key1\n$file_b ++ $key2\n";

    if ($opt_u) { $title .= "Score_cutoff = $sc_limit"; }

    my $ea1 = read_error_area("$file_a");
    my $ea2 = read_error_area("$file_b");
    my $ea3 = read_error_area("$file_c");

    my $max_sen1 = read_max_sen (1, $roc_type, "$file_a");
    my $max_sen2 = read_max_sen (1, $roc_type, "$file_b");
    my $max_sen3 = read_max_sen (1, $roc_type, "$file_c");

    my $max_sen = $max_sen1;
    if ($max_sen2 > $max_sen) { $max_sen = $max_sen2; }
    if ($max_sen3 > $max_sen) { $max_sen = $max_sen3; }

    if ($opt_j) { $max_sen = $opt_j; }

    my $min_spe1 = read_min_spe (1, $roc_type, "$file_a");
    my $min_spe2 = read_min_spe (1, $roc_type, "$file_b");
    my $min_spe3 = read_min_spe (1, $roc_type, "$file_c");

    my $min_spe = $min_spe1;
    if ($min_spe2 < $min_spe) { $min_spe = $min_spe2; }
    if ($min_spe3 < $min_spe) { $min_spe = $min_spe3; }

    $key1 .= " ** Error Area = $ea1";
    $key2 .= " ** Error Area = $ea2";
    $key3 .= " ** Error Area = $ea3";

    my $outplot = "$file_a\_three.ps";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";
    
    #print GP "set terminal postscript  default color solid 14\n";
    print GP "set terminal postscript  default color 14\n";
    print GP "set linestyle 1 lt 1 lw 4\n";
    print GP "set linestyle 2 lt 2 lw 4\n";
    print GP "set linestyle 3  lt 3 lw 3\n";
    print GP "set linestyle 4 lt 4 lw 4\n";
    print GP "set linestyle 5 lt 5 lw 4\n";
    print GP "set linestyle 6 lt 6 lw 4\n";
    print GP "set linestyle 7 lt 7 lw 4\n";
    print GP "set linestyle 8 lt 8 lw 4\n";
    print GP "set linestyle 9 lt 9 lw 4\n";
    print GP "set linestyle 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set linestyle 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set linestyle 33 lt 3 lw 2 pt 3 ps 1.2\n";
    print GP "set linestyle 44 lt 4 lw 2 pt 4 ps 1.2\n";
    print GP "set linestyle 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set linestyle 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set linestyle 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set linestyle 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set linestyle 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set title '$title'\n"; 
    print GP "set ylabel '$ylabel'\n";
    print GP "set xlabel '$xlabel'\n";

    print GP "plot [0:$max_sen] [$min_spe:1.0+0.2*(1.0-$min_spe)/0.99]  '$file_b' using $tag1:$tag2 title '$key2' with lines ls 3, '$file_c' using $tag1:$tag2 title '$key3' with lines ls 2, '$file_a' using $tag1:$tag2 title '$key1' with lines ls 1\n";

    #print GP "plot [0:$max_sen] [$min_spe:1.0+0.2*(1.0-$min_spe)/0.99]  '$file_b' using $tag1:$tag3 title '$key2' with lines ls 3, '$file_c' using $tag1:$tag3 title '$key3' with lines ls 2, '$file_a' using $tag1:$tag3 title '$key1' with lines ls 1\n";
    close(GP);

    system ("gv -landscape -magstep -2 $outplot&\n");
}


sub read_error_area {

    my ($file) = @_;

    my $ea;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#\# error area = (\S+)/) { 
	    $ea = $1;
	}
    }
    
    if ($ea =~ /^\d+\.\d$/) { $ea .= "0"; }

    close (FILE);

    return $ea;

}

sub read_head {

    my ($file, $title_ref) = @_;

    my $title;
    my $head;
    my $head_a;
    my $head_b;
    my $n_ali = 0;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#\# (.+)/) { 
	    $head = $1; last;
	}
    }

    #print "HEAD $head\n";
    if    ($head =~ /.+\/(\S+)\s+(.+)/) { $head_a = $1; $head_b = $2; }
    elsif ($head =~ /(\S+)\s+(.+)/)     { $head_a = $1; $head_b = $2; }

    #print "HEAD_b$head_b\n";
    if ($head_b =~ /total=\s*(\d+)/) { $n_ali = $1; }

    #print "HEAD_a $head_a\n";
    if    ($head_a =~ /^(\S+)\.stk/) { $title = $1; }
    elsif ($head_a =~ /^(\S+)\.fa/)  { $title = $1; }
    else                             { $title = $head_a; }

    if ($head_a =~ /\.blast/)     { $title .= " ** $n_ali BLASTN ALIGNMENTS"; }
    else                          { $title .= " ** $n_ali STRUCTURAL ALIGNMENTS"; }

    if    ($head_a =~ /\.qrna/)   { $head_a = "QRNA 1.2b";   }
    elsif ($head_a =~ /\.eqrna/)  { $head_a = "eQRNA 2.0.3"; }
    elsif ($head_a =~ /\.ddbrna/) { $head_a = "ddbRNA";      }
    elsif ($head_a =~ /\.rnaz/)   { $head_a = "RNAz";        }
    elsif ($head_a =~ /\.msari/)  { $head_a = "MSARI";       }

    close (FILE);

    $$title_ref = $title;

    return $head_a;

}

sub read_max_sen {

    my ($tag, $roc_type, $file) = @_;

    my $max_sen;
    my $spe;

    my $rna_sen;
    my $cod_sen;
    my $rna_spe;
    my $rna_spe1;
    my $rna_spe2;
    my $cod_spe;
    my $cod_spe1;
    my $cod_spe2;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#/) { 
	    next;
	}
	elsif (/\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
	    $rna_sen  = $1;
	    $rna_spe1 = $2;
	    $rna_spe2 = $3;

	    $cod_sen  = $4;
	    $cod_spe1 = $5;
	    $cod_spe2 = $6;

	    if    ($tag =~ /^1$/) { $rna_spe = $rna_spe1; $cod_spe = $cod_spe1; }
	    elsif ($tag =~ /^2$/) { $rna_spe = $rna_spe2; $cod_spe = $cod_spe2; }
	    else { print "your choices are '1' or '2' not $tag\n"; die; }

	    if ($roc_type =~ /^RNA$/) {
		$max_sen = $rna_sen;
		$spe     = $rna_spe;
	    }
	    elsif ($roc_type =~ /^COD$/) {
		$max_sen = $cod_sen;
		$spe     = $cod_spe;
	    }
	    else { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

	    if ($opt_s) { if ($spe >= $sp_limit) { last; } }
	    else { last; }
	}
    }

    close (FILE);

    #print "MAX SEN $max_sen SPE $spe SPE_LIMIT $sp_limit\n";
    return $max_sen;

}

sub read_target_sen {

    my ($tag, $roc_type, $file, $target_spe) = @_;

    my $ret_sen = -1.0;
    my $prv_sen = -1.0;
    my $prv_spe = -1.0;
    my $prv_cut = -1.0;

    my $rna_sen;
    my $cod_sen;

    my $rna_spe;
    my $rna_spe1;
    my $rna_spe2;
    my $cod_spe;
    my $cod_spe1;
    my $cod_spe2;

    my $sen;
    my $spe;
    my $cutoff;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#/) { 
	    next;
	}
	elsif (/(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {

	    $cutoff = $1;

	    $rna_sen  = $2;
	    $rna_spe1 = $3;
	    $rna_spe2 = $4;

	    $cod_sen  = $5;
	    $cod_spe1 = $6;
	    $cod_spe2 = $7;

	    if    ($tag =~ /^1$/) { $rna_spe = $rna_spe1; $cod_spe = $cod_spe1; }
	    elsif ($tag =~ /^2$/) { $rna_spe = $rna_spe2; $cod_spe = $cod_spe2; }
	    else { print "your choices are '1' or '2' not $tag\n"; die; }

	    if ($roc_type =~ /^RNA$/) {
		$sen = $rna_sen;
		$spe = $rna_spe;
	    }
	    elsif ($roc_type =~ /^COD$/) {
		$sen = $cod_sen;
		$spe = $cod_spe;
	    }
	    else { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

	    $sen = int($sen*10000)/10000;
	    $spe = int($spe*10000)/10000;

	    #print "T_SPE $sen $spe\n";
	    if ($spe > $target_spe) { 
		if ($prv_sen > -1.0) { $ret_sen = 0.5 * ($sen + $prv_sen); }
		else                 { $ret_sen = $sen;                    }

		$ret_sen .= " ** $sen [$spe:$cutoff]"."-"."$prv_sen [$prv_spe:$prv_cut]"; 
		last; 
	    }

	    $prv_sen = $sen;
	    $prv_spe = $spe;
	    $prv_cut = $cutoff;
	}
    }

    close (FILE);

    return $ret_sen;

}

sub read_min_spe {

    my ($tag, $roc_type, $file) = @_;

    my $min_spe;
    my $rna_spe;
    my $rna_spe1;
    my $rna_spe2;
    my $cod_spe;
    my $cod_spe1;
    my $cod_spe2;

    if ($opt_s) { $min_spe = $sp_limit; }
    else {
	open (FILE,"$file") || die;
	while (<FILE>) {
	    
	    if (/^\#/) { 
		next;
	    }
	    elsif (/\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+(\S+)\s+(\S+)/) {

		$rna_spe1 = $1;
		$rna_spe2 = $2;
		$cod_spe1 = $3;
		$cod_spe2 = $4;

		if    ($tag =~ /^1$/) { $rna_spe = $rna_spe1; $cod_spe = $cod_spe1; }
		elsif ($tag =~ /^2$/) { $rna_spe = $rna_spe2; $cod_spe = $cod_spe2; }
		else { print "your choices are '1' or '2' not $tag\n"; die; }

		if ($roc_type =~ /^RNA$/) {
		    $min_spe = $rna_spe;
		}
		elsif ($roc_type =~ /^COD$/) {
		    $min_spe = $cod_spe;
		}
		else { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }
		
		last;
	    }
	}
	
	close (FILE);
    }
    
    return $min_spe;   
}

sub read_target_spe {

    my ($tag, $roc_type, $file, $target_sen) = @_;

    my $ret_spe = -1.0;
    my $prv_spe = -1.0;
    my $prv_sen = -1.0;
    my $prv_cut = -1.0;

    my $rna_sen;
    my $cod_sen;

    my $rna_spe;
    my $rna_spe1;
    my $rna_spe2;
    my $cod_spe;
    my $cod_spe1;
    my $cod_spe2;

    my $spe;
    my $sen;
    my $cutoff;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#/) { 
	    next;
	}
	elsif (/(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
	    $cutoff = $1;

	    $rna_sen  = $2;
	    $rna_spe1 = $3;
	    $rna_spe2 = $4;

	    $cod_sen  = $5;
	    $cod_spe1 = $6;
	    $cod_spe2 = $7;
	    
	    if    ($tag =~ /^1$/) { $rna_spe = $rna_spe1; $cod_spe = $cod_spe1; }
	    elsif ($tag =~ /^2$/) { $rna_spe = $rna_spe2; $cod_spe = $cod_spe2; }
	    else { print "your choices are '1' or '2' not $tag\n"; die; }
	    
	    if ($roc_type =~ /^RNA$/) {
		$sen = $rna_sen;
		$spe = $rna_spe;
	    }
	    elsif ($roc_type =~ /^COD$/) {
		$sen = $cod_sen;
		$spe = $cod_spe;
	    }
	    else { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

	    $sen = int($sen*10000)/10000;
	    $spe = int($spe*10000)/10000;

	    #print "T_SEN $sen $spe\n";
	    if ($sen < $target_sen) { 
		if ($prv_spe > -1.0) { $ret_spe = 0.5 * ($prv_spe + $spe); }
		else                 { $ret_spe = $spe;                    }

		$ret_spe .= " ** $spe [$sen:$cutoff]"."-"."$prv_spe [$prv_sen:$prv_cut]"; 
		last; 
	    }

	    $prv_sen = $sen;
	    $prv_spe = $spe;
	    $prv_cut = $cutoff;
	}
    }

    close (FILE);

    return $ret_spe;

}

sub roc_file {

    my ($roc_type, $file, $filesh, $roc_file, $type, 
	$min_cutoff, $max_cutoff, $inc_cutoff, 
	$min_id, $max_id, $min_gc, $max_gc, $max_n_ali,
	$ret_total, $ret_total_sh) = @_;

    my $total    = 0;
    my $total_sh = 0;
    my @rna;
    my @rna_sh;
    my @cod;
    my @cod_sh;
    
    my $error_area;

    my $rna_sen_zero  = -1;
    my $rna_spe1_zero = -1;
    my $rna_spe2_zero = -1;
    my $cod_sen_zero  = -1;
    my $cod_spe1_zero = -1;
    my $cod_spe2_zero = -1;

    for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) { 
	$rna[$cutoff]    = 0;
	$cod[$cutoff]    = 0;
	$rna_sh[$cutoff] = 0;
	$cod_sh[$cutoff] = 0;
    }
    
    open (OUT,">$roc_file") || die;
    
    if ($type =~ /^qrna$/) {
	parse_qrna ("$file",   \@rna,    \@cod,    \$total,    $min_cutoff, $max_cutoff, $inc_cutoff, $min_id, $max_id, $min_gc, $max_gc, $max_n_ali);
	parse_qrna ("$filesh", \@rna_sh, \@cod_sh, \$total_sh, $min_cutoff, $max_cutoff, $inc_cutoff, $min_id, $max_id, $min_gc, $max_gc, $max_n_ali);
    }
    elsif ($type =~ /^ddbrna$/) {
	parse_ddbrna ("$file",   \@rna,    \$total,    $min_cutoff, $max_cutoff, $inc_cutoff, $max_n_ali);
	parse_ddbrna ("$filesh", \@rna_sh, \$total_sh, $min_cutoff, $max_cutoff, $inc_cutoff, $max_n_ali);
    }
    elsif ($type =~ /^RNAz$/) {
	parse_RNAz ("$file",   \@rna,    \$total,    $min_cutoff, $max_cutoff, $inc_cutoff, $min_id, $max_id, $max_n_ali);
	parse_RNAz ("$filesh", \@rna_sh, \$total_sh, $min_cutoff, $max_cutoff, $inc_cutoff, $min_id, $max_id, $max_n_ali);
    }
    
    if (0) {
	print "TOTAL $total\n";
	for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
	    print "CUTOFF ", $cutoff/$inc_cutoff+$min_cutoff, " $rna[$cutoff] $cod[$cutoff]\n";
	}
	print "TOTAL_SH $total_sh\n";
	for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
	    print "CUTOFF ", $cutoff/$inc_cutoff+$min_cutoff, " $rna_sh[$cutoff] $cod_sh[$cutoff]\n";
	}
    }

    if ($roc_type =~ /^RNA$/) {
	$error_area = cal_error_area(\@rna, \@rna_sh, $total, $total_sh, $min_cutoff, $max_cutoff, $inc_cutoff); 
    }
    elsif ($roc_type =~ /^COD$/) {
	$error_area = cal_error_area(\@cod, \@cod_sh, $total, $total_sh, $min_cutoff, $max_cutoff, $inc_cutoff); 
    }
    else { print "your choices are 'RNA' or 'COD' not $roc_type\n"; die; }

    print OUT "## $file total=$total total_sh=$total_sh\n";
    print OUT "## error area = $error_area \n";
    for (my $cutoff = 0; $cutoff <= $inc_cutoff*($max_cutoff-$min_cutoff); $cutoff ++) {
	
	if ($cutoff/$inc_cutoff >= $sc_limit-$min_cutoff) {
	    
	    print OUT $cutoff/$inc_cutoff+$min_cutoff, "\t$rna[$cutoff]\t$cod[$cutoff]\t$total\t$rna_sh[$cutoff]\t$cod_sh[$cutoff]\t$total_sh\t";
	    
	    if ($total > 0 && $total_sh > 0) {
		print OUT $rna[$cutoff]/$total, "\t", 
		1.0-($rna_sh[$cutoff]/$total_sh), "\t", 
		1.0-($rna_sh[$cutoff]/$total_sh)-($cod[$cutoff]/$total), "\t", 
		$cod[$cutoff]/$total, "\t",
		1.0-($cod_sh[$cutoff]/$total_sh),"\t",
		1.0-($cod_sh[$cutoff]/$total_sh)-($rna[$cutoff]/$total),"\n";
	    }
	    else { 	
		print OUT "\n";
	    }

	    if ($total > 0 && $cutoff/$inc_cutoff+$min_cutoff == 0) { 
		$rna_sen_zero = $rna[$cutoff]/$total; 
		$cod_sen_zero = $cod[$cutoff]/$total; 
		$rna_spe1_zero = 1.0-($rna_sh[$cutoff]/$total_sh); 
		$rna_spe2_zero = 1.0-($rna_sh[$cutoff]/$total_sh) - $cod[$cutoff]/$total; 
		$cod_spe1_zero = 1.0-($cod_sh[$cutoff]/$total_sh); 
		$cod_spe2_zero = 1.0-($cod_sh[$cutoff]/$total_sh) - $rna[$cutoff]/$total; 
	    }
	}
	
    }
    close (OUT);
    
    print "\nfile   $file\nfilesh $filesh\n";
    print "total    = $total\n";
    print "total_sh = $total_sh\n";

    if ($total != $total_sh) {
	print "\nWarning: different number of alignments in file $file\ntotal = $total, total_sh = $total_sh\n"; 
    }
    
    $rna_sen_zero  = int($rna_sen_zero*10000)/10000;
    $rna_spe1_zero = int($rna_spe1_zero*10000)/10000;
    $rna_spe2_zero = int($rna_spe2_zero*10000)/10000;
    $cod_sen_zero  = int($cod_sen_zero*10000)/10000;
    $cod_spe1_zero = int($cod_spe1_zero*10000)/10000;
    $cod_spe2_zero = int($cod_spe2_zero*10000)/10000;

    if ($roc_type =~ /^RNA$/) {
	print "CUTOFF_ZERO sen = $rna_sen_zero spe1 = $rna_spe1_zero spe2 = $rna_spe2_zero\n"; 
    }
    elsif ($roc_type =~ /^COD$/) {
	print "CUTOFF_ZERO sen = $cod_sen_zero spe1 = $cod_spe1_zero spe2 = $cod_spe2_zero\n"; 
    }

    my $scale = 1000;
    my $jump = 1;
    my $target_spe = 0.990;
    for (my $s = $scale; $s >= $target_spe*$scale; ) {
	my $spe = $s/$scale;
	print "SENSITIVITY for $spe\tSPECIFICITY = ", read_target_sen("1", $roc_type, $roc_file, $spe), "\n";
	#print "SENSITIVITY for $spe\tSPECIFICITY = ", read_target_sen("2", $roc_type, $roc_file, $spe), "\n";
	$s -= $jump;
    }

    my $target_sen = 0.60;
    print "\nSPECIFICITY for $target_sen\tSENSITIVITY = ", read_target_spe(1, $roc_type, $roc_file, $target_sen), "\n";
    #print "SPECIFICITY for $target_sen\tSENSITIVITY = ", read_target_spe(2, $roc_type, $roc_file, $target_sen), "\n";

    plot_roc($roc_type, $roc_file);
    
    $$ret_total    = $total;
    $$ret_total_sh = $total_sh;
}

sub sens_and_spec_at_cutoff {

    my ($tag, $file, $type, $target_cutoff, $sen_ref, $spe_ref) = @_;

    my $sen = -1;
    my $spe = -1;
    my $cutoff;

    my $type_sen;
    my $type_spe;

    open (FILE,"$file") || die;
    while (<FILE>) {

	if (/^\#/) { 
	    next;
	}
	elsif (/(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
	    $cutoff = $1;

	   # print "CUTOFF $cutoff\n";

	    my $rna_sen  = $2;
	    my $rna_spe1 = $3;
	    my $rna_spe2 = $4;
	    my $cod_sen  = $5;
	    my $cod_spe1 = $6;
	    my $cod_spe2 = $7;

	    my $rna_spe;
	    my $cod_spe;

	    if    ($tag =~ /^1$/) { $rna_spe = $rna_spe1; $cod_spe = $cod_spe1; }
	    elsif ($tag =~ /^2$/) { $rna_spe = $rna_spe2; $cod_spe = $cod_spe2; }
	    else { print "your choices are '1' or '2' not $tag\n"; die; }

	    if ($type =~ /^RNA$/) {
		$type_sen = $rna_sen;
		$type_spe = $rna_spe;
	    }
	    elsif ($type =~ /^COD$/) {
		$type_sen = $cod_sen;
		$type_spe = $cod_spe;
	    }

	    if ($cutoff <= $target_cutoff) { 
		$sen = $type_sen;
		$spe = $type_spe;

		#print "SEN $cutoff $sen $spe\n";
	    }


	    $$sen_ref = $sen;
	    $$spe_ref = $spe;
	}
    }

    close (FILE);
    
    #print "T_SEN $target_cutoff $sen $spe\n";

    $$sen_ref = $sen;
    $$spe_ref = $spe;
}
