% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lf.vd.R
\name{lf.vd}
\alias{lf.vd}
\title{Construct a VDFR regression term}
\usage{
lf.vd(
  X,
  argvals = seq(0, 1, l = ncol(X)),
  vd = NULL,
  integration = c("simpson", "trapezoidal", "riemann"),
  L = NULL,
  basistype = c("s", "te", "t2"),
  transform = NULL,
  mp = TRUE,
  ...
)
}
\arguments{
\item{X}{matrix containing variable-domain functions. Should be \eqn{N x J},
where \eqn{N} is the number of subjects and \eqn{J} is the maximum number of time
points per subject. Most rows will have \code{NA} values in the right-most
columns, corresponding to unobserved time points.}

\item{argvals}{indices of evaluation of \code{X}, i.e. \eqn{(t_{i1},.,t_{iJ})} for
subject \eqn{i}. May be entered as either a length-\code{J} vector, or as
an \code{N} by \code{J} matrix. Indices may be unequally spaced. Entering
as a matrix allows for different observations times for each subject.}

\item{vd}{vector of values of containing the variable-domain width (\eqn{T_i}
above). Defaults to the \code{argvals} value corresponding to the last
non-\code{NA} element of \eqn{X_i(t)}.}

\item{integration}{method used for numerical integration. Defaults to \code{"simpson"}'s rule
for calculating entries in \code{L}. Alternatively and for non-equidistant grids,
\code{"trapezoidal"} or \code{"riemann"}.}

\item{L}{an optional \code{N} by \code{ncol(argvals)} matrix giving the weights for the numerical
integration over \code{t}. If present, overrides \code{integration}.}

\item{basistype}{character string indicating type of bivariate basis used.
Options include \code{"s"} (the default), \code{"te"}, and \code{"t2"},
which correspond to \code{mgcv::s}, \code{mgcv::te}, and \code{mgcv::t2}.}

\item{transform}{character string indicating an optional basis transformation;
see Details for options.}

\item{mp}{for \code{transform=="linear"} or \code{transform=="quadratic"},
\code{TRUE} to use multiple penalties for the smooth (one for each marginal
basis). If \code{FALSE}, penalties are concatonated into a single
block-diagonal penalty matrix (with one smoothing parameter).}

\item{...}{optional arguments for basis and penalization to be passed to the
function indicated by \code{basistype}. These could include, for example,
\code{"bs"}, \code{"k"}, \code{"m"}, etc. See \code{\link{te}} or
\code{\link{s}} for details.}
}
\value{
a list with the following entries
   \item{call}{a \code{call} to \code{s} or \code{te}, using the appropriately constructed
     weight matrices}
   \item{data}{data used by the \code{call}, which includes the matrices indicated
     by \code{argname}, \code{Tindname}, and \code{LXname}}
   \item{L}{the matrix of weights used for the integration}
   \item{argname}{the name used for the \code{argvals} variable in the \code{formula}
     used by \code{mgcv::gam}}
   \item{Tindname}{the name used for the \code{Tind} variable in the \code{formula}
     used by \code{mgcv::gam}}
   \item{LXname}{the name of the \code{by} variable used by \code{s} or \code{te}
     in the \code{formula} for \code{mgcv::gam}}
}
\description{
This function defines the a variable-domain functional regression term
for inclusion in an \code{\link[mgcv]{gam}}-formula (or \code{\link[mgcv]{bam}} or
\code{\link[mgcv]{gamm}} or \code{gamm4::gamm} as constructed by
\code{\link{pfr}}. These are functional predictors for which each function is
observed over a domain of different width.
The default is the term \eqn{1/T_i\int_0^{T_i}X_i(t)\beta(t,T_i)dt},
where \eqn{X_i(t)} is a functional predictor of length \eqn{T_i} and \eqn{\beta(t,T_i)}
is an unknown bivariate coefficient function. Various domain transformations
are available, such as lagging or domain-standardizing the coordinates, or
parameterizing the interactions; these often result in improved model fit.
Basis choice is fully customizable using the options of
\code{\link[mgcv]{s}} and \code{\link[mgcv]{te}}.
}
\details{
The variable-domain functional regression model uses the term
   \eqn{\frac1{T_i}\int_0^{T_i}X_i(t)\beta(t,T_i)dt} to incorporate a
   functional predictor with subject-specific domain width. This term imposes
   a smooth (nonparametric) interaction between \eqn{t} and \eqn{T_i}. The domain
   of the coefficient function is the triangular (or trapezoidal) surface
   defined by \eqn{{t,T_i: 0\le t\le T_i}}. The default basis uses
   bivariate thin-plate regression splines.
   
   Different basis transformations can result in different properties; see
   Gellar, et al. (2014) for a more complete description. We make five basis
   transformations easily accessible using the \code{transform} argument.
   This argument is a character string that can take one of the following
   values:
   \enumerate{
     \item \code{"lagged"}: transforms \code{argvals} to \code{argvals - vd}
     \item \code{"standardized"}: transforms \code{argvals} to \code{argvals/vd},
       and then rescales \code{vd} linearly so it ranges from 0 to 1
     \item \code{"linear"}: first transforms the domain as in
       \code{"standardized"}, then parameterizes the interaction with
       \code{"vd"} to be linear
     \item \code{"quadratic"}: first transforms the domain as in
       \code{"standardized"}, then parameterizes the interaction with
       \code{"vd"} to be quadratic
     \item \code{"noInteraction"}: first transforms the domain as in
       \code{"standardized"}, then reduces the bivariate basis to univariate
       with no effect of \code{vd}. This would be equivalent to using
       \code{\link{lf}} on the domain-standardized predictor functions.
   }
   
   The practical effect of using the \code{"lagged"} basis is to increase
   smoothness along the right (diagonal) edge of the resultant estimate.
   The practical effect of using a \code{"standardized"} basis is to allow
   for greater smoothness at high values of \eqn{T_i} compared to lower
   values.
   
   These basis transformations rely on the basis constructors
   available in the \code{mgcvTrans} package. For more specific control over
   the transformations, you can use \code{bs="dt"} and/or \code{bs="pi"};
   see \code{\link{smooth.construct.dt.smooth.spec}} or
   \code{\link{smooth.construct.pi.smooth.spec}} for an explanation of the
   options (entered through the \code{xt} argument of \code{lf.vd}/\code{s}).
   
   Note that tensor product bases are only recommended when a standardized
   transformation is used. Without this transformation, just under half of
   the "knots" used to define the basis will fall outside the range of the
   data and have no data available to estimate them. The penalty allows
   the corresponding coefficients to be estimated, but results may be
   unstable.
}
\examples{
\dontrun{
  data(sofa)
  fit.vd1 <- pfr(death ~ lf.vd(SOFA) + age + los,
                 family="binomial", data=sofa)
  fit.vd2 <- pfr(death ~ lf.vd(SOFA, transform="lagged") + age + los,
                 family="binomial", data=sofa)
  fit.vd3 <- pfr(death ~ lf.vd(SOFA, transform="standardized") + age + los,
                 family="binomial", data=sofa)
  fit.vd4 <- pfr(death ~ lf.vd(SOFA, transform="standardized",
                               basistype="te") + age + los,
                 family="binomial", data=sofa)
  fit.vd5 <- pfr(death ~ lf.vd(SOFA, transform="linear", bs="ps") + age + los,
                 family="binomial", data=sofa)
  fit.vd6 <- pfr(death ~ lf.vd(SOFA, transform="quadratic", bs="ps") + age + los,
                 family="binomial", data=sofa)
  fit.vd7 <- pfr(death ~ lf.vd(SOFA, transform="noInteraction", bs="ps") + age + los,
                 family="binomial", data=sofa)
  
  ests <- lapply(1:7, function(i) {
    c.i <- coef(get(paste0("fit.vd", i)), n=173, n2=173) 
    c.i[(c.i$SOFA.arg <= c.i$SOFA.vd),]
  })
  
  # Try plotting for each i
  i <- 1
  lims <- c(-2,8)
  if (requireNamespace("ggplot2", quietly = TRUE) &
      requireNamespace("RColorBrewer", quietly = TRUE)) {
        est <- ests[[i]]
        est$value[est$value<lims[1]] <- lims[1]
        est$value[est$value>lims[2]] <- lims[2]
        ggplot2::ggplot(est, ggplot2::aes(SOFA.arg, SOFA.vd)) +
          ggplot2::geom_tile(ggplot2::aes(colour=value, fill=value)) +
          ggplot2::scale_fill_gradientn(  name="", limits=lims,
                    colours=rev(RColorBrewer::brewer.pal(11,"Spectral"))) +
          ggplot2::scale_colour_gradientn(name="", limits=lims,
                    colours=rev(RColorBrewer::brewer.pal(11,"Spectral"))) +
          ggplot2::scale_y_continuous(expand = c(0,0)) +
          ggplot2::scale_x_continuous(expand = c(0,0)) +
          ggplot2::theme_bw()
  }
}
  
}
\references{
Gellar, Jonathan E., Elizabeth Colantuoni, Dale M. Needham, and
   Ciprian M. Crainiceanu. Variable-Domain Functional Regression for Modeling
   ICU Data. Journal of the American Statistical Association,
   109(508):1425-1439, 2014.
}
\seealso{
\code{\link{pfr}}, \code{\link{lf}}, mgcv's
   \code{\link{linear.functional.terms}}.
}
\author{
Jonathan E. Gellar <JGellar@mathematica-mpr.com>
}
