#' @title Solution path generation via the Wild Contrast Maximisation method
#' @description This function arranges all possible change-points in the mean of the input vector in the order of importance, via the Wild Binary Segmentation 2 method.
#' @details The Wild Contrast Maximisation (WCM) algorithm generates a nested sequence of candidate models by identifying large gaps in the solution path generated by WBS2, which aids the model selection step in the presence of large random fluctuations due to serial dependence. See Cho and Fryzlewicz (2023) for further details.
#' @param x A numeric vector containing the data to be processed.
#' @param type The type of change-point models fitted to the data; currently the class of piecewise constant signals (\code{type = "const"}) is supported.
#' @param M The maximum number of data sub-samples drawn at each recursive stage of the algorithm. The default is \code{M = 100}. 
#' @param min.d The minimum distance between candidate change-point estimators; 
#' if \code{min.d = NULL}, it is set to be \code{max(20, 10 + ceiling(log(length(x))^1.1)}.
#' @param Q The maximum number of allowable change-points. 
#' The default is \code{Q = floor(log(length(x))^1.9)}.
#' @param max.iter The maximum number of candidate change-point models considered; if a model with the number of change-point estimators exceeding \code{Q} is required to generate the sequence of required candidate models, this argument is ignored. The default is \code{max.iter = 5}.
#' @return An S3 object of class \code{cptpath}, which contains the following fields: 
#' \item{solutions.nested}{\code{TRUE}, i.e., the change-point outputs are nested}
#' \item{solution.path}{Locations of possible change-points in the mean of \code{x}, arranged in decreasing order of change-point importance; this is not used by \code{\link{model.gsa}}}
#' \item{solution.set}{A list of candidate change-point models. Each model contains possible change-points in the mean of \code{x}; this is used by \code{\link{model.gsa}}}
#' \item{x}{Input vector \code{x}}
#' \item{type}{The type of the change-point model considered, which has value "const" here}
#' \item{M}{Input parameter \code{M}}
#' \item{cands}{Matrix of dimensions \code{Q} by 4. The first two columns are (start, end)-points of the detection intervals of the corresponding possible change-point location in the third column. The fourth column is a measure of strength of the corresponding possible change-point. The order of the rows is the same as the order returned in \code{solution.path}}
#' \item{method}{The method used, which has value "wcm" here}
#' @references H. Cho & P. Fryzlewicz (2024) Multiple change point detection under serial dependence: Wild contrast maximisation and gappy Schwarz algorithm. \emph{Journal of Time Series Analysis}, 45(3): 479--494.
#' @seealso \code{\link{model.gsa}}
#' @examples
#' set.seed(111)
#' f <- rep(c(0, 5, 2, 8, 1, -2), c(100, 200, 200, 50, 200, 250))
#' x <- f + arima.sim(list(ar = c(.75, -.5), ma = c(.8, .7, .6, .5, .4, .3)), n = length(f), sd = 1)
#' sol.wcm(x)$solution.set
#' @export
sol.wcm <- function(x, type = 'const', M = 100, min.d = NULL, Q = floor(log(length(x))^1.9), max.iter = 5) {

  if(type != 'const') stop("Currently, sol.wcm only supports the piecewise constant model type")
  
  n <- length(x)
  if(is.null(min.d)) min.d <- max(20, 10 + ceiling(log(n)^1.1))
  if(n < 2 * min.d + 1) stop("x is too short for sol.wcm, or min.d is too large")
  double.cusum <- FALSE
  
  res <- t(wcm(x, M, min.d, 1))
  res <- res[order(- res[, 4]), ]
  colnames(res) <- c('s', 'e', 'b', 'cusum')
  
  Q <- min(Q, dim(res)[1])
  if(Q < 1) stop("x is too short for sol.wcm, or min.d is too large")
  z <- log(res[1:Q, 4])
  cs <- cumsum(z[1:Q])
  ecp.seq <- res[1:Q, 3]
  
  ecp.list <- list()
  current <- c()
  niter <- ii <- 0
  if(!double.cusum){
    for(k in sort(sort(abs(diff(z)), decreasing = TRUE, index.return = TRUE)$ix[1:max.iter])){
      new.cp <- sort(ecp.seq[(ii + 1):k])
      current <- sort(c(current, new.cp))
      ecp.list <- c(ecp.list, list(current))
      ii <- k
    }
  } else{
    k0 <- which.max(abs(diff(z[(ii + 1):Q]))) + ii 
    while(ii < Q - 1 & niter < max.iter){
      if(double.cusum){
        eval <- ((Q - ii)/(1:(Q - ii - 1)) * ((Q - ii - 1):1))^.5 * (cs[(ii + 1):(Q - 1)] - (ii > 0)*cs[max(1, ii)] - (cs[Q] - (ii > 0)*cs[max(1, ii)])/(Q - ii) * (1:(Q - ii - 1)))
        k <- which.max(eval) + ii
      } else k <- k0
      new.cp <- sort(ecp.seq[(ii + 1):k])
      current <- sort(c(current, new.cp))
      ecp.list <- c(ecp.list, list(current))
      ii <- k
      k0 <- which.max(abs(diff(z[(ii + 1):Q]))) + ii 
      niter <- niter + 1
    }
  }  
  
  ret <- list()	  
  ret$solutions.nested = TRUE
  ret$solution.path = ecp.seq
  ret$solution.set = ecp.list
  ret$x = x
  ret$type = "const"
  ret$M = M
  ret$cands = res[1:Q,, drop = FALSE]  
  ret$method = "wcm"
  class(ret) <- "cptpath"
  
  return(ret)
  
}
  
#' @title Estimating change-points in the piecewise-constant mean of a noisy data sequence with auto-regressive noise via gappy Schwarz algorithm
#' @description This function estimates the number and locations of change-points in the piecewise-constant mean of a noisy data sequence with auto-regressive noise via gappy Schwarz algorithm
#' from a candidate model sequence generated by \code{sol.wcm}.
#' @details From the largest to the smallest (i.e. empty) candidate models generated by \code{sol.wcm}, 
#' gappy Schwarz algorithm locally evaluates the Schwarz criterion (SC, under piecewise constant signal + AR(p) noise model, with the AR order p to be determined adaptively) and its modification SC0
#' on each segment determined by the next smallest candidate model. It selects the larger model as the final model if over each segment, all newly introduced estimators are deemed `significant' according to SC and SC0; see Cho and Fryzlewicz (2023) for details.
#' @param cptpath.object A solution-path object, returned by a \code{sol.wcm} routine. Note that the field \code{cptpath.object$x} contains the input data sequence. 
#' @param p.max The maximum AR order. The default is \code{p.max = 10}. 
#' @param pen Penalty used for the Schwarz criterion. \code{log(length(cptpath.object$x))^1.01} is used as default. 
#' @return An S3 object of class \code{cptmodel}, which contains the following fields: 
#' \item{solution.path}{The solution path method used to obtain \code{cptpath.object}, here its value is \code{"wcm"}}
#' \item{model.selection}{The model selection method used to return the final change-point estimators object, here its value is \code{"gsa"}}
#' \item{no.of.cpt}{The number of estimated change-points in the piecewise-constant mean of the vector \code{cptpath.object$x}}
#' \item{cpts}{The locations of estimated change-points in the piecewise-constant mean of the vector \code{cptpath.object$x}. These are the end-points of the corresponding constant-mean intervals}
#' \item{est}{An estimate of the piecewise-constant mean of the vector \code{cptpath.object$x}; the values are the sample means of the data (replicated a suitable number of times) between each pair of consecutive detected change-points}
#' @references H. Cho & P. Fryzlewicz (2024) Multiple change point detection under serial dependence: Wild contrast maximisation and gappy Schwarz algorithm. \emph{Journal of Time Series Analysis}, 45(3): 479--494.
#' @seealso \code{\link{sol.wcm}}
#' @examples 
#' set.seed(111)
#' f <- rep(c(0, 5, 2, 8, 1, -2), c(100, 200, 200, 50, 200, 250))
#' x <- f + arima.sim(list(ar = c(.75, -.5), ma = c(.8, .7, .6, .5, .4, .3)), n = length(f), sd = 1)
#' model.gsa(sol.wcm(x))
#' @export
model.gsa <- function(cptpath.object, p.max = 10, pen = log(length(cptpath.object$x))^1.01){
  
  if(!("cptpath" %in%  class(cptpath.object))) stop("A cptmodel class object has to be supplied in the first argument.")
  if(cptpath.object$method != 'wcm') stop(paste0("model.gsa won't work on cptpath.object produced with solution generating functions other than sol.wcm"))
  
  mt <- cptpath.object$type
  if(mt != 'const') stop("Currently, model.gsa only supports the piecewise constant model type")
  
  x <- cptpath.object$x
  n <- length(x)
  ecp.list <- cptpath.object$solution.set
  
  
  ll <- length(ecp.list)
  while(ll >= 1){
    if(ll == 1) current <- numeric(0) else current <- ecp.list[[ll - 1]]
    new.cp <- setdiff(ecp.list[[ll]], current)
    
    out <- rep(0, length(current) + 1)
    brks <- c(0, current, n)
    for(jj in 1:(length(current) + 1)){
      int <- (brks[jj] + 1):brks[jj + 1]
      ecp <- new.cp[new.cp %in% int]
      if(length(ecp) > 0){
        loc.brks <- c(brks[jj], ecp, brks[jj + 1])
        if(length(int) > p.max + log(n) + length(ecp) + 1 & max(diff(loc.brks)) > p.max + 1 + log(n)){
          ms <- ms.sc(x, s = brks[jj], e = brks[jj + 1], ecp = ecp, p.max = p.max, pen = pen)
          if(ms$cp.sc >= ms$ncp.sc) out[jj] <- -1 else out[jj] <- 1
        } else out[jj] <- -1
      } 
    }
    if(any(out < 0)) flag <- FALSE else flag <- TRUE
    # else{
    #   rcm <- refine.chp.model(ecp.list[[ll]], x, c = .5)
    #   ms <- ms.sc(x, s = 0, e = n, ecp = rcm$rcp, p.max = p.max, pen = pen)
    #   if(ms$cp.sc >= ms$ncp.sc) flag <- FALSE 
    # } 
    if(flag) break
    ll <- ll - 1
  }
  
  if(ll == 0){
    cp <- integer(0) 
    # cp.info <- matrix(0, nrow = 0, ncol = 4)
    # colnames(cp.info) <- colnames(res)
  } else{
    cp <- ecp.list[[ll]]
    # rcp <- refine.chp.model(cp, x, c = .5)$rcp
    # cp.info <- res[1:length(cp),, drop = FALSE]
  }
  
  ret <- structure(list(solution.path = 'wcm',
                        model.selection = 'gsc',
                        no.of.cpt = length(cp),
                        cpts = cp, 
                        est = mean_from_cpt(x, cp)), 
                   class = 'cptmodel')

  return(ret)
}

###################

#' wcm
#' @keywords internal
#' @noRd
wcm <- function(x, M = 100, min.d = 0, top.scale = 0){
  
  n <- length(x)
  if (n <= 2 * min.d + 1) return(matrix(NA, 4, 0)) else {
    if(top.scale == 1){
      tmp <- rbind(systematic_cusums(x[1:round(n/2)], M = M, min.d = min.d)$max.val, 
                   systematic_cusums(x[round(n/4) + 1:round(n/2)], M = M, min.d = min.d)$max.val + c(rep(round(n/4), 3), 0), 
                   systematic_cusums(x[(round(n/2) + 1):n], M = M, min.d = min.d)$max.val + c(rep(round(n/2), 3), 0))
      cpt <- t(tmp[which.max(tmp[, 4]), , drop = FALSE])
    } else cpt <- t(systematic_cusums(x, M = M, min.d = min.d)$max.val)
    return(cbind(cpt, wcm(x = x[1:cpt[3]], M = M, min.d = min.d, top.scale = 0),
                 wcm(x = x[(cpt[3] + 1):n], M = M, min.d = min.d, top.scale = 0) + c(rep(cpt[3], 3), 0)))
  }
  
}

#' local model comparison routine in gsa
#' @keywords internal
#' @noRd
ms.sc <- function(x, s, e, ecp, p.max = 10, pen = NULL){
  
  n <- length(x)

  brks <- c(s, ecp, e)
  cp.lik.seq <- ncp.lik.seq <- cp.sc.seq <- ncp.sc.seq <- rep(0, p.max + 1)
  
  for(ii in 1:(p.max + 1)){
    pp <- ii - 1
    
    len <- e - max(pp, s)
    y <- x[(max(pp, s) + 1):e]
    R <- L <- c()
    if(pp > 0) for(kk in 1:pp) L <- cbind(L, x[(max(pp, s) + 1):e - kk])
    for(kk in 1:(length(brks) - 1)){
      if(brks[kk + 1] <= pp) next
      tmp <- rep(0, len)
      tmp[max(1, brks[kk] - max(pp, s) + 1):(brks[kk + 1] - max(pp, s))] <- 1
      R <- cbind(R, tmp)
    }
    X <- cbind(L, R)
    lm.fit1 <- stats::lm(y ~ 0 + X)
    cp.lik.seq[ii] <- loglik1 <- -len/2*log(2*pi*sum(resid(lm.fit1)^2)/len) - len/2
    cp.sc.seq[ii] <- (pp + length(ecp) + 1) * pen - loglik1
    
    if(pp > 0) z <- y - L %*% coef(lm.fit1)[1:pp] else z <- y
    
    ncp.lik.seq[ii] <- loglik2 <- -len/2*log(2*pi*sum((z - mean(z))^2)/len) - len/2
    ncp.sc.seq[ii] <- (pp + 1) * pen - loglik2
  }
 
  p <- which.min(cp.sc.seq[-1]) 

  out <- list(cp.sc = cp.sc.seq[p + 1], ncp.sc = ncp.sc.seq[p + 1], p = p,
              cp.lik = cp.lik.seq, ncp.lik = ncp.lik.seq,
              cp.sc.seq = cp.sc.seq, ncp.sc.seq = ncp.sc.seq)
  return(out)
}

#' #' Refinement of change point location estimators
#' #' @keywords internal 
#' #' @noRd
#' refine.chp.model <- function(ecp, x, c = .5){
#'   
#'   n <- length(x)
#'   rcp <- ecp * 0
#'   nres <- c()
#'   for(i in 1:length(ecp)){
#'     if(i == 1){
#'       if(i < length(ecp)) int <- 1:floor(ecp[i]*(1 - c) + ecp[i + 1]*c) else int <- 1:n
#'     } else{
#'       if(i < length(ecp)) int <- floor(ecp[i]*(1 - c) + ecp[i - 1]*c + 1):floor(ecp[i]*(1 - c) + ecp[i + 1]*c) else int <- floor(ecp[i]*(1 - c) + ecp[i - 1]*c + 1):n
#'     }
#'     if(length(int) < 2){
#'       if(i == 1) int <- 1:ecp[i + 1] else if(i == length(ecp)) int <- (ecp[i - 1] + 1):n else int <- (ecp[i - 1] + 1):ecp[i + 1]
#'     }
#'     csx <- cumsum(x[int])
#'     len <- length(int)
#'     evalx <- (len/((1:(len - 1)) * ((len - 1):1)))^.5 * (csx[1:(len - 1)] - csx[len]/len * (1:(len - 1)))
#'     rcp[i] <- int[1] + which.max(abs(evalx)) - 1
#'     nres <- rbind(nres, c(min(int), rcp[i], max(int), max(abs(evalx))))
#'   }
#'   
#'   return(list(rcp = rcp, nres = nres))
#' }

