/*
 * Tegra Audio Hub. The audio integrated component on Tegra to support multiple
 * interfaces to audio devices on system.
 *
 * Copyright 2013 Google Inc. All rights reserved.
 * Copyright (c) 2013, NVIDIA CORPORATION.  All rights reserved.
 *
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 */

#ifndef __DRIVERS_SOUND_TEGRA_AHUB_H__
#define __DRIVERS_SOUND_TEGRA_AHUB_H__

#include "drivers/bus/i2s/tegra.h"
#include "drivers/common/fifo.h"
#include "drivers/sound/route.h"

struct TegraXbarRegs;
struct TegraApbifRegs;

// Crossbar (to route data inside audio hub).
typedef struct TegraAudioHubXbar
{
	struct TegraXbarRegs *regs;
} TegraAudioHubXbar;

// AMBA Peripheral Bus Interface (provides DMA FIFO to send data into hub).
typedef struct TegraAudioHubApbif
{
	TxFifoOps ops;
	struct TegraApbifRegs *regs;
	uint32_t full_mask;
	size_t capacity_in_word;  // FIFO capacity in words.
} TegraAudioHubApbif;

typedef struct TegraAudioHub
{
	SoundRouteComponent component;

	TegraAudioHubXbar *xbar;  // Crossbar (to route data inside AHUB).
	TegraAudioHubApbif *apbif;  // AMBA Peripheral Bus Interface.
	TegraI2s *i2s;  // Activate Tegra I2S peripheral.
} TegraAudioHub;

TegraAudioHubXbar *new_tegra_audio_hub_xbar(uintptr_t regs);
TegraAudioHubApbif *new_tegra_audio_hub_apbif(uintptr_t regs,
					      size_t capacity_in_word);
TegraAudioHub *new_tegra_audio_hub(TegraAudioHubXbar *xbar,
				   TegraAudioHubApbif *apbif,
				   TegraI2s *i2s);

/*
 * Tegra AHUB Registers Definition
 */

enum {
	TEGRA_AUDIOCIF_CTRL_FIFO_THRESHOLD_SHIFT = 24,
	TEGRA_AUDIOCIF_CTRL_FIFO_THRESHOLD_MASK_US = 0x3f,
	TEGRA_AUDIOCIF_CTRL_FIFO_THRESHOLD_MASK =
			TEGRA_AUDIOCIF_CTRL_FIFO_THRESHOLD_MASK_US <<
			TEGRA_AUDIOCIF_CTRL_FIFO_THRESHOLD_SHIFT,

	/* Channel count minus 1 */
	TEGRA_AUDIOCIF_CTRL_AUDIO_CHANNELS_SHIFT = 20,
	TEGRA_AUDIOCIF_CTRL_AUDIO_CHANNELS_MASK_US = 0xf,
	TEGRA_AUDIOCIF_CTRL_AUDIO_CHANNELS_MASK =
			TEGRA_AUDIOCIF_CTRL_AUDIO_CHANNELS_MASK_US <<
			TEGRA_AUDIOCIF_CTRL_AUDIO_CHANNELS_SHIFT,

	/* Channel count minus 1 */
	TEGRA_AUDIOCIF_CTRL_CLIENT_CHANNELS_SHIFT = 16,
	TEGRA_AUDIOCIF_CTRL_CLIENT_CHANNELS_MASK_US = 0xf,
	TEGRA_AUDIOCIF_CTRL_CLIENT_CHANNELS_MASK =
			TEGRA_AUDIOCIF_CTRL_CLIENT_CHANNELS_MASK_US <<
			TEGRA_AUDIOCIF_CTRL_CLIENT_CHANNELS_SHIFT,

	TEGRA_AUDIOCIF_BITS_4 = 0,
	TEGRA_AUDIOCIF_BITS_8 = 1,
	TEGRA_AUDIOCIF_BITS_12 = 2,
	TEGRA_AUDIOCIF_BITS_16 = 3,
	TEGRA_AUDIOCIF_BITS_20 = 4,
	TEGRA_AUDIOCIF_BITS_24 = 5,
	TEGRA_AUDIOCIF_BITS_28 = 6,
	TEGRA_AUDIOCIF_BITS_32 = 7,

	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT = 12,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_MASK =
		7 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_4 =
		TEGRA_AUDIOCIF_BITS_4  << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_8 =
		TEGRA_AUDIOCIF_BITS_8  << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_12 =
		TEGRA_AUDIOCIF_BITS_12 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_16 =
		TEGRA_AUDIOCIF_BITS_16 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_20 =
		TEGRA_AUDIOCIF_BITS_20 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_24 =
		TEGRA_AUDIOCIF_BITS_24 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_28 =
		TEGRA_AUDIOCIF_BITS_28 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_32 =
		TEGRA_AUDIOCIF_BITS_32 << TEGRA_AUDIOCIF_CTRL_AUDIO_BITS_SHIFT,

	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT = 8,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_MASK =
		7 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_4 =
		TEGRA_AUDIOCIF_BITS_4  << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_8 =
		TEGRA_AUDIOCIF_BITS_8  << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_12 =
		TEGRA_AUDIOCIF_BITS_12 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_16 =
		TEGRA_AUDIOCIF_BITS_16 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_20 =
		TEGRA_AUDIOCIF_BITS_20 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_24 =
		TEGRA_AUDIOCIF_BITS_24 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_28 =
		TEGRA_AUDIOCIF_BITS_28 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,
	TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_32 =
		TEGRA_AUDIOCIF_BITS_32 << TEGRA_AUDIOCIF_CTRL_CLIENT_BITS_SHIFT,

	TEGRA_AUDIOCIF_EXPAND_ZERO = 0,
	TEGRA_AUDIOCIF_EXPAND_ONE = 1,
	TEGRA_AUDIOCIF_EXPAND_LFSR = 2,

	TEGRA_AUDIOCIF_CTRL_EXPAND_SHIFT = 6,
	TEGRA_AUDIOCIF_CTRL_EXPAND_MASK = 3 << TEGRA_AUDIOCIF_CTRL_EXPAND_SHIFT,
	TEGRA_AUDIOCIF_CTRL_EXPAND_ZERO =
		TEGRA_AUDIOCIF_EXPAND_ZERO << TEGRA_AUDIOCIF_CTRL_EXPAND_SHIFT,
	TEGRA_AUDIOCIF_CTRL_EXPAND_ONE =
		TEGRA_AUDIOCIF_EXPAND_ONE  << TEGRA_AUDIOCIF_CTRL_EXPAND_SHIFT,
	TEGRA_AUDIOCIF_CTRL_EXPAND_LFSR =
		TEGRA_AUDIOCIF_EXPAND_LFSR << TEGRA_AUDIOCIF_CTRL_EXPAND_SHIFT,

	TEGRA_AUDIOCIF_STEREO_CONV_CH0 = 0,
	TEGRA_AUDIOCIF_STEREO_CONV_CH1 = 1,
	TEGRA_AUDIOCIF_STEREO_CONV_AVG = 2,

	TEGRA_AUDIOCIF_CTRL_STEREO_CONV_SHIFT = 4,
	TEGRA_AUDIOCIF_CTRL_STEREO_CONV_MASK =
			3 << TEGRA_AUDIOCIF_CTRL_STEREO_CONV_SHIFT,
	TEGRA_AUDIOCIF_CTRL_STEREO_CONV_CH0 =
			TEGRA_AUDIOCIF_STEREO_CONV_CH0 <<
			TEGRA_AUDIOCIF_CTRL_STEREO_CONV_SHIFT,
	TEGRA_AUDIOCIF_CTRL_STEREO_CONV_CH1 =
			TEGRA_AUDIOCIF_STEREO_CONV_CH1 <<
			TEGRA_AUDIOCIF_CTRL_STEREO_CONV_SHIFT,
	TEGRA_AUDIOCIF_CTRL_STEREO_CONV_AVG =
			TEGRA_AUDIOCIF_STEREO_CONV_AVG <<
			TEGRA_AUDIOCIF_CTRL_STEREO_CONV_SHIFT,

	TEGRA_AUDIOCIF_CTRL_REPLICATE = 3,

	TEGRA_AUDIOCIF_DIRECTION_TX = 0,
	TEGRA_AUDIOCIF_DIRECTION_RX = 1,

	TEGRA_AUDIOCIF_CTRL_DIRECTION_SHIFT = 2,
	TEGRA_AUDIOCIF_CTRL_DIRECTION_MASK =
			1 << TEGRA_AUDIOCIF_CTRL_DIRECTION_SHIFT,
	TEGRA_AUDIOCIF_CTRL_DIRECTION_TX =
			TEGRA_AUDIOCIF_DIRECTION_TX <<
			TEGRA_AUDIOCIF_CTRL_DIRECTION_SHIFT,
	TEGRA_AUDIOCIF_CTRL_DIRECTION_RX =
			TEGRA_AUDIOCIF_DIRECTION_RX <<
			TEGRA_AUDIOCIF_CTRL_DIRECTION_SHIFT,

	TEGRA_AUDIOCIF_TRUNCATE_ROUND = 0,
	TEGRA_AUDIOCIF_TRUNCATE_CHOP = 1,

	TEGRA_AUDIOCIF_CTRL_TRUNCATE_SHIFT = 1,
	TEGRA_AUDIOCIF_CTRL_TRUNCATE_MASK =
			1 << TEGRA_AUDIOCIF_CTRL_TRUNCATE_SHIFT,
	TEGRA_AUDIOCIF_CTRL_TRUNCATE_ROUND =
			TEGRA_AUDIOCIF_TRUNCATE_ROUND <<
			TEGRA_AUDIOCIF_CTRL_TRUNCATE_SHIFT,
	TEGRA_AUDIOCIF_CTRL_TRUNCATE_CHOP =
			TEGRA_AUDIOCIF_TRUNCATE_CHOP  <<
			TEGRA_AUDIOCIF_CTRL_TRUNCATE_SHIFT,

	TEGRA_AUDIOCIF_MONO_CONV_ZERO = 0,
	TEGRA_AUDIOCIF_MONO_CONV_COPY = 1,

	TEGRA_AUDIOCIF_CTRL_MONO_CONV_SHIFT = 0,
	TEGRA_AUDIOCIF_CTRL_MONO_CONV_MASK =
			1 << TEGRA_AUDIOCIF_CTRL_MONO_CONV_SHIFT,
	TEGRA_AUDIOCIF_CTRL_MONO_CONV_ZERO =
			TEGRA_AUDIOCIF_MONO_CONV_ZERO <<
			TEGRA_AUDIOCIF_CTRL_MONO_CONV_SHIFT,
	TEGRA_AUDIOCIF_CTRL_MONO_CONV_COPY =
			TEGRA_AUDIOCIF_MONO_CONV_COPY <<
			TEGRA_AUDIOCIF_CTRL_MONO_CONV_SHIFT,

	/* Registers within TEGRA_AUDIO_CLUSTER_BASE */

	TEGRA_AHUB_CHANNEL_CTRL = 0x0,
	TEGRA_AHUB_CHANNEL_CTRL_STRIDE = 0x20,
	TEGRA_AHUB_CHANNEL_CTRL_COUNT = 4,
	TEGRA_AHUB_CHANNEL_CTRL_TX_EN = 1 << 31,
	TEGRA_AHUB_CHANNEL_CTRL_RX_EN = 1 << 30,
	TEGRA_AHUB_CHANNEL_CTRL_LOOPBACK = 1 << 29,

	TEGRA_AHUB_CHANNEL_CTRL_TX_THRESHOLD_SHIFT = 16,
	TEGRA_AHUB_CHANNEL_CTRL_TX_THRESHOLD_MASK_US = 0xff,
	TEGRA_AHUB_CHANNEL_CTRL_TX_THRESHOLD_MASK =
			TEGRA_AHUB_CHANNEL_CTRL_TX_THRESHOLD_MASK_US <<
			TEGRA_AHUB_CHANNEL_CTRL_TX_THRESHOLD_SHIFT,

	TEGRA_AHUB_CHANNEL_CTRL_RX_THRESHOLD_SHIFT = 8,
	TEGRA_AHUB_CHANNEL_CTRL_RX_THRESHOLD_MASK_US = 0xff,
	TEGRA_AHUB_CHANNEL_CTRL_RX_THRESHOLD_MASK =
			TEGRA_AHUB_CHANNEL_CTRL_RX_THRESHOLD_MASK_US <<
			TEGRA_AHUB_CHANNEL_CTRL_RX_THRESHOLD_SHIFT,

	TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_EN = 1 << 6,

	TEGRA_PACK_8_4 = 2,
	TEGRA_PACK_16 = 3,

	TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_SHIFT = 4,
	TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_MASK_US = 3,
	TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_MASK =
		TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_MASK_US <<
		TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_SHIFT,
	TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_8_4 =
		TEGRA_PACK_8_4 << TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_SHIFT,
	TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_16 =
		TEGRA_PACK_16 << TEGRA_AHUB_CHANNEL_CTRL_TX_PACK_SHIFT,

	TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_EN = 1 << 2,

	TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_SHIFT = 0,
	TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_MASK_US = 3,
	TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_MASK =
		TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_MASK_US <<
		TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_SHIFT,
	TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_8_4 =
		TEGRA_PACK_8_4 << TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_SHIFT,
	TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_16 =
		TEGRA_PACK_16 << TEGRA_AHUB_CHANNEL_CTRL_RX_PACK_SHIFT,

	/* TEGRA_AHUB_CHANNEL_CLEAR */

	TEGRA_AHUB_CHANNEL_CLEAR = 0x4,
	TEGRA_AHUB_CHANNEL_CLEAR_STRIDE = 0x20,
	TEGRA_AHUB_CHANNEL_CLEAR_COUNT = 4,
	TEGRA_AHUB_CHANNEL_CLEAR_TX_SOFT_RESET = 1 << 31,
	TEGRA_AHUB_CHANNEL_CLEAR_RX_SOFT_RESET = 1 << 30,

	TEGRA_AHUB_CHANNEL_STATUS = 0x8,
	TEGRA_AHUB_CHANNEL_STATUS_STRIDE = 0x20,
	TEGRA_AHUB_CHANNEL_STATUS_COUNT = 4,
	TEGRA_AHUB_CHANNEL_STATUS_TX_FREE_SHIFT = 24,
	TEGRA_AHUB_CHANNEL_STATUS_TX_FREE_MASK_US = 0xff,
	TEGRA_AHUB_CHANNEL_STATUS_TX_FREE_MASK =
			TEGRA_AHUB_CHANNEL_STATUS_TX_FREE_MASK_US <<
			TEGRA_AHUB_CHANNEL_STATUS_TX_FREE_SHIFT,
	TEGRA_AHUB_CHANNEL_STATUS_RX_FREE_SHIFT = 16,
	TEGRA_AHUB_CHANNEL_STATUS_RX_FREE_MASK_US = 0xff,
	TEGRA_AHUB_CHANNEL_STATUS_RX_FREE_MASK =
			TEGRA_AHUB_CHANNEL_STATUS_RX_FREE_MASK_US <<
			TEGRA_AHUB_CHANNEL_STATUS_RX_FREE_SHIFT,
	TEGRA_AHUB_CHANNEL_STATUS_TX_TRIG = 1 << 1,
	TEGRA_AHUB_CHANNEL_STATUS_RX_TRIG = 1 << 0,

	TEGRA_AHUB_CHANNEL_TXFIFO = 0xc,
	TEGRA_AHUB_CHANNEL_TXFIFO_STRIDE = 0x20,
	TEGRA_AHUB_CHANNEL_TXFIFO_COUNT = 4,

	TEGRA_AHUB_CHANNEL_RXFIFO = 0x10,
	TEGRA_AHUB_CHANNEL_RXFIFO_STRIDE = 0x20,
	TEGRA_AHUB_CHANNEL_RXFIFO_COUNT = 4,

	TEGRA_AHUB_CIF_TX_CTRL = 0x14,
	TEGRA_AHUB_CIF_TX_CTRL_STRIDE = 0x20,
	TEGRA_AHUB_CIF_TX_CTRL_COUNT = 4,
	/* Uses field from TEGRA_AUDIOCIF_CTRL_* */

	TEGRA_AHUB_CIF_RX_CTRL = 0x18,
	TEGRA_AHUB_CIF_RX_CTRL_STRIDE = 0x20,
	TEGRA_AHUB_CIF_RX_CTRL_COUNT = 4,
	/* Uses field from TEGRA_AUDIOCIF_CTRL_* */

	TEGRA_AHUB_CONFIG_LINK_CTRL = 0x80,
	TEGRA_AHUB_CONFIG_LINK_CTRL_MASTER_FIFO_FULL_CNT_SHIFT = 28,
	TEGRA_AHUB_CONFIG_LINK_CTRL_MASTER_FIFO_FULL_CNT_MASK_US = 0xf,
	TEGRA_AHUB_CONFIG_LINK_CTRL_MASTER_FIFO_FULL_CNT_MASK =
		TEGRA_AHUB_CONFIG_LINK_CTRL_MASTER_FIFO_FULL_CNT_MASK_US <<
		TEGRA_AHUB_CONFIG_LINK_CTRL_MASTER_FIFO_FULL_CNT_SHIFT,
	TEGRA_AHUB_CONFIG_LINK_CTRL_TIMEOUT_CNT_SHIFT = 16,
	TEGRA_AHUB_CONFIG_LINK_CTRL_TIMEOUT_CNT_MASK_US = 0xfff,
	TEGRA_AHUB_CONFIG_LINK_CTRL_TIMEOUT_CNT_MASK =
		TEGRA_AHUB_CONFIG_LINK_CTRL_TIMEOUT_CNT_MASK_US <<
		TEGRA_AHUB_CONFIG_LINK_CTRL_TIMEOUT_CNT_SHIFT,
	TEGRA_AHUB_CONFIG_LINK_CTRL_IDLE_CNT_SHIFT = 4,
	TEGRA_AHUB_CONFIG_LINK_CTRL_IDLE_CNT_MASK_US = 0xfff,
	TEGRA_AHUB_CONFIG_LINK_CTRL_IDLE_CNT_MASK =
		TEGRA_AHUB_CONFIG_LINK_CTRL_IDLE_CNT_MASK_US <<
		TEGRA_AHUB_CONFIG_LINK_CTRL_IDLE_CNT_SHIFT,
	TEGRA_AHUB_CONFIG_LINK_CTRL_CG_EN = 1 << 2,
	TEGRA_AHUB_CONFIG_LINK_CTRL_CLEAR_TIMEOUT_CNTR = 1 << 1,
	TEGRA_AHUB_CONFIG_LINK_CTRL_SOFT_RESET = 1 << 0,

	TEGRA_AHUB_MISC_CTRL = 0x84,
	TEGRA_AHUB_MISC_CTRL_AUDIO_ACTIVE = 1 << 31,
	TEGRA_AHUB_MISC_CTRL_AUDIO_CG_EN = 1 << 8,
	TEGRA_AHUB_MISC_CTRL_AUDIO_OBS_SEL_SHIFT = 0,
	TEGRA_AHUB_MISC_CTRL_AUDIO_OBS_SEL_MASK =
			0x1f << TEGRA_AHUB_MISC_CTRL_AUDIO_OBS_SEL_SHIFT,

	TEGRA_AHUB_APBDMA_LIVE_STATUS = 0x88,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_RX_CIF_FIFO_FULL = 1 << 31,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_TX_CIF_FIFO_FULL = 1 << 30,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_RX_CIF_FIFO_FULL = 1 << 29,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_TX_CIF_FIFO_FULL = 1 << 28,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_RX_CIF_FIFO_FULL = 1 << 27,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_TX_CIF_FIFO_FULL = 1 << 26,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_RX_CIF_FIFO_FULL = 1 << 25,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_TX_CIF_FIFO_FULL = 1 << 24,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_RX_CIF_FIFO_EMPTY = 1 << 23,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_TX_CIF_FIFO_EMPTY = 1 << 22,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_RX_CIF_FIFO_EMPTY = 1 << 21,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_TX_CIF_FIFO_EMPTY = 1 << 20,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_RX_CIF_FIFO_EMPTY = 1 << 19,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_TX_CIF_FIFO_EMPTY = 1 << 18,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_RX_CIF_FIFO_EMPTY = 1 << 17,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_TX_CIF_FIFO_EMPTY = 1 << 16,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_RX_DMA_FIFO_FULL = 1 << 15,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_TX_DMA_FIFO_FULL = 1 << 14,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_RX_DMA_FIFO_FULL = 1 << 13,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_TX_DMA_FIFO_FULL = 1 << 12,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_RX_DMA_FIFO_FULL = 1 << 11,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_TX_DMA_FIFO_FULL = 1 << 10,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_RX_DMA_FIFO_FULL = 1 << 9,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_TX_DMA_FIFO_FULL = 1 << 8,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_RX_DMA_FIFO_EMPTY = 1 << 7,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH3_TX_DMA_FIFO_EMPTY = 1 << 6,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_RX_DMA_FIFO_EMPTY = 1 << 5,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH2_TX_DMA_FIFO_EMPTY = 1 << 4,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_RX_DMA_FIFO_EMPTY = 1 << 3,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH1_TX_DMA_FIFO_EMPTY = 1 << 2,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_RX_DMA_FIFO_EMPTY = 1 << 1,
	TEGRA_AHUB_APBDMA_LIVE_STATUS_CH0_TX_DMA_FIFO_EMPTY = 1 << 0,

	TEGRA_AHUB_I2S_LIVE_STATUS = 0x8c,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S4_RX_FIFO_FULL = 1 << 29,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S4_TX_FIFO_FULL = 1 << 28,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S3_RX_FIFO_FULL = 1 << 27,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S3_TX_FIFO_FULL = 1 << 26,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S2_RX_FIFO_FULL = 1 << 25,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S2_TX_FIFO_FULL = 1 << 24,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S1_RX_FIFO_FULL = 1 << 23,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S1_TX_FIFO_FULL = 1 << 22,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S0_RX_FIFO_FULL = 1 << 21,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S0_TX_FIFO_FULL = 1 << 20,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S4_RX_FIFO_ENABLED = 1 << 19,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S4_TX_FIFO_ENABLED = 1 << 18,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S3_RX_FIFO_ENABLED = 1 << 17,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S3_TX_FIFO_ENABLED = 1 << 16,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S2_RX_FIFO_ENABLED = 1 << 15,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S2_TX_FIFO_ENABLED = 1 << 14,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S1_RX_FIFO_ENABLED = 1 << 13,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S1_TX_FIFO_ENABLED = 1 << 12,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S0_RX_FIFO_ENABLED = 1 << 11,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S0_TX_FIFO_ENABLED = 1 << 10,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S4_RX_FIFO_EMPTY = 1 << 9,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S4_TX_FIFO_EMPTY = 1 << 8,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S3_RX_FIFO_EMPTY = 1 << 7,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S3_TX_FIFO_EMPTY = 1 << 6,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S2_RX_FIFO_EMPTY = 1 << 5,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S2_TX_FIFO_EMPTY = 1 << 4,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S1_RX_FIFO_EMPTY = 1 << 3,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S1_TX_FIFO_EMPTY = 1 << 2,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S0_RX_FIFO_EMPTY = 1 << 1,
	TEGRA_AHUB_I2S_LIVE_STATUS_I2S0_TX_FIFO_EMPTY = 1 << 0,

	TEGRA_AHUB_DAM_LIVE_STATUS = 0x90,
	TEGRA_AHUB_DAM_LIVE_STATUS_STRIDE = 0x8,
	TEGRA_AHUB_DAM_LIVE_STATUS_COUNT = 3,
	TEGRA_AHUB_DAM_LIVE_STATUS_TX_ENABLED = 1 << 26,
	TEGRA_AHUB_DAM_LIVE_STATUS_RX1_ENABLED = 1 << 25,
	TEGRA_AHUB_DAM_LIVE_STATUS_RX0_ENABLED = 1 << 24,
	TEGRA_AHUB_DAM_LIVE_STATUS_TXFIFO_FULL = 1 << 15,
	TEGRA_AHUB_DAM_LIVE_STATUS_RX1FIFO_FULL = 1 << 9,
	TEGRA_AHUB_DAM_LIVE_STATUS_RX0FIFO_FULL = 1 << 8,
	TEGRA_AHUB_DAM_LIVE_STATUS_TXFIFO_EMPTY = 1 << 7,
	TEGRA_AHUB_DAM_LIVE_STATUS_RX1FIFO_EMPTY = 1 << 1,
	TEGRA_AHUB_DAM_LIVE_STATUS_RX0FIFO_EMPTY = 1 << 0,

	TEGRA_AHUB_SPDIF_LIVE_STATUS = 0xa8,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_USER_TX_ENABLED = 1 << 11,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_USER_RX_ENABLED = 1 << 10,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_DATA_TX_ENABLED = 1 << 9,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_DATA_RX_ENABLED = 1 << 8,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_USER_TXFIFO_FULL = 1 << 7,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_USER_RXFIFO_FULL = 1 << 6,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_DATA_TXFIFO_FULL = 1 << 5,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_DATA_RXFIFO_FULL = 1 << 4,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_USER_TXFIFO_EMPTY = 1 << 3,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_USER_RXFIFO_EMPTY = 1 << 2,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_DATA_TXFIFO_EMPTY = 1 << 1,
	TEGRA_AHUB_SPDIF_LIVE_STATUS_DATA_RXFIFO_EMPTY = 1 << 0,

	TEGRA_AHUB_I2S_INT_MASK = 0xb0,
	TEGRA_AHUB_DAM_INT_MASK = 0xb4,
	TEGRA_AHUB_SPDIF_INT_MASK = 0xbc,
	TEGRA_AHUB_APBIF_INT_MASK = 0xc0,
	TEGRA_AHUB_I2S_INT_STATUS = 0xc8,
	TEGRA_AHUB_DAM_INT_STATUS = 0xcc,
	TEGRA_AHUB_SPDIF_INT_STATUS = 0xd4,
	TEGRA_AHUB_APBIF_INT_STATUS = 0xd8,
	TEGRA_AHUB_I2S_INT_SOURCE = 0xe0,
	TEGRA_AHUB_DAM_INT_SOURCE = 0xe4,
	TEGRA_AHUB_SPDIF_INT_SOURCE = 0xec,
	TEGRA_AHUB_APBIF_INT_SOURCE = 0xf0,
	TEGRA_AHUB_I2S_INT_SET = 0xf8,
	TEGRA_AHUB_DAM_INT_SET = 0xfc,
	TEGRA_AHUB_SPDIF_INT_SET = 0x100,
	TEGRA_AHUB_APBIF_INT_SET = 0x104,

	TEGRA_AHUB_AUDIO_RX = 0x0,
	TEGRA_AHUB_AUDIO_RX_STRIDE = 0x4,
	TEGRA_AHUB_AUDIO_RX_COUNT = 17,
};

#endif /* __DRIVERS_SOUND_TEGRA_AHUB_H__ */
