// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with the package.



#include "drawarea.h"
#include "krot.h"
#include "krotData.h"
#include "scrollbar.h"
#include "workarea.h"

#include <kpopmenu.h>

#include <qlayout.h>

#include <sys/types.h>


WorkArea::WorkArea( KRotData *krotdata, QWidget *parent, const char *name )
    : QWidget( parent, name ),
      average( krotdata->averaging() ),
      krotData( krotdata )
      
{
    KROT_LAUNCH( "Launching WorkArea constructor." );
    QBoxLayout *vlayout = new QVBoxLayout( this );
    QBoxLayout *hlayout = new QHBoxLayout();
    vlayout->addLayout( hlayout );
    drawarea = new DrawArea( krotData, this, "DrawArea" );
    hlayout->addWidget( drawarea );
    vscroll = new KRotScrollBar( KRotScrollBar::Vertical, this, "Vertical ScrollBar" );
    vscroll->setFixedWidth( 15 );
    hlayout->addWidget( vscroll );
    hscroll = new KRotScrollBar( KRotScrollBar::Horizontal, this, "Horizontal ScrollBar" );
    hscroll->setFixedHeight( 15 );
    vlayout->addWidget( hscroll );
    vlayout->activate();

    connect( drawarea, SIGNAL( freq( const int64_t ) ),
	     this,     SLOT( setFreq( const int64_t ) ) );
    connect( drawarea, SIGNAL( assignment( const AssignedFrequency& ) ),
	     this,     SLOT( newAssignment( const AssignedFrequency& ) ) );
    connect( hscroll, SIGNAL( valueChanged( int ) ),
	     drawarea,  SLOT( newPosition( int ) ) );
    connect( hscroll, SIGNAL( valueChanged( int ) ),
	     drawarea,  SLOT( newIntensity( int ) ) );
    connect( hscroll, SIGNAL( rightButtonPressed() ),
	     this,  SLOT( selectRange() ) );

    // create range selection popup
    popup = new QPopupMenu();
    connect( popup, SIGNAL( activated( int ) ), this, SLOT( showRange( int ) ) );
}



WorkArea::~WorkArea()
{
    delete popup;
}



bool WorkArea::findRange( int64_t pos )
{
    DataRanges::iterator r = ranges.begin();
    while( ranges.end() != r ) {
	if( r->start <= pos && pos <= r->stop ) {
	    current = *r;
	    KROT_DEBUG1( "Current range now starts at %lld", current.start );
	    return true;
	}
	r++;
    }
    KROT_DEBUG2( "No range containing %lld found, current still starts at %lld", pos, current.start );
    return false;
}


void WorkArea::limits( const DataRanges& r )
{
    KROT_LAUNCH( "Launching WorkArea::limits" );
    ranges = r;
    if( ranges.empty() ) {
	// zero everything
	shifts[ 0 ] = shifts[ 1 ] = 0;
	hscroll->setRange( 0, 0 );
	hscroll->setSteps( 0, 0 );
	vscroll->setRange( 0, 0 );
	vscroll->setSteps( 0, 0 );
	drawarea->setDisplayRange( DataRange() );
    } else {
	int64_t val = current.start + vscroll->value();
	// find old range
	if( ! findRange( val ) ) {
	    // if no range contains the current value, go to the first range
	    current = ranges.front();
	}
	showCurrentRange();
    }
    return;
}



void WorkArea::selectRange()
{
    popup->clear();
    int titleId = popup->insertItem( i18n( "Frequency Ranges" ), -2, 0 );
    popup->setItemEnabled( titleId, false );
    popup->insertSeparator( 1 );
    DataRanges::iterator r = ranges.begin();
    int id = 0;
    while( ranges.end() != r ) {
	if( r->start != r->stop ) {
	    char buffer[64];
#if SIZEOF_LONG >= 8
	    sprintf( buffer, "%ld - %ld", r->start, r->stop );
#else
	    sprintf( buffer, "%lld - %lld", r->start, r->stop );
#endif
	    popup->insertItem( buffer, id, id + 2 );
	    id++;
	}
	r++;
    }
    popup->exec( QCursor::pos() );
}



void WorkArea::setAveraging( const unsigned int newAveraging )
    // Calculate new scrollbar value, overwrite old averaging value by new one,
    // set scrollbar limits, and finally set scrollbar to the new value
    // calculated right before (v.s.).
{
    KROT_LAUNCH( "Launching WorkArea::setAveraging" );
    int64_t val = static_cast< int64_t >( static_cast< double >( average ) / newAveraging
					  * ( current.start + hscroll->value() + drawarea->width() / 2 ) );
    average = newAveraging;
    limits( ranges );
    hscroll->setValue( val - current.start + drawarea->width() / 2 );
    return;
}



void WorkArea::setPosition( int64_t pos )
{
    KROT_LAUNCH( "Launching WorkArea::setPosition" );
    if( current.start > pos || pos > current.stop )
	if( ! findRange( pos ) ) // What is best to do here ?
	    pos = current.start; // Probably we should throw an exception ?
    hscroll->setValue( pos - current.start );
    return;
}



void WorkArea::showRange( int id )
{
#warning Need to do better here
    if( 0 <= id ) {
	current = ranges[ id ];
	showCurrentRange();
    }
}



void WorkArea::showCurrentRange()
{
    // update frequency scrollbar
    int start = drawarea->width() / 2;
    int stop = ( current.stop - current.start ) / average - drawarea->width() / 2;
    hscroll->setRange( start, std::max( start, stop ) );
    hscroll->setSteps( 1, drawarea->width() / 2 );
    // update intensity scrollbar
    int min = drawarea->height() / 2;
    int max = ( current.max - current.min ) - drawarea->height() / 2;
    vscroll->setRange( min, std::max( min, max ) );
    vscroll->setSteps( 1, drawarea->height() / 2 );
    // tell the drawarea
    drawarea->setDisplayRange( current );
    drawarea->update();
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
