// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_DRAWING_H
#define KROT_DRAWING_H



#include "krotData.h"
#include "transition.h"

#include <qevent.h>
#include <qwidget.h>

#include <sys/types.h>


class DataSet;
class DialogLineSelection;



/**
 * Drawing canvas.
 *
 * @short Drawing area widget.
 * @author Jochen Kpper
 * @version 1999/02/28
 */
class DrawArea : public QWidget
{
    Q_OBJECT

public:

    /**
     * Constructor.
     *
     * @param parent Passed to QWidget constructor.
     * @param name   Passed to QWidget constructor.
     */
    DrawArea( KRotData *krotdata, QWidget *parent=0, const char *name=0 );

    /**
     * New range to display
     *
     * @param r New DataRange to use for display
     */
    void setDisplayRange( const DataRange& r );

    
protected:

    /**
     * Compute the actual intensity value of the dataset pointed to by iter.
     * Takes into account averaging and the value returned by the dataset
     * operator[].
     *
     * @param  set Reference of @ref DataSet.
     * @param  freq Frequency to calculate intensity at.
     * @return Intensity.
     */
    int computeIntensity( const DataSet& set, const int64_t freq ) const;

    /**
     * Sends a signal about the new frequency the mouse is at.
     */
    void mouseMoveEvent( QMouseEvent * );

    /**
     * Handles user selection of range to show simulated lines of and
     * sends signal about assignment frequency (mouse button 2 pressed).
     */
    void mousePressEvent( QMouseEvent * );

    /**
     * Handles user selection of range to show simulated lines of.
     * In here the signal to the outside world is send.
     */
    void mouseReleaseEvent( QMouseEvent * );

    /**
     * Repaints the entire canvas.
     */
    void paintEvent( QPaintEvent * );

    /**
     * Check start frequency whenever we are resized.
     */
    void resizeEvent( QResizeEvent * );


protected slots:
    
    void newIntensity( int value );

    /**
     * Calculate start frequency from new scrollbar value.
     *
     * @param value Scrollbar value.
     */
    void newPosition( int value );


protected:

    /**
     * Check wether the actual start frequency is within the global frequency range.
     * If it isn't, set it to the closer global range boundary.
     */
    void checkStartFreq();


signals:

    /**
     * Sends every assignment made to the world.
     * These assignment need to be stored and should be editable and saveable.
     */
    void assignment( const AssignedFrequency& ass );

    /**
     * Emitted whenever the user selects a range of simulated lines to make an
     * assignment.
     *
     * @param start Beginning of the range (lower frequency).
     * @param start End of the range (higher frequency).
     */
    void lineSelectionRange( const int64_t start, const int64_t stop );

    /**
     * Emitted whenever the mouse was moved inside the @ref DrawArea.
     *
     * @param freq Actual mouse cursor frequency.
     */
    void freq( const int64_t freq );


protected:

    /**
     * Line selection dialog.
     */
    DialogLineSelection *lineSelection;

    /**
     * The KRotData to display.
     */
    KRotData *krotData;

    /**
     * Frequency and intesity of the leftmost & lowest point in the canvas.
     */
    int64_t min, start;

    /**
     * Range to display.
     */
    DataRange range;
    
    /**
     * Last positions where left/middle/right button were pressed.
     */
    QPoint posPressed[3];
};



#include <drawarea_inline.h>



#endif



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
