// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_CALCULATIONPARAMETER_INLINE_H
#define KROT_CALCULATIONPARAMETER_INLINE_H



#include "constants.h"



inline CalculationParameter::CalculationParameter()
    : InertialConstants()
{
    defaultValues();
    return;
}



inline CalculationParameter::CalculationParameter( const CalculationParameter& p)
    : InertialConstants( p )
{
    for( ParameterFloat i=fBegin; i<fEnd; incr( i ) ) {
	parameterFloat[ i ] = p.parameterFloat[ i ];
	fitParameterFloat[ i ] = p.fitParameterFloat[ i ];
    }
    for( ParameterInt i=iBegin; i<iEnd; incr( i ) ) {
	parameterInt[ i ] = p.parameterInt[ i ];
	fitParameterInt[ i ] = p.fitParameterInt[ i ];
    }
    return;
}



inline CalculationParameter::~CalculationParameter()
{
    return;
}



inline CalculationParameter& CalculationParameter::operator=( const CalculationParameter& p)
{
    if( this == &p )
	return *this;
    InertialConstants::operator=( p );
    for( int i=0; i<fEnd; i++ ) {
	parameterFloat[ i ] = p.parameterFloat[ i ];
	fitParameterFloat[ i ] = p.fitParameterFloat[ i ];
    }
    for( int i=0; i<iEnd; i++ ) {
	parameterInt[ i ] = p.parameterInt[ i ];
	fitParameterInt[ i ] = p.fitParameterInt[ i ];
    }
    return *this;
}



inline bool CalculationParameter::axisTilting() const
{
    for( ParameterFloat i=AxisTiltPhi; i<=AxisTiltChi; incr( i ) )
	if( fitParameterFloat[ i ] || ( abs( parameterFloat[ i ] ) > Constants::eps() ) )
	    return true;
    return false;
}




inline double CalculationParameter::constant( const ParameterFloat p ) const
{
    return parameterFloat[ p ];
}



inline double& CalculationParameter::constant( const ParameterFloat p )
{
    return parameterFloat[ p ];
}



inline int CalculationParameter::constant( const ParameterInt p ) const
{
    return parameterInt[ p ];
}



inline int& CalculationParameter::constant( const ParameterInt p )
{
    return parameterInt[ p ];
}



inline void CalculationParameter::defaultValues()
{
    InertialConstants::defaultValues();
    parameterFloat[ Temperature ] = parameterFloat[ Temperature2 ] = 1.0;
    parameterFloat[ TemperatureWeight ] = 0.0;
    parameterFloat[ Origin ]= 0.0;
    parameterFloat[ LPU ] = 5.0;
    parameterFloat[ HybridA ] = 100.0;
    parameterFloat[ HybridB ] = 0.0;
    parameterFloat[ HybridC ] = 0.0;
    parameterInt[ Jmin ] = 0;
    parameterInt[ Jmax ] = 15;
    parameterInt[ DKmax ] = 1;
    parameterInt[ NSSWee ] = parameterInt[ NSSWeo ] = parameterInt[ NSSWoe ] = parameterInt[ NSSWoo ] = 1;
    for( int i=0; i<fEnd; i++ )
	fitParameterFloat[ i ] = false;
    for( int i=0; i<iEnd; i++ )
	fitParameterInt[ i ] = false;
    return;
}



inline bool CalculationParameter::fit() const
{
    for( int i=0; i<fEnd; i++ )
	if( fitParameterFloat[ i ] )
	    return true;
    for( int i=0; i<iEnd; i++ )
	if( fitParameterInt[ i ] )
	    return true;
    return InertialConstants::fit();
}




inline bool CalculationParameter::fit( const ParameterFloat fp ) const
{
    return fitParameterFloat[ fp ];
}



inline bool& CalculationParameter::fit( const ParameterFloat fp )
{
    return fitParameterFloat[ fp ];
}



inline bool CalculationParameter::fit( const ParameterInt fp ) const
{
    return fitParameterInt[ fp ];
}



inline bool& CalculationParameter::fit( const ParameterInt fp )
{
    return fitParameterInt[ fp ];
}



inline InertialConstants::InertialConstants()
{
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=A; i<end; incr( i ) ) {
	    value[ s ][ i ] = 0.0;
	    fitValue[ s ][ i ] = false;
	}
    return;
}



inline InertialConstants::InertialConstants( const InertialConstants& ic )
{
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=A; i<end; incr( i ) ) {
	    value[ s ][ i ] = ic.value[ s ][ i ];
	    fitValue[ s ][ i ] = ic.fitValue[ s ][ i ];
	}
    return;
}



inline InertialConstants& InertialConstants::operator=( const InertialConstants& ic )
{
    if( this == &ic ) // object identity ?
	return *this;
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=A; i<end; incr( i ) ) {
	    value[ s ][ i ] = ic.value[ s ][ i ];
	    fitValue[ s ][ i ] = ic.fitValue[ s ][ i ];
	}
    return *this;
}



inline InertialConstants::~InertialConstants()
{
    return;
}



inline bool InertialConstants::centrifugalDistortion() const
{
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=DJ; i<=dK; incr( i ) )
	    if( fitValue[ s ][ i ] || ( abs( value[ s ][ i ] ) > Constants::eps() ) )
		return true;
    return false;
}




inline double& InertialConstants::constant( const State s, const Constant c )
{
    return value[ s ][ c ];
}



inline double InertialConstants::constant( const State s, const Constant c ) const
{
    return value[ s ][ c ];
}



inline void InertialConstants::defaultValues()
{
    for( Constant c=begin; c<end; incr( c ) ) {
	value[ GroundState ][ c ] = 0.0;
	value[ ExcitedState ][ c ] = 0.0;
    }
    for( Constant c=begin; c<end; incr( c ) ) {
	    fitValue[ GroundState ][ c ] = false;
	    fitValue[ ExcitedState ][ c ] = false;
    }
    return;
}



inline bool InertialConstants::fit() const
{
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=begin; i<end; incr( i ) )
	    if( fitValue[ s ][ i ] )
		return true;
    return false;
}




inline bool& InertialConstants::fit( const State s, const Constant c )
{
    return fitValue[ s ][ c ];
}



inline bool InertialConstants::fit( const State s, const Constant c ) const
{
    return fitValue[ s ][ c ];
}



inline bool InertialConstants::linearDistortion() const
{
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=DX; i<=DZ; incr( i ) )
	    if( fitValue[ s ][ i ] || ( abs( value[ s ][ i ] ) > Constants::eps() ) )
		return true;
    return false;
}




// global functions

inline void incr( CalculationParameter::ParameterFloat& p )
{
    p = static_cast< CalculationParameter::ParameterFloat >( p + 1 );
    return;
}



inline void incr( CalculationParameter::ParameterInt& p )
{
    p = static_cast< CalculationParameter::ParameterInt >( p + 1  );
    return;
}



inline void incr( InertialConstants::State& s )
{
    s = static_cast< InertialConstants::State >( s +  1 );
    return;
}



inline void incr( InertialConstants::Constant& c )
{
    c = static_cast< InertialConstants::Constant >( c + 1 );
    return;
}



#endif
