//
// Copyright (C) 1999, Mark W J Redding <grawlfang@tesco.net>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "Konstellation.h"
#include "FullMap.h"
#include "StarDetails.h"

extern "C"
{
#include "Stars.h"
}

#include <qmsgbox.h>


Konstellation::Konstellation(QWidget* p, const char* n)
	: KTopLevelWidget(n)
{
	currView=(mapView*)0;
	m_bslast=m_bscat=0;

	setupMenu();
	setupStatusBar();
	resize(400,300);
	sView = new QScrollView(this);
	sView->setGeometry(1,30,360,180);
	sView->setVScrollBarMode(QScrollView::Auto);
	sView->setHScrollBarMode(QScrollView::Auto);
	sView->setBackgroundColor(black);

	lZoom = new QLabel(this);
	lZoom->setAutoResize(true);
	lZoom->setText(klocale->translate("Zoom"));

	lMag = new QLabel(this);
	lMag->setAutoResize(true);
	lMag->setText(klocale->translate("Mag."));

	sZoom = new QSlider(this);
	sZoom->setRange(0,100);
	sZoom->setSteps(1,1);
	sZoom->setOrientation(QSlider::Horizontal);
	sZoom->setTickmarks(QSlider::NoMarks);
	sZoom->setTracking(false);
	sZoom->setValue(0);	
	connect(sZoom,SIGNAL(valueChanged(int)),SLOT(setZoom(int)));

	sMag = new QSlider(this);
	sMag->setRange(0,14);
	sMag->setSteps(1,1);
	sMag->setOrientation(QSlider::Horizontal);
	sMag->setTickmarks(QSlider::NoMarks);
	sMag->setTracking(false);
	sMag->setValue(4);	
	connect(sMag,SIGNAL(valueChanged(int)),SLOT(setMagnitude(int)));

	sd = new StarDetails(this);
	hideDetails();

	connect(sd,SIGNAL(quit()),SLOT(hideDetails()));

	m_scale=0;
	m_mag=4.0;
	fullMap();
	setZoom((int)m_scale);
	setMagnitude((int)m_mag);	
	currView->repaint();
}

Konstellation::~Konstellation()
{
	delete sZoom;
	delete sMag;
	delete sd;
}

void
Konstellation::resizeEvent(QResizeEvent* e)
{
	int aw = width();
	int ah = height();

	sView->resize(aw-1,ah-80);

	lZoom->setGeometry(1,ah-33,40,10);
	sZoom->setGeometry(41,ah-33,aw,10);
	lMag->setGeometry(1,ah-46,40,10);
	sMag->setGeometry(41,ah-46,aw/2,10);

	KTopLevelWidget::resizeEvent(e);
}

void
Konstellation::closeEvent(QCloseEvent*)
{
	exitKonst();
}

void
Konstellation::paintEvent(QPaintEvent*)
{
}

void
Konstellation::locate(double x , double y)
{
QString xp(256);

	xp.sprintf("%.2f,%.2f",x,y);
	m_StatusBar->changeItem(xp.data(),0);
}

void Konstellation::centre(int x , int y)
{
	sView->center(x,y);
}

void
Konstellation::setupMenu()
{
	m_FileMenu = new QPopupMenu();
	m_gm_undo = m_FileMenu->insertItem(klocale->translate("Undo"),this,SLOT(undoConnect()));
	m_gm_exit = m_FileMenu->insertItem(klocale->translate("Exit"),this,SLOT(exitKonst()));

	m_ViewMenu = new QPopupMenu();
	m_vm_fmap = m_ViewMenu->insertItem(klocale->translate("Full Map"),this,SLOT(fullMap()));

	m_HelpMenu = new QPopupMenu();
	m_hm_about = m_HelpMenu->insertItem(klocale->translate("About"),this,SLOT(helpAbout()));

	m_Menu = new KMenuBar(this);
	m_Menu->insertItem(klocale->translate("File"),m_FileMenu);
	m_Menu->insertItem(klocale->translate("View"),m_ViewMenu);
	m_Menu->insertItem(klocale->translate("Help"),m_HelpMenu);

	setMenu(m_Menu);
}

void
Konstellation::setupStatusBar()
{
	m_StatusBar = new KStatusBar(this);
	setStatusBar(m_StatusBar);
	m_StatusBar->insertItem("000.00,000.00",0);
	m_StatusBar->insertItem("       ",1);
	m_StatusBar->insertItem("       ",2);
	m_StatusBar->insertItem("            ",3);
	m_StatusBar->insertItem("000.00,000.00",4);
	m_StatusBar->insertItem("                ",5);
	m_StatusBar->insertItem(" ",6);
	m_StatusBar->enable(KStatusBar::Show);
}

void
Konstellation::exitKonst()
{
	emit quit();
}

void
Konstellation::helpAbout()
{
	QMessageBox::about(this,"Konstellation 0.1",
				"Copyright (C) 1999, Mark W J Redding\n"
				"Konstellation comes with ABSOLUTELTY NO WARRANTY\n"
				"This is free software and you are welcome to redistribute it.\n"
				"Contact the author at <grawlfang@tesco.net>\n");
}

void
Konstellation::setZoom(int zoom)
{
QString sc(8);

	m_scale=(double)zoom;
	double zlev=1.0+((double)zoom*0.25);
	currView->setScale(zlev);
	sc.sprintf("%.2f",zlev);
	m_StatusBar->changeItem(sc,1);	
}


void
Konstellation::setMagnitude(int mag)
{
QString sc(8);

	m_mag=(double)mag;
	double zmag=7.0-(double)mag/2.0;
	currView->setMagnitude(zmag);
	sc.sprintf("%.1f",zmag);
	m_StatusBar->changeItem(sc,2);
}

void
Konstellation::countStars(int s)
{
QString sc(5);

	sc.sprintf("%d",s);
	m_StatusBar->changeItem(sc,3);
}

void
Konstellation::showPosition(mapView *view)
{
QString sc(256);

	double x=(double)(sView->contentsX()-7)/view->getScale();
	double y=(double)(sView->contentsY()-7)/view->getScale();
	x=(360.0-x)/15.0;
	if(y>90.0)
		y=0.0-(y-90.0);
	else
		y=90.0-y;
	if(x>24.0) x=24.0;
	if(y>90.0) y=90.0;

	sc.sprintf("%.2f,%.2f",x,y);
	m_StatusBar->changeItem(sc,4);
}

void
Konstellation::cleanView()
{
	if(currView)
	{
		sView->removeChild(currView);
		delete currView;
		currView=(mapView*)0;
	}
}

void
Konstellation::fullMap()
{
	cleanView();
	currView = new FullMap(this,m_scale,m_mag);
	sView->addChild(currView);
	currView->repaint();
}

void
Konstellation::showStar(long bs, int x, int y)
{
	if(!bs)
	{
		m_bslast=0;
		return;
	}

	m_bslast=m_bscat;
	m_bscat=bs;

	QPopupMenu *pop = new QPopupMenu();
	pop->insertItem(klocale->translate("Detail"),this,SLOT(showDetail()));
	pop->insertItem(klocale->translate("Connect"),this,SLOT(connectUp()));
	pop->popup(QPoint(x,y));
	pop->exec();
}

void
Konstellation::connectUp()
{
int sts;

	m_StatusBar->changeItem(" ",5);
	if(!m_bslast) return;
	if(!m_bscat) return;
	sts=::ConnectStars(m_bscat,m_bslast);
	m_StatusBar->changeItem(klocale->translate("connected"),5);
	m_undo_bscat=m_bscat;
	m_undo_bslast=m_bslast;
	m_bscat=m_bslast=0;
	currView->repaint();
}

void
Konstellation::showDetail()
{
	sd->view(m_bscat);
	sd->show();
}

void
Konstellation::hideDetails()
{
	sd->hide();
}

void
Konstellation::undoConnect()
{
int sts;

	if(!m_undo_bscat|!m_undo_bslast) return;
	sts=::UnconnectStars(m_undo_bscat,m_undo_bslast);
	m_undo_bscat=m_undo_bslast=0;
	currView->repaint();
}
