/*
 * readpcf.c    - read PCF font file (PCF is common format)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "font.h"

/*
 * First, read TOC from PCF and build internal PCF INFO
 *      Linked list of PCFREC, as below.
 */

typedef struct _PCFREC  *PCFPTR ;

typedef struct _PCFREC {
    PCFPTR  next   ;
    CARD32  tag    ;
    CARD32  format ;
    CARD32  size   ;
    CARD32  offset ;
    PBYTE   block  ;
} PCFREC ;

/*
 * freePcf - dispose internal representation of PCF
 */

static  void freePcf(PCFPTR pcf)
{
    PCFPTR  nxt ;

    for (nxt = NULL ; pcf != NULL ; pcf = nxt) {
        nxt = pcf->next ;
	free(pcf) ;
    }
}

/*
 * readPcf - read PCF and build internal representation
 */
 
#define PCF_VERSION     0x70636601L

typedef struct _TABREC {
    CARD32  tag    ;
    CARD32  format ;
    CARD32  size   ;
    CARD32  offset ;
} TABREC, TABPTR ;

static  PCFPTR readPcf(FILE *fp)
{
    CARD32  version, count, i ;
    TABREC  tab      ;
    PCFPTR  ptr, top, bot ;

    /*
     * Check version and number of table entires
     *      OS/2 have LSB first CARD32.
     */
     
    if (fread(&version, sizeof(CARD32), 1, fp) != 1) {
        printf("failed to read version\n") ;
        return(NULL) ;
    }
    if (fread(&count, sizeof(CARD32), 1, fp) != 1) {
        printf("failed to read ntables\n") ;
        return(NULL) ;
    }

#ifdef  DEBUG
    printf("PCF Verison %08x, %d tables\n", version, count) ;
#endif
    
    if (version != PCF_VERSION) {
        printf("PCF version mismatch\n") ;
	return(NULL) ;
    }
    if (count == 0) {
        printf("no tables\n") ;
	return(NULL) ;
    }

    /*
     * Read table entries and build internal list
     *      OS/2 have LSB first CARD32.
     */
     
    top = bot = NULL ;
    
    for (i = 0 ; i < count ; i++) {
        if (fread(&tab, sizeof(TABREC), 1, fp) != 1) {
	    printf("prematured EOF\n") ;
	    break ;
	}
	if ((ptr = (PCFPTR) malloc(sizeof(PCFREC) + tab.size)) == NULL) {
	    printf("failed to alloc. data block\n") ;
	    break ;
	}
	memset(ptr, 0, sizeof(PCFREC) + tab.size) ;
	
	ptr->next   = NULL            ;
	ptr->tag    = tab.tag         ;
	ptr->format = tab.format      ;
	ptr->size   = tab.size        ;
	ptr->offset = tab.offset      ;
	ptr->block  = (PBYTE) &ptr[1] ;

        if (top == NULL) {
	    top = bot = ptr ;
	} else {
	    bot->next = ptr ;
	    bot = ptr       ;
	}
    }
    
    /*
     * Read Data Blocks
     *      Byte/bit order of those block unchanged.
     */

    for (ptr = top ; ptr != NULL ; ptr = ptr->next) {
        fseek(fp, ptr->offset, SEEK_SET)    ;
	fread(ptr->block, 1, ptr->size, fp) ;
    } 

#ifdef DEBUG
    for (ptr = top ; ptr != NULL ; ptr = ptr->next) {
        printf("Block %08x %08x Size %8d at %08x\n",
	    ptr->tag, ptr->format, ptr->size, ptr->block) ;
    } 
    fflush(stdout) ;
#endif    

    return(top) ;
}

/*
 * getPcf - refer to PCF Data Blcok specified with TAG
 */

#define PCF_PROPERTIES              (1L<<0)
#define PCF_ACCELERATORS            (1L<<1)
#define PCF_METRICS                 (1L<<2)
#define PCF_BITMAPS                 (1L<<3)
#define PCF_INK_METRICS             (1L<<4)
#define PCF_BDF_ENCODINGS           (1L<<5)
#define PCF_SWIDTHS                 (1L<<6)
#define PCF_GLYPH_NAMES             (1L<<7)
#define PCF_BDF_ACCELERATORS        (1L<<8)

static  PCFPTR  getPcf(PCFPTR top, CARD32 tag)
{
    PCFPTR  ptr ;

    for (ptr = top ; ptr != NULL ; ptr = ptr->next) {
        if (ptr->tag == tag) {
	    return(ptr) ;
        }
    }
    return(NULL) ;
}

/*
 * Formatting Data in Memory
 */

#define PCF_FORMAT_MASK         0xffffff00L
#define PCF_GLYPH_PAD_MASK      0x00000003L
#define PCF_BYTE_MASK           0x00000004L
#define PCF_BIT_MASK            0x00000008L
#define PCF_SCAN_UNIT_MASK      0x00000030L
#define PCF_MASKBITS            0x0000003fL

#define MSBFIRST    0
#define LSBFIRST    1

#define PCF_BYTE_ORDER(f)       (((f) & PCF_BYTE_MASK)?MSBFIRST:LSBFIRST)
#define PCF_BIT_ORDER(f)        (((f) & PCF_BIT_MASK)?MSBFIRST:LSBFIRST)
#define PCF_GLYPH_PAD_INDEX(f)  ((f) & PCF_GLYPH_PAD_MASK)
#define PCF_GLYPH_PAD(f)        (1<<PCF_GLYPH_PAD_INDEX(f))
#define PCF_SCAN_UNIT_INDEX(f)  (((f) & PCF_SCAN_UNIT_MASK) >> 4)
#define PCF_SCAN_UNIT(f)        (1<<PCF_SCAN_UNIT_INDEX(f))
#define PCF_FORMAT_BITS(f)      ((f) & PCF_MASKBITS)

#define PCF_FORMAT_MATCH(a,b) (((a)&PCF_FORMAT_MASK) == ((b)&PCF_FORMAT_MASK))

#define PCF_DEFAULT_FORMAT      0x00000000L
#define PCF_INKBOUNDS           0x00000200L
#define PCF_ACCEL_W_INKBOUNDS   0x00000100L
#define PCF_COMPRESSED_METRICS  0x00000100L

static CARD8   fmtCARD8(CARD8 data, CARD32 format)
{
    return(data) ;
}

static  CARD16  fmtCARD16(CARD16 data, CARD32 format)
{
    CARD16  new ;
    PBYTE   sp, dp ;
    
    if (PCF_BYTE_ORDER(format) == LSBFIRST) {
        return(data) ;
    }

    sp = (PBYTE) &data ;
    dp = (PBYTE) &new  ;

    dp[0] = sp[1] ;
    dp[1] = sp[0] ;
    
    return(new) ;
}

static  CARD32  fmtCARD32(CARD32 data, CARD32 format)
{
    CARD32  new ;
    PBYTE   sp, dp ;

    if (PCF_BYTE_ORDER(format) == LSBFIRST) {
        return(data) ;
    }

    sp = (PBYTE) &data ;
    dp = (PBYTE) &new  ;

    dp[0] = sp[3] ;
    dp[1] = sp[2] ;
    dp[2] = sp[1] ;
    dp[3] = sp[0] ;
    
    return(new) ;
}

/*
 * procPcfProps - Process PCF Font Properties
 */

#pragma pack(1)

typedef struct _PROPREC {
    CARD32  name  ;
    CARD8   isstr ;
    CARD32  value ;
} PROPREC, *PROPPTR ;

#pragma pack()
    
static  BOOL    procPcfProps(PCFPTR pcf, FontPtr pFont)
{
    int     i ;
    CARD32  format, nprops, lenstr ;
    PBYTE   ptr    ;
    PROPPTR pProp  ;
    PBYTE   pStr   ;
    
    /*
     * Refer to Property Block
     */
        
    if ((pcf = getPcf(pcf, PCF_PROPERTIES)) == NULL) {
        printf("no Property Block\n") ;
	return(FALSE) ;
    }
    ptr = pcf->block ;

#ifdef  DEBUG
    printf("Prop. Block %08x %08x Size %8d at %08x\n",
	    pcf->tag, pcf->format, pcf->size, pcf->block) ;
    fflush(stdout) ;
#endif

    /*
     * Map to Property Block
     */
     
    format = *(CARD32 *) ptr ; ptr += sizeof(CARD32) ;

    if (! PCF_FORMAT_MATCH(format, PCF_DEFAULT_FORMAT)) {
        printf("bad format in Props\n") ;
        return(FALSE) ;
    }

    nprops = *(CARD32 *) ptr ; ptr += sizeof(CARD32) ;
    nprops = fmtCARD32(nprops, format) ;
    pProp = (PROPPTR) ptr ; ptr += (sizeof(PROPREC) * nprops) ;

#ifdef  DEBUG
    printf("Found %d Properties @ %08x\n", nprops, pProp) ; fflush(stdout) ;
#endif

    for (i = 0 ; i < nprops ; i++) {
        pProp[i].name  = fmtCARD32(pProp[i].name,  format) ;
        pProp[i].value = fmtCARD32(pProp[i].value, format) ;
        pProp[i].isstr = fmtCARD8(pProp[i].isstr, format)  ;
    }
    
    (CARD32) ptr += 3 ; (CARD32) ptr &= 0xfffffffc ;    /* WRAP */
    lenstr = *(CARD32 *) ptr ; ptr += sizeof(CARD32) ;
    lenstr = fmtCARD32(lenstr, format) ;
    pStr = ptr ; ptr += lenstr ;

#ifdef  DEBUG
    printf("Found %d String Block @ %08x\n", lenstr, pStr) ; fflush(stdout) ;
#endif
    
    if (((CARD32) ptr - (CARD32) pcf->block) > pcf->size) {
        printf("bad size in Props\n") ;
	return(FALSE) ;
    }

#ifdef  DEBUG
    printf("%d Index at %08x, %d String at %08x\n", nprops, lenstr, pProp, pStr) ;
    fflush(stdout) ;        
#endif

    /*
     * Register Properties
     */
    
#ifdef  DEBUG
    for (i = 0 ; i < nprops ; i++) {
	if (pProp[i].isstr) {
	    printf("%s : [%s]\n", &pStr[pProp[i].name], &pStr[pProp[i].value]) ;
	} else {
	    printf("%s : %d\n", &pStr[pProp[i].name], pProp[i].value) ;
	}
    }
#endif

    AllocFontProp(pFont, nprops) ;
    for (i = 0 ; i < nprops ; i++) {
	if (pProp[i].isstr) {
	    AddStrProp(pFont, &pStr[pProp[i].name], &pStr[pProp[i].value]) ;
	} else {
	    AddIntProp(pFont, &pStr[pProp[i].name], pProp[i].value) ;
	}
    }
    return(TRUE) ;
}

/*
 * Converting Bit/Byte Orders
 */
 
unsigned char _reverse_byte[0x100] = {
    0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,
    0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,
    0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
    0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,
    0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,
    0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
    0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,
    0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,
    0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
    0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,
    0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,
    0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
    0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,
    0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,
    0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
    0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,
    0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,
    0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
    0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,
    0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,
    0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
    0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,
    0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,
    0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
    0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,
    0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,
    0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
    0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,
    0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,
    0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
    0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,
    0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff
};

static void BitOrderInvert(PBYTE buff, int len)
{
    while (len > 0) {
        *buff = _reverse_byte[*buff & 0xff] ;
	buff++ ;
	len--  ;
    }
}

static void SwapTwoBytes(PBYTE buff, int len)
{
    BYTE    tmp ;
    
    while (len > 0) {
        tmp = buff[0]     ;
	buff[0] = buff[1] ;
	buff[1] = tmp     ;
	buff += 2 ;
	len  -= 2 ;
    }
}

static void SwapFourBytes(PBYTE buff, int len)
{
    BYTE    tmp ;
    
    while (len > 0) {
        tmp = buff[0]     ;
	buff[0] = buff[3] ;
	buff[3] = tmp     ;
        tmp = buff[1]     ;
	buff[1] = buff[2] ;
	buff[2] = tmp     ;
	buff += 4 ;
	len  -= 4 ;
    }
}

/*
 * Metrics on PCF
 */
 
#pragma pack(1)
typedef struct _METSREC {
    BYTE    left    ;
    BYTE    right   ;
    BYTE    width   ;
    BYTE    ascent  ;
    BYTE    descent ;
} METSREC ;

typedef struct _METLREC {
    INT16   left    ;
    INT16   right   ;
    INT16   width   ;
    INT16   ascent  ;
    INT16   descent ;
    INT16   attr    ;
} METLREC ;
#pragma pack()

static xCharInfo *getMetricsShort(METSREC *base, int n, xCharInfo *m)
{
    METSREC *p = base + n ;
    
    m->leftSideBearing  = (INT16) (p->left    - 0x80) ;
    m->rightSideBearing = (INT16) (p->right   - 0x80) ;
    m->characterWidth   = (INT16) (p->width   - 0x80) ;
    m->ascent           = (INT16) (p->ascent  - 0x80) ;
    m->descent          = (INT16) (p->descent - 0x80) ;
    m->attributes       = (INT16) (0                ) ;
}

static xCharInfo *getMetricsLong(METLREC *base, int n, xCharInfo *m)
{
    METLREC *p = base + n ;

    m->leftSideBearing  = p->left    ;
    m->rightSideBearing = p->right   ;
    m->characterWidth   = p->width   ;
    m->ascent           = p->ascent  ;
    m->descent          = p->descent ;
    m->attributes       = p->attr    ;
}

/*
 * procPcfCharInfo - Process PCF CharInfo (metrics + glyph)
 *      metrics, glyph is indexed with Encoding Table
 */

static  BOOL    procPcfCharInfo(PCFPTR pcf, FontPtr pFont)
{
    PBYTE   ptr ;
    PCFPTR  pcfEnc,  pcfMet,  pcfGly  ;
    CARD32  eformat, mformat, gformat ;
    CARD32  ecounts, mcounts, gcounts ;
    CARD16  *eBase  ;       /* Base of Encoding Info        */
    METLREC *mlBase ;       /* Base of Long  Metrics Info   */
    METSREC *msBase ;       /* Base of Short Metrics Info   */
    CARD32  *goBase ;       /* Base of Glyph Offset Table   */
    BYTE    *gbBase ;       /* Base of Glyph Bitmap Area    */
    CARD32  gbSizes[4] ;
    CARD32  gbLength   ;
    int     enc, idx, row, col ;
    xCharInfo   metrics ;
    int     padbytes, h, w, i, j, slen, dlen ;
    PBYTE   sp, dp ;

    /*
     * Refer to Metrics
     *      mformat <- Format of Metrics Block
     *      mcounts <- Number of Metrics
     *      mlBase  <- Base of Metrics Info. for Long  Format
     *      msBase  <- Base of Metrics Info. for Short Format
     */

    if ((pcfMet = getPcf(pcf, PCF_METRICS)) == NULL) {
        printf("no Metrics Block\n") ;
	return(FALSE) ;
    }

#ifdef  DEBUG
    printf("Met. Block %08x %08x Size %8d at %08x\n",
	    pcfMet->tag, pcfMet->format, pcfMet->size, pcfMet->block) ;
    fflush(stdout) ;
#endif

    ptr = pcfMet->block ;
    mformat = *(CARD32 *) ptr ; ptr += sizeof(CARD32) ;

    mlBase = NULL ;
    msBase = NULL ;

    if (PCF_FORMAT_MATCH(mformat, PCF_DEFAULT_FORMAT)) {
        mcounts = fmtCARD32( *(CARD32 *) ptr, mformat) ;
	ptr += sizeof(CARD32) ;
	mlBase = (METLREC *) ptr ;
    } else if (PCF_FORMAT_MATCH(mformat, PCF_COMPRESSED_METRICS)) {
        mcounts = fmtCARD16( *(CARD16 *) ptr, mformat) ;
	ptr += sizeof(CARD16) ;
	msBase = (METSREC *) ptr ;
    } else {
        printf("bad metrics format\n") ;
	return(FALSE) ;
    }
    
    if (mlBase) {
        for (idx = 0 ; idx < mcounts ; idx++) {
	    mlBase[idx].left    = fmtCARD16(mlBase[idx].left,    mformat) ;
	    mlBase[idx].right   = fmtCARD16(mlBase[idx].right,   mformat) ;
	    mlBase[idx].width   = fmtCARD16(mlBase[idx].width,   mformat) ;
	    mlBase[idx].ascent  = fmtCARD16(mlBase[idx].ascent,  mformat) ;
	    mlBase[idx].descent = fmtCARD16(mlBase[idx].descent, mformat) ;
	    mlBase[idx].attr    = fmtCARD16(mlBase[idx].attr,    mformat) ;
	}
    }
    
#ifdef  DEBUG
    if (msBase != NULL) {
        printf("%d Short Metrics at %08x\n", mcounts, msBase) ;
    } else {
        printf("%d Long  Metrics at %08x\n", mcounts, mlBase) ;
    }
    fflush(stdout) ;
#endif

    /*
     * Refer to Glyph Bimatp
     *      gformat <- Format of Bitmap Block
     *      gcounts <- Numger of Glyphs
     *      goBase  <- Base of Glyph Offsets
     *      gbBase  <- Base of Glyph Bitmaps
     */
     
    if ((pcfGly = getPcf(pcf, PCF_BITMAPS)) == NULL) {
        printf("no Bitmap Block\n") ;
	return(FALSE) ;
    }

#ifdef  DEBUG
    printf("Gly. Block %08x %08x Size %8d at %08x\n",
	    pcfGly->tag, pcfGly->format, pcfGly->size, pcfGly->block) ;
    fflush(stdout) ;
#endif

    ptr = pcfGly->block ;
    gformat = *(CARD32 *) ptr ; ptr += sizeof(CARD32) ;

    if (! PCF_FORMAT_MATCH(gformat, PCF_DEFAULT_FORMAT)) {
        printf("bad Bitmap Format\n") ;
	return(FALSE) ;
    }
    
    gcounts = fmtCARD32( *(CARD32 *) ptr, gformat) ; ptr += sizeof(CARD32) ;
    padbytes = PCF_GLYPH_PAD(gformat)  ;

    goBase = (CARD32 *) ptr ; ptr += (sizeof(CARD32) * gcounts) ;

    gbSizes[0] = fmtCARD32( *(CARD32 *) ptr, gformat) ; ptr += sizeof(CARD32) ;
    gbSizes[1] = fmtCARD32( *(CARD32 *) ptr, gformat) ; ptr += sizeof(CARD32) ;
    gbSizes[2] = fmtCARD32( *(CARD32 *) ptr, gformat) ; ptr += sizeof(CARD32) ;
    gbSizes[3] = fmtCARD32( *(CARD32 *) ptr, gformat) ; ptr += sizeof(CARD32) ;
    gbLength = gbSizes[PCF_GLYPH_PAD_INDEX(gformat)] ;
    
    gbBase = ptr ;
                
    if (gcounts != mcounts) {
        printf("unmatched Metrics/Glyph counts\n") ;
	return(FALSE) ;
    }

    /*
     * Adjusting CARD32 value of Indexes
     */

    for (idx = 0 ; idx < gcounts ; idx++) {
        goBase[idx] = fmtCARD32( goBase[idx], gformat) ;
    }

    /*
     * Adjusting Bit/Byteorder of Bitmaps
     */
    
    if (PCF_BIT_ORDER(gformat) != MSBFIRST) {
        BitOrderInvert(gbBase, gbLength) ;
    }
    if (PCF_BYTE_ORDER(gformat) != MSBFIRST) {
        switch(PCF_SCAN_UNIT(gformat)) {
	case 1 :
	    break ;
	case 2 :
	    SwapTwoBytes(gbBase, gbLength) ;
	    break ;
	case 4 :
	    SwapFourBytes(gbBase, gbLength) ;
	    break ;
        }
    }
    
    /*
     * Refer to Encoding
     *      eformat <- Format of Encoding Block
     *      ecounts <- Number of Encodings
     *      eBase   <- Base of Encoding Info.
     */
        
    if ((pcfEnc = getPcf(pcf, PCF_BDF_ENCODINGS)) == NULL) {
        printf("no Encoding Block\n") ;
	return(FALSE) ;
    }

#ifdef  DEBUG
    printf("Enc. Block %08x %08x Size %8d at %08x\n",
	    pcfEnc->tag, pcfEnc->format, pcfEnc->size, pcfEnc->block) ;
    fflush(stdout) ;
#endif

    ptr = pcfEnc->block ;
    eformat = *(CARD32 *) ptr ; ptr += sizeof(CARD32) ;

    if (! PCF_FORMAT_MATCH(eformat, PCF_DEFAULT_FORMAT)) {
        printf("bad encoding format\n") ;
	return(FALSE) ;
    }

    pFont->firstCol = fmtCARD16( *(CARD16 *) ptr, eformat) ;
    ptr += sizeof(CARD16) ;
    pFont->lastCol  = fmtCARD16( *(CARD16 *) ptr, eformat) ;
    ptr += sizeof(CARD16) ;
    pFont->firstRow = fmtCARD16( *(CARD16 *) ptr, eformat) ;
    ptr += sizeof(CARD16) ;
    pFont->lastRow  = fmtCARD16( *(CARD16 *) ptr, eformat) ;
    ptr += sizeof(CARD16) ;
    ptr += sizeof(CARD16) ; /* Skip Default Char */

    ecounts = (pFont->lastRow - pFont->firstRow + 1)
                * (pFont->lastCol - pFont->firstCol + 1) ;
    eBase   = (CARD16 *) ptr ;

    for (enc = 0 ; enc < ecounts ; enc++) {
        eBase[enc] = fmtCARD16( eBase[enc], eformat) ;
    }

#ifdef  DEBUG
    printf("%d Encodings at %08x\n", ecounts, eBase) ;
    printf("\tRows %d - %d, Cols %d - %d\n",
        pFont->firstRow, pFont->lastRow, pFont->firstCol, pFont->lastCol) ;
#endif

    /*
     * Save metrics + glyph into CharInfo
     */

#define bump(row, col)  \
    if ((col += 1) > pFont->lastCol) {  \
        col = pFont->firstCol ;         \
	row += 1 ;                      \
    }
    
    row = pFont->firstRow ;
    col = pFont->firstCol ;
    for (enc = 0 ; enc < ecounts ; enc++) {

        if (eBase[enc] == 0xffff) {
            bump(row, col) ;
            continue ;
        }
        idx = eBase[enc] ;

#ifdef  DEBUG
        printf("Char %02x %02x, enc %d, idx %d\n", row, col, enc, idx) ;
        fflush(stdout) ;
#endif
        if (msBase) {
            getMetricsShort(msBase, idx, &metrics) ;
        } else {
            getMetricsLong(mlBase, idx, &metrics)  ;
        }
        ptr = AllocCharInfo(pFont, row, col,
	        metrics.leftSideBearing, metrics.rightSideBearing,
	        metrics.ascent, metrics.descent,
		metrics.characterWidth, metrics.attributes) ;
        if (ptr == NULL) {
	    bump(row, col) ;
	    continue ;
	}

        w = metrics.rightSideBearing -  metrics.leftSideBearing ;
        h = metrics.ascent + metrics.descent ;
        dlen = (w + 7) / 8 ;
        slen = ((dlen + padbytes - 1) / padbytes) * padbytes ;
	sp = gbBase + goBase[idx] ;
	dp = ptr ;
	
#ifdef  DEBUG
        printf("Glyph %d w %d h %d offset %d\n", idx, w, h, goBase[idx]) ;
	printf("Copying from %08x to %08x, s-len %d d-len %d\n",
	        sp, dp, slen, dlen) ;
	fflush(stdout) ;
#endif

	for (i = 0 ; i < h ; i++) {
	    for (j = 0 ; j < dlen ; j++) {
	        dp[j] = sp[j] ;
	    }
	    sp += slen ;
	    dp += dlen ;
	}
	bump(row, col) ;
    }
    return(TRUE) ;
}

/*
 * File access routines for PCF font file
 */

FontPtr ReadPcfFont(FILE *fp)
{
    FontPtr     pFont   ;       /* Generating Font Data */
    FontPropPtr pProp   ;
    PCFPTR      pPcf    ;

    /*
     * Create Font Entry
     */

    if ((pFont = CreateFontInfo()) == NULL) {
        return(NULL) ;
    }

    /*
     * Read PCF Font File
     */

    if ((pPcf = readPcf(fp)) == NULL) {
        FreeFontInfo(pFont) ;
        return(NULL) ;
    }
    
    /*
     * Get Font Properties
     */

    if (procPcfProps(pPcf, pFont) != TRUE) {
        FreeFontInfo(pFont) ;
	freePcf(pPcf)       ;
	return(NULL) ;
    }

    /*
     * Get PCF CharInfo (Metrics + Glyph)
     */

    if (procPcfCharInfo(pPcf, pFont) != TRUE) {
        FreeFontInfo(pFont) ;
	freePcf(pPcf)       ;
	return(NULL) ;
    }
    
    /*
     * Final Computing ...
     */

    if ((pProp = GetFontProp(pFont, "FONT")) != NULL) {
        SetFontName(pFont, pProp->str) ;
    }
    ComputeBoundings(pFont) ;
    
    return(pFont) ;
}
