/*
 * futil.c - utilities handling font info. (internal form)
 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>
#include <string.h>
#include "font.h"

/*
 * Some font have empty glyph.  For those font, use DUMMY area for
 * glyph space
 */

static  char    dummy[2] = { 0 , 0 } ;

/*
 * FreeFontInfo - dispose FontRec and linked structures
 *  Uses sub-functions
 *      FreeFontProp    dispose font property
 *      FreeCharInfo    dispose charater info.
 */

static void FreeFontProp(FontPropPtr pfp)
{
    if (pfp == NULL) {
        return ;
    }
    if (pfp->name) {
        free(pfp->name) ;
    }
    if (pfp->str) {
        free(pfp->str) ;
    }   
}

static  void    FreeCharInfo(CharInfoPtr pci)
{
    if (pci == NULL) {
        return ;
    }
    if ((pci->glyph) && (pci->glyph != dummy)) {
        free(pci->glyph) ;
    }
}

void FreeFontInfo(FontPtr pFont)
{
    int         i, j ;
    FontPropPtr	pfp ;
    CharInfoPtr pci ;

    if (pFont == NULL) {
        return ;
    }
    if (pFont->fontname) {
        free(pFont->fontname) ;
    }
    if (pfp = pFont->pFP) {
        for (i = 0 ; i < pFont->numProp ; i++) {
	    FreeFontProp(&pfp[i]) ;
	}
        free(pFont->pFP) ;
    }
    for (i = 0 ; i < 256 ; i++) {
        if (pci = pFont->pCI[i]) {
	    for (j = 0 ; j < 256 ; j++) {
	        FreeCharInfo(&pci[j]) ;
	    }
	    free(pci) ;
	}
    }
    free(pFont) ;
    return ;
}

/*
 * CreateFontInfo - Create font info. entry
 */

FontPtr CreateFontInfo(void)
{
    FontPtr     pFont ;

    if (pFont = (FontPtr) malloc(sizeof(FontRec))) {
        memset(pFont, 0, sizeof(FontRec)) ;
    }
    pFont->firstRow = MAXSHORT ;
    pFont->lastRow  = 0 ;
    pFont->firstCol = MAXSHORT ;
    pFont->lastCol  = 0 ;

    return(pFont) ;
}

/*
 * SetFontName - Register font name
 */

char *SetFontName(FontPtr pFont, char *name)
{
    if ((pFont == NULL) || (name == NULL) || (*name == '\0')) {
        return(NULL) ;
    }
    if (pFont->fontname) {
        free(pFont->fontname) ;
    }

    pFont->fontname = strdup(name) ;
    
    return(pFont->fontname) ;
}

/*
 * AllocFontProp - allocate space for Font Properies
 */

#define PROP_STEP   20

FontPropPtr AllocFontProp(FontPtr pFont, int nprops)
{
    int         bytes = sizeof(FontPropRec) * (nprops + PROP_STEP) ;
    FontPropPtr pFP ;

    if ((pFont == NULL) || (nprops <= 0)) {
        return(NULL) ;
    }    
    if ((pFP = malloc(bytes)) == NULL) {
        return(NULL) ;
    }
    memset(pFP, 0, bytes) ;
    
    pFont->sizProp = nprops ;
    pFont->numProp = 0 ;
    pFont->pFP = pFP ;

    return(pFP) ;
}

/*
 * ReallocFontProp - enlarge prop area
 */

static FontPropPtr ReallocFontProp(FontPtr pFont)
{
    FontPropPtr pFP ;
    int         bytes = sizeof(FontPropRec) * (pFont->sizProp + PROP_STEP) ;

    if ((pFont == NULL) || (pFont->pFP == NULL)) {
        return(NULL) ;
    }
    if ((pFP = realloc(pFont->pFP, bytes)) == NULL) {
        return(NULL) ;
    }
    memset(&pFP[pFont->sizProp], 0, (sizeof(FontPropRec) * PROP_STEP)) ;
    pFont->pFP = pFP ;
    pFont->sizProp += PROP_STEP ;
    
    return(pFP) ;
}

/*
 * GetFontProp - refer to name'd property
 */

FontPropPtr GetFontProp(FontPtr pFont, char *name)
{
    int             i ;
    FontPropPtr     pFP ;

    if ((pFont == NULL) || (pFont->pFP == NULL)) {
        return(NULL) ;
    }
    if ((name == NULL) || (*name == '\0')) {
        return(NULL) ;
    }

    for (i = 0, pFP = pFont->pFP ; i < pFont->numProp ; i++, pFP++) {
        if (strcmp(pFP->name, name) == 0) {
	    return(pFP) ;
	}
    }
    return(NULL) ;
}
 
/*
 * AddStrProp - add property with string value
 */

FontPropPtr AddStrProp(FontPtr pFont, char *name, char *str)
{
    FontPropPtr     pFP ;

    if ((pFont == NULL) || (pFont->pFP == NULL)) {
        return(NULL) ;
    }
    if ((name == NULL) || (*name == '\0')) {
        return(NULL) ;
    }
    if ((str == NULL) || (*str == '\0')) {
        str = "" ;
    }

    if (pFont->numProp >= pFont->sizProp) {
        if (ReallocFontProp(pFont) == NULL) {
	    return(NULL) ;
	}
    }
    if ((pFP = GetFontProp(pFont, name)) == NULL) {
        pFP = &pFont->pFP[pFont->numProp] ;
        pFP->name = strdup(name) ;
        pFont->numProp += 1 ;
    }
    if (pFP->str) {
        free(pFP->str) ;
    }
    pFP->str = strdup(str)  ;
    pFP->val = 0 ;
    return(pFP) ;	    
}

/*
 * AddIntProp - add property with integer value
 */

FontPropPtr AddIntProp(FontPtr pFont, char *name, long  val)
{
    FontPropPtr     pFP ;

    if ((pFont == NULL) || (pFont->pFP == NULL)) {
        return(NULL) ;
    }
    if ((name == NULL) || (*name == '\0')) {
        return(NULL) ;
    }

    if (pFont->numProp >= pFont->sizProp) {
        if (ReallocFontProp(pFont) == NULL) {
	    return(NULL) ;
	}
    }
    if ((pFP = GetFontProp(pFont, name)) == NULL) {
        pFP = &pFont->pFP[pFont->numProp] ;
        pFP->name = strdup(name) ;
        pFont->numProp += 1 ;
    }
    if (pFP->str) {
        free(pFP->str) ;
    }
    pFP->str = NULL ;
    pFP->val = val  ;
    return(pFP) ;	    
}

/*
 * AllocCharInfo - Set metrics and allocate glyph area
 */

char *AllocCharInfo(FontPtr pFont, int row, int col,
            int left, int right, int ascent, int descent,
	    int width, int attr)
{
    int         w, h, bytes ;
    CharInfoPtr pci ;
    char        *p ;

    if (pFont == NULL) {
        return(NULL) ;
    }
    if ((row < 0) || (row >= 256)) {
        return(NULL) ;
    }
    if ((col < 0) || (col >= 256)) {
        return(NULL) ;
    }

    if (pFont->pCI[row] == NULL) {
        if ((pci = malloc(sizeof(CharInfoRec) * 256)) == NULL) {
	    return(NULL) ;
	}
	memset(pci, 0, (sizeof(CharInfoRec) * 256)) ;
	pFont->pCI[row] = pci ;
    }
    pci = pFont->pCI[row] ;

    h = descent + ascent ;
    w = right - left ;
    bytes = ((w + 7) >> 3) * h ;

    if (bytes == 0) {
        p = dummy ;
    } else if ((p = malloc(bytes)) == NULL) {
        return(NULL) ;
    }
    if (pci[col].glyph) {
        free(pci[col].glyph) ;
    }
    memset(p, 0, bytes) ;
    pci[col].glyph = p  ;
    pci[col].metrics.leftSideBearing  = left  ;
    pci[col].metrics.rightSideBearing = right ;
    pci[col].metrics.ascent  = ascent  ;
    pci[col].metrics.descent = descent ;
    pci[col].metrics.characterWidth = width   ;
    pci[col].metrics.attributes = attr ;

    pFont->firstRow = min(pFont->firstRow, row) ;
    pFont->lastRow  = max(pFont->lastRow,  row) ;
    pFont->firstCol = min(pFont->firstCol, col) ;
    pFont->lastCol  = max(pFont->lastCol,  col) ;

    return(p) ;    
}

/*
 * GetCharInfo - Refer to Char Info.
 */

CharInfoPtr GetCharInfo(FontPtr pFont, int row, int col)
{
    CharInfoPtr     pci ;

    if (pFont == NULL) {
        return(NULL) ;
    }
    if ((row < 0) || (row >= 256)) {
        return(NULL) ;
    }
    if ((col < 0) || (col >= 256)) {
        return(NULL) ;
    }

    if ((pci = pFont->pCI[row]) == NULL) {
        return(NULL) ;
    }
    if (pci[col].glyph == NULL) {
        return(NULL) ;
    }
    return(&pci[col]) ;
}

/*
 * ComputeBoundings - compute min/max boundings of the font
 */

void ComputeBoundings(FontPtr pFont)
{
    int         row, col ;
    CharInfoPtr pci      ;
    xCharInfo   *minp = &pFont->minbound ;
    xCharInfo   *maxp = &pFont->maxbound ;

    minp->ascent  = MAXSHORT ;
    minp->descent = MAXSHORT ;
    minp->leftSideBearing  = MAXSHORT ;
    minp->rightSideBearing = MAXSHORT ;
    minp->characterWidth  = MAXSHORT ;
    minp->attributes      = 0xffff ;    /* all bits on */

    maxp->ascent  = MINSHORT ;
    maxp->descent = MINSHORT ;
    maxp->leftSideBearing  = MINSHORT ;
    maxp->rightSideBearing = MINSHORT ;
    maxp->characterWidth  = MINSHORT ;
    maxp->attributes      = 0x0000 ;    /* all bits off */

#define MINMAX(field) \
	if (minp->field > pci->metrics.field) \
	     minp->field = pci->metrics.field; \
	if (maxp->field < pci->metrics.field) \
	     maxp->field = pci->metrics.field;

    for (row = pFont->firstRow ; row <= pFont->lastRow ; row++) {
        for (col = pFont->firstCol ; col <= pFont->lastCol ; col++) {
	    if ((pci = GetCharInfo(pFont, row, col)) == NULL) {
	        continue ;
	    }
	    MINMAX(ascent);
	    MINMAX(descent);
	    MINMAX(leftSideBearing);
	    MINMAX(rightSideBearing);
	    MINMAX(characterWidth);
            minp->attributes &= pci->metrics.attributes;
            maxp->attributes |= pci->metrics.attributes;
	}
    }
#undef  MINMAX
}

/*
 * ComputeWeight - compute weight of the font
 */

long ComputeWeight(FontPtr pFont)
{
    return (0L) ;   /* may be not used on XFONT program */
}

/*
 * DumpFontInfo - Dump font info. for test
 */

static  void dumpmetrics(char *name, xCharInfo *mp)
{
    printf("%s\n", name) ;
    printf("left %d, right %d, ascent %d, descent %d, width %d\n",
                mp->leftSideBearing,
                mp->rightSideBearing,
                mp->ascent,
                mp->descent,
                mp->characterWidth) ;
}

static void dumpprop(FontPropPtr pFP)
{
    if (pFP->str) {
        printf("%s : %s\n", pFP->name, pFP->str) ;
    } else {
        printf("%s : %ld\n", pFP->name, pFP->val) ;
    }
}

static void dumpglyph(CharInfoPtr pci)
{
    int     w, h, r, b, bpr ;
    char    *p ;
    
    h = pci->metrics.descent + pci->metrics.ascent ;
    w = pci->metrics.rightSideBearing - pci->metrics.leftSideBearing ;
    bpr = (w + 7) >> 3 ;

    for (r = 0 ; r < h ; r++) {
        p = pci->glyph + (r * bpr) ;
	for (b = 0 ; b < w ; b++) {
	    putchar((p[b>>3] & (1<<(7-(b&7)))) ? '#' : '-') ;
	}
	putchar('\n') ;
    }
}

static  char buff[32] ;

void DumpFontInfo(FontPtr pFont)
{
    int         i, j ;
    CharInfoPtr pci ;

    printf("%s : col %d - %d row %d - %d\n", pFont->fontname,
                            pFont->firstCol, pFont->lastCol,
                            pFont->firstRow, pFont->lastRow) ;
    dumpmetrics("minbound", &pFont->minbound) ;
    dumpmetrics("maxbound", &pFont->maxbound) ;
    
    for (i = 0 ; i < pFont->numProp ; i++) {
        dumpprop(&pFont->pFP[i]) ;
    }

    for (i = pFont->firstRow ; i <= pFont->lastRow ; i++) {
	for (j = pFont->firstCol ; j <= pFont->lastCol ; j++) {
	    if (pci = GetCharInfo(pFont, i, j)) {
	        sprintf(buff, "\nCHAR %02x %02x", i, j) ;
	        dumpmetrics(buff, &(pci[i].metrics)) ;
		dumpglyph(pci) ;
	    }
	}
    }
}
