/* [wscrdrv.c wk 31.10.91] Simple Full Screen Handler
 *	Copyright (c) 1988-94 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * (originally developed for W)
 *
 * History:
 * 27.01.93 wk	C Set/2 Support
 * 06.05.93 wk	Moved GUI support to wpmdrv.c
 */


#include <wk/tailor.h>

#if !UNIX  /* see wscrans.c for an UNIX compatible version */

#define DBG_SCANCODES 0


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/environ.h>
#define INCL_ALT_KEYCODES 1
#include <wk/keys.h>
#include <wk/hwacc.h>
#if MSDOS
#elif OS2
#elif DOS386
#elif DOS16RM
#elif EMX
#elif WINNT
#else
    #error Operating system must either be MSDOS, OS2, WINNT or EMX
#endif
#include <wk/mouse.h>
#include <wk/bios.h>
#if EMX && !OS20
    #include <os2.h>
#elif defined(OS20)
    #define INCL_DOSPROCESS 1
    #include <os2.h>
    #define INCL_WK_WSCRDRV_GUI 1
#elif defined(OS2)
    #define INCL_DOSPROCESS 1
    #define INCL_KBD 1
    #define INCL_VIO 1
    #if OS20
       #define INCL_DOSDEVIOCTL 1
    #endif
    #include <os2.h>
    #define INCL_WK_WSCRDRV_GUI 1
    /* #define HAVE_OLD_OS2_HEADERS 1 */
#elif WINNT
#else
    #include <bios.h>
    #include <dos.h>
  #if __ZTC__
    #include <sound.h>
  #endif
    #define HAB long
#endif
#include <wk/wscrdrv.h>

/****** constants **********/
#define MAX_DOSKEYS  20  /* our own pretype buffer for MSDOS */

/******* types *************/
typedef struct {
	char c;
	char a;
    } cell_t;

#if __IBMC__
typedef volatile cell_t * crtptr_t;
#elif __WATCOMC__ && DOS386
typedef volatile cell_t *crtptr_t;
#else
typedef volatile cell_t _far *crtptr_t;
#endif

/******** globals **********/
#if OS2 || EMX
    static crtptr_t vioBuffer ;
    static int hvio;
#endif
#if !OS2
    static struct {
	ushort chr,scan;
	int state;
    } kbdBuf[MAX_DOSKEYS]; /* array to store the keystrokes */
    static int kbdBufCnt;  /* number of keys in buffer */
    static crtptr_t screenBase ;
#endif

static int isColor ;		  /* flag for color-mode */
static struct {
	int sx, sy, sxMul2;
	int x2, y2 ;
	int crsX, crsY ;
	int crsForm ;
	int crsActiv ;
	int size ;		  /*total size*/
	int buttons;		  /* # of mouse button s or 0 when no mouse*/
    } screen ;			  /* Screen Parameters*/


static int crtSaveFlag ;

static struct {
	int videoMode;
	ushort cursorForm;
	int crsX, crsY;
	struct { int px, py; } mouse;
	ushort orgScrSize;	    /* # of elem in orgScr */
	ushort orgScrRows;
	ushort orgScrCols;
	cell_t *orgScr; 	    /* area to save org Screen */
    } save ;			    /* save original screen Environment*/


static cell_t *image ;	   /* area to store the image */
static int translateCodes;

static volatile int *breakFlagAddr;

/********** prototypes *********/
static cell_t GetRndCell();
static int GenericInkey( ushort *rVkey, ushort *rVal,
		      ushort *rScan, ushort *rRpt, ushort *rFlags );
static ushort VkeyCodeMap( int keycode, ushort scan );

/*********** defined functions ********/
/*********** functions ****************/

/****************
 * Returns: 0 = standard color screen
 *	    1 = standard mono screen
 */


int ScrDrvOpen()
{
    int err=0, cols, rows, pages , i ;
  #if OS20
  #elif EMX && !OS20
    USHORT * _Seg16 vBuffer;
    ULONG vioBufferSize;
  #elif OS2
    crtptr_t vBuffer;
    ushort vioBufferSize;
  #endif

    save.videoMode = -1 ;

  #if DEBUG
    xassert( sizeof (cell_t) == 2 );
  #endif
  #if OS2 || EMX
   #if EMX && !OS20
    if( _osmode == OS2_MODE ) {
   #endif
	save.videoMode = -1;
	hvio = 0;
	isColor = WKBiosGetVideoMode( &cols, &rows ) != 7;
   #if EMX && !OS20
    }
    else {  /* _osmode == DOS_MODE */
   #endif
  #endif
  #if !OS2  /* DOS or EMX continued */
	isColor = 1 ; /* default*/
	save.videoMode = WKBiosGetVideoMode( &cols, &rows );
	switch( save.videoMode ) {
	  case 0x07: isColor = 0; save.videoMode = -1;break;
	  case 0x02: case 0x03: case 0x23: case 0x4D: case 0x17:
	  case 0x4F: case 0x57: case 0x19: case 0x1A: case 0x24:
	  case 0x45: case 0x32: case 0x2A: case 0x4E: case 0x40:
	  case 0x18: case 0x33: case 0x22: case 0x44: case 0x43:
	  case 0x26: case 0x5A: case 0x34:  save.videoMode = -1; break;
	  default:
	    rows = 25;
	    WKBiosSetVideoMode( 3 ) ; /* change Vidomode to Standard*/
	    break ;
	}
	kbdBufCnt = 0;
   #if EMX && !OS20
    }
   #endif
  #endif /* DOS */

    /**** Setup Screen Parameters ****/
    screen.sx = cols ;
    screen.sy = rows ;
    screen.sxMul2 = cols*2 ;
    screen.x2 = cols -1 ;
    screen.y2 = rows -1 ;
    screen.size = screen.sx * screen.sy ;
    if( !(image = calloc( screen.size, sizeof *image )) )
	{ err = -1; goto retlab ; }

  #if OS20
    vioBuffer = (crtptr_t)WKVideoGetBuffer( NULL );
  #elif OS2 || EMX
   #if EMX && !OS20
    if( _osmode == OS2_MODE ) {
   #endif
	 #if !EMX || OS20
	    VioGetBuf( (PULONG)&vBuffer, (PUSHORT)&vioBufferSize, hvio);
	    vioBuffer = (crtptr_t)vBuffer;
	 #endif
   #if EMX && !OS20
    }
    else
	screenBase = HW_MakePtrToMemory( isColor ? 0xb8000 : 0xb0000 ,
							  screen.size*2 );
   #endif
  #else
    screenBase = HW_MakePtrToMemory( isColor ? 0xb8000 : 0xb0000,
							  screen.size*2 );
  #endif

    save.orgScrSize = screen.size;
    save.orgScrRows = screen.sy;
    save.orgScrCols = screen.sx;
    if( !(save.orgScr = malloc( save.orgScrSize * sizeof *save.orgScr)) )
	{ err = -1; goto retlab ; }
    ScrDrvMouseOn();
    ScrDrvSaveScreen();
    ScrDrvShowCrs( 1 ); /* initial painting of cursor */
    ScrDrvHideCrs();	/* and hide him */
    ScrDrvRepaint();	/* Sync with image */
    MouseShow();

  retlab:
    if( err && save.videoMode != -1 )  /* Reset Videomode on error*/
	WKBiosSetVideoMode( save.videoMode ) ;
    if( err )
	Error(4,"Can't open CRT: not enough core");
    return isColor ? 0 : 1;
}



#ifdef OS2
HAB
#else
int
#endif
ScrDrvGetActiveHAB(void)
{
    return 0;
}


/****************
 *  enables or disables code translation
 */

void ScrDrvCodeTrans( int mode )
{
    translateCodes = mode ;
}

void ScrDrvRegBreakFlag( volatile int *flag )
{
    breakFlagAddr = flag;
}

void ScrDrvClose()
{
    ScrDrvRegBreakFlag( NULL );
    FREE( image );
    FREE( save.orgScr );
}



void ScrDrvQrySize( int *x, int *y )
{
    *x = screen.sx ;
    *y = screen.sy ;
}


void ScrDrvSetCrs( int x, int y )
{
    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
    screen.crsY=y;
    screen.crsX=x ;
    if( screen.crsActiv )
	WKBiosSetCursorPos( 0, screen.crsY, screen.crsX ) ;
}


void ScrDrvHideCrs()
{
    screen.crsActiv = 0 ;
    WKBiosSetCursorPos( 0, screen.sy, 0) ;  /* will hide cursor */
}

void ScrDrvMouseOn()
{
    screen.buttons = MouseInitialize();
    MousePut(save.mouse.px, save.mouse.py);
}


void ScrDrvMouseOff()
{
    MouseGet( &save.mouse.px, &save.mouse.py );
    MouseTerminate();
    screen.buttons = 0;
}


void ScrDrvShowCrs( int form )
{
    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
    if( form == screen.crsForm )
	;
    else if( form == 1 ) {
      #if OS2
	WKBiosSetCursorForm( 0x0c0f ) ;
      #else
	WKBiosSetCursorForm( isColor? 0x0607:  0x0b0c ) ;
      #endif
	screen.crsForm = 1 ;
    } else if( form == 2 ) {
      #if OS2
	WKBiosSetCursorForm( 0x070f ) ;
      #else
	WKBiosSetCursorForm( isColor? 0x0307:0x060c ) ;
      #endif
	screen.crsForm = 2 ;
    }
    screen.crsActiv = 1 ;
    WKBiosSetCursorPos( 0, screen.crsY, screen.crsX ) ;
}


void ScrDrvSaveScreen()
{
    cell_t *mPtr;
    crtptr_t sPtr;
    ushort cnt;

  #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB to switch to output Screen */
    int dmy1, dmy2;	/* by calling video Bios */
    WKBiosGetVideoMode( &dmy1, &dmy2 );
  #endif
    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if !OS2
    MouseHide();
  #endif
    mPtr = save.orgScr;
  #if OS2
    if( !vioBuffer )
	return;
    sPtr = vioBuffer;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(mPtr++) = *(sPtr++);
  #else
    sPtr = screenBase;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(mPtr++) = *(sPtr++);
  #endif
  #if !OS2
    MouseShow();
  #endif
    save.cursorForm = WKBiosGetCursorForm() ;
    WKBiosGetCursorPos( 0, &save.crsY, &save.crsX );
}


void ScrDrvRestScreen()
{
    cell_t *mPtr;
    crtptr_t sPtr;
    ushort cnt;


  #if __ZTC__ && DEBUG && !OS2	/* Tell ZDB to switch to output Screen */
    int dmy1, dmy2;	/* by calling video Bios */
    WKBiosGetVideoMode( &dmy1, &dmy2 );
  #endif
  #if !OS2
    MouseHide();
  #endif
    mPtr = save.orgScr;
  #if OS2
    if( !vioBuffer )
	return;
    sPtr = vioBuffer;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(sPtr++) = *(mPtr++);
  #else
    sPtr = screenBase;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(sPtr++) = *(mPtr++);
  #endif
  #if !OS2
    MouseShow();
  #endif
    WKBiosSetCursorForm( save.cursorForm ) ;
    WKBiosSetCursorPos( 0, save.crsY, save.crsX ) ;
}



void ScrDrvShow()
{
  #if OS20
    MouseHide();
    WKVideoShowBuffer( screen.size*2 );
    MouseShow();
  #elif OS2
    MouseHide();
    VioShowBuf( 0, screen.size*2, hvio);
    MouseShow();
  #endif
}


void ScrDrvRepaint()
{
    int i;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
  #if !OS2
    MouseHide();
  #endif
    for(i=0; i < screen.size; i++ ) {
      #if OS2
	vioBuffer[i] = image[i];
      #else /* DOS*/
	screenBase[i] = image[i];
      #endif
    }

  #if !OS2
    MouseShow();
  #endif
}



/****************
 * Query the Mouse
 * Returns: Number of Buttons or 0 if no Mouse available in this screen Area
 */

int ScrDrvQryMouse( int *btn, int *px, int *py )
{
    *btn = MouseGet( px, py );
    return screen.buttons;
}


void ScrDrvReadCell( int x, int y, int *c, int *a )
{
    int i;

    i = y * screen.sx + x ;
    if( i >= 0 && i < screen.size ) {
	*c = translateCodes ? MapIbm2Iso( image[i].c, 1 ) : image[i].c;
	*a = image[i].a;
    }
}


void ScrDrvWriteCell( int x, int y, int c, int a )
{
    int i;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    i = y * screen.sx + x ;
    if( c & 0x80 )
	if( translateCodes )
	    c = MapIso2Ibm(c,1);
    if( i >= 0 && i < screen.size ) {
	if( image[i].c != c || image[i].a != a ) {
	    image[i].c = c ;
	    image[i].a = a ;
	  #if !OS2
	    MouseHideCond( x, y, x+1, y);
	  #endif
	    #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB to switch to output Screen */
	    {
	      int dmy1, dmy2;	  /* by calling video Bios */
	      WKBiosGetVideoMode( &dmy1, &dmy2 );
	    }
	    #endif
	  #if OS2
	    vioBuffer[i] = image[i];
	  #else /* DOS*/
	    screenBase[i] = image[i];
	  #endif
	  #if !OS2
	    MouseShow();
	  #endif
	}
    }
}


void ScrDrvWriteStr( int x, int y, const char *buf, size_t len, int a )
{
    int i, anyupd, c;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; len && i < screen.size; len--, i++, buf++ ) {
	    c = *buf;
	    if( c & 0x80 )
		if( translateCodes )
		    c = MapIso2Ibm(c,1);
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		    #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB */
		    {
		      int dmy1, dmy2;	  /* by calling video Bios */
		      WKBiosGetVideoMode( &dmy1, &dmy2 );
		    }
		    #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	}
  #if !OS2
    if( anyupd )
	MouseShow();
  #endif
}


/****************
 * Einen String der laenge len ausgeben und rechts mit blanks
 * bis auf laenge maxlen auffuellen. Attribute werden nicht geschrieben.
 * der String wird aber nur solange ausgegeben bis auch maxlen erreicht ist;
 * nur ein eventuell kuerzerer String wird mit blanks aufgefuellt.
 */

void ScrDrvWriteStrFill( int x, int y, const char *buf,
			 size_t len, size_t maxlen )
{
    int i, anyupd, c;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 ) {
	for( ; maxlen && len && i < screen.size; len--, maxlen--,i++, buf++ ) {
	    c = *buf;
	    if( c & 0x80 )
		if( translateCodes )
		    c = MapIso2Ibm(c,1);
	    if( image[i].c != c ) {
		image[i].c = c ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		    #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB */
		    {
		      int dmy1, dmy2;	  /* by calling video Bios */
		      WKBiosGetVideoMode( &dmy1, &dmy2 );
		    }
		    #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	}
	for( ; maxlen && i < screen.size; maxlen--,i++ ) {
	    if( image[i].c != ' ' ) {
		image[i].c = ' ' ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		    #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB */
		    {
		      int dmy1, dmy2;	  /* by calling video Bios */
		      WKBiosGetVideoMode( &dmy1, &dmy2 );
		    }
		    #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	}
    }
  #if !OS2
    if( anyupd )
	MouseShow();
  #endif
}



void ScrDrvWriteNChar( int x, int y, int c, int a, size_t n )
{
    int i, anyupd;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( c & 0x80 )
	if( translateCodes )
	    c = MapIso2Ibm(c,1);
    if( i >= 0 )
	for( ; n && i < screen.size; n--, i++ ) {
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+n, y);
		  #endif
		    #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB */
		    {
		      int dmy1, dmy2;	  /* by calling video Bios */
		      WKBiosGetVideoMode( &dmy1, &dmy2 );
		    }
		    #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio) ;
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	}
  #if !OS2
    if( anyupd )
	MouseShow();
  #endif
}


/****************
 * Nur Attribute schreiben
 */

void ScrDrvWriteNAttr( int x, int y, int a, size_t n )
{
    int i, anyupd;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; n && i < screen.size; n--, i++ ) {
	    if( image[i].a != a ) {
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+n, y);
		  #endif
		    #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB */
		    {
		      int dmy1, dmy2;	  /* by calling video Bios */
		      WKBiosGetVideoMode( &dmy1, &dmy2 );
		    }
		    #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio) ;
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	}
  #if !OS2
    if( anyupd )
	MouseShow();
  #endif
}



/****************
 * Mode = 1 : Put Crt into Pause Mode
 *	  0 : Restore CRT if CRT  is in Pause Mode
 */

void ScrDrvPauseCrt( int mode )
{
  #if OS2 && 0
    ushort x,y;
  #else
    crtptr_t sPtr;
  #endif
    ushort cnt;
    cell_t cell;
    int i;

    if( !crtSaveFlag && !mode )
	return;
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    if( mode ) {
      #if __ZTC__ && DEBUG && !OS2  /* Tell ZDB to switch to output Screen */
	int dmy1, dmy2;     /* by calling video Bios */
	WKBiosGetVideoMode( &dmy1, &dmy2 );
      #endif
	cell.c = ' ';
	cell.a = 7;
      #if !OS2
	MouseHide();
      #endif
	cnt  = save.orgScrSize;
      #if OS2 && 0
	if( !crtSaveFlag ) {
	    for( y=0 ; y < save.orgScrRows ; y++ )
		VioWrtNCell( (PCH)&cell, 2*screen.sx, y, 0, hvio ) ;
	} else {
	    for(i=0; i < 3; i++ ) {/* write some random cells */
		cell = GetRndCell();
		i = rand() % cnt;
		VioWrtCellStr( (PCH)&cell, 2, i / screen.sx, i % screen.sx,hvio);
	    }
	}
      #elif OS2
	sPtr = vioBuffer;
	if( !crtSaveFlag ) {
	    while( cnt-- )  /* clear screen at first call */
		*(sPtr++) = cell;
	    cnt  = save.orgScrSize;
	}
	for(i=0; i < 3; i++ ) /* write some random cells */
	    sPtr[ rand() % cnt ] = GetRndCell();
      #else
	sPtr = screenBase;
	if( !crtSaveFlag ) {
	    while( cnt-- )  /* clear screen at first call */
		*(sPtr++) = cell;
	    cnt  = save.orgScrSize;
	}
	for(i=0; i < 3; i++ ) /* write some random cells */
	    sPtr[ rand() % cnt ] = GetRndCell();
      #endif
      crtSaveFlag = 1;
    }
    else if( crtSaveFlag ) {
	crtSaveFlag = 0;
	ScrDrvRepaint();
  #if !OS2
	MouseShow();
  #endif
    }
}




/**********
 * Put one Cell to the screen
 */


static cell_t GetRndCell()
{
    cell_t val;

    switch( rand() % 1000 ) {
      case 1: val.c = '.';        break;
      case 2: val.c = '.';        break;
      case 3: val.c = '.';        break;
      case 4: val.c = '.';        break;
      case 5: val.c = '.';        break;
      case 6: val.c = '.';        break;
      case 7: val.c = '.';        break;
      case 8: val.c = '.';        break;
      case 9: val.c = '+';        break;
      case 10:val.c = '+';       break;
      case 11:val.c = '+';       break;
      case 12:val.c = '*';       break;
      case 13:val.c = '.';       break;
      case 14:val.c = 'w';       break;
      case 15:val.c = '.';       break;
      case 16:val.c = '*';       break;
      case 17:val.c = 'W';       break;
      case 18:val.c = '.';       break;
      default:val.c = ' ';        break;
    }
    val.a = (rand() % 6)+1;
    return val;
}


/****************
 * Ermittelt die aktuellen Togglekey states:
 * Bit 0 = Scroll Lock
 *     1 = Num Lock
 *     2 = Caps Lock
 */

unsigned ScrDrvGetToggleKeys()
{
    unsigned state;

  #if OS20
    state = WKKeyboardGetToggle();
  #elif OS2
    KBDINFO kbi;

    state = 0;
    kbi.cb = sizeof kbi;
    if( !KbdGetStatus( &kbi, 0 ) )
	state = (kbi.fsState >> 4) & 0x07;
  #elif EMX && !OS20
    state = 0; /* no way yet found to determine */
  #elif __WATCOMC__
    state = (_bios_keybrd(_KEYBRD_SHIFTSTATUS) >> 4) & 0x07;
  #else
    state = (bioskey(0x02) >> 4) & 0x07;
  #endif
    return state;
}


int ScrDrvInkey()
{
    static ushort repeatCnt;
    static int	  repeatChr;
    int keycode;
    ushort vkey, val, scan, rpt, flags;

    if( repeatCnt ) {
	repeatCnt--;
	return repeatChr;
    }

    keycode = GenericInkey(&vkey, &val, &scan, &rpt, &flags );
    if( keycode && rpt > 1 ) {
	repeatChr = keycode;
	repeatCnt = rpt - 1 ;
    }
    return keycode;
}


int ScrDrvGetKey( ushort *rVkey, ushort *rChr,
		  ushort *rScan, ushort *rRpt, ushort *rFlags )
{
    int keycode;
    ushort vkey, val, scan, rpt, flags;

    keycode = GenericInkey(&vkey, &val, &scan, &rpt, &flags );

    if( rVkey )
	*rVkey = vkey;
    if( rChr )
	*rChr  = val;
    if( rScan )
	*rScan = scan;
    if( rRpt )
	*rRpt  = rpt;
    if( rFlags )
	*rFlags =flags;
    return keycode;
}


static int GenericInkey( ushort *rVkey, ushort *rVal,
		      ushort *rScan, ushort *rRpt, ushort *rFlags )
{
    ushort flags, vkey, chr, rpt, scan;
    int keycode, c, state, i, msX, msY, msEvent;
    unsigned msBtn;
    static unsigned lastMsBtn;
  #if OS20
    unsigned kstate;
  #elif OS2
    KBDKEYINFO kbci;
  #endif

    msEvent = 0;
    keycode = 0;
    vkey = chr = scan = rpt = flags = 0;
    /* testen ob ein Zeichen vorhanden ist */
  #if OS20
    if( WKKeyboardGetChar( 0, &chr, &scan, &kstate ) )
	goto endLabel; /* fehler (z.B. kein focus */
  #elif OS2
    if( KbdCharIn( &kbci, IO_NOWAIT, 0 ) )
	goto endLabel; /* fehler (z.B. kein focus */
    if( !(kbci.fbStatus & ~(
			  #if OS20 || !defined(HAVE_OLD_OS2_HEADERS)
			    KBDTRF_SHIFT_KEY_IN | KBDTRF_INTERIM_CHAR_IN |
			    KBDTRF_CONVERSION_REQUEST
			  #else
			    SHIFT_KEY_IN | INTERIM_CHAR_IN |
			    CONVERSION_REQUEST
			  #endif
			    )))
	goto endLabel; /* not a useful character */
    #if DBG_SCANCODES
      printf("K=**** vkey=**** chr=%4x scan=%4x state=%016b\n",
		       kbci.chChar, kbci.chScan, kbci.fsState );
    #endif
    scan = kbci.chScan;
    chr  = kbci.chChar;
  #else
    ScrDrvPoll();
    if( kbdBufCnt ) {
	chr   = kbdBuf[0].chr;
	scan  = kbdBuf[0].scan;
	state = kbdBuf[0].state;
	for(i=1; i < kbdBufCnt; i++ )
	    kbdBuf[i-1] = kbdBuf[i];
	kbdBufCnt--;
    }
    else if( msBtn = MouseGet( &msX, &msY ) ) {
	if( lastMsBtn == msBtn )
	    keycode = 0;
	else {
	    msEvent = 1;
	    lastMsBtn = msBtn;
	  #if EMX && !OS20
	    state = 0; /* ?? */
	  #elif __WATCOMC__
	    state = _bios_keybrd(_KEYBRD_SHIFTSTATUS);
	  #else
	    state = bioskey(0x02);
	  #endif
	    if( msBtn & MOUSE_LEFT )
		keycode = K_BTN_LEFT;
	    else if( msBtn & MOUSE_RIGHT )
		keycode = K_BTN_RIGHT;
	    else if( msBtn & MOUSE_LEFTDBL )
		keycode = K_BTN_LEFTDBL;
	    else /* MOUSE_RIGHTDBL */
		keycode = K_BTN_RIGHTDBL;
	}
    }
    else {
	lastMsBtn = 0;
	goto endLabel;
    }

    #if DBG_SCANCODES
      printf("K=**** vkey=**** chr=%4x scan=%4x state=%016b\n",
		       chr , scan, state );
    #endif
  #endif

    rpt  = 1;
    if( msEvent ) {
	vkey = keycode;
	flags |= K_FLAG_VIRT | K_FLAG_MOUSE;
    }
    else {
	if( chr && chr != 0xe0 )
	    keycode = chr & 0xff;
	else
	    keycode = 256 + (scan&0xff);
	if( vkey = VkeyCodeMap( chr == 224 ? 224 : keycode, scan ) )
	    flags |= K_FLAG_VIRT;
	if( chr  )
	    flags |= K_FLAG_CHAR;
	if( scan )
	    flags |= K_FLAG_SCAN;
    }

  #if OS20
    flags |= kstate;
  #elif OS2
    #if HAVE_OLD_OS2_HEADERS || OS20
	if( kbci.fsState & (RIGHTSHIFT| LEFTSHIFT) )
	    flags |= K_FLAG_SHIFT;
	if( kbci.fsState & CONTROL )
	    flags |= K_FLAG_CTRL;
	if( kbci.fsState & ALT )
	    flags |= K_FLAG_ALT;
    #else
	if( kbci.fsState & (KBDSTF_RIGHTSHIFT| KBDSTF_LEFTSHIFT) )
	    flags |= K_FLAG_SHIFT;
	if( kbci.fsState & KBDSTF_CONTROL )
	    flags |= K_FLAG_CTRL;
	if( kbci.fsState & KBDSTF_ALT )
	    flags |= K_FLAG_ALT;
    #endif
  #else
    if( state & ( 0x01| 0x02) )
	flags |= K_FLAG_SHIFT;
    if( state & 0x04 )
	flags |= K_FLAG_CTRL;
    if( state & 0x08 )
	flags |= K_FLAG_ALT;
  #endif
    #if DBG_SCANCODES
      printf("K=%4x vkey=%4x chr=%4x scan=%4x flags=%016b\n",
		keycode, vkey, chr, scan, flags );
    #endif

  endLabel:
    *rVkey = vkey;
    *rVal  = chr;
    *rScan = scan;
    *rRpt  = rpt;
    *rFlags =flags;
    return keycode;
}


static ushort VkeyCodeMap( int keycode , ushort scan )
{
    switch( keycode ) {
      case K_A0: keycode = '0'; break;
      case K_A1: keycode = '1'; break;
      case K_A2: keycode = '2'; break;
      case K_A3: keycode = '3'; break;
      case K_A4: keycode = '4'; break;
      case K_A5: keycode = '5'; break;
      case K_A6: keycode = '6'; break;
      case K_A7: keycode = '7'; break;
      case K_A8: keycode = '8'; break;
      case K_A9: keycode = '9'; break;
      case K_F1: case K_F2: case K_F3: case K_F4: case K_F5:
      case K_F6: case K_F7: case K_F8: case K_F9:
      case K_F10: keycode  = K_F1 + keycode - K_F1 ; break;
      case K_SF1: case K_SF2: case K_SF3: case K_SF4: case K_SF5:
      case K_SF6: case K_SF7: case K_SF8: case K_SF9:
      case K_SF10: keycode  = K_F1 + keycode - K_SF1 ; break;
      case K_CF1: case K_CF2: case K_CF3: case K_CF4: case K_CF5:
      case K_CF6: case K_CF7: case K_CF8: case K_CF9:
      case K_CF10: keycode  = K_F1 + keycode - K_CF1 ; break;
      case K_AF1: case K_AF2: case K_AF3: case K_AF4: case K_AF5:
      case K_AF6: case K_AF7: case K_AF8: case K_AF9:
      case K_AF10: keycode  = K_F1 + keycode - K_AF1 ; break;
      case K_F11: case K_SF11: case K_CF11: case K_AF11: keycode= K_F11; break;
      case K_F12: case K_SF12: case K_CF12: case K_AF12: keycode= K_F12; break;

      case K_HOME:
      case K_CHOME: keycode = K_PAD_7; break;
      case K_AHOME: keycode = K_HOME ; break;

      case K_UP:
      case K_CUP:   keycode = K_PAD_8; break;
      case K_AUP:   keycode = K_UP   ; break;

      case K_PGUP:
      case K_CPGUP:  keycode = K_PAD_9; break;
      case K_APGUP:  keycode = K_PGUP ; break;

      case K_LEFT:
      case K_CLEFT:  keycode = K_PAD_4; break;
      case K_ALEFT:  keycode = K_LEFT ; break;

      case K_PAD_5:
      case K_PAD_C5: keycode = K_PAD_5 ; break;

      case K_RIGHT:
      case K_CRIGHT: keycode = K_PAD_6; break;
      case K_ARIGHT: keycode = K_RIGHT; break;

      case K_END:
      case K_CEND:  keycode = K_PAD_1; break;
      case K_AEND:  keycode = K_END;   break;

      case K_DOWN:
      case K_CDOWN: keycode = K_PAD_2; break;
      case K_ADOWN: keycode = K_DOWN ; break;

      case K_PGDN:
      case K_CPGDN: keycode = K_PAD_3; break;
      case K_APGDN: keycode = K_PGDN ; break;

      case K_INS:
      case K_CINS:  keycode = K_PAD_INS; break;
      case K_AINS:  keycode = K_INS  ; break;

      case K_DEL:
      case K_CDEL:  keycode = K_PAD_DEL; break;
      case K_ADEL:  keycode = K_DEL  ; break;


      case K_TAB:      keycode = scan == 15  ? K_VK_TAB : 0; break;
      case (256+148): /* c-TAB */
      case (256+165): /* a-Tab */
      case K_BACKTAB:  keycode = K_VK_TAB   ; break;

      case K_RUBOUT:   keycode = scan == 14  ? K_VK_RUBOUT : 0; break;
      case K_CRUBOUT:  keycode = scan == 14  ? K_VK_RUBOUT : 0; break;
      case K_ARUBOUT:  keycode = K_VK_RUBOUT; break;

      case K_ESCAPE    : keycode = K_VK_ESCAPE; break;
      case ' '         :
      case K_BREAK     : break;

      case 13 :
      case 10 :
	keycode = scan == 224 ? K_PAD_RETURN :
		  scan == 28  ? K_VK_RETURN  : 0;
	break;

      case (256+28):  /* Alt-Return */
	keycode = scan == 224 ? K_PAD_RETURN : K_VK_RETURN ;
	break;
      case K_PAD_ARETURN: keycode = K_PAD_RETURN; break;

      case '-':  keycode = scan == 74 ?  K_PAD_MINUS:0; break;
      case K_PAD_CMINUS:
      case K_PAD_AMINUS: keycode = K_PAD_MINUS; break;

      case '+':          keycode = scan == 78 ?  K_PAD_PLUS:0; break;
      case K_PAD_CPLUS:  case K_PAD_APLUS: keycode = K_PAD_PLUS; break;

      case '/':          keycode = scan == 224?  K_PAD_DIV :0; break;
      case K_PAD_CDIV :  case K_PAD_ADIV : keycode = K_PAD_DIV ; break;

      case '*':          keycode = scan == 55 ?  K_PAD_MUL :0; break;
      case K_PAD_CMUL :  case K_PAD_AMUL : keycode = K_PAD_MUL ; break;

      case K_AA:
      case K_AB:
      case K_AC:
      case K_AD:
      case K_AE:
      case K_AF:
      case K_AG:
      case K_AH:
      case K_AI:
      case K_AJ:
      case K_AK:
      case K_AL:
      case K_AM:
      case K_AN:
      case K_AO:
      case K_AP:
      case K_AQ:
      case K_AR:
      case K_AS:
      case K_AT:
      case K_AU:
      case K_AV:
      case K_AW:
      case K_AX:
      case K_AY:
      case K_AZ: break;

      case '.' : /* english driver */
      case ',' : /* german driver */
		 keycode = scan == 83  ?  K_PAD_DEL :0; break;
      case '0' : keycode = scan == 82  ?  K_PAD_INS :0; break;
      case '1' : keycode = scan == 79  ?  K_PAD_1   :0; break;
      case '2' : keycode = scan == 80  ?  K_PAD_2   :0; break;
      case '3' : keycode = scan == 81  ?  K_PAD_3   :0; break;
      case '4' : keycode = scan == 75  ?  K_PAD_4   :0; break;
      case '5' : keycode = scan == 76  ?  K_PAD_5   :0; break;
      case '6' : keycode = scan == 77  ?  K_PAD_6   :0; break;
      case '7' : keycode = scan == 71  ?  K_PAD_7   :0; break;
      case '8' : keycode = scan == 72  ?  K_PAD_8   :0; break;
      case '9' : keycode = scan == 73  ?  K_PAD_9   :0; break;


      case 224:
	switch( scan ) {
	  case	0: keycode = 0;      ; break;
	  case 146:
	  case 82: keycode = K_INS   ; break;
	  case 147:
	  case 83: keycode = K_DEL   ; break;
	  case 117:
	  case 79: keycode = K_END   ; break;
	  case 145:
	  case 80: keycode = K_DOWN  ; break;
	  case 118:
	  case 81: keycode = K_PGDN  ; break;
	  case 115:
	  case 75: keycode = K_LEFT  ; break;
	  case 116:
	  case 77: keycode = K_RIGHT ; break;
	  case 119:
	  case 71: keycode = K_HOME  ; break;
	  case 141:
	  case 72: keycode = K_UP    ; break;
	  case 132:
	  case 73: keycode = K_PGUP  ; break;
	}
	break;

      case 0:


      default: keycode = 0; break;
    }
    return keycode;
}





/****************
 * Poll Fuktion sollte zumindest bei DOS hauefig aufgerufen werden.
 * Hintergrund:
 * Leider hat die DOS Key Queue keine Infos ueber den
 * akt. shiftstate, allso muessen wir versuchen diese
 * Infos selber zu speichern
 */

void ScrDrvPoll(void)
{
#if !OS2
    int state, c;
  #if DOS386 && __ZTC__
    static short _far *fp;
  #elif EMX && !OS20
    static short *fp;
  #endif

  #if 0
    while( bioskey(0x11) ) { /* zortech funktioniert mit x11 nicht */
  #elif DOS386 && __ZTC__    /* also hackermaessig */
    if( !fp )
	fp = _x386_mk_protected_ptr( 0x0041a );

    while( fp[0] != fp[1] ) {
  #elif EMX && !OS20
    if( !fp )
	fp = HW_MakePtrToMemory( 0x0041a , 2 * sizeof *fp );

    while( fp[0] != fp[1] ) {
  #elif DOS386 && __WATCOMC__
    while( *(short*)0x41a != *(short*)0x41c ) {
  #else
    while( *(short*)MK_FP(0x40,0x1a) != *(short*)MK_FP(0x40,0x1c) ) {
  #endif
      #if EMX && !OS20
	state = 0;  /* ?? */
	c = _read_kbd(0, 1, 0 );
	if( c != -1 ) {
	    if( !c )
		c = (_read_kbd(0, 1, 0 ) << 8) & 0xff00;
	    else
		c &= 0xff;
	 }
	 else
	    c = 0;
      #elif __WATCOMC__
	state = _bios_keybrd(_KEYBRD_SHIFTSTATUS);
	c = _bios_keybrd(_NKEYBRD_READ);
      #else
	state = bioskey(0x02);
	c = bioskey(0x10);
      #endif
	if( kbdBufCnt+1 < MAX_DOSKEYS ) {
	    kbdBuf[kbdBufCnt].chr  = c & 0xff;
	    kbdBuf[kbdBufCnt].scan = (c >> 8) & 0xff;
	    kbdBuf[kbdBufCnt].state= state;
	    kbdBufCnt++;
	}
	else /* buffer overflow */
	  #if __ZTC__
	    sound_beep(1172); /* 880 Hz */
	  #else
	    putc('\a', stderr);
	  #endif
    }
#endif
}


#endif /*!UNIX*/
/**** end of file ***/
