/* [sema.c wk 27.7.90] Semaphore function
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <wk/tailor.h>
RCSID("$Id: sema.c,v 1.6 1995/03/08 16:57:29 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <direct.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/locking.h>
#include <share.h>
#include <io.h>

#ifdef OS2
    #define INCL_DOS
    #define INCL_NOPM
    #include <os2.h>
#else
    #include <dos.h>
#endif

#include <wk/sema.h>
#include <wk/file.h>

/****** constants *********/
#define MAX_HANDLES	10
#define FILE_NAME "WK_SEM.TBL"
#define WAIT_INTERVAL	50L /* ms */

/******** typedefs ********/
typedef struct {
	int  useFlag;	    /*0: unused*/
	int  recNr ;	    /* recordnumber of the semaphore*/
	int  used;	    /* number of waits for that semaphore*/
	char name[SEMA_MAXLEN+1] ;
    } semaCtrl_t;

typedef struct {
	ushort openCount;
	short  value;
	ushort waiting;     /* number of stations waiting*/
	char   name[SEMA_MAXLEN+1];
    } semaRecord_t ;

/******* globals **********/
static int initOkay = 0 ;
static semaCtrl_t *semaTable ;
static int fileHandle ;
static char semaFileName[F_MAX_PATH] ; /*full path name of File*/

#ifdef OS2
static FILELOCK syncLockAddr = { 0x50000001, 1L } ;
#endif

/****** prototypes ********/
static void DoInit(void);
static void CleanUp(void);
static int Lock(int);
static int Unlock(int);
static int ReadRecord( int, semaRecord_t *, int);
static int WriteRecord( int, semaRecord_t *, int);
static int AppendRecord( int, semaRecord_t *);

/***** def. Function ******/
#define EnterCritical() 	  /* z.Z. Dummies*/
#define LeaveCritical()

/******* Functions ********/


#ifdef DOCUMENTATION
@Summary Sema...	Semaphore Functions
 #include <wk/sema.h>

 int SemaOpen( const char *name, short value, int *handle );
 int SemaClose( int handle ) ;
 int SemaCount( int handle, ushort *openCount );
 int SemaValue( int handle, short *semaValue );
 int SemaSignal( int handle );
 int SemaWait( int handle, long timeOutValue );
 const char *SemaErrorText(int errorCode );

@Description
 Semaphore Funktionen; local (z.Z. noch nicht untersttzt) und
 Netzwerksweit.
  Der Name kann eine Bel. Lnge bis zu SEMA_MAXLEN haben und aus
 beliebigen ASCII-Zeichen bestehen; soll es sich um ein netzwerks-
 weites Semaphore handlen, so mssen die ersten beiden Zeichen ein
 '/' sein.
  SemaCount liefert den aktuellen OpenCount des Semaphores zurck.
  SemaValue liefert den aktuellen Wert des Semphores zurck.
  SemaSignal inkrementiert das Semphore.
  SemaWait dekrementiert das Semphore und wartet falls der Wert dann
 negativ ist. ist die in Millisekunden angegebene Zeit abgelaufen, so
 kehrt die Funktion  mit dem Fehler ESEMA_TIMEOUT zurck; wobei
 das Semapore zuvor wieder inkrementiert wird.
@Return Value
 0 wenn kein Fehler, sonst Fehlercode ESEMA_xxxxxx
@Notes
 Netzwerksweite Semaphore werden implementiert durch eine Datei
 die alle Semaphore enthlt. Der Name dieser Datei ist WK_SEM.TBL
 und wird im aktuellen Directory erzeugt oder falls die Environment
 Variable WK_SEMA gesetzt ist, in dem dort angegebenen Directory.
 Der Dateiname wird in jedem Fall in einen vollstndigen Pfadnamen
 umgewandelt.
 Ist in dieser Datei ein Semphore offen, so ist auch die Datei
 offen; d.h. die Datei kann nur gelscht werden, wenn kein Semaphore
 benutzt wird. Die Datei darf sich keinesfalls auf einem lokalen
 Laufwerk befinden !
#endif /*DOCUMENTATION*/




int SemaOpen( const char *name, short value, int *handle )
{
    int err , records , i , found ;
    semaRecord_t rec ;
    long hl ;

    *handle = -1 ;
    err = 0 ;
    if( !initOkay )
	DoInit() ;

    /** validate name **/
    if( strlen(name) > SEMA_MAXLEN || strlen(name) < 2 )
	return ESEMA_INVALID_NAME ;
    if( !(*name == '/' && name[1] == '/' ) )
        bug("local semaphores not implemented");

    /** retrieve a new handle **/
    EnterCritical() ;
    for(i=0; i < MAX_HANDLES; i++ )
	if( !semaTable[i].useFlag ) {
	    semaTable[i].useFlag++ ;
	    *handle = i ;
	    break ;
	}
    LeaveCritical() ;
    if( *handle == -1 ) {
	err = ESEMA_OUT_OF_HANDLES;
	goto retlab ;
    }
    strcpy( semaTable[*handle].name, name ) ;

    /** open semaphore File **/
    if( fileHandle == -1 )
	if( (fileHandle = sopen( semaFileName, O_BINARY|O_RDWR|O_CREAT,
				       SH_DENYNO,S_IWRITE|S_IREAD )) == -1 ) {
	    error(1000,GetStr(1), semaFileName) ;
	    err = ESEMA_GENERAL_ERROR ;
	    goto retlab ;
	}
    if( (hl=filelength(fileHandle)) == -1 ) {
	err = ESEMA_GENERAL_ERROR ;
	goto retlab ;
    }

    /** search for a record with the given semaphore **/
    records = (int)( hl / sizeof(semaRecord_t) );
    for(found=i=0; i < records && !found && !err ; i++ )
	if( ReadRecord( fileHandle, &rec, i ) )
	    break ;
	else if( !strcmp( name, rec.name ) ) {
	    if( !Lock(fileHandle) ) {
		if( ReadRecord( fileHandle, &rec, i ) )
		    err = ESEMA_GENERAL_ERROR ;
		else if( !rec.openCount ) {
		    /** semaphore is not open **/
		    if( value < 1 )
			err = ESEMA_INVALID_VALUE ;
		    else {
			/** recreate semaphore **/
			semaTable[*handle].recNr = i ;
			semaTable[*handle].used = 0 ;
			rec.openCount = 1 ;
			rec.value = value ;
			rec.waiting = 0 ;
			if( WriteRecord( fileHandle, &rec, i ) )
			    err = ESEMA_GENERAL_ERROR ;
			else
			    found++ ;
		    }
		}
		else {
		    /** semaphore is open **/
		    semaTable[*handle].recNr = i ;
		    semaTable[*handle].used  = i ;
		    rec.openCount++ ;
		    if( WriteRecord( fileHandle, &rec, i ) )
			err = ESEMA_GENERAL_ERROR ;
		    else
			found++ ;
		}
		Unlock(fileHandle);
	    }
	}

    if( !err && !found ) {
	/** create a new semaphore at the end of the File **/
	if( value < 1 )
	    err = ESEMA_INVALID_VALUE ;
	else if( Lock(fileHandle) )
	    err = ESEMA_TIMEOUT ;
	else {
	    if( (hl=filelength(fileHandle)) == -1 )
		err = ESEMA_GENERAL_ERROR ;
	    else {
		records = (int)( hl / sizeof(semaRecord_t) );
		rec.openCount = 1 ;
		rec.value = value ;
		rec.waiting = 0 ;
		strcpy( rec.name, name ) ;
		if( AppendRecord( fileHandle, &rec ) )
		    err = ESEMA_GENERAL_ERROR ;
		else {
		    semaTable[*handle].recNr = records ;
		    semaTable[*handle].used  = 0;
		}
	    }
	    Unlock(fileHandle);
	}
    }


  retlab:
    if( err && *handle != -1 ) {
	EnterCritical() ;
	semaTable[*handle].useFlag = 0 ;
	LeaveCritical() ;
    }

    return err ;
}



int SemaClose( int handle )
{
    int err ;
    semaCtrl_t *ctrl ;
    semaRecord_t rec ;

    err = 0;
    if( handle < 0 && handle >= MAX_HANDLES )
	err = ESEMA_INVALID_HANDLE ;
    else if( !semaTable[handle].useFlag )
	err = ESEMA_INVALID_HANDLE ;
    else {
	ctrl = semaTable+handle ;
	xassert( fileHandle != -1 ) ;
	if( !Lock(fileHandle) ) {
	    if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
		err = ESEMA_GENERAL_ERROR ;
	    else {
		 if( rec.openCount )
		     rec.openCount-- ;
		 if( WriteRecord( fileHandle, &rec, ctrl->recNr ) )
		     err = ESEMA_GENERAL_ERROR ;
	    }
	    Unlock(fileHandle) ;
	}
	else
	    err = ESEMA_TIMEOUT ;

    }

    if( !err ) {
	EnterCritical() ;
	ctrl->useFlag = 0 ;
	LeaveCritical() ;
    }

    return err ;
}



int SemaCount( int handle, ushort *count )
{
    int  err ;
    semaCtrl_t *ctrl ;
    semaRecord_t rec ;

    err = 0;
    *count = 0 ;
    if( handle < 0 && handle >= MAX_HANDLES )
	err = ESEMA_INVALID_HANDLE ;
    else if( !semaTable[handle].useFlag )
	err = ESEMA_INVALID_HANDLE ;
    else {
	ctrl = semaTable+handle ;
	xassert( fileHandle != -1 ) ;
	if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
	    err = ESEMA_GENERAL_ERROR ;
	else
	    *count = rec.openCount ;
    }

    return err ;
}


int SemaValue( int handle, short *value )
{
    int err ;
    semaCtrl_t *ctrl ;
    semaRecord_t rec ;

    err = 0;
    if( handle < 0 && handle >= MAX_HANDLES )
	err = ESEMA_INVALID_HANDLE ;
    else if( !semaTable[handle].useFlag )
	err = ESEMA_INVALID_HANDLE ;
    else {
	ctrl = semaTable+handle ;
	xassert( fileHandle != -1 ) ;
	if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
	    err = ESEMA_GENERAL_ERROR ;
	else
	    *value = rec.value ;
    }

    return err ;
}


int SemaSignal( int handle )
{
    int err ;
    semaCtrl_t *ctrl ;
    semaRecord_t rec ;

    err = 0;
    if( handle < 0 && handle >= MAX_HANDLES )
	err = ESEMA_INVALID_HANDLE ;
    else if( !semaTable[handle].useFlag )
	err = ESEMA_INVALID_HANDLE ;
    else {
	ctrl = semaTable+handle ;
	xassert( fileHandle != -1 ) ;
	if( !Lock(fileHandle) ) {
	    if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
		err = ESEMA_GENERAL_ERROR ;
	    else {
		if( rec.value == 0x7fff )
		    err = ESEMA_INVALID_VALUE ;
		else {
		    rec.value++ ;
		    if( WriteRecord( fileHandle, &rec, ctrl->recNr ) )
			err = ESEMA_GENERAL_ERROR ;
		}
	    }
	    Unlock(fileHandle) ;
	}
	else
	    err = ESEMA_TIMEOUT ;
    }

    if( !err )
	ctrl->used-- ;
    return err ;
}


int SemaWait( int handle, long timeout )
{
    int okay , err ;
    semaCtrl_t *ctrl ;
    semaRecord_t rec ;

    err = 0;
    if( handle < 0 && handle >= MAX_HANDLES )
	err = ESEMA_INVALID_HANDLE ;
    else if( !semaTable[handle].useFlag )
	err = ESEMA_INVALID_HANDLE ;
    else {
	ctrl = semaTable+handle ;
	xassert( fileHandle != -1 ) ;
	if( !Lock(fileHandle) ) {
	    if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
		err = ESEMA_GENERAL_ERROR ;
	    else {
		if( rec.value == 0x8000 )  /* (negativ)*/
		    err = ESEMA_INVALID_VALUE ;
		else {
		    rec.value-- ;
		    rec.waiting++ ;
		    if( WriteRecord( fileHandle, &rec, ctrl->recNr ) )
			err = ESEMA_GENERAL_ERROR ;
		}
	    }
	    Unlock(fileHandle) ;
	}
	else
	    err = ESEMA_TIMEOUT ;
    }

    if( !err ) {
	/* wait for semaphore to get positiv */
	okay = 0 ;
	while( !err && !okay && timeout >= 0L )
	    if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
		err = ESEMA_GENERAL_ERROR ;
	    else if( rec.value < 0 ) {
		Sleep(WAIT_INTERVAL);
		timeout -= WAIT_INTERVAL ;
	    }
	    else {
		if( !Lock(fileHandle) ) {
		    if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
			err = ESEMA_GENERAL_ERROR ;
		    else if( rec.value >= 0 && rec.waiting == 1 ) {
			rec.waiting = 0 ;
			if( WriteRecord( fileHandle, &rec, ctrl->recNr ) )
			    err = ESEMA_GENERAL_ERROR ;
			else
			    okay++ ;
		    }
		    Unlock(fileHandle);
		}
		else
		    err = ESEMA_TIMEOUT ;
	    }

	if( !err && !okay ) {	 /* did a timeout occur ?*/
	    if( !Lock(fileHandle) ) {
		if( ReadRecord( fileHandle, &rec, ctrl->recNr ) )
		    err = ESEMA_GENERAL_ERROR ;
		else if( rec.waiting > 0 ) {
		    rec.waiting-- ;
		    rec.value++ ;
		    WriteRecord( fileHandle, &rec, ctrl->recNr ) ;
		}
		Unlock(fileHandle);
	    }
	    err = ESEMA_TIMEOUT ;
	}
    }

    if( !err )
	ctrl->used++ ;
    return err ;
}




const char *SemaErrorText(int err)
{
    const char *p ;
    static char buffer[30] ;

    switch( err ) {
      case ESEMA_GENERAL_ERROR : p = "General error" ; break;
      case ESEMA_INVALID_NAME  : p = "Invalid name" ; break;
      case ESEMA_INVALID_HANDLE: p = "Invalid handle" ; break;
      case ESEMA_INVALID_VALUE : p = "Invalid value" ; break;
      case ESEMA_TIMEOUT       : p = "Timeout" ; break;
      case ESEMA_OUT_OF_HANDLES: p = "Out of handles" ; break;
      default: sprintf(buffer, "error %d", err ) ; p = buffer ; break ;
    }
    return p ;
}



/***********************************
 ****** Helper Functions ***********
 ***********************************/

static void DoInit()
{
    char *p ;

    semaTable = xcalloc( MAX_HANDLES, sizeof *semaTable ) ;

    if( p = getenv("WK_SEMA") ) {
	strcpy( semaFileName, p );
        strcat( semaFileName, "/" ) ;
    }
    else
        *semaFileName = '\0' ;
    strcat( semaFileName, FILE_NAME ) ;
    FilenameMkAbs( NULL, semaFileName );
    fileHandle = -1 ;	/* not open*/
    AddCleanUp( CleanUp, NULL );

    initOkay = 1 ;
}


static void CleanUp(void *dummy)
{
    int i ;
    semaCtrl_t *ctrl ;

    for(i=0; i < MAX_HANDLES ; i++ ) {
	ctrl = semaTable + i ;
	if( ctrl->useFlag ) {
	    while( ctrl->used > 0 )
		if( SemaSignal( i ) )
		    break ;
	    SemaClose(i) ;
	}
    }
}



/*
 *  Lockes the File temporary
 *  Lock sync byte to indicate file is at the moment in use
 *  returns: 0: okay
 */

static int Lock( int handle )
{
    int cnt , err ;
  #ifndef OS2
    union REGS	 iRegs, oRegs;
  #endif

  #ifdef OS2
    for(cnt = 0; cnt < 20 ; cnt++ )
	if( err = DosFileLocks((ushort)handle,NULL, &syncLockAddr ) ) {
	    /** perform randow delay **/
	    Sleep( (long)rand() % 100L + 50L );
	}
	else
	    break ;
  #else
    for(cnt = 0; cnt < 20 ; cnt++ ) {
	iRegs.x.ax = 0x5c00;  /* lock region*/
	iRegs.x.bx = handle ;
	iRegs.x.cx = 0x5000 ;
	iRegs.x.dx = 0x0001 ;
	iRegs.x.si = 0x0000 ;
	iRegs.x.di = 0x0001 ;
	intdos( &iRegs, &oRegs ) ;
	if( oRegs.x.cflag ) {
	    err = 1 ;
	    /** perform randow delay **/
	    Sleep( (long)rand() % 100L + 50L );
	}
	else {
	    err = 0 ;
	    break ;
	}
    }
  #endif

    return err ;
}


/*
 *  UnLockes the File temporary
 *  UnLock sync Byte
 *  returns: 0: okay
 */

static int Unlock( int handle)
{
  #ifdef OS2
    return DosFileLocks((ushort)handle, &syncLockAddr, NULL ) ;
  #else
    union REGS	 iRegs, oRegs;

    iRegs.x.ax = 0x5c01;  /* unlock region*/
    iRegs.x.bx = handle ;
    iRegs.x.cx = 0x5000 ;
    iRegs.x.dx = 0x0001 ;
    iRegs.x.si = 0x0000 ;
    iRegs.x.di = 0x0001 ;
    intdos( &iRegs, &oRegs ) ;
    return oRegs.x.cflag ? 1 : 0 ;
  #endif
}



/*
 * Read Record nr
 * Returns: 0 on success
 */

static int ReadRecord( int handle, semaRecord_t *buffer , int nr )
{
    if( lseek( handle, (long)nr * sizeof *buffer, SEEK_SET ) == -1L )
	return -1 ;
    if( read( handle, (char*)buffer, sizeof *buffer ) != sizeof *buffer )
	return -1 ;
    return 0 ;
}


/*
 * Write Record nr
 * Returns: 0 on success
 */

static int WriteRecord( int handle, semaRecord_t *buffer , int nr )
{
    if( lseek( handle, (long)nr * sizeof *buffer, SEEK_SET ) == -1L )
	return -1 ;
    if( write( handle, (char*)buffer, sizeof *buffer ) != sizeof *buffer )
	return -1 ;
    return 0 ;
}

/*
 * Append a Record
 * Returns: 0 on success
 */

static int AppendRecord( int handle, semaRecord_t *buffer )
{
    if( lseek( handle, 0L, SEEK_END ) == -1L )
	return -1 ;
    if( write( handle, (char*)buffer, sizeof *buffer ) != sizeof *buffer )
	return -1 ;
    return 0 ;
}


#ifdef TEST
/***********************************
 ********* Testfunktionen **********
 ***********************************/

char *wk_usage[] =
{
    "sema name ",
    ""
} ;



void main( int argc, char **argv )
{
    char   *s ;
    int err ;
    ushort count ;
    short value ;
    int handle ;
    char sema[128+1] ;

    set_ctrl_break_handler() ;

    for( s=""; --argc && **++argv == '-' && *s != '-'; )
        for( s = *argv + 1 ; *s && *s != '-' ; s++ )
	    switch( *s ) {
              case '?': usage(1);
	      default:	usage(0) ;
	    }
    if( argc != 1 )
	usage(0) ;

    strcpy( sema, *argv );
    if( err=SemaOpen( sema , 1, &handle ) )
        error(2,"Error SemaOpen: %s", SemaErrorText(err) ) ;
    printf( "Semaphore: %s\n", sema  ) ;

    if( err=SemaCount( handle, &count ) )
        error(2,"Error SemaCount: %s", SemaErrorText(err) ) ;
    printf( "OpenCount: %u\n", count ) ;

    if( err=SemaValue( handle, &value ) )
        error(2,"Error SemaValue: %s", SemaErrorText(err) ) ;
    printf( "Value ...: %d\n", value ) ;


    printf( "Waiting for Semaphore " ) ;
    do {
        putchar('.') ;
	err=SemaWait( handle, 1000L ) ;
    } while( err == ESEMA_TIMEOUT ) ;
    if( err && err != ESEMA_TIMEOUT )
        error(2,"Error SemaWait: %s", SemaErrorText(err) ) ;
    puts( "\nOkay ! starting action" ) ;


    puts( "Hit any key ..." ) ;
    while( !inkey() )
	;

    puts( "action ended !" ) ;
    if( err=SemaSignal( handle ) )
        error(2,"Error SemaSignal: %s", SemaErrorText(err) ) ;
    puts( "Signaled okay !") ;

    if( err=SemaClose( handle ) )
        error(2,"Error SemaClose: %s", SemaErrorText(err) ) ;
    puts( "Close okay !") ;

    exit(0) ;
} /* end main() */


#endif /* TEST*/
/**** bottom of file ****/
