/* [lwpsema.c wk 15.01.94] Implements semaphores.
 *	Copyright (c) 1994 by Werner Koch (dd9jn)
 * This file is part of WkLib.
 *
 * WkLib is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * WkLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This simple approach implements semaphores by
 * locking all functions during critical operations
 * with an real OS-Semaphore. I selected this method
 * so that I can centralize the semaphore support or
 * practical disable it on systems where there is no
 * need for it.
 */

#include <wk/tailor.h>
RCSID("$Id: lwpsema.c,v 1.5 2001/08/10 08:21:29 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <wk/lwp.h>
#if defined(OS20)
    #define INCL_NOCOMMON 1
    #define INCL_DOSSEMAPHORES 1
    #include <os2.h>
#endif


#ifndef CLK_TCK
  #define CLK_TCK CLOCKS_PER_SEC
#endif

/************************************************
 ************  local variables	*****************
 ************************************************/

static t_sema unusedSemas;  /* linked list of unused Semaphores */


/************************************************
 ************  local functions	 ****************
 ************************************************/
#if OS20
static HMTX serializeSemaphore;

static void RequestSerialization(void)
{
    int rc;
    if( (rc = DosRequestMutexSem( serializeSemaphore , 1000 )) )
        Error(8,"sema: error requesting internal serialization: rc=%d", rc );
}

static void ReleaseSerialization(void)
{
    int rc;
    if( (rc = DosReleaseMutexSem( serializeSemaphore )) )
        Error(4,"sema: error releasing internal serialization: rc=%d", rc );
}

#else	/* all other OS */
#define RequestSerialization()	   do {    } while(0)
#define ReleaseSerialization()	   do {    } while(0)
#endif

/************************************************
 ************  public functions  ****************
 ************************************************/

t_sema CreateSema( int count )
{
    t_sema s;

    if( s = unusedSemas )
	unusedSemas = s->next;
    else
	s = xmalloc( sizeof *s );
    s->next = NULL;
    s->count = s->initial = count;
    return s;
}



void CloseSema( t_sema s )
{
    s->next = unusedSemas;
    unusedSemas = s;
}

void SignalSema( t_sema s )
{
    RequestSerialization();
    s->count++;
    ReleaseSerialization();
    LwpYield();
}

void DecrementSema( t_sema s )
{
    RequestSerialization();
    s->count--;
    ReleaseSerialization();
    LwpYield();
}



void ResetSema( t_sema s )
{
    RequestSerialization();
    s->count = s->initial;
    ReleaseSerialization();
}

void WaitSema( t_sema s )
{
    WaitSemaTimed( s, 0 );
}



/****************
 * Returns: 0 = Okay, !0 = timed out
 */

int WaitSemaTimed( t_sema s, ulong timeout )
{
    clock_t endTime=0;

    LwpYield();
    RequestSerialization();
    s->count--;
    if( s->count < 0 ) {
	if( timeout )
	    endTime = clock() + timeout / (1000 / CLK_TCK);
	while( s->count < 0 ) {
	    ReleaseSerialization();
	    RelinquishThread();
	    if( timeout && s->count < 0 && clock() > endTime )
		return -1;
	    RequestSerialization();
	}
    }
    ReleaseSerialization();
    return 0;
}

int TestSema( t_sema s )
{
    LwpYield();
    RequestSerialization();
    s->count--;
    if( s->count < 0 ) {
	s->count++;
	ReleaseSerialization();
	return -1;
    }
    ReleaseSerialization();
    return 0;
}

int QuerySema( t_sema s )
{
    int count;

    LwpYield();
    RequestSerialization();
    count = s->count;
    ReleaseSerialization();
    return count;
}


/*** end of file ***/
