/* [cmprread.c wk 31.10.90] Compress Read Funtions
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * ------------------------------------------------------------
 * Aufbau des Compress Infos im File
 *  byte 0 = kein Compress
 *	 1 = Standard nibble Compress
 *	     (andere Werte  sind fr Erweiterungen reserviert)
 *  data : Abhngig vom Kompressionsalgorithmus
 * ------------------------------------------------------------
 * Algorithms:
 * ------------------------------------------------------------
 *  Standard nibble compress: ( mode = 1, length of data = 14 )
 *	Es wird nibbleweise gespeichert
 *	Folgende Bedeutung kommte den einzelnen nibbles zu:
 *	Wert
 *	0 - D = Index in die Tabelle der 14 hufigsten Zeichen
 *	E     = Flag: das folgende Nibble gibt einen Wiederholungs-
 *		faktor an, mit dem das bernchste Zeichen wiederholt wird
 *		Ist das zu Wiederholende Zeichen in der Tabelle vorhanden
 *		so wird mit einem Bias von 4 abgespeichert, das sich dann
 *		erst bei einer 4-fachen Wiederholung der Wiederholungsfaktor
 *		lohnt;	handelt es sich um ein anders Zeichen, so wird mit
 *		einem Bias von 2 gespeichert, da sich bei diesen Zeichen
 *		bereits bei einer einfachen Wiederholung ein Vorteil ergibt.
 *	F     = Flag: die beiden folgenden Nibbles ergeben das darzustellende
 *		Zeichen
 *	Steuerfunktionen:
 *	EOFFlag:    (wg. eof mitten im nibble bentigt)
 *	Folgt auf nach Repeatflag und einem repeatcount von 0 wieder ein
 *	Repeatflag, so ist diese Kombination das EOF-Flag
 *	also: E 0 E = EOF-Flag
 * ------------------------------------------------------------
 ***************************************************************
 * History:
 * 30.03.93 wk	CompressOpen gibt jetzt Fehler zurueck bei outOfMemory
 */

#include <wk/tailor.h>
RCSID("$Id: cmprread.c,v 1.8 1995/03/08 16:53:46 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <wk/cmpr.h>

#if __MSC__
#pragma check_stack(off)
#endif

/****** constants *********/
/******** typedefs ********/
typedef struct {
	int	mode ;	    /* Compress Mode*/
	byte	tbl[14];    /* Tabelle mit den 14 hufigsten Zeichen*/
	FILE	*st;	    /* Handle of File*/

	int	cnt ;	    /* Flag: nibble gespeichert*/
	byte	nib ;	    /* gespeichertes Nibble ( wenn cnt true )*/

	int	rptcnt;     /* Anzahl der noch zu wiederholenden Zeichen*/
	int	rptchr;     /* zu wiederholendes Zeichen*/

	int	eof;	    /* eof/error flag*/

    } cW1_t ;	    /* working area for compress mode 1 */


/******* globals **********/
/****** prototypes ********/
static byte GetNibble( cW1_t *);
/***** def. Function ******/
/******* Functions ********/


#ifdef DOCUMENTATION
@Summary CompressOpen
 #include <wk/cmpr.h>
 void *CompressOpen( FILE *stream );
@Description
 CompressInformationen lesen und einen Handle ( void Pointer )
 auf Compressinfo zurckgeben.
@Return Value
 Returns NULL wenn Infos nicht gelesen werden konnten
@See Also
 CompressSeek CompressClose CompressRead
#endif /*DOCUMENTATION*/


void *CompressOpen( FILE *stream )
{
    cW1_t *ctrl ;
    int err, c ;

    err = 0 ;
    if( ctrl = malloc( sizeof *ctrl ) ) {
	ctrl->st = stream ; /* save Stream for later use */
	if( (c = getc(stream)) == EOF )
	    err++ ;
	else if( c && c != 1 ) {
            Fatal("Unsupported Compressmode %d", c ) ;
	    err++ ;
	}
	else if( ctrl->mode = (byte)c ) {
	    if( !fread( ctrl->tbl, sizeof ctrl->tbl, 1 , stream   ) )
		err++ ;
	}
    }
    else
	err++;

    if( err )
	FREE( ctrl ) ;
    else
	CompressSeek(ctrl, -1L) ; /* Init Buffers */

    return (void *)ctrl ;
}

#ifdef DOCUMENTATION
@Summary CompressClose
 #include <wk/cmpr.h>
 void CompressClose( void *p );
@Description
 Nach Beendigung des CompressLesens muss diese
 Funktion aufgerufen werdn, um interne resourcen wieder
 freizugeben; die Datei wird dadurch aber nicht geschlossen
 Ein Aufruf mit einem NULL-Pointer ist mglich und bewirk nichts.
@See Also
 CompressOpen
#endif /*DOCUMENTATION*/

void CompressClose( void *p )
{
    FREE(p) ;
}





#ifdef DOCUMENTATION
@Summary CompressSeek
 #include <wk/cmpr.h>

 long CompressSeek( void *ctrl, long off );
@Description
 Positionieren im File ( hnlich fseek )
 Wenn off == -1L dann wird nur der aktuelle Offset
		 zurckgegeben und keine neu Positionierung
		 vorgenommen.
 SeekEnd/SeekCur ist nicht anwendbar, da dann mit Sicherheit kein
 gltiger Offset getroffen wird.
@Return Value
 neuer (bzw alter)  Offset im File oder -1L bei einem Fehler
@See Also
 CompressOpen CompressRead
#endif /*DOCUMENTATION*/

long CompressSeek( void *ctrl, long off )
{
    ((cW1_t*)ctrl)->cnt = 0 ;
    ((cW1_t*)ctrl)->rptcnt = 0 ;
    ((cW1_t*)ctrl)->eof = 0 ;
    clearerr( ((cW1_t*)ctrl)->st ) ;

    if( off >= 0L )
	if( fseek( ((cW1_t*)ctrl)->st, off, SEEK_SET ) )
	    ((cW1_t*)ctrl)->eof++ ;

    return ((cW1_t*)ctrl)->eof ? -1L : ftell( ((cW1_t*)ctrl)->st ) ;
}




#ifdef DOCUMENTATION
@Summary CompressRead
 #include <wk/cmpr.h>

 int CompressRead( void *ctrl );
@Description
 Lesen eines Zeichens (Byte) , vorher muss
 CompressOpen() erfolgreich durchgefhrt worden sein
 Es wird ab der letzten FilePosition gelesen. Diese Fileposition
 kann wie mit CompressSeek() eingestellt werden, dabei ist aber
 darauf zu achten das auf eine gltige Position angesteuert wird.

 Nach einem Fehler ist CompressSeek immer ntig !
@Return Value
 Returns: character or EOF if error or eof
@See Also
 CompressOpen CompressSeek
#endif /*DOCUMENTATION*/


int CompressRead( void *ctrl )
{
    int c=0;  /* avoid compiler warning about uninitialized use */
    byte n ;

    if( !((cW1_t*)ctrl)->mode ) {
	if( (c = getc(((cW1_t*)ctrl)->st)) == EOF )
	    ((cW1_t*)ctrl)->eof++ ;
    }
    else { /* compress mode */
	if( ((cW1_t*)ctrl)->rptcnt ) {
	    c = ((cW1_t*)ctrl)->rptchr ;
	    ((cW1_t*)ctrl)->rptcnt-- ;
	}
	else {
	    n = GetNibble(((cW1_t*)ctrl)) ;
	    if( n < 14 )
		c = ((cW1_t*)ctrl)->tbl[ n ] ;
	    else if( n == 14 ) {   /* repeat flag*/
		((cW1_t*)ctrl)->rptcnt = GetNibble( ((cW1_t*)ctrl) ) ;
		n = GetNibble( ((cW1_t*)ctrl) ) ;
		if( n < 14 ) {
		    c = ((cW1_t*)ctrl)->tbl[ n ] ;
		    ((cW1_t*)ctrl)->rptcnt += 3 ; /* one will be delivered imm.*/
		}
		else if( n == 14 ) {
		    ((cW1_t*)ctrl)->eof++ ;
		    /* laut definition muss dann hier der repeatcount*/
		    /* 0 sein; eine berprfung erfolgt aber nicht.*/
		    /* Andere Werte als 0 sind fr Erweiterungen vorgesehen.*/
		}
		else {
		    n = GetNibble(((cW1_t*)ctrl)) ;
		    c = n << 4 | GetNibble(((cW1_t*)ctrl)) ;
		    ((cW1_t*)ctrl)->rptcnt += 1 ;   /* one will be delivered imm.*/
		}
		((cW1_t*)ctrl)->rptchr = c ;
	    }
	    else {  /* escape flag*/
		n = GetNibble(((cW1_t*)ctrl)) ;
		c = n << 4 | GetNibble(((cW1_t*)ctrl)) ;
	    }
	}
    }

    return ((cW1_t*)ctrl)->eof ? EOF : c & 0xff ;
}



/*
 * Vom Eingabe Stream einen Nibble lesen
 * Bei einem Fehler wird ctrl->eof gesetzt.
 */


static byte GetNibble( cW1_t *ctrl )
{
    byte n = 0;  /* avoid compiler warning about uninitialized use */
    int c;

    if( !ctrl->cnt ) {
	if( (c = getc(ctrl->st)) == EOF )
	    ctrl->eof++ ;
	else {
	    n = (byte)((c & 0xf0) >> 4) ;
	    ctrl->nib = (byte)(c & 0x0f) ;
	    ctrl->cnt++ ;
	}
    }
    else  {
	n = ctrl->nib ;
	ctrl->cnt-- ;
    }

    return n ;
}

/********* bottom of file ***************/
