/* [xinterface.c wk 02.03.95] X 11 Interface for W
 *	Copyright (c) 1995 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *--------------------------------------------------------------------
 *
 *
 * History:
 */


#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <wk/lib.h>
#include <wk/keys.h>
#include <wk/string.h>

#include "xinterface.h"
#include "jnxicon"

#ifdef DEBUG
#define DBG_PRINT   Info
#else
#define DBG_PRINT   DummyInfo
#endif

#define  MIN_SCREEN_ROWS 12
#define  MIN_SCREEN_COLS 80  /* I think this is minimum in other modules */

struct xinterface_globals glo = { 0 }; /* define structure */

typedef struct {
    const char *name;
    char **value;
} APP_PARAM;


static struct {
    char *font;
    int  bdWidth;   /* border width */
    char *display_name;
    char *geometry;
    char *geometry_cline;
    char *geometry_rsrc;
    char *toggleCaret;
} prm = { "fixed", 1 };

static APP_PARAM app_resources[] = {
  { "font",         &prm.font          },
  { "geometry",     &prm.geometry_rsrc },
  { "toggleCaret", &prm.toggleCaret },
  { NULL }
};


static APP_PARAM app_options[] = {
  { "-display",     &prm.display_name   },
  { "-d",           &prm.display_name   },
  { "-geometry",    &prm.geometry_cline },
  { "-g",           &prm.geometry_cline },
  { "-fn",          &prm.font           },
  { NULL }
};

/* Table to map PC colors to X colors (allocated on demand)
 *  0 = Black	   8 = Dark grey	 High nibble is background
 *  1 = Blue	   9 = Light blue	 Low  nibble is foreground
 *  2 = Green	   A = Light green
 *  3 = Cyan	   B = Light cyan
 *  4 = Red	   C = Light red
 *  5 = Magenta    D = Light magenta
 *  6 = Brown	   E = Yellow
 *  7 = White	   F = Bright white
 */
static struct {
    const char *name;
    ulong pixel;
} colortable[16] = {
  { "black"         },
  { "blue"          },
  { "dark green"    },
  { "cyan"          },
  { "red"           },
  { "magenta"       },
  { "brown"         },
  { "light gray"    },
  { "dim gray"      },
  { "light blue"    },
  { "green"         },
  { "light cyan"    },
  { "red2"          },
  { "magenta2"      },
  { "yellow"        },
  { "white"         }
};



static int saved_argc;
static char **saved_argv;
static int available;
static void (*exposureHandler)(void);
static void (*notifyHandler)(int);

static struct {
    int valid;
    unsigned state;
    unsigned keycode;
} lastkey;

static struct {
    int type;	/* 0 = none, 1 = replacing, 2 = inserting caret */
    int x,y;	/* current positions */
    GC	gc;
} caret;

static void Cleanup(void);
static int  WaitEvent(void);
static void ProcessButtonEvent( XButtonEvent *event, int pressed );


#ifndef DEBUG
static void
DummyInfo(const char*s,...)
{
}
#endif


static void
Cleanup()
{
    if( glo.display ) {
	XCloseDisplay(glo.display);
	glo.display = NULL;
    }
}


/****************
 * Parse the args and remove the used ones
 */

void
JnxParseArgs( int *c_argc, char ***c_argv )
{
    int argc = *c_argc;
    char **argv = *c_argv;
    const char *s;
    int i, j;

    for(i=1; (i+1) < argc; i++ ) {
	for(j=0; s = app_options[j].name; j++ ) {
	    if( !strcmp(argv[i],s) ) {
		*app_options[j].value = argv[++i];
		for(j=i+1; j < argc; j++ ) /* shift argv */
		    argv[j-2] = argv[j];
		argv[j-2] = NULL;
		argc -= 2;
		i -= 3;
		break;
	    }
	}
    }
    *c_argc = saved_argc = argc;
    *c_argv = saved_argv = argv;
}



void
JnxInitApp( int x, int y, unsigned width, unsigned height )
{
    const char *s;
    char *p;
    int i;
    Colormap default_cmap;
    XColor  color;
    XSizeHints xsh;
    XWMHints  xwmh;
    char default_geometry[50];
    int bitmask;
    XGCValues gcv;
    XSetWindowAttributes xswa;

    RegisterCleanup(Cleanup);

    /* open the display */
    if( !(glo.display = XOpenDisplay(prm.display_name)) )
	Error(3,"can't open display '%s'", XDisplayName(prm.display_name));
    available = 1;

    /* get resources */
    for(i=0; s = app_resources[i].name; i++ )
	if( p = XGetDefault(glo.display,wklibApplicationName,s) )
	    *app_resources[i].value = p;

    /* set colors etc. */
    if( !(glo.fontStruct = XLoadQueryFont(glo.display,prm.font)) )
	Error(3,"display %s cannot load font %s", DisplayString(glo.display),
								    prm.font );

    glo.fontWidth =  glo.fontStruct->max_bounds.width;
    glo.fontHeight=  glo.fontStruct->ascent + glo.fontStruct->descent;
  #if 0 /* how to handle this */
    /* add additional padding (1,1) */
    glo.fontWidth++;
    glo.fontHeight++;
  #endif

    default_cmap = DefaultColormap(glo.display,DefaultScreen(glo.display));
    for(i=0; i < DIM(colortable); i++ )  {
	if( !XParseColor( glo.display, default_cmap,
			  colortable[i].name, &color) ||
	    !XAllocColor(glo.display, default_cmap, &color) ) {
	    Fatal("failed to allocate color '%s'", colortable[i].name );
	    colortable[i].pixel =
		    i ? WhitePixel(glo.display,DefaultScreen(glo.display))
		      : BlackPixel(glo.display,DefaultScreen(glo.display));
	}
	else
	    colortable[i].pixel = color.pixel;
    }
    glo.bgPix = colortable[0].pixel;
    glo.fgPix = colortable[DIM(colortable)-1].pixel;

    /* fill XSizeHints */
    xsh.flags = PPosition | PSize | PMinSize;
    xsh.height	   = height;
    if( xsh.height < MIN_SCREEN_ROWS * glo.fontHeight )
	xsh.min_height = MIN_SCREEN_ROWS * glo.fontHeight;
    else
	xsh.min_height = xsh.height;
    if( !xsh.height )
	xsh.height = xsh.min_height;
    xsh.width	   = width;
    if( xsh.width < MIN_SCREEN_COLS * glo.fontWidth )
	xsh.min_width = MIN_SCREEN_COLS * glo.fontWidth;
    else
	xsh.min_width = xsh.width;
    if( !xsh.width )
	xsh.width = xsh.min_width;
    xsh.x = x;
    xsh.y = y;

    /* setup the geometry strings */
    sprintf(default_geometry,"%dx%d%+d%+d",
			      xsh.width, xsh.height, xsh.x, xsh.y);
    if( prm.geometry_cline )
	prm.geometry = prm.geometry_cline;
    else if( prm.geometry_rsrc )
	prm.geometry = prm.geometry_rsrc;
    else
	prm.geometry = default_geometry;

    /* parse geometry */
    bitmask = XGeometry(glo.display,DefaultScreen(glo.display),
			prm.geometry, default_geometry, prm.bdWidth,
			glo.fontWidth, glo.fontHeight,
			0,0, &xsh.x, &xsh.y, &xsh.width, &xsh.height);
		     /* !!!-- adder - but I don't know how to handle it */
		     /* when drawing with ImageString etc. */
    if( bitmask & ( XValue | YValue ) )
	xsh.flags |= USPosition;
    if( bitmask & (WidthValue | HeightValue) )
	xsh.flags |= USSize;
    glo.screenHeight = xsh.height;
    glo.screenWidth  = xsh.width;

    /* create the main window */
    glo.mainWin = XCreateSimpleWindow(glo.display,
				      DefaultRootWindow(glo.display),
				      xsh.x, xsh.y, xsh.width, xsh.height,
				      prm.bdWidth, glo.bgPix, glo.fgPix );
    XStoreName(glo.display, glo.mainWin, wklibApplicationName);
    XSetIconName(glo.display, glo.mainWin, wklibApplicationName);
    XSetNormalHints(glo.display, glo.mainWin, &xsh);


    /* give some more hints */
    xwmh.flags = InputHint | StateHint;
    xwmh.input = True;
    xwmh.initial_state = NormalState;
    if( xwmh.icon_pixmap = XCreateBitmapFromData(glo.display, glo.mainWin,
				jnxicon_bits, jnxicon_width, jnxicon_height ) )
	xwmh.flags |= IconPixmapHint;
    else
	Error(0,"error creating icon");

  #warning Must implemt hits for wmaker.
#if 0
    xwmh.window_group = ...


You must define the WM_CLASS (XSetClassHint()) and the
CLIENT_LEADER or XWMHints.window_group properties, which are
automatically set by most applications that use Xt (Motif, Athena ...), but if
you use plain Xlib you must set them by hand.

Also you must make a call to XSetCommand(dpy, leader, argv, argc);
#endif

    XSetWMHints(glo.display, glo.mainWin, &xwmh );

    /* set attributes */
    xswa.colormap = DefaultColormap(glo.display,DefaultScreen(glo.display));
    xswa.bit_gravity = NorthWestGravity;
    XChangeWindowAttributes(glo.display,glo.mainWin,
			    (CWColormap | CWBitGravity), &xswa );

    /* create graphics context */
    gcv.font = glo.fontStruct->fid;
    gcv.foreground = glo.fgPix;
    gcv.background = glo.bgPix;
    glo.gc = XCreateGC(glo.display, glo.mainWin,
		       (GCFont | GCForeground | GCBackground), &gcv );
    /* and one for the caret */
    caret.type = 0;
    gcv.function = GXxor;
    gcv.background = BlackPixel(glo.display,DefaultScreen(glo.display));
    gcv.foreground = WhitePixel(glo.display,DefaultScreen(glo.display));

    /* My old server was somewhat strange: XFillRectangele with mode GXxor
     * used the background color to fill. This kludge is for such servers
     */
    if( prm.toggleCaret && !stricmp(prm.toggleCaret,"true") ) {
	XGCValues tmp;
	tmp.foreground = gcv.foreground;
	gcv.foreground = gcv.background;
	gcv.background = tmp.foreground;
    }

    caret.gc = XCreateGC(glo.display, glo.mainWin,
			  (GCFunction | GCForeground | GCBackground), &gcv );

    /* select exposure events */
    XSelectInput(glo.display, glo.mainWin,
		  ExposureMask |
		  KeyPressMask |
		  StructureNotifyMask |
		  ButtonPressMask |
		  ButtonReleaseMask
		);
    /* map main window */
    XMapWindow(glo.display, glo.mainWin);

    /* get the attributes and save them for later use */
    if( !XGetWindowAttributes(glo.display, glo.mainWin, &glo.mainWinAttr) )
	Error(2,"error getting window attributes");
    DBG_PRINT("Screen: w=%u h=%u  font: w=%u h=%u",
				    glo.screenWidth, glo.screenHeight,
				    glo.fontWidth, glo.fontHeight );
}


/****************
 * Tell wether the X windows system is available and initialized
 * (used for applications to test, wether they should run on X)
 */

int
JnxAvailable()
{
    return available;
}

void
JnxRegisterExposureHandler( void (*f)(void) )
{
    exposureHandler = f;
}

void
JnxRegisterNotifyHandler( void (*f)(int) )
{
    notifyHandler = f;
}

/****************
 * Return the screensize for the current font
 * in number of characters
 */

void
JnxQueryScreenSize( unsigned *x, unsigned *y )
{
    *x = glo.screenWidth / glo.fontWidth;
    *y = glo.screenHeight / glo.fontHeight;
    if( *x < MIN_SCREEN_COLS )
	*x = MIN_SCREEN_COLS;
    if( *y < MIN_SCREEN_ROWS )
	*y = MIN_SCREEN_ROWS;
}

/****************
 * Add a new event mask to the window
 */

void
JnxActivateEvent( Window xid, long mask )
{
    XWindowAttributes xwa;
    XSetWindowAttributes xswa;

    if( XGetWindowAttributes(glo.display, xid, &xwa) ) {
	xswa.event_mask = xwa.your_event_mask | mask;
	XChangeWindowAttributes(glo.display, xid, CWEventMask, &xswa);
    }
}


/****************
 * Remove an event mask from the window
 */

void
JnxIgnoreEvent( Window xid, long mask )
{
    XWindowAttributes xwa;
    XSetWindowAttributes xswa;

    if( XGetWindowAttributes(glo.display, xid, &xwa) ) {
	xswa.event_mask = xwa.your_event_mask & ~mask;
	XChangeWindowAttributes(glo.display, xid, CWEventMask, &xswa);
    }
}

/****************
 * Wait for an event (with timeout).
 * Returns: 0  := have event
 *	    1  := no Event (timeout or EINTR)
 *	    -1 := application should terminate
 */

int
JnxWaitEvent()
{
    fd_set rfds, xfds;
    int rc;
    struct timeval tv;

    FD_ZERO(&rfds);
    FD_SET(ConnectionNumber(glo.display), &rfds);
    FD_ZERO(&xfds);
    FD_SET(ConnectionNumber(glo.display), &xfds);
    tv.tv_sec = 1;
    tv.tv_usec = 0;
    if( (rc = select(FD_SETSIZE, &rfds, NULL, &xfds, &tv)) == -1 ) {
	if( errno != EINTR )
	    Error(1002,"JnxWaitEvent: select failed");
	return 1; /* interrupt */
    }
    else if( !rc )
	return 1;
    else
	return WaitEvent();
}

static int
WaitEvent()
{
    XNextEvent(glo.display, &glo.event);
    if( glo.event.xany.window == glo.mainWin ) {
	switch( glo.event.type ) {
	  case Expose:
	    DBG_PRINT("Event=Expose: count %d\n", glo.event.xexpose.count);
	    if( !glo.event.xexpose.count ) {
		if( exposureHandler ) {
		    (*exposureHandler)();
		}
	    }
	    break;
	  case KeyPress:
	    if( glo.event.xkey.keycode == 54
		&& ((glo.event.xkey.state & (ShiftMask|ControlMask|Mod1Mask))
							     == ControlMask)){
		/* control - c */
		raise(SIGINT);
		break;
	    }
	    if( lastkey.valid )
		DBG_PRINT("Event=KeyPress: overrun");
	    lastkey.valid = 1;
	    lastkey.state = glo.event.xkey.state;
	    lastkey.keycode = glo.event.xkey.keycode;
	    break;

	  case ButtonRelease:
	    ProcessButtonEvent(&glo.event.xbutton,0);
	    break;
	  case ButtonPress:
	    ProcessButtonEvent(&glo.event.xbutton,1);
	    break;

	  case MotionNotify:
	    glo.pointer_x = glo.event.xmotion.x;
	    glo.pointer_y = glo.event.xmotion.y;
	    glo.pointer_state = glo.event.xmotion.state;
	    DBG_PRINT("Event=MotionNotify: x=%d y=%d state=%x",
			    glo.event.xmotion.x,
			    glo.event.xmotion.y,
			    glo.event.xmotion.state );
	    break;

	  case ConfigureNotify:
	    if( glo.screenWidth != glo.event.xconfigure.width ||
		glo.screenHeight != glo.event.xconfigure.height ) {
		DBG_PRINT("Event=ConfigureNotify: w=%d h=%d",
				glo.event.xconfigure.width,
				glo.event.xconfigure.height );
		glo.screenWidth = glo.event.xconfigure.width;
		glo.screenHeight= glo.event.xconfigure.height;
		/* correct the size */
		glo.screenWidth = ((glo.screenWidth +  glo.fontWidth - 1)
				    / glo.fontWidth) * glo.fontWidth;
		glo.screenHeight =((glo.screenHeight + glo.fontHeight -1)
				    / glo.fontHeight)* glo.fontHeight;
		if(    glo.screenWidth != glo.event.xconfigure.width
		    || glo.screenHeight!= glo.event.xconfigure.height ) {
		    DBG_PRINT("            corrected: w=%d h=%d",
			    glo.screenWidth, glo.screenHeight );
		    XResizeWindow(glo.display, glo.mainWin,
			      glo.screenWidth, glo.screenHeight );
		}
		if( notifyHandler )
		    (*notifyHandler)(1);
	    }
	    break;

	  default: /* none */;
	}
	fflush(stderr);
    }
    return 0;
}


/****************
 * handle the buttonPress (pressed=1) and ButtonRelease (pressed = 0) events.
 */
static void
ProcessButtonEvent(XButtonEvent *e,  int pressed )
{
    int idx;

  #if 0
    {
	const char *s;
	s = e->button == 1? "left" : e->button==2? "middle" : e->button==3 ?
			    "right" :"none";
	printf("%s button %s x=%d x=%d state=%x\n",
		  s, pressed? "pressed ": "released", e->x, e->y, e->state );
    }
  #endif
    if( (idx = e->button -1) > 2 || idx < 0 )
	return; /* forget this one */


    glo.pointer_x = e->x;
    glo.pointer_y = e->y;
    glo.pointer_state = e->state;
    if( pressed ) {
	if( !glo.any_button_down ) {
	    glo.any_button_down = 1;
	    JnxActivateEvent( glo.mainWin, ButtonMotionMask );
	}
	glo.btn[idx].pressed = 1;
	glo.btn[idx].reported= 0;
	glo.btn[idx].last.state = e->state;
	glo.btn[idx].last.x = e->x;
	glo.btn[idx].last.y = e->y;
	glo.btn[idx].last.t = e->time;
	if( notifyHandler )
	    (*notifyHandler)(11+idx*2);
    }
    else { /* released */
	glo.btn[idx].pressed = 0;
	if( !glo.btn[0].pressed && !glo.btn[1].pressed && !glo.btn[2].pressed){
	    /*JnxIgnoreEvent( glo.mainWin, ButtonMotionMask );*/
	    glo.any_button_down = 0;
	}
	if( notifyHandler )
	    (*notifyHandler)(10+idx*2);
    }
}



/****************
 * Returns the current mouse position and the current state of the buttons.
 * int the high
 */
unsigned
JnxReadMouse( int *x, int *y )
{
    unsigned state;

    if( !glo.fontWidth || !glo.fontHeight )
	return 0; /* oops */

    while( XPending( glo.display ) )
	WaitEvent();
    *x = glo.pointer_x / glo.fontWidth;
    *y = glo.pointer_y / glo.fontHeight;
    if( !glo.any_button_down )
	return 0;

    state = 0;
    if( glo.btn[0].pressed )
	state |= 1;
    if( glo.btn[1].pressed )
	state |= 2;
    if( glo.btn[2].pressed )
	state |= 4;
    return state;
}


/****************
 * Read the last keycode.
 * Returns: 0 := no keycode available
 */

int
JnxReadKeyCode( unsigned *keycode, unsigned *state )
{
    unsigned n;

    while( XPending( glo.display ) )
	WaitEvent();
    if( !lastkey.valid ) { /* have a look at the pointer */
	int idx=0;
	if( glo.btn[idx].pressed && !glo.btn[idx].reported )
	    n = K_BTN_LEFT;
	else if( glo.btn[++idx].pressed && !glo.btn[idx].reported )
	    n = K_BTN_MIDDLE;
	else if( glo.btn[++idx].pressed && !glo.btn[idx].reported )
	    n = K_BTN_RIGHT;
	else
	    return 0;
	glo.btn[idx].reported = 1;
	*keycode = n;
	n = 128; /* mark mouse click */
	if( glo.btn[idx].last.state & ShiftMask )
	    n |= (1<<0);
	if( glo.btn[idx].last.state & ControlMask )
	    n |= (1<<1);
	if( glo.btn[idx].last.state & Mod1Mask )
	    n |= (1<<2);
	if( glo.btn[idx].last.state & Mod1Mask )
	    n |= (1<<3); /* This is AltGr on my old XFree86 */
	if( glo.btn[idx].last.state & 0x2000 )
	    n |= (1<<3); /* This is AltGr on my XFree86 3.3 with the pc102 */
	*state = n;
	return 1;
    }
    lastkey.valid = 0;
    n = lastkey.keycode;
    if( n <= 8 )
	return 0;   /* throw away */
    else if( n <= 96 )
	n -= 8;  /* this is the offset to the PC scancodes */
    else switch( n ) {
      case  97: /*Home*/
      case  98: /*Up*/
      case  99: /*Prior*/
      case 100: /*Left*/
		n += 5; break;
	/* 101	  Begin*/
      case 102: /*Right*/
      case 103: /*End*/
      case 104: /*Down*/
      case 105: /*Next*/
      case 106: /*Insert*/
      case 107: /*Delete*/
		n += 4; break;
      case 108: /*KP_Enter*/
		n = 96; break;
      case 110: /*Pause*/
		n = 119; break;
      case 111: /*Print*/
		n = 99; break;
      case 112: /*KP_Divide*/
		n = 98; break;
	/* 113:   Mode_switch*/
	/* 114:   Break*/
      case 115: /* Super_L */
      case 116: /* Super_R */
      case 117: /* Hyper_R */
	break;

      default:
	return 0;   /* throw away */
    }
    *keycode = n;
    n = 0;
    if( lastkey.state & ShiftMask )
	n |= (1<<0);
    if( lastkey.state & ControlMask )
	n |= (1<<1);
    if( lastkey.state & Mod1Mask )
	n |= (1<<2);
    if( lastkey.state & Mod1Mask )
	n |= (1<<3); /* This is AltGr on my old XFree86 */
    if( lastkey.state & 0x2000 )
	n |= (1<<3); /* This is AltGr on my XFree86 3.3 with the pc102 */
    *state   = n;
    return 1;
}


void
JnxClearScreen()
{
}

void
JnxSetTitle( int mode, const char *s )
{
    if( !mode )
	XStoreName(glo.display, glo.mainWin, s );
    else if( mode == 1 )
	XSetIconName(glo.display, glo.mainWin, s );
}


/*
 * We use PC colors
 *  0 = Black	   8 = Dark grey	 High nibble is background
 *  1 = Blue	   9 = Light blue	 Low  nibble is foreground
 *  2 = Green	   A = Light green
 *  3 = Cyan	   B = Light cyan
 *  4 = Red	   C = Light red
 *  5 = Magenta    D = Light magenta
 *  6 = Brown	   E = Yellow
 *  7 = White	   F = Bright white
 */

void
JnxSetColor( unsigned colorAttr )
{
    ulong fg, bg;
    int flushed=0;

    bg = colortable[(colorAttr & 0xf0)>>4].pixel;
    if( bg != glo.bgPix ) {
	JnxWriteChar(0,0,-1); /* flush */
	flushed++;
	XSetBackground(glo.display, glo.gc, bg );
	glo.bgPix = bg;
    }
    fg = colortable[colorAttr & 0x0f].pixel;
    if( fg != glo.fgPix ) {
	if( !flushed )
	    JnxWriteChar(0,0,-1); /* flush */
	XSetForeground(glo.display, glo.gc, fg );
	glo.fgPix = fg;
    }
}

void
JnxSetDefaultColor()
{
   /* not used */
}


/****************
 * Set the Caret (the cursor used in text fields)
 * x and y is the position and caretType is one of:
 *  0 := do not show the caret
 *  1 := use the caret used for replacing text
 *  2 := use the caret used for inserting text.
 */

void
JnxSetCaret(int x, int y, int caretType)
{
    unsigned height;

    if( caret.type ) { /* remove old caret */
	if( caretType == caret.type && x == caret.x && y == caret.y ) {
	    return; /* already set at this position */
	}
	if( caret.type == 1 )
	    height = 0;
	else
	    height = glo.fontHeight / 3;
	height += glo.fontStruct->descent;
	XFillRectangle(glo.display, glo.mainWin, caret.gc,
						 caret.x * glo.fontWidth,
	    glo.fontStruct->ascent + glo.fontStruct->descent
				   + caret.y * glo.fontHeight - height,
					       glo.fontWidth, height );
    }

    if( caretType ) {
	if( caretType == 1 )
	    height = 0;
	else
	    height = glo.fontHeight / 3;
	height += glo.fontStruct->descent;
	XFillRectangle(glo.display, glo.mainWin, caret.gc,
						   x * glo.fontWidth,
	    glo.fontStruct->ascent + glo.fontStruct->descent
				   + y * glo.fontHeight - height,
					   glo.fontWidth, height );
	caret.x = x;
	caret.y = y;
    }
    caret.type = caretType;
}


/****************
 * Write a character with the current colors at position x,y
 * Using -1 for c flushes the internal buffering.
 */

void
JnxWriteChar( int x, int y, int c )
{
    static byte buf[100];
    static int	n, x1, y1;
    int ct, cx=0, cy=0; /* cx,cy are controlled by ct, init to avoid gcc warn*/

    if( n >= DIM(buf) || c == -1 ) {
	if( n ) {
	  flush:
	    if( caret.type && caret.y == y1
		&& caret.x >= x1 && caret.x < x1 + n ) {
		ct = caret.type;
		cx = caret.x;
		cy = caret.y;
		JnxSetCaret(0,0,0);
	    }
	    else
		ct = 0;
	    /*printf("draw: x=%2d y=%2d n=%3d c=%d\n", x1, y1, n, c );*/
	    x1 *= glo.fontWidth;
	    y1 *= glo.fontHeight;
	    y1 += glo.fontStruct->ascent;
	    XDrawImageString(glo.display, glo.mainWin, glo.gc, x1, y1, buf, n);
	    n = 0;
	    if( ct )
		JnxSetCaret(cx,cy,ct);
	}
	if( c == -1 )
	    return;
    }

    if( !n ) {
	x1 = x;
	y1 = y;
    }
    else if( y1 != y || x1 + n != x )
	goto flush; /* ;-)) */

    /* we use iso8879-1 for output.
     * all control characters are filtered away and replaced by a centered dot;
     * otherwise the Xserver garbles the output.
     */
    if( !(c & 0x60) || c == 127 )
	buf[n++] = 0xb7;
    else
	buf[n++] = c;
}

void
JnxStoreBytes( char *text, size_t size )
{
    XStoreBytes(glo.display, text, size );
}

void *
JnxFetchBytes( size_t *r_nbytes )
{
    void *p;
    int nbytes;

    p = XFetchBytes(glo.display, &nbytes );
    *r_nbytes = nbytes;
    return p;
}

void
JnxFreeFetchedBytes( void *p )
{
    if( p )
	XFree(p);
}


/*** eof ***/
