/* [wscreen.c wk 3.7.91] W-Editor Screen Kit
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 27.08.92 wk	in ShowMessage.. buf vergroessert und lastMsgLen
 *		begrenzt (assert schlug heute beim Speichern an)
 * 03.01.93 wk	some cleanups
 */

#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <time.h>
#include <wk/lib.h>
#include <wk/string.h>
#include <wk/mouse.h>
#include <wk/file.h>
#include <wk/direc.h>
#include <wk/bios.h>
#include <wk/wscrdrv.h>
#ifdef GNU_C_LIB_OS2
    #ifndef CLOCKS_PER_SEC
	#define CLOCKS_PER_SEC 1
    #endif
#endif

#include "w.h"
#include "wcmd.h"
#include "wscreen.h"
#include "wkbddrv.h"
#include "wfile.h"
#include "wmove.h"
#ifdef UNIX
#include "jnx.h"
#endif
#include "hilite.h"

/****** constants *********/
/******* typedefs ********/
/******* globals **********/
static int isInitialized;
static int codeTranslation;
static int monochromePalette;

static char *names[] = {
    "Text", "Mark", "Cursor", "Cmdline", "Status", "Status2",
    "Message", "Message2", "Browse", "Overscan",
    "Hilite1", "Hilite2", "Hilite3", "Hilite4", "Hilite5",
    "Hilite6", "Hilite7", "Hilite8", "Hilite9", NULL
};
static char *colors[16] = {
   "Black", "Blue", "Green", "Cyan", "Red", "Magenta", "Brown", "White",
   "Dark-Grey", "Light-Blue", "Light-Green", "Light-Cyan", "Light-Red",
   "Light-Magenta", "Yellow", "Bright-White"
};
static int sxText, syText, pyText, atText, atText2, atText3, atText4;
static int sxBrow, syBrow, pyBrow, atBrow;
static int sxInfo, pyInfo, atInfo, atInfo2;
static int sxCmd, pyCmd, atCmd;
static int sxMsg, pyMsg, atMsg, atMsg2;
static int atHilite[9] = { 0x74, 0x75, 0x76, 0x71, 0x72, 0x73, 0x7a,
			   0x7b, 0x7c };

static int fileHd, browseFileHd;

static int textUpdLocked;

static const char **textTblBuffer;
static ushort *lenTblBuffer;
static int lastMsgLen;
static const char *topText;
static const char *bottomText;
#ifdef UNIX
static const char mouseMenu[] =
		    "<> ln pg";
#else
static const char mouseMenu[] =
		    "\x1b" "\x1a" " \x18" "\x19" " \x1e" "\x1f";
#endif
static int mouseMenuPos[3]; /* jeweils 2 positionen breit */
static byte *hiliteBuffer; /* malloced of length sxText */

/******* prototypes *******/
static void UpdInfo( void );
static void UpdCmdLine( void);
static void UpdText( void );
static void UpdBrowse(void);
static void UpdCursor(void);
static void DrawMouseMenu(void);
static void NotifyHandler( int notification );
static void Initialize(int,int);
static void CleanUp( void *dummy );
/******* functions ********/


/****************
 * Create a screen
 * mode 0 = standard
 *	1 = Codetranslation ISO <--> CP 850
 * bwFlag = true: use B/W colors
 * Returns: Handle
 */

void
MakeScreen( int mode, int bwFlag )
{
    static int sentinel;
    int i;

  #if W_FULL
    if( sentinel )
	Bug("Only 1 Screen at the moment");
    sentinel++;
  #endif
  #ifdef __linux__
    if( mode == 1 ) {
	/* disable font switching = assume latin-1 is loaded*/
	mode = 0;
	VTLinuxSetCharSet( 1 );
    }

  #endif
    ScreenSetTrans(mode);
    for(i=0; i < DIM(atHilite); i++ )
	HiliteSetColor(i, atHilite[i]);
    Initialize( mode, bwFlag );
}


/****************
 * Die Zeichenumsetzungstabelle ndern.
  * mode 0 ::= Standard
 *	1 ::= Latin-1
 */

void
ScreenSetTrans( int mode )
{
  #ifdef UNIX
    topText    = "=== Top of file ===";
    bottomText = "=== Bottom of file ===";
    codeTranslation = 0;
  #else
  #if W_FULL
    if( !mode ) {
  #endif
	topText    = "\xcd\xcd\xcd Top of file \xcd\xcd\xcd";
	bottomText = "\xcd\xcd\xcd Bottom of file \xcd\xcd\xcd";
	codeTranslation = 0;
  #if W_FULL
    }
    else {
	topText    = "\x9b\x9b\x9b Top of file \x9b\x9b\x9b";
	bottomText = "\x9b\x9b\x9b Bottom of file \x9b\x9b\x9b";
	codeTranslation = 1;
    }
    if( isInitialized ) /* driver noch umschalten */
	ScrDrvCodeTrans( mode );
  #endif
  #endif
}


void LockTextUpdate( int state )
{
    textUpdLocked = state;
}

void RedrawScreen()
{
    lastMsgLen = sxMsg;
    UpdateScreen();
  #if W_FULL
    DrawMouseMenu();
  #endif
    ShowMessage(NULL);
    ScrDrvRepaint();
    ScrDrvShow();
}

void RemoveScreen()
{
    ScrDrvMouseOff();
    ScrDrvRestScreen();
    ScrDrvShow();
}

void EndRemoveScreen()
{
    ScrDrvSaveScreen();
    ScrDrvMouseOn();
}


/****************
 * bind fileHandle to the Screen
 */

void
BindScreen( int fHd )
{
    UpdateFileRing( fHd );
    fileHd = fHd;
    UpdateWindowTitle(fHd);
}


void
UpdateWindowTitle( int fHd )
{
    if( fileHd != fHd )
	return;
  #ifdef UNIX
    if( JnxAvailable() ) {
	const char *name;
	unsigned flags;
	int isdir;
	char *p, *base, *dir;

	name = GetFileInfo( fHd, &flags );
	base = basename(name);
	if( isdir = ((flags & WFILE_INT) && !strcmp( name, ".dir" )) )
	    dir = xstrdup(GetDirPattern());
	else
	    dir = dirname(name);
	p = xmalloc( strlen(wklibApplicationName) + strlen(base) +
						    strlen(dir) + 20);
	sprintf(p,"%s - %s%s", wklibApplicationName,
		    base, flags & WFILE_CHG? " (edited)":"" );
	if( *dir )
	    sprintf(p+strlen(p), " %s %s", isdir?"of":"in", dir );

	JnxSetTitle(0,p);
	JnxSetTitle(1,base);
	free(base);
	free(dir);
	free(p);
    }
  #endif
}

/****************
 * Bind a filehandle to the supplementary Text Window;
 * this will open this window. if fhd is -1, the binding will be released
 * and this window will disappear.
 */
#if W_FULL
void BindSupplementaryScreen( int fhd )
{
    if( fhd == -1 ) { /* close the browse window */
	if( browseFileHd != -1 ) {
	    pyText -= syBrow;
	    syText += syBrow;
	    browseFileHd = -1;
	}
	return;
    }
    if( browseFileHd == -1 ) { /* open the browse window */
	if( syText < 10 )
	    return ;  /* screen is too small */
	sxBrow = sxText;
	syBrow = 5;
	pyBrow = pyText;
	pyText += syBrow;
	syText -= syBrow;
    }
    browseFileHd = fhd;
}
#endif



void UpdateScreen()
{
    UpdCursor();
    UpdInfo();
    UpdText();
  #if W_FULL
    if( browseFileHd != -1 )
	UpdBrowse();
  #endif
    UpdCmdLine();
    ScrDrvShow();
}

#if W_FULL
void IdleScreen( int mode )
{
    static long lastHit, interval;
    static int saved;

    if( !mode ) {
	lastHit = clock();
	saved = 0;
	interval = GetSetOption( SETOPT_CRTSAVE );
    }
    else if( interval ) {
	if( !saved ) {
	    if( clock() - lastHit > interval * CLOCKS_PER_SEC ) {
		ScrDrvPauseCrt( 1 );
		saved = 1;
		lastHit = clock();
		ScrDrvShow();
	    }
	}
	else if( clock() - lastHit > CLOCKS_PER_SEC/5 ) {
	    ScrDrvPauseCrt( 1 );
	    ScrDrvShow();
	}
    }
}
#endif


/****************
 * Den Filehandle des aktuellen Screens zurueckgeben.
 */

int QryScreenFile()
{
    return fileHd;
}

int QrySupplementaryScreenFile()
{
    return browseFileHd;
}


/****************
 * Die InfoZeile des aktuellen Screens anzeigen:
 */

static void UpdInfo()
{
    char buf[60], *p, cbuf[3+1], kprefix, name[45];
    int x, i, textVal, attr, format;
    unsigned flag;
    ushort nbytes, reclen;
    size_t n;
    ulong filY, winY;	    /* akt.zeile, abstand window vom fileanfang*/
    ushort filX, winX;	    /* dito pos in Zeile */
    ushort crsY, crsX, cmdX, cmdNbytes, cmdWin, cmdCrs;
    int topVis;


    Filename2Str( GetFileInfo2( fileHd, &flag, &format, &reclen ),
						 name, DIM(name) );
    GetAllFilePos( fileHd, &filX, &filY, &winX, &winY, &crsX, &crsY );
    topVis = winY ? 0 : 1;
    GetPtr2CmdLine(fileHd,&cmdX);
    GetAllCmdLineInfo( fileHd, &cmdNbytes, &cmdX, &cmdWin, &cmdCrs );
    GetPtr2Line( fileHd, &nbytes );
    if( nbytes > winX )
	nbytes -= winX;

    if( crsX < nbytes ) {
	ScrDrvReadCell( crsX, crsY+pyText+topVis, &textVal, &attr);
	sprintf( cbuf, " %02X", textVal&0xff );
    }
    else
	strcpy( cbuf, " --" );

    attr = flag & WFILE_CHG ? atInfo2 : atInfo;

    kprefix = KybrdDrvGetPrefix();
    n = sprintf( buf, "%c %s%s%u %lu %c%s%u%c%s %s%s",
		  kprefix ? kprefix : ' ',
		  GetLastEofState() ? "EOF ":"",
		  winX ? "\x1b":"",
		  filX+1, filY+1,
		  codeTranslation? '<':'(',
		  cmdWin ? "\x1b":"", cmdX+1,
		  codeTranslation? '>':')',
		  QryReservedMemoryState() ? "": " LowMem",
		  flag & WFILE_INSERT ? "Insert" : "Replace", cbuf );

    for(x=0; x < sxInfo && name[x] ; x++ )
	ScrDrvWriteCell( x, pyInfo, name[x], attr );
  #if W_FULL
    if( flag & WFILE_INT ) {
	if( !strcmp( name, ".dir" ) ) {
	    for(i=0; x < sxInfo-n && i < 4 ; x++, i++)
		ScrDrvWriteCell( x, pyInfo, " of "[i] , atInfo );
	    p = Filename2Str( GetDirPattern(), name, DIM(name)-8 );
	    for(; x < sxInfo-n && *p ; x++, p++)
		ScrDrvWriteCell( x, pyInfo, *p , atInfo );
	}
    }
  #endif
    if( flag & WFILE_NEW && !( flag & WFILE_INT ) )
	for(i=0; x < sxInfo-n && i < 6 ; x++, i++)
	    ScrDrvWriteCell( x, pyInfo, " (new)"[i] , atInfo );
  #if W_FULL
    if( flag & WFILE_VIEW  )
	for(i=0; x < sxInfo-n && i < 7 ; x++, i++)
	    ScrDrvWriteCell( x, pyInfo, " (view)"[i] , atInfo );
    else if( (flag & WFILE_LOCK) && (flag & WFILE_RO)  )
	for(i=0; x < sxInfo-n && i < 8 ; x++, i++)
	    ScrDrvWriteCell( x, pyInfo, " (inUse)"[i] , atInfo );
    else if( flag & WFILE_LOCK	)
	for(i=0; x < sxInfo-n && i < 9 ; x++, i++)
	    ScrDrvWriteCell( x, pyInfo, " (locked)"[i] , atInfo );
    else
  #endif
	 if( flag & WFILE_RO  )
	for(i=0; x < sxInfo-n && i < 5 ; x++, i++)
	    ScrDrvWriteCell( x, pyInfo, " (ro)"[i] , atInfo );

    if( flag & WFILE_NOTAB )
	for(i=0; x < sxInfo-n && i < 9 ; x++, i++)
	    ScrDrvWriteCell(  x, pyInfo, " (noTabs)"[i] , atInfo );
    if( format == WFILE_FMT_FIX )
	for(i=0; x < sxInfo-n && i < 6 ; x++, i++)
	    ScrDrvWriteCell(  x, pyInfo, " (fix)"[i] , atInfo );
    else if( format == WFILE_FMT_2PR )
	for(i=0; x < sxInfo-n && i < 6 ; x++, i++)
	    ScrDrvWriteCell(  x, pyInfo, " (prf)"[i] , atInfo );
    else if( format == WFILE_FMT_BIN )
	for(i=0; x < sxInfo-n && i < 6 ; x++, i++)
	    ScrDrvWriteCell(  x, pyInfo, " (bin)"[i] , atInfo );
    for(; x < sxInfo-n ; x++)
	ScrDrvWriteCell( x, pyInfo, ' ', atInfo );
    for( x= sxInfo-n, p = buf; x < sxInfo; x++, p++ )
	ScrDrvWriteCell( x, pyInfo, *p, atInfo );
}



/****************
 * Die CommandLine anzeigen:
 */

static void UpdCmdLine()
{
    char *p;
    int x;
    ushort pos, win, crs, nbytes;


    p = GetAllCmdLineInfo( fileHd, &nbytes, &pos, &win, &crs );
    x=0;
    if( nbytes >= win ) {
	p += win;
	nbytes -= win;
	if( nbytes > sxCmd )
	    nbytes = sxCmd;
	ScrDrvWriteStr( x, pyCmd, p, nbytes, atCmd );
	x += nbytes;
    }
    if( x < sxCmd )
	ScrDrvWriteNChar( x, pyCmd, ' ', atCmd, sxCmd - x );
}


/****************
 * Den Cursor anzeigen
 */

static void UpdCursor()
{
    unsigned flags;
    ulong  filY, winY;
    ushort filX, winX, crsY, crsX;

    GetFileInfo( fileHd, &flags );
    if( flags & WFILE_INCMD ) {
	GetAllCmdLineInfo( fileHd, &crsY /*dummy*/, &filX, &winX, &crsX );
	ScrDrvSetCrs( crsX, pyCmd );
    }
    else {
	GetAllFilePos( fileHd, &filX, &filY, &winX, &winY, &crsX, &crsY);
	if( !winY )
	    crsY++;
	ScrDrvSetCrs( crsX, crsY+pyText );
    }
    ScrDrvShowCrs( flags & WFILE_INSERT? 2 : 1 );
}


void GetScreenSize( int *sx, int *sy )
{
    *sx = sxText;
    *sy = syText;
}


ushort GetCmdLineLen()
{
    return sxCmd;
}




/****************
 * UpdText() den Text neu anzeigen
 */

static void UpdText( )
{
    static const char **text;
    static ushort *lenTbl;
    int mtype, x, y, i, topVis, chr, at, sy, sx;
    enum { sTOP, sTEXT, sBOTTOM, sCLEAR } state;
    size_t ml1, ml2, mlHelp;
    ushort mp1, mp2;
    const char *p;
    ushort len;
    unsigned fileFlags;
    ulong  filY, winY;
    ushort filX, winX, crsY, crsX;
    size_t n;

    if( textUpdLocked )
	return ;

    text   = textTblBuffer;
    lenTbl = lenTblBuffer;

    GetAllFilePos( fileHd, &filX, &filY, &winX, &winY, &crsX, &crsY );
    topVis = winY ? 0 : 1;
    sx = sxText;
    sy = topVis ? (syText-1) : syText;
    GetFileInfo( fileHd, &fileFlags );

  #if USE_EMS
    mtype = GetLineBlockInit( fileHd, winY, sy+1 , &ml1, &mp1, &ml2, &mp2);
  #else
    mtype = GetLineBlock( fileHd, winY, text+topVis, lenTbl+topVis,
			  sy+1 , &ml1, &mp1, &ml2, &mp2);
  #endif

    if( mtype ) {
	ulong xl1, xl2;
	ushort xp1, xp2;

	xl1 = MarkStart( &xp1 );
	xl2 = MarkEnd( &xp2 );
	if( xl1 < winY )
	    ml1 = 0;
	if( xl2 >= winY+sy )
	    ml2 = sy-1;
    }
    else
	ml1 = ml2 = sy+10;  /* invalidate mark info */

    if( topVis ) {
	ml1++;
	ml2++;
	state = sTOP;
    }
    else
	state = sTEXT;

    for( y=0; y < syText; ) {
	switch( state ) {
	  case sTOP:
	    ScrDrvWriteStr( 0, y+pyText, topText, n=strlen(topText), atText4);
	    if( n < sx )
	       ScrDrvWriteNChar( n, y+pyText, ' ', atText4, sx - n );
	    y++;
	    state = sTEXT;
	    break;

	  case sBOTTOM:
	    ScrDrvWriteStr( 0, y+pyText, bottomText,
					 n=strlen(bottomText), atText4 );
	    if( n < sx )
		ScrDrvWriteNChar( n, y+pyText, ' ', atText4, sx - n );
	    y++;
	    state = sCLEAR;
	    break;

	  case sCLEAR:
	    ScrDrvWriteNChar( 0, y+pyText, ' ', atText4, sx);
	    y++;
	    break;

	  case sTEXT:
	  #if USE_EMS	/* nacheinander holen */
	    if( !(p = GetLineBlockNext( &len )) )
		state = sBOTTOM;
	    else {
	  #else
	    if( !(p = text[y]) )
		state = sBOTTOM;
	    else { /* }*/
		len = lenTbl[y];
	  #endif
		if( len > winX ) {
		    p += winX;
		    len -= winX;
		}
		else
		    len = 0;
		if( len > sx )
		    len = sx;
		mlHelp =  y >= ml1 && y <= ml2 ;
		if( !mlHelp ) {
		    if( fileFlags & WFILE_HILITE ) {
			const byte *attbuf;
			if( attbuf = HiliteLine(fileHd, winY+y-topVis, winX,
							 hiliteBuffer, len  ))
			    ScrDrvWriteAttrStr( 0, y+pyText, p, len, attbuf);
			else
			    ScrDrvWriteStr( 0, y+pyText, p, len, atText);
		    }
		    else
			ScrDrvWriteStr( 0, y+pyText, p, len, atText);
		    if( len < sx )
			ScrDrvWriteNChar( len, y+pyText, ' ',
					  atText4, sx - len );

		}
		else if( mtype == MARKTYPE_LINE ) {
		    ScrDrvWriteStr( 0, y+pyText, p, len, atText2);
		    if( len < sx )
			ScrDrvWriteNChar( len, y+pyText, ' ',
					  atText2, sx - len );
		}
		else if( mtype == MARKTYPE_BLOCK ) {
		    ScrDrvWriteStrFill( 0, y+pyText, p, len, sx);
		    x = 0;
		    i = (mp1 - winX);
		    if( i > 0 ) {
			if( i > sx )
			    i = sx;
			ScrDrvWriteNAttr( x, y+pyText, atText, i );
			x += i;
			i = (mp2 - mp1) + 1;
		    }
		    else
			i = (mp2 - winX) + 1;
		    if( i > 0 && x < sx ) {
			if( i+x > sx )
			    i = sx-x;
			ScrDrvWriteNAttr( x, y+pyText, atText2, i );
			x += i;
		    }
		    if( x < sx ) {
			i = sx - x;
			ScrDrvWriteNAttr( x, y+pyText, atText, i );
		    }
		}
		else {	/* MARKTYPE_CHAR */
		    for(x=0; x < len ; x++, p++ ) {
			if( ml1 == ml2 )
			    at = x >= mp1 && x <= mp2 ? atText2:atText;
			else if( y == ml1 )
			    at = x >= mp1 ? atText2:atText;
			else if( y == ml2 )
			    at = x <= mp2 ? atText2:atText;
			else
			    at = atText2;
			ScrDrvWriteCell( x, y+pyText, *p, at );
		    }
		    for(; x < sx ; x++ ) { /* clear rest of line */
			at = atText;
			ScrDrvWriteCell( x, y+pyText, ' ', at );
		    }
		}
		y++;
	    }
	    break;

	  default: Bug("UpdText state=%d", state);
	}
    }


    if( fileFlags & WFILE_INCMD ) {
	ScrDrvReadCell( crsX, crsY+pyText+topVis, &chr, &at );
	ScrDrvWriteCell( crsX, crsY+pyText+topVis, chr, atText3 );
    }

}


/****************
 * UpdBrowse() um das Browse Window anzuzeigen
 * (1. Zeile = Cursorposition)
 */

#if W_FULL
static void UpdBrowse(void)
{
    static const char **text;
    static ushort *lenTbl;
    int  y, sy, sx;
    size_t ml1, ml2;
    ushort mp1, mp2;
    const char *p;
    ushort len;
    ulong  filY, winY;
    ushort filX, winX, crsY, crsX;



    text   = textTblBuffer;
    lenTbl = lenTblBuffer;

    GetAllFilePos( browseFileHd, &filX, &filY, &winX, &winY, &crsX, &crsY );
    sx = sxBrow;
    sy = syBrow;

    GetLineBlock( browseFileHd, filY, text, lenTbl,
			  sy+1 , &ml1, &mp1, &ml2, &mp2);
    if( monochromePalette )
	sy--;
    for( y=0; y < sy; ) {
	if( !(p = text[y]) )
	    break;
	else {
	    len = lenTbl[y];
	    if( len > winX ) {
		p += winX;
		len -= winX;
	    }
	    else
		len = 0;
	    if( len > sx )
		len = sx;
	    ScrDrvWriteStr( 0, y+pyBrow, p, len, atBrow);
	    if( len < sx )
		ScrDrvWriteNChar( len, y+pyBrow, ' ',
				  atBrow, sx - len );
	    y++;
	}
    }
    for( ; y < sy; y++ )
	ScrDrvWriteNChar( 0, y+pyBrow, ' ', atBrow, sx);
    if( monochromePalette )
	ScrDrvWriteNChar( 0, y+pyBrow, topText[0], atBrow, sx );
}
#endif


/****************
 * Eine Message anzeigen, die alte wird geloescht, wird NULL
 * angegeben, so wird nur die alte geloescht. Message dar nicht
 * laenger als 80 Zeichen werden.
 */

void ShowMessage( const char *s, ... )
{
    va_list arg_ptr ;
    char buf[81];
    int x;

    for(x=0; x < lastMsgLen; x++ )
	ScrDrvWriteCell( x, pyMsg, ' ', atMsg );
    if( s ) {
	va_start( arg_ptr, s ) ;
	lastMsgLen = vsprintf( buf, s, arg_ptr) ;
	va_end(arg_ptr);
	if( lastMsgLen >= sxMsg )
	    lastMsgLen = sxMsg - 1;
	for(x=0; x < lastMsgLen; x++ )
	    ScrDrvWriteCell( x, pyMsg, buf[x], atMsg );
    }
    ScrDrvShow();
}


/****************
 * Same as ShowMessage, but in another color
 */

void ShowMessageAsInfo( const char *s, ... )
{
    va_list arg_ptr ;
    char buf[120];
    int x;

    for(x=0; x < lastMsgLen; x++ )
	ScrDrvWriteCell( x, pyMsg, ' ', atMsg );
    if( s ) {
	va_start( arg_ptr, s ) ;
	lastMsgLen = vsprintf( buf, s, arg_ptr) ;
	xassert( lastMsgLen < DIM(buf) );
	va_end(arg_ptr);
	if( lastMsgLen >= sxMsg )
	    lastMsgLen = sxMsg -1;
	for(x=0; x < lastMsgLen; x++ )
	    ScrDrvWriteCell( x, pyMsg, buf[x], atMsg2 );
    }
    ScrDrvShow();
}


/****************
 * Funktion wie ShowMessage, aber Ausgabe erfolgt direkt via BIOS Call
 * und kann deswegen im Critical errorhandler benutzt wird.
 */

#if MSDOS || DOS386 || DOS16RM
void ShowMessageViaBios( const char *s, ... )
{
    va_list arg_ptr ;
    char buf[120];
    int x;
    int oldY, oldX;

    WKBiosGetCursorPos( 0, &oldY, &oldX );
    for(x=0; x < lastMsgLen; x++ ) {
	WKBiosSetCursorPos( 0, pyMsg, x );
	WKBiosPutCell( 0, ' ', 7 );
    }
    if( s ) {
	va_start( arg_ptr, s ) ;
	lastMsgLen = vsprintf( buf, s, arg_ptr) ;
	va_end(arg_ptr);
	xassert( lastMsgLen < DIM(buf) );
	if( lastMsgLen >= sxMsg )
	    lastMsgLen = sxMsg -1;
	for(x=0; x < lastMsgLen; x++ ) {
	    WKBiosSetCursorPos( 0, pyMsg, x );
	    WKBiosPutCell( 0, buf[x], 12 );
	}
    }
    WKBiosSetCursorPos( 0, oldY, oldX );
}
#endif



/****************
 * Switch automark mode on; that means, the mark will be extended with
 * the mouse movement, until all mousebuttont are release.
 */

void ScreenAutoMark()
{
  #if W_FULL
    int hd, markMode, i, btn;
    ulong  lnr, lastLnr;
    ushort pos, lastPos;

    markMode = MarkInfo( &hd );
    if( hd != fileHd )
	markMode = 0;
    lastLnr = ULONG_MAX;
    lastPos = USHRT_MAX;
    while( markMode ) {
	i = ScreenMousePos( &btn, &lnr, &pos );
	if( !btn )
	    markMode = 0;
	else if( i != 2 )
	    ; /* wait til mouse is back in textArea */
	else if( lnr != lastLnr || pos != lastPos ) {
	    lastLnr = lnr;
	    lastPos = pos;
	    Move2Line( fileHd, lnr );
	    Move2Column( fileHd, pos );
	    SetMark( fileHd, markMode );
	    UpdText();
	    ScrDrvShow();
	}
    }
  #endif
}



/****************
 * Position des Mousecursors bestimmen
 * Returns: 0 - Mouse nicht in einem aktiven Window
 *	    1 - Mouse in CommandLine, ( lineNr ist dann undefiniert )
 *	    2 - Mouse im textfenster
 */

int ScreenMousePos( int *btn, ulong *retlnr, ushort *retpos )
{
  #if W_FULL
    unsigned fileFlags;
    int px, py;
    ulong  filY, winY;
    ushort filX, winX, crsY, crsX;

    if( ScrDrvQryMouse( btn, &px, &py ) ) {
	if( py == pyCmd ) {
	    GetAllCmdLineInfo( fileHd, &crsY /*dummy*/, &filX, &winX, &crsX );
	    *retpos = winX + px;;
	    return 1;
	}
	else if( py >= pyText && py < pyText+syText ) {
	    py -= pyText;
	    GetAllFilePos( fileHd, &filX, &filY, &winX, &winY, &crsX, &crsY );
	    if( !winY && !py )
		return 0;
	    GetFileInfo( fileHd, &fileFlags );
	    *retlnr = winY? (winY + py) : py-1;
	    *retpos = winX + px;
	    return 2;
	}
    }
  #endif
    return 0;
}



/****************
 * Falls eine Menue aktiv ist, gibt diese Funktion die ID des Menuepunktes
 * in abhaengigkeit von der (globale) screenpos zurueck.
 * Returns: 0 = kein Menupunkt
 *	    1 = Menupunkt 1
 *	    2 = Menupunkt 2
 *	    3 = Menupunkt 3
 */

int ScreenQryMenuID( int x, int y )
{
    int i;

  #if W_FULL
    if( y == pyMsg && x >= sxMsg )  /* inside Mousemenu */
	for(i=0; i < 3; i++ )
	    if( x == mouseMenuPos[i] || x == mouseMenuPos[i]+1 )
		return i+1;
  #endif
    return 0;
}


#if W_FULL
static void
DrawMouseMenu()
{
    if( mouseMenuPos[0] >= sxMsg )
	ScrDrvWriteStr( mouseMenuPos[0], pyMsg,
			mouseMenu, strlen(mouseMenu), atMsg2 );
}
#endif


/****************
 * Change the screen lay out, (will only register it for next output)
 * Screen layouts are assigned by number: ( from top to bottom )
 * -1 = use current layout
 * 0 = Standard layout ( this is same as 1 )
 * 1 = text, cmd, info, msg
 * 2 = info, msg, cmd, text
 * 3 = info, msg, text, cmd
 * 4 = msg, info, text, cmd
 * 5 = info, text, cmd, msg
 * 6 = msg, info, text, cmd
 * 7 = msg, info, cmd, text
 */

int ScreenChangeLayout( int layoutID )
{
    int btn,x,y, err=0;
    int oldBrowseFileHd;
    static int currentLayout = 0;

    ScrDrvQrySize( &x, &y );
    if( y < 6 || x < 80 )
	Bug("Inv. Screensize (%d*%d)", x,y);
  #if W_FULL
    oldBrowseFileHd = browseFileHd;
    BindSupplementaryScreen( -1 ); /* switch off */
  #endif
    syText = y-3;
    sxText= sxCmd = sxInfo= sxMsg = x;

    if( layoutID == -1 )
	layoutID = currentLayout;

    switch( layoutID ) {
      case 0:
      case 1:	/* text, cmd, info, msg */
	pyText= 0;
	pyCmd = y-3;
	pyInfo= y-2;
	pyMsg = y-1;
	break;
    #if W_FULL
      case 2:	/* info, msg, cmd, text */
	pyInfo= 0;
	pyMsg = 1;
	pyCmd = 2;
	pyText= 3;
	break;

      case 3:	/* info, msg, text, cmd */
	pyInfo= 0;
	pyMsg = 1;
	pyText= 2;
	pyCmd = y-1;
	break;

      case 4:	/* msg, info, text, cmd */
	pyMsg = 0;
	pyInfo= 1;
	pyText= 2;
	pyCmd = y-1;
	break;

      case 5:	/* info, text, cmd, msg */
	pyInfo= 0;
	pyText= 1;
	pyCmd = y-2;
	pyMsg = y-1;
	break;

      case 6:	/* msg, info, text, cmd */
	pyMsg = 0;
	pyInfo= 1;
	pyText= 2;
	pyCmd = y-1;
	break;

      case 7:	/* msg, info, cmd, text */
	pyMsg = 0;
	pyInfo= 1;
	pyCmd = 2;
	pyText= 3;
	break;

      case 8:	/* info, text, cmd, msg */
	pyInfo= 0;
	pyText= 1;
	pyCmd = y-2;
	pyMsg = y-1;
	break;
    #endif
      default: err = ERR_INVARG; break;
    }
    currentLayout = layoutID;

  #if W_FULL
    if( ScrDrvQryMouse( &btn, &x, &y ) ) {
	mouseMenuPos[0] = sxMsg - 8;
	mouseMenuPos[1] = sxMsg - 5;
	mouseMenuPos[2] = sxMsg - 2;
	sxMsg -= 8;
    }
    BindSupplementaryScreen( oldBrowseFileHd ); /* restore old state */
  #endif

    free(textTblBuffer);
    free(lenTblBuffer);
    textTblBuffer = xmalloc( (syText+1) * sizeof *textTblBuffer );
    lenTblBuffer = xmalloc( (syText+1) * sizeof *lenTblBuffer );
    free(hiliteBuffer);
    hiliteBuffer = xmalloc( sxText + 5 );

    return err;
}


/****************
 * change the screen colors, ( screenRedraw needed )
 * expects a string containing 8 colordefinitions  in hex
 * or the name of the sreenelement and the two colors as text
 * to change one element after the other (case insensitive)
 * e.g.:  to change the Cmdline:
 *	 "cmdline  bright white on green"
 * the colors are:
 * "Black", "Blue", "Green", "Cyan", "Red", "Magenta", "Brown", "White",
 * "Dark-grey", "Light-blue", "Light-green", "Light-cyan", "Light-red",
 * "Light-magenta", "Yellow", "Bright-white"
 *
 *  1. Color (Text): text
 *  2. Color (Marked): marked text
 *  3. Color (Cursor): cursor text
 *  4. Color (Cmdline): cmdline
 *  5. Color (Status): infoline
 *  6. Color (Status2): infoline modified
 *  7. Color (Message): messageline as error
 *  8. Color (Message2): messageline as info
 *  9. Color (Browse): Browse Window
 * 10. Color (Overscan): text with no charcters
 * 11. Color (Hilite1): 1. color for syntax highlighting
 * 12. Color (Hilite2): 2. color for syntax highlighting
 * 13. Color (Hilite3): 3. color for syntax highlighting
 * 14. Color (Hilite4): 4. color for syntax highlighting
 * 15. Color (Hilite5): 5. color for syntax highlighting
 * 16. Color (Hilite6): 6. color for syntax highlighting
 * 17. Color (Hilite7): 7. color for syntax highlighting
 * 18. Color (Hilite8): 8. color for syntax highlighting
 * 19. Color (Hilite9): 9. color for syntax highlighting
 *
 * or the keywords "mono", "colour" or "color" to select a default palette.
 * Hilite colors may only be set with the new syntax.
 */

int ScreenChangeColors( const char *str )
{
    int n, i, err=0, bg, fg;
    byte values[10];

    /* try the one-by-one format */
    str += strspn( str, " \t\n" ); /* skip leading blanks */
    i = strcspn( str, " \t\n" );
    for(n=0; names[n]; n++ )
	if( i == strlen(names[n]) && !strnicmp( str, names[n], i) ) {
	    /* found the colorname (we will use the new format) */
	    str += i;
	    str += strspn( str, " \t\n" ); /* skip blanks */
	    i = strcspn( str, " \t\n" );
	    for(fg=0; fg < 16; fg++ )
		if( i==strlen(colors[fg]) && !strnicmp(str, colors[fg], i) )
		    break;
	    if( fg == 16 )
		return ERR_INVARG;
	    str += i;
	    str += strspn( str, " \t\n" ); /* skip blanks */
	    if( toupper(*str) != 'O' || toupper(str[1]) != 'N'
		|| !strchr( " \t\n", str[2]) )
		return ERR_INVARG;
	    str += 3;
	    str += strspn( str, " \t\n" ); /* skip blanks */
	    i = strcspn( str, " \t\n" );
	    for(bg=0; bg < 16; bg++ )
		if( i==strlen(colors[bg]) && !strnicmp(str, colors[bg], i) )
		    break;
	    if( bg == 16 )
		return ERR_INVARG;
	    str += i;
	    str += strspn( str, " \t\n" ); /* skip blanks */
	    if( *str )
		return ERR_INVARG;
	    i = (bg << 4) | fg;
	    switch(n) {
	      case 0: atText = i;
		      HiliteSetColor(-1, i);
		      break;
	      case 1: atText2= i; break;
	      case 2: atText3= i; break;
	      case 3: atCmd  = i; break;
	      case 4: atInfo = i; break;
	      case 5: atInfo2= i; break;
	      case 6: atMsg  = i; break;
	      case 7: atMsg2 = i; break;
	      case 8: atBrow = i; break;
	      case 9: atText4= i; break;
	      default:
		xassert(n < 19);
		atHilite[n-10] = i;
		HiliteSetColor(n-10, i);
		break;
	    }
	    return 0;
	}


    /* no: try the old format */
    monochromePalette = 0;
    if( !strcmpl( str, "mono" ) ) {
	str = "0f7070700f098f0f070f";
	monochromePalette = 1;
    }
    else if( !strcmpl( str, "color" ) || !strcmpl( str, "colour" ) ) {
	/*str = "1f4f702f0f0c0c0b301f";  <-- PE2 style */
	str = "1f4f702f0f0c0c0b301f";
	str = "7040073f0f0c0c0b3070";
    }
    for( n = 0; n < 10 && !err ; n++ ) {
	str += strspn( str, " ,\t\n" ); /* skip Blanks and commas */
	if( !*str )
	    err = ERR_INVARG;
	else if( isxdigit( *str ) && isxdigit( str[1] ) ) {
	    values[n] = 16*(isdigit(*str) ? (*str-'0'):(toupper(*str)-'A'+10));
	    str++;
	    values[n] += isdigit(*str) ? (*str-'0'):(toupper(*str)-'A'+10);
	    str++;
	}
	else
	    err = ERR_INVARG;
    }

    if( !err ) {
	atText = values[0];
	HiliteSetColor(-1, atText);
	atText2= values[1];
	atText3= values[2];
	atCmd  = values[3];
	atInfo = values[4];
	atInfo2= values[5];
	atMsg  = values[6];
	atMsg2 = values[7];
	atBrow = values[8];
	atText4= values[9];
    }

    return err;
}


const char *
GetScreenColorDef( int n )
{
    static char buffer[60];
    int i;

    switch(n) {
      case 0: i = atText ; break;
      case 1: i = atText2; break;
      case 2: i = atText3; break;
      case 3: i = atCmd  ; break;
      case 4: i = atInfo ; break;
      case 5: i = atInfo2; break;
      case 6: i = atMsg  ; break;
      case 7: i = atMsg2 ; break;
      case 8: i = atBrow ; break;
      case 9: i = atText4; break;
      default:
	if(n < 19)
	    i = atHilite[n-10];
	else
	    return NULL;
    }
    sprintf(buffer,"set color %-8s %s on %s",
		    names[n], colors[i & 15], colors[((i & 0xf0)>>4)& 15] );
    return buffer;
}




/****************
 *  Returns: 0 = Standard codes
 *	     1 = ISO-8859-1 Code Translation active
 */

int ScreenQryCodeTrans(void)
{
    return codeTranslation;
}


static void
NotifyHandler( int notification )
{
    switch( notification ) {
      case 1: /* screen size changed */
	ScreenChangeLayout(-1); /* does the screen site change for us */
	RedrawScreen();
	break;

      default: /* none */ ;
    }
}


/****************
 * Initialisieren des screen Subsystems, kann beliebig oft aufgerufen werden,
 * da Abfrage darauf intern in diesem Modul
 */

static void
Initialize( int mode, int bwFlag )
{
    int y, x;
    const char *p;

    if( isInitialized )
	return;

    browseFileHd = -1;

    x = ScrDrvOpen();
    ScrDrvRegBreakFlag( &sigIntPending );
    ScrDrvRegNotifyHandler( NotifyHandler );
  #if W_FULL
    ScrDrvCodeTrans( mode );
  #endif
    if( !bwFlag )
	bwFlag = x;

    ScreenChangeColors( bwFlag ? "mono" : "color" );
    ScreenChangeLayout(0); /* setup Standard layout */
    ScrDrvShowCrs( 1 );
  #ifdef USE_VT_LINUX
    if( VTLinuxGetCharSet() == 2 ) {
	topText    = "\x92\x9a\x9a Top of file \x9a\x9a\x98";
	bottomText = "\x92\x9a\x9a Bottom of file \x9a\x9a\x98";
    }
  #endif

    isInitialized++;
    AddCleanUp( CleanUp, NULL );
    LockTextUpdate(1);
    for(y=0; y < syText; y++ )
	ScrDrvWriteNChar( 0, y+pyText, ' ', atText, sxText );
    ScrDrvWriteNChar( 0, pyCmd, ' ', atCmd, sxCmd );
    ScrDrvWriteNChar( 0, pyInfo, ' ', atInfo, sxInfo );
    ScrDrvWriteNChar( 0, pyMsg, ' ', atMsg, sxMsg );
  #if W_FULL
    DrawMouseMenu();
    ScrDrvWriteStr( 0, pyText, CopyRight(10), strlen(CopyRight(10)), atText );
    for( p = CopyRight(31), y=pyText+1; *p ; p += p[x]? (x+1) : x, y++ ) {
	for(x=0; p[x] && p[x] != '\n'; x++ )
	    ;
	ScrDrvWriteStr( 0, y, p, x, atText );
    }
    ScrDrvWriteStr( 0, pyCmd , CopyRight(22), strlen(CopyRight(22)), atCmd  );
    ScrDrvWriteStr( 0, pyInfo, CopyRight(21), strlen(CopyRight(21)), atInfo );
  #endif
    ScrDrvShow();
}



static void CleanUp( void *dummy )
{
    if( !isInitialized )
	return;

    free(textTblBuffer);
    free(lenTblBuffer);
    ScrDrvRestScreen();
    ScrDrvShow();
    ScrDrvClose();
    isInitialized = 0;
}


/*** bottom of file ***/
