/* [wscrdrv.c wk 31.10.91] Simple Full Screen Handler
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * ***** Special W version ********
 *
 * History:
 * 27.01.93 wk	C Set/2 Support
 * 06.05.93 wk	Moved GUI support to wpmdrv.c
 * 28.02.95 wk	rearranged keyboard handling
 */


#include "wtailor.h"

#if !UNIX  /* see wscrans.c for an UNIX compatible version */

#define DBG_SCANCODES 0


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/environ.h>
#include <wk/keys.h>
#include <wk/hwacc.h>
#if MSDOS
#elif OS2
#elif DOS386
#elif DOS16RM
#elif EMX
#elif WINNT
#else
    #error Operating system must either be MSDOS, OS2, or EMX
#endif
#include <wk/mouse.h>
#include <wk/bios.h>
#if EMX
    #include <os2.h>
#elif defined(OS20)
    #define INCL_DOSPROCESS 1
    #include <os2.h>
    #define INCL_WK_WSCRDRV_GUI 1
#elif defined(OS2)
    #define INCL_DOSPROCESS 1
    #define INCL_KBD 1
    #define INCL_VIO 1
    #if OS20
       #define INCL_DOSDEVIOCTL 1
    #endif
    #include <os2.h>
    #define INCL_WK_WSCRDRV_GUI 1
#elif WINNT && !__CYGWIN32__
#else
    #include <bios.h>
    #include <dos.h>
  #if __ZTC__
    #include <sound.h>
  #endif
    #define HAB long
#endif
#include <wk/wscrdrv.h>

/****** constants **********/
#define MAX_DOSKEYS  20  /* our own pretype buffer for MSDOS */

/******* types *************/
#ifdef WINNT
typedef CHAR_INFO cell_t;
#else
typedef struct {
	char c;
	char a;
    } cell_t;
#endif

#if __IBMC__
typedef volatile cell_t * crtptr_t;
#elif __WATCOMC__ && DOS386
typedef volatile cell_t *crtptr_t;
#else
typedef volatile cell_t _far *crtptr_t;
#endif

/******** globals **********/
#if OS2 || EMX
    static crtptr_t vioBuffer ;
    static int hvio;
#elif WINNT
    static HANDLE con_out, con_in;
    static int vio_changed;
    SMALL_RECT viorect;
#endif
#if !OS2 && !WINNT
    static struct {
	ushort chr,scan;
	int state;
    } kbdBuf[MAX_DOSKEYS]; /* array to store the keystrokes */
    static int kbdBufCnt;  /* number of keys in buffer */
    static crtptr_t screenBase ;
#endif

static int isColor ;		  /* flag for color-mode */
static struct {
	int sx, sy, sxMul2;
	int x2, y2 ;
	int crsX, crsY ;
	int crsForm ;
	int crsActiv ;
	int size ;		  /*total size*/
	int buttons;		  /* # of mouse button s or 0 when no mouse*/
    } screen ;			  /* Screen Parameters*/


static int crtSaveFlag ;

static struct {
	int videoMode;
	ushort cursorForm;
	int crsX, crsY;
	struct { int px, py; } mouse;
	ushort orgScrSize;	    /* # of elem in orgScr */
	ushort orgScrRows;
	ushort orgScrCols;
	cell_t *orgScr; 	    /* area to save org Screen */
    } save ;			    /* save original screen Environment*/


static cell_t *image ;	   /* area to store the image */
static int translateCodes;

static volatile int *breakFlagAddr;

#ifdef WINNT
  #define VIDEOPAGE ((int)con_out)
#else
  #define VIDEOPAGE 0
#endif

/********** prototypes *********/
static cell_t GetRndCell();

/*********** defined functions ********/
/*********** functions ****************/

/****************
 * Returns: 0 = standard color screen
 *	    1 = standard mono screen
 */


int
ScrDrvOpen()
{
    int err=0, cols, rows, pages , i ;
  #if OS20
  #elif EMX
    USHORT * _Seg16 vBuffer;
    ULONG vioBufferSize;
  #elif OS2
    crtptr_t vBuffer;
    ushort vioBufferSize;
  #elif WINNT
  #endif

    save.videoMode = -1 ;
  #if DEBUG && !WINNT
    xassert( sizeof (cell_t) == 2 );
  #endif
  #if OS2 || EMX
   #if EMX
    if( _osmode == OS2_MODE ) {
   #endif
	save.videoMode = -1;
	hvio = 0;
	isColor = WKBiosGetVideoMode( &cols, &rows ) != 7;
   #if EMX
    }
    else {  /* _osmode == DOS_MODE */
   #endif
  #endif
  #if !OS2  && !WINNT /* DOS or EMX continued */
	isColor = 1 ; /* default*/
	save.videoMode = WKBiosGetVideoMode( &cols, &rows );
	switch( save.videoMode ) {
	  case 0x07: isColor = 0; save.videoMode = -1;break;
	  case 0x02: case 0x03: case 0x23: case 0x4D: case 0x17:
	  case 0x4F: case 0x57: case 0x19: case 0x1A: case 0x24:
	  case 0x45: case 0x32: case 0x2A: case 0x4E: case 0x40:
	  case 0x18: case 0x33: case 0x22: case 0x44: case 0x43:
	  case 0x26: case 0x5A: case 0x34:  save.videoMode = -1; break;
	  default:
	    rows = 25;
	    WKBiosSetVideoMode( 3 ) ; /* change Vidomode to Standard*/
	    break ;
	}
	kbdBufCnt = 0;
   #if EMX
    }
   #endif
  #elif WINNT

/*  con_out = CreateFile( "CONOUT$", */
    if( (con_out = GetStdHandle(STD_OUTPUT_HANDLE)) == INVALID_HANDLE_VALUE ) {
	Error(0,"GetStdHandle failed: rc=%d", GetLastError() );
	err =-1;
	goto leave;
    }
    if( (con_in = GetStdHandle(STD_INPUT_HANDLE)) == INVALID_HANDLE_VALUE ) {
	Error(0,"GetStdHandle failed: rc=%d", GetLastError() );
	err =-1;
	goto leave;
    }
    save.videoMode = -1;
    isColor = 1; /* FIXME!*/
    {	CONSOLE_SCREEN_BUFFER_INFO info;

	if( !GetConsoleScreenBufferInfo( con_out, &info ) ) {
	    Error(0,"GetConsoleScreenBufferInfo failed: rc=%d", GetLastError());
	    err =-1;
	    goto leave;
	}
	cols = info.dwSize.X;
	rows = info.dwSize.Y;
    }
  #endif /* !winnt */

    /**** Setup Screen Parameters ****/
    screen.sx = cols ;
    screen.sy = rows ;
    screen.sxMul2 = cols*2 ;
    screen.x2 = cols -1 ;
    screen.y2 = rows -1 ;
    screen.size = screen.sx * screen.sy ;
    if( !(image = calloc( screen.size, sizeof *image )) )
	{ err = -1; goto leave; }

  #if OS20
    vioBuffer = (crtptr_t)WKVideoGetBuffer( NULL );
  #elif WINNT
    vio_changed = 0;
    viorect.Left  = screen.sx;
    viorect.Right = 0;
    viorect.Top = screen.sy;
    viorect.Bottom = 0;
  #elif OS2 || EMX
   #if EMX
    if( _osmode == OS2_MODE ) {
   #endif
	 #if !EMX
	    VioGetBuf( (PULONG)&vBuffer, (PUSHORT)&vioBufferSize, hvio);
	    vioBuffer = (crtptr_t)vBuffer;
	 #endif
   #if EMX
    }
    else
	screenBase = HW_MakePtrToMemory( isColor ? 0xb8000 : 0xb0000 ,
							  screen.size*2 );
   #endif
  #else
    screenBase = HW_MakePtrToMemory( isColor ? 0xb8000 : 0xb0000,
							  screen.size*2 );
  #endif

    save.orgScrSize = screen.size;
    save.orgScrRows = screen.sy;
    save.orgScrCols = screen.sx;
    if( !(save.orgScr = malloc( save.orgScrSize * sizeof *save.orgScr)) )
	{ err = -1; goto leave ; }
    ScrDrvMouseOn();
    ScrDrvSaveScreen();
    ScrDrvShowCrs( 1 ); /* initial painting of cursor */
    ScrDrvHideCrs();	/* and hide him */
    ScrDrvRepaint();	/* Sync with image */
    MouseShow();

  leave:
    if( err && save.videoMode != -1 )  /* Reset Videomode on error*/
	WKBiosSetVideoMode( save.videoMode ) ;
    if( err )
	Error(4,"Can't open CRT: not enough core");
    return isColor ? 0 : 1;
}




#if INCL_WK_WSCRDRV_GUI
HAB ScrDrvGetActiveHAB(void)
{
    return 0;
}
#endif


#ifdef WINNT

unsigned
WKKeyboardGetToggle()
{
  #if 0
    KBDINFO kbi;

    kbi.cb = sizeof kbi;
    if( !KbdGetStatus( &kbi, 0 ) )
	return (kbi.fsState >> 4) & 0x07;
    else
	return 0;
  #endif
	return 0;
}


/****************
 * Get a character from the keyboard
 * mode 0 ::= no wait
 *	1 ::= wait
 *
 * Returns: 0 = Character available
 *	    !0 = no Character or error
 */

int
WKKeyboardGetChar(int mode, ushort *chr, ushort *scan, unsigned *state)
{
    INPUT_RECORD irec;
    ulong n;

    if( !mode ) {
	if( !GetNumberOfConsoleInputEvents(con_in, &n ) ) {
	    Error(0,"GetNumberOfConsoleInputEvents failed: rc=%d",
						GetLastError() );
	    return 1;
	}
	if( !n )
	    return 2; /* no evenets */
    }

  repeat:
    if( !ReadConsoleInput(con_in, &irec, 1, &n ) ) {
	Error(0,"ReadConsoleInput failed: rc=%d", GetLastError() );
	return 1;
    }

    switch( irec.EventType ) {
      case KEY_EVENT:
	if( !irec.Event.KeyEvent.bKeyDown ) {
	    if( mode )
		goto repeat;
	    return 2;
	}
      #if 0
	Info("vkey=%3hu scan=%3hu char=%3u (%c) rep=%hu state=%lx",
		irec.Event.KeyEvent.wVirtualKeyCode,
		irec.Event.KeyEvent.wVirtualScanCode,
		irec.Event.KeyEvent.uChar.AsciiChar,
		irec.Event.KeyEvent.uChar.AsciiChar >=' '
		&& irec.Event.KeyEvent.uChar.AsciiChar <='z' ?
		irec.Event.KeyEvent.uChar.AsciiChar : '?',
		irec.Event.KeyEvent.wRepeatCount,
		irec.Event.KeyEvent.dwControlKeyState  );
      #endif
	if( scan )
	    *scan = irec.Event.KeyEvent.wVirtualScanCode;
	if( chr ) {
	    if( irec.Event.KeyEvent.dwControlKeyState & ENHANCED_KEY )
		*chr = 224;
	    else
		*chr  = irec.Event.KeyEvent.uChar.AsciiChar;
	}
	if( state ) {
	    *state  = 0;
	    if( irec.Event.KeyEvent.dwControlKeyState & SHIFT_PRESSED )
		*state |= K_FLAG_SHIFT;
	    if( irec.Event.KeyEvent.dwControlKeyState
		& (LEFT_CTRL_PRESSED | RIGHT_CTRL_PRESSED) )
		*state |= K_FLAG_CTRL;
	    if( irec.Event.KeyEvent.dwControlKeyState & LEFT_ALT_PRESSED )
		*state |= K_FLAG_ALT;
	    if( irec.Event.KeyEvent.dwControlKeyState & RIGHT_ALT_PRESSED )
		*state |= K_FLAG_COMPOSE;
	}
	return 0;
      default:
	if( mode )
	    goto repeat; /* we want a char and nothing else */
	return 2; /* unsupported event */
    }
}


#endif /* WINNT */





/****************
 *  enables or disables code translation
 */

void ScrDrvCodeTrans( int mode )
{
    translateCodes = mode ;
}

void ScrDrvRegBreakFlag( volatile int *flag )
{
    breakFlagAddr = flag;
}

void
ScrDrvRegNotifyHandler( void (*f)(int) )
{
    /* dummy */
}

void ScrDrvClose()
{
    ScrDrvRegBreakFlag( NULL );
    FREE( image );
    FREE( save.orgScr );
}



void ScrDrvQrySize( int *x, int *y )
{
    *x = screen.sx ;
    *y = screen.sy ;
}


void ScrDrvSetCrs( int x, int y )
{
    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
    screen.crsY=y;
    screen.crsX=x ;
    if( screen.crsActiv )
	WKBiosSetCursorPos( VIDEOPAGE, screen.crsY, screen.crsX ) ;
}


void ScrDrvHideCrs()
{
    screen.crsActiv = 0 ;
    WKBiosSetCursorPos( VIDEOPAGE, screen.sy, 0) ;  /* will hide cursor */
}

void ScrDrvMouseOn()
{
    screen.buttons = MouseInitialize();
    MousePut(save.mouse.px, save.mouse.py);
}


void ScrDrvMouseOff()
{
    MouseGet( &save.mouse.px, &save.mouse.py );
    MouseTerminate();
    screen.buttons = 0;
}


void ScrDrvShowCrs( int form )
{
    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
    if( form == screen.crsForm )
	;
    else if( form == 1 ) {
      #if OS2
	WKBiosSetCursorForm( 0x0c0f ) ;
      #elif WINNT
	/*WKBiosSetCursorForm( 0x0c0f )*/ ;
      #else
	WKBiosSetCursorForm( isColor? 0x0607:  0x0b0c ) ;
      #endif
	screen.crsForm = 1 ;
    } else if( form == 2 ) {
      #if OS2
	WKBiosSetCursorForm( 0x070f ) ;
      #elif WINNT
	/*WKBiosSetCursorForm( 0x070f )*/ ;
      #else
	WKBiosSetCursorForm( isColor? 0x0307:0x060c ) ;
      #endif
	screen.crsForm = 2 ;
    }
    screen.crsActiv = 1 ;
    WKBiosSetCursorPos( VIDEOPAGE, screen.crsY, screen.crsX ) ;
}


void ScrDrvSaveScreen()
{
    cell_t *mPtr;
    crtptr_t sPtr;
    ushort cnt;

  #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB to switch to output Screen */
    int dmy1, dmy2;	/* by calling video Bios */
    WKBiosGetVideoMode( &dmy1, &dmy2 );
  #endif
    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if !OS2
    MouseHide();
  #endif
    mPtr = save.orgScr;
  #if OS2
    if( !vioBuffer )
	return;
    sPtr = vioBuffer;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(mPtr++) = *(sPtr++);
  #elif WINNT
  #else
    sPtr = screenBase;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(mPtr++) = *(sPtr++);
  #endif
  #if !OS2
    MouseShow();
  #endif
    save.cursorForm = WKBiosGetCursorForm() ;
    WKBiosGetCursorPos( 0, &save.crsY, &save.crsX );
}


void ScrDrvRestScreen()
{
    cell_t *mPtr;
    crtptr_t sPtr;
    ushort cnt;


  #if __ZTC__ && DEBUG && !OS2	/* Tell ZDB to switch to output Screen */
    int dmy1, dmy2;	/* by calling video Bios */
    WKBiosGetVideoMode( &dmy1, &dmy2 );
  #endif
  #if !OS2
    MouseHide();
  #endif
    mPtr = save.orgScr;
  #if OS2
    if( !vioBuffer )
	return;
    sPtr = vioBuffer;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(sPtr++) = *(mPtr++);
  #elif WINNT
  #else
    sPtr = screenBase;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(sPtr++) = *(mPtr++);
  #endif
  #if !OS2
    MouseShow();
  #endif
    WKBiosSetCursorForm( save.cursorForm ) ;
    WKBiosSetCursorPos( VIDEOPAGE, save.crsY, save.crsX ) ;
}



void
ScrDrvShow()
{
  #if OS20
    MouseHide();
    WKVideoShowBuffer( screen.size*2 );
    MouseShow();
  #elif OS2
    MouseHide();
    VioShowBuf( 0, screen.size*2, hvio);
    MouseShow();
  #elif WINNT
    if( vio_changed ) {
	COORD i_size, i_pos;
	SMALL_RECT rect;

	i_size.X = screen.sx;
	i_size.Y = screen.sy;
	i_pos.X = viorect.Left;
	i_pos.Y = viorect.Top;
	if( !WriteConsoleOutput(con_out, image, i_size, i_pos, &viorect ) )
	    Error(0,"WriteConsoleOutput failed: rc=%d", GetLastError());
	vio_changed = 0;
	viorect.Left  = screen.sx;
	viorect.Right = 0;
	viorect.Top = screen.sy;
	viorect.Bottom = 0;
    }
  #endif
}


void
ScrDrvRepaint()
{
    int i;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
#ifdef WINNT
  vio_changed = 0;
  viorect.Left	= screen.sx;
  viorect.Right = 0;
  viorect.Top = screen.sy;
  viorect.Bottom = 0;
#else
  #if OS2
    if( !vioBuffer )
	return;
  #endif
  #if !OS2
    MouseHide();
  #endif
    for(i=0; i < screen.size; i++ ) {
      #if OS2
	vioBuffer[i] = image[i];
      #else /* DOS*/
	screenBase[i] = image[i];
      #endif
    }

  #if !OS2
    MouseShow();
  #endif
#endif
}



/****************
 * Query the Mouse
 * Returns: Number of Buttons or 0 if no Mouse available in this screen Area
 */

int
ScrDrvQryMouse( int *btn, int *px, int *py )
{
    *btn = MouseGet( px, py );
    return screen.buttons;
}


void
ScrDrvReadCell( int x, int y, int *c, int *a )
{
    int i;

    i = y * screen.sx + x ;
    if( i >= 0 && i < screen.size ) {
      #ifdef WINNT
	*c = translateCodes ? MapIbm2Iso( image[i].Char.AsciiChar, 1 )
			    : image[i].Char.AsciiChar;
	*a = image[i].Attributes;
      #else
	*c = translateCodes ? MapIbm2Iso( image[i].c, 1 ) : image[i].c;
	*a = image[i].a;
      #endif
    }
}


void
ScrDrvWriteCell( int x, int y, int c, int a )
{
    int i;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    i = y * screen.sx + x ;
    if( c & 0x80 )
	if( translateCodes )
	    c = MapIso2Ibm(c,1);
    if( i >= 0 && i < screen.size ) {
      #ifdef WINNT
	if( image[i].Char.AsciiChar != c || image[i].Attributes != a ) {
	    image[i].Char.AsciiChar = c ;
	    image[i].Attributes = a ;
	    vio_changed = 1;
	    if( x < viorect.Left )
		viorect.Left = x;
	    if( x > viorect.Right )
		viorect.Right = x;
	    if( y < viorect.Top )
		viorect.Top = y;
	    if( y > viorect.Bottom )
		viorect.Bottom = y;
	}
      #else
	if( image[i].c != c || image[i].a != a ) {
	    image[i].c = c ;
	    image[i].a = a ;
	  #if !OS2
	    MouseHideCond( x, y, x+1, y);
	  #endif
	  #if OS2
	    vioBuffer[i] = image[i];
	  #else /* DOS*/
	    screenBase[i] = image[i];
	  #endif
	  #if !OS2
	    MouseShow();
	  #endif
	}
      #endif
    }
}


void
ScrDrvWriteStr( int x, int y, const char *buf, size_t len, int a )
{
    int i, anyupd, c;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; len && i < screen.size; len--, i++, buf++ ) {
	    c = *buf;
	    if( c & 0x80 )
		if( translateCodes )
		    c = MapIso2Ibm(c,1);
	  #ifdef WINNT
	   if( image[i].Char.AsciiChar != c || image[i].Attributes != a ) {
	       image[i].Char.AsciiChar = c ;
	       image[i].Attributes = a ;
	       vio_changed = 1;
	       if( x < viorect.Left )
		   viorect.Left = x;
	       if( x > viorect.Right )
		   viorect.Right = x;
	       if( y < viorect.Top )
		   viorect.Top = y;
	       if( y > viorect.Bottom )
		   viorect.Bottom = y;
	    }
	    if( ++x > screen.x2 ) { x=0; y++; }
	  #else
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	  #endif
	}
  #if !OS2 && !WINNT
    if( anyupd )
	MouseShow();
  #endif
}

void
ScrDrvWriteAttrStr( int x, int y, const char *buf,
		    size_t len, const byte *abuf)
{
    int i, anyupd, c, a;


    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; len && i < screen.size; len--, i++, buf++, abuf++ ) {
	    c = *buf;
	    a = *abuf;
	    if( c & 0x80 )
		if( translateCodes )
		    c = MapIso2Ibm(c,1);
	  #ifdef WINNT
	   if( image[i].Char.AsciiChar != c || image[i].Attributes != a ) {
	       image[i].Char.AsciiChar = c ;
	       image[i].Attributes = a ;
	       vio_changed = 1;
	       if( x < viorect.Left )
		   viorect.Left = x;
	       if( x > viorect.Right )
		   viorect.Right = x;
	       if( y < viorect.Top )
		   viorect.Top = y;
	       if( y > viorect.Bottom )
		   viorect.Bottom = y;
	    }
	    if( ++x > screen.x2 ) { x=0; y++; }
	  #else
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	  #endif
	}
  #if !OS2 && !WINNT
    if( anyupd )
	MouseShow();
  #endif
}

/****************
 * Einen String der laenge len ausgeben und rechts mit blanks
 * bis auf laenge maxlen auffuellen. Attribute werden nicht geschrieben.
 * der String wird aber nur solange ausgegeben bis auch maxlen erreicht ist;
 * nur ein eventuell kuerzerer String wird mit blanks aufgefuellt.
 */

void ScrDrvWriteStrFill( int x, int y, const char *buf,
			 size_t len, size_t maxlen )
{
    int i, anyupd, c;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 ) {
	for( ; maxlen && len && i < screen.size; len--, maxlen--,i++, buf++ ) {
	    c = *buf;
	    if( c & 0x80 )
		if( translateCodes )
		    c = MapIso2Ibm(c,1);
	  #ifdef WINNT
	    if( image[i].Char.AsciiChar != c  ) {
	       image[i].Char.AsciiChar = c ;
	       vio_changed = 1;
	       if( x < viorect.Left )
		   viorect.Left = x;
	       if( x > viorect.Right )
		   viorect.Right = x;
	       if( y < viorect.Top )
		   viorect.Top = y;
	       if( y > viorect.Bottom )
		   viorect.Bottom = y;
	    }
	    if( ++x > screen.x2 ) { x=0; y++; }
	  #else
	    if( image[i].c != c ) {
		image[i].c = c ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	  #endif
	}
	for( ; maxlen && i < screen.size; maxlen--,i++ ) {
	  #ifdef WINNT
	    if( image[i].Char.AsciiChar != ' ' ) {
	       image[i].Char.AsciiChar = ' ';
	       vio_changed = 1;
	       if( x < viorect.Left )
		   viorect.Left = x;
	       if( x > viorect.Right )
		   viorect.Right = x;
	       if( y < viorect.Top )
		   viorect.Top = y;
	       if( y > viorect.Bottom )
		   viorect.Bottom = y;
	    }
	    if( ++x > screen.x2 ) { x=0; y++; }
	  #else
	    if( image[i].c != ' ' ) {
		image[i].c = ' ' ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+len, y);
		  #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio);
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	  #endif
	}
    }
  #if !OS2 && !WINNT
    if( anyupd )
	MouseShow();
  #endif
}



void ScrDrvWriteNChar( int x, int y, int c, int a, size_t n )
{
    int i, anyupd;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( c & 0x80 )
	if( translateCodes )
	    c = MapIso2Ibm(c,1);
    if( i >= 0 )
	for( ; n && i < screen.size; n--, i++ ) {
	  #ifdef WINNT
	    if( image[i].Char.AsciiChar != c || image[i].Attributes != a ) {
	       image[i].Char.AsciiChar = c ;
	       image[i].Attributes = a ;
	       vio_changed = 1;
	       if( x < viorect.Left )
		   viorect.Left = x;
	       if( x > viorect.Right )
		   viorect.Right = x;
	       if( y < viorect.Top )
		   viorect.Top = y;
	       if( y > viorect.Bottom )
		   viorect.Bottom = y;
	    }
	    if( ++x > screen.x2 ) { x=0; y++; }
	  #else
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+n, y);
		  #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio) ;
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	  #endif
	}
  #if !OS2 && !WINNT
    if( anyupd )
	MouseShow();
  #endif
}


/****************
 * Nur Attribute schreiben
 */

void ScrDrvWriteNAttr( int x, int y, int a, size_t n )
{
    int i, anyupd;

    if( crtSaveFlag )
	ScrDrvPauseCrt(0);
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    anyupd = 0;
    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; n && i < screen.size; n--, i++ ) {
	  #ifdef WINNT
	    if( image[i].Attributes != a ) {
	       image[i].Attributes = a ;
	       vio_changed = 1;
	       if( x < viorect.Left )
		   viorect.Left = x;
	       if( x > viorect.Right )
		   viorect.Right = x;
	       if( y < viorect.Top )
		   viorect.Top = y;
	       if( y > viorect.Bottom )
		   viorect.Bottom = y;
	    }
	    if( ++x > screen.x2 ) { x=0; y++; }
	  #else
	    if( image[i].a != a ) {
		image[i].a = a ;
		if( !anyupd ) {
		    anyupd++;
		  #if !OS2
		    MouseHideCond( x, y, x+n, y);
		  #endif
		}
	      #if OS2 && 0
		VioWrtCellStr( (PCH)&image[i], 2, i/screen.sx, i%screen.sx,hvio) ;
	      #elif OS2
		vioBuffer[i] = image[i];
	      #else /* DOS*/
		screenBase[i] = image[i];
	      #endif
	    }
	  #endif
	}
  #if !OS2 && !WINNT
    if( anyupd )
	MouseShow();
  #endif
}



/****************
 * Mode = 1 : Put Crt into Pause Mode
 *	  0 : Restore CRT if CRT  is in Pause Mode
 */

void ScrDrvPauseCrt( int mode )
{
  #if OS2 && 0
    ushort x,y;
  #else
    crtptr_t sPtr;
  #endif
    ushort cnt;
    cell_t cell;
    int i;

    if( !crtSaveFlag && !mode )
	return;
  #if OS2
    if( !vioBuffer )
	return;
  #endif
    if( mode ) {
      #if __ZTC__ && DEBUG && !OS2 && !WINNT  /* Tell ZDB to switch to output Screen */
	int dmy1, dmy2;     /* by calling video Bios */
	WKBiosGetVideoMode( &dmy1, &dmy2 );
      #endif
      #ifdef WINNT
	cell.Char.AsciiChar = ' ';
	cell.Attributes = 7;
      #else
	cell.c = ' ';
	cell.a = 7;
      #endif
      #if !OS2 && !WINNT
	MouseHide();
      #endif
	cnt  = save.orgScrSize;
      #if OS2 && 0
	if( !crtSaveFlag ) {
	    for( y=0 ; y < save.orgScrRows ; y++ )
		VioWrtNCell( (PCH)&cell, 2*screen.sx, y, 0, hvio ) ;
	} else {
	    for(i=0; i < 3; i++ ) {/* write some random cells */
		cell = GetRndCell();
		i = rand() % cnt;
		VioWrtCellStr( (PCH)&cell, 2, i / screen.sx, i % screen.sx,hvio);
	    }
	}
      #elif OS2
	sPtr = vioBuffer;
	if( !crtSaveFlag ) {
	    while( cnt-- )  /* clear screen at first call */
		*(sPtr++) = cell;
	    cnt  = save.orgScrSize;
	}
	for(i=0; i < 3; i++ ) /* write some random cells */
	    sPtr[ rand() % cnt ] = GetRndCell();
      #elif WINNT
	/* FIXME */;
      #else
	sPtr = screenBase;
	if( !crtSaveFlag ) {
	    while( cnt-- )  /* clear screen at first call */
		*(sPtr++) = cell;
	    cnt  = save.orgScrSize;
	}
	for(i=0; i < 3; i++ ) /* write some random cells */
	    sPtr[ rand() % cnt ] = GetRndCell();
      #endif
      crtSaveFlag = 1;
    }
    else if( crtSaveFlag ) {
	crtSaveFlag = 0;
	ScrDrvRepaint();
  #if !OS2 && !WINNT
	MouseShow();
  #endif
    }
}




/**********
 * Put one Cell to the screen
 */


static cell_t
GetRndCell()
{
    cell_t val;
#ifdef WINNT
  #define X val.Char.AsciiChar
#else
  #define X val.c
#endif
    switch( rand() % 1000 ) {
      case 1:  X = '.';        break;
      case 2:  X = '.';        break;
      case 3:  X = '.';        break;
      case 4:  X = '.';        break;
      case 5:  X = '.';        break;
      case 6:  X = '.';        break;
      case 7:  X = '.';        break;
      case 8:  X = '.';        break;
      case 9:  X = '+';        break;
      case 10: X = '+';       break;
      case 11: X = '+';       break;
      case 12: X = '*';       break;
      case 13: X = '.';       break;
      case 14: X = 'w';       break;
      case 15: X = '.';       break;
      case 16: X = '*';       break;
      case 17: X = 'W';       break;
      case 18: X = '.';       break;
      default: X = ' ';        break;
    }
    X = (rand() % 6)+1;
    return val;
#undef X
}


/****************
 * Ermittelt die aktuellen Togglekey states:
 * Bit 0 = Scroll Lock
 *     1 = Num Lock
 *     2 = Caps Lock
 */

unsigned ScrDrvGetToggleKeys()
{
    unsigned state;

  #if OS20 || WINNT
    state = WKKeyboardGetToggle();
  #elif OS2
    KBDINFO kbi;

    state = 0;
    kbi.cb = sizeof kbi;
    if( !KbdGetStatus( &kbi, 0 ) )
	state = (kbi.fsState >> 4) & 0x07;
  #elif EMX || UNIX
    state = 0; /* no way yet found to determine */
  #elif __WATCOMC__
    state = (_bios_keybrd(_KEYBRD_SHIFTSTATUS) >> 4) & 0x07;
  #else
    state = (bioskey(0x02) >> 4) & 0x07;
  #endif
    return state;
}


/****************
 * Get the scancode, the value, and the status;
 * look at the mouse buttons.
 * when a mouse buuton is hit, K_FLAG_MOUSE is set in flags
 * and rVal contains the keycode K_BTN_xxxxx
 *
 * Returns wether a keyvalue is available or not
 */

int
ScrDrvBasicInkey( ushort *rVal, ushort *rScan, unsigned *rFlags )
{
    ushort flags, chr, scan;
    int c, state, i, msX, msY;
    unsigned msBtn;
  static unsigned lastMsBtn;
  #if OS20 || WINNT
    unsigned kstate;
  #elif OS2
    KBDKEYINFO kbci;
  #endif

    chr = scan = flags = 0;
  #if OS20 || WINNT
    if( WKKeyboardGetChar( 0, &chr, &scan, &kstate ) )
	return 0;/* fehler (z.B. kein focus */
  #elif OS2
    if( KbdCharIn( &kbci, IO_NOWAIT, 0 ) )
	return 0;/* fehler (z.B. kein focus */
    if( !(kbci.fbStatus & ~(
			    KBDTRF_SHIFT_KEY_IN | KBDTRF_INTERIM_CHAR_IN |
			    KBDTRF_CONVERSION_REQUEST
			    )))
	return 0;/* not a useful character */
    scan = kbci.chScan;
    chr  = kbci.chChar;
  #else
    ScrDrvPoll();
    if( kbdBufCnt ) {
	chr   = kbdBuf[0].chr;
	scan  = kbdBuf[0].scan;
	state = kbdBuf[0].state;
	for(i=1; i < kbdBufCnt; i++ )
	    kbdBuf[i-1] = kbdBuf[i];
	kbdBufCnt--;
    }
    else if( msBtn = MouseGet( &msX, &msY ) ) {
	if( lastMsBtn == msBtn )
	    return 0;

	lastMsBtn = msBtn;
      #if EMX
	state = 0; /* ?? */
      #elif __WATCOMC__
	state = _bios_keybrd(_KEYBRD_SHIFTSTATUS);
      #else
	state = bioskey(0x02);
      #endif
	flags |= K_FLAG_MOUSE;
	if( msBtn & MOUSE_LEFT )
	    chr = K_BTN_LEFT;
	else if( msBtn & MOUSE_RIGHT )
	    chr = K_BTN_RIGHT;
	else if( msBtn & MOUSE_LEFTDBL )
	    chr = K_BTN_LEFTDBL;
	else /* MOUSE_RIGHTDBL */
	    chr = K_BTN_RIGHTDBL;
    }
    else {
	lastMsBtn = 0;
	return 0;
    }
  #endif


  #if OS20 || WINNT
    flags |= kstate;
  #elif OS2
    if( kbci.fsState & (KBDSTF_RIGHTSHIFT| KBDSTF_LEFTSHIFT) )
	flags |= K_FLAG_SHIFT;
    if( kbci.fsState & KBDSTF_CONTROL )
	flags |= K_FLAG_CTRL;
    if( kbci.fsState & KBDSTF_ALT )
	flags |= K_FLAG_ALT;
  #else
    if( state & ( 0x01| 0x02) )
	flags |= K_FLAG_SHIFT;
    if( state & 0x04 )
	flags |= K_FLAG_CTRL;
    if( state & 0x08 )
	flags |= K_FLAG_ALT;
  #endif

    *rVal  = chr;
    *rScan = scan;
    *rFlags =flags;
    return 1;
}



/****************
 * Poll Fuktion sollte zumindest bei DOS hauefig aufgerufen werden.
 * Hintergrund:
 * Leider hat die DOS Key Queue keine Infos ueber den
 * akt. shiftstate, allso muessen wir versuchen diese
 * Infos selber zu speichern
 */

void ScrDrvPoll(void)
{
#if !OS2 && !WINNT
    int state, c;
  #if DOS386 && __ZTC__
    static short _far *fp;
  #elif EMX
    static short *fp;
  #endif

  #if 0
    while( bioskey(0x11) ) { /* zortech funktioniert mit x11 nicht */
  #elif DOS386 && __ZTC__    /* also hackermaessig */
    if( !fp )
	fp = _x386_mk_protected_ptr( 0x0041a );

    while( fp[0] != fp[1] ) {
  #elif EMX
    if( !fp )
	fp = HW_MakePtrToMemory( 0x0041a , 2 * sizeof *fp );

    while( fp[0] != fp[1] ) {
  #elif DOS386 && __WATCOMC__
    while( *(short*)0x41a != *(short*)0x41c ) {
  #else
    while( *(short*)MK_FP(0x40,0x1a) != *(short*)MK_FP(0x40,0x1c) ) {
  #endif
      #if EMX
	state = 0;  /* ?? */
	c = _read_kbd(0, 1, 0 );
	if( c != -1 ) {
	    if( !c )
		c = (_read_kbd(0, 1, 0 ) << 8) & 0xff00;
	    else
		c &= 0xff;
	 }
	 else
	    c = 0;
      #elif __WATCOMC__
	state = _bios_keybrd(_KEYBRD_SHIFTSTATUS);
	c = _bios_keybrd(_NKEYBRD_READ);
      #else
	state = bioskey(0x02);
	c = bioskey(0x10);
      #endif
	if( kbdBufCnt+1 < MAX_DOSKEYS ) {
	    kbdBuf[kbdBufCnt].chr  = c & 0xff;
	    kbdBuf[kbdBufCnt].scan = (c >> 8) & 0xff;
	    kbdBuf[kbdBufCnt].state= state;
	    kbdBufCnt++;
	}
	else /* buffer overflow */
	  #if __ZTC__
	    sound_beep(1172); /* 880 Hz */
	  #else
	    putc('\a', stderr);
	  #endif
    }
#endif
}


#endif /*!UNIX*/
/**** end of file ***/
