/* [wfile.c wk 4.7.91] W-Editor File handling
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 *  9.05.92 wk	Neue Funktion: GetPrevFileHandle()
 * 21.05.92 wk	Neue Option in RenameFile: "." als filename
 * 30.05.92 wk	Added ExitList Function for OS/2
 *  6.12.92 wk	Unlock now sets file to r/o, user may reset to r/w
 *		added caseinsensive searchmode
 * 20.12.92 wk	AllocFile um startline und maxlines erweitert.
 * 03.01.93 wk	some cleanups
 * 12.01.93 wk	removed tabmode, cause meaning has changed
 * 16.05.93 wk	added lineId logic
 * 07.12.93 wk	Patches for OS/2 Netware Requestor Problems
 * 14.12.93 wk	Noch mehr patches; verlagert in Macro FixSlash()
 * 06.03.95 wk	UN*X fixes
 */

#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#if __ZTC__
#include <direct.h>
#endif
#include <ctype.h>
#include <setjmp.h>
#include <time.h>
#include <errno.h>
#if UNIX
#include <sys/stat.h>
#endif
#include <unistd.h>
#if OS2 || OS20
  #define INCL_DOSPROCESS 1
  #include <os2.h>
#endif
#include <wk/lib.h>
#include <wk/file.h>
#include <wk/string.h>
#include <wk/io.h>
#if __ZTC__ && !OS2 && USE_EMS
  #include <handle.h>
  #define USE_HANDLES 1
#else
  #define __handle    /* undef this */
#endif

#include "w.h"
#include "wfile.h"
#include "wcmd.h"
#include "wscreen.h"

/****** constants *********/
#define VIRTUAL 0   /* 1 = use a virtual memory manager for the file */
#if W_FULL
#define MARKSTACK_SIZE 10
#define LABELTBL_SIZE  26   /* for A .. Z */
#else
#define MARKSTACK_SIZE 5
#define LABELTBL_SIZE  1    /* for A .. Z */
#endif


#define VIEWLINES 200 /* load in chunks of 200 lines */


/******* typedefs ********/

/*
 * Der zu editierende Text wird in der folgenden Struktur gehalten:
 * Es handelt sich um eine doppelt verkettete liste, wobei jedes Element eine
 * Textzeile enthaelt. 'len' gibt die Anzahl der gueltigen Zeichen in 'd' an;
 * Falls 'len' 0 ist so ist in 'd' ein Byte vorhanden, hierbei handelt es sich
 * dann aber um ein ungueltiges byte. Das Zeilenende Zeichen wird nicht mit
 * abgespeichert.
 * Das erste Element der Liste hat als 'prv' NULL, das letze hat
 * als 'nxt' NULL.
 */

typedef struct t_line {
	struct t_line __handle *nxt; /* double linked list: pointer to next */
	struct t_line __handle *prv; /* 	    and to previous line */
	struct {
	  #if VIRTUAL
	    unsigned  len : 15;    /* len of line */
	    unsigned virt : 1;	  /* d is a virtual block */
	  #else
	    ushort len;
	  #endif
	} x;
	char   d[1];		/* dynamicly extended: size is len */
    } line_t;

#if VIRTUAL
typedef struct {    /* replaces d in line_t	*/
	byte	flag;	/* true if in tempfile, false in original file */
	long	offset; /* offset in file */
	ulong	nlines; /* number of lines represented by this block */
    } virtual_t;
#endif



/****************
 * Diese Struktur enthaelt alle Infos ueber den File
 */

typedef struct {
	char *name;		/* name of file  or NULL if unused slot */
	FILE *stream;		/* Stream of file or NULL if closed */
      #if VIRTUAL
	int virtual;		/* virtuality flag */
      #endif
	int intern;		/* Is internal Flag */
	int  view;		/* a view file */
	struct {
	    unsigned readOptions;
	    int     format;
	    ushort  fixRecLen;
	    long    offset;
	} vm;			/* for view mode */
	int  new;		/* This is a new file */
	int  ro;		/* readonly file */
	int  chg;		/* contents of file was changed */
	int  ring;		/* file is in ring */
	int  lock;		/* sema lock */
	int  noTab;		/* save file without tab compressing */
	int  format;		/* WFILE_FMT_xxxx */
	ushort fixRecLen;	/* fixed reclen, valid only for format .._FIX */
	ushort tabs[MAX_TABS];	/* tab array for this file */
	ushort tabcnt;		/* number of valid tabs */
	ushort lmargin, rmargin, dmargin;
	int  insertMode;	/* Cursor is in insertMode */
	int  inCmdFlag; 	/* Cursor is in Command */
	ushort cmdWin;		/* Window position in commandLine */
	ushort cmdPos;		/* Current Position in commandLine */
	ushort cmdCrs;		/* Cursor Position in commandLine */
	char *cmdline;		/* 0-terminated String, */
				/* Buffer size = MAX_LINELEN+1 */
	char *editBuf;		/* Editbuffer of size: MAX_LINELEN+1 */
	int    editBufValid;	/* Buffer is valid */
	ushort editBufLen;	/* cur Len of EditBuf */
	ulong  editBufLine;	/* line of Editbuf */
	ushort winX;		/* Position of window in file */
	ulong  winY;		/* LineNr of window in file */
	ushort crsX, crsY;	/* Position of textCursor in window */
	ushort maxLen;		/* max. Len of lines */
	ushort nattrib; 	/* number of additioanl attribute bytes */
				/* at the end of the line */
				/* the first one is always used for */
				/* syntax highlighting */
	int    lineIdOff;	/* 0 = no lineIds, > 0 offset of lineId */
	ushort lineIdRover;	/* just during file loading */

	ulong  nlines;		/* total number of lines */

	/* controls the file storage */
	line_t __handle *first; /* first Element in linked list or NULL */
	line_t __handle *crs;	/* pointer to current Element or NULL */
	line_t __handle *hilite_crs;   /* a second pointer  */
	ushort pos;		/* Position of Cursor in 'crs' */
	ulong  lnr ;		/* number of cur line: 0... */
	ulong  hilite_lnr ;	/* for hilite_crs */
	int  hiliteFlag;
	char hiliteType[5+1];	/* type for highlighting or "" for default */
      #ifdef UNIX
	struct stat statbuf;
	int statbuf_ok; 	/* statbuf is valid */
      #endif
	char *profile_var;
    } file_t;

typedef struct {
	int    type;	    /* MARKTYPE_...  oder 0 */
	int    hd;	    /* handle of mark */
	byte   anchorLine;  /* 0 = startLine is anchor */
	byte   anchorPos;   /* 0 = startPos  is anchor */
	ulong  startLine;   /* Mark Start Line */
	ushort startPos;    /* Mark Start Pos in dieser Zeile */
	ulong  endLine;     /* Mark End Cursor */
	ushort endPos;	    /* Mark End Pos in dieser Zeile */
    } mark_t;		    /* es gilt immer: start <= end !! */

/******* globals **********/
static file_t fileTbl[MAX_FILES];
static int    fileRing[MAX_FILES];
static int unnamedFile = -1;	/* handle of unnamed file */

static mark_t mark;
static mark_t markStack[MARKSTACK_SIZE];
static int markStackInd;
static struct {
	int fhd;	/* handle of file, or -1 if label not set */
	ulong lnr;
	ushort pos;
    } labelTbl[LABELTBL_SIZE]; /* indexed by A..Z */

static const char whiteSpaces[] = " \t\n" ;
#if VIRTUAL
static ulong spillChunk = 100;	/* may change depeneding on available memory*/
static char *spillFileName;	/* NULL if unused */
static FILE *spillStream;
#endif
static char *viewBuffer;

static int sortCompareCounter;
static jmp_buf sortCompareJmpBuf;
static enum { dALFA, dNUM, dNAME, dEXT, dSIZE, dDATE, dALFACI } sortMode;
static int directorySortKludge;
static int directorySortReverse;

#if USE_EMS
static struct {
	file_t *file;
	line_t __handle *line;
	ulong lnr, editLnr;
    } getBlock;
#endif

/******* prototypes *******/
#if OS2
VOID APIENTRY PanicExit(ushort termCode );
#endif


static void CleanUp( void *dummy );

static int GetMoreLines( int fhd );
static size_t ReadLenPrefixed( char *buffer, FILE *st, int *err);
static size_t ReadBinary( char *buffer, FILE *st, ushort reclen );

static char *MakeFileName( const char *name );
static int IsInternalName( const char *name );

static void Append2Unnamed( int fhd, char *data, ushort dataLen );

static ushort GetLine( int fileHandle, char *buffer, ushort bufferLen );
static int PutLine( int fileHandle, char *buffer, ushort bufferLen );
#ifdef __GNUC__
static int SortLinesCompareA( const void *a, const void *b );
static int SortLinesCompareB( const void *a, const void *b );
static int SortLinesCompareDir( const void *a, const void *b );
#else
static int SortLinesCompareA( line_t __handle **a, line_t __handle **b );
static int SortLinesCompareB( line_t __handle **a, line_t __handle **b );
static int SortLinesCompareDir( line_t __handle **a, line_t __handle **b );
#endif
#if __IBMC__
   void     _Optlink qsort( void *, size_t, size_t,
			    int ( * _Optlink __compare )( const void *, const void * ) );
#endif

/******* functions ********/

/* NetWare Requester in der OS/2 DOS-BOX veraendert
 * in einigen Versionen den bei sopen() uebergebenen
 * filename; deswegen wird er hier wieder zurueckgesetzt
 */
#define FixSlash(a) do { char *s; for(s=(a); *s; s++ ) \
				      if( *s == '\\' ) \
					   *s = '/';   \
		    } while(0)

#if OS2
VOID APIENTRY PanicExit(ushort termCode )
{
    int mode;

    mode = 1; /* direct called by DOSExitList */
    if( termCode != TC_EXIT )
	CleanUp( &mode );
    CleanupPMClipBoard();
    DosExitList(EXLST_EXIT,0);
}
#endif

static void
CleanUp( void *parm )
{
    int i, mode = *(int*)parm;
    file_t *file;
    line_t __handle *line;
    FILE *st;

    st = NULL ;
    for( i=0; i < MAX_FILES ; i++ )
	if( (file=fileTbl+i)->name ) {
	  #if W_FULL
	    if( file->lock && !file->ro && mode != 1 )
		ResetFileLock( file->name );
	  #endif
	    if( file->intern || !file->chg )
		continue;
	  #if W_FULL
	    if( !st ) {
		puts("\n\n\n\nDumping changed files to 'w__panic.dmp'");
		st = fopen("w__panic.dmp", "a" );
		fprintf( st, "[START W PANIC DUMP]\n" );
	    }
	    if( !st )
		break;
	    fprintf( st, "[BEGIN W PANIC DUMP of '%s']\n", file->name );
	    for( line = file->first; line; line = line->nxt ) {
		if( fwrite( line->d, line->x.len, 1, st ) != 1 )
		    break;
		fputc('\n', st );
	    }
	    fprintf( st, "[END W PANIC DUMP of '%s']\n", file->name );
	  #endif
	}
    if( st )
	fclose(st);

    if( mode != 1 )
	for( i=0; i < MAX_FILES ; i++ )
	    ReleaseFile(i);
}

/****************
 * Look at the last line of the file for a pattern
 *  "*-*wedit:abc*-*"
 * where abc is the value we are interested in. If we found
 * such a value and if we have a variable "ZPrf_abc" then execute
 * the contents of this variable.
 * If this fails, check whether this file would be executed by a
 * command interpreter and return the "ZPrfSh".  This function can
 * then later try to figure out the command interpreter and set some
 * highlighting
 */
static char *
GetProfileName( int fhd )
{
    const char *line, *s;
    ushort nbytes;
    char *buf;

    SeekLine( fhd, ULONG_MAX );
    line = GetPtr2Line( fhd, &nbytes );
    if( nbytes < 13 )
	goto topline_check;
    s = MemStr( line, "*-*wedit:", nbytes );
    if( !s )
	goto topline_check;
    nbytes -= s - line;
    nbytes -= 9;
    line = s + 9;
    for(s=line; nbytes; s++, nbytes-- ) {
	if( isspace(*s) )
	    goto topline_check;
	if( *s == '*' )
	    break;
    }
    if( s==line || nbytes < 3 || !( s[1] == '-' && s[2] == '*') )
	return NULL;
    nbytes = s - line;
    buf = xmalloc( 5 + nbytes + 1 );
    strcpy( buf, "ZPrf_" );
    memcpy( buf+5, line, nbytes );
    buf[5+nbytes] = 0;
    return buf;

  topline_check:
    SeekLine( fhd, 0 );
    line = GetPtr2Line( fhd, &nbytes );
    if( nbytes > 4 && (!strncmp( line, "#!/",3) || !strncmp( line, "# !/",4)))
	return xstrdup( "ZPrfSh" );
    return NULL;
}

/****************
 * Eine File allokieren, die internen Strukturen bereitstellen und
 * einlesen.
 *  modeFlag 1 = so wird ScrBind durchgefuehrt und
 *		 die erste Seite waehrend des Aufbaus angezeigt.
 *	     2 = kein Bind und fehler wenn file nicht existiert
 *	     3 = kein Bind und fehler wenn file existiert
 * Falls bit 4 gesetzt ist, so werden tabs nicht expandiert
 * startline gibt die Zeile an, ab der gelesen werden soll
 * 0 = von anfang an, 1 = auch von anfang an, 2 ab Zeile 2 ...
 * maxlines gibt die max. anzahl von zu lesenden Zeilen an;
 *   0 = alle
 * Falls Bit 5 gesetzt ist, so werden die Zeilen mit einer iineId
 * versehen.
 * Bit 6 kennzeichnet viewmode
 *
 * Returns: ErrorCode und internen Handle
 * Falls der ErrorCode ERR_FINUSE ist, so gibt *fhd den
 *  Filehandle dieses Files zurueck.
 */

int AllocFile( int *retfhd, const char *name, int modeFlag,
	       int format, ushort fixRecLen ,
	       ulong startline, ulong maxlines )
{
    static int cleanupmode = 0;
    static int firstCall=0;
    int err, i, internFlag, unnamedFlag, set2ro, binOpn, notabexp;
    int makeLineIds, viewMode;
    char *fullName, *buffer;
    size_t nbytes, helppos;
    ulong lnr=0;
    int sx, sy, shwFlg=0, scrfhd, fhd, failOnNew, failOnExist;
    file_t *file;
    unsigned readOptions;
    char fileNameStr[35];

    err=0;
    scrfhd=0; /* avoid compiler warning; is controlled by <mode> */
    fhd = 0;  /* avoid warning; controlled by ?? */
    if( !firstCall ) {
	firstCall++;
	for(i=0; i < LABELTBL_SIZE; i++ )
	    labelTbl[i].fhd = -1;
	for(i=0; i < MAX_FILES; i++ )
	    fileRing[i] = -1;
      #if !__IBMC__
	AddCleanUp( CleanUp, &cleanupmode );
      #endif
      #if OS2
	DosExitList(EXLST_ADD | 0x9f00, (PFNEXITLIST)PanicExit );
      #endif			/* !-- one below those used by PM */
    }

    internFlag = unnamedFlag = failOnNew = failOnExist = 0;
    notabexp = modeFlag & (1<<4);
    makeLineIds = modeFlag & (1<<5);
    if( viewMode = modeFlag & (1 << 6) ) {
	makeLineIds = 0;
	failOnNew = 1;
    }

    modeFlag &= 0x0f;
    if( modeFlag == 2 ) {
	failOnNew = 1;
	modeFlag = 0;
    }
    else if( modeFlag == 3 ) {
	failOnExist = 1;
	modeFlag = 0;
    }

    if( (i=IsInternalName( name )) != -1 ) {
	internFlag++;
	if( !i )
	    unnamedFlag++;
    }

    if( fixRecLen > MAX_LINELEN ||
	(  format != WFILE_FMT_STD &&
	   format != WFILE_FMT_FIX &&
	   format != WFILE_FMT_BIN &&
	   format != WFILE_FMT_2PR )  )
	err = ERR_USPFFMT;
    else if( (format == WFILE_FMT_FIX || format == WFILE_FMT_BIN) &&
	     (fixRecLen < 1 || fixRecLen > MAX_LINELEN) )
	err = ERR_INVRLEN;


    if( !err ) {
	fullName = NULL;
	fullName = MakeFileName( name );
	/* hier testen ob schon im Editor vorhanden */
	if( (i = GetFileHandle( name )) != -1 ) {
	    *retfhd = fhd = i; /* return its filehandle */
	    err = ERR_FINUSE;
	}

	if( err )
	    free( fullName );
	else if( !internFlag && access( fullName, R_OK ) ) {
	    if( !access( fullName, F_OK ) ) { /* but only if file exists */
		free( fullName );
		err = ERR_READPER;
	    }
	}
	if( !err ) {
	    /* get an empty fileslot */
	    for(file=NULL,i=0; i < MAX_FILES-(unnamedFlag? 0 : 1); i++ )
		if( !fileTbl[i].name ) {
		    file = fileTbl + i;
		    *retfhd = fhd = i;
		    break;
		}
	    if( !file ) {
		err = ERR_FILEHD;
		free( fullName );
	    }
	}
    }

    if( !err ) {
	char *s;

	binOpn = GetSetOption( SETOPT_READBINARY ) || format != WFILE_FMT_STD;
	if( file->intern = internFlag )
	    file->stream = NULL;
	else {
	    s = fsopen2( fullName, binOpn ? "fb" : "ft" , &file->stream);
	    if( s ) {
		free(fullName);
		fullName = s;
	    }
	    else if( !access( fullName, F_OK ) )
		err = ERR_OPNFIL;
	    FixSlash(fullName);
	}
	file->name = fullName;
	Filename2Str( fullName, fileNameStr, DIM(fileNameStr) );
	file->new = file->stream ? 0 : 1;
	file->chg = 0;
	file->ring = 1;
	file->lock = 0;
	file->ro = 0;
	file->view = 0;
	file->format = format;
	file->fixRecLen = fixRecLen;
	set2ro = file->stream? access( fullName, W_OK ) : 0;
      #ifdef UNIX
	file->statbuf_ok = file->stream
			   && !fstat( fileno(file->stream), &file->statbuf);
      #endif
	FixSlash(fullName);

	if( viewMode ) {
	    file->view = 1;
	    maxlines = VIEWLINES*2;
	    set2ro = 1;
	}
	if( startline ) { /* fix offset */
	    startline--;
	    set2ro = 1;   /* set to readonly to prevent inadvertently save */
	}
	if( maxlines )
	    set2ro = 1;   /* set to readonly to prevent inadvertently save */


	file->cmdline = buffer = xmalloc( MAX_LINELEN+1 );
	file->editBuf = xmalloc( MAX_LINELEN+1 );
	file->editBufValid=0;
	file->insertMode = 1;
	file->inCmdFlag = 1;	/* Cursor will be in CommandLine */
	file->cmdWin = 0;	/* At first Position */
	file->cmdPos = 0;	/* At first Position */
	file->cmdCrs = 0;	/* At first Position */
	file->crsX = 0;
	file->crsY = 0;
	file->winX = 0;
	file->winY = 0;
	file->tabcnt = 0;	/* es sind noch keien Tabs gesetzt */
	file->noTab = 0;	/* default is: save with tabcompressing */
	file->maxLen = 0;
	file->nlines = 0;

	file->first = NULL;
	file->crs = NULL;
	file->hilite_crs = NULL;
	file->lnr = 0;
	file->pos = 0;
	file->profile_var = NULL;
	file->hiliteFlag = 0;
	*file->hiliteType = 0;
	file->nattrib = 1;  /* for the syntax highlighting */
	if( makeLineIds ) {
	    file->lineIdOff = 1;
	    file->nattrib += sizeof(ushort);
	    file->lineIdRover = 1;
	}
	else {
	    file->lineIdOff = 0;
	    file->lineIdRover = 0;
	}

	if( modeFlag ) {
	    scrfhd = QryScreenFile();
	    BindScreen( fhd );
	    GetScreenSize( &sx, &sy );
	    shwFlg = 0;
	}

	if( err )
	    ;
	else if( file->new || internFlag ) {
	    if( failOnNew )
		err = ERR_NOFILE;
	    else {
		err = InsertLine( fhd, "", 0 );
		lnr = 1;
	    }
	}
	else if( failOnExist )
	    err = ERR_FILEXIST;
	else {
	    readOptions = F_NFILE_LF;
	    if( GetSetOption( SETOPT_TABEXPAND ) && !notabexp )
		readOptions |= F_NFILE_TAB;
	    if( binOpn )
		readOptions |= F_NFILE_SKIP_CR;
	    helppos = 0;
	    lnr = 0;
	    if( viewMode ) {
		file->vm.readOptions = readOptions;
		file->vm.format = format;
		file->vm.fixRecLen = fixRecLen;
		file->vm.offset = 0;
	    }
	    while( nbytes = format == WFILE_FMT_STD ?
				FExtRead( buffer, MAX_LINELEN, file->stream,
				      readOptions, 0,
					   &helppos ) :
			    format == WFILE_FMT_2PR ?
				ReadLenPrefixed( buffer, file->stream, &i ) :
				ReadBinary( buffer, file->stream,fixRecLen )
		 ) {
		if( sigIntPending ) {
		    err = ERR_CMDINT;
		    break;
		}
		if( startline ) {
		    startline--;
		    continue;
		}
		if( err = InsertLine( fhd, buffer, nbytes-1 ) )
		    break;
		if( !(++lnr % 400) || lnr == 20 ) {
		    ShowMessageAsInfo("Reading %s (%lu) ...",
				       fileNameStr, lnr );
		    SigIntPoll(); /* do a poll, may be DOS-Break is off */
		}
		if( modeFlag && !shwFlg ) {
		    if( (int)lnr == sy ) {
			ulong actLine;

			actLine = file->lnr;
			SeekLine( fhd,	0 );
			file->chg = 0;
			*buffer = 0; /* clear buffer, cause it's the cmdline */
			UpdateScreen();
			shwFlg++;
			SeekLine( fhd, actLine );
		    }
		}
		if( maxlines )
		    if( lnr >= maxlines )
			break;
	    }
	    if( format != WFILE_FMT_2PR ) {
		if( ferror(file->stream) )
		    err = ERR_REDFIL;
	    } else {
		if( i )
		    err = ERR_REDFIL;
	    }
	    if( err || !file->view ) {
		fclose(file->stream);
		file->stream = NULL;
	    }
	    ShowMessage( NULL );
	}
	*buffer = 0; /* clear cmdline */

	if( err ) { /* err may be produced by InsertLine ( ERR_NOMEM ) */
	    if( modeFlag )
		BindScreen( scrfhd );
	    ReleaseFile( fhd );
	}
	else {
	    if( !lnr )	/* file was of size 0 */
		InsertLine( fhd, "", 0 );
	    file->chg = 0;
	    file->ro  = set2ro;
	    file->lineIdRover = 0;
	    file->profile_var = GetProfileName( fhd );
	    SeekLine( fhd,  0 );
	    PosLine( fhd, 0 );
	  #if W_FULL
	    if( !file->view ) {
		if( !file->intern && GetSetOption( SETOPT_AUTOLOCK ) )
		    SetFileFlag( fhd, WFILE_LOCK ); /*try to lock the file*/
		if( GetSetOption( SETOPT_LOCKCHECK ) )
		    if( CheckFileLock( file->name ) )	/* locked: */
			SetFileFlag( fhd, WFILE_LOCK ); /* force setting of inUse*/
	    }
	  #endif

	}
    }

    return err;
}



/****************
 * Used in view mode to fetch lines.
 */

static int GetMoreLines( int fhd )
{
    int err = 0, count, i;
    size_t nbytes, helppos=0;
    file_t *file;

    if( !viewBuffer )
	viewBuffer = xmalloc( MAX_LINELEN+1 );
    file = fileTbl+fhd;
    count=0;
    file->ro = 0;
    while( nbytes = file->vm.format == WFILE_FMT_STD ?
			FExtRead( viewBuffer, MAX_LINELEN, file->stream,
			      file->vm.readOptions, 0,
				   &helppos ) :
		    file->vm.format == WFILE_FMT_2PR ?
			ReadLenPrefixed( viewBuffer, file->stream, &i ) :
			ReadBinary( viewBuffer, file->stream,
					    file->vm.fixRecLen )
	 ) {
	if( sigIntPending ) {
	    err = ERR_CMDINT;
	    break;
	}
	if( err = InsertLine( fhd, viewBuffer, nbytes-1 ) )
	    break;
	if( ++count == VIEWLINES )
	    break;
    }

    if( count == VIEWLINES ) {
	line_t __handle *line, __handle *next;
	int found=0;

	for( line = file->first; line && count; line = next, count-- ) {
	    if( line == file->crs )
		found++;
	    next = line->nxt;
	  #if USE_HANDLES
	    handle_free( line );
	  #else
	    free( line );
	  #endif
	    file->nlines--;
	}
	line->prv = NULL;
	file->first = line;
	if( found )
	    file->crs = line;

    }

    file->ro = 1;

    return err;
}

/****************
 * Einen File lesen, wobei jede Zeile durch eine Laengenangabe
 * gekennzeichnet ist.
 * Returns: Anzahl der bytes exklusive Header + 1;
 *	    oder 0 falls ein Fehler oder EOF aufgetreten ist
 *	err gibt eventuellen Format fehler zurueck:
 *	    1 = invalid Header
 *	    2 = data line shorter than specified in header
 *	    3 = line too long
 *	    4 = line too short
 */

static size_t ReadLenPrefixed( char *buffer, FILE *st, int *err)
{
    ushort len;

    *err = 0;
    if( fread( &len , sizeof len , 1, st ) != 1 ) {
	if( ferror(st) )
	    *err = 1; /* invalid header */
	return 0;     /* terminate */
    }
  #if !WKLIB_BIG_ENDIAN
    len = (len << 8) | ((len >> 8) & 0xff);
  #endif
    if( len > MAX_LINELEN ) {
	*err = 3; /* line too long */
	return 0;     /* terminate */
    }
    else if( len < 2 ) {
	*err = 4; /* line too short */
	return 0;     /* terminate */
    }
    len -= 2;
    if( fread( buffer , len , 1, st ) != 1 ) {
	*err = 2;     /* line too short */
	return 0;     /* terminate */
    }
    return len+1;
}


/****************
 * Wie Fwrite aber mit dem Unterschie, dass imm er n+1 bytes zurueckgegeben
 * werden, und nur bei Fehler oder EOF 0
 */

static size_t ReadBinary( char *buffer, FILE *st, ushort reclen )
{
    size_t len;

    len = fread( buffer , 1, reclen, st );
    return len ? (len+1):0;
}



void
ExecFilesProfile( int fhd )
{
    file_t *file = fileTbl + fhd;

    if( file->profile_var )
	Cmd_ExecuteString( file->profile_var, ARGTYPE_VAR );
    else
	Cmd_ExecuteString( "ZPrf_default", ARGTYPE_VAR );
}


/****************
 * rename an active file
 * checks to see if this file already exists
 * Special Option: if "." is given for the new name
 * the newName will be the original filename, but
 * in the current directory.
 */

int RenameFile( int fhd, const char *newName )
{
    int err, internFlag;
    char *name;
    file_t *file;
    const char *s;

    /* optionprocessing */
    s = newName;
    while( *s ) {
	if( isspace( *s ) )
	    s++;
	else if( *s == '-' ) {
	    if( *++s == '-' ) {
		s++;
		while( isspace( *s ) )
		    s++;
		break; /* while loop */
	    }
	    for( ; *s && !isspace(*s); s++ )
		return ERR_INVOPT;
	}
	else
	    break;  /* while loop */
    }
    newName = s;
    /* end optionprocessing */

    err = 0;
    file = fileTbl + fhd;
  #if W_FULL
    if( file->view )
	return ERR_VIEW;
    if( *newName == '.' && !newName[1] ) {
	char *p;

	p = xmalloc( F_MAX_PATH );
	getcwd( p, F_MAX_PATH-1 );
	strcat( p, "/" );
	strcat( p, FileGetFName( file->name ) );
	name = MakeFileName( p );
	free(p);
    }
    else
	name = MakeFileName( newName );
  #else
    name = MakeFileName( newName );
  #endif
    internFlag = 0;
    if( IsInternalName( name ) != -1 )
	internFlag++;
    if( FileCmpName( file->name , name ) ) /* same file ? */
	if( GetFileHandle( name ) != -1 ) /* or alrady in use */
	    err = ERR_FINUSE;
    if( err )
	free(name);
    else {
      #if W_FULL
	if( file->lock && !file->ro )
	    ResetFileLock( file->name );
      #endif
	free(file->name);
	file->name = name;
	file->lock = 0;
	file->intern = internFlag;
	file->chg = 1; /* name change implies file changed */
	file->new = internFlag? 0 : access( name, F_OK );
	file->ro =  internFlag || file->new ? 0 : access( name, W_OK );
	FixSlash(name);
      #if W_FULL
	if( !file->intern && GetSetOption( SETOPT_AUTOLOCK ) )
	    SetFileFlag( fhd, WFILE_LOCK ); /*try to lock the file*/
	if( GetSetOption( SETOPT_LOCKCHECK ) )
	    if( CheckFileLock( file->name ) )	/* locked: */
		SetFileFlag( fhd, WFILE_LOCK ); /* force setting of inUse*/
      #endif
    }

    return err;
}



/****************
 * Eine file wieder freigeben.
 * Returns: ErrorCode
 */

void ReleaseFile( int fHd )
{
    file_t *file;
    line_t __handle *line, __handle *next;
    int i;

    file = fileTbl + fHd;
    if( file->name ) {
      #if W_FULL
	if( file->lock && !file->ro )	 /* delete the lock file */
	    ResetFileLock( file->name );
      #endif
	if( mark.type && mark.hd == fHd )
	    mark.type = 0;
	if( fHd == unnamedFile )
	    unnamedFile = -1;
	/* invalidate all entries in the mark stack */
	for(i=0; i < markStackInd; i++ )
	    if( markStack[i].type && markStack[i].hd == fHd )
		markStack[i].type = 0;
	/* remove all labels from this file */
	for(i=0; i < LABELTBL_SIZE; i++ )
	    if( labelTbl[i].fhd == fHd )
		labelTbl[i].fhd = -1;

	for( line = file->first; line; line = next ) {
	    next = line->nxt;
	  #if USE_HANDLES
	    handle_free(line);
	  #else
	    free(line);
	  #endif
	}
	file->first = NULL;

	if( file->stream )
	    fclose( file->stream );
	FREE( file->name );
	FREE( file->cmdline );
	FREE( file->editBuf );
	FREE( file->profile_var );
    }
}

void ReleaseAllFiles()
{
    int i;

    for(i=0; i < MAX_FILES; i++ )
	if( fileTbl[i].name )
	    ReleaseFile( i );
}



/****************
 * This Functions can be called to assure that the linked lists
 * are okay;
 */

int CheckFileBuffers()
{
    int i;
    file_t *file;
    line_t __handle *line, *last;
    char name[35];

    for( i=0; i < MAX_FILES ; i++ )
	if( (file=fileTbl+i)->name ) {
	    Filename2Str( file->name, name, DIM(name) );
	    ShowMessageAsInfo("Checking buffer '%s'", name );
	    for( last = line = file->first; line; line = line->nxt ) {
		if( last != line->prv )
		    if( line != last ) { /* line->prv at first line is NULL */
			ShowMessage("Buffer '%s' corrupted !", name );
			return ERR_UNKNOWN ;
		    }
		last = line;
	    }
	}
    ShowMessageAsInfo("Data structures are consistent" );
    return 0;
}




/****************
 * Get the Handle of the File with name.
 * Returns: filehandle or -1 if file not loaded
 */

int GetFileHandle( const char *name )
{
    int fhd, i;
    char *fullName;

    fullName = MakeFileName( name );
    fhd = -1;
    for(i=0; i < MAX_FILES; i++ )
	if( fileTbl[i].name )
	    if( !FileCmpName( fileTbl[i].name, fullName ) ) {
		fhd = i;
		break;
	    }
    free( fullName );
    return fhd;
}



/****************
 * Diese Funktion gibt den naechsten belegten filehandle zurueck,
 * falls nur ein File im Editor ist, wird dies der uebergebene
 * filehandle sein.
 * returns: filehandle
 */

int GetNextFileHandle( int fhd )
{
    int i;

    xassert( fhd >= 0 );
    for(i=fhd-1; i >= 0 ; i--)
	if( fileTbl[i].name )
	    return i;
    for(i=MAX_FILES-1; i >= fhd; i-- )
	if( fileTbl[i].name )
	    return i;
    BUG(); return 0; /* NEVER REACHED */
}


/****************
 * Diese Funktion gibt den vorigen belegten filehandle zurueck,
 * falls nur ein File im Editor ist, wird dies der uebergebene
 * filehandle sein.
 * returns: filehandle
 */

int GetPrevFileHandle( int fhd )
{
    int i;

    xassert( fhd >= 0 );
    for(i=fhd+1; i < MAX_FILES; i++ )
	if( fileTbl[i].name )
	    return i;
    for( i=0; i <= fhd; i++ )
	if( fileTbl[i].name )
	    return i;
    BUG(); return 0; /* NEVER REACHED */
}



void UpdateFileRing( int fhd )
{
  #if 0
    int i, j, f0;

    if( (f0 = fileRing[0]) == fhd )
	return ; /* ist bereits an erster Stelle */
    /* den ersten mal nach ganz hinten (damit nicht immer die gleichen kommen)*/
    /* file suchen */
    j = MAX_FILES - 1;
    for(i=0; i < MAX_FILES; i++ )
	if( fileRing[i] == fhd )
	    j = i;
	    break;
	}
    for(j=i; j > 0; j-- )
	fileRing[j] = fileRing[j-1];
    fileRing[0] = fhd;
  #endif
}



const char *GetFileInfo( int fHd, unsigned *flags )
{
    int format;
    ushort reclen;

    return GetFileInfo2( fHd, flags, &format, &reclen );
}


const char *GetFileInfo2( int fHd, unsigned *flags,
			  int *format, ushort *reclen )
{
    unsigned flg;
    file_t *file;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;

    flg = 0;
    if( file->new )
	flg |= WFILE_NEW;
    if( file->chg )
	flg |= WFILE_CHG;
    if( file->ro  )
	flg |= WFILE_RO;
    if( file->noTab )
	flg |= WFILE_NOTAB;
    if( file->intern )
	flg |= WFILE_INT;
    if( file->insertMode )
	flg |= WFILE_INSERT;
    if( file->inCmdFlag )
	flg |= WFILE_INCMD;
    if( file->ring )
	flg |= WFILE_RING;
    if( file->lock )
	flg |= WFILE_LOCK;
    if( file->view )
	flg |= WFILE_VIEW;
    if( file->hiliteFlag )
	flg |= WFILE_HILITE;

    *flags = flg;
    *format = file->format;
    *reclen = file->fixRecLen;
    return file->name;
}

/****************
 * Returns: NULL := Hiliting not enabled
 *	    string := with the type of highlighting, where an empty string
 *		      stands for default highlighting.
 */
const char *
GetFilesHiliteType( int fHd )
{
    file_t *file;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;

    if( !file->hiliteFlag )
	return NULL;
    return file->hiliteType;
}

/****************
 * Set the type of highlighting. This does not enable hiliting.
 */
void
SetFilesHiliteType( int fHd, const char *newtype )
{
    file_t *file;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;
    memcpy(file->hiliteType, newtype, DIM(file->hiliteType));
}


/****************
 * Returns the extension of the file
 */

const char *GetFilesExt( int fHd )
{
    file_t *file;
    const char *p;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;

    p = file->name + strlen( file->name ) - FileExtCheck( file->name );

    return  *p == '.' ? (p+1) : p;
}




/****************
 * Die Tab settings des aktuellen Files setzen.
 * Die Tabs werden aus dem uebergebene Array kopiert.
 * Die Tabwerte muessen aufsteigend sortiert sein, 0 ist nicht zulaessig,
 * da dort ein implizierter tabstop vorhanden ist.
 * es werden nur die gueltigen Tabs uebernommen, ab dem ersten ungueltigen wird
 * der rest uebergangen.
 */

void SetFilesTabSettings( int fHd, ushort *tabArray, ushort cnt )
{
    file_t *file;
    ushort last, i;

    file = fileTbl + fHd;
    if( cnt >= MAX_TABS )
	cnt = MAX_TABS;
    for(last=i=0; i < cnt; i++ )
	if( tabArray[i] > last && tabArray[i] < MAX_LINELEN)
	    last = file->tabs[i] = tabArray[i];
	else
	    break;  /* not in ascending order - skip the rest */
    file->tabcnt = i;
}


/****************
 * Die Tab settings des aktuellen Files holen:
 * Die Tabs werden in das uebergebene Array kopiert, max. aber arrSize tabs
 * Returns: Anzahl der gueltigen Eintraege im Array.
 */

ushort GetFilesTabSettings( int fHd, ushort *tabArray, ushort arrSize )
{
    file_t *file;
    ushort tabcnt;

    file = fileTbl + fHd;
    tabcnt = file->tabcnt;
    if( arrSize < tabcnt )
	tabcnt = arrSize;

    memcpy( tabArray, file->tabs, tabcnt* sizeof( ushort ) );
    return tabcnt;
}



void SetFilesMargins( int fHd, ushort left, ushort right, ushort indent )
{
    file_t *file;

    file = fileTbl + fHd;
    file->lmargin = left;
    file->rmargin = right;
    file->dmargin = indent;
}


void GetFilesMargins( int fHd, ushort *left, ushort *right, ushort *indent )
{
    file_t *file;

    file = fileTbl + fHd;
    *left   = file->lmargin;
    *right  = file->rmargin;
    *indent = file->dmargin;
}


#ifdef UNIX
struct stat *
GetFileStat( int fHd )
{
    file_t *file;

    file = fileTbl + fHd;
    return file->statbuf_ok ? &file->statbuf : NULL;
}
#endif


void
SetFileFlag( int fHd, int flag )
{
    file_t *file;

    file = fileTbl + fHd;
    switch( flag ) {
      case WFILE_CHG:
	if( !file->chg ) {
	    file->chg = 1;
	    UpdateWindowTitle(fHd);
	}
	else
	    file->chg = 1;
	break;
      case WFILE_INSERT:file->insertMode = 1; break;
      case WFILE_INCMD: file->inCmdFlag  = 1; break;
      case WFILE_NOTAB: file->noTab = 1; break;
      case WFILE_HILITE:file->hiliteFlag= 1; break;
      case WFILE_RING:	file->ring = 1; break;
      case WFILE_INT:	file->intern = 1; break;
      case WFILE_RO:
	if( !file->lock )
	    file->ro = 1;
	break;
      case WFILE_NEW:	file->new = 1; break;
      case WFILE_LOCK:
       #if W_FULL
	if( file->intern )
	    ;  /* no need to lock this files */
	else if( !file->lock ) { /* do it only when not locked */
	    ResetFileFlag( fHd, WFILE_RO );
	    if( !file->ro ) { /* okay, try to lock */
		if( SetFileLock( file->name ) )
		    file->ro = 1; /* in use, so: set readonly and lock */
		file->lock = 1;
	    }
	}
       #endif
	break;
    }
}


void ResetFileFlag( int fHd, int flag )
{
    file_t *file;

    file = fileTbl + fHd;
    switch( flag ) {
      case WFILE_INSERT:file->insertMode = 0; break;
      case WFILE_INCMD: file->inCmdFlag  = 0; break;
      case WFILE_CHG:
	if( file->chg ) {
	    file->chg = 0;
	    UpdateWindowTitle(fHd);
	}
	else
	    file->chg = 0;
	break;
      case WFILE_NOTAB: file->noTab = 0; break;
      case WFILE_HILITE: file->hiliteFlag = 0; break;
      case WFILE_RING:	file->ring = 0; break;
      case WFILE_NEW:	file->new = 0; break;
      case WFILE_RO:
	if( file->view )
	    ; /* can't change */
	else if( file->lock )
	    ;  /* must be changed via Reset Lock Flag */
	else if( file->intern )
	    file->ro = 0;
	else if( !access( file->name, W_OK ) || !access( file->name, F_OK ) )
	    file->ro = 0;
	else if( file->new && !access( file->name, W_OK ) )
	    file->ro = 0;
	FixSlash(file->name);
	break;
      case WFILE_LOCK:
       #if W_FULL
	if( file->lock ) {
	    if( file->ro ) {
		file->lock = 0;
		ResetFileFlag( fHd, WFILE_RO );
	    }
	    else {
		ResetFileLock( file->name );
		file->lock = 0;
	    }
	}
       #endif
	break;
    }
}


void
WeditUnlockFile( int fHd )
{
    file_t *file;

    file = fileTbl + fHd;
    if( !file->intern && file->lock ) {
	ResetFileLock( file->name );
	file->lock = 0;
	ResetFileFlag( fHd, WFILE_RO );
	if( CheckFileLock( file->name ) )   /* locked: */
	    SetFileFlag( fHd, WFILE_LOCK ); /* force setting of inUse*/
	else
	    SetFileFlag( fHd, WFILE_RO );   /* it's more secure to set this */
					    /* file r/o when manually unlocked*/
    }
}



/****************
 * Return Ptr to Commandline of file with handle fHd
 * if this file is not valid ( may be just released from editor )
 * NULL is returned to indicated this situation
 */

char *GetPtr2CmdLine( int fHd, ushort *pos )
{
    xassert( fHd >= 0 && fHd < MAX_FILES );
    if( fileTbl[fHd].name ) {
       *pos = fileTbl[fHd].cmdPos;
       return  fileTbl[fHd].cmdline;
    } else
	return NULL;
}



/****************
 * Retrieve Informations about the commandline
 * Returns: Ptr to CmdLine
 */

char *GetAllCmdLineInfo( int fhd, ushort *nbytes,
			 ushort *pos, ushort *win, ushort *crs )
{
    file_t *file;

    file = fileTbl + fhd;
    *pos = file->cmdPos;
    *win = file->cmdWin;
    *crs = file->cmdCrs;
    *nbytes = strlen( file->cmdline );
    return file->cmdline;
}


void SetAllCmdLineInfo( int fhd, ushort pos, ushort win, ushort crs )
{
    file_t *file;
    ushort n;
    char *p;

    file = fileTbl + fhd;
    xassert( pos <= MAX_LINELEN );
    file->cmdPos = pos;
    file->cmdWin = win;
    file->cmdCrs = crs;
    n = strlen( p=file->cmdline );
    for(p +=n; n < pos; n++, p++ )
	*p = ' ';
    *p = '\0';
}




/****************
 * Get Position of textcursor.
 * Returns: 0;
 */

int GetTextCrsPos( int fHd, ushort *x, ushort *y )
{
    xassert( fHd >= 0 && fHd < MAX_FILES );
    *x = fileTbl[fHd].crsX;
    *y = fileTbl[fHd].crsY;
    return 0;
}




/****************
 * Fill actual line into Edit Buffer an return an pointer to this Buffer
 * nbytes, will return the valid len of the buffer
 * Falls im editbuffer noch eine  gueltige Zeile mit einer anderen zeilennr.
 * steht, so wird diese zuerst zurueckgeschrieben.
 */

char *GetEditBuf( int fHd, ushort *nbytes )
{
    file_t *file;
    ulong actLine;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;

    actLine = file->lnr;
    if( file->editBufValid && actLine != file->editBufLine )
	FlushEditBuf( fHd );

    if( !file->editBufValid ) {
	file->editBufLen = GetLine( fHd, file->editBuf, MAX_LINELEN );
	file->editBufLine = actLine;
	if( !file->ro )
	    file->editBufValid = 1;
    }

    *nbytes = file->editBufLen;
    return file->editBuf;
}


/****************
 * Die aktuelle laenge des editBuffers neu setzen
 */

void SetEditBuf( int fHd, ushort nbytes )
{
    file_t *file;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;

    xassert( nbytes <= MAX_LINELEN );
    file->editBufLen = nbytes ;
}


/****************
 * Den EditBuffer fuer ungueltig erklaeren
 */

void ClearEditBuf( int fHd )
{
    file_t *file;

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;
    file->editBufValid = 0;
}



/****************
 * Flushs the editbufffer, if there is a valid line in it
 * veraendert die aktuelle Zeile nicht.
 * Danach ist der Buffer nich mehr gueltig.
 * Sondermode: wenn fileHandle -1 ist, so werden alle geflushed
 */

void FlushEditBuf( int fHd )
{
    file_t *file;
    ulong actLine;

    if( fHd == -1 ) {
	int i;

	for(i=0; i < MAX_FILES; i++ )
	    if( fileTbl[i].name )
		FlushEditBuf(i);
	return;
    }

    xassert( fHd >= 0 && fHd < MAX_FILES );
    file = fileTbl + fHd;
    if( file->editBufValid ) {
	actLine = file->lnr;
	SeekLine( fHd, file->editBufLine );
	PutLine( fHd, file->editBuf, file->editBufLen );
	file->editBufValid = 0;
	SeekLine( fHd, actLine );
    }
}


/****************
 * Returns number of active files.
 */

int GetFileCount()
{
    int i, n;

    for( n=i=0; i < MAX_FILES ; i++ )
	if( fileTbl[i].name )
	    n++;
    if( unnamedFile != -1 && n == 1 )
	n--;
    return n;
}


/****************
 * Speicher fuer einen Filenamen allokieren, und den Filenamem mit
 * einem absoluten Path versehen. Kann mit free wieder freigegeben werden.
 * Returns: Pointer to allocated string with full filename.
 */

static char *
MakeFileName( const char *name )
{
    char *buf, *newbuf;
    size_t len;
  #if UNIX
    int nloop=0;
    struct stat sb;
  #endif

    name += strspn( name, whiteSpaces ); /* skip Blanks */
  #if OS2  /* wg. HPFS sind auch blanks im Namen moeglich */
    len = strlen( name );
  #else
    len = strcspn( name, whiteSpaces );
  #endif
    xassert( len < F_MAX_PATH );
  #if UNIX
    buf = NULL;
    if( *name == '~' && name[1] )  {
	const char *s;

	if( (s = getenv("HOME")) && *s ) {
	    buf = xmalloc(strlen(s)+len+3);
	    strcpy(buf,s);
	    mem2str(buf+strlen(buf),name+1,len);
	}
    }
    if( !buf ) {
	buf = xmalloc( F_MAX_PATH+1 );
	mem2str( buf, name, len+1 );
    }

    /* hmmm, readlink() does not follow a link chain
     * but the man pages says ELOOP is an errocode, so it
     * should do it -- Il do it myself here
     */
    while( !lstat( buf, &sb ) && S_ISLNK(sb.st_mode) ) { /* this is a symlink */
	int nchars;
	int size = strlen(buf)+10;
	char *buf2;
	if( size < 100 )
	    size = 100;

	for(;;) {
	    buf2 = xmalloc( size );
	    nchars = readlink( buf, buf2, size-1);
	    if( nchars < size )
		break;
	    free(buf2);
	    size *= 2;
	}
	if( nchars > 0 ) { /* found it */
	    char *p = buf;
	    buf = buf2;
	    free(p);
	    buf[nchars] = 0; /* terminate the string */
	    if( ++nloop > 20 ) {
		Error(0,"%s: loop in symlinks", name );
		break;
	    }
	}
	else
	    break;
    }

  #else
    buf = xmalloc( F_MAX_PATH+1 );
    mem2str( buf, name, len+1 );
  #endif

    if( IsInternalName( buf ) == -1 )
	len = strlen( FilenameMkAbs(NULL, buf) );
    newbuf = realloc( buf, len+1 );  /* shrink buffer */

    return newbuf ? newbuf : buf;
}


/****************
 * Gibt 0 zurueck, falls kewin ScratchFile, ansonsten wird die
 * Nummer des Scratchfiles zurueckgegeben
 */

int IsScratchFile( const char *name )
{
    int i;

    if( (i=IsInternalName( name )) > 100) {
	i -= 100;
	return i;
    }
    return 0;
}


int IsInternalFileName( const char *name )
{
    char *fullName;
    int i;

    fullName = MakeFileName( name );
    i = IsInternalName( fullName ) != -1 ;
    free( fullName);
    return i;
}


/****************
 * Testen, ob es sich um einen internen Namen handelt.
 * wenn das Ergebiss > 99 ist, so handelt es sich um einen scratch file
 * Returns: -1 if not internal Name
 *	    index into internal Name Table
 */

static int IsInternalName( const char *name )
{
    int i;
    static const char *names[] = {
		".unnamed" ,
		".keydefs" ,
		".dir"     ,      /* Directory listing */
		".editlist",      /* activ files in editor */
		".history" ,      /* history list */
		".work"    ,      /* temp work file */
		".varlist" ,      /* list of all known variabels*/
		".ascii"   ,      /* ascii table */
		".tree"    ,      /* directory tree */
		".clipboard",
		".info",
		NULL  };
    for(i=0; names[i]; i++ )
	if( !FileCmpName( names[i], name ) )
	    return i;
    if( *name == '.')
	if( isdigit( name[1] ) ) {
	    i = atoi(name+1);
	    if( i > 0 &&  i < 100 )
		return i + 100;
	}
    return -1;
}


/****************
 * This function handles the internal file unnamed.
 * If its doesn't exist, it will be created
 */

static void Append2Unnamed( int srcfhd, char *data, ushort dataLen )
{
  #if W_FULL
    int err=0, maxLines;

    if( !QryReservedMemoryState() ) {
	return; /* don't do so */
    }
    maxLines = GetSetOption( SETOPT_BACKUP );
    if( maxLines < 1 )
	return;
    maxLines += 2; /* need to add some */

    if( unnamedFile == -1 ) { /* Create file */
	if( err = AllocFile( &unnamedFile, ".unnamed", 0,
			     WFILE_FMT_STD, 0 , 0,0	 ) ) {
	    ShowMessage("can't create '.unnamed': %s", GetErrorString(err) );
	    unnamedFile = -1;
	    return;
	}
	ResetFileFlag( unnamedFile, WFILE_RING );
    }

    if( QryScreenFile() == unnamedFile || srcfhd == unnamedFile )
	return; /* not if active */


    if( GetFileTotLines( unnamedFile ) > maxLines ) {
	SeekLine( unnamedFile, 0 );
	DeleteLine( unnamedFile );
    }
    SeekLine( unnamedFile, ULONG_MAX );
    InsertLine( unnamedFile, data, dataLen );
  #endif
}


/***********************************************************
 **** Funktionen fuer das Textarray ************************
 ***********************************************************/



/****************
 * Eine Zeile hinter der aktuellen Position einfuegen, neu aktuelle
 * Position ist dann diese Zeile und dort das erste Byte.
 * Ist noch keine Zeile vorhanden, so wird diese als erste Zeile eingefuegt
 * Returns : ErrorCode
 */

int
InsertLine( int fileHandle, const char *data, ushort dataLen )
{
    file_t *file;
    line_t __handle *line, __handle *next, __handle *prev;
    ulong oldLnr;
    int err, i;

    file = fileTbl +fileHandle;
    if( file->ro )
	return ERR_EDTRO;
    err = 0;
    /* first allocate memory for the new line */
  #if USE_HANDLES
    line = handle_calloc( sizeof(*line)+dataLen-(dataLen?1:0)+file->nattrib);
  #else
    line = calloc( 1, sizeof(*line)+dataLen-(dataLen?1:0)+file->nattrib );
  #endif
    if( !line )
	err = ERR_NOMEM;
    else {
	line->x.len = dataLen;
	memcpy( line->d, data, dataLen );
	memset( line->d+dataLen, 0, file->nattrib );
	if( file->lineIdOff ) {
	  #if ALIGNMENT_REQUIRED
	    memcpy( line->d+dataLen+file->lineIdOff,
				    &file->lineIdRover, sizeof(ushort));
	  #else
	    *(ushort*)(line->d+dataLen+file->lineIdOff) = file->lineIdRover;
	  #endif
	    if( file->lineIdRover )
		file->lineIdRover++;
		/* ein Wrap-Around vonn ffff nach 0 macht nichts, da
		 * sich dadurch das taggen mit lineIds abschalten und weitere
		 * Zeilen nicht mehr getagged werden (ich will nicht zu viel
		 * Platz fuer diese lineIds verbrauche, 65535 Zeilen sind
		 * ja wohl ausreichend
		 */
	}
	if( !file->first ) { /* this is the first line to be inserted */
	    file->first = line;
	    file->crs = file->first;
	    file->lnr = oldLnr = 0;
	    file->hilite_crs = NULL;
	}
	else {
	    if( !file->crs ) { /* no cursor set */
		file->crs = file->first;
		file->lnr = oldLnr = 0;
		BUG();
	    }
	    prev = file->crs;
	    next = prev->nxt;
	    prev->nxt = line;
	    line->prv = prev;
	    if( next ) {
		line->nxt = next;
		next->prv = line;
	    }
	    file->crs = line;
	    oldLnr = file->lnr;
	    file->lnr++;
	    /* assume that this part is visible (better performance) */
	    file->hilite_crs = file->crs;
	    file->hilite_lnr = file->lnr;
	}
	file->pos  = 0; /* set cursor to first byte */
	file->crsX = 0;
	file->nlines++;
	if( dataLen > file->maxLen )
	    file->maxLen = dataLen;
	if( !file->view )
	    SetFileFlag( fileHandle, WFILE_CHG );

	/* update the mark */
	if( mark.type && fileHandle == mark.hd ) {
	    if( oldLnr < mark.startLine ) {
		mark.startLine++;
		mark.endLine++;
	    }
	    else if( file->lnr <= mark.endLine ) {
		mark.endLine++;
	    }
	}

	/* update the mark stack */
	for(i=0; i < markStackInd; i++ )
	    if( markStack[i].type && markStack[i].hd == fileHandle ) {
		if( oldLnr < markStack[i].startLine ) {
		    markStack[i].startLine++;
		    markStack[i].endLine++;
		}
		else if( file->lnr <= markStack[i].endLine ) {
		    markStack[i].endLine++;
		}
	    }

	/* update the labels */
	for(i=0; i < LABELTBL_SIZE; i++ )
	    if( labelTbl[i].fhd == fileHandle ) {
		if( oldLnr < labelTbl[i].lnr )
		    if( labelTbl[i].lnr )
			labelTbl[i].lnr++;
	    }

    }

    return err;
}


/****************
 * Eine Zeile loeschen
 * Position ist die aktuelle Zeile
 * Ist noch keine Zeile vorhanden, so passiert nichts
 * Returns : ErrorCode
 */

int
DeleteLine( int fileHandle )
{
    file_t *file;
    line_t __handle *line, __handle *next, __handle *prev;
    int err, i;

    file = fileTbl+fileHandle;
    if( file->ro )
	return ERR_EDTRO;
    err = 0;
    line = file->crs ;
    if( line ) {
	prev = line->prv;
	next = line->nxt;
	if( prev && next ) {
	    prev->nxt = next;
	    next->prv = prev;
	    file->crs = next;
	}
	else if( prev ) { /* last element in list */
	    prev->nxt = NULL;
	    file->crs = prev;
	    if( file->lnr )
		file->lnr--;
	}
	else if( next ) { /* first element in list */
	    file->first = next;
	    next->prv = NULL;
	    file->crs = file->first;
	    file->lnr = 0;
	    file->hilite_crs = NULL;
	}
	else {	/* only element in list */
	    file->first = NULL;
	    file->crs = NULL;
	}
	if( !file->view )
	    Append2Unnamed( fileHandle, line->d, line->x.len );
	/* assume that this part is visible (better performance) */
	file->hilite_crs = file->crs;
	file->hilite_lnr = file->lnr;

	file->pos = 0;
	xassert( file->nlines );
	file->nlines--;
      #if USE_HANDLES
	handle_free( line );
      #else
	free( line );
      #endif
	if( !file->view )
	    SetFileFlag( fileHandle, WFILE_CHG );

	/* update the mark */
	if( mark.type && fileHandle == mark.hd ) {
	    if( file->lnr < mark.startLine ) {
		if( mark.startLine ) {
		    mark.startLine--;
		    mark.endLine--;
		}
	    }
	    else if( file->lnr <= mark.endLine ) {
		if( mark.startLine == mark.endLine )
		    mark.type = 0;  /* delete mark */
		else
		    mark.endLine--;
	    }
	}

	/* update the mark stack */
	for(i=0; i < markStackInd; i++ )
	    if( markStack[i].type && markStack[i].hd == fileHandle ) {
		if( file->lnr < markStack[i].startLine ) {
		    if( markStack[i].startLine ) {
			markStack[i].startLine--;
			markStack[i].endLine--;
		    }
		}
		else if( file->lnr <= markStack[i].endLine ) {
		    if( markStack[i].startLine == markStack[i].endLine )
			markStack[i].type = 0;	/* delete mark */
		    else
			markStack[i].endLine--;
		}
	    }

	/* update the labels */
	for(i=0; i < LABELTBL_SIZE; i++ )
	    if( labelTbl[i].fhd == fileHandle ) {
		if( file->lnr < labelTbl[i].lnr )
		    if( labelTbl[i].lnr )
			labelTbl[i].lnr--;
	    }
    }
    return err;
}



/****************
 * Alle Zeilen des Files lschen
 * und eine Leere wieder einfgen.
 */

int DeleteAllLines( int fhd )
{
    int err=0,i;
    file_t *file;
    line_t __handle *line, __handle *next;

    err = 0;
    file = fileTbl+fhd;
    if( file->ro )
	return ERR_EDTRO;
    for( line = file->first; line; line = next ) {
	next = line->nxt;
      #if USE_HANDLES
	handle_free( line );
      #else
	free( line );
      #endif
    }
    file->first = NULL;
    file->lnr = 0;
    file->hilite_crs = NULL;
    file->maxLen = 0;
    file->nlines = 0;
    InsertLine( fhd, "", 0 );
    if( mark.type && mark.hd == fhd )
	mark.type = 0;
    /* invalidate all entries in the mark stack */
    for(i=0; i < markStackInd; i++ )
	if( markStack[i].type && markStack[i].hd == fhd )
	    markStack[i].type = 0;
    /* remove all labels from this file */
    for(i=0; i < LABELTBL_SIZE; i++ )
	if( labelTbl[i].fhd == fhd )
	    labelTbl[i].fhd = -1;
    return err;
}


/****************
 * SeekLine() - Cursor auf bestimmen Zeile positionieren
 * 0 = Vor die erste Zeile positionieren
 * ULONG_MAX = auf letze Ziele positionieren.
 * Ist die Zeilenicht vrhanden, so wird auf die letzte Zeile
 * positioniert.
 * Returns:  0 : Okay
 *	     1 : letzte Zeile erreicht
 */

int
SeekLine( int fileHandle, ulong newLnr )
{
  #if VIRTUAL
    line_t __handle *line;
    ulong lnr;
    file_t *file;
    virtual_t v;

    file = fileTbl+fileHandle;

    if( !file->crs ) {
	file->crs = file->first;
	file->lnr = 0;
    }

    if( file->first ) {  /* work only if lines allocated */
	line = file->crs;
	lnr = file->lnr;
	if( lnr < newLnr ) { /* go forward */
	    while( lnr < newLnr && line->nxt ) {
		line = line->nxt;
		if( line->x.virt ) {
		    memcpy( &v, line->d, sizeof v );
		    if( lnr + v.nlines < newLnr )
			lnr += v.nlines;
		    else {  /* the target line is swapped */
			SwapIn( line );
		    }
		}
		else
		    lnr++;
	    }
	}
	else {		/* go back */
	    while( lnr > newLnr && line->prv ) {
		lnr--;
		line = line->prv;
	    }
	}
	file->crs = line;
	file->lnr = lnr;
	return line->nxt ? 0 : 1 ;
    }
    else
	return 0;
  #else  /* not virtual */
    line_t __handle *line;
    ulong lnr;
    file_t *file;

    file = fileTbl+fileHandle;

    if( !file->crs ) {
	file->crs = file->first;
	file->lnr = 0;
    }

    if( file->first ) {  /* work only if lines allocated */
	line = file->crs;
	lnr = file->lnr;
	if( lnr < newLnr ) { /* go forward */
	    while( lnr < newLnr && line->nxt ) {
		lnr++;
		line = line->nxt;
	    }
	    if( file->view && !line->nxt ) {
		file->crs = line;
		file->lnr = lnr;
		GetMoreLines(fileHandle);
	    }
	}
	else {		/* go back */
	    while( lnr > newLnr && line->prv ) {
		lnr--;
		line = line->prv;
	    }
	    if( file->view ) {
		    ; /* reload............!!!!!!! */
	    }
	}
	file->crs = line;
	file->lnr = lnr;
	return line->nxt ? 0 : 1 ;
    }
    else
	return 0;
  #endif
}



/****************
 * Die Position innerhalb der aktuellen Zeile setzen
 */

void PosLine( int fileHandle, ushort newPos )
{
    file_t *file;

    file = fileTbl+fileHandle;
    file->pos = newPos;
}


void SeekAndPosLine( int fileHandle, ushort newPos, ulong newLine )
{
    file_t *file;

    file = fileTbl+fileHandle;
    file->pos = newPos;
    SeekLine( fileHandle, newLine );
}


/****************
 * GetLine() - Get a line and copy it into a buffer for editing
 * If the line is too long, only the first bytes will be copied
 * If there are no lines, function will return 0;
 * Returns: number of valid bytes in line
 */

static ushort GetLine( int fileHandle, char *buffer, ushort bufferLen )
{
    ushort nbytes;
    line_t __handle *line;
    file_t *file;

    file = fileTbl+fileHandle;

    nbytes = 0;
    if( file->first ) {
	if( !file->crs )
	    file->crs = file->first;
	line = file->crs;
	if( bufferLen < line->x.len )
	    nbytes = bufferLen;
	else
	    nbytes = line->x.len;
	memcpy( buffer, line->d, nbytes );
    }
    return nbytes;
}


/****************
 * PutLine() - Set line as data into current line
 * Returns: ErrorCode
 */

static int
PutLine( int fileHandle, char *buffer, ushort bufferLen )
{
    int err;
    line_t __handle *line, __handle *newline;

    file_t *file;

    file = fileTbl+fileHandle;
    if( file->ro )
	return ERR_EDTRO;

    err = 0;
    if( !(line = file->crs) )
	err = InsertLine( fileHandle, buffer, bufferLen );
    else if( bufferLen == line->x.len ) { /* very simple in this case */
	Append2Unnamed( fileHandle, line->d, line->x.len );
	if( memcmp( line->d, buffer, bufferLen ) )
	    SetFileFlag( fileHandle, WFILE_CHG );
	memcpy( line->d, buffer, bufferLen );
	/* lineIds are retained without any additional code */
	/* but we have to reset the hilite-state: */
	((byte*)line->d)[line->x.len] = 0;
    }
    else {
	Append2Unnamed( fileHandle, line->d, line->x.len );
      #if USE_HANDLES
	newline = handle_calloc( sizeof(*newline)
				 + bufferLen-(bufferLen?1:0)+file->nattrib);
      #else
	newline = calloc( 1, sizeof(*newline)
			     + bufferLen-(bufferLen?1:0)+file->nattrib);
      #endif
	if( !newline )
	    err = ERR_NOMEM;
	else {
	    if( newline->prv = line->prv )
		newline->prv->nxt = newline;
	    if( newline->nxt = line->nxt )
		newline->nxt->prv = newline;
	    newline->x.len = bufferLen;
	    memcpy( newline->d, buffer, bufferLen );
	    if( file->nattrib )
		memcpy(newline->d+bufferLen,line->d+line->x.len,file->nattrib);
	    ((byte*)newline->d)[newline->x.len] = 0; /* reset state */

	    if( file->first == line )
		file->first = newline;
	    if( file->crs == line )
		file->crs = newline;
	    if( file->hilite_crs == line )
		file->hilite_crs = newline;
	  #if USE_HANDLES
	    handle_free( line );
	  #else
	    free( line ); /* throw away the old line */
	  #endif
	    SetFileFlag( fileHandle, WFILE_CHG );
	}
    }
    return err;
}



/****************
 * Get current line and position, pos my be NULL
 */

ulong GetFilePos( int fileHandle, ushort *pos )
{
    file_t *file;

    file = fileTbl + fileHandle;
    if( pos )
	*pos = file->pos;
    return file->lnr;
}


ulong GetFileTotLines( int fhd )
{
    file_t *file;

    file = fileTbl + fhd;
    return file->nlines;
}



/****************
 * Get All Position Infos of the file
 * Returns: Total Number of lines
 */

ulong GetAllFilePos( int fhd, ushort *filX, ulong *filY,
			      ushort *winX, ulong *winY,
			      ushort *crsX, ushort *crsY )
{
    file_t *file;

    file = fileTbl + fhd;
    *filX = file->pos;
    *filY = file->lnr;
    *winX = file->winX;
    *winY = file->winY;
    *crsX = file->crsX;
    *crsY = file->crsY;
    return file->nlines;
}



void SetTextAreaPos( int fhd, ushort winX, ulong  winY,
			      ushort crsX, ushort crsY )
{
    file_t *file;

    file = fileTbl + fhd;
    file->winX = winX;
    file->winY = winY;
    file->crsX = crsX;
    file->crsY = crsY;
}



/****************
 * Sortieren eines Buffers
 * Es werden nur die markierten Zeilen sortiert,
 * Falls eine Blockmark vorhanden ist, so werden nur die damit
 * markierten Spalten zum Vergelich herangezogen.
 * Funktion fuehrt eine intern Flush durch !!
 * Folgende Optionen werden erkannt
 * -r  = reverse Sort
 * -R  = reverse only; skip sort
 * -n  = numerischer Vergleich ( atol() )
 * -c  = caseinsensitive sort
 * -d  = Directory Sort (name, but directories come first)
 * -dn = Directory Sort (name)
 * -de = Directory Sort (extension, name)
 * -ds = Directory Sort (size)
 * -dd = Directory Sort (date,time)
 * --  = keine weiteren Optionen
 */

int
SortLines( int fileHandle, const char *s )
{
    line_t __handle *line, *firstLine, *lastLine;
    line_t __handle **ptrArr;  /* will be allocated */
    file_t *file;
    ulong lnr=0, nlines, startline, endline;
    volatile /* due to longjmp */ size_t n;
    struct {
	int rev:1;
	int num:1;
	int dir:1;
	int nocase:1;
	int nosort:1;
    } o;
    int err = 0;

    xassert( fileHandle >= 0 && fileHandle < MAX_FILES );
    memset( &o, 0, sizeof o );
    sortMode = dALFA;
    directorySortKludge = 1;
    directorySortReverse = 0;
    while( *s ) {
	if( isspace( *s ) )
	    s++;
	else if( *s == '-' ) {
	    if( *++s == '-' ) {
		s++;
		while( isspace( *s ) )
		    s++;
		break; /* while loop */
	    }
	    for( ; *s && !isspace(*s); s++ ) {
		switch( *s ) {
		  case 'R' : o.nosort++; /* fall thru */
		  case 'r' : o.rev++; directorySortReverse = 1; break;
		  case 'c' : o.nocase++; break;
		  case 'd' : o.dir++;
		    switch( s[1] ) {
		      case 'd' : sortMode = dDATE; s++; break;
		      case 's' : sortMode = dSIZE; s++; break;
		      case 'e' : sortMode = dEXT ; s++; break;
		      case 'n' : directorySortKludge = 0; /* fall thru */
		      default  : sortMode = dNAME; s++; break;
		    }
		    break;
		  case 'n' : o.num++; sortMode = dNUM; break;
		  default  : return ERR_INVOPT;
		}
	    }
	}
	else
	    break;  /* while loop */
    }
    if( *s )
	return ERR_INVARG;

    if( !o.dir && o.nocase )
	sortMode = dALFACI;

    if( !o.dir && mark.type != MARKTYPE_LINE && mark.type != MARKTYPE_BLOCK )
	return ERR_NOMRK;
    if( !o.dir && mark.type && mark.hd != fileHandle )
	return ERR_MRKCFL;

    FlushEditBuf( fileHandle );
    file = fileTbl + fileHandle;
    if( file->ro && !file->intern ) /* sorting ro internal files is okay */
	return ERR_EDTRO;
    if( file->nlines < 2 )
	return 0;   /* da gibts dann nichts zu sortieren */
    if( mark.startLine == mark.endLine && mark.type )
	return 0;   /* da gibts dann nichts zu sortieren */
    if( mark.type ) {
	startline = mark.startLine;
	endline = mark.endLine;
    }
    else {
	startline = 0;
	endline = file->nlines-1;
    }

    if( !file->crs ) {
	file->crs = file->first;
	file->lnr = 0;
    }

    /* positionieren auf die ersten Zeile */
    if( line = file->first ) {
	line = file->crs;
	lnr = file->lnr;
	if( lnr < startline  ) { /* go forward */
	    while( lnr < startline && line->nxt ) {
		lnr++;
		line = line->nxt;
	    }
	}
	else {		/* go back */
	    while( lnr > startline && line->prv ) {
		lnr--;
		line = line->prv;
	    }
	}
    }

    /* setup an Array of ptrs to the lines */
    nlines = endline - startline + 1;
    xassert( nlines > 1 );
    if( nlines * sizeof *ptrArr > UINT_MAX - 100 )
	return ERR_MRK2LRG;

    ptrArr = calloc( nlines, sizeof *ptrArr );
    if( !ptrArr )
	return ERR_NOMEM;

    firstLine = line->prv;
    for(n=0; lnr <= endline; lnr++ ) {
	if( !line ) {
	    free( ptrArr );
	    return ERR_BUG;
	}
	ptrArr[n++] = line;
	line = line->nxt;
    }
    lastLine = line;

    if( !o.nosort )
	/* Sortieren */
	if( !(err = setjmp( sortCompareJmpBuf )) )
	    qsort( ptrArr, n, sizeof(*ptrArr),
	     #if __IBMC__
	      (int ( * _Optlink )( const void *, const void * ))
					 (
	     #endif
					       o.dir ?
						SortLinesCompareDir :
					   mark.type == MARKTYPE_LINE ?
						SortLinesCompareA :
						SortLinesCompareB
	     #if __IBMC__
		)
	     #endif
								      ) ;
    if( !err ) {
	/* linked List updaten */
	if( o.rev ) {
	    n = nlines - 1;
	    if( firstLine )
		firstLine->nxt = ptrArr[n];
	    else
		file->first = ptrArr[n];
	    ptrArr[n]->prv = firstLine;

	    for(; n ; n-- ) {
		ptrArr[n]->nxt = ptrArr[n-1];
		ptrArr[n-1]->prv = ptrArr[n];
	    }

	    if( lastLine )
		lastLine->prv = ptrArr[0];
	    ptrArr[0]->nxt = lastLine;
	} else {
	    if( firstLine )
		firstLine->nxt = ptrArr[0];
	    else
		file->first = ptrArr[0];
	    ptrArr[0]->prv = firstLine;

	    for(n=1; n < nlines; n++ ) {
		ptrArr[n-1]->nxt = ptrArr[n];
		ptrArr[n]->prv = ptrArr[n-1];
	    }

	    if( lastLine )
		lastLine->prv = ptrArr[n-1];
	    ptrArr[n-1]->nxt = lastLine;
	}

	/* resync */
	file->crs = NULL;
	SeekLine( fileHandle, file->lnr );


	/* Ready */
	SetFileFlag( fileHandle, WFILE_CHG );
    }

    free( ptrArr );

    return err;
}


/****************
 *  Vergleich bei LineMark, eigene Funktion, um einen Vergleich zu sparen
 */

static int
#ifdef __GNUC__
SortLinesCompareA( const void *xa, const void *xb )
{
    line_t __handle **a = (line_t **)xa;
    line_t __handle **b = (line_t **)xb;
#else
SortLinesCompareA( line_t __handle **a, line_t __handle **b )
{
#endif
    ushort la, lb;
    char *pa, *pb;

    if( ++sortCompareCounter > 500 ) {
	sortCompareCounter = 0;
	SigIntPoll();
	if( sigIntPending )
	    longjmp( sortCompareJmpBuf, ERR_CMDINT );
    }
    la = (*a)->x.len;
    lb = (*b)->x.len;
    pa = (*a)->d;
    pb = (*b)->d;
    if( !la && !lb )
	return 0;
    else if( sortMode == dNUM ) {
	long sa, sb;

	sa = Atoln( pa, la );
	sb = Atoln( pb, lb );
	if( sa < sb )
	    return -1;
	if( sa > sb )
	    return 1;
	return 0;
    }
    else if( sortMode == dALFACI ) {
	if( la == lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( toupper(*pa) < toupper(*pb) )
		    return -1;
		else if( toupper(*pa) != toupper(*pb) )
		    return 1 ; /* a is greater */
	    }
	    return 0;
	}
	else if( la < lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( toupper(*pa) < toupper(*pb) )
		    return -1;
		else if( toupper(*pa) != toupper(*pb) )
		    return 1 ; /* a is greater */
	    }
	    return -1;
	}
	else {	/* lb < la */
	    for(; lb; lb--, pa++, pb++ ) {
		if( toupper(*pa) < toupper(*pb) )
		    return -1;
		else if( toupper(*pa) != toupper(*pb) )
		    return 1 ; /* a is greater */
	    }
	    return 1;
	}
    }
    else {
	if( la == lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( *pa < *pb )
		    return -1;
		else if( *pa != *pb )
		    return 1 ; /* a is greater */
	    }
	    return 0;
	}
	else if( la < lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( *pa < *pb )
		    return -1;
		else if( *pa != *pb )
		    return 1 ; /* a is greater */
	    }
	    return -1;
	}
	else {	/* lb < la */
	    for(; lb; lb--, pa++, pb++ ) {
		if( *pa < *pb )
		    return -1;
		else if( *pa != *pb )
		    return 1 ; /* a is greater */
	    }
	    return 1;
	}
    }
}


/****************
 *  Vergleich bei BlockMark
 */

static int
#ifdef __GNUC__
SortLinesCompareB( const void *xa, const void *xb )
{
    line_t __handle **a = (line_t **)xa;
    line_t __handle **b = (line_t **)xb;
#else
SortLinesCompareB( line_t __handle **a, line_t __handle **b )
{
#endif
    ushort la, lb;
    char *pa, *pb;

    if( ++sortCompareCounter > 500 ) {
	sortCompareCounter = 0;
	SigIntPoll();
	if( sigIntPending )
	    longjmp( sortCompareJmpBuf, ERR_CMDINT );
    }
    la = (*a)->x.len;
    pa = (*a)->d;
    if( la >= mark.startPos ) {
	pa += mark.startPos;
	la -= mark.startPos;
	if( la > mark.endPos - mark.startPos + 1 )
	    la = mark.endPos - mark.startPos + 1;
    }
    else
	la = 0;
    lb = (*b)->x.len;
    pb = (*b)->d;
    if( lb >= mark.startPos ) {
	pb += mark.startPos;
	lb -= mark.startPos;
	if( lb > mark.endPos - mark.startPos + 1 )
	    lb = mark.endPos - mark.startPos + 1;
    }
    else
	lb = 0;

    if( !la && !lb )
	return 0;
    else if( sortMode == dNUM ) {
	long sa, sb;

	sa = Atoln( pa, la );
	sb = Atoln( pb, lb );
	if( sa < sb )
	    return -1;
	if( sa > sb )
	    return 1;
	return 0;
    }
    else if( sortMode == dALFACI ) {
	if( la == lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( toupper(*pa) < toupper(*pb) )
		    return -1;
		else if( toupper(*pa) != toupper(*pb) )
		    return 1 ; /* a is greater */
	    }
	    return 0;
	}
	else if( la < lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( toupper(*pa) < toupper(*pb) )
		    return -1;
		else if( toupper(*pa) != toupper(*pb) )
		    return 1 ; /* a is greater */
	    }
	    return -1;
	}
	else {	/* lb < la */
	    for(; lb; lb--, pa++, pb++ ) {
		if( toupper(*pa) < toupper(*pb) )
		    return -1;
		else if( toupper(*pa) != toupper(*pb) )
		    return 1 ; /* a is greater */
	    }
	    return 1;
	}
    }
    else {
	if( la == lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( *pa < *pb )
		    return -1;
		else if( *pa != *pb )
		    return 1 ; /* a is greater */
	    }
	    return 0;
	}
	else if( la < lb ) {
	    for(; la; la--, pa++, pb++ ) {
		if( *pa < *pb )
		    return -1;
		else if( *pa != *pb )
		    return 1 ; /* a is greater */
	    }
	    return -1;
	}
	else {	/* lb < la */
	    for(; lb; lb--, pa++, pb++ ) {
		if( *pa < *pb )
		    return -1;
		else if( *pa != *pb )
		    return 1 ; /* a is greater */
	    }
	    return 1;
	}
    }
}


/****************
 *  Vergleich (LineMark) fuer Directory Mode, der SortMode wird der
 * globalen Variablen sortMode entnommen
 * Aufbau einer DirZeile, DIR immer zuerst
 *
 * "12.03.92 14:15:12 ssssssss cc name..."
 * oder
 * "12.03.92 14:15:12    <DIR> cc name..."
 */

static int
#ifdef __GNUC__
SortLinesCompareDir( const void *xa, const void *xb )
{
    line_t __handle **a = (line_t **)xa;
    line_t __handle **b = (line_t **)xb;
#else
SortLinesCompareDir( line_t __handle **a, line_t __handle **b )
{
#endif
    ushort la, lb;
    char *pa, *pb;
    long sa, sb;
    time_t ta, tb;
    struct tm tm;

    if( ++sortCompareCounter > 500 ) {
	sortCompareCounter = 0;
	SigIntPoll();
	if( sigIntPending )
	    longjmp( sortCompareJmpBuf, ERR_CMDINT );
    }
    la = (*a)->x.len;
    lb = (*b)->x.len;
    pa = (*a)->d;
    pb = (*b)->d;
    if( la < 31 || lb < 31) /* kann eigentlich nicht sein ... */
	return 0;

    /* handle ".." entry, which is always the first one */
    if( pa[30] == '.' && pa[31]== '.' && !pa[32] )
	return directorySortReverse? 1: -1;

    switch( sortMode ) {
      case dSIZE:
	sa = atol( pa+18 );
	sb = atol( pb+18 );
	if( sa < sb )
	    return -1;
	if( sa > sb )
	    return 1;
	break;
      case dDATE: /* "12.03.92 14:15:12    <DIR> cc name..."*/
	tm.tm_mday = ATOI_2(pa);
	tm.tm_mon  = ATOI_2(pa+3) -1 ;
	tm.tm_year = ATOI_2(pa+6);
	tm.tm_hour = ATOI_2(pa+9);
	tm.tm_min  = ATOI_2(pa+12);
	tm.tm_sec  = ATOI_2(pa+15);
	tm.tm_isdst = -1; /* unkown */
	ta = mktime( &tm );
	tm.tm_mday = ATOI_2(pb);
	tm.tm_mon  = ATOI_2(pb+3) -1 ;
	tm.tm_year = ATOI_2(pb+6);
	tm.tm_hour = ATOI_2(pb+9);
	tm.tm_min  = ATOI_2(pb+12);
	tm.tm_sec  = ATOI_2(pb+15);
	tm.tm_isdst = -1; /* unkown */
	tb = mktime( &tm );
	if( ta < tb )
	    return -1;
	if( ta > tb )
	    return 1;
	break;

      case dEXT:
      default:
	break;
    }

    if( directorySortKludge ) {
	/* previous compare results in equal, so go on and sort them by name */
	if( pa[22] == 'D' && pb[22] != 'D' )
	    return -1;
	else if( pa[22] != 'D' && pb[22] == 'D' )
	    return 1;
    }

    la -= 30; pa += 30;
    lb -= 30; pb += 30;
    if( la == lb ) {
	for(; la; la--, pa++, pb++ ) {
	    if( *pa < *pb )
		return -1;
	    else if( *pa != *pb )
		return 1 ; /* a is greater */
	}
	return 0;
    }
    if( la < lb ) {
	for(; la; la--, pa++, pb++ ) {
	    if( *pa < *pb )
		return -1;
	    else if( *pa != *pb )
		return 1 ; /* a is greater */
	}
	return -1;
    }
    /* lb < la */
    for(; lb; lb--, pa++, pb++ ) {
	if( *pa < *pb )
	    return -1;
	else if( *pa != *pb )
	    return 1 ; /* a is greater */
    }
    return 1;
}





/****************
 * GetLineBlock()  Die Funktion liefert ein Array mit Pointern
 * auf die Daten der Zeilen und Infos ueber die Markierungen
 * keine Zeilen werden durch NULL-Poiner dargestellt.
 *  ml1 = index im Array, mit dem Anfang der markierung
 *  mp1 = Position in der Zeile ( nicht bei linemark )
 *  ml2 = index im Array, mit dem Ende der markierung
 *  mp2 = Position in der Zeile ( nicht bei linemark )
 * Istg ein EditBuffer aktiv, so wird die Zeile aus diesem geholt.
 * Return Value: falls mark vorhanden: markType, sonst 0
 *
 * NOTE: Fuer EMS gibt die Funktion den Pointer nicht zurueck
 */

#if USE_EMS
int
GetLineBlockInit( int fileHandle, ulong newLnr, size_t blockSize,
		  size_t *ml1, ushort *mp1, size_t *ml2, ushort *mp2 )
{
    line_t __handle *line;
    ulong lnr, editLnr;
    size_t n;
    int marktype;
    file_t *file;

    xassert( fileHandle >= 0 && fileHandle < MAX_FILES );
    file = fileTbl + fileHandle;
    editLnr = file->editBufValid? file->editBufLine:ULONG_MAX;

    if( !file->crs ) {
	file->crs = file->first;
	file->lnr = 0;
    }

    if( line = file->first ) {
	line = file->crs;
	lnr = file->lnr;
	if( lnr < newLnr ) { /* go forward */
	    while( lnr < newLnr && line->nxt ) {
		lnr++;
		line = line->nxt;
	    }
	}
	else {		/* go back */
	    while( lnr > newLnr && line->prv ) {
		lnr--;
		line = line->prv;
	    }
	}
    }

    getBlock.file = file;
    getBlock.line = line;
    getBlock.lnr  = lnr;
    getBlock.editLnr = editLnr;
    /* das ist zwar selsam codiert, aber in  der originalen Funktion */
    /* genauso codiert - warum also sich Gedanken machen */
    marktype = 0;
    for(n=0; n < blockSize; n++ ) {
	if( mark.type && mark.hd == fileHandle ) {
	    if( !marktype &&
		(lnr >= mark.startLine && lnr <= mark.endLine) ) {
		*ml1 = n;
		*mp1 = mark.startPos;
		marktype = mark.type;
	    }
	    if( marktype && lnr <= mark.endLine ) {
		*ml2 = n;
		*mp2 = mark.endPos;
		marktype = mark.type;
	    }
	}
	lnr++;
    }

    return marktype;
}


const char *
GetLineBlockNext( ushort *retlen )
{
    const char *p;

    if( getBlock.line ) {
	if( getBlock.lnr == getBlock.editLnr ) {
	    p	    = getBlock.file->editBuf;
	    *retlen = getBlock.file->editBufLen;
	}
	else {
	    p	     = getBlock.line->d;
	    *retlen  = getBlock.line->x.len;
	}
	getBlock.line = getBlock.line->nxt;
	getBlock.lnr++;
    }
    else
	p = NULL;
    return p;
}

#endif


int
GetLineBlock( int fileHandle, ulong newLnr,
	      const char **block, ushort *len, size_t blockSize,
	      size_t *ml1, ushort *mp1, size_t *ml2, ushort *mp2 )
{
    line_t __handle *line;
    ulong lnr=0, editLnr;
    size_t n;
    int marktype;
    file_t *file;

    xassert( fileHandle >= 0 && fileHandle < MAX_FILES );
    file = fileTbl + fileHandle;
    editLnr = file->editBufValid? file->editBufLine:ULONG_MAX;

    if( !file->crs ) {
	file->crs = file->first;
	file->lnr = 0;
    }

    if( line = file->first ) {
	line = file->crs;
	lnr = file->lnr;
	if( lnr < newLnr ) { /* go forward */
	    while( lnr < newLnr && line->nxt ) {
		lnr++;
		line = line->nxt;
	    }
	    if( file->view && !line->prv ) {
		GetMoreLines(fileHandle);
	    }
	}
	else {		/* go back */
	    while( lnr > newLnr && line->prv ) {
		lnr--;
		line = line->prv;
	    }
	    if( file->view && !line->prv ) {
		    ; /* reload */
	    }
	}
    }

    marktype = 0;
    for(n=0; n < blockSize; n++ ) {
	if( line ) {
	    if( lnr == editLnr ) {
		block[n] = file->editBuf;
		len[n]	 = file->editBufLen;
	    }
	    else {
		block[n] = line->d;
		len[n]	 = line->x.len;
	    }
	    if( mark.type && mark.hd == fileHandle ) {
		if( !marktype &&
		    (lnr >= mark.startLine && lnr <= mark.endLine) ) {
		    *ml1 = n;
		    *mp1 = mark.startPos;
		    marktype = mark.type;
		}
		if( marktype && lnr <= mark.endLine ) {
		    *ml2 = n;
		    *mp2 = mark.endPos;
		    marktype = mark.type;
		}
	    }
	    line = line->nxt;
	    if( file->view && line ) {
		    ; /* reload */
	    }
	    lnr++;
	}
	else
	    block[n] = NULL;
    }

    return marktype;
}




/****************
 * GetPtr2Line()  Die Funktion eine Pointer auf die aktuelle Zeile zurueck.
 * ist diese im Editbuffer, so wird der entsprechende Pointer zurueckgegeben.
 * Return Value: Pointer to line and in nbytes valid length
 */

const char *GetPtr2Line( int fileHandle, ushort *length )
{
    file_t *file;
    line_t __handle *line;

    file = fileTbl + fileHandle;

    if( !file->crs ) {
	file->crs = file->first;
	file->lnr = 0;
    }

    line = file->crs;
    if( file->editBufValid && file->lnr == file->editBufLine ) {
	*length  = file->editBufLen;
	return	file->editBuf;
    }
    else {
	*length  = line->x.len;
	return (char*)line->d;
    }
}


/****************
 * Die Funktion eine Pointer auf die <seeklnr>nte Zeile zurueck.
 * ist diese im Editbuffer, so wird der entsprechende Pointer zurueckgegeben.
 * returns: 0 := okay
 *	    1 := last line reached or error
 *    rbuf   points to the line (or NULL if not available)
 *    nbytes valid length
 *    state  current hilite state of line
 * the returned values are only valid until the next call to a File functions
 */

int
GetHiliteLine( int fileHandle,
	       ulong seeklnr, const char **rbuf,
	       ushort *length, int *state, unsigned *flags)
{
    file_t *file;
    line_t __handle *line;
    ulong lnr;

    file = fileTbl + fileHandle;
    *rbuf = NULL;
    xassert(file->nattrib);
    if( !file->first )
	return 1;
    if( !file->hilite_crs ) {
	file->hilite_crs = file->first;
	file->hilite_lnr = 0;
    }

    line = file->hilite_crs;
    lnr  = file->hilite_lnr;
    if( lnr < seeklnr ) { /* go forward */
	while( lnr < seeklnr && line->nxt ) {
	    lnr++;
	    line = line->nxt;
	}
    }
    else {	    /* go back */
	while( lnr > seeklnr && line->prv ) {
	    lnr--;
	    line = line->prv;
	}
    }
    file->hilite_crs = line;
    file->hilite_lnr = lnr;

    *flags = 0;
    if( file->editBufValid && lnr == file->editBufLine ) {
	*length  = file->editBufLen;
	*rbuf = file->editBuf;
	*flags |= 1;  /* edit line */
    }
    else {
	*length  = line->x.len;
	*rbuf = (char*)line->d;
    }
    *state = ((byte*)line->d)[line->x.len];
    return line->nxt? 0 : 1;
}


/****************
 *  Den Hilite State der angegebene Zeile setzen
 * returns: 0 := okay
 *	    1 := last line reached
 */
int
SetHiliteState( int fhd, ulong seeklnr, int state )
{
    file_t *file;
    line_t __handle *line;
    ulong lnr;

    file = fileTbl + fhd;
    if( !file->first )
	return 1;
    if( !file->hilite_crs ) {
	file->hilite_crs = file->first;
	file->hilite_lnr = 0;
    }

    line = file->hilite_crs;
    lnr  = file->hilite_lnr;
    if( lnr < seeklnr ) { /* go forward */
	while( lnr < seeklnr && line->nxt ) {
	    lnr++;
	    line = line->nxt;
	}
    }
    else {	    /* go back */
	while( lnr > seeklnr && line->prv ) {
	    lnr--;
	    line = line->prv;
	}
    }
    file->hilite_crs = line;
    file->hilite_lnr = lnr;

    ((byte*)line->d)[line->x.len] = state;
    return line->nxt? 0 : 1;
}


void SaveFilePos( int fhd , saveFilePos_t *pos )
{
    file_t *file;

    file = fileTbl+fhd;
    pos->fhd = fhd;
    pos->lnr = file->lnr;
    pos->pos  = file->pos;
    pos->x    = file->crsX;
    pos->y    = file->crsY;
}



void RestFilePos( saveFilePos_t *pos )
{
    file_t *file;

    file = fileTbl+pos->fhd;
    file->crsX = pos->x;
    file->crsY = pos->y;
    SeekLine( pos->fhd, pos->lnr );
    PosLine( pos->fhd, pos->pos );
}


/****************
 * Set a Mark at actual Pos:
 * Ist dies die erste markierung, so wird start und end gesetzt;
 * bei allen weiteren wird nur noch end gesetzt.
 *  markType : 0 = Clear marks
 *	       MARKTYPE_LINE
 *	       MARKTYPE_BLOCK
 * Returns: 0 = Okay
 *	    ERR_MRKCFL	= Mark Conflict
 */

int SetMark( int fileHandle, int marktype )
{
    file_t *file;

    file = fileTbl+fileHandle;
    if( !marktype )
	mark.type = 0;
    else if( !mark.type ) {    /* no mark yet set */
	mark.type = marktype;
	mark.hd   = fileHandle;
	mark.startLine = mark.endLine = file->lnr;
	mark.startPos = mark.endPos = file->pos;
	mark.anchorLine = mark.anchorPos = 0;
    }
    else if( mark.type == marktype && fileHandle == mark.hd ) {
	if( !mark.anchorLine ) {
	    if( file->lnr >= mark.startLine )
		mark.endLine = file->lnr;
	    else {
		mark.anchorLine = 1;
		mark.endLine = mark.startLine;
		mark.startLine = file->lnr;
	    }
	} else {
	    if( file->lnr >= mark.endLine ) {
		mark.startLine = mark.endLine;
		mark.endLine = file->lnr;
		mark.anchorLine = 0;
	    }
	    else
		mark.startLine = file->lnr;
	}
	if( !mark.anchorPos ) {
	    if( file->pos >= mark.startPos )
		mark.endPos  = file->pos;
	    else {
		mark.anchorPos = 1;
		mark.endPos = mark.startPos;
		mark.startPos = file->pos;
	    }
	} else {
	    if( file->pos >= mark.endPos ) {
		mark.startPos = mark.endPos;
		mark.endPos = file->pos;
		mark.anchorPos = 0;
	    }
	    else
		mark.startPos  = file->pos;
	}
    }
    else
	return ERR_MRKCFL;
    return 0;
}


/****************
 * Adjust a block mark: Shift to the left or the right
 * according to value
 */

void AdjustBlockMark( int fhd, int value )
{
    if( mark.type == MARKTYPE_BLOCK && mark.hd == fhd ) {
	if( value > 0 ) {
	    if( value + mark.endPos < MAX_LINELEN ) {
		mark.startPos += value;
		mark.endPos += value;
	    }
	}
	else {
	    if( mark.startPos >= (-value) ) {
		mark.startPos += value;
		mark.endPos += value;
	    }
	}

    }
}



/****************
 * Put the mark onto the Stack , Mark will not be deleted !
 */

int PushMark()
{
    int err=0;

    if( markStackInd < MARKSTACK_SIZE )
	markStack[markStackInd++] = mark;
    else
	err = ERR_PUSHMRK;

    return err;
}


/****************
 * Replace the mark with the last pushed mark
 */

int PopMark()
{
    int err=0;

    if( markStackInd )
	mark = markStack[--markStackInd];
    else
	err = ERR_POPMRK;

    return err;
}


void ResetMarkStack()
{
    markStackInd = 0;
}


/****************
 * Get informations about the mark
 *	retFhd: if ! NULL returns handle of the file with the mark
 * Returns: marktype
 */

int MarkInfo( int *retFHd )
{
    if( retFHd )
	*retFHd = mark.hd;
    return mark.type;
}

ulong MarkStart( ushort *pos )
{
    if( pos )
	*pos = mark.startPos;
    return mark.startLine;
}


ulong MarkEnd( ushort *pos )
{
    if( pos )
	*pos = mark.endPos;
    return mark.endLine;
}



/****************
 * Set a label to the file with handle fhd at its actual position.
 * if fhd is -1, the label will be removed, labels must have values
 * from 'A' to 'Z'.
 * returns: errorcode
 */

#if W_FULL
int FileSetLabel( int label, int fhd )
{
    int err=0;
    file_t *file;

    label -= 'A';
    if( label >= 0 && label < LABELTBL_SIZE ) {
	labelTbl[label].fhd = fhd;
	if( fhd != -1 ) {
	    file = fileTbl+fhd;
	    labelTbl[label].lnr = file->lnr;
	    labelTbl[label].pos = file->pos;
	}
    }
    else
	err = ERR_INVLAB;

    return err;
}


int FileGetLabel( int label, int *fhd, ulong *lnr, ushort *pos )
{
    int err=0;

    label -= 'A';
    if( label >= 0 && label < LABELTBL_SIZE ) {
	if( (*fhd = labelTbl[label].fhd) != -1 ) {
	    *lnr = labelTbl[label].lnr;
	    *pos = labelTbl[label].pos;
	}
	else
	    err = ERR_NOLABEL;
    }
    else
	err = ERR_INVLAB;

    return err;
}




/****************
 * Anhand einer gegeben lineID nach der Zeile suchen.
 * Returns: errorCode und in retLineNr die nr der gefundenen
 *			  Zeile (0 based!)
 */

int GetLineNrFromLineId( int fileHandle, ulong *retLineNr )
{
    line_t __handle *line;
    ulong lnr;
    file_t *file;
    ushort lineId;
    int i;

    file = fileTbl+fileHandle;
    if( !file->lineIdOff )
	return ERR_NOLINEID;   /* ist fuer den file gar nicht enabled */
    if( *retLineNr > 0xffff )
	return ERR_NOLINEID;   /* so hohe Ids gibt es nicht */
    lineId = *retLineNr;
    if( !lineId )
	return ERR_NOLINEID;   /* 0 gibt es nicht */

    for( i=100,line=file->first, lnr = 0; line; line = line->nxt, lnr++ ) {
	if(
	  #if ALIGNMENT_REQUIRED
	    !memcmp( line->d+line->x.len+1, &lineId, sizeof(ushort))
	  #else
	    *(ushort*)(line->d+line->x.len+1) == lineId
	  #endif
	    ) {
	    *retLineNr = lnr;
	    return 0;
	}
	if( !--i ) { /* gelegentlich mal nach SigInt sehen */
	    SigIntPoll();
	    if( sigIntPending )
		return ERR_CMDINT;
	    i = 100;
	}
    }
    return ERR_NOLINEID;
}



/****************
 * Falls LineIds enabled sin, diese wieder neu setzen, so
 * dass die ZeilenNummern mit den LineIds wieder uebereinstimmen
 * (wird von Save benutzt, um die lineIds wieder mit den Zeilennummern
 * des gesepicherten Files zu syncronisieren)
 */

void RefreshLineIds( int fileHandle )
{
    line_t __handle *line;
    file_t *file;
    ushort lineId;
    int i;

    file = fileTbl+fileHandle;
    if( !file->lineIdOff )
	return ;	       /* ist fuer den file gar nicht enabled */
    lineId = 1;
    for( i=100,line=file->first; line; line = line->nxt ) {
	  #if ALIGNMENT_REQUIRED
	    memcpy( line->d+line->x.len+1, &lineId, sizeof(ushort));
	  #else
	    *(ushort*)(line->d+line->x.len+1) = lineId;
	  #endif
	    if( lineId )
		lineId++;
	    else { /* nach 65000 Zeilen mal Ctrl-Break checks */
		if( !--i ) { /* gelegentlich mal nach SigInt sehen */
		    SigIntPoll();
		    if( sigIntPending )
			return;
		    i = 100;
		}
	    }
    }
}



#endif


/*** bottom of file ***/
