/* [update.c wk 24.07.94] An update utility
 *	Copyright (c) 1994 by Werner Koch (dd9jn)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 14.03.95 wk	now uses a large buffer to copy the files
 */

#include <wk/tailor.h>
RCSID("$Id: update.c,v 1.6 1997/03/24 19:52:08 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <wk/file.h>
#include <wk/direc.h>
#include <wk/string.h>
#include <wk/environ.h>

/****** constants *********/
#define BUFFERSIZE 32000
/******** typedefs ********/

static void DoUpdate( const char *src, const char *dst);
/******* globals **********/
static struct {
	int tty;
	int quiet;
	int verbose;
	int basename;
	int destfile;
	int noexec;
	int todos;
    } opt ;

static int errorflag;
static char *buffer;

/******* Functions ********/

const char *
CopyRight( int level )
{
    const char *p;
    switch( level ) {
      case 10:
      case 0:	p = "update - v1.02; "
		    "Copyright 1994,1996 by Werner Koch (dd9jn)" ; break;
      case 13:	p = "update"; break;
      case 14:	p = "1.02"; break;
      case 1:
      case 11:	p = "Usage: update [options] files destination (-h for help)";
		break;
      case 2:
      case 12:	p =
    "Syntax: update [options] files destination to\n"
    "Copy files to destination directory if file is newer\n"
    "Options summary:\n"
    " -b = use basename of sourcefile\n"
    " -f = destination is a file\n"
    " -todos = translate by using DOZE conventions\n"
    " -n = no execution\n"
    " -q = quiet mode\n"
    " -v = verbose listing\n"
    " -h = help\n";
	break;
      default: p = "?";
    }

    if( !level ) { fputs( p,stderr ); putc('\n',stderr); }
    else if( level == 1 ) { fputs( p, stderr ); putc( '\n', stderr );exit(3);}
    else if( level == 2 ) {  /* help */ puts( p ); exit(0); }

    return p;
}

int main( int argc, char **argv )
{
    char *s;
    char *destdir, *cwd;

    if( ArgExpand( &argc, &argv, 4 | 1 ) )
	Error(4,GetStr(12));

    for( s=""; --argc && **++argv == '-' && *s != '-'; )
	for( s = *argv + 1 ; *s && *s != '-' ; s++ )
	    switch( *s ) {
	      case 'v': opt.verbose++; break;
	      case 'f': opt.destfile++; break;
	      case 'b': opt.basename++; break;
	      case 'q': opt.quiet++; break;
	      case 'n': opt.noexec++; break;
	      case 't':
		if( !strcmp(s,"todos") ) {
		    opt.todos++;
		    while(*s) s++; s--;
		}
		else
		    Error(3,GetStr(15),s );
		break;
	      case 'h' :
	      case '?' : CopyRight(0) ; CopyRight(2); break;
	      default  : Error(3,GetStr(15),s );
	    }
    if( argc < 2 )
	CopyRight(1);
    if( opt.destfile && argc != 2 )
	Error(3,"error: option -f needs exactly two filenames");
    if( opt.destfile && opt.basename )
	Error(3,"error: option -f can't be combined with -b");

    if( !opt.quiet ) {
	if( (opt.tty = IsTerminal(0)) )
	    setvbuf(stdout,NULL,_IONBF,0);
    }
    else
	opt.verbose = 0 ;
    buffer = xmalloc(BUFFERSIZE);

    /* get the destination directory */
    destdir = xstrdup(argv[--argc]);
    if( opt.destfile ) {
	if( *destdir && destdir[strlen(destdir)-1] == '/' )
	    Error(2,"It seems that '%s' is a directory", destdir );
    }
    else if( !*destdir || destdir[strlen(destdir)-1] != '/' )
	Error(2,"I'm not sure wether '%s' is a directory", destdir );
    cwd = NULL;
    /* process the files */
    for( ; argc ; argc--, argv++ ) {
	s = *argv;
	if( *s && s[strlen(s)-1] == '/' ) {
	    if( !opt.quiet )
		Info("Switching to source directory '%s'", s );
	    if( !cwd )
		cwd = cwdname();
	    if( ChangeDirectory( s ) )
		Error(2,"can't switch to '%s'", s );
	}
	else
	    DoUpdate( s, destdir );
    }
    if( cwd ) {  /* need to switch back for MSDOS */
	if( ChangeDirectory( cwd ) )
	    Error(2,"can't switch back to '%s'", cwd );
	free(cwd);
    }
    free(buffer);
    return errorflag? 1:0;
} /* end main() */


static void
DoUpdate( const char *src, const char *dstdir)
{
    FILE *fpin, *fpout;
    char *dst;
    char *s;
    struct stat srcstat, dststat;
    size_t n;

    if( stat(src, &srcstat) ) {
	Error(1000,"error: can't stat '%s'", src);
	errorflag=1;
	return;
    }

    if( opt.destfile )
	dst = xstrdup(dstdir);
    else if( opt.basename ) {
	s = basename(src);
	dst = xmalloc( strlen(dstdir) + strlen(s) + 1 );
	strcpy(stpcpy(dst,dstdir),s);
	free(s);
    }
    else {
	dst = xmalloc( strlen(dstdir) + strlen(src) + 1 );
	strcpy(stpcpy(dst,dstdir),src);
    }

    if( stat(dst, &dststat) ) {
	if( errno != ENOENT ) {
	    Error(1000,"error: can't stat '%s'", dst);
	    errorflag=1;
	    free(dst);
	    return;
	}
	if( opt.verbose )
	    if( opt.destfile )
		Info("'%s' does not exist", dst );
	    else
		Info("'%s' not in '%s'", src, dstdir );
    }
    else if( dststat.st_mtime >= srcstat.st_mtime ) {
	if( opt.verbose )
	    Info("'%s' needs no update", dst );
	free(dst);
	return;
    }

    if( opt.noexec ) {
	Info("'%s' would be copied to '%s'", src, dst );
	free(dst);
	return;
    }
    if( !opt.quiet )
	Info("%s -> %s", src, dst );

    fpin = xfopen( src, "rb" );
    fpout= xfopen( dst, "wb" );

    while( n = fread(buffer,1, BUFFERSIZE, fpin) ) {
	if( n == EOF )
	    Error(2,"error reading '%s'", src);
	if( opt.todos ) {
	    byte *p;

	    for(p=buffer; n; n--, p++ )
		if( *p == '\n' ) {
		    putc('\r', fpout );
		    putc('\n', fpout );
		}
		else if( *p & 128 )
		    putc(MapIso2Ibm(*p,0), fpout );
		else
		    putc(*p, fpout );
	}
	else {
	    if( fwrite(buffer,1,n,fpout) != n )
		Error(2,"error writing '%s'", dst);
	}
    }

    fclose(fpout);
    fclose(fpin);
  #ifdef UNIX
    if( chmod(dst, srcstat.st_mode  ) )
	Error(1000,"%s: warning: chmod failed", dst);
  #endif
    free(dst);
}


/*** bottom of file ***/
