/* [ldsh.c wk 03.11.93] a linker frontend
 *	Copyright (c) 1993,1997 by Werner Koch (dd9jn)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * BUGS: if else endif kann durch einen Kommentar in einem else.zweig
 *	 durcheinandergebracht werden.
 */

#include <wk/tailor.h>
RCSID("$Id: ldsh.c,v 1.25 1997/08/25 09:14:45 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/string.h>
#include <wk/direc.h>
#include <wk/file.h>
#include <wk/io.h>


#ifndef UNIX
#define TMP_PREFIX "$LDS"
#else
#define TMP_PREFIX ".#ld"
#endif


/******** typedefs ********/
typedef enum {
    tUNKNOWN = 0,
    tLINK386,	/* standard OS/2 2.0 linker (32-bit)*/
    tLINK,	/* Standard linker (OS/2 version) */
    tLINKdos,	/* Standard Linker (MSDOS) */
    tBLINK,	/* Zortech OS/2 Linker */
    tBLINKdos,	/* Zortech MSDOS Linker */
    tBLINKx,	/* Zortech MSDOS LINKER using DOSX */
    tWLINK,	/* watcom linker */
    tLD
} linker_t;

typedef enum {
    nUNKNOWN = 0,
    mOBJ,
    mEXE,
    mLIB,
    mDEF,
    mRES,
    mIMP,
    mEXP,
    mMAP,
    mOTH
} mtype_t;


typedef struct filelist_s {
    mtype_t type;
    struct filelist_s *next;
    char name[1];
} filelist_t;


typedef struct fplist {
    struct fplist *next;
    FILE *fp;
    char *name;     /* malloced */
} *t_fplist;


typedef struct s_macro {
    struct s_macro *next;
    const char *value;
    char name[1];
} *MACRO;

/******* globals **********/
static struct {
	int verbose;
	int noExec;
	int backslash;
	long stackSize;
	int  stackSizeForced;
	int setDebug;
	int ignore;
	int windowapi;
	int nodefault;
	int nocase;
	char *implibName;
	char *linkerOptions;
	char *machine;
	int forcemap;
	int posix;
	int profile;
	int debug;
	int syslink;
	int nolinkinfo;
	char *configfile;
	const char *toolprefix;
    } opt ;

static struct {
    const char *s;
    mtype_t type;
    int flag;	/* bit 0:  unix, bit 1: dos etc.  bit 2: cross unix to dos */
} extTable[] = {
    { ".o"    , mOBJ, 1 },
    { ".a"    , mLIB, 1 },
    { ".sa"   , mLIB, 1 },
    { ".so"   , mEXE, 1 },
    { ".obj"  , mOBJ, 2 },
    { ".lib"  , mLIB, 2 },
    { ".imp"  , mLIB, 2 }, /* translated to mIMP for netware */
    { ".exp"  , mEXP, 2 },
    { ".exe"  , mEXE, 2 },
    { ".dll"  , mEXE, 0 },
    { ".nlm"  , mEXE, 0 },
    { ".def"  , mDEF, 0 },
    { ".map"  , mMAP, 0 },
    { ".res"  , mRES, 0 },
    { ""      , mEXE, 1 },
    { NULL, 0 }
} ;

static enum {
    oOS2,
    oOS20,
    oMSDOS,
    oNETWARE,
    oUNIX,
    oWINDOWS
} ostype;
static int isOS2Version2;
static int isDosExtender;
static int createDLL;
static char *target;
static char *target_CPU;
static char *target_COMPANY;
static char *target_SYSTEM;
static MACRO macrolist;

static char *resourceCompiler = "rc";

static linker_t  linker;
static const char *linker_command;
static filelist_t *fileTable;
static int linkerReturnCode;

static t_fplist tempfiles;

static char  *infoStringArea;
static size_t infoStringAreaUsed;
static size_t infoStringAreaSize;

static int import_lib_ready;

/****** prototypes ********/
static void Cleanup( void *dummy );
static char *ChangeSlash( char *string );
static void FindLinker(void);
static void SetupFiletype(void);
#ifdef UNIX
  static void CallDLLTool(void);
#endif
static void CallLinker(void);
static void CallResourceLinker(void);
static void CallImportLinker(void);
static void RunUNIXLinker(const char *);
static void BuildMSLink( FILE *fp );
static void BuildWatcomLink( FILE *fp );
static void WatcomExtractFromDef( FILE *fpout, FILE *fpin, const char *key );
static void WatcomExtractOneFromDef( FILE *fpout, FILE *fpin, const char *key);
static filelist_t *PreprocessFile( filelist_t *f );
static void AddInfoString(const char*string);
static const char *CreateInfoObj( const char *infostring, size_t len );
static char *CreateTargetSpec(void);
static void CopyFile( const char *src, const char *dst );

/******* Functions ********/


const char *
CopyRight( int level )
{
    const char *p;
    switch( level ) {
      case 10:
      case 0:	p = "ldsh - v1.10b; "
		    "Copyright 1993-97 by Werner Koch (dd9jn)" ; break;
      case 14:	p =	    "1.10b"; break;
      case 1:
      case 11:	p = "Usage: ldsh [options] modules (-h for help)";
		break;
      case 2:
      case 12:	p =
    "Syntax: ldsh [options] modules\n"
    "LDSH calls a linker depending on the Compiler and the OS\n";
	break;
      case 13:	p = "ldsh"; break;
      default:	p = WklibCopyRight(level);
    }
    ShowCopyRight(level);
    return p;
}

static ARGPARSE_OPTS main_opts[] = {
    { 'v', "verbose",   0   , "Tell what we are doing" },
    { 'd', "debug",     0   , "Increase debug level" },
    { 'm', "force-map" ,0   , "force creation of a map file" },
    { 'g', NULL        ,0   , "enable debugging" },
    { 'w', "window-api",1   , "enable windowapi" },
    { 'i', "ignore"  ,  2   , "retain an incomplete run file" },
    { 's', "stacksize", 3   , "set stacksize to n bytes" },
    { 'n', "dry-run",   0   , "do not execute commands" },
    { 'L', NULL,        2   , "create import library <name> for the DLL" },
    { 'C', "config",    2   , "use this configurationfile" },
    { 'b', "machine",   2|8 , "force the use of gcc (optional with arg -b)" },
    { 'B', "exec-prefix", 2 , "prefix used to call special tools" },
    { 'D', NULL,        2   , "insert string <s> with key <k> into the\n"
			      "executable (e.g. -D name=joe" },
    { 501, "jn",        0   , "no default library search" },
    { 502, "ji",        0   , "be case in-sensitive" },
    { 503, "jx",        0   , "do not create a linker-info object" },
    { 601, "syslink" ,  0   , "force usage of system linker" },
    { 602, "Wl"      ,  2   , "pass string to the linker" },
    { 603, "posix"   ,  0   , "use posix linker (for HP)" },
    { 604, "pg"      ,  0   , "enable profiling" },
    {0} };


void
main( int argc, char **argv )
{
    char *s;
    filelist_t *f, *fhelp;
    int others;
    ARGPARSE_ARGS pargs;

    if( ArgExpand( &argc, &argv, 4 | 1 ) )
	Error(4,GetStr(12));

    pargs.argc = &argc;
    pargs.argv = &argv;
    pargs.flags= 32; /* enable kludge to allow single dashed long options */
    while( ArgParse( &pargs, main_opts) ) {
	s = pargs.r.ret_str;
	switch( pargs.r_opt ) {
	  case 'v': opt.verbose++; break;
	  case 'd': opt.debug++; break;
	  case 'm': opt.forcemap++; break;
	  case 'n': opt.noExec++; break;
	  case 'g': opt.setDebug++; break;
	  case 'w': opt.windowapi++; break;
	  case 'i': opt.ignore++; break;
	  case 's':
	    opt.stackSizeForced++;
	    opt.stackSize = pargs.r.ret_long;
	    break;
	  case 'L': opt.implibName = pargs.r.ret_str; break;
	  case 'C': opt.configfile = pargs.r.ret_str; break;
	  case 'b': opt.machine = pargs.r_type? pargs.r.ret_str : ""; break;
	  case 'B': opt.toolprefix = pargs.r.ret_str; break;
	  case 'D': AddInfoString(pargs.r.ret_str); break;

	  case 501: opt.nodefault++; break;
	  case 502: opt.nocase++; break;
	  case 503: opt.nolinkinfo++; break;
	  case 601: opt.syslink++; break;
	  case 602: opt.linkerOptions = pargs.r.ret_str; break;
	  case 603: opt.posix++; break;
	  case 604: opt.profile++; break;
	  case -2 : /* invalid option */
	    if( pargs.r.ret_str && !strncmp(pargs.r.ret_str,"Wl",2) ) {
		/* Oh what a kludge to make thinks like "-Wl,-egal" work */
		opt.linkerOptions = pargs.r.ret_str+2;
		break;
	    }
	  default : pargs.err = 2; break;
	}
    }
    if( !argc )
	CopyRight(1);

    target = getenv("TARGET");

    AddCleanUp( Cleanup, NULL );

    /* collect all files */
    fhelp = NULL;
    others = 0;
    for( ; argc ; argc--, argv++ ) {
	if( s = strchr( *argv, '=' ) ) {
	    MACRO m;

	    /* environment from commandline */
	    *s++ = 0;
	    /* Some special macros */
	    if( !strcmp(*argv, "TARGET") )
		target = s;
	    m = xmalloc( sizeof *m + strlen(*argv) );
	    strcpy(m->name,*argv);
	    m->value = s;
	    m->next = macrolist;
	    macrolist = m;
	}
	else if( !strcmp(*argv, "--") )
	    others = !others;
	else {
	    f = xcalloc( 1, sizeof *f + strlen(*argv) );
	    strcpy( f->name, *argv );
	    f->next = NULL;
	    if( others )
		f->type = mOTH;
	    if( !fhelp )
		fileTable = fhelp = f;
	    else {
		fhelp->next = f;
		fhelp = f;
	    }
	}
    }


    /* get COMPANY and SYSTEM from TARGET */
    if( !target )
	Error(2,"no TARGET specified");
    target_CPU = xstrdup(target);
    if( s = strchr(target_CPU,'-') ) {
	*s++ = 0;
	if( *s == '-' ) {
	    target_COMPANY = "";
	    target_SYSTEM = s+1;
	}
	else {
	    target_COMPANY = s;
	    if( s = strchr(target_COMPANY,'-') ) {
		*s++ = 0;
		target_SYSTEM = s;
	    }
	}
    }
    if( !target_CPU || !*target_CPU
	|| !target_COMPANY
	|| !target_SYSTEM || !*target_SYSTEM )
	Error(2,"malformed TARGET=%s", target);

    /* make LinkDate */
    s = xmalloc(100);
    sprintf(s,"LinkDate=%s", JD2Str2(-1,3,-1,100,NULL));
    AddInfoString(s);
    free(s);

    FindLinker();
    SetupFiletype();
  #ifdef UNIX
    if( createDLL && (	  !strcmp( target_SYSTEM, "mingw32")
		       || !strcmp( target_SYSTEM, "cygwin32")
		       || !strcmp( target_SYSTEM, "winnt") ) )
	CallDLLTool();
  #endif
    CallLinker();
    if( !linkerReturnCode )
	CallResourceLinker();
    if( opt.implibName && !linkerReturnCode && !import_lib_ready )
	CallImportLinker();
    if( linkerReturnCode > 255 )
	linkerReturnCode = 4;
    exit(linkerReturnCode) ;
}


static void
Cleanup( void *dummy )
{
    t_fplist r, rn;
    filelist_t *f, *fhelp;
    MACRO m, m2;

    for(r = tempfiles; r ; r = rn ) {
	rn = r->next;
	if( r->fp )
	    fclose(r->fp);
	if( opt.debug > 2 )
	   Info("debuginfo: retaining '%s'", r->name );
	else
	    remove( r->name );
	free( r->name );
	free(r);
    }

    /* release memory */
    for(f=fileTable; f; f = fhelp ) {
	fhelp = f->next;
	free(f);
    }
    for(m= macrolist; m; m = m2 ) {
	m2 = m->next;
	free(m);
    }
    macrolist = NULL;
    free(infoStringArea);
}




static char *
ChangeSlash( char *string )
{
    char *s;

    if( opt.backslash )
	for(s=string; *s ; s++ )
	    if( *s == '/' )
		*s = '\\';
    return string;
}


/****************
 * Bestimmt den Typ des Linkers
 */

static void
FindLinker()
{
    char *os;

    ostype = oOS2;
    isOS2Version2 = 1;
    os = target_SYSTEM;

    if( opt.verbose )
	Info("os='%s' cpu='%s'", os, target_CPU );
    if( !strcmp( os, "os2") && !strcmp(target_CPU,"i386") ) {
	linker = tLINK386;
    }
    else if( !strcmp( os, "os2" ) ) {
	isOS2Version2 = 0;
	linker = tLINK;
    }
    else if( !strcmp( os, "netware" ) ) {
	linker = tWLINK;
	ostype = oNETWARE;
    }
    else if( !strcmp( os, "unix" ) ) {
	linker = tLD;
	ostype = oUNIX;
    }
    else if( !strcmp( os, "mingw32") && !strcmp(target_CPU,"i386") ) {
	if( opt.verbose )
	    linker_command="gcc -v";
	else
	    linker_command="gcc";
	if( !opt.toolprefix )
	    opt.toolprefix="i386--mingw32";
	linker = tLD;
	ostype = oUNIX;
    }
    else if( !strcmp( os, "msdos" ) && !strcmp(target_CPU,"i386")) {
	linker = tLINKdos;
	resourceCompiler = "rc_d";
	ostype = oMSDOS;
	isDosExtender=1;
    }
    else if( !strcmp( os, "msdos" ) ) {
	linker = tLINKdos;
	ostype = oMSDOS;
	resourceCompiler = "rc_d";
    }
    else if( !strcmp( os, "windoze" ) ) {
	linker = tLINKdos;
	resourceCompiler = "rc_d";
	ostype = oWINDOWS;
    }
    else {  /* assume a UNIX system */
	linker = tLD;
	ostype = oUNIX;
    }

    if( opt.syslink )
	;
    else if( !strcmp(target_COMPANY,"ztc") ) {
	if( linker == tLINKdos )
	    linker = getenv("DPMI_AVAILABLE")? tBLINKx : tBLINKdos;
    }
    else if( !strcmp(target_COMPANY,"wtc") )
	linker = tWLINK;

    if( linker == tLINK386 ||
	linker == tLINK ||
	linker == tLINKdos ||
	linker == tBLINK ||
	linker == tBLINKdos ||
	linker == tBLINKx ||
	linker == tWLINK  )
	opt.backslash = 1;

    if( ostype == oMSDOS && opt.windowapi )
	ostype = oWINDOWS;

    if( !opt.stackSizeForced && !opt.stackSize && linker == tLINK386 )
	opt.stackSize = 100000;
    else if( !opt.stackSizeForced && !opt.stackSize &&
				  ostype == oOS2 && isOS2Version2 )
	opt.stackSize = 100000;
}

static void
SetupFiletype()
{
    filelist_t *f, *f2, *f1;
    const char *ext, *s;
    int i;

    createDLL=0;
    for(f=fileTable; f; f = f->next ) {
	ChangeSlash( f->name );
	if( f->type != mOTH ) { /* special case, already set */
	    ext = f->name + strlen(f->name) - FileExtCheck( f->name );
	    for(i=0; s = extTable[i].s; i++ )
		if( !FileCmpName( s, ext ) ) {
		    f->type = extTable[i].type;
		    /* fix the type for netware */
		    if( ostype == oNETWARE && f->type == mLIB
			&& !FileCmpName( ".imp", ext ) )
			f->type = mIMP;
		    if( f->type==mEXE && !FileCmpName( s, ".dll" ) )
			createDLL=1;
		    break;
		}
	}
    }

    i = 0;
    for(f=fileTable; f; f = f->next )
	if( !f->type )
	    Error(0,"warning: '%s': module type is unknown", f->name);
	else if( f->type == mEXE )
	    i++;
    if( i > 1 )
	Error(0,"warning: more than one EXE or DLL file specified");
    else if( !i )  { /* take the name of the first obj file as run file */
	for(f=fileTable; f; f = f->next )
	    if( f->type == mOBJ )
		break;
	if( !f )
	    Error(2,"error: no run file specified");
	s = f->name;
	f = xcalloc( 1, sizeof *f + strlen(s) + 4 );
	strcpy( f->name, s );
	FileExtAppend(f->name, "exe");
	f->type = mEXE;
	f->next = fileTable;
	fileTable = f;
    }

    if( opt.forcemap ) {
	for(f=fileTable; f; f = f->next )
	    if( f->type == mMAP )
		break;
	if( !f ) { /* no map file - create on */
	    for(f=fileTable; f; f = f->next )
		if( f->type == mEXE )
		    break;
	    if( !f )
		BUG();
	    s = f->name;
	    f = xcalloc( 1, sizeof *f + strlen(s) + 4 );
	    strcpy( f->name, s );
	    FileExtAppend(f->name, "map");
	    f->type = mMAP;
	    f->next = fileTable;
	    fileTable = f;
	}
    }

    if( opt.implibName ) {
	for(i=0,f=fileTable; f; f = f->next )
	    if( f->type == mDEF )
		i++;
	if( !i || !createDLL )
	    Error(2,"error: need DLL and DEF-File to create Import-Library");
    }

    /* Def-Files eventuell preprocessen */
    for(i=0,f1=NULL,f=fileTable; f; f1=f, f = f->next )
	if( f->type == mDEF )
	    if( f2 = PreprocessFile(f) ) { /* austauschen */
		f2->next = f->next;
		if( !f1 )
		    fileTable = f2;
		else
		    f1->next = f2;
		free(f);
		f = f2;
	    }

    if( !opt.nolinkinfo )
	if( s = CreateInfoObj( infoStringArea, infoStringAreaUsed )) {
	    f = xcalloc( 1, sizeof *f + strlen(s) );
	    strcpy( f->name, s );
	    ChangeSlash( f->name );
	    f->type =mOBJ;
	    f->next = fileTable;
	    fileTable = f;
	}
}


/****************
 * This is used with mingw32 (or cygwin32 ) to create the dll export object
 * and the import library (when used with option -L)
 *
 * NOTE: We need "dlltool"  somewhere in the path
 *
 * Example: to create the dll "foo.dll" we use this command:
 *   dlltool --output-exp temp.o --output-lib implibname.a
 *	     --as i386--mingw32-as --def foo.def --dllname foo.dll foo.o
 * "--output-lib" is only used with option -L.
 * "--def" is only used if we have a def file
 * "foo" is the first object file encountered and only used if there is no .def.
 */
#ifdef UNIX
static void
CallDLLTool()
{
    filelist_t *f;
    const char *expFile=NULL, *objFile=NULL, *defFile=NULL;
    char *nameFile=NULL, *baseFile=NULL;
    char *s, *p;
    t_fplist fpl;

    /* Get the def file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mDEF ) {
	    defFile = f->name;
	    break;
	}
    if( !defFile ) { /* no def file, so take the first object file */
	for(f=fileTable; f; f = f->next )
	    if( f->type == mOBJ )
		objFile = f->name;
	if( !objFile )
	    Error(2,"error: need .def or .o file");
    }
    for(f=fileTable; f; f = f->next )
	if( f->type == mEXE ) {
	    nameFile = basename(f->name);
	    break;
	}
    if( !nameFile )
	Error(2,"error: need .dll file to get the name of the DLL");

    /* We cannot use this function because dlltools is very lazy, by
     * creating a tempfile by prefixing this filename with a "t" ---
     * What we would get ist this "t/tmp/..." and that always, because
     * the CreateTmpFile() functions return absolute filenames ....
     *
     * if( !(p = CreateTmpFile3( "", TMP_PREFIX, ".exp" )) )
     *	 Error(2,GetStr(8));
     *
     * So we a more simple scheme.
     */
    p = xmalloc(50);
    sprintf(p, "exp-%d.tmp", (int)getpid() );
    fpl = xcalloc( 1, sizeof *fpl );
    fpl->name = p;
    fpl->next = tempfiles;
    tempfiles = fpl;
    expFile = p;
    /* and insert the exp file into the list of object files */
    f = xcalloc( 1, sizeof *f + strlen(expFile) );
    strcpy( f->name, expFile );
    ChangeSlash( f->name );
    f->type =mOBJ;
    f->next = fileTable;
    fileTable = f;

    if( !(baseFile = CreateTmpFile3( NULL, TMP_PREFIX, ".base" )) )
	Error(2,GetStr(8));
    fpl = xcalloc( 1, sizeof *fpl );
    fpl->name = baseFile;
    fpl->next = tempfiles;
    tempfiles = fpl;

    s = xmalloc( 120
		 + strlen(expFile)
		 + strlen(defFile? defFile : objFile)
		 + strlen(nameFile)
		 + strlen(baseFile)
		 + (opt.implibName? strlen(opt.implibName):0)
	       );
    sprintf(s,"dlltool %s--as %s%sas --output-exp %s %s%s "
	      "--dllname %s %s%s %s",
	       opt.verbose? "-v ":"",
	       opt.toolprefix? opt.toolprefix : "",
	       opt.toolprefix? "-"            : "",
	       expFile,
	       opt.implibName? "--output-lib ":"",
	       opt.implibName? opt.implibName :"",
	       nameFile,
	       defFile? "--def ":"",
	       defFile? defFile :"",
	       objFile? objFile :"" );

    if( opt.noExec )
	printf("Command: '%s'\n", s );
    else {
	Info("(%.100s)", s );
	linkerReturnCode = system(s);
	if( opt.debug > 1)
	    Info("dbg: system(%.100s) returned %d", s, linkerReturnCode );
	if( linkerReturnCode )
	    Error(2,"first dlltool call failed (rc=%d)", linkerReturnCode );
    }

    /* now run the unix linker to produce the base relocation table */
    sprintf(s,"-Wl,--base-file -Wl,%s", baseFile );
    RunUNIXLinker(s);
    if( linkerReturnCode )
	Error(2,"first link failed (rc=%d)", linkerReturnCode );

    sprintf(s,"dlltool %s--as %s%sas --output-exp %s %s%s "
	      "--dllname %s --base-file %s %s%s %s",
	       opt.verbose? "-v ":"",
	       opt.toolprefix? opt.toolprefix : "",
	       opt.toolprefix? "-"            : "",
	       expFile,
	       opt.implibName? "--output-lib ":"",
	       opt.implibName? opt.implibName :"",
	       nameFile,
	       baseFile,
	       defFile? "--def ":"",
	       defFile? defFile :"",
	       objFile? objFile :"" );

    if( opt.noExec )
	printf("Command: '%s'\n", s );
    else {
	Info("(%.100s)", s );
	linkerReturnCode = system(s);
	if( opt.debug > 1)
	    Info("dbg: system(%.100s) returned %d", s, linkerReturnCode );
	if( linkerReturnCode )
	    Error(2,"second dlltool call failed (rc=%d)", linkerReturnCode );
    }

    free(s);
    free(nameFile);
}
#endif /* UNIX */

/****************
 * Create a responsefile and call the linker
 */

static void
CallLinker()
{
    FILE *fp;
    char *filename, *s;
    filelist_t *f;
    t_fplist fpl, fperr;

    fpl = fperr = NULL;
    if( linker == tLD ) {   /* dummies */
	filename = xstrdup("<commandline>");
	fp = stdout;
    }
    else if( opt.noExec ) {
	filename = xstrdup("<responsefile>");
	fp = stdout;
    }
    else {
	if( !(filename = CreateTmpFile( TMP_PREFIX )) )
	    Error(2,GetStr(8));
	fpl = xcalloc( 1, sizeof *fpl );
	fpl->name = filename;
	fpl->next = tempfiles;
	tempfiles = fpl;

	ChangeSlash(filename);
	if( !(fp = fopen( filename, "w" )) )
	    Error(2,GetStr(9), filename);
	fpl->fp = fp;
    }

    s = NULL;
    if( linker == tLINK386 ) {
	BuildMSLink( fp );
	s = xmalloc( 20 + strlen(filename) );
	sprintf(s,"link386 @%s", filename );
    }
    else if( linker == tLINK ) {
	BuildMSLink( fp );
	s = xmalloc( 20 + strlen(filename) );
	sprintf(s,"link @%s", filename );
    }
    else if( linker == tLINKdos ) {
	BuildMSLink( fp );
	s = xmalloc( 20 + strlen(filename) );
	sprintf(s,"link_d  @%s", filename );
    }
    else if( linker == tBLINK ) {
	BuildMSLink( fp );
	s = xmalloc( 20 + strlen(filename) );
	sprintf(s,"blink @%s", filename );
    }
    else if( linker == tBLINKdos ) {
	BuildMSLink( fp );
	s = xmalloc( 20 + strlen(filename) );
	sprintf(s,"blink_d @%s", filename );
    }
    else if( linker == tBLINKx ) {
	BuildMSLink( fp );
	s = xmalloc( 20 + strlen(filename) );
	sprintf(s,"blinkx @%s", filename );
    }
    else if( linker == tWLINK ) {
	fperr = xcalloc( 1, sizeof *fperr );
	if( !(fperr->name = CreateTmpFile( TMP_PREFIX )) )
	    Error(2,GetStr(8));
	fperr->next = tempfiles;
	tempfiles = fperr;
	ChangeSlash(fperr->name);

	BuildWatcomLink( fp );
	s = xmalloc( 40 + strlen(filename) + strlen(fperr->name) );
	sprintf(s,"wlink @%s >%s", filename, fperr->name );
    }
    else if( linker == tLD ) {
	;
    }
    else
	Bug("The Linker for this System is not supported");

    if( linker == tLD )
	RunUNIXLinker(NULL);
    else if( opt.noExec )
	printf("Command: '%s'\n", s );
    else {
	fclose(fp);
	if( fpl )
	   fpl->fp = NULL;
	if( opt.verbose )
	    Info("(%s)", s );
	linkerReturnCode = system(s);
	if( opt.debug > 1)
	    Info("dbg: system(%.100s) returned %d", s, linkerReturnCode );
	if( linkerReturnCode && !opt.ignore ) {
	    /* delete the (incomplete) exe file */
	    for(f=fileTable; f; f = f->next )
		if( f->type == mEXE ) {
		    remove(f->name);
		    break;
		}
	}
	if( linker == tWLINK ) { /* show the error file */
	    int n = 0;

	    free(s);
	    s = xmalloc(1000);
	    fperr->fp = xfopen(fperr->name, "r");
	    while( fgets(s, 999, fperr->fp ) ){
		/* filter out Warnings like :
		 * "Warning(1027): <..> redefinition of _demoModeFlag ignored"
		 * these are due to a watcom compiler bug.
		 * (creates EXTDEF instead of COMDEF records)
		 */
		if( !strncmp(s, "Warning(1027):", 14 ) )
		    n++;
		else
		    fputs(s,stdout);
	    }
	    if( n )
		Info("Watcom Warning(1027) occured %d times", n );
	    fclose(fperr->fp); fperr->fp = NULL;
	}
    }
    free(s);

}



static void
CallResourceLinker()
{
    filelist_t *f;
    const char *exeFile=NULL, *resFile=NULL;
    char *s;

    for(f=fileTable; f; f = f->next )
	if( f->type == mRES ) {
	    resFile = f->name;
	    break;
	}
    if( !resFile ) /* no resource file specified */
	return;
    for(f=fileTable; f; f = f->next )
	if( f->type == mEXE ) {
	    exeFile = f->name;
	    break;
	}
    if( !exeFile )
	BUG();

    s = xmalloc( 20 + strlen(resFile) + strlen(exeFile) );
    sprintf(s,"%s %s %s", resourceCompiler, resFile, exeFile );

    if( opt.noExec )
	printf("Command: '%s'\n", s );
    else {
	Info("(%s)", s );
	linkerReturnCode = system(s);
	if( opt.debug > 1)
	    Info("dbg: system(%.100s) returned %d", s, linkerReturnCode );
    }
    free(s);

}


static void
CallImportLinker()
{
    filelist_t *f;
    const char *defFile=NULL, *outname;
    char *s;
    int emximp;

    for(f=fileTable; f; f = f->next )
	if( f->type == mDEF ) {
	    defFile = f->name;
	    break;
	}
    if( !defFile ) /* no def file specified */
	return;

    if( emximp = !strcmp(target_COMPANY,"emx") ) {
	/* emximp needs ".lib" as extension, but we like ".imp"
	 * (this conflicts with emxs usage of ".imp") -- so we write to an
	 * temp-file and copy it later
	 */
	t_fplist fpl;
	char *p;

	if( !(p = CreateTmpFile3( NULL, TMP_PREFIX, ".lib" )) )
	    Error(2,GetStr(8));
	fpl = xcalloc( 1, sizeof *fpl );
	fpl->name = p;
	fpl->next = tempfiles;
	tempfiles = fpl;
	outname = p;
    }
    else
	outname = opt.implibName;
    s = xmalloc( 30 + strlen(outname) + strlen(defFile) );
    sprintf(s, emximp ? "emximp -o %s %s"  :
			"implib /NOLOGO /NOI %s %s", outname, defFile );

    if( opt.noExec )
	printf("Command: '%s'\n", s );
    else {
	linkerReturnCode = system(s);
	if( opt.debug > 1)
	    Info("dbg: system(%.100s) returned %d", s, linkerReturnCode );
	if( !linkerReturnCode && emximp )
	    CopyFile( outname, opt.implibName );
    }
    free(s);

}


/****************
 * Function to run the UNIX linker via cc
 */

static void
RunUNIXLinker( const char *more_opts )
{
    filelist_t *f;
    size_t n;
    char *commandline, *p;

    /* count total size of all filenames */
    for(n=0,f=fileTable; f; f = f->next )
	n += strlen(f->name) + 1;
    /* add something for the options */
    n += 146;
    if( opt.machine )
	n += strlen(opt.machine) + 4;
    if( opt.linkerOptions )
	n += strlen(opt.linkerOptions);
    if( linker_command )
	n += strlen( linker_command );
    if( opt.toolprefix )
	n += strlen( opt.toolprefix );
    if( more_opts )
	n += strlen( more_opts );
    commandline = xmalloc(n);
    if( opt.machine ) {
	p = stpcpy(commandline,"gcc ");
	if( *opt.machine )
	    p = stpcpy(stpcpy(stpcpy(p,"-b "),opt.machine)," ");
    }
    else if( linker_command && opt.toolprefix )
	p = stpcpy(stpcpy(stpcpy(stpcpy(
		   commandline,opt.toolprefix),"-"),linker_command)," ");
    else if( linker_command )
	p = stpcpy(stpcpy(commandline,linker_command)," ");
    else
	p = stpcpy(commandline,opt.posix ? "c89 " : "cc ");
    if( opt.profile )
	p = stpcpy(p, "-pg ");
    if( createDLL && !strcmp(target_SYSTEM,"mingw32") )
	p = stpcpy(p,"-dll ");
    if( opt.windowapi && !strcmp(target_SYSTEM,"mingw32") )
	p = stpcpy(p,"-windows ");

    if( opt.linkerOptions )
	p = stpcpy(stpcpy(stpcpy(p,"-Wl"),opt.linkerOptions)," ");

    if( more_opts )
	p = stpcpy(stpcpy(p,more_opts)," ");


    p = stpcpy(p,"-o ");
    /* get the run file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mEXE )
	    break;
    p = stpcpy(stpcpy(p," "),f->name);


    /* get the object files */
    for(f=fileTable; f; f = f->next )
	if( f->type == mOBJ )
	    p = stpcpy(stpcpy(p," "),f->name);

    /* get the libraries */
    for(f=fileTable; f; f = f->next )
	if( f->type == mLIB ) {
	    if( !opt.nodefault && strlen(f->name) > 3
					     && !memcmp(f->name,"lib", 3) ) {
		/* simulate unix ld behaviour */
		p = stpcpy(stpcpy(p," -l"),f->name+3);
		if( p[-1] == 'a' && p[-2] == '.' )
		   p -= 2;
		*p = 0;
	    }
	    else
		p = stpcpy(stpcpy(p," "),f->name);
	}
    /* add all the files with module type OTH */
    /* used to pass arbitrary strings to the linker */
    for(f=fileTable; f; f = f->next )
	if( f->type == mOTH )
	    p = stpcpy(stpcpy(p," "),f->name);

    if( opt.noExec )
	printf("Command: '%s'\n", commandline );
    else {
	Info("(%.500s)", commandline );
	linkerReturnCode = system(commandline);
	if( opt.debug > 1)
	    Info("dbg: system(%.400s) returned %d",
				    commandline, linkerReturnCode );
    }
    free(commandline);
}


/****************
 * Function to build a response file for MSDOS Linker or Link386
 */

static void BuildMSLink( FILE *fp )
{
    filelist_t *f;

    /* write the flags */
    if( linker == tBLINKdos || linker == tBLINKx )
	;
    else if( ostype == oMSDOS )
	fprintf(fp,"/NOLOGO %s/NOE ", opt.nocase? "":"/NOI " );
    else
	fprintf(fp,"/NOLOGO %s/NOE /PM:%s ", opt.nocase? "":"/NOI ",
					     opt.windowapi ? "PM":"VIO" );

    if( opt.setDebug )
	fprintf(fp,"/CO ");
    if( opt.nodefault )
	fprintf(fp,"/NOD ");
    if( opt.stackSize )
	fprintf(fp,"/ST:%ld ", opt.stackSize );
    if( opt.forcemap )
	if( linker == tLINK386 )
	    fputs("/MAP:full ", fp);
	else
	    fputs("/MAP ", fp);

    /* get the object files */
    for(f=fileTable; f; f = f->next )
	if( f->type == mOBJ )
	    fprintf(fp,"%s +\n", f->name );
    putc('\n',fp);

    /* get the run file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mEXE )
	    break;
    fprintf(fp,"%s\n", f ? f->name : "" );

    /* get the list file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mMAP )
	    break;
    fprintf(fp,"%s\n", f ? f->name : "nul" );

    /* get the libraries */
    for(f=fileTable; f; f = f->next )
	if( f->type == mLIB )
	    fprintf(fp,"%s +\n", f->name );
    putc('\n',fp);

    /* get the definition file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mDEF )
	    break;
    fprintf(fp,"%s\n", f ? f->name : "" );
}

/****************
 * Function to build a response file for Watcom-LINK
 */

static void BuildWatcomLink( FILE *fp )
{
    filelist_t *f;

    if( opt.verbose )
	fputs("option verbose\n", fp );
    else
	fputs("option quiet\n", fp );
    if( opt.nocase )
	fputs("option nocaseexact\n", fp );
    else
	fputs("option caseexact\n", fp );
    if( opt.setDebug )
	fputs("debug all\n", fp );
    if( ostype == oMSDOS )
	fputs(isDosExtender?"system dos4g\n":"system dos\n", fp );
    else if( ostype == oNETWARE )
	fprintf(fp,"system netware\n" );
    else if( ostype == oWINDOWS )
	fprintf(fp,"system windows%s\n", createDLL? "_dll":"" );
    else if( ostype == oOS2 && !isOS2Version2 )
	fprintf(fp,"system os2%s\n", createDLL? " dll initinstance":""   );
    else if( ostype == oOS2 && opt.windowapi && createDLL )
	fprintf(fp,"system os2v2 dll initinstance\n"  );
    else if( ostype == oOS2 && opt.windowapi )
	fprintf(fp,"system os2v2_pm\n" );
    else
	fprintf(fp,"system os2v2%s\n", createDLL? " dll initinstance":""  );
    if( opt.nodefault )
	fputs("option nodefaultlibs\n", fp );
    if( opt.stackSize )
	fprintf(fp,"option stack=%ld\n", opt.stackSize );
    if( ostype == oOS2 && createDLL )
	fprintf(fp,"option manyautodata\n");
    if( opt.linkerOptions )
	fprintf(fp,"%s\n", opt.linkerOptions);

    /* get the object files */
    for(f=fileTable; f; f = f->next )
	if( f->type == mOBJ )
	    fprintf(fp,"file %s\n", f->name );
    /* get the run file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mEXE )
	    break;
    if( f )
	fprintf(fp,"name %s\n", f->name );

    /* get the list file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mMAP )
	    break;
    if( f )
	fprintf(fp,"option map=%s\n", f->name );

    /* get the libraries */
    for(f=fileTable; f; f = f->next )
	if( f->type == mLIB )
	    fprintf(fp,"library %s\n", f->name );

    /* get the import files */
    for(f=fileTable; f; f = f->next )
	if( f->type == mIMP )
	    fprintf(fp,"import @%s\n", f->name );

    /* get the export files */
    for(f=fileTable; f; f = f->next )
	if( f->type == mEXP )
	    fprintf(fp,"export @%s\n", f->name );

    /* get the definition file */
    for(f=fileTable; f; f = f->next )
	if( f->type == mDEF )
	    break;
    if( f ) {
	FILE *fpin = xfopen( f->name, "r" );
	WatcomExtractFromDef(fp, fpin, "IMPORTS");
	WatcomExtractFromDef(fp, fpin, "EXPORTS");
	WatcomExtractOneFromDef(fp, fpin, "LIBRARY");
	WatcomExtractOneFromDef(fp, fpin, "DESCRIPTION");
	if( ferror(fpin) )
	    Error(2,GetStr(2), f->name);
	fclose(fpin);
    }
}



/****************
 * Die durch Key bestimmten Sektion aus einem def-file herausholen
 * (der Preprocessor ist bereits druebergelaufen) und im Watcomformat
 * auf fpout ausgeben.
 * Der Syntax ist noch sehr eingeschrnkt: Der Key wird nur
 * erkannt, wenn er am Anfang der Zeile steht.
 *
 * Syntax EXPORTS:
 *
 *    entryname [=internalname] [@ord [RESIDENTNAME]]
 *
 * Syntax IMPORTS:
 *
 *    [internalname=]modulename.entry
 *
 */

static void
WatcomExtractFromDef( FILE *fpout, FILE *fpin, const char *key )
{
    char *buffer = xmalloc(256);
    char *p, *p2;
    ulong lnr = 0;
    char *iname, *ename;
    ulong ord=0; /* init to avoid gcc warning */
    int has_ord, has_resname;
    int c;

    if( opt.debug > 1 )
	Info("WatcomExtractFromDef(%s)", key);
    rewind(fpin);
    /* key suchen */
    while( fgets(buffer,255,fpin) ) {
	lnr++;
	if( !(p = strchr(buffer, '\n')) )
	    Error(2,"DEF-file: line %lu is too long", lnr );
	*p = 0;
	StripTrailingWSpaces(buffer);
	if( opt.debug > 2 )
	    Info("line1='%.60s'", buffer);
	if( !strcmp(buffer, key) )
	    break;
    }
    while( fgets(buffer,255,fpin) ) {
	/* (wird ja nur ausgefuehrt, wenn der key gefunden wurde) */
	lnr++;
	if( !(p = strchr(buffer, '\n')) )
	    Error(2,"[.def]:%lu: line too long", lnr );
	*p = 0;
	if( opt.debug > 2 )
	    Info("line2='%.60s'", buffer);
	if( *buffer && !isspace(*(byte*)buffer) )
	    break;  /* Zeichen in der ersten Spalte -- fertig */
	StripWSpaces(buffer);
	iname = ename = NULL;
	has_ord = has_resname = 0;
	p = buffer;
	if( !*p )
	    ;	/* skip empty line */
	else if( !strcmp(key, "EXPORTS") ) {
	    if( *p == '@' || *p == '=' )
		goto syntax_error;
	    ename = p;
	    while( *p && !isspace(*(byte*)p) )
		p++;
	    c = *p;
	    if( isspace(*(byte*)p) ) {
		*p++ = 0;
		while( isspace(*(byte*)p) )
		    p++;
		c = *p;
	    }
	    if( c )
		*p++ = 0;
	    if( c == '=' ) {
		StripLeadingWSpaces(p);
		if( !*p || *p == '@' || *p == '=' )
		    goto syntax_error;
		iname = p;
		while( *p && !isspace(*(byte*)p) )
		    p++;
		c = *p;
		if( isspace(*(byte*)p) ) {
		    *p++ = 0;
		    while( isspace(*(byte*)p) )
			p++;
		    c = *p;
		}
		if( c )
		    *p++ = 0;
	    }
	    if( c == '@' ) {
		if( !*p || *p == '@' || *p == '=' )
		    goto syntax_error;
		ord = strtoul(p, &p2, 10);
		p = p2;
		has_ord = 1;
		StripLeadingWSpaces(p);
		if( !strcmp(p,"RESIDENTNAME") ) {
		    has_resname = 1;
		    p += 12;
		}
	    }
	    if( *p )
		goto syntax_error;

	    fprintf(fpout,"export %s", ename );
	    if( has_ord )
		fprintf(fpout,".%lu", ord );
	    if( iname )
		fprintf(fpout,"=%s", iname );
	    if( has_resname )
		fputs(" RESIDENT", fpout);
	    putc('\n', fpout);
	}
	else if( !strcmp(key, "IMPORTS") ) {
	    if( p2 = strchr( p, '=' ) ) {
		*p2 = 0;
		StripWSpaces(p);
		iname = p;
		p = p2+1;
	    }
	    StripWSpaces(p);
	    if( !*p || strpbrk( p, " \t=@" ) )
		goto syntax_error;
	    ename = p;

	    fputs("import", fpout);
	    if( iname )
		fprintf(fpout," %s", iname );
	    fprintf(fpout," %s\n", ename );
	}
	else
	    BUG();
    }

    free(buffer);
    return;
  syntax_error:
    Error(2,"[.def]:%lu: syntax error", lnr );
}


static void WatcomExtractOneFromDef( FILE *fpout, FILE *fpin, const char *key )
{
    char *buffer = xmalloc(256);
    char *p, *p2;
    ulong lnr = 0;

    if( opt.debug > 1 )
	Info("WatcomExtractOneFromDef(%s)", key);
    rewind(fpin);
    /* key suchen */
    while( fgets(buffer,255,fpin) ) {
	lnr++;
	if( !(p = strchr(buffer, '\n')) )
	    Error(2,"DEF-file: line %lu is too long", lnr );
	*p = 0;
	if( opt.debug > 2 )
	    Info("line='%.60s'", buffer);
	if( (p = strpbrk( buffer, " \t=" )) )
	    *p++ = 0;
	else
	    p = "";
	if( !strcmp(buffer, key) ) {
	    if( !strcmp(key, "DESCRIPTION") ) {
		if( !(p = strchr( p, '\'' )) )
		    goto syntax_error;
		if( !(p2 = strchr( p+1, '\'' )) || p2[1] )
		    goto syntax_error;
		fprintf(fpout,"option description %s\n", p);
	    }
	    else if( !strcmp(key, "LIBRARY") ) {
		while( isspace( *p ) )
		    p++;
		if( p2 = strpbrk( p, " \t" ) )
		    *p2 = 0;
		fprintf(fpout,"option modname=%s\n", p);
	    }
	    else
		BUG();
	    break;
	}
    }

    free(buffer);
    return;
  syntax_error:
    Error(2,"[.def]:%lu: syntax error", lnr );
}


/****************
 * testen ob der angegebene File (aus der filetable)
 * Preprocessoranweisung hat. Wenn dem so ist, diesen
 * Preprocessen und den filenamen durch einen temp. File ersetzen
 * Folgende Erweiterungen des Files sind zulssing:
 * Anweisungen stehen immer in der ersten Spalte und beginnen
 * mit einem Prozentzeichen:
 *	 %<blank> Kommentar
 *	 %if <bedingung>
 *	 %elif <bedingung>
 *	 %else
 *	 %endif
 *	 %%
 * Kommentarzeilen werden vollstaending entfernt.
 * "%%" ist keine Preprocessorline, sondern wird durch ein einzelnes %
 * ersetzt.
 * Bedingung kann zu Zeit nur sein:
 *    OS2	::= fuer OS/2 Def-Files
 *    WINDOWS3	::= fuer Windows Def-Files
 * Desweiteren werden Makros innerhalb einer Zeile ersetzt,
 * diese werden "$(makroname)" notiert, wobei der Makroname casesensitive
 * ist. Die Sequence "$(" dient z.Z. dem erkenne des Makros (Dies wird
 * irgendwanneinmal auf normalen Shell-Syntax umgestellt, z.Z. reicht
 * dieses Verfahren aber aus und es ist auch nicht ntigt, Strings zu erkennen
 * um in diesen keine Expansion vorzunehmen -- in Strings kommen z.Z. nur
 * Ident Anweisungen vor und diese entahlten kein "$(".
 * Die Lnge eines Makronamens ist z.Z. auf 255 Zeichen limitiert.
 *
 * Verschachtelungen mit Else sind z.Z. noch nicht impl.
 * Returns: neue filelist oder NULL wenn nicht notwendig
 */

static filelist_t *PreprocessFile( filelist_t *f )
{
    FILE *fp, *outfp;
    int lfseen, c;
    char *outname;
    long line;
    t_fplist fpl;
    filelist_t *f2;
    enum {
	sCOPY,	    /* copy characters to the outstream */
	sCOPYNEXT,  /* set state sCOPY for the next character */
	sSKIP,	    /* do not copy */
	sPERCENT,   /* PERCENT seen in first column */
	sCONDLINE_NO,  /* This is a line of type "%?name " */
	sCONDLINE_YES  /* This is a line of type "%?name " */
    } state;
    enum {
      cUNKNOWN = 0,
      cCOMMENT,
      cTRUE,
      cFALSE,
      cIF,
      cELIF,
      cELSE,
      cENDIF,
      cIN_KEYWORD,
      cIN_IF,
      cIN_ELIF,
      cIN_ELSE,
      cERROR
    } cond=0, lastcond;
    char buf[20];
    char macrobuf[256];
    int macroidx;
    int idx=0;

    fp = xfopen( f->name, "r" );
    /* erstmal sehen, ob denn ueberhaupt preprocess anweisungen vorhanden sind*/
    lfseen = 1;
    state = sCOPY;
    while( (c=getc(fp)) != EOF ) {
	if( lfseen && c == '%' ) {
	    state = sPERCENT;
	    break;
	}
	lfseen = c == '\n';
    }
    if( ferror(fp) )
	Error(2,GetStr(2), f->name);
    if( state == sCOPY ) {  /* nothing to do */
	fclose(fp);
	return NULL;
    }

    /* jetzt mal los */
    if( !(outname = CreateTmpFile3( NULL, TMP_PREFIX, ".def" )) )
	Error(2,GetStr(8));
    fpl = xcalloc( 1, sizeof *fpl );
    fpl->name = outname;
    fpl->next = tempfiles;
    tempfiles = fpl;
    if( !(outfp = fopen( outname, "w" )) )
	Error(2,GetStr(9), outname);
    fpl->fp = outfp;

    rewind(fp);
    lfseen = 1;
    state = sCOPY;
    lastcond = cUNKNOWN;
    line = 1;
    macroidx = -1;
    while( (c=getc(fp)) != EOF ) {
	if( c == '\n' )
	    line++;
	switch( state ) {
	  case sCOPYNEXT:
	    state = sCOPY;
	    /* fall thru */
	  case sCOPY:
	  case sSKIP:
	    if( lfseen && c == '%' ) {
		state = sPERCENT;
		cond = cUNKNOWN;
	    }
	    lfseen = c == '\n';
	    break;

	  case sCONDLINE_YES:
	  case sCONDLINE_NO:
	    if( c == '\n' ) {
		lfseen = 1;
		state = sCOPY;
	    }
	    break;

	  case sPERCENT:
	    if( c == '\n' ) {
		lfseen = 1;
		if( cond == cIN_KEYWORD ) {
		    buf[idx] = 0;
		    if( !strcmp( buf, "else" ) )
			cond = cELSE;
		    else if( !strcmp( buf, "endif" ) )
			cond = cENDIF;
		}
		switch( cond ) {
		  case cCOMMENT:
		  case cUNKNOWN:
		    state = sCOPYNEXT; /* not the linefeed */
		    break;
		  case cIN_IF:
		  case cIN_ELIF:
		    buf[idx]= 0;
		    /* fall thru */
		  case cIF:
		  case cELIF:
		    if( !strcmp( buf, "OS2" ) || !strcmp(buf, "OS20") )
			idx = 1;
		    else if( !strcmp(buf, "WINDOWS3")||!strcmp(buf, "MSDOS") )
			idx = 2;
		    else
			Error(2,"%s:%ld: unknown ldsh-preprocess operator '%s'",
						f->name, line, buf);
		    if( (idx==1 && ostype == oOS2 ) ||
			(idx==2 && (ostype == oMSDOS ||ostype==oWINDOWS)
						    ) ) {
			cond = cTRUE;
			state = sCOPYNEXT;
		    }
		    else {
			cond = cFALSE;
			state = sSKIP;
		    }
		    lastcond = cond;
		    break;

		  case cELSE:
		    if( lastcond == cTRUE )
			state = sSKIP;
		    else if( lastcond == cFALSE )
			state = sCOPYNEXT;
		    else
			Error(2,"%s:%ld: confused about this ELSE",
							   f->name, line);
		    lastcond = cond;
		    break;
		  case cENDIF:
		    state = sCOPYNEXT;
		    break;
		  default:
		    Error(2,"%s:%ld: invalid ldsh-preprocess line",
							   f->name, line);
		}
	    }
	    else {
		switch( cond )	{
		  case cUNKNOWN:  /* first time (2. character on line ) */
		    if( isspace(c) )
			cond = cCOMMENT;
		    else if( c == '%' )  /* keine Preprocessor line */
			state = sCOPY;
		    else  {
			idx = 0;
			buf[idx++] = c;
			cond = cIN_KEYWORD;
		    }
		    break;

		  case cIN_KEYWORD:
		    if( isspace(c) ) {
			buf[idx] = 0;
			idx = 0;    /* used later on */
			if( *buf == '?' && buf[1] == '!' && buf[2] ) {
			    cond = cUNKNOWN;
			    state= !strcmp( buf+2, target_SYSTEM ) ?
				      sCONDLINE_NO : sCONDLINE_YES;
			}
			else if( *buf == '?' && buf[1] ) {
			    cond = cUNKNOWN;
			    state= !strcmp( buf+1, target_SYSTEM ) ?
				      sCONDLINE_YES : sCONDLINE_NO;
			}
			else if( !strcmp( buf, "if" ) )
			    cond = cIN_IF;
			else if( !strcmp( buf, "elif" ) )
			    cond = cIN_ELIF;
			else if( !strcmp( buf, "else" ) )
			    cond = cELSE;
			else if( !strcmp( buf, "endif" ) )
			    cond = cENDIF;
			else
			    cond = cERROR;
		    }
		    else if( idx < DIM(buf)-1 )
			buf[idx++] = c;
		    else
			cond = cERROR;
		    break;

		  case cIN_IF:
		  case cIN_ELIF:
		    if( isspace(c) ) {
			if( !idx )
			    buf[idx++] = c;
			else {
			    buf[idx] = 0;
			    cond = cond == cIN_IF?  cIF : cELIF;
			}
		    }
		    else if( idx < DIM(buf)-1 )
			buf[idx++] = c;
		    else
			cond = cERROR;
		    break;


		  case cCOMMENT:
		  case cERROR:
		    break;

		  case cIF:
		  case cELIF:
		  case cELSE:
		  case cENDIF:
		    if( !isspace(c) )
			cond = cERROR;
		    break;

		  default: Info("%d %d", cond, state);BUG();
		}

	    }
	    break;

	  default: BUG();
	}
	if( state == sCOPY || state == sCONDLINE_YES ) {
	    if( macroidx != -1 ) {
		if( macroidx == -2 ) {
		    if( c == '(' )
			macroidx = 0; /* start copying */
		    else {
			putc('$', outfp);
			putc(c, outfp);
			macroidx = -1;
		    }
		}
		else if( macroidx >= DIM(macrobuf)-2)
		    Error(2,"%s:%ld: macro too long", f->name, line);
		else if( c == ')' ) { /* ready , substitute macro */
		    MACRO m;
		    const char *s;

		    macrobuf[macroidx] = 0;
		    macroidx = -1;
		    for(m=macrolist; m; m = m->next )
			if( !strcmp(m->name, macrobuf) ) {
			    fputs(m->value, outfp);
			    break;
			}
		    if( !m ) { /* not on cmdline - try environment */
			if( s = getenv(macrobuf) )
			    fputs(s, outfp);
		    }
		}
		else
		    macrobuf[macroidx++] = c;
	    }
	    else if( c == '$' )
		macroidx = -2;
	    else
		putc(c, outfp);
	}
    }
    if( macroidx != -1 )
	Error(2,"%s:%ld: unterminated macro", f->name, line);
    if( ferror(fp) )
	Error(2,GetStr(2), f->name);
    fclose(fp);
    fclose(outfp);
    if( fpl )
	fpl->fp = NULL;

    /* und den Namen des deffiles ersetzen */
    f2 = xcalloc( 1, sizeof *f2 + strlen(outname) );
    strcpy( f2->name, outname );
    ChangeSlash( f2->name );
    f2->type = f->type;
    return f2;
}


static void AddInfoString(const char*string)
{
    size_t n;

    if( !infoStringArea ) {
	infoStringArea = xmalloc(infoStringAreaSize=200);
	infoStringAreaUsed=0;
    }
    n = strlen(string)+1;
    if( infoStringAreaUsed + n >= infoStringAreaSize )
	infoStringArea = xrealloc(infoStringArea, infoStringAreaSize += n+200);
    memcpy(infoStringArea+infoStringAreaUsed, string, n);
    infoStringAreaUsed += n;
}




/****************
 * Create the Linke Info Object file
 * Returns: name of created object file
 */

static const char *
CreateInfoObj( const char *infostring, size_t infostringlen)
{
    t_fplist f;
    ulong datalen, ul;
    ushort us;
    FILE *fpcfg;
    char *line=NULL, *p, *p2;
    int c, any, okay, any2;
    char *libid, *filter;
    const char *s;

    filter = NULL;
    libid = CreateTargetSpec();
    if( opt.verbose )
	Info("Looking for .LinkerInfo=%s", libid);
    /* find the section in the configuration file */
    if( opt.configfile )
	fpcfg = xfopen( opt.configfile, "r" );
    else {
	if( !(p=efopen2(
			#ifdef UNIX
			    "=.:/usr/local/lib:/usr/lib"
			#else
			    ""
			#endif
			     , "ldsh.cfg", "r", &fpcfg)) ) {
	    Error(0,"warning: configuration file 'ldsh.cfg' not found");
	    free(libid);
	    return NULL;
	}
	if( opt.verbose )
	    Info("in configuration file '%s'", p );
	free(p);
    }
    line = xmalloc(1000);
    while( fgets(line, 999, fpcfg ) ) {
	if( *line != '.' )
	    continue;
	if( !(p=strchr(line+1,'=')) )
	    continue;
	*p++ = 0;
	if( strcmp(line+1, "LinkerInfo") )
	    continue;
	if( p2 = strpbrk(p," \t\n\r") )
	    *p2 = 0;
	if( !strcmp(p, libid) )
	    break;
    }
    if( ferror(fpcfg) )
	Error(2,GetStr(2), "ldsh.cfg");
    if( feof(fpcfg) ) {
	Info("linker_info not available for '%s'", libid);
	fclose(fpcfg);
	free(line);
	free(libid);
	return NULL;
    }

    datalen = infostringlen+3;

    f = xcalloc( 1, sizeof *f );
    if( !(f->name = CreateTmpFile( TMP_PREFIX )) )
	Error(2,GetStr(8));
    f->next = tempfiles;
    tempfiles = f;
    if( !(f->fp = fopen( f->name, "wb" )) )
	Error(2,GetStr(9), f->name);

    FREE(filter);
    any = any2 = okay = 0;
    while( fgets(line, 999, fpcfg ) ) {
	if( *line == '#' )
	    continue;
	if( *line == '.' ) {
	    if( !any2 && !strncmp(line+1, "LinkerInfo",10) )
		continue; /* skip other LinkerInfos*/
	    any2 = 1;
	    if( !strncmp(line+1,"filter",6) ) {
		if( any )
		    Error(2,".filter in invalid context");
		filter = xstrdup(line+8);
		StripWSpaces(filter);
		continue;
	    }
	    else if( filter ) {
		if( !strncmp( line+1, "retlif", 6 ) ) {
		    okay++;
		    break;
		}
	    }
	    else
		break;
	}
	any2 = 1;
	/* process this line */
	if( filter ) {
	    int esc = 0;
	    for(p=line;*p;p ++)  {
		if( esc ) {
		    switch(*p) {
		      case 's': {
			    size_t n;

			    fputs(".string \"", f->fp);
			    s = infostring;
			    for(n=0; n < infostringlen ; n++, s++ ) {
				if( !*s )
				    fputs("\"\n\t.string \"", f->fp);
				else
				    putc(*s,f->fp);
			    }
			    fputs("\"\n", f->fp);

			    /* forget about the rest of the line */
			    p = line + strlen(line) -1;
			}
			break;
		      case 'b': {
			    size_t n;
			    int i;

			    fputs(".byte ", f->fp);
			    s = infostring;
			    i=0;
			    for(n=0; n < infostringlen ; n++, s++ ) {
				if( ++i > 10 ) {
				    fputs("\n\t.byte ",f->fp);
				    i = 1;
				}
				else if( i > 1 )
				    putc(',',f->fp);
				fprintf(f->fp,"%#02x", *s);
			    }
			    putc('\n',f->fp);

			    /* forget about the rest of the line */
			    p = line + strlen(line) -1;
			}
			break;
		      case '@':
			putc('@',f->fp);
			break;
		      default:
			putc('@',f->fp);
			putc(*p,f->fp);
		    }
		    esc = 0;
		}
		else if( *p == '@' )
		    esc=1;
		else
		    putc(*p,f->fp);
	    }
	}
	else {
	    any = 1;
	    for(p=line;;p += 2) {
		while(isspace(*p))
		    p++;
		if( !*p )
		    break;
		if( !p[1] || !(!p[2] || isspace(p[2])) )
		    Error(2,"syntax error in LinkerInfo section of 'ldsh.cfg'");
		if( *p == ':' ) { /* controlcode */
		    switch(p[1]) {
		      case '0':
			us = datalen;
			fwrite( &us, 2, 1, f->fp );
			break;
		      case '1':
			fwrite( &datalen, 4, 1, f->fp );
			break;
		      case '2':
			us = 4 + datalen;
			fwrite( &us, 2, 1, f->fp );
			break;
		      case '3':
			us = 6 + datalen;
			fwrite( &us, 2, 1, f->fp );
			break;
		      case '4':
			fwrite( infostring, infostringlen, 1, f->fp );
			putc(0,f->fp);
			putc(0,f->fp);
			putc(0,f->fp);
			break;
		      case '5':
			putc(0,f->fp); /* checksum (0 is always a valid one)*/
			break;
		      case '6':
			ul = (datalen + 3) & ~3u;  /* round up */
			fwrite( &ul, 4, 1, f->fp );
			break;
		      case '7':
			fwrite( infostring, infostringlen, 1, f->fp );
			putc(0,f->fp);
			putc(0,f->fp);
			putc(0,f->fp);
			/* write trailing bytes */
			ul = (datalen + 3) & ~3u;  /* round up */
			for( ; ul > datalen ; ul-- )
			    putc(0x90,f->fp);
			break;
		      default:
			Error(2,"invalid controlcode '%c'"
				" in LinkerInfo section of 'ldsh.cfg'", p[1]);
		    }
		}
		else if( isxdigit(p[0]) && isxdigit(p[1]) ) { /* normal hex-byte */
		    c = strtoul(p,NULL,16);
		    putc(c,f->fp);
		}
		else
		    Error(2,"invalid value in LinkerInfo section of 'ldsh.cfg'");
	    }
	}
    }
    if( ferror(fpcfg) )
	Error(2,GetStr(2), "ldsh.cfg");
    if( filter && !okay )
	Error(2,"unclosed .filter block");
    fclose(fpcfg);
    free(line);
    if( ferror(f->fp) )
	Error(2,GetStr(3), f->name);
    fclose(f->fp); f->fp = NULL;
    free(libid);
    if( filter ) {
	t_fplist f2;
	char *buf;
	const char *s1, *s2;

	f2 = xcalloc( 1, sizeof *f2 );
	if( !(f2->name = CreateTmpFile( TMP_PREFIX )) )
	    Error(2,GetStr(8));
	f2->next = tempfiles;
	tempfiles = f2;

	buf = xmalloc(strlen(filter) +strlen(f2->name) + strlen(f->name) + 20);
	p = buf;
	if( !(s1=strstr(filter, "%1")) )
	    Error(2,"arg 1 missing in '%s'", filter);
	if( !(s2=strstr(filter, "%2")) )
	    Error(2,"arg 2 missing in '%s'", filter);
	if( s1 < s2 ) {
	    memcpy(p,filter,s1-filter);
	    p = stpcpy(p+(s1-filter),f->name);
	    memcpy(p,s1+2,s2-s1);
	    p = stpcpy(p+(s2-s1-2),f2->name);
	    memcpy(p,s2+2,filter+strlen(filter)-s2);
	}
	else {
	    memcpy(p,filter,s2-filter);
	    p = stpcpy(p+(s2-filter),f2->name);
	    memcpy(p,s2+2,s1-s2);
	    p = stpcpy(p+(s1-s2-2),f->name);
	    memcpy(p,s1+2,filter+strlen(filter)-s1);
	}
	if( opt.debug)
	    Info("running: %s",buf);
	if( system(buf) )
	    Error(2,"cmd '%s' failed", buf);
	free(buf);
	free(filter);
	return f2->name;
    }
    else
	return f->name;
}

/****************
 * we use the target to extract the libid, by looking for the last occurence
 * of TARGET in an object filename and than use theis sting without the
 * filename.
 * Returns: malloced string
 */

static char *
CreateTargetSpec()
{
    filelist_t *f;
    const char *s, *s1;
    char *p, *p2;

    s = NULL;
    /* get an object files */
    for(f=fileTable; f && !s ; f = f->next )
	if( f->type == mOBJ ) {
	    /* case insensitive because of MSDOS filesystem ... */
	    for(s1=f->name; s1 = stristr(s1,target); s = s1, *s1? s1++ : s)
		;
	}
    if( !s ) {
	Error(0,"warning: target not found in any .o; trying the libs");
	/* or try a library */
	for(f=fileTable; f && !s ; f = f->next )
	    if( f->type == mLIB ) {
		/* case insensitive because of MSDOS filesystem ... */
		for(s1=f->name; s1 = stristr(s1,target); s = s1, *s1? s1++ : s)
		    ;
	    }
    }
    if( !s )
	Error(2,"oops, didn't found the target in any file name");
    p = xstrdup(s);
    if( opt.backslash )
	for(p2=p; *p2; p2++ )
	    if( *p2 == '\\' )
		*p2 = '/';

    if( p2=strrchr(p,'/') )
	*p2=0;
    /* remove a trailing "-d", because Debug versions are usualy the same */
    if( strlen(p) > 2 && !stricmp(p+strlen(p)-2,"-d") )
	p[strlen(p)-2] = 0;
    return p;
}


static void
CopyFile( const char *src, const char *dst )
{
    FILE *fp1, *fp2;
    int c;

    fp1 = xfopen(src,"rb");
    fp2 = xfopen(dst,"wb");

    while( (c=getc(fp1)) != EOF )
	putc(c, fp2);
    if( ferror(fp1) )
	Error(2,GetStr(2), src);
    if( ferror(fp2) )
	Error(2,GetStr(3), dst);

    fclose(fp1);
    fclose(fp2);
}

/*** bottom of file ***/
