/* [ipf2doc.c wk 25.9.92] IPF Sourcen in Plain Text konvertieren
 *	Copyright (c) 1992 by Werner Koch (dd9jn)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 20.12.92 wk	Um Index erweitert
 * 21.07.94 wk	Indexfilename in Index ist natrlich unsinn.
 */

#include <wk/tailor.h>
RCSID("$Id: ipf2doc.c,v 1.4 1996/05/19 17:00:16 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/file.h>
#include <wk/string.h>

/****** constants *********/
#define MAX_LINELEN   200
#define MAX_HDRLEVELS  6
#define MAX_INDICES   300
#define MAX_TAGLEN   100
#define MAGIC_USERDOC_ARG 9876
#define NOBRKSPACE   '\xff'

static struct {
	const char *name;  /* String excluding the & and the . */
	const byte  code;  /* Code (compilers codepage) */
    } symbolTable[] = {
/* acute		*/  { "aa"        , ''                     },
/* a circumflex 	*/  { "ac"        , ''                     },
/* a umlaut		*/  { "ae"        , ''                     },
/* A umlaut		*/  { "Ae"        , ''                     },
/* a grave		*/  { "ag"        , ''                     },
/* ae ligature		*/  { "aelig"     , ''                     },
/* AE ligature		*/  { "AElig"     , ''                     },
/* Alpha		*/  { "Alpha"     , 'A'                     },
/* Ampersand		*/  { "amp"       , '&'                     },
/* and			*/  { "and"       , '^'                     },
/* angstrom		*/  { "angstrom"  , ''                     },
/* a overcircle 	*/  { "ao"        , ''                     },
/* A overcircle 	*/  { "Ao"        , ''                     },
/* Apostrophe		*/  { "apos"      , '\''                    },
/* ASCII code 185	*/  { "bx2022"    , ''                     },
/* ASCII code 186	*/  { "bx2020"    , ''                     },
/* ASCII code 187	*/  { "bx0022"    , ''                     },
/* ASCII code 188	*/  { "bx2002"    , ''                     },
/* ASCII code 200	*/  { "bx2200"    , ''                     },
/* ASCII code 201	*/  { "bx0220"    , ''                     },
/* ASCII code 202	*/  { "bx2202"    , ''                     },
/* ASCII code 203	*/  { "bx0222"    , ''                     },
/* ASCII code 204	*/  { "bx2220"    , ''                     },
/* ASCII code 205	*/  { "bx0202"    , ''                     },
/* ASCII code 206	*/  { "bx2222"    , ''                     },
/* Asterisk		*/  { "asterisk"  , '*'                     },
/* At sign		*/  { "atsign"    , '@'                     },
/* Back slash		*/  { "bslash"    , '\\'                    },
/* Back slash		*/  { "bsl"       , '\\'                    },
/* Beta 		*/  { "Beta"      , ''                     },
/* box ascender 	*/  { "bxas"      , ''                     },
/* box ascender 	*/  { "bxbj"      , ''                     },
/* box cross		*/  { "bxcr"      , ''                     },
/* box cross		*/  { "bxcj"      , ''                     },
/* box descender	*/  { "bxde"      , ''                     },
/* box descender	*/  { "bxtj"      , ''                     },
/* box horizontal	*/  { "bxh"       , ''                     },
/* box lower-left	*/  { "bxll"      , ''                     },
/* box lower-right	*/  { "bxlr"      , ''                     },
/* box right junction	*/  { "bxri"      , ''                     },
/* box right junction	*/  { "bxrj"      , ''                     },
/* box upper-left	*/  { "bxul"      , ''                     },
/* box upper-right	*/  { "bxur"      , ''                     },
/* box vertical 	*/  { "bxv"       , ''                     },
/* c cedilla		*/  { "cc"        , ''                     },
/* C cedilla		*/  { "Cc"        , ''                     },
/* Caret symbol 	*/  { "caret"     , '^'                     },
/* Close double quote	*/  { "cdq"       , '\"'                    },
/* Close French dblquote*/  { "cdqf"      , ''                     },
/* Close single quote	*/  { "csq"       , '\''                    },
/* Comma		*/  { "comma"     , ','                     },
/* Colon		*/  { "colon"     , ':'                     },
/* Dash 		*/  { "dash"      , '-'                     },
/* degree		*/  { "degree"    , ''                     },
/* degree		*/  { "deg"       , ''                     },
/* divide		*/  { "divide"    , ''                     },
/* Dollar sign		*/  { "dollar"    , '$'                     },
/* dot			*/  { "dot"       , ''                     },
/* Down arrow		*/  { "darrow"    , ''                     },
/* e acute		*/  { "ea"        , ''                     },
/* E acute		*/  { "Ea"        , ''                     },
/* e circumflex 	*/  { "ec"        , ''                     },
/* e umlaut		*/  { "ee"        , ''                     },
/* e grave		*/  { "eg"        , ''                     },
/* Em dash		*/  { "emdash"    , '-'                     },
/* En dash		*/  { "endash"    , '-'                     },
/* Equal sign		*/  { "eq"        , '='                     },
/* Equal sign		*/  { "equals"    , '='                     },
/* Equal sign		*/  { "eqsym"     , '='                     },
/* Exclamation point	*/  { "xclm"      , '!'                     },
/* Exclamation point	*/  { "xclam"     , '!'                     },
/* function of		*/  { "fnof"      , ''                     },
/* Greater than 	*/  { "gtsym"     , '>'                     },
/* Greater than 	*/  { "gt"        , '>'                     },
/* Greater than 	*/  { "gesym"     , '>'                     },
/* House		*/  { "house"     , ''                     },
/* Hyphen		*/  { "hyphen"    , '-'                     },
/* i acute		*/  { "ia"        , ''                     },
/* i circumflex 	*/  { "ic"        , ''                     },
/* i umlaut		*/  { "ie"        , ''                     },
/* i grave		*/  { "ig"        , ''                     },
/* inverted exclamation */  { "inve"      , ''                     },
/* inverted question	*/  { "invq"      , ''                     },
/* Left arrow		*/  { "larrow"    , ''                     },
/* Left brace		*/  { "lbrace"    , '{'                     },
/* Left brace		*/  { "lbrc"      , '{'                     },
/* Left bracket 	*/  { "lbracket"  , '['                     },
/* Left bracket 	*/  { "lbrk"      , '['                     },
/* Left parenthesis	*/  { "lpar"      , '('                     },
/* Left parenthesis	*/  { "lparen"    , '('                     },
/* logical not		*/  { "lnot"      , ''                     },
/* M dash		*/  { "mdash"     , '-'                     },
/* Minus sign		*/  { "minus"     , '-'                     },
/* Mu			*/  { "mu"        , ''                     },
/* N dash		*/  { "ndash"     , '-'                     },
/* n tidle		*/  { "nt"        , ''                     },
/* N tidle		*/  { "Nt"        , ''                     },
/* not symbol		*/  { "lnot"      , ''                     },
/* not symbol		*/  { "notsym"    , ''                     },
/* Number sign		*/  { "numsign"   , '#'                     },
/* o acute		*/  { "oa"        , ''                     },
/* o circumflex 	*/  { "oc"        , ''                     },
/* o grave		*/  { "og"        , ''                     },
/* o umlaut		*/  { "oe"        , ''                     },
/* O umlaut		*/  { "Oe"        , ''                     },
/* one fourth		*/  { "frac14"    , ''                     },
/* one half		*/  { "frac12"    , ''                     },
/* Open double quote	*/  { "odq"       , '\"'                    },
/* Open French dblquote */  { "odqf"      , ''                     },
/* Open single quote	*/  { "osq"       , '`'                     },
/* Percent		*/  { "percent"   , '%'                     },
/* Period		*/  { "per"       , '.'                     },
/* Plus sign		*/  { "plus"      , '+'                     },
/* plusminus		*/  { "plusmin"   , ''                     },
/* plusminus		*/  { "pm"        , ''                     },
/* pound sterling	*/  { "Lsterling" , ''                     },
/* Required blank	*/  { "rbl"       , ' '                     },
/* Right brace		*/  { "rbrace"    , '}'                     },
/* Right brace		*/  { "rbrc"      , '}'                     },
/* Right bracket	*/  { "rbracket"  , ']'                     },
/* Right bracket	*/  { "rbrk"      , ']'                     },
/* Right parenthesis	*/  { "rpar"      , ')'                     },
/* Right parenthesis	*/  { "rparen"    , ')'                     },
/* Semicolon		*/  { "semi"      , ';'                     },
/* shaded box 1/4 dots	*/  { "box14"     , ''                     },
/* shaded box 1/2 dots	*/  { "box12"     , ''                     },
/* shaded box 3/4 dots	*/  { "box34"     , ''                     },
/* Slash		*/  { "slash"     , '/'                     },
/* Slash		*/  { "slr"       , '/'                     },
/* solid box		*/  { "BOX"       , ''                     },
/* solid box bottomhalf */  { "BOXBOT"    , ''                     },
/* Split vertical bar	*/  { "splitvbar" , '|'                     },
/* square bullet	*/  { "sqbul"     , ''                     },
/* superscript 2	*/  { "sup2"      , ''                     },
/* Tilde		*/  { "tilde"     , '~'                     },
/* u acute		*/  { "ua"        , ''                     },
/* u circumflex 	*/  { "uc"        , ''                     },
/* u grave		*/  { "ug"        , ''                     },
/* u umlaut		*/  { "ue"        , ''                     },
/* U umlaut		*/  { "Ue"        , ''                     },
/* Underscore		*/  { "us"        , '_'                     },
/* underscored a	*/  { "aus"       , ''                     },
/* underscored o	*/  { "ous"       , ''                     },
/* y umlaut		*/  { "ye"        , ''                     },
	{ NULL, 0 }
    };

static const char whiteSpaces[] = " \t\n\f";

/******** typedefs ********/
typedef size_t atom_t;

typedef struct {
	ushort h[MAX_HDRLEVELS];  /* gueltig sind alle > 0 */
	atom_t id, title;
	ushort lnr, nlines;	  /* linenumber (1...), # of lines */
    } indexTbl_t;


/****** prototypes ********/
static void Initialize(void);
static void Err( const char *str );
static void Wrn( const char *str );
static void PreProcess(FILE *, const char *fname);
static void Process( int c );
static void ProcessChar(int c);
static void ProcessStr(const char *s);
static void FlushLine(void);
static void AddNewLine(void);
static void PutNewLine(void);
static void BeginTag(void);
static void EndTag(void);
static void EndSymbol(void);
static void Tag_template(int,char*,int);
static void Tag_h(int,char*,int);
static void Tag_p(int,char*,int);
static void Tag_userdoc(int,char*,int);
static void Tag_list(int,char*,int);
static void Tag_listitem(int,char*,int);
static void Tag_note(int,char*,int);
static void Tag_lines(int,char*,int);
static void Tag_parml(int,char*,int);
static void Tag_pitem(int,char*,int);

static void PrintIndex(void);
static void WriteIndexFile(void);

static void MakeAtomTable( size_t size );
static void DestroyAtomTable(void);
static atom_t AddAtom( const char *s );
static atom_t GetAtom( const char *name );
static const char *GetAtomName( atom_t hd );

/******* globals **********/

static struct {
	const char *name;  /* String excluding the : and the . */
	void (*fnc)(int, char *, int); /* pointer to function */
	const int arg;	  /* arg to pass to function */
    } tagTable[] = {
	{ "acviewport"  , Tag_template              , 0   },
	{ "artlink"     , Tag_template              , 0   },
	{ "artwork"     , Tag_template              , 0   },
	{ "caution"     , Tag_template              , 0   },
	{ "cgraphic"    , Tag_lines                 , 0   },
	{ "color"       , Tag_template              , 0   },
	{ "ctrl"        , Tag_template              , 0   },
	{ "ctrldef"     , Tag_template              , 0   },
	{ "ddf"         , Tag_template              , 0   },
	{ "dl"          , Tag_template              , 0   },
	{ "docprof"     , Tag_template              , 0   },
	{ "fig"         , Tag_template              , 0   },
	{ "figcap"      , Tag_template              , 0   },
	{ "font"        , Tag_template              , 0   },
	{ "fn"          , Tag_template              , 0   },
	{ "h1"          , Tag_h                     , 1   },
	{ "h2"          , Tag_h                     , 2   },
	{ "h3"          , Tag_h                     , 3   },
	{ "h4"          , Tag_h                     , 4   },
	{ "h5"          , Tag_h                     , 5   },
	{ "h6"          , Tag_h                     , 6   },
	{ "hide"        , Tag_template              , 0   },
	{ "hp1"         , Tag_template              , 1   },
	{ "hp2"         , Tag_template              , 2   },
	{ "hp3"         , Tag_template              , 3   },
	{ "hp4"         , Tag_template              , 4   },
	{ "hp5"         , Tag_template              , 5   },
	{ "hp6"         , Tag_template              , 6   },
	{ "hp7"         , Tag_template              , 7   },
	{ "hp8"         , Tag_template              , 8   },
	{ "hp9"         , Tag_template              , 9   },
	{ "i1"          , Tag_template              , 1   },
	{ "i2"          , Tag_template              , 2   },
	{ "icmd"        , Tag_template              , 0   },
	{ "isyn"        , Tag_template              , 0   },
	{ "li"          , Tag_listitem              , 0   },
	{ "lines"       , Tag_lines                 , 1   },
	{ "link"        , Tag_template              , 0   },
	{ "lm"          , Tag_template              , 0   },
	{ "lp"          , Tag_template              , 0   },
	{ "note"        , Tag_note                  , 0   },
	{ "nt"          , Tag_template              , 0   },
	{ "ol"          , Tag_list                  , 2   },
	{ "p"           , Tag_p                     , 0   },
	{ "parml"       , Tag_parml                 , 0   },
	{ "pbutton"     , Tag_template              , 0   },
	{ "pd"          , Tag_pitem                 , 1   },
	{ "pt"          , Tag_pitem                 , 0   },
	{ "rm"          , Tag_template              , 0   },
	{ "sl"          , Tag_list                  , 0   },
	{ "table"       , Tag_template              , 0   },
	{ "title"       , Tag_template              , 0   },
	{ "ul"          , Tag_list                  , 1   },
	{ "userdoc"     , Tag_userdoc ,MAGIC_USERDOC_ARG  },
	{ "warning"     , Tag_template              , 0   },
	{ "xmp"         , Tag_lines                 , 2   },
	{ NULL		, NULL }
    } ;



static struct {
	int tty;
	int quiet;
	int verbose;
	char *indexFileName;
    } opt ;

static struct {
	byte line[MAX_LINELEN+1]; /* Zwichenspeichern einer Zeile */
	int  n;     /* Anzahl Zeichen in line */
	struct {
	    int no; /* current number of header ( 1... ) */
	} h[MAX_HDRLEVELS];
	int  hlvl;    /* current header level (0..) */
	char tagBuf[MAX_TAGLEN+1]; /* used to parse a tag */
	char indexBuf[MAX_LINELEN+1]; /* Zwichenspeichern des Index Titles */
	int  tagLen;		   /* current length of tagBuf */
	int  userdocSeen;  /* flag, indicating that we are behind userdoc */
	int  doBreak;	/* flag set by ".br" control word and some tags */
	int  doUnderline; /* underline the next line: 1 = * 2 = = 3 = - */
	int  doAddIndex;  /* index komplettieren */
	int  listLevel; /* 0 = not in list */
	int  parmIdent; /* ident for parmlist, wenn negativ wird er erst */
			/* fuer die naechste Zeile angewendet */
	int  addIdent;	/* wird z.b durch parm,Ident gesetzt */
	enum {
	    mFMT = 0,	/* format mode */
	    mLEFT,	/* left align */
	    mCENTER,	/* center line */
	    mRIGHT,	/* right align */
	    mGRAPH,	/* no format monospaced font */
	    mXMP	/* no format, monospaced font, idendet by 2 spaces */
	} mode;     /* formatierungsmodus */
	struct {
	    ushort left;  /* number of blanks at the left */
	    ushort ident; /* additional blanks at start of a new Paragraph */
	    ushort len;   /* length of printable line */
	} margin;
	enum {
	    sIDLE,  /* idle = not inside userdoc */
	    sCHAR,  /* processing a character */
	    sTAG,   /* processing a tag */
	    sSYM    /* processing a symbol */
	} state;    /* state excluding the preprocessed control words */
    } fmt;  /* Zusammenfassung aller aktuellen formatierungsoptionen */


static ulong linesPrinted;

/****************
 * Zum Festhalten von Strings wird ein Speicherbereich benutzt,
 * indem diese hintereinander abgelegt sind, somit sit es mglich
 * ber einen einfachen Index auf eine String zuzugreifen. (AtomTable)
 */

static char   *atomTable;     /* table of Strings */
static atom_t atomTableSize;  /* current size of table */
static atom_t atomTableMax;   /* Max size of table */


/******************
 * Array  zum Speichern der Indices
 */
static indexTbl_t indexTbl[MAX_INDICES];
static size_t	  indexTblSize;

/******* Functions ********/
#define isSpcOrPct(c)  (isspace((c))||ispunct((c)))

const char *CopyRight( int level )
{
    const char *p;
    switch( level ) {
      case 10:
      case 0:	p = "IPF2DOC - Ver. 0.53; "
		    "Copyright 1992 by Werner Koch (dd9jn)" ; break;
      case 14:	p =		   "0.53"; break;
      case 1:
      case 11:	p = "Usage: ipf2doc [options] [file](-h for help)";
		break;
      case 2:
      case 12:	p =
    "Syntax: ipf2doc [options] [file]\n\n"
    "Converts an IPF sourceformat to plain text\n"
    "Takes input from stdin if file is ommitted\n"
    "Sends output to stdout\n"
    "Options summary:\n"
    " -q = quiet mode\n"
    " -v = verbose infomations\n"
    " -i<file> = name of index file\n"
    " -h = help\n";
	break;
      case 13:	p = "ipf2doc"; break;
      default: p = "?";
    }

    if( !level ) { fputs( p, stderr ); putc('\n',stderr); }
    else if( level == 1 ) { fputs( p, stderr ); putc( '\n', stderr );exit(3);}
    else if( level == 2 ) {  /* help */ puts( p ); exit(0); }

    return p;
}

void main( int argc, char **argv )
{
    char *s;

    for( s=""; --argc && **++argv == '-' && *s != '-'; )
	for( s = *argv + 1 ; *s && *s != '-' ; s++ )
	    switch( *s ) {
	      case 'v': opt.verbose++; break;
	      case 'q': opt.quiet++; break;
	      case 'i':
		opt.indexFileName = s + 1 ;
		while( *s ) s++ ; s -- ;
		break;
	      case 'h' :
	      case '?' : CopyRight(0) ; CopyRight(2); break;
	      default  : Error(3,GetStr(15),s );
	    }
    if( argc > 1 )
	CopyRight(1) ;

    if( !opt.quiet ) {
	if( opt.tty = IsTerminal(0) )
	    setvbuf(stdout,NULL,_IONBF,0);
	CopyRight(0);
    }
    else
	opt.verbose = 0 ;

    if( !opt.quiet && !argc )
	fprintf(stderr,"taking input from stdin ...\n");

    Initialize();
    if( argc )
	PreProcess(xfopen(*argv, "r"),*argv);
    else
	PreProcess(stdin,"-");
    if( indexTblSize ) { /* fix up # of lines */
	indexTbl[indexTblSize-1].nlines =
	    linesPrinted - indexTbl[indexTblSize-1].lnr + 1;
    }
    PrintIndex();
    if( opt.indexFileName )
	WriteIndexFile();

    DestroyAtomTable();
    exit(0) ;
} /* end main() */


/****************
 * Die globalen Variablen initialisieren
 */

static void Initialize(void)
{
    int i;

    MakeAtomTable( 5000 );
    indexTblSize = 0;
    fmt.n = 0;
    for(i=0; i < MAX_HDRLEVELS; i++ )
	fmt.h[i].no = 0; /* header tag will increment it */
    fmt.hlvl = 0;
    fmt.userdocSeen = 0;
    fmt.doBreak = 0;
    fmt.doUnderline = 0;
    fmt.doAddIndex = 0;
    fmt.listLevel = 0;
    fmt.parmIdent = 0;
    fmt.addIdent = 0;
    fmt.mode  = mFMT;
    fmt.state = sIDLE;

    fmt.margin.left = 9;
    fmt.margin.len  = 65;
    fmt.margin.ident= 0; /* gibt es gar nicht */
    linesPrinted = 0;
    xassert( fmt.margin.len < MAX_LINELEN );
}


static void Err( const char *str )
{
    Error(2,"error: %s", str );
}

static void Wrn( const char *str )
{
    Error(0,"warning: %s", str );
}


/****************
 * Zeichen einlesen zund ControlWords behandeln.
 * Ruft sich rekursiv auf fuer imbedded files
 */

static void PreProcess( FILE *st, const char *fname )
{
    int c;	/* current character */
    int nlFlag; /* c is the first character in a new line */
    char imbedName[F_MAX_PATH]; /* name of imbedded file */
    size_t n;	    /* help var */
    FILE *imbedSt;  /* stream of imbedded file */
    enum { sINIT,	    /* initial state */
	   sPOSCTRLWORD,    /* possible a control word */
	   sPOSBREAK,	    /* possible control word: break */
	   sPOSIMBED,	    /* possible control word: imbed */
	   sSKIP,	    /* skip rest of line */
	   sGETFILENAME     /* wait for filename */
	 } state;

    state = sINIT;
    nlFlag = 1;
    while( (c=getc(st)) != EOF ) {
	if( c == '\n' ) {
	    if( state == sINIT )
		Process(c);
	    else if( state == sGETFILENAME ) { /* imbed file */
		StripWSpaces(imbedName);
		if( imbedSt = fopen( imbedName, "r" ) ) {
		    if( opt.verbose )
			fprintf(stderr, "imbedding '%s'\n", imbedName);
		    PreProcess( imbedSt, imbedName );
		    fclose(imbedSt);
		}
		else
		    Error(0,"error: can't open '%s' - skipped", imbedName);
	    }
	    nlFlag = 1;
	    state = sINIT;
	    continue;
	}
	switch( state ) {
	  case sINIT:
	    if( nlFlag && c == '.' )
		state = sPOSCTRLWORD;
	    else
		Process(c);
	    nlFlag = 0;
	    break;

	  case sSKIP:
	    break;

	  case sPOSCTRLWORD:
	    if( c == '*' )
		state = sSKIP;	/* comment */
	    else if( c == 'b' )
		state = sPOSBREAK;
	    else if( c == 'i' )
		state = sPOSIMBED;
	    else {
		Process('.');
		Process(c);
	    }
	    break;

	  case sPOSBREAK:
	    if( c == 'r' ) {
		fmt.doBreak++;	/* nicht vollkommen richtig, da break */
		state = sSKIP;	/* zu ignorieren ist, falls weiterer Text */
	    }			/* in der Zeile folgt */
	    else {
		Process('.');
		Process('b');
		Process(c);
		state = sINIT;
	    }
	    break;

	  case sPOSIMBED:
	    if( c == 'm' ) {
		*imbedName = 0;
		state = sGETFILENAME;
	    }
	    else {
		Process('.');
		Process('i');
		Process(c);
		state = sINIT;
	    }
	    break;

	  case sGETFILENAME:
	    if( isspace(c) && !*imbedName )
		break;	/* still waiting for begin of filename */
	    else if( (n=strlen(imbedName)) < DIM(imbedName)-1 ) {
		imbedName[n++] = c;
		imbedName[n] = 0;
	    }
	    break;


	  default: BUG();
	}
    }
}


static void Process( int c )
{
    switch( fmt.state ) {
      case sIDLE:   /* wait for :userdoc. or do nothing more */
	if( !fmt.userdocSeen && c == ':' )
	    BeginTag();
	break;

      case sCHAR:
	if( c == ':' )
	    BeginTag();
	else if( c == '&' ) {
	    if( fmt.state == sSYM )
		Err("nested symbols");
	    fmt.state = sSYM;
	    fmt.tagLen = 0;
	}
	else
	    ProcessChar(c);
	break;

      case sTAG:
	if( c == '.' ) {
	    fmt.tagBuf[fmt.tagLen] = 0; /* append string terminator */
	    EndTag();
	}
	else if( !fmt.tagLen && isspace(c) ) { /* ": " seems not to be a tag*/
	    if( fmt.userdocSeen ) {
		fmt.state = sCHAR;
		ProcessChar(':');
		ProcessChar(' ');
	    }
	    else
		fmt.state = sIDLE;
	}
	else {
	    fmt.tagBuf[fmt.tagLen] = c;
	    if( ++fmt.tagLen >= MAX_TAGLEN )
		Err("tag too long");
	}
	break;

      case sSYM:
	if( c == '.' ) {
	    fmt.tagBuf[fmt.tagLen] = 0; /* append string terminator */
	    EndSymbol();
	}
	else {
	    fmt.tagBuf[fmt.tagLen] = c;
	    if( ++fmt.tagLen >= MAX_TAGLEN )
		Err("symbol too long");
	}
	break;

      default: BUG();
    }
}


/****************
 * Falls Zeile Zeichen enthlt, diese Zeile ausgeben
 */

static void FlushLine()
{
    int i;

    for(i=0; i < fmt.n ; i++ )
	if( !isspace(fmt.line[i]) ) {
	    fmt.doBreak++;
	    ProcessChar('\n');
	    break;
	}
}

static void AddNewLine()
{
    fmt.doBreak++;
    ProcessChar('\n');
}


static void ProcessStr( const char *s )
{
    while( *s )
	ProcessChar( *s++ );
}


static void ProcessChar(int c)
{
    int i,n;


    if( fmt.doAddIndex ) {
	if( !*fmt.indexBuf && isspace(c) )
	    ; /* skip the leading blanks */
	else if( (n = strlen( fmt.indexBuf )) < MAX_LINELEN-1 ) {
	    fmt.indexBuf[n] = (byte)c;
	    fmt.indexBuf[n+1] = 0;
	}
    }

    if( fmt.doBreak && c == '\n' ) {
	fmt.doBreak = 0;
	n = fmt.n;
	PutNewLine();
	if( fmt.doAddIndex ) {	/* fix up indextable */
	    xassert( indexTblSize );
	    StripWSpaces(fmt.indexBuf);
	    indexTbl[indexTblSize-1].title = AddAtom(fmt.indexBuf);
	    fmt.doAddIndex = 0;
	}
	if( fmt.doUnderline ) {
	    c = fmt.doUnderline == 1 ? '*' :
		fmt.doUnderline == 2 ? '=' : '-' ;
	    for(i=0; i < n; i++ )
		fmt.line[i] = c;
	    fmt.n = i;
	    PutNewLine();
	    fmt.doUnderline = 0;
	}
    }
    else if( fmt.mode == mFMT ) {
	if( c == '\n' )
	    c = ' ';
	if( !fmt.n && isspace(c) )
	    ; /* skip spaces at the beginning of a line */
	else
	    fmt.line[fmt.n++] = c;
	if( fmt.n >= fmt.margin.len - fmt.addIdent )  {
	    n = fmt.n ;
	    if( !isSpcOrPct(fmt.line[n-1]) ) { /* split line */
		for(i=n-2; i >= 0 ; i-- )
		    if( isSpcOrPct(fmt.line[i]) )
			break;
		if( i > 1 ) { /* long enough to split */
		    fmt.n = i+1;
		    PutNewLine();
		    while( i < n )
			if( isspace(fmt.line[i]) )
			    i++;
			else
			    break;
		    while( i < n )
			fmt.line[fmt.n++] = fmt.line[i++];
		}
		else {	/* can't split */
		    PutNewLine();
		}
	    }
	    else
		PutNewLine();
	}
    }
    else { /* graphics or lines */
	if( c == '\n' )
	    PutNewLine();
	else {
	    fmt.line[fmt.n++] = c;
	    if( fmt.n >= fmt.margin.len - fmt.addIdent )
		PutNewLine();
	}
    }
}


/****************
 * ! Private to ProcessChar !
 * Eine neue Zeile beginnen, dabei eine linken rand lassen
 * Hier wird auch das alignment vorgenommen
 */

static void PutNewLine()
{
    int c, n, k;

    for(n=0; n < fmt.margin.left + fmt.addIdent ; n++ )
	putchar(' ');

    n = 0;  /* default ist aber mit dem ersten zeichen anfangen */
    if( fmt.mode == mXMP ) {
	putchar(' ');
	putchar(' ');
    }
    else if( fmt.mode == mRIGHT ) {
	for(k=0; k < fmt.n; k++ )
	    if( isspace(fmt.line[k]) )
		break;
	    else if ( fmt.line[k] == (NOBRKSPACE & 0xff)  )
		break;
	/* k zeigt nun auf das erste Zeichen das kein Space ist */
	/** ... work .... **/
    }
    else if( fmt.mode == mCENTER ) {
	/** ... work .... **/
    }

    for(; n < fmt.n; n++ ) {
	c = fmt.line[n];
	if( c == NOBRKSPACE )
	    c = ' ' ;
	putchar( c );
    }
    putchar('\n');
    linesPrinted++;
    fmt.n = 0;
    if( fmt.parmIdent < 0 ) {
	fmt.parmIdent = -fmt.parmIdent;
	fmt.addIdent  = fmt.parmIdent;
    }
}




/****************
 * Initialisierung fuer die Tagverarbeitung
 */

static void BeginTag(void)
{
    if( fmt.state == sSYM )
	Bug("tag encountered while parsing a symbol");
    if( fmt.state == sTAG && fmt.userdocSeen )
	Err("nested tags");
    fmt.state = sTAG;
    fmt.tagLen = 0;
}



/****************
 * Tag is in tagBufffer, process it.
 * the colon and the dot are nor included in the buffer
 */

static void EndTag()
{
    const char *pc;
    char *tag, *p;
    int n, i , endFlg;

    tag = fmt.tagBuf;
    if( endFlg = *tag == 'e' )
	tag++;
    /* go, find the first blank */
    n = strcspn( tag, whiteSpaces );
    if( !n )
	Err("invalid tag found");
    p = tag + n;
    if( *p ) /* may be there are some args */
	p += strspn( p+1, whiteSpaces ) +1;
    tag[n] = 0;  /* terminate the name */

    for( i=0; pc = tagTable[i].name ; i++ )
	if( !strcmp( pc, tag ) )
	    break;
    if( !pc )
	Err("invalid tag found");
    StripWSpaces(p);
    tagTable[i].fnc( endFlg, p, tagTable[i].arg );
    if( tagTable[i].arg != MAGIC_USERDOC_ARG )
	fmt.state = sCHAR;
}


/****************
 * Symbol is in tagBufffer, process it.
 * the ampersand and the dot are not included in the buffer
 */

static void EndSymbol()
{
    const char *pc;
    char  *sym;
    int i;

    sym = fmt.tagBuf;
    if( !*sym )
	Err("invalid symbol found");

    for( i=0; pc = symbolTable[i].name ; i++ )
	if( !strcmp( pc, sym ) )
	    break;
    fmt.state = sCHAR;
    if( !pc ) {
	Wrn("invalid symbol");
	ProcessChar( ' ' );
    }
    else
	ProcessChar( symbolTable[i].code );
}


/****************
 * Funktionen zum Verarbeiten der user tags; alle diese Tag funcktzionen haben
 * den gleichen aufrufsyntax: endFlag, gigt an, das es sich um
 * das endekennzzeichen dieses Secion handelt, eine Pointer in fmt.tagBuf
 * der auf dem ersten Zeichen eventueller argumente steht (oder auf '\0')
 * und einem Integer wert, der der tagTable entnommen wurde.
 * userdoc hat die besonderheit, das es den neuen state selbst setzt
 * Der TagBuffer darf veraendert werden.
 */

static void Tag_template( int endFlag ,char *str ,int arg)
{
}


static void Tag_userdoc(int endFlag ,char *str ,int arg)
{
    if( !endFlag ) {
	if( fmt.userdocSeen )
	    Err("'userdoc' nested");
	fmt.userdocSeen = 1;
	fmt.state = sCHAR;
    }
    else {
	FlushLine();
	AddNewLine();
	fmt.state = sIDLE;  /* skip the rest of the files */
    }
}


static void Tag_h(int endFlag ,char *str ,int arg)
{
    char buf[10], *p;
    int i;

    if( endFlag )
	return;   /* optional */
    xassert( arg > 0 && arg <= MAX_HDRLEVELS );
    arg--;
    if( arg > fmt.hlvl+1 )
	Err("misplaced header tag");

    FlushLine();
    fmt.mode = mFMT;
    fmt.parmIdent = 0;
    fmt.addIdent = 0;
    for( i = arg ? 2 : 4; i; i-- )  /* some lines in front of the headers */
	AddNewLine();

    for(i=arg+1; i < MAX_HDRLEVELS; i++ )
	fmt.h[i].no = 0; /* reset all lower levels */
    fmt.hlvl = arg;
    fmt.h[arg].no++;
    for( i=0; i <= arg; i++ ) {
	sprintf(buf, i? ".%d":"%d",fmt.h[i]);
	for(p=buf; *p ; p++ )
	    ProcessChar(*p);
    }
    if( !arg ) /* der erste level wird mit punkt dargestellt ( "1." ) */
       ProcessChar('.');  /* die weiteren ohne punkt am ende ("1.1") */
    ProcessChar(NOBRKSPACE);
    ProcessChar(NOBRKSPACE);
    fmt.doBreak++;
    fmt.doUnderline = arg+1;
    /* Die Argumente ein wenig bearbeiten */

    if( !*str ) {
	if( indexTblSize < MAX_INDICES ) {
	    indexTbl[indexTblSize].id = AddAtom("");
	    indexTbl[indexTblSize].title = AddAtom("");
	    for( i=0; i < MAX_HDRLEVELS; i++ )
		indexTbl[indexTblSize].h[i] = fmt.h[i].no;
	    indexTblSize++;
	}
	else
	    Error(2,"out of memory in indextable");
	fmt.doAddIndex++;
	*fmt.indexBuf = 0;
	return;
    }

  #if 0 /* z.Z. wird nur der erste Index benutzt */
    while( str = strstr( str , "id=" ) ) {
  #else
    if( str = strstr( str , "id=" ) ) {
  #endif
	str += 3;
	if( *str == '\'' ) {
	    p = ++str;
	    if( str = strchr( p, '\'' ) )
		*str++ = 0;
	    else
		p = NULL;
	}
	else if( *str ) {
	    p = str;
	    if( str = strchr( p, ' ' ) )
		*str++ = 0;
	    else
		for(str=p; *++str; )
		    ;
	}
	else
	    p = NULL;

	if( p ) {
	    if( indexTblSize ) { /* fix up # of lines */
		indexTbl[indexTblSize-1].nlines =
		    linesPrinted - indexTbl[indexTblSize-1].lnr + 1;
	    }
	    if( indexTblSize < MAX_INDICES ) {
		indexTbl[indexTblSize].id = AddAtom(p);
		indexTbl[indexTblSize].title = AddAtom("");
		indexTbl[indexTblSize].lnr   = linesPrinted+1;
		indexTbl[indexTblSize].nlines = 0; /* unknown at this point */
		for( i=0; i < MAX_HDRLEVELS; i++ )
		    indexTbl[indexTblSize].h[i] = fmt.h[i].no;
		indexTblSize++;
	    }
	    else
		Error(2,"out of memory in indextable");
	    fmt.doAddIndex++;
	    *fmt.indexBuf = 0;
	}
    }
}



static void Tag_p(int endFlag ,char *str ,int arg)
{
    int i;

    if( endFlag )
	return;   /* optional */

    FlushLine();
    fmt.parmIdent = 0;
    fmt.addIdent = 0;
    AddNewLine();
    for(i=0; i < fmt.margin.ident; i++ )
	ProcessChar(NOBRKSPACE);  /* with ident */
}


static void Tag_note(int endFlag ,char *str ,int arg)
{
    int i;

    if( endFlag )
	return;   /* optional */

    FlushLine();
    AddNewLine();
    for(i=4; i < fmt.listLevel * 4; i++ )
	ProcessChar(NOBRKSPACE);  /* with ident */
    ProcessStr("Note:");
    ProcessChar(NOBRKSPACE);
    ProcessChar(NOBRKSPACE);
}


/****************
 *  fuer die list tags: arg = 0 -> :sl.
 *			arg = 1 -> :ul.
 *			arg = 2 -> :ol.
 */

static void Tag_list(int endFlag ,char *str ,int arg)
{
    if( endFlag ) {
	if( fmt.listLevel )
	    fmt.listLevel--;
	return;
    }
    FlushLine();
    AddNewLine();
    fmt.listLevel++;
}


/****************
 * Tag listItem ( :li. )
 */

static void Tag_listitem(int endFlag ,char *str ,int arg)
{
    int i;

    if( endFlag )
	return;    /* nicht notwendig */

    FlushLine();
    for(i=4; i < fmt.listLevel * 4; i++ )
	ProcessChar(NOBRKSPACE);  /* with ident */
}

/****************
 *  arg = 0 -> :cgraphic.
 *  arg = 1 -> :lines.
 *  arg = 2 -> :xmp.
 */

static void Tag_lines(int endFlag ,char *str ,int arg)
{
    FlushLine();
    if( endFlag ) {
	fmt.mode = mFMT;
	if( !arg )	      /* for :ecgraphic.  */
	    AddNewLine();
	return;
    }
    if( !arg )
	AddNewLine();

    fmt.mode = arg == 2 ? mXMP	:
	       arg == 1 ? mLEFT : mGRAPH;
}


static void Tag_parml(int endFlag ,char *str ,int arg)
{
    if( endFlag ) {
	FlushLine();
	fmt.parmIdent = 0;
	fmt.addIdent = 0;
    } else {
	fmt.parmIdent = 0;
	fmt.addIdent = 0;
	FlushLine();
    }
}



/****************
 * arg = 0 --> :pt.
 * arg = 1 --> :pd.
 */

static void Tag_pitem(int endFlag ,char *str ,int arg)
{
    int i;

    if( endFlag )
	return;

    if( !arg ) {
	/* parameter tag */
	FlushLine();
	fmt.parmIdent = 0;
	fmt.addIdent = 0;
    } else {
	/* parameter definition */
	fmt.addIdent = 0;
	fmt.parmIdent = -10;
	for( i=fmt.n; i ; i-- )
	    if( !isspace( fmt.line[i-1] ) )
		break;
	fmt.n = i;
	if( i >= 10)
	    ProcessChar(NOBRKSPACE);
	for( ; i < 10; i ++ )
	    ProcessChar(NOBRKSPACE);
    }
}


/************************************/
/******** Index ausgeben ************/
/************************************/

static void PrintIndex()
{
    size_t n;
    int i,j, lvl;
    const char *p;

    putchar('\n');
    putchar('\n');
    for(i=0; i < fmt.margin.left; i++ )
	putchar(' ');
    puts( "TABLE OF CONTENTS\n" );
    for(n=0; n < indexTblSize; n++ ) {
	for( lvl=0; lvl < MAX_HDRLEVELS; lvl++ )
	    if( !indexTbl[n].h[lvl] )
		break;

	if( lvl < 2 )
	    putchar('\n');
	if( lvl < 3 )
	    putchar('\n');

	/* left margins */
	for(i=0; i < fmt.margin.left; i++ )
	    putchar(' ');

	/* ident the first 4 levels */
	for(j=0,i=1; i < lvl && i < 4; i++ ) {
	    putchar(' ');
	    putchar(' ');
	    j += 2;
	}

	/* print the header numbers */
	for( i=0; i < lvl; i++ )
	    j += printf( i? ".%d":"%d",indexTbl[n].h[i]);
	p = GetAtomName( indexTbl[n].title );
	/* fill up with spaces */
	if( *p ) {
	    putchar(' ');
	  #if 0
	    for( ; j < 28; j++ )
		putchar('.');
	  #endif
	    putchar(' ');
	}
	/* print the title */
	puts(p);
    }
}


static void WriteIndexFile()
{
    size_t n;
    FILE *st;
    const char *p;

    st = xfopen( opt.indexFileName, "w" );
    for(n=0; n < indexTblSize; n++ ) {
	for( p = GetAtomName( indexTbl[n].title ); *p ; p++ )
	    fputc( *p == ':' ? '@' : *p , st );
	fprintf( st, "::%u(%u)\n",
	    indexTbl[n].lnr,
	    indexTbl[n].nlines	);
    }
    fclose(st);
}


/************************************/
/********* Helper functions *********/
/************************************/

static void MakeAtomTable( size_t size )
{
    xassert( size > 1 );
    atomTable = xmalloc( size );
    atomTableMax = size ;
    *atomTable = 0xff;
    atomTableSize = 1 ;
}

static void DestroyAtomTable(void)
{
    free(atomTable);
    atomTableMax = 0 ;
    atomTableSize = 0 ;
}


static atom_t AddAtom( const char *s )
{
    char *p;
    size_t n;

    if( n = GetAtom( s ) )
	return n; /* already in table */
    n = atomTableSize;
    p = atomTable + n;
    for(; *s && atomTableSize < atomTableMax; atomTableSize++ )
	*p++ = *s++;
    if( atomTableSize == atomTableMax )
	Error(2,"out of memory in atomtable");
    *p = 0;
    atomTableSize++;
    return n;
}


static atom_t GetAtom( const char *name )
{
    const char *p, *s;
    size_t n;

    /* we do not use the 1. entry */
    /* so we can use 0 as a NotFoundIndicator */
    p = atomTable;
    n = 0;
    do {
	s = name;
	p++;
	n++;
	for( ; *p == *s && *s && n < atomTableSize; p++, s++, n++ )
	    ;
	if( !*p && !*s ) { /* found */
	    for(; name != s; s--, n-- )
		;
	    return n;  /* return index of start of string */
	}
	for( ; *p && n < atomTableSize; p++, n++ )
	    ;
    } while( n < atomTableSize );
    return 0;
}


static const char *GetAtomName( atom_t hd )
{
    if( hd && hd < atomTableSize )
	return atomTable + hd;
    Bug("No Atom with handle %x", hd);
    /*NOTREACHED*/
}


/*** bottom of file ***/
