/* [wvar.c wk 6.9.91] W-Editor Variable handling
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 *  9.05.92 wk	Neue Variablen ZNLines und ZCpop und die allokierten Buffer
 *		fuer die Rueckgabge numerische Variablenwerte auf 12 gekrzt.
 *  6.12.92 wk	Added operators sameCI and diffCI
 * 18.12.92 wk	New Operators julian, daytime and vars: ZCTime, ZTime, ZDate
 * 03.01.93 wk	some cleanups, moved _envptr to function
 * 04.04.93 wk	ProcessInfos erweitert um '*'
 * 15.05.93 wk	new operator getword
 * 10.06.93 wk	removed ZRegister
 * 07.03.95 wk	Added OPERATOR_EXIST
 * 07.03.95 wk	Added OPERATOR_INEDITOR
 */

#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#ifdef UNIX
#include <pwd.h>
#include <sys/utsname.h>
#endif
#if __ZTC__
#include <direct.h>
#endif
#if __IBMC__ || __WATCOMC__
  #include <process.h>
#endif
#include <wk/lib.h>
#include <wk/file.h>
#include <wk/string.h>
#include <sys/types.h>
#include <wk/io.h>
#include <wk/environ.h>

#include "w.h"
#include "wcmd.h"
#include "wscreen.h"
#include "wfile.h"
#include "wmove.h"
#include "jnx.h"
#include <wk/wscrdrv.h> /* prototype ScrDrvGetToggleKeys() */

/****** constants *********/
#define STACKSIZE 50
#define VARROOM_CHUNK 20    /* make room for 20 variables with one call */

#define  ZNAMEID_PATH	    1
#define  ZNAMEID_DIR	    2
#define  ZNAMEID_CWD	    3
#define  ZNAMEID_LINE	    4
#define  ZNAMEID_COLUMN     5
#define  ZNAMEID_EOF	    6
#define  ZNAMEID_WRDATCRS   7
#define  ZNAMEID_CHRATCRS   8
#define  ZNAMEID_VERSION    9
#define  ZNAMEID_PGMNAME   10
#define  ZNAMEID_COPYRIGHT 11
#define  ZNAMEID_LOADDIR   12
#define  ZNAMEID_OS	   13
#define  ZNAMEID_CTOP	   14
#define  ZNAMEID_FND	   15
#define  ZNAMEID_NLINES    16
#define  ZNAMEID_CMDLINE   17
#define  ZNAMEID_INCMD	   18
#define  ZNAMEID_CPOP	   19
#define  ZNAMEID_ERR	   20
#define  ZNAMEID_INSERT    21
#define  ZNAMEID_NUMLCK    22
#define  ZNAMEID_SCRLCK    23
#define  ZNAMEID_CAPLCK    24
#define  ZNAMEID_INMARK    25
#define  ZNAMEID_DATE	   27
#define  ZNAMEID_TIME	   28
#define  ZNAMEID_CTIME	   29
#define  ZNAMEID_PID	   30
#define  ZNAMEID_UID	   31
#define  ZNAMEID_REALNAME  32
#define  ZNAMEID_NODENAME  33
#define  ZNAMEID_MACHINE   34
#define  ZNAMEID_SYSNAME   35
#define  ZNAMEID_DOMAINNAME 36
#define  ZNAMEID_SCREEN    37
#define  ZNAMEID_ISODATE   38

/******* typedefs ********/
typedef struct {
	byte type;	/* type of Value ( ARGTYPE_xxx ) */
	char *value;	/* malloced value */
    } stack_t;

typedef struct {
	char *name;	/* malloced or NULL if not in use */
	char *value;	/* malloced */
    } var_t;

/******* globals **********/
#if W_FULL
static int isInitialized;
static stack_t stack[STACKSIZE];
static int stackInd;
static var_t *varTbl;
static size_t varTblSize;
static struct {
	const char *name;
	byte  nameId;
    } zVarNames[] = {
	{ "ZPath"       , ZNAMEID_PATH         } ,
	{ "ZDir"        , ZNAMEID_DIR          } ,
	{ "ZCwd"        , ZNAMEID_CWD          } ,
	{ "ZLine"       , ZNAMEID_LINE         } ,
	{ "ZColumn"     , ZNAMEID_COLUMN       } ,
	{ "ZNLines"     , ZNAMEID_NLINES       } ,
	{ "ZErr"        , ZNAMEID_ERR          } ,
	{ "ZFnd"        , ZNAMEID_FND          } ,
	{ "ZEof"        , ZNAMEID_EOF          } ,
	{ "ZWrdAtCrs"   , ZNAMEID_WRDATCRS     } ,
	{ "ZChrAtCrs"   , ZNAMEID_CHRATCRS     } ,
	{ "ZVersion"    , ZNAMEID_VERSION      } ,
	{ "ZPgmName"    , ZNAMEID_PGMNAME      } ,
	{ "ZLoadDir"    , ZNAMEID_LOADDIR      } ,
	{ "ZPID"        , ZNAMEID_PID          } ,
      #ifdef UNIX
	{ "ZUID"        , ZNAMEID_UID          } ,
	{ "ZRealName"   , ZNAMEID_REALNAME     } ,
	{ "Znodename"   , ZNAMEID_NODENAME     } ,
	{ "Zdomainname" , ZNAMEID_DOMAINNAME   } ,
	{ "Zsysname"    , ZNAMEID_SYSNAME      } ,
	{ "Zmachine"    , ZNAMEID_MACHINE      } ,
      #endif
	{ "ZOS"         , ZNAMEID_OS           } ,
	{ "ZScreen"     , ZNAMEID_SCREEN       } ,
	{ "ZCTime"      , ZNAMEID_CTIME        } ,
	{ "ZDate"       , ZNAMEID_DATE         } ,
	{ "ZISODate"    , ZNAMEID_ISODATE      } ,
	{ "ZTime"       , ZNAMEID_TIME         } ,
	{ "ZCmdLine"    , ZNAMEID_CMDLINE      } ,
	{ "ZInCmd"      , ZNAMEID_INCMD        } ,
	{ "ZInMark"     , ZNAMEID_INMARK       } ,
	{ "ZInsert"     , ZNAMEID_INSERT       } ,
	{ "ZNumLck"     , ZNAMEID_NUMLCK       } ,
	{ "ZScrLck"     , ZNAMEID_SCRLCK       } ,
	{ "ZCapLck"     , ZNAMEID_CAPLCK       } ,
	{ "ZCopyRight"  , ZNAMEID_COPYRIGHT    } ,
	{ "ZCpop"       , ZNAMEID_CPOP         } ,   /* special !*/
	{ "ZCtop"       , ZNAMEID_CTOP         } ,
	{ NULL		, 0 } };

static int varInd_ZWordDel = -1 ; /* for faster access of this variable */
#endif
/******* prototypes *******/
static int Compare( int operator, int *retBool );
static void RemoveArgs( int operator );
static int Calculate( int operator );
static int PushNumberValue( long val );
static int PushValue( const char *value, int argType );
static int DupValue(void);
static char *PeekValue(int);
static int SwapValue( void );
static char *PopValue(void);
static int PutVar( const char *name, const char *value );
static char *GetVar( const char *name, int argType );
static void DoInit(void);
static void CleanUp( void *);
/******* functions ********/

#if W_FULL
/****************
 * the Enter command: Put an Value onto the calc stack
 */

int Cmd_Enter( cmd_t *cmd )
{
    return PushValue( cmd->arg.string, cmd->type );
}


/****************
 * Assign last element on the stack to a variable or print it if no args
 * Remove Element from stack
 */

int Cmd_Assign( cmd_t *cmd )
{
    int err=0;
    char *val, *buf;

    if( !(val = PopValue()) )
	err = ERR_CSTKUDR;
    else if( cmd->type == ARGTYPE_NO ) {
	err = Cmd_Put( val );
	free( val );
    }
    else if( cmd->type == ARGTYPE_VARENV ) {
	buf = xmalloc( strlen(cmd->arg.string) + 1 + strlen(val) + 1 );
	strcpy( buf, cmd->arg.string );
	strcat( buf, "=" );
	strcat( buf, val );
	free( val );
	if( putenv( buf ) )
	    err = ERR_PUTENV;
	free( buf );
    }
    else {  /* ARGTYPE_VAR */
	err = PutVar( cmd->arg.string , val );
	free( val );
    }

    return err;
}


int Cmd_Jmp( cmd_t *cmd )
{
    int err=0;
    int value;

    if( cmd->nxtCmd ) {
	if( !(err = Compare( cmd->arg.number, &value )) ) {
	    RemoveArgs( cmd->arg.number );
	    if( value ) {
		if( !(err = ExecOneCmdBlock( cmd->nxtCmd )) )
		    err = ERR_PSEUDO_END;
	    }
	    else
		err = ERR_PSEUDO_SKIP;
	}
    }
    else
	err = ERR_LOOP;
    return err;
}



int Cmd_Call( cmd_t *cmd )
{
    int err=0;
    int value;

    if( cmd->nxtCmd ) {
	if( !(err = Compare( cmd->arg.number, &value )) ) {
	    RemoveArgs( cmd->arg.number );
	    if( value )
		err = ExecOneCmdBlock( cmd->nxtCmd );
	    if( !err )
		err = ERR_PSEUDO_SKIP;
	}
    }
    else
	err = ERR_LOOP;
    return err;
}


int Cmd_While( cmd_t *cmd )
{
    int err=0, ex;
    int value;

    if( cmd->nxtCmd ) {
	for(ex=0; !ex && !err; ) {
	    if( !(err = Compare( cmd->arg.number, &value )) ) {
		if( value ) {
		    err = ExecOneCmdBlock( cmd->nxtCmd );
		    if( err )
			RemoveArgs( cmd->arg.number );
		}
		else {
		    RemoveArgs( cmd->arg.number );
		    err = ERR_PSEUDO_SKIP;
		}
	    }
	}
    }
    else
	err = ERR_LOOP;
    return err;
}


int Cmd_Until( cmd_t *cmd )
{
    int err=0, ex;
    int value;

    if( cmd->nxtCmd ) {
	for(ex=0; !ex && !err; ) {
	    if( err = ExecOneCmdBlock( cmd->nxtCmd ) )
		;
	    else if( !(err = Compare( cmd->arg.number, &value )) ) {
		if( value ) {
		    RemoveArgs( cmd->arg.number );
		    err = ERR_PSEUDO_SKIP;
		}
	    }
	}
    }
    else
	err = ERR_LOOP;
    return err;
}



int Cmd_Calc( cmd_t *cmd )
{
    return Calculate( cmd->arg.number );
}


/****************
 * Eine Expression berechenen:
 * Dies Expression kann nur ein vergleich sein, ansonsten wird ein fehler
 * zurckgegeben. Der Stack wird nicht gendert !!!
 *
 * Returns: ErrorCode
 */

static int Compare( int operator, int *retBool )
{
    int err = 0;
    char *val1, *val2;
    long a, b;

    switch( operator ) {
      case OPERATOR_NOP: /* set to true, will be used for unconditional jumps */
	*retBool = 1;
	break;

      case OPERATOR_TRUE:
      case OPERATOR_FALSE:
	if( !(val1 = PeekValue(1)) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    *retBool = atol( val1 ) != 0;
	    if( operator == OPERATOR_FALSE )
		*retBool = !*retBool;
	    free( val1 );
	}
	break;


      case OPERATOR_LT:
      case OPERATOR_LE:
      case OPERATOR_EQ:
      case OPERATOR_NE:
      case OPERATOR_GE:
      case OPERATOR_GT:
      case OPERATOR_SAME:
      case OPERATOR_DIFF:
      case OPERATOR_SAMECI:
      case OPERATOR_DIFFCI:
	if( !(val2 = PeekValue(1)) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else if( !(val1 = PeekValue(2)) ) {
	    free( val2 );
	    err = ERR_CSTKUDR;	/* stack underflow */
	}
	else {
	    a = atol( val1 );
	    b = atol( val2 );

	    if( operator == OPERATOR_LT )
		*retBool = a < b;
	    else if( operator == OPERATOR_LE )
		*retBool = a <= b;
	    else if( operator == OPERATOR_EQ )
		*retBool = a == b;
	    else if( operator == OPERATOR_NE )
		*retBool = a != b;
	    else if( operator == OPERATOR_GE )
		*retBool = a >= b;
	    else if( operator == OPERATOR_GT )
		*retBool = a > b;
	    else if( operator == OPERATOR_SAME )
		*retBool = !strcmp( val1, val2 );
	    else if( operator == OPERATOR_DIFF )
		*retBool = strcmp( val1, val2 );
	    else if( operator == OPERATOR_SAMECI )
		*retBool = !strcmpl( val1, val2 );
	    else  /* OPERATOR_DIFFCI */
		*retBool = strcmpl( val1, val2 );

	    free( val1 );
	    free( val2 );
	}
	break;

      default: err = ERR_INVOP; break;
    }
    return err;
}




/****************
 * Perform a calculation on the calculation stack.
 */

static int Calculate( int operator )
{
    int err = 0;
    char *val1, *val2, *buf, *p;
    long a, b;
    int i;
    char numbuf[32+1+5]; /* 32 is for binary conversion + soem security */
    size_t n;
    ulong ul;

    switch( operator ) {
      case OPERATOR_NOP: /* this is a true NOP in a calculation */
	break;
      case OPERATOR_TRUE:
      case OPERATOR_FALSE:
      case OPERATOR_LT:
      case OPERATOR_LE:
      case OPERATOR_EQ:
      case OPERATOR_NE:
      case OPERATOR_GE:
      case OPERATOR_GT:
      case OPERATOR_SAME:
      case OPERATOR_DIFF:
      case OPERATOR_SAMECI:
      case OPERATOR_DIFFCI:
	if( !(err = Compare( operator, &i )) ) {
	    RemoveArgs( operator );
	    err = PushValue( i? "1": "0", ARGTYPE_STRING );
	}
	break;

      case OPERATOR_ADD:
      case OPERATOR_SUB:
      case OPERATOR_MLT:
      case OPERATOR_DIV:
      case OPERATOR_MOD:
	if( !(val2 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else if( !(val1 = PopValue()) ) {
	    free( val2 );
	    err = ERR_CSTKUDR;	/* stack underflow */
	}
	else {
	    a = atol( val1 );
	    free( val1 );
	    b = atol( val2 );
	    free( val2 );

	    if( operator == OPERATOR_ADD )
		a += b;
	    else if( operator == OPERATOR_SUB )
		a -= b;
	    else if( operator == OPERATOR_MLT )
		a *= b;
	    else if( !b )
		err = ERR_DIVZERO;
	    else if( operator == OPERATOR_DIV )
		a /= b;
	    else /* OPERATOR_MOD */
		a %= b;

	    if( !err )
		err = PushNumberValue( a );
	}
	break;

      case OPERATOR_LEN:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    i = strlen( val1 );
	    free( val1 );
	    err = PushNumberValue( i );
	}
	break;

      case OPERATOR_DUP:
	err = DupValue();
	break;

      case OPERATOR_EVL:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    err = PushValue( val1, ARGTYPE_STRING );
	    free( val1 );
	}
	break;

      case OPERATOR_TPATH:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	  #if MSDOSFILESYSTEM
	    for(buf=val1; *buf; buf++ )
		if( *buf == '/' )
		    *buf = '\\';
	  #endif
	    err = PushValue( val1, ARGTYPE_STRING );
	    free( val1 );
	}
	break;

      case OPERATOR_VAL:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    /* supported formats are: "$n", "nnh", "nnH", "0xn" for hex */
				 /*    n   for decimal */
	    /* leading white spaces are skipped, conversion stops at the */
	    /* first unregonizible character */
	    StripWSpaces( val1 );
	    p = val1;
	    if( !*p )
		strcpy(numbuf, "0");
	    else if( *p == '$' ) { /* hex string */
		a = strtoul( p+1, NULL, 16 );
		sprintf(numbuf,"%lu", a );
	    }
	    else if( *p == '0' && toupper(p[1]) == 'X' ) {
		a = strtoul( p+2, NULL, 16 );
		sprintf(numbuf,"%lu", a );
	    }
	    else {  /* check for trailing character */
		n = strspn( p, "0123456789abcdefABCDEF" );
		if( n && toupper(p[n])=='H'){
		    a = strtoul( p, NULL, 16 );
		    sprintf(numbuf,"%lu", a );
		}
		else {
		    a = strtol( p, NULL, 10 );
		    sprintf(numbuf,"%ld", a );
		}
	    }
	    free( val1 );
	    err = PushValue( numbuf, ARGTYPE_STRING );
	}
	break;

      case OPERATOR_VALB:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    a = strtoul( val1, NULL, 2 );
	    sprintf(numbuf,"%lu", a );
	    free( val1 );
	    err = PushValue( numbuf, ARGTYPE_STRING );
	}
	break;

      case OPERATOR_VALX:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    a = strtoul( val1, NULL, 16 );
	    sprintf(numbuf,"%lu", a );
	    free( val1 );
	    err = PushValue( numbuf, ARGTYPE_STRING );
	}
	break;


      case OPERATOR_HEX:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    a = atol( val1 );
	    free( val1 );
	    sprintf(numbuf,"%lx", a );
	    err = PushValue( numbuf, ARGTYPE_STRING );
	}
	break;


      case OPERATOR_BIN:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    a = atol( val1 );
	    free( val1 );
	  #if 0     /* seems %b isn't ANSI (GNUC 2.1) */
	    sprintf(numbuf,"%lb", a );
	  #else    /* so I have to monkey around */
	    ul = a;
	    p = numbuf;
	    do {
		*p++ = (ul % 2) + '0';
		ul /= 2;
	    } while( ul );
	    *p = 0;
	    strrev( numbuf );
	  #endif
	    err = PushValue( numbuf, ARGTYPE_STRING );
	}
	break;


      case OPERATOR_CAT:
	if( !(val2 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else if( !(val1 = PopValue()) ) {
	    free( val2 );
	    err = ERR_CSTKUDR;	/* stack underflow */
	}
	else {
	    buf = xmalloc( strlen(val2)+strlen(val1)+1 );
	    strcpy( buf, val1 );
	    strcat( buf, val2 );
	    free( val1 );
	    free( val2 );
	    err = PushValue( buf, ARGTYPE_STRING );
	    free(buf);
	}
	break;

      case OPERATOR_SWAP:
	err = SwapValue();
	break;

      case OPERATOR_POP:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    free( val1 );
	}
	break;

      case OPERATOR_JULIAN:
	sprintf(numbuf,"%lu", TodaysJD() );
	err = PushValue( numbuf, ARGTYPE_STRING );
	break;

      case OPERATOR_TMOFDAY:
	sprintf(numbuf,"%lu", TimeOfDay() );
	err = PushValue( numbuf, ARGTYPE_STRING );
	break;


      case OPERATOR_CLEAR:
	while( val1 = PopValue() )
	    free( val1 );
	break;


      case OPERATOR_GETWORD:
	if( !(val1 = PopValue()) )  /* get word delimiters */
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    int fileHd;
	    const char *txt;
	    ushort pos, n, nbytes;

	    fileHd = QryScreenFile();
	    txt = GetPtr2Line( fileHd, &nbytes );
	    GetFilePos( fileHd, &pos );
	    if( nbytes < pos )
		txt = "";
	    else {
		for(n=pos; n < nbytes && !isspace(txt[n]) && txt[n] &&
					      !strchr(val1,txt[n]); n++ )
		    ;
		n -= pos;
		val2 = xmalloc( n+1 );
		memcpy( val2, txt+pos, n ); val2[n] = '\0';
		err = PushValue( val2, ARGTYPE_STRING );
		free(val2);
	    }
	    free(val1);
	}
	break;

      case OPERATOR_EXIST:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    i = access( val1, F_OK );
	    free( val1 );
	    err = PushValue( i? "0": "1", ARGTYPE_STRING );
	}
	break;

      case OPERATOR_INEDITOR:
	if( !(val1 = PopValue()) )
	    err = ERR_CSTKUDR;	/* stack underflow */
	else {
	    i = GetFileHandle( val1 ) != -1;
	    free( val1 );
	    err = PushValue( i? "1": "0", ARGTYPE_STRING );
	}
	break;


      default: err = ERR_INVOP; break;
    }
    return err;
}



/****************
 * This function removes the correct number of Args from the calculation
 * stack. To be used together with compare.
 */

static void RemoveArgs( int operator )
{
    switch( operator ) {
      case OPERATOR_NOP:
	break;
      case OPERATOR_LT:
      case OPERATOR_LE:
      case OPERATOR_EQ:
      case OPERATOR_NE:
      case OPERATOR_GE:
      case OPERATOR_GT:
      case OPERATOR_SAME:
      case OPERATOR_DIFF:
      case OPERATOR_SAMECI:
      case OPERATOR_DIFFCI:
	free( PopValue() );
	/* fall thru to remove the second arg */
      case OPERATOR_TRUE:
      case OPERATOR_FALSE:
	free( PopValue() );
	break;
      default: BUG();
    }
}


/****************
 * Pops the top value from the Calulation Stack, converted to a ulong
 * Returns: 0 = okay, or errorCode
 */

int CalcStackPopULong( ulong * val )
{
    char *str;
    long val1;

    if( str = PopValue() ) {
	val1 = atol( str );
	free( str );
	if( val1 < 0 )
	    return ERR_VALRNG;
	*val = (ulong)val1;
	return 0;
    }
    else
	return ERR_CSTKUDR;  /* stack underflow */
}
#endif

/****************
 * Den Wert einer Variablen holen.
 * Paremter ist das Cmd.
 * Returns: Pointer to an string, this string must be free via:
 *	    FreeVarValue()
 */

char *VarValue( cmd_t *cmd )
{
  #if W_FULL
    return GetVar( cmd->arg.string, cmd->type );
  #else
    return xstrdup("");
  #endif
}

char *VarValueStr( const char *str, int argType )
{
  #if W_FULL
    return GetVar( str, argType );
  #else
    return xstrdup("");
  #endif
}

void FreeVarValue( char *p )
{
    free( p );
}



/****************
 * Special Function to get a string with the word delimters
 * NOTE: don't use any function which may change the contents of
 * variables, because the returned string may be a pointer into the
 * VariabelTable
 */

const char *GetWordDelimiters()
{
  #if W_FULL
    if( varInd_ZWordDel == -1 )
	return "";
    else
	return varTbl[varInd_ZWordDel].value ;
  #else
    return "";
  #endif
}



/****************
 *  same as PushValue, but value is a long
 */
#if W_FULL
static int PushNumberValue( long val )
{
    char buf[20];

    sprintf( buf, "%ld", val );
    return PushValue( buf, ARGTYPE_STRING );
}


/****************
 *  Push a value onto the calculation stack:
 * if it is a variable, push its value, if it is a stack, push itself
 * argType may be : ARGTYPE_STRING, ARGFTYPE_VAR or ARGTYPE_VARENV
 * Retrurns: 0 = Okay or errorcode
 */

static int PushValue( const char *value, int argType  )
{
    if( !isInitialized )
	DoInit();

    if( stackInd < STACKSIZE ) {
	xassert( argType == ARGTYPE_VAR || argType == ARGTYPE_VARENV ||
		 argType == ARGTYPE_STRING );
	stack[stackInd].type = (byte)argType;
	stack[stackInd].value= xstrdup( value );
	stackInd++;
	return 0;
    }
    else
	return ERR_CSTKOVR;
}



/****************
 * Get the value from the stack but does not remove it from the stack
 * Returns: Pointer to value ( this ptr must be free ! )
 *	  or NULL if no values on the stack
 */

static char *PeekValue( int deep )
{
    char *value;
    int  valType;

    xassert( deep );
    if( stackInd >= deep ) {
	value = stack[stackInd-deep].value;
	xassert( value );
	valType = stack[stackInd-deep].type;
	if( valType == ARGTYPE_VAR || valType == ARGTYPE_VARENV )
	    value = GetVar( value, valType );
	else
	    value = xstrdup(value);
    }
    else
	value = NULL;
    return value;
}


static int SwapValue( void )
{
    stack_t h;

    if( stackInd >= 2 ) {
	h = stack[stackInd-1];
	stack[stackInd-1] = stack[stackInd-2];
	stack[stackInd-2] = h;
	return 0;
    }
    else
	return ERR_CSTKUDR;  /* stack underflow */
}


/****************
 * Same as PeekValue, but removes value from the stack
 * returned ptr must be free !
 */

static char *PopValue()
{
    char *value;

    if( value = PeekValue(1) ) {
	stackInd--;
	free( stack[stackInd].value );
    }
    return value;
}



/****************
 * Duplicate the last value on the stack
 */

static int DupValue()
{
    if( !isInitialized )
	DoInit();

    if( !stackInd )
	return ERR_CSTKUDR;
    else if( stackInd < STACKSIZE ) {
	stack[stackInd].type  = stack[stackInd-1].type;
	stack[stackInd].value = xstrdup( stack[stackInd-1].value);
	stackInd++;
	return 0;
    }
    else
	return ERR_CSTKOVR;
}



/****************
 * Put a variable into the variable Table
 * if value is an empty string, the variable will be removed
 * Variable die mit Z anfangen sind ungltig mit folgenden Ausnahmen:
 *   - ZPrf_<xxxx>
 *   - ZPrfSh
 *   - ZEditPrc
 *   - ZEditPrc_<xxxx>
 *   - ZWordDel
 *   - ZDefEditOpt
 *   - ZDefSaveOpt
 * Returns: ErrorCode
 */

static int PutVar( const char *name, const char *value )
{
    size_t n;
    var_t *var;
    int found, create, err, zWordDel;
    char *newVal;

    zWordDel = !strcmp( name, "ZWordDel" );
    if( !*name || toupper(*name) == 'Z' )
	if( strncmp( name, "ZEditPrc", 8 ) &&
	    strncmp( name, "ZPrf_", 5 ) &&
	    strncmp( name, "ZPrfSh", 6 ) &&
	    strncmp( name, "ZDefEditOpt", 11 ) &&
	    strncmp( name, "ZDefSaveOpt", 11 ) && !zWordDel )
	    return ERR_INVVAR;

    if( !isInitialized )
	DoInit();
    found = create = err = 0;
    for( n=0, var = varTbl; n < varTblSize; n++, var++ )
	if( var->name )
	    if( !strcmp( var->name, name ) ) {
		found++;
		break;
	    }
    if( found && !*value ) { /* delete */
	FREE( var->name );
	free( var->value);
	if( zWordDel )
	    varInd_ZWordDel = -1;
    }
    else if( *value ) {
	if( !found )
	    for( n=0, var = varTbl; n < varTblSize; n++, var++ )
		if( !var->name ) {
		    create++;
		    found++;
		    break;
		}

	if( !found ) { /* go and create or extend the table */
	    var = realloc( varTbl, (varTblSize+VARROOM_CHUNK)*sizeof *var );
	    if( !var )
		err = ERR_VARROOM;
	    else {
		varTbl = var;
		var = varTbl + varTblSize; /* this can be used for a new var */
		found++;
		create++;
		varTblSize += VARROOM_CHUNK;
		for(n=0; n < VARROOM_CHUNK; n++ )
		    var[n].name = NULL;
	    }
	}

	if( found ) {
	    if( create ) {
		if( !(var->name = strdup( name )) )
		    err = ERR_VARROOM;
		else if( !(var->value = strdup( value )) )
		    err = ERR_VARROOM;
		else {
		    if( zWordDel )
			varInd_ZWordDel = var - varTbl;
		}
	    }
	    else if( strcmp(var->value, value) )  {
		if( !(newVal = strdup( value )) )
		    err = ERR_VARROOM;
		else {
		    free( var->value );
		    var->value = newVal;
		}
	    }
	}
    }
    return err;
}


/****************
 *  Get a variable value
 */

static char *
GetVar( const char *txt, int argType )
{
    char *value, *drvBuf, *dirBuf;
    const char *delim;
    unsigned flags;
    ushort pos, nbytes;
    int fileHd, found,i;
    size_t n;
    var_t *var;
    time_t tclock;

    fileHd = QryScreenFile();
    value = NULL;
    if( argType == ARGTYPE_VAR ) {
	for(found=i=0; zVarNames[i].name; i++ )
	    if( !strcmp( zVarNames[i].name, txt ) ) {
		found = zVarNames[i].nameId;
		break;
	    }
	switch( found ) {
	  case ZNAMEID_PATH:
	    txt = GetFileInfo( fileHd, &flags );
	    break;
	  case ZNAMEID_DIR:
	    txt = GetFileInfo( fileHd, &flags );
	    if( !strcmp( txt, ".dir" ) )
		txt = GetDirPattern();
	    value = xmalloc( F_MAX_PATH );
	    drvBuf = xmalloc( F_MAX_DRIVE);
	    dirBuf = xmalloc( F_MAX_DIR  );
	    FilenameSplit( txt, drvBuf, dirBuf, NULL, NULL );
	    FilenameMake( value, drvBuf, dirBuf, NULL, NULL );
	    free( dirBuf);
	    free( drvBuf);
	    break;
	  case ZNAMEID_CWD:
	    value = xmalloc( F_MAX_PATH );
	    getcwd( value, F_MAX_PATH-1 );
	    strcat( value, "/" );
	    FilenameCompr( value );
	    break;
	  case ZNAMEID_LINE:
	    value = xmalloc( 12 );
	    sprintf(value,"%lu", GetFilePos( fileHd,NULL )+1 );
	    break;
	  case ZNAMEID_NLINES:
	    value = xmalloc( 12 );
	    sprintf(value,"%lu", GetFileTotLines( fileHd ) );
	    break;
	  case ZNAMEID_COLUMN:
	    value = xmalloc( 12 );
	    GetFilePos( fileHd, &pos );
	    sprintf(value,"%u", pos+1 );
	    break;
	  case ZNAMEID_EOF:
	    txt = GetLastEofState() ? "1":"0";
	    break;
	  case ZNAMEID_FND:
	    txt = GetLastFoundState() ? "1" : "0";
	    break;
	  case ZNAMEID_ERR:
	    value = xmalloc( 12 );
	    sprintf(value,"%d", GetLastCmdError() );
	    break;
	  case ZNAMEID_PID:
	  #ifdef __ZTC__
	    txt = "0";
	  #else
	    value = xmalloc( 12 );
	    sprintf(value,"%d", getpid() );
	  #endif
	    break;
	#ifdef UNIX
	  case ZNAMEID_UID: {
		struct passwd *passwd;

		txt = getenv("LOGNAME");
		if( txt && (passwd = getpwnam(txt)) ) {
		    value = xmalloc( 12 );
		    sprintf(value,"%d", (int)passwd->pw_uid );
		}
		else
		    txt = "";
	    }
	    break;
	  case ZNAMEID_REALNAME: {
		struct passwd *passwd;

		txt = getenv("LOGNAME");
		if( txt && (passwd = getpwnam(txt)) )
		    value = xstrdup(passwd->pw_gecos);
		else
		    txt = "";
	    }
	    break;
	  case ZNAMEID_SYSNAME: {
		struct utsname uts;

		if( !uname( &uts ) )
		    value = xstrdup(uts.sysname);
		else
		    txt = "";
	    }
	    break;
	  case ZNAMEID_NODENAME: {
		struct utsname uts;

		if( !uname( &uts ) )
		    value = xstrdup(uts.nodename);
		else
		    txt = "";
	    }
	    break;
	  case ZNAMEID_MACHINE: {
		struct utsname uts;

		if( !uname( &uts ) )
		    value = xstrdup(uts.machine);
		else
		    txt = "";
	    }
	    break;
	  case ZNAMEID_DOMAINNAME: {
		struct utsname uts;

		if( !uname( &uts ) )
		    value = xstrdup(uts.domainname);
		else
		    txt = "";
	    }
	    break;
	#endif /*UNIX*/

	  case ZNAMEID_SCREEN:
	  #ifdef UNIX
	    if( JnxAvailable() )
		txt = "X";
	    else
		txt = "con";
	  #else
	    txt = "direct";
	  #endif
	    break;
	  case ZNAMEID_WRDATCRS:
	  case ZNAMEID_CHRATCRS:
	    txt = GetPtr2Line( fileHd, &nbytes );
	    GetFilePos( fileHd, &pos );
	    if( nbytes < pos )
		txt = "";
	    else {
		if( found == ZNAMEID_WRDATCRS ) {
		    n=pos;
		    delim = GetWordDelimiters();
		    for(; n < nbytes && !isspace(txt[n]) && txt[n] &&
						!strchr(delim,txt[n]); n++ )
			;
		    n -= pos;
		}
		else
		    n=1;
		value = xmalloc( n+1 );
		memcpy( value, txt+pos, n ); value[n] = '\0';
	    }
	    break;
	  case ZNAMEID_INMARK:
	    {
		int fhd2, type;
		ulong lnr, line1, line2;
		ushort pos, pos1, pos2;

		txt = "0";
		type = MarkInfo( &fhd2 );
		if( type && fhd2 == fileHd ) {
		    lnr = GetFilePos( fileHd, &pos );
		    line1 = MarkStart( &pos1);
		    line2 = MarkEnd( &pos2);
		    if( lnr >= line1 && lnr <= line2 ) {
			if( type == MARKTYPE_LINE )
			    txt = "1";
			else if( pos >= pos1 && pos <= pos2 )
			    txt = "1";
		    }
		}
	    }
	    break;


	  case ZNAMEID_INCMD:
	    GetFileInfo( fileHd, &flags );
	    txt = flags & WFILE_INCMD ? "1" : "0";
	    break;
	  case ZNAMEID_INSERT:
	    GetFileInfo( fileHd, &flags );
	    txt = flags & WFILE_INSERT? "1" : "0";
	    break;
	  case ZNAMEID_SCRLCK:
	    txt = ScrDrvGetToggleKeys() & 1 ? "1" : "0";
	    break;
	  case ZNAMEID_NUMLCK:
	    txt = ScrDrvGetToggleKeys() & 2 ? "1" : "0";
	    break;
	  case ZNAMEID_CAPLCK:
	    txt = ScrDrvGetToggleKeys() & 4 ? "1" : "0";
	    break;
	  case ZNAMEID_CMDLINE:
	    value = xstrdup( GetPtr2CmdLine( fileHd, &nbytes ) );
	    StripWSpaces( value );
	    break;
	  case ZNAMEID_VERSION:
	    txt = CopyRight(14);
	    break;
	  case ZNAMEID_PGMNAME:
	    txt = CopyRight(13);
	    break;
	  case ZNAMEID_COPYRIGHT:
	    txt = CopyRight(10);
	    break;
	  case ZNAMEID_LOADDIR:
	    txt = CopyRight(21);
	    break;
	  case ZNAMEID_OS:
	  #ifdef UNIX /* unix is always unix, but there is a Zsysname instead*/
	    txt = "UNIX";
	  #elif OS2
	    txt = "OS/2";
	  #elif MSDOS
	    txt = "MSDOS";
	  #else
	    txt = CopyRight(22);
	  #endif
	    break;
	  case ZNAMEID_CPOP:
	    if( !(value = PopValue() ))
		txt = "";
	    break;
	  case ZNAMEID_CTOP:
	    if( !(value = PeekValue(1) ))
		txt = "";
	    break;
	  case ZNAMEID_CTIME:
	    time( &tclock );
	    value = xstrdup(asctime(localtime(&tclock)));
	    value[24] = 0; /* strip lf */
	    break;

	  case ZNAMEID_DATE:
	    time( &tclock );
	    value = xmalloc( 10 );
	    strftime(value,10, "%d.%m.%y", localtime(&tclock) );
	    break;

	  case ZNAMEID_ISODATE:
	    time( &tclock );
	    value = xmalloc( 20 );
	    strftime(value,20, "%Y-%m-%d", localtime(&tclock) );
	    break;

	  case ZNAMEID_TIME:
	    time( &tclock );
	    value = xmalloc( 10 );
	    strftime(value,10, "%H:%M:%S", localtime(&tclock) );
	    break;

	  default:  /* look through the table */
	    for( found=0, n=0, var = varTbl; n < varTblSize; n++, var++ )
		if( var->name )
		    if( !strcmp( var->name, txt ) ) {
			found++;
			txt = var->value;
			break;
		    }
	    if( !found )
		txt = "";   /* default to an empty string */
	    break;
	}
    }
    else if( argType == ARGTYPE_VARENV ) {
	if( !(txt = getenv( txt )) )
	    txt = "";
    }

    if( !value )
	value = xstrdup( txt );
    return value;
}


/****************
 * ffnet den file: .varlist und schreibe alle variablen hinein.
 */

int
Cmd_VarList( cmd_t *cmd )
{
    int err=0, n, fhd;
    const char *name;
    char *buffer, *varVal;
    var_t *var;
  #if OS2 || UNIX
    int seq, pid, first, i;
    ushort tc, rc;
    static char t0[] = "----[Jobs]----";
  #endif
    static char t1[] = "----[Environment]----";
    static char t2[] = "----[User defined]----";
    static char t3[] = "----[System defined]----";
    static char t4[] = "----[Set options]----";
    static char t5[] = "----[Stack]----";

    if( !(err=Cmd_Edit(".varlist")) ) {
	fhd = QryScreenFile();
	ResetFileFlag( fhd, WFILE_RO );
	DeleteAllLines( fhd );
	buffer = xmalloc( MAX_LINELEN );

      #if OS2 || UNIX
	first = seq = 0;
	for(;(n=GetDetachedProcInfo( seq, buffer+23, MAX_LINELEN-25,
					       &pid, &tc, &rc )) != -1;seq++ )
	    if( n&0x7f ) {
		if( !first ) {
		    InsertLine( fhd, t0, sizeof(t0)-1 );
		    first++;
		}
		if( (n&0x7f) == 3 ) /* complete */
		    sprintf( buffer, "%c%d complete %5u-%u", n&0x80? '*':' ',
							     seq+1, tc, rc );
		else
		    sprintf( buffer, "%c%d %s pid %d", n&0x80? '*':' ', seq+1,
			     (n&0x7f) == 2 ? "running": "starting", pid );
		for(i=strlen(buffer); i < 23; i++ )
		    buffer[i] = ' ';

		InsertLine( fhd, buffer, strlen(buffer) );

	    }
      #endif

	for(n=1; varVal = PeekValue(n); n++ ) {
	    if( n==1 )
		InsertLine( fhd, t5, sizeof(t5)-1 );
	    sprintf( buffer, "%2d: '%.70s'", n, varVal );
	    InsertLine( fhd, buffer, strlen(buffer) );
	    free( varVal );
	}


	InsertLine( fhd, t2, sizeof(t2)-1 );
	for( n=0, var = varTbl; n < varTblSize; n++, var++ )
	    if( var->name ) {
		xassert( strlen(var->name)+strlen(var->value)+6 < MAX_LINELEN);
		sprintf( buffer, "%s = '%s'", var->name, var->value );
		InsertLine( fhd, buffer, strlen(buffer) );
	    }
	InsertLine( fhd, t3, sizeof(t3)-1 );
	for( n=0; zVarNames[n].name; n++ ) {
	    if( zVarNames[n].nameId == ZNAMEID_CPOP )
		continue; /* can't show this var, because its access would */
			  /* change the calculation stack */
	    varVal = GetVar(zVarNames[n].name, ARGTYPE_VAR);
	    sprintf( buffer, "%s = '%s'", zVarNames[n].name, varVal );
	    InsertLine( fhd, buffer, strlen(buffer) );
	    free( varVal );
	}
	InsertLine( fhd, t4, sizeof(t4)-1 );
	for( n = 1; name = GetSetOptionString(n); n++ )
	    InsertLine( fhd, name, strlen(name) );
	for( n = 0; name = GetScreenColorDef(n); n++ )
	    InsertLine( fhd, name, strlen(name) );

	InsertLine( fhd, t1, sizeof(t1)-1 );
	for( name = GetEnvironmentStrings(); *name; name += strlen(name)+1 )
	    InsertLine( fhd, name, strlen(name) );

	free(buffer);
	SeekLine( fhd, 0 );
	if( GetFileTotLines( fhd ) > 1 )
	    DeleteLine( fhd );
	Move2Pos( fhd, 0, 0 );
	ResetFileFlag( fhd, WFILE_CHG );
	SetFileFlag( fhd, WFILE_RO );
    }
    return err;
}


static void DoInit()
{
    if( isInitialized )
	return;
    AddCleanUp( CleanUp, NULL );
    isInitialized++;
}


static void CleanUp( void *dummy )
{
    size_t n;
    var_t *var;

    if( !isInitialized )
	return;

    for( n=0, var = varTbl; n < varTblSize; n++, var++ )
	if( var->name ) {
	    free( var->name );
	    free( var->value );
	}
    varTblSize = 0;
    free( varTbl );
    while( stackInd ) {
	stackInd--;
	free( stack[stackInd].value );
    }
    isInitialized = 0;
}
#endif

/*** bottom of file ***/
