/* [wscrans.c wk 01.02.94] Full Screen Handler using ANSI
 *	Copyright (c) 1994 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * See wscrdrv.c for an OS/2 and MSDOS compatible version.
 * This is a replacement for the file wscrdrv.c, which
 * is completly disabled for UNIX.
 *
 * History:
 */


#include "wtailor.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/environ.h>
#include <wk/keys.h>
#include <wk/wscrdrv.h>
#include <wk/mouse.h>
#include <wk/io.h>
#if __WATCOMC__
    #include <bios.h>
#endif
#if OS20
    #include <wk/bios.h>
#endif

#include "jnx.h"
#include "wkbddrv.h"

/****** constants **********/
#define MAX_DOSKEYS  20  /* our own pretype buffer */


/****************
 * Translation tables PC Colors to ANSI Colors;
 * one for background and one for foreground.
 * This Tables are internal to AnsiColor()
 */
static const char * const ansiBG[16] = {
	/*  0 */   "\x1b[0;40;"    ,
	/*  1 */   "\x1b[0;44;"    ,
	/*  2 */   "\x1b[0;42;"    ,
	/*  3 */   "\x1b[0;46;"    ,
	/*  4 */   "\x1b[0;41;"    ,
	/*  5 */   "\x1b[0;45;"    ,
	/*  6 */   "\x1b[0;43;"    ,
	/*  7 */   "\x1b[0;47;"    ,

	/*  8 */   "\x1b[0;40;5;"    ,
	/*  9 */   "\x1b[0;44;5;"    ,
	/*  A */   "\x1b[0;42;5;"    ,
	/*  B */   "\x1b[0;46;5;"    ,
	/*  C */   "\x1b[0;41;5;"    ,
	/*  D */   "\x1b[0;45;5;"    ,
	/*  E */   "\x1b[0;43;5;"    ,
	/*  F */   "\x1b[0;47;5;"
    };
static const char * const ansiFG[16] = {
	/*  0 */   "30m"    ,
	/*  1 */   "34m"    ,
	/*  2 */   "32m"    ,
	/*  3 */   "36m"    ,
	/*  4 */   "31m"    ,
	/*  5 */   "35m"    ,
	/*  6 */   "33m"    ,
	/*  7 */   "37m"    ,

	/*  8 */   "30;1m"  ,
	/*  9 */   "34;1m"  ,
	/*  A */   "32;1m"  ,
	/*  B */   "36;1m"  ,
	/*  C */   "31;1m"  ,
	/*  D */   "35;1m"  ,
	/*  E */   "33;1m"  ,
	/*  F */   "37;1m"
    };

/******* types *************/
typedef struct {
	byte c;
	byte a;
    } cell_t;


/******** globals **********/
static struct {
    ushort chr,scan;
    int state;
} kbdBuf[MAX_DOSKEYS]; /* array to store the keystrokes */
static int kbdBufCnt;  /* number of keys in buffer */

static int use_xwindows;
static int use_vcs;  /* virtual console memory (linux) */
static int fd_vcs;   /* handle of the vcs */
static int cursorFormHack;
static int isColor ;		  /* flag for color-mode */
static struct {
	int sx, sy, sxMul2;
	int x2, y2 ;
	int crsX, crsY ;
	int crsForm ;
	int crsActiv ;
	int size ;	    /*total size in bytes */
	int nchg;	    /*items in change list */
	struct {
	    int buttons;	/* # of mouse button s or 0 when no mouse */
	    int x,y;		/* current position */
	} mouse;
} screen ;		/* Screen Parameters*/

static cell_t *image ;	    /* area to store the image */
static ulong *changed;	    /* array of longs with bitwise coded change flags*/
static struct {
    char *image;	 /* area to store the original image */
    int crsX, crsY ;
} save;

static int translateCodes;
static volatile int *breakFlagAddr;
static void (*notifier)(int);

/********** prototypes *********/
static void ExposureHandler(void);
static void NotifyHandler(int);
static void SetCursorPos(void);

/*********** defined functions ********/

#define MarkChange(i)	do { changed[(i)/32] |= (1 << ((i) % 32)); } while(0)


#define AnsiOut(c)	do {	if( (byte)c < 32 || c == 127 )	   \
				    putc('~',stdout);  \
				else		       \
				    putc(c, stdout);   \
			} while(0)
#define AnsiFlush()	do { fflush(stdout); } while(0)
#define AnsiBell()	do { putc('\a',stdout); fflush(stdout); } while(0)
#define AnsiColor(a)	do { fputs( ansiBG[((a) & 0xf0)>>4], stdout ); \
			     fputs( ansiFG[(a) & 0x0f], stdout );      \
			} while(0)
#define AnsiClear()	do { fputs( "\x1b[2J"    , stdout); } while(0)
#define AnsiHome()	do { fputs( "\x1b[H"     , stdout); } while(0)
#ifdef __linux__
    /* "\e[?25l" will hide the cursor  and "\e[?25h" will show the cursor */
#define AnsiSave()	do { fputs( "\x1b[s\x1b[?25l", stdout); } while(0)
#define AnsiRest()	do { fputs( "\x1b[u\x1b[?25h", stdout); } while(0)
#else
#define AnsiSave()	do { fputs( "\x1b[s", stdout); } while(0)
#define AnsiRest()	do { fputs( "\x1b[u", stdout); } while(0)
#endif
#define AnsiPos(x,y)	do {					  \
			    printf("\x1b[%d;%df", (y)+1, (x)+1 ); \
			} while(0)

#define AnsiCrsForm1()	do { if( cursorFormHack ) \
				  fputs( "\x1b#c", stdout); } while(0)
#define AnsiCrsForm2()	do { if( cursorFormHack ) \
			     fputs( "\x1b#C", stdout); } while(0)


/*********** functions ****************/

/****************
 * Returns: 0 = standard color screen
 *	    1 = standard mono screen
 */


int ScrDrvOpen()
{
    int err=0, cols, rows;
  #if USE_VT_LINUX
    ushort x, y;
  #endif

    image = NULL;
    changed = NULL;
  #ifdef UNIX
    use_xwindows = JnxAvailable();
  #endif
  #if DEBUG
    xassert( sizeof (cell_t) == 2 );
  #endif

  #if USE_VT_LINUX
    if( use_xwindows )	{
	JnxQueryScreenSize( &cols, &rows );
    }
    else if( (fd_vcs = VTLinuxOpenVCS( &x, &y )) != -1 ) {
	use_vcs = 1;
	cols = x;
	rows = y;
    }
    else {
	if( VTLinuxGetScreenSize(1, &x, &y ) ) {
	    Error(0,"Error retrieving screensize - using 80*20");
	    x = 80;
	    y = 20;
	}
	cols = x;
	rows = y;
    }
  #elif defined(UNIX)
    if( use_xwindows )
	JnxQueryScreenSize( &cols, &rows );
    else  {
	cols = 80;
	rows = 24;
    }
  #else
    cols = 80;
    rows = 24;
  #endif
    isColor = 1;

    /**** Setup screen constants ****/
    screen.sx = cols ;
    screen.sy = rows ;
    screen.sxMul2 = cols*2 ;
    screen.x2 = cols -1 ;
    screen.y2 = rows -1 ;
    screen.size = screen.sx * screen.sy ;
    screen.nchg = (screen.size+31) / 32;
    if( !(image = calloc( screen.size, sizeof *image )) ) {
	err = -1;
	goto giveup;
    }
    if( !(changed = calloc( screen.nchg+1, sizeof *changed)) ) {
	err = -1;
	goto giveup;
    }
    if( use_vcs ) {
	if( !(save.image = malloc( screen.size * 2 )) ) {
	    err = -1;
	    goto giveup;
	}
    }

    if( use_xwindows ) {
	JnxRegisterExposureHandler( ExposureHandler );
	JnxRegisterNotifyHandler( NotifyHandler );
    }

    ScrDrvMouseOn();
    ScrDrvSaveScreen();
  #ifdef __linux__
    if( use_vcs ) { /* do we have the hacked drivers/char/console.c */
	cell_t x;

	fputs( "\x1b[H" "y" "\x1b#c", stdout);
	fflush(stdout);
	lseek(fd_vcs, 4, 0 );
	read(fd_vcs, &x, 2 );
	if( x.c == 'y' ) {
	  #ifdef DEBUG
	    Info("joh, wir haben den cursor form hack");
	  #endif
	    cursorFormHack = 1;
	}
    }
  #endif
    ScrDrvShowCrs( 1 ); /* initial painting of cursor */
    ScrDrvHideCrs();	/* and hide him */
    ScrDrvRepaint();	/* Sync with image */
    MouseShow();

  giveup:
    if( err ) {
	FREE(image);
	FREE(changed);
	Error(4,"Can't open CRT: not enough core");
    }
    return isColor ? 0 : 1;
}

void ScrDrvClose()
{
    ScrDrvRegBreakFlag( NULL );
    FREE( image );
    FREE(changed);
}



/****************
 * Stub for OS/2 / Windows Version
 */

long ScrDrvGetActiveHAB(void)
{
    return 0;
}


void ScrDrvSaveScreen()
{
    if( use_vcs ) {
	byte x;

	lseek(fd_vcs, 2, 0 );
	read(fd_vcs, &x, 1 );
	save.crsX = x;
	read(fd_vcs, &x, 1 );
	save.crsY = x;
	read(fd_vcs, save.image, screen.size * 2 );
    }
}


void ScrDrvRestScreen()
{
    if( use_xwindows )
	;
    else if( use_vcs ) {
	lseek(fd_vcs, 4, 0 );
	write(fd_vcs, save.image, screen.size * 2 );
	ScrDrvSetCrs( save.crsX, save.crsY );
    }
    else {
	AnsiHome();
	AnsiClear(); /* better than nothing */
    }
}



/****************
 *  enables or disables code translation
 */

void ScrDrvCodeTrans( int mode )
{
    translateCodes = mode ;
}

void ScrDrvRegBreakFlag( volatile int *flag )
{
    breakFlagAddr = flag;
}


void
ScrDrvRegNotifyHandler( void (*f)(int) )
{
    notifier = f;
}



void ScrDrvQrySize( int *x, int *y )
{
    *x = screen.sx ;
    *y = screen.sy ;
}


void ScrDrvSetCrs( int x, int y )
{
    screen.crsY=y;
    screen.crsX=x ;
    if( screen.crsActiv )
	SetCursorPos() ;
}


void ScrDrvHideCrs()
{
    screen.crsActiv = 0 ;
    SetCursorPos();
}

void ScrDrvMouseOn()
{
    screen.mouse.buttons = MouseInitialize();
    MousePut(screen.mouse.x, screen.mouse.y);
}


void ScrDrvMouseOff()
{
    MouseGet( &screen.mouse.x, &screen.mouse.y );
    MouseTerminate();
    screen.mouse.buttons = 0;
}


void ScrDrvShowCrs( int form )
{
    if( form == screen.crsForm )
	;   /* already set */
    else if( form == 1 ) {
	screen.crsForm = 1;
	if( !use_xwindows )
	    AnsiCrsForm1();
    }
    else if( form == 2 ) {
	screen.crsForm = 2;
	if( !use_xwindows )
	    AnsiCrsForm2();
    }
    screen.crsActiv = 1 ;
    SetCursorPos();
}


static void
ExposureHandler()
{
    ScrDrvRepaint();
    ScrDrvShow();
}



static void
NotifyHandler(int notification )
{
    switch( notification ) {
      case 1: /* ConfigureNotify (size changed) */
	JnxSetCaret( 0, 0, 0 ); /* remove the caret */
	ScrDrvClose();
	ScrDrvOpen();
	if( notifier )
	    (*notifier)(1);
	break;
      case 10: /* Left Button release */
      case 11: /* Left Button press */
      case 12: /* Right Button release */
      case 13: /* Right Button press */
      case 14: /* Middle Button release */
      case 15: /* Middle Button press */
	break;
      default: /* none */;
    }
}


void
ScrDrvShow()
{
    int n,i,j,x,x2,y,y2,any;
    byte lasta;
    cell_t *cp;
    ulong bitmap;

    x2=y2=0; lasta=0; /*(only to avoid compiler warning)*/
    any = 0;
    if( use_xwindows ) {
	for(n=0; n < screen.nchg; n++ ) {
	    if( bitmap = changed[n] ) {
		i = n * 32;
		y = i / screen.sx;
		x = i % screen.sx;
		if( !any ) {
		    /* get the first Attribute */
		    lasta= image[i].a;
		    JnxClearScreen();
		    JnxSetColor(lasta);
		    any++;
		}
		for(cp = image + i,j=0; j < 32; j++, i++, cp++,x++ ) {
		    if( x >= screen.sx ) {
			x = 0;
			y++;
			if( y >= screen.sy )
			    break;
		    }
		    if( bitmap & (1<<j) ) {
			if( lasta != cp->a ) {
			    lasta = cp->a;
			    JnxSetColor(lasta);
			}
			JnxWriteChar(x,y,cp->c);
		    }
		}
		changed[n] = 0;
	    }
	}
	if( any ) {
	    JnxWriteChar(0,0,-1); /* flush */
	}
    }
    else if( use_vcs ) {
	for(n=0; n < screen.nchg; n++ ) {
	    if( changed[n] )  {
		for(; n < screen.nchg; n++ )
		    changed[n] = 0;
		lseek(fd_vcs, 4, 0);
		write(fd_vcs, image, screen.size * sizeof *image);
		break;
	    }
	}
    }
    else {
	for(n=i=0; n < screen.nchg; n++ ) {
	    if( changed[n] ) {
		y = i / screen.sx;
		x = i % screen.sx;
		if( !any ) {
		    /* get the first Attribute */
		    lasta= image[i].a;
		    AnsiSave();
		    AnsiPos(x,y);
		    AnsiColor(lasta);
		    any++;
		}
		else if( !(x == x2 && y == y2) )
		    AnsiPos(x,y);
		for(cp = image + i,j=0; j < 32; j++, i++, cp++,x++ ) {
		    if( i < screen.size ) {
			if( lasta != cp->a ) {
			    lasta = cp->a;
			    AnsiColor(lasta);
			}
			AnsiOut(cp->c);
		    }
		}
		while( x >= screen.sx ) {
		    x = 0;
		    y++;
		}
		x2 = x;
		y2 = y;
		changed[n] = 0;
	    }
	    else
		i += 32;
	}
	if( any ) {
	    AnsiRest();
	    AnsiFlush();
	}
    }
}


void ScrDrvRepaint()
{
    int i;

    for(i=0; i < screen.nchg; i++ )
	changed[i] = ~0;
  #if 0
    if( use_vcs )
	printf("\x1b%s", getenv("W_CHAR"));  /* switch G0 to NULL_Map */
	/*  G0_charset:
	 * \x1b(0 := GRAF_MAP
	 * \x1b(B := NORM_MAP
	 * \x1b(U := NULL_MAP
	 * \x1b(K := USER_MAP
	 *   use ) to switch G1_charset
	 */
   #endif
}



/****************
 * Query the Mouse
 * Returns: Number of Buttons or 0 if no Mouse available in this screen Area
 */

int
ScrDrvQryMouse( int *btn, int *px, int *py )
{
  #ifdef UNIX
    if( use_xwindows ) {
	unsigned bstate;
	bstate = JnxReadMouse(px, py);
	*btn = 0;
	if( bstate & 1 )
	    *btn |= MOUSE_LEFT;
	if( bstate & 4 )
	    *btn |= MOUSE_RIGHT;
	return 3;
    }
  #endif
    *btn = MouseGet( px, py );
    return screen.mouse.buttons;
}



/****************
 * Read a cell from the screen, that is, get it from the image
 */

void ScrDrvReadCell( int x, int y, int *c, int *a )
{
    int i;

    i = y * screen.sx + x ;
    if( i >= 0 && i < screen.size ) {
	*c = translateCodes ? MapIbm2Iso( image[i].c, 1 ) : image[i].c;
	*a = image[i].a;
    }
}



/****************
 * Write a Cell to the image
 */

void ScrDrvWriteCell( int x, int y, int c, int a )
{
    int i;

    i = y * screen.sx + x ;
    if( (c & 0x80) && translateCodes ) /* translate that character */
	c = MapIso2Ibm(c,1);
    if( i >= 0 && i < screen.size ) {
	if( image[i].c != c || image[i].a != a ) {
	    image[i].c = c;
	    image[i].a = a;
	    MarkChange(i);
	}
    }
}


void
ScrDrvWriteStr( int x, int y, const char *buf, size_t len, int a )
{
    int i, c;

    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; len && i < screen.size; len--, i++, buf++ ) {
	    c = *buf;
	    if( (c & 0x80) && translateCodes )
		c = MapIso2Ibm(c,1);
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		MarkChange(i);
	    }
	}
}


void
ScrDrvWriteAttrStr( int x, int y, const char *buf,
		    size_t len, const byte *abuf )
{
    int i, c, a;

    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; len && i < screen.size; len--, i++, buf++, abuf++ ) {
	    c = *buf;
	    a = *abuf;
	    if( (c & 0x80) && translateCodes )
		c = MapIso2Ibm(c,1);
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		MarkChange(i);
	    }
	}
}


/****************
 * Einen String der laenge len ausgeben und rechts mit Blanks
 * bis auf laenge maxlen auffuellen. Attribute werden nicht geschrieben.
 * der String wird aber nur solange ausgegeben bis auch maxlen erreicht ist;
 * nur ein eventuell kuerzerer String wird mit blanks aufgefuellt.
 */

void ScrDrvWriteStrFill( int x, int y, const char *buf,
			 size_t len, size_t maxlen )
{
    int i, c;

    i = y * screen.sx + x ;
    if( i >= 0 ) {
	for( ; maxlen && len && i < screen.size; len--, maxlen--,i++, buf++ ) {
	    c = *buf;
	    if( (c & 0x80) && translateCodes )
		c = MapIso2Ibm(c,1);
	    if( image[i].c != c ) {
		image[i].c = c ;
		MarkChange(i);
	    }
	}
	for( ; maxlen && i < screen.size; maxlen--,i++ ) {
	    if( image[i].c != ' ' ) {
		image[i].c = ' ' ;
		MarkChange(i);
	    }
	}
    }
}



void ScrDrvWriteNChar( int x, int y, int c, int a, size_t n )
{
    int i;

    i = y * screen.sx + x ;
    if( (c & 0x80) && translateCodes )
	c = MapIso2Ibm(c,1);
    if( i >= 0 )
	for( ; n && i < screen.size; n--, i++ ) {
	    if( image[i].c != c || image[i].a != a ) {
		image[i].c = c ;
		image[i].a = a ;
		MarkChange(i);
	    }
	}
}


/****************
 * Nur Attribute schreiben
 */

void ScrDrvWriteNAttr( int x, int y, int a, size_t n )
{
    int i;

    i = y * screen.sx + x ;
    if( i >= 0 )
	for( ; n && i < screen.size; n--, i++ ) {
	    if( image[i].a != a ) {
		image[i].a = a ;
		MarkChange(i);
	    }
	}
}



/****************
 * This is only a dummy
 */

void ScrDrvPauseCrt( int mode )
{
}




static void
SetCursorPos()
{
    if( use_xwindows ) {
	if( screen.crsActiv )
	    JnxSetCaret( screen.crsX, screen.crsY, screen.crsForm==1? 1 : 2 );
	else
	    JnxSetCaret( 0, 0, 0 );
    }
    else {
	if( screen.crsActiv )
	    AnsiPos(screen.crsX, screen.crsY);
	else
	    AnsiPos(0, screen.y2);
	AnsiFlush();
    }
}


/****************
 * Ermittelt die aktuellen Togglekey states:
 * Bit 0 = Scroll Lock
 *     1 = Num Lock
 *     2 = Caps Lock
 */

unsigned
ScrDrvGetToggleKeys()
{
    unsigned state, flags;

    if( use_xwindows )
	return 0;  /* fixme */

  #if USE_VT_LINUX
    flags = VTLinuxGetKeyState();
    /* bit 4 := scroll lock active
     *	   5 := num lock active
     *	   6 := caps lock active
     */
    state = (flags >> 4) & 7;
  #else
    state = 0;
  #endif
    return state;
}




/****************
 * Get the scancode, the value, and the status;
 * look at the mouse buttons.
 * when a mouse buuton is hit, K_FLAG_MOUSE is set in flags
 * and rVal contains the keycode K_BTN_xxxxx
 *
 * Returns wether a keyvalue is available or not
 */

int
ScrDrvBasicInkey( ushort *rVal, ushort *rScan, unsigned *rFlags )
{
    ushort flags, chr, scan;
    int state, i, msX, msY;
    unsigned msBtn;
  static unsigned lastMsBtn;
  #if OS20
    unsigned kstate;
  #endif

    chr = scan = flags = 0;
  #if OS20
    if( WKKeyboardGetChar( 0, &chr, &scan, &kstate ) )
	return 0;/* fehler (z.B. kein focus */
    flags |= kstate;
  #else
    ScrDrvPoll();
    if( kbdBufCnt ) {
	chr   = kbdBuf[0].chr;
	scan  = kbdBuf[0].scan;
	state = kbdBuf[0].state;
	for(i=1; i < kbdBufCnt; i++ )
	    kbdBuf[i-1] = kbdBuf[i];
	kbdBufCnt--;
    }
    else if( msBtn = MouseGet( &msX, &msY ) ) {
	if( lastMsBtn == msBtn )
	    return 0;

	lastMsBtn = msBtn;
      #if EMX  || OS2|| UNIX
	state = 0; /* ?? */
      #elif __WATCOMC__
	state = _bios_keybrd(_KEYBRD_SHIFTSTATUS);
      #else
	state = bioskey(0x02);
      #endif
	flags |= K_FLAG_MOUSE;
	if( msBtn & MOUSE_LEFT )
	    chr = K_BTN_LEFT;
	else if( msBtn & MOUSE_RIGHT )
	    chr = K_BTN_RIGHT;
	else if( msBtn & MOUSE_LEFTDBL )
	    chr = K_BTN_LEFTDBL;
	else /* MOUSE_RIGHTDBL */
	    chr = K_BTN_RIGHTDBL;
    }
    else {
	lastMsBtn = 0;
	return 0;
    }


    if( state & ( 0x01| 0x02) )
	flags |= K_FLAG_SHIFT;
    if( state & 0x04 )
	flags |= K_FLAG_CTRL;
    if( state & 0x08 )
	flags |= K_FLAG_ALT;
  #endif

    *rVal  = chr;
    *rScan = scan;
    *rFlags =flags;
    return 1;
}







/****************
 * Poll Fuktion sollte zumindest bei DOS hauefig aufgerufen werden.
 * Hintergrund:
 * Leider hat die DOS Key Queue keine Infos ueber den
 * akt. shiftstate, allso muessen wir versuchen diese
 * Infos selber zu speichern
 */

void ScrDrvPoll(void)
{
    if( use_xwindows )
	return;

  #if USE_VT_LINUX
    VTLinuxPoll();
  #elif OS20
    return;
  #else
    int state, c;

    if( c = Inkey() ) {
	state = 0;
	if( kbdBufCnt+1 < MAX_DOSKEYS ) {
	    kbdBuf[kbdBufCnt].chr  = c & 0xff;
	    kbdBuf[kbdBufCnt].scan = (c >> 8) & 0xff;
	    kbdBuf[kbdBufCnt].state= state;
	    kbdBufCnt++;
	}
	else /* buffer overflow */
	    AnsiBell();
    }
  #endif
}

/**** end of file ***/
