/* [wkbddrv.c wk 24.11.91] Keyboard driver
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 06.11.94 wk	Auf Wunsch von Peter ein nice eingebaut fr MS-Doof
 */


#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/lib.h>
#include <wk/string.h>
#define INCL_ALT_KEYCODES 1
#include <wk/keys.h>
#include <wk/mouse.h>
#include <wk/environ.h>
#include <wk/wscrdrv.h>

#include "w.h"
#include "wcmd.h"
#include "wkbddrv.h"
#include "wscreen.h"
#include "jnx.h"


#if USE_VT_LINUX
static struct {
    ushort  scan;
    ushort  baseCode;	   /* from the US PC keyboard layout */
    byte  std,shift,meta;  /* value mapping for the German Keyboard (latin-1)*/
} scanCodeMap[] = {
  /* Must be sorted by .scan */
  {   1, K_ESCAPE			},
  {   2, '1',   '1', '!'                },
  {   3, '2',   '2', '\"',''           },
  {   4, '3',   '3', '', ''           },
  {   5, '4',   '4', '$'                },
  {   6, '5',   '5', '%'                },
  {   7, '6',   '6', '&'                },
  {   8, '7',   '7', '/', '{'           },
  {   9, '8',   '8', '(', '['           },
  {  10, '9',   '9', ')', ']'           },
  {  11, '0',   '0', '=', '}'           },
  {  12, '-',   '', '?', '\\'          },
  {  13, '=',   '', '`',               },
  {  14, K_RUBOUT			},
  {  15, K_TAB				},
  {  16, 'Q',   'q', 'Q', '@'           },
  {  17, 'W',   'w', 'W'                },
  {  18, 'E',   'e', 'E'                },
  {  19, 'R',   'r', 'R'                },
  {  20, 'T',   't', 'T'                },
  {  21, 'Y',   'z', 'Z'                },
  {  22, 'U',   'u', 'U'                },
  {  23, 'I',   'i', 'I'                },
  {  24, 'O',   'o', 'O'                },
  {  25, 'P',   'p', 'P'                },
  {  26, '[',   '', ''                },
  {  27, ']',   '+', '*', '~'           },
  {  28, K_RETURN			},
  /* 29,  left-ctrl */
  {  30, 'A',   'a', 'A'                },
  {  31, 'S',   's', 'S'                },
  {  32, 'D',   'd', 'D'                },
  {  33, 'F',   'f', 'F'                },
  {  34, 'G',   'g', 'G'                },
  {  35, 'H',   'h', 'H'                },
  {  36, 'J',   'j', 'J'                },
  {  37, 'K',   'k', 'K'                },
  {  38, 'L',   'l', 'L'                },
  {  39, ';',   '', ''                },
  {  40, '\'',  '', ''                },
  {  41, '^',   '^', ''                },
  /* 42, left-shift */
  {  43, '\\',  '#', '\''               },
  {  44, 'Z',   'y', 'Y'                },
  {  45, 'X',   'x', 'X'                },
  {  46, 'C',   'c', 'C'                },
  {  47, 'V',   'v', 'V'                },
  {  48, 'B',   'b', 'B'                },
  {  49, 'N',   'n', 'N'                },
  {  50, 'M',   'm', 'M'                },
  {  51, ',',   ',', ';'                },
  {  52, '.',   '.', ':'                },
  {  53, '/',   '-', '_'                },
  /* 54, right-shift */
  {  55, K_PAD_MUL			},
  /* 56, left-alt */
  {  57, ' ',   ' ', ' '                },
  {  58, K_CAPLCK			},
  {  59, K_F1				},
  {  60, K_F2				},
  {  61, K_F3				},
  {  62, K_F4				},
  {  63, K_F5				},
  {  64, K_F6				},
  {  65, K_F7				},
  {  66, K_F8				},
  {  67, K_F9				},
  {  68, K_F10				},
  {  69, K_NUMLCK			},
  {  70, K_SCROLL			},
  {  71, K_PAD_7			},
  {  72, K_PAD_8			},
  {  73, K_PAD_9			},
  {  74, K_PAD_MINUS			},
  {  75, K_PAD_4			},
  {  76, K_PAD_5			},
  {  77, K_PAD_6			},
  {  78, K_PAD_PLUS			},
  {  79, K_PAD_1			},
  {  80, K_PAD_2			},
  {  81, K_PAD_3			},
  {  82, K_PAD_INS			},
  {  83, K_PAD_DEL			},
  /* 84  not used */
  /* 85  not used */
  {  86, '<',      '<', '>', '|'        },  /*(German 102-Key)*/
  {  87, K_F11				},
  {  88, K_F12				},
  /* 89  not used */
  /* 90  not used */
  /* 91  not used */
  /* 92  not used */
  /* 93  not used */
  /* 94  not used */
  /* 95  not used */
  {  96, K_PAD_RETURN			},
  /* 97, right ctrl */
  {  98, K_PAD_DIV			},
  {  99, K_PRINT			},
  /*100, right alt */
  /*101  not used */
  { 102, K_HOME 			},
  { 103, K_UP				},
  { 104, K_PGUP 			},
  { 105, K_LEFT 			},
  { 106, K_RIGHT			},
  { 107, K_END				},
  { 108, K_DOWN 			},
  { 109, K_PGDN 			},
  { 110, K_INS				},
  { 111, K_DEL				},
  /*112  not used */
  /*113  not used */
  /*114  not used */
  { 115, K_SUPER_L			},
  { 116, K_SUPER_R			},
  { 117, K_HYPER_R			},
  /*118  not used */
  { 119, K_PAUSE			},
};

/* Shift keys are not included: */
/*   29,  left-ctrl */
  /* 42, left -shift */
  /* 54, right shift */
  /* 56, left alt */
  /* 71, right alt */
  /* 97, right ctrl */

#else

static struct {
    ushort  scan;
    ushort  baseCode;	  /* from the US PC keyboard layout */
} scanCodeMap[] = {
  /* Must be sorted by .scan */
  {   1 , K_ESCAPE	  }, {	 2 , '1'             },
  {   3 , '2'             }, {   4 , '3'             },
  {   5 , '4'             }, {   6 , '5'             },
  {   7 , '6'             }, {   8 , '7'             },
  {   9 , '8'             }, {  10 , '9'             },
  {  11 , '0'             }, {  12 , '-'             },
  {  13 , '='             }, {  14 , K_RUBOUT        },
  {  15 , K_TAB 	  }, {	16 , 'Q'             },
  {  17 , 'W'             }, {  18 , 'E'             },
  {  19 , 'R'             }, {  20 , 'T'             },
  {  21 , 'Y'             }, {  22 , 'U'             },
  {  23 , 'I'             }, {  24 , 'O'             },
  {  25 , 'P'             }, {  26 , '['             },
  {  27 , ']'             }, {  28 , K_RETURN        },
  {  30 , 'A'             }, {  31 , 'S'             },
  {  32 , 'D'             }, {  33 , 'F'             },
  {  34 , 'G'             }, {  35 , 'H'             },
  {  36 , 'J'             }, {  37 , 'K'             },
  {  38 , 'L'             }, {  39 , ';'             },
  {  40 , '\''            }, {  41 , '`'             },
  {  43 , '\\'            }, {  44 , 'Z'             },
  {  45 , 'X'             }, {  46 , 'C'             },
  {  47 , 'V'             }, {  48 , 'B'             },
  {  49 , 'N'             }, {  50 , 'M'             },
  {  51 , ','             }, {  52 , '.'             },
  {  53 , '/'             }, {  55 , K_PAD_MUL       },
  {  57 , ' '             }, {  59 , K_F1            },
  {  60 , K_F2		  }, {	61 , K_F3	     },
  {  62 , K_F4		  }, {	63 , K_F5	     },
  {  64 , K_F6		  }, {	65 , K_F7	     },
  {  66 , K_F8		  }, {	67 , K_F9	     },
  {  68 , K_F10 	  }, {	71 , K_PAD_7	     },
  {  72 , K_PAD_8	  }, {	73 , K_PAD_9	     },
  {  74 , K_PAD_MINUS	  }, {	75 , K_PAD_4	     },
  {  76 , K_PAD_5	  }, {	77 , K_PAD_6	     },
  {  78 , K_PAD_PLUS	  }, {	79 , K_PAD_1	     },
  {  80 , K_PAD_2	  }, {	81 , K_PAD_3	     },
  {  82 , K_PAD_INS	  }, {	83 , K_PAD_DEL	     },
  {  84 , K_F1		  }, {	85 , K_F2	     },
  {  86 , K_F3		  }, {	87 , K_F4	     },
  {  88 , K_F5		  }, {	89 , K_F6	     },
  {  90 , K_F7		  }, {	91 , K_F8	     },
  {  92 , K_F9		  }, {	93 , K_F10	     },
  {  94 , K_F1		  }, {	95 , K_F2	     },
  {  96 , K_F3		  }, {	97 , K_F4	     },
  {  98 , K_F5		  }, {	99 , K_F6	     },
  { 100 , K_F7		  }, { 101 , K_F8	     },
  { 102 , K_F9		  }, { 103 , K_F10	     },
  { 104 , K_F1		  }, { 105 , K_F2	     },
  { 106 , K_F3		  }, { 107 , K_F4	     },
  { 108 , K_F5		  }, { 109 , K_F6	     },
  { 110 , K_F7		  }, { 111 , K_F8	     },
  { 112 , K_F9		  }, { 113 , K_F10	     },
  { 114 , K_PAD_MUL	  }, { 115 , K_PAD_4	     },
  { 116 , K_PAD_6	  }, { 117 , K_PAD_1	     },
  { 118 , K_PAD_3	  }, { 119 , K_PAD_7	     },
  { 120 , '1'             }, { 121 , '2'             },
  { 122 , '3'             }, { 123 , '4'             },
  { 124 , '5'             }, { 125 , '6'             },
  { 126 , '7'             }, { 127 , '8'             },
  { 128 , '9'             }, { 129 , '0'             },
  { 130 , '/'             }, { 131 , '='             },
  { 132 , K_PAD_9	  }, { 133 , K_F11	     },
  { 134 , K_F12 	  }, { 135 , K_F11	     },
  { 136 , K_F12 	  }, { 137 , K_F11	     },
  { 138 , K_F12 	  }, { 139 , K_F11	     },
  { 140 , K_F12 	  }, { 141 , K_PAD_8	     },
  { 142 , K_PAD_MINUS	  }, { 143 , K_PAD_5	     },
  { 144 , K_PAD_PLUS	  }, { 145 , K_PAD_2	     },
  { 146 , K_PAD_INS	  }, { 147 , K_PAD_DEL	     },
  { 149 , K_PAD_DIV	  }, { 150 , K_PAD_MUL	     },
  { 151 , K_HOME	  }, { 152 , K_UP	     },
  { 153 , K_PGUP	  }, { 155 , K_LEFT	     },
  { 157 , K_RIGHT	  }, { 159 , K_END	     },
  { 160 , K_DOWN	  }, { 161 , K_PGDN	     },
  { 162 , K_INS 	  }, { 163 , K_DEL	     },
  { 164 , K_PAD_DIV	  }, { 166 , K_PAD_RETURN    },
  { 224 , K_RETURN	  },
};

#endif

typedef struct deadkey {
    struct deadkey *next;
    byte one,sone;
    byte two,stwo;
    byte value, scan, state;
} *DEADKEY;




static void Cleanup( void *dummy );
static char *ParseKeyDefValue( char *p, int *ret_value, int mode );
static int isInitialized;
static char prefixNextKey;
static int translateCodes;
static DEADKEY deadkeys;
static ushort	multi_key;
static ushort	pendingMulti;
static ushort	pendingScan;
static unsigned pendingFlag;
static ushort	pendingChar;
static int	pendingBaseCode;
static unsigned debugflags;


#if USE_VT_LINUX
static int
GetBaseCode( ushort scan, unsigned kstate, ushort *ret_chr )
{
    int i, left=0, right=DIM(scanCodeMap)-1;

    kstate &= 15; /* we only need the lower 4 bits */
    do {
	i = (left+right)/2;
	if( scan < scanCodeMap[i].scan )
	    right = i-1;
	else if( scan > scanCodeMap[i].scan )
	    left = i+1;
	else {
	    if( !ret_chr )
		;
	    else if( !kstate )
		*ret_chr = scanCodeMap[i].std;
	    else if( kstate == 1 )
		*ret_chr = scanCodeMap[i].shift;
	    else if( kstate == 8 || kstate == 4 || kstate == 12 )
		*ret_chr = scanCodeMap[i].meta;
	    else
		*ret_chr = 0;
	    return scanCodeMap[i].baseCode;
	}
    } while( right >= left );

    return 0;
}

#else	/* not USE_VT_LINUX */

static int
GetBaseCode( ushort scan )
{
    int i, left=0, right=DIM(scanCodeMap)-1;

    do {
	i = (left+right)/2;
	if( scan < scanCodeMap[i].scan )
	    right = i-1;
	else if( scan > scanCodeMap[i].scan )
	    left = i+1;
	else
	    return scanCodeMap[i].baseCode;
    } while( right >= left );

    return 0;
}

/****************
 * Special handling for the pad keys
 * Our definition is:
 *    In ancient PC times there where only the pad-keys.
 */

static int
RemapBaseCode( int code, unsigned kstate )
{

    switch( code ) {
      case K_PAD_7   : code = K_HOME; break;
      case K_PAD_8   : code = K_UP; break;
      case K_PAD_9   : code = K_PGUP; break;
      case K_PAD_4   : code = K_LEFT; break;
      case K_PAD_6   : code = K_RIGHT; break;
      case K_PAD_1   : code = K_END;  break;
      case K_PAD_2   : code = K_DOWN; break;
      case K_PAD_3   : code = K_PGDN; break;
      case K_PAD_INS : code = K_INS ; break;
      case K_PAD_DEL : code = K_DEL ; break;
      default: break;
    }
    return code;
}

static int
RemapScanCode( int chr, unsigned kstate )
{
    int code;

    switch( chr ) {
      case 47  : code = K_PAD_DIV ; break;
      default: code = K_PAD_RETURN;
    }
    return code;
}
#endif /* USE_VT_LINUX */

static void
Cleanup( void *dummy )
{
    DEADKEY d, d2;
    for( d = deadkeys; d; d = d2 ) {
	d2 = d->next;
	free(d);
    }
    deadkeys=NULL;
}


void
KybrdDrvOpen( int mode )
{
    if( !isInitialized ) {
	AddCleanUp( Cleanup, NULL );
	isInitialized = 1;
    }
    KybrdDrvSetTrans( mode );
  #ifdef UNIX
    if( JnxAvailable() )
	return; /*fixme*/
  #endif
  #if USE_VT_LINUX
    if( VTLinuxOpen() ) {
	Error(0,"Sorry, this will only run on the console");
	Error(0,"       - check, that you have r/w access to /dev/tty?");
	Error(2,"       - or try the X version by using the proper option");
    }
  #endif
}


/****************
 * Die translationtable auswaehlen:
 * mode 0 := standard
 *	1 := Latin-1
 */

void
KybrdDrvSetTrans( int mode )
{
    translateCodes = mode;
}


void
KybrdDrvSetDebug( unsigned flags )
{
    debugflags = flags;
}


void
KybrdDrvSetPrefix( char pref )
{
    prefixNextKey = pref;
}

char
KybrdDrvGetPrefix()
{
    return prefixNextKey;
}



int
KybrdDrvGetKeyId( int *ret_value )
{
    ushort chr, scan;
    unsigned kstate, ks, flag;
    int baseCode, keyid;
    DEADKEY d;

  #if USE_VT_LINUX
    if( JnxAvailable() ) {
	if( !JnxReadKeyCode( &kstate, &flag ) )
	    return 0;
	scan = kstate;
	/*printf("scan=%3d  flag=%04x\n", scan, flag ); */
    }
    else {
	if( !VTLinuxReadScan( &scan, &flag ) )	{
	    Sleep(0);  /* be nice */
	    return 0;
	}
    }

    chr = 0;
    if( flag & 128 )  /* mouse! */
	baseCode = scan;
    else /* get the base-key from the scan-code */
	baseCode = GetBaseCode(scan, flag, &chr);

    /* map flag to kstate */
    kstate = 0;
    if( flag & 1 )
	kstate |= K_FLAG_SHIFT;
    if( flag & 2 )
	kstate |= K_FLAG_CTRL;
    if( flag & 4 )
	kstate |= K_FLAG_ALT;
    if( flag & 128 )
	kstate |= K_FLAG_MOUSE;

    if( !pendingMulti && multi_key && scan == multi_key )  {
	pendingMulti = 1;
	return 0;
    }

    if( pendingMulti && !chr )
	return 0;  /* ingore all non-character keys */

    if( !pendingScan && multi_key && !pendingMulti )
	;
    else if( !pendingScan ) {
	pendingMulti = 0;
	if( scan && !(kstate & ~(K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT)) ) {
	    ks = kstate & (K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT);
	    for(d=deadkeys; d; d = d->next )
		if( d->one == scan && d->sone == ks ) {
		    pendingScan = scan;
		    pendingChar = chr;
		    pendingFlag = ks;
		    pendingBaseCode = baseCode;
		    return 0;
		}
	}
    }
    else {
	pendingMulti = 0;
	d = NULL;
	if( scan && !(kstate & ~(K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT)) ) {
	    ks = kstate & (K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT);
	    for(d=deadkeys; d; d = d->next )
		if( d->two == scan && d->stwo == ks &&
		    d->one == pendingScan &&
		    d->sone== pendingFlag )
		    break;
	}
	if( d ) {
	    scan = d->scan;
	    kstate = d->state;
	    chr = d->value;
	    baseCode = GetBaseCode(scan, 0, NULL);
	}
	else {
	    scan = pendingScan;
	    kstate = pendingFlag;
	    chr = pendingChar;
	    baseCode = pendingBaseCode;
	}
	pendingScan = 0;
    }



    /* remap some basecodes to K_VK_xxx values */
    if( baseCode == K_ESCAPE )
	baseCode = K_VK_ESCAPE;
    else if( baseCode == K_RUBOUT )
	baseCode = K_VK_RUBOUT;
    else if( baseCode == K_TAB )
	baseCode = K_VK_TAB;
    else if( baseCode == K_RETURN )
	baseCode = K_VK_RETURN;
  #else
    if( !ScrDrvBasicInkey( &chr, &scan, &kstate ) )  {
      #if OS2
	Sleep(60);  /* be nice */
      #elif __WATCOMC__ && MSDOS
	Sleep(0);  /* be nice */
      #endif
	return 0;
    }
    if( !pendingScan ) {
	if( scan && !(kstate & ~(K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT)) ) {
	    ks = kstate & (K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT);
	    for(d=deadkeys; d; d = d->next )
		if( d->one == scan && d->sone == ks ) {
		    pendingScan = scan;
		    pendingChar = chr;
		    pendingFlag = ks;
		    return 0;
		}
	}
    }
    else {
	d = NULL;
	if( scan && !(kstate & ~(K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT)) ) {
	    ks = kstate & (K_FLAG_SHIFT|K_FLAG_CTRL|K_FLAG_ALT);
	    for(d=deadkeys; d; d = d->next )
		if( d->two == scan && d->stwo == ks &&
		    d->one == pendingScan &&
		    d->sone== pendingFlag )
		    break;
	}
	if( d ) {
	    scan = d->scan;
	    kstate = d->state;
	    chr = d->value;
	}
	else {
	    scan = pendingScan;
	    kstate = pendingFlag;
	    chr = pendingChar;
	}
	pendingScan = 0;
    }



    if( kstate & K_FLAG_MOUSE )
	baseCode = chr;
    else {
	/* get the base-key from the scan-code */
	if( scan == 86 && chr )
	    baseCode = 0; /* special for german kbd (same scan for '<' & s-f3)*/
	else
	    baseCode = GetBaseCode(scan);
	if( chr == 224 )
	    baseCode = RemapBaseCode(baseCode, kstate );
	else if( scan == 224 )
	    baseCode = RemapScanCode(chr, kstate );

	/* remap some basecodes to K_VK_xxx values */
	if( baseCode == K_ESCAPE )
	    baseCode = K_VK_ESCAPE;
	else if( baseCode == K_RUBOUT )
	    baseCode = K_VK_RUBOUT;
	else if( baseCode == K_TAB )
	    baseCode = K_VK_TAB;
	else if( baseCode == K_RETURN )
	    baseCode = K_VK_RETURN;
    }
  #endif

    if( debugflags & 1 ) {
	const char *s;

	fprintf(stderr, "scan %3d  char %3d  key %3d", (int)scan, (int)chr, baseCode );
	if( kstate & K_FLAG_SHIFT )
	    fputs( " shift", stderr);
	if( kstate & K_FLAG_CTRL )
	    fputs( " ctrl", stderr);
	if( kstate & K_FLAG_ALT   )
	    fputs( " alt ", stderr);
	if( kstate & K_FLAG_COMPOSE )
	    fputs( " comp", stderr);
	if( *(s = WklibKeyCode2String(baseCode)) )
	    fprintf(stderr, " (%s)\n", WklibKeyCode2String(baseCode) );
	else if( baseCode < 256 )
	    fprintf(stderr, " (%c)\n", baseCode );
	else
	    putc('\n',stderr);
    }

    /* create the keyid */
    keyid = baseCode;

    /* check for the special mouse actions */
    if( kstate & K_FLAG_MOUSE ) {
	int msBtn, msX, msY, i;

	ScrDrvQryMouse( &msBtn, &msX, &msY );
	if( i = ScreenQryMenuID( msX, msY ) ) {
	     i--;
	     xassert( i < 4 );
	     keyid = K_USER_FNC0 + i*2;
	     if( msBtn & (MOUSE_RIGHT|MOUSE_RIGHTDBL))
		 keyid++;
	 }
    }

    if( !keyid )  /* may be the alt-3-digit-sequence */
	keyid = K_VK_COMPOSED; /* and not modifiers */
    else if( prefixNextKey ) {
	switch( prefixNextKey ) {
	  case 's': case 'S': keyid |= KEYMOD_SHIFT; break;
	  case 'c': case 'C': keyid |= KEYMOD_CTRL;  break;
	  case 'a': case 'A': keyid |= KEYMOD_ALT;   break;
	  case 'u': case 'U': keyid |= KEYMOD_USR;   break;
	  case 'x': case 'X': keyid |= KEYMOD_EXT;   break;
	}
	prefixNextKey = 0;
    }
    else {
	if( kstate & K_FLAG_SHIFT )
	    keyid |= KEYMOD_SHIFT;
	if( kstate & K_FLAG_CTRL )
	    keyid |= KEYMOD_CTRL;
	if( kstate & K_FLAG_ALT   )
	    keyid |= KEYMOD_ALT;
    }


    if( ret_value ) {
	*ret_value = chr & 0xff;
	if( translateCodes && (*ret_value & 0x80) )
	    *ret_value = MapIbm2Iso( *ret_value, 1 );
    }

    if( keyid == K_EXIT )   /* issued by GUI CloseWindow */
	exit(0);

    return keyid;
}


/****************
 * Define a key (to change keyboard layout), the buffer contains the
 * definition string -- this function is allowed to change the buffer!
 * NOTE: Currently only used with linux console mode.
 * Syntax for the defintionstring (all values are in standard c notation):
 *
 *  "scan" <scancode> <std>|"-" <shift>|"-" <meta>|"."
 *
 *  "compose" modifier <first> modifier <second> <value>
 *	      [ modifier <scan> ]
 *
 *  "multikey"  multikey_scancode
 *
 *  with modifier := ["shift"|"ctrl"|"alt"]
 *
 * where the constant "scan" identifies this as a scancode definition,
 * <scancode> is the keyboard scancode
 * <std>      the value to produce with no shift keys
 * <shift>    the value to produce with shift-key pressed
 * <meta>     the value to produce with the meta-key (right-alt-key) pressed
 * <first>    the scancode of the first key pressed
 * <second>   the scancode of the second key pressed
 * <result>   and this result is produced
 *	      (when there is no valid secondscan for a firstscan,
 *	       first is used and second is ignored)
 *	       a result of 0 deletes this entry.
 *
 *	   (values may also be given as character enclosed in single quotes)
 */

int
KybrdDrvDefKey(char *buffer)
{
    char *p, *p2;
    int i;
    int scan, std,shift,meta, one,sone,two,stwo,result, fakescan, sfakescan;

    for(p = buffer; isspace(*p) ; p++ )
	;
    if( p2 = strpbrk(p, " \t\n") )
	*p2++ = 0;
    else
	return ERR_INVARG;
    if( !stricmp(p,"scan") ) {
	p = p2;
	if( !(p = ParseKeyDefValue(p, &scan,0)) ) /* scancode */
	    return ERR_INVARG;
	if( !(p = ParseKeyDefValue(p, &std,1)) ) /* std */
	    return ERR_INVARG;
	if( !(p = ParseKeyDefValue(p, &shift,1)) ) /* shift */
	    return ERR_INVARG;
	if( !(p = ParseKeyDefValue(p, &meta,1)) ) /* meta */
	    return ERR_INVARG;
	/* ignore the rest */
      #if USE_VT_LINUX /* used only here */
	/* find the scancode */
	for(i=0; i < DIM(scanCodeMap); i++ )
	    if( scanCodeMap[i].scan == scan )
		break;
	if( i == DIM(scanCodeMap) )
	    return ERR_INVKEY;
	if( std != -1 )
	    scanCodeMap[i].std = std;
	if( shift != -1 )
	    scanCodeMap[i].shift = shift;
	if( meta != -1 )
	    scanCodeMap[i].meta = meta;
      #endif
    }
    else if( !stricmp(p,"compose") ) {
	DEADKEY d, d2;

	p = p2;
	sone = stwo = sfakescan = 0;
	fakescan = 0;

	if( !(p = ParseKeyDefValue(p, &one,2))	)
	    return ERR_INVARG;
	if( one <= -10 ) {
	    sone = one==-10 ? K_FLAG_SHIFT :
		   one==-11 ? K_FLAG_CTRL  :
			      K_FLAG_ALT ;
	    if( !(p = ParseKeyDefValue(p, &one,0))  )
		return ERR_INVARG;
	}
	if( !(p = ParseKeyDefValue(p, &two,2)) )
	    return ERR_INVARG;
	if( two <= -10 ) {
	    stwo = two==-10 ? K_FLAG_SHIFT :
		   two==-11 ? K_FLAG_CTRL  :
			      K_FLAG_ALT ;
	    if( !(p = ParseKeyDefValue(p, &two,0))  )
		return ERR_INVARG;
	}
	if( !(p = ParseKeyDefValue(p, &result,1)) || result == -1 )
	    return ERR_INVARG;
	while( isspace(*p) )
	    p++;
	if( *p ) { /* parse faked scancode */
	    if( !(p = ParseKeyDefValue(p, &fakescan,2))  )
		return ERR_INVARG;
	    if( fakescan <= -10 ) {
		sfakescan = fakescan==-10 ? K_FLAG_SHIFT :
			    fakescan==-11 ? K_FLAG_CTRL  :
					   K_FLAG_ALT ;
		if( !(p = ParseKeyDefValue(p, &fakescan,0))  )
		    return ERR_INVARG;
	    }
	}

	for( d2=NULL,d = deadkeys; d; d2 = d, d = d->next )
	    if( d->one == one && d->sone == sone &&
		d->two == two && d->stwo == stwo )
		break;
	if( !result && !fakescan && !sfakescan ) { /* delete the entry */
	    if( d ) {
		if( d2 )
		    d2->next = d->next;
		else
		    deadkeys = d->next;
		free(d);
	    }
	}
	else {
	    if( !d ) { /* add a new entry */
		d = xmalloc( sizeof *d );
		d->next = deadkeys;
		deadkeys = d;
	    }
	    d->one = one;
	    d->sone = sone;
	    d->two = two;
	    d->stwo = stwo;
	    d->value = result & 0xff;
	    d->scan = fakescan & 0xff;
	    d->state = sfakescan & 0xff;
	}
    }
    else if( !stricmp( p, "composekey") ) {
	int smulti;

	p = p2;
	if( !(p = ParseKeyDefValue(p, &smulti, 0 )) )
	    return ERR_INVARG;
	multi_key = smulti;
    }
    else
	return ERR_INVARG;

   return 0;
}


/****************
 * mode 0 := parse a scancode
 *	1 := parse a value
 *	2 := parse a scancode or a string
 * Returns: NULL := Error
 *	 *ret_value >= 0 := scancode or keyvalue
 *	 *ret_value == -1 := ignore
 *	 *ret_value == -10 := string "shift" (only mode 2)
 */

static char *
ParseKeyDefValue( char *p, int *ret_value, int mode )
{
    char *p2, *p3;
    long hl;

    while( isspace(*p) )
	p++;
    if( p2 = strpbrk(p, " \t\n") )
	*p2++ = 0;
    else
	p2 = p + strlen(p);

    if( !*p || (*p == '-' && !p[1]) ) {
	if( !mode || mode == 2 )
	    return NULL;
	*ret_value = -1; /* ignore */
    }
    else if( (*p == '\'' && p[1] && p[2]=='\'' && !p[3]) ) {
	if( !mode || mode == 2 )
	    return NULL;
	*ret_value = ((byte*)p)[1];
    }
    else if( mode == 2 && !stricmp(p, "shift") )
	*ret_value = -10;
    else if( mode == 2 && !stricmp(p, "ctrl") )
	*ret_value = -11;
    else if( mode == 2 && !stricmp(p, "alt") )
	*ret_value = -12;
    else {
	hl = strtol(p, &p3, 0 );
	if( *p3 || hl < 0 || hl > 255 )
	    return NULL; /* invalid arg */
	*ret_value = hl;
    }
    return p2;
}


/****************
 * Returns a string with the key definition
 * start enumerate with seq=0,
 * mode 1: only compose.
 * Returns pointer to  static buffer  or NULL at end.
 */

const char *
KybrdDrvEnumKeyDef(int mode, int seq)
{
    static char buffer[60];
    int i;
    DEADKEY d;

    xassert(mode==1);

    for(d=deadkeys,i=1; d; d = d->next,i++ )
	if( i > seq )
	    break;
    if( !d )
	return NULL;
    sprintf(buffer, "compose %6s%3d %6s%3d  %3d",
			d->sone & K_FLAG_SHIFT ? "shift" :
			d->sone & K_FLAG_CTRL  ? "ctrl" :
			d->sone & K_FLAG_ALT   ? "alt" : "",
			d->one,
			d->stwo & K_FLAG_SHIFT ? "shift" :
			d->stwo & K_FLAG_CTRL  ? "ctrl" :
			d->stwo & K_FLAG_ALT   ? "alt" : "",
			d->two, d->value );
    if( d->state || d->scan )
	sprintf(buffer+strlen(buffer),"  %6s%3d",
			d->state & K_FLAG_SHIFT ? "shift" :
			d->state & K_FLAG_CTRL	? "ctrl" :
			d->state & K_FLAG_ALT	? "alt" : "",
			d->scan );
    return buffer;
}

/**** end of file ***/
