/* [vt_linux.c wk 02.03.95] Linux Keyboard driver
 *	Copyright (c) 1995 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *--------------------------------------------------------------------
 *
 * This is the quick and dirty way to handle the keyboard under
 * Linux. I have no time to hack a good key interface, so I choose this
 * way, knowning it will only work on a console.
 *
 * IMPORTANT: The Main Module should catch all signals, so a cleanup
 * will be done to switch the keyboard back into a usable mode.
 *
 * Here is a list of important signals:
 *	SIGALRM, SIGHUP, SIGINT, SIGQUIT, SIGILL, SIGTRAP,
 *	SIGABRT, SIGIOT, SIGFPE, SIGKILL, SIGUSR1, SIGSEGV,
 *	SIGUSR2, SIGPIPE, SIGTERM, SIGSTKFLT, SIGCHLD,
 *	SIGCONT, SIGSTOP, SIGTSTP, SIGTTIN, SIGTTOU
 *
 * This code may be used to check for a console on stdin:
 *
 *	    struct stat chkbuf;
 *	    int major, minor;
 *
 *	    fstat(0, &chkbuf);
 *	    major = chkbuf.st_rdev >> 8;
 *	    minor = chkbuf.st_rdev & 0xff;
 *
 *	    * console major num is 4, minor 64 is the first serial line *
 *	    if( !(major == 4 && minor < 64) ) {
 *		fputs("Error: not a console\n", stderr);
 *		exit(4);
 *	    }
 *
 * Note: This is not applicable, as we use "dev/console" ("dev/tty0")
 *
 *
 * History:
 */


#include "wtailor.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <ctype.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <termios.h>
#ifndef __linux__
  #error This is a Linux-only-module !
#endif
#include <sys/ioctl.h>
#include <linux/kd.h>
#include <linux/vt.h>
#include <linux/keyboard.h>

#include <wk/lib.h>
#include "wkbddrv.h"
#include "werr.h"

#include "fonts.h"


#define DBG_PRINT   DummyInfo /* Info */

/* signals for Linux's process control of consoles */
#define SIG_RELEASE    SIGTSTP	/* SIGWINCH */
#define SIG_ACQUIRE    SIGCONT	/* SIGUSR1 */

static int fd = -1;	/* this is "/dev/tty0" */
static int console_number;
static int using_vcs;
static int disableFontSwitching;
static struct {
    int kbdmode;
    struct termios tio;
} oldvalue;

static struct {
    int shift,ctrl,alt,meta;
} state;


#define KBD_QUEUE_LEN 20
static struct {
    ushort   scan;
    unsigned kstate;
} kbdQueue[KBD_QUEUE_LEN]; /* array to store the keystrokes */
static int kbdQueueCnt;


static void CleanUp(void *dummy);
static void SwitchFont( int mode );
static void AcquireVT(int sig);
static void ReleaseVT(int sig);

static void
DummyInfo(const char*s,...)
{
}



/****************
 * Open the Linux keyboard
 * Returns: 0 := okay
 */

int
VTLinuxOpen()
{
    struct termios tio;
    char *s;

    s = ttyname(0);
    if( !s ) {
	Error(0,"not connected to a console device");
	return -1;
    }
    if( !(strlen(s) >= 9 && !memcmp(s, "/dev/tty", 8) && isdigit(s[8])) ) {
	Error(0,"'%s' is not a console device", s);
	return -1;
    }
    console_number = atoi(s+8);
    if( console_number < 1 || console_number > 63 ) {
	Error(0,"'%s' is an invalid console device", s);
	return -1;
    }
    if( (fd = open(s, O_RDONLY|O_NONBLOCK)) < 0 ) {
	Error(1000,"can't open %s",s);
	return -1;
    }

    if( ioctl(fd, KDGKBMODE, &oldvalue.kbdmode) ) {
	Error(1000,"can't ioctl(KDGKBMODE)");
	close(fd); fd = -1;
	return -1;
    }
    tcgetattr(fd, &oldvalue.tio );
    tcgetattr(fd, &tio);
    tio.c_lflag     &= ~ (ICANON | ECHO | ISIG );
    tio.c_iflag     = 0;
    tio.c_cc[VMIN]  = 1;    /* one char at a time */
    tio.c_cc[VTIME] = 1;    /* 0.1 sec intercharacter timeout */

    AddCleanUp(CleanUp,NULL);

    tcsetattr(fd, TCSAFLUSH, &tio);
    if( ioctl(fd, KDSKBMODE, K_MEDIUMRAW ) ) {
	Error(1000,"can't ioctl(KDSKBMODE)");
	close(fd); fd = -1;
	return -1;
    }

    memset(&state,0,sizeof state);
    kbdQueueCnt = 0;
    return 0;
}



/****************
 * Try to open the virtual console screen (see VCS(4))
 * Returns: -1 not available
 *	    handle of the vcs
 */

int
VTLinuxOpenVCS( ushort *retX, ushort *retY )
{
    int vcs;
    struct {char lines, cols, x, y;} scrn;
    char buf[15];
    struct vt_mode vt_mode;

    sprintf(buf,"/dev/vcsa%d", console_number);
    if( (vcs = open(buf, O_RDWR)) == -1 )
	return -1;
    read(vcs, &scrn, 4);
    *retX = scrn.cols;
    *retY = scrn.lines;

    /* we must have process control over the vC */
    vt_mode.mode = VT_PROCESS;
    vt_mode.waitv = 0;
    vt_mode.relsig = SIG_RELEASE;
    vt_mode.acqsig = SIG_ACQUIRE;
    vt_mode.frsig = 0;
    signal(SIG_RELEASE, ReleaseVT);
    signal(SIG_ACQUIRE, AcquireVT);
    if( ioctl(fd, VT_SETMODE, &vt_mode) )
	Error(1002,"Init VT mode failed");
    using_vcs = 1;
    /* we need to change the font */
    SwitchFont(1);
    return vcs;
}



/****************
 * Returns: 0 = no vcs
 *	    1 = vcs raw
 *	    2 = vcs with Switchdmode
 */

int
VTLinuxGetCharSet()
{
    return using_vcs? 2 : 0;
}


/****************
 * Set the characeterset for the vcs mode
 * mode 0 := standard
 *	1 := do not switch font
 */

void
VTLinuxSetCharSet( int mode )
{
    disableFontSwitching = mode == 1;
}



/****************
 * switch the font:
 * mode: 0 = restore font
 *	 1 = switch to Latin1 font
 */

static void
SwitchFont( int mode )
{
    static char *savearea = NULL;
    static char *fontarea = NULL;
    static int saved = 0;

    if( !using_vcs )
	return;
    if( disableFontSwitching )
	return;
    if( !savearea )
	savearea = xmalloc( 8192 );
    if( !fontarea ) { /* load the font from the internal table */
	int i,j;
	fontarea = xcalloc( 8192, 1 );
	for(i=j=0; i < 8192; i += 32, j += 16)
	    memcpy(fontarea+i, font_8x16_iso_latin_1+j, 16);
    }
    if( mode ) { /* switch to latin 1 */
	if( !saved ) {
	    /* save the last font */
	    DBG_PRINT("save font");
	    if( ioctl( fd, GIO_FONT, savearea ) )
		Error(1002,"cannot save current font");
	    saved = 1;
	}
	/* do the switch */
	DBG_PRINT("load font");
	if( ioctl( fd, PIO_FONT, fontarea ) )
	    Error(1002,"cannot load font");
    }
    else if(saved) {/* restore saved font */
	DBG_PRINT("restore font");
	if( ioctl( fd, PIO_FONT, savearea ) )
	    Error(1002,"cannot restore font");
	saved = 0;
    }
    else
	DBG_PRINT("no saved font");

}


void
VTLinuxClose()
{
    CleanUp(NULL);
}



/****************
 * Try to get the Scancode from the Keyboard; this function will
 * not block.
 * Switching to another VT is handled here by looking at
 * ALT-CTRL-F[1..12].
 * Bits in <kstate>: Bit 0 := in shift state
 *			 1 := in ctrl state
 *			 2 := in alt state
 *			 3 := in meta state
 *			 4 := scroll lock active
 *			 5 := num lock active
 *			 6 := caps lock active
 *			 7 := reserved
 *		     bit 8 := left shift key is currently pressed
 *		     bit 9 := left ctrl key is currently pressed
 *		     bit 10:= left alt key is currently pressed
 *		     bit 11:= reserved
 *		     bit 12:= right shift key is currently pressed
 *		     bit 13:= right ctrl key is currently pressed
 *		     bit 14:= right alt key is currently pressed
 *		     bit 15:= reserved
 *
 * Return: 0 := no keypress
 *	   1 := scancode available
 */

int
VTLinuxReadScan( ushort *ret_scan, unsigned *ret_kstate )
{
    int i;

    VTLinuxPoll();
    if( kbdQueueCnt ) {
	*ret_scan = kbdQueue[0].scan;
	*ret_kstate = kbdQueue[0].kstate;
	for(i=1; i < kbdQueueCnt; i++ )
	    kbdQueue[i-1] = kbdQueue[i];
	kbdQueueCnt--;
	return 1;
    }
    return 0;
}


static int
ReadScan( ushort *ret_scan, unsigned *ret_kstate )
{
    byte scan;
    int rc, skip, key, pressed;

    rc = key = 0;
    if( fd != -1 && read(fd, &scan, 1) > 0 ) {
	key = scan & 0x7f;
	pressed = !(scan & 0x80);
	skip = 0;
	switch( key ) {
	  case 42: /* left shift */
	  case 54: /* right shift */
	    skip++;
	    if( !(state.shift && pressed) )
		state.shift = pressed;
	    break;

	  case 29: /* left ctrl */
	  case 97: /* right ctrl */
	    skip++;
	    if( !(state.ctrl && pressed) )
	       state.ctrl = pressed;
	    break;

	  case 56: /* left alt */
	    skip++;
	    if( !(state.alt && pressed) )
	       state.alt = pressed;
	    break;
	  case 100: /* right alt */
	    skip++;
	    if( !(state.meta && pressed) )
	       state.meta = pressed;
	    break;

	  default: break;
	}
	if( !skip && pressed ) {
	    rc = 1; /* valid key */
	    if( state.alt && state.ctrl && !state.shift ) {
		/* special actions on Alt-Ctrl and a pressed key */
		if( key >= 59 && key <= 68 ) {
		    ioctl(fd,VT_ACTIVATE,key-58);
		    rc = 0; /* caller should not see this code */
		}
		else if( key >= 87 && key <= 88 ) {
		    VTLinuxSwitchConsole(key-86+10);
		    rc = 0; /* caller should not see this code */
		}
		else if( key == 111 || key ==83 ) {
		    /* alt-ctrl-del */
		    raise(SIGQUIT); /* at least we should die */
		    rc = 0; /* caller should not see this code */
		}
	    }
	    else if( !state.alt && state.ctrl && !state.shift && key == 46 ) {
		/* ctrl-C */
		raise(SIGINT);
		rc = 0;
	    }
	}
    }

    if( ret_scan )
	*ret_scan = key;
    if( ret_kstate )
	*ret_kstate = VTLinuxGetKeyState();

    return rc;
}


/****************
 * Return the last keystate.
 */

unsigned
VTLinuxGetKeyState()
{
    unsigned kstate = 0;

    if( state.shift )
	kstate |= 1<<0;
    if( state.ctrl )
	kstate |= 1<<1;
    if( state.alt   )
	kstate |= 1<<2;
    if( state.meta )
	kstate |= 1<<3;
    return kstate;
}


/*******************
 * Returns: # no of characters in queue
 */

int
VTLinuxPoll()
{
    ushort scan;
    unsigned kstate;

    while( ReadScan( &scan, &kstate ) ) {
	if( kbdQueueCnt < KBD_QUEUE_LEN-1 ) {
	     kbdQueue[kbdQueueCnt].scan = scan;
	     kbdQueue[kbdQueueCnt].kstate = kstate;
	     kbdQueueCnt++;
	}
	else { /* buffer overflow */
	    if( fd != -1 )
		write(fd,"\a",1);
	    break;
	}
   }
   return kbdQueueCnt;
}



/****************
 * Try to wait for an event
 * Returns: -1 := error
 *	     0 := input available
 *	     1 := interrupt or timeout
 */
int
VTLinuxWait()
{
    fd_set fds;
    int rc;
    struct timeval tv;

    if( kbdQueueCnt )
	return 0; /* do not wait in this case */
    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    tv.tv_sec = 1;
    tv.tv_usec = 0;
    if( (rc = select(FD_SETSIZE, &fds, NULL, NULL, &tv)) == -1 )
	return errno == EINTR ? 1 : -1;
    return rc? 0 : 1;
}

int
VTLinuxGetScreenSize( int fd, ushort *x, ushort *y )
{
    struct winsize ws;

    if( ioctl(fd, TIOCGWINSZ, &ws ) >= 0 ) {
	*x = ws.ws_col;
	*y = ws.ws_row;
	return 0;
    }
    return -1;
}

int
VTLinuxSwitchConsole( int no )
{
    if( no == console_number )
	return 0;
    if( no < 1 || no > 12 )
	return ERR_INVARG;
    memset(&state,0,sizeof state);
    kbdQueueCnt = 0;
    SwitchFont(0);
    DBG_PRINT("switch to %d", no);
    if( ioctl(fd,VT_ACTIVATE,no) )
	return ERR_UNKNOWN;
 #if 0	/* handled in AcquireVT() */
    DBG_PRINT("waiting for %d", console_number);
    while( ioctl(fd,VT_WAITACTIVE,console_number) )
	DBG_PRINT("interrupted");; /* was interrupted */
    DBG_PRINT("active again");
    SwitchFont(1);
 #endif
    return 0;
}


static void
AcquireVT(int sig)
{
    signal(SIG_ACQUIRE, AcquireVT);
    DBG_PRINT("acquire VT");
    if( ioctl(fd, VT_RELDISP, VT_ACKACQ) )     /* switch acknowledged */
	 Error(1000,"VT release (ack) failed");
    SwitchFont(1);
    DBG_PRINT("acquired");
}

static void
ReleaseVT(int sig)
{
    signal(SIG_RELEASE, ReleaseVT );

    DBG_PRINT("release VT");
    SwitchFont(0);
    if( ioctl(fd, VT_RELDISP, 1) )  /* switch ok by me */
	Error(1000,"VT release (OK) failed");
    DBG_PRINT("released");
}


static void
CleanUp(void *dummy)
{
    if( fd == -1 )
	return;
    DBG_PRINT("vt_linux: cleanup");
    if( using_vcs ) {
	struct vt_mode vt_mode;

	vt_mode.mode = VT_AUTO;
	DBG_PRINT("reset VT mode");
	if( ioctl(fd, VT_SETMODE, &vt_mode) )
	    Error(1000,"Reset VT mode failed");
	signal( SIG_RELEASE, SIG_IGN);
	signal( SIG_ACQUIRE, SIG_IGN);
    }
    DBG_PRINT("reset font");
    SwitchFont(0);
    ioctl(fd, KDSKBMODE, oldvalue.kbdmode);
    tcsetattr(fd, 0, &oldvalue.tio );
    close(fd);
    fd = -1;
    using_vcs = 0;
}


/*** eof ***/
