/* [hilite.c wk 06.10.95] W-Editor Syntax higlighting
 *	Copyright (c) 1995 by Werner Koch (dd9jn)
 * This file is part of the W-Editor.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This modules works close togteher with wscreen.
 * Currently only C syntax is supported.
 */

#include "wtailor.h"
RCSID("$Id: hilite.c,v 1.10 1999/10/16 10:42:07 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <wk/lib.h>
#include <wk/string.h>

#include "w.h"
#include "wfile.h"


#define USE_CACHE 1

static struct {
    int   used;
    ulong lnr;
    int   state;
    size_t bufsize; /* allocated length */
    byte  *buffer;  /* malloced */
    size_t buflen;  /* current length */
} cache[30];
static int cache_fhd = -1; /* cache is invalid */
static int hilite_type = 0;

static byte std_color, colors[9];

static void State2Color( byte *buffer, size_t len );


/****************
 * Das syntax-highlighting zurcksetzen. Dies wird durch ein zurcksetzen
 * aller Zeilen erreicht.
 */
void
HiliteReset( int fhd )
{
    ulong lnr=0;

    cache_fhd = -1;
    while( !SetHiliteState( fhd, lnr, 0 ) )
	lnr++;
}




/****************
 * Used by screen module to tell us the colors.
 * (index starts with 0)
 * color is only a byte
 * idx -1 is used for the standard attribute
 */

void
HiliteSetColor( int idx, int color )
{
    if( idx >=0 && idx < DIM(colors) )
	colors[idx] = color;
    else if( idx == -1 )
	std_color = color;
    cache_fhd = -1; /* reset cache */
}



/****************
 * map state to attributes
 * My colormap is:
 * No. Color			Usage
 * 1   Blue on Bright-White	comments
 * 2   Magenta on Bright-White	strings, SGML tags
 * 3   Brown on White		SGML MDs
 * 4   Blue on White		MD Strings
 * 5   Green on Bright-White	attribute values
 * 6   red   on bright-White	Entities
 * 7   Light-Green on White
 * 8   Light-Cyan on White
 * 9   Light-Red on White
 */
static void
State2Color( byte *buffer, size_t len )
{
    size_t n;
    byte *p;

    if( hilite_type == 3 )  { /* sgml */
	for(n=0,p=buffer; n < len ; n++, p++ ) {
	    switch(*p) {
	      case 2:  *p = colors[0]; break; /* comments */
	      case 3: /* tag */
	      case 9:
		       *p = colors[1]; break;
	      case 7: /* attrib value */
	      case 8:
	      case 10:
	      case 11:
		       *p = colors[4]; break;
	      case 4: /* MD */
		       *p = colors[2]; break;
	      case 5: /* String in MD */
	      case 6: /* String in MD */
		       *p = colors[3]; break;
	      case 12: *p = colors[5]; break; /* entity */
	      default: *p = std_color; break;
	    }
	}
    }
    else if( hilite_type == 4 )  { /* scm */
	for(n=0,p=buffer; n < len ; n++, p++ ) {
	    switch(*p) {
	      case 2: *p = colors[0]; break; /* comments */
	      case 3: *p = colors[1]; break; /* string */
	      case 4: *p = colors[5]; break; /* hashed constant */
	      default: *p = std_color; break;
	    }
	}
    }
    else if( hilite_type == 7 ) { /* perl */
	for(n=0,p=buffer; n < len ; n++, p++ ) {
	    switch(*p) {
	      case 2:  *p = colors[0]; break;  /* C: comments */
	      case 3:  *p = colors[1]; break;  /* C: strings */
	      case 4:  *p = colors[2]; break;
	      case 5:  *p = colors[0]; break;  /* inline dox */
	      case 6:  *p = colors[4]; break;  /* C: keywords */
	      case 7:  *p = colors[5]; break;  /* C: cpp words */
	      case 8:  *p = colors[6]; break;
	      case 9:  *p = colors[7]; break;
	      case 10: *p = colors[8]; break;
	      default: *p = std_color; break;
	    }
	}
    }
    else {
	for(n=0,p=buffer; n < len ; n++, p++ ) {
	    switch(*p) {
	      case 2:  *p = colors[0]; break;  /* C: comments */
	      case 3:  *p = colors[1]; break;  /* C: strings */
	      case 4:  *p = colors[2]; break;
	      case 5:  *p = colors[3]; break;
	      case 6:  *p = colors[4]; break;  /* C: keywords */
	      case 7:  *p = colors[5]; break;  /* C: cpp words */
	      case 8:  *p = colors[6]; break;
	      case 9:  *p = colors[7]; break;
	      case 10: *p = colors[8]; break;
	      default: *p = std_color; break;
	    }
	}
    }
}


/****************
 * Find C or ML keywords
 * Mode: 0 = C
 *	 6 = ML
 *	 7 = perl
 */
static void
find_c_keywords( int mode,
		 const char *line, ushort len,	byte *buffer, ushort buflen )
{
    struct keywords {
	const char *name;
	int len;
    } ;
    static struct keywords c_words[] = {
	{  "asm",       3 },
	{  "auto",      4 },
	{  "break",     5 },
	{  "case",      4 },
	{  "char",      4 },
	{  "const",     5 },
	{  "continue",  8 },
	{  "default",   7 },
	{  "do",        2 },
	{  "else",      4 },
	{  "enum",      4 },
	{  "extern",    6 },
	{  "for",       3 },
	{  "goto",      4 },
	{  "if",        2 },
	{  "inline",    6 },
	{  "int",       3 },
	{  "long",      4 },
	{  "register",  8 },
	{  "return",    6 },
	{  "short",     5 },
	{  "sizeof",    6 },
	{  "static",    6 },
	{  "struct",    6 },
	{  "switch",    6 },
	{  "typedef",   7 },
	{  "typeof",    6 },   /* gcc */
	{  "union",     5 },
	{  "unsigned",  8 },
	{  "void",      4 },
	{  "volatile",  8 },
	{  "while",     5 },
	{  "__asm__",   7 },
	{  "__attribute__",  13 },
	{  "__inline__",  10 },
	{  "__typeof__",  10 },
	{  "__extension__",  13 },
	{ NULL, 0 }
    };
    static struct keywords ml_words[] = {
	{ "and",        3     },
	{ "as",         2     },
	{ "asr",        3     },
	{ "assert",     6     },
	{ "begin",      5     },
	{ "class",      5     },
	{ "constraint", 10    },
	{ "do",         2     },
	{ "done",       4     },
	{ "downto",     6     },
	{ "else",       4     },
	{ "end",        3     },
	{ "exception",  9     },
	{ "external",   8     },
	{ "false",      5     },
	{ "for",        3     },
	{ "fun",        3     },
	{ "function",   8     },
	{ "functor",    7     },
	{ "if",         2     },
	{ "in",         2     },
	{ "include",    7     },
	{ "inherit",    7     },
	{ "initializer",11    },
	{ "land",       4     },
	{ "lazy",       4     },
	{ "let",        3     },
	{ "lor",        3     },
	{ "lsl",        3     },
	{ "lsr",        3     },
	{ "lxor",       4     },
	{ "match",      5     },
	{ "method",     6     },
	{ "mod",        3     },
	{ "module",     6     },
	{ "mutable",    7     },
	{ "new",        3     },
	{ "object",     6     },
	{ "of",         2     },
	{ "open",       4     },
	{ "or",         2     },
	{ "parser",     6     },
	{ "private",    7     },
	{ "rec",        3     },
	{ "sig",        3     },
	{ "struct",     6     },
	{ "then",       4     },
	{ "to",         2     },
	{ "true",       4     },
	{ "try",        3     },
	{ "type",       4     },
	{ "val",        3     },
	{ "virtual",    7     },
	{ "when",       4     },
	{ "while",      5     },
	{ "with",       4     },
	{ NULL, 0 }
    };
    static struct keywords perl_words[] = {
	{ "cmp"       , 3  },
	{ "continue"  , 8  },
	{ "do"        , 2  },
	{ "else"      , 4  },
	{ "elsif"     , 5  },
	{ "eq"        , 2  },
	{ "for"       , 3  },
	{ "foreach"   , 7  },
	{ "gt"        , 2  },
	{ "if"        , 2  },
	{ "last"      , 4  },
	{ "le"        , 2  },
	{ "local"     , 5  },
	{ "lt"        , 2  },
	{ "my"        , 2  },
	{ "ne"        , 2  },
	{ "next"      , 4  },
	{ "redo"      , 4  },
	{ "sub"       , 3  },
	{ "unless"    , 6  },
	{ "while"     , 5  },
	{ NULL, 0 }
    };
    int i, j, k, okay=0;
    static struct keywords *words;
    int max_wordlen;

    if( mode == 7 ) {
	words = perl_words;
	max_wordlen = 8;
    }
    else if( mode == 6 ) {
	words = ml_words;
	max_wordlen = 11;
    }
    else {
	words = c_words;
	max_wordlen = 13;
    }

    if( len < buflen )
	buflen = len;  /* just in case ... */

    if( !mode ) { /*first look for cpp keywords */
	static struct {
	    const char *name;
	    int len;
	    int special;
	} cppwords[] = {
	    {  "define",    6, 0 },
	    {  "if",        2, 0 },
	    {  "ifdef",     5, 0 },
	    {  "ifndef",    6, 0 },
	    {  "elif" ,     4, 0 },
	    {  "else" ,     4, 0 },
	    {  "endif",     5, 0 },
	    {  "error",     5, 0 },
	    {  "include",   7, 0 },
	    {  "line",      4, 0 },
	    {  "pragma",    6, 0 },
	    {  "warning",   7, 0 }, /* gcc */
	    {  "defined",   7, 1 }, /* this does only work on a line with a #*/
	    { NULL, 0, 0 }
	};

	for(i=0; i < buflen; i++ ) {
	    if( buffer[i] != 1 || isspace(line[i]) )
		continue;
	    if( line[i] == '#' ) {
		okay++;
		continue;
	    }
	    if( !okay )
		break;
	    if( isalpha(line[i]) || line[i]=='_' ) {
		/* may be the start of an identifier, find end */
		for(j=i+1; j < buflen && buffer[j] == 1
			   && (isalpha(line[j]) || line[j]=='_'); )
		    j++;
		j -= i;
		if( j >= 2 && j <= 7 ) { /* maybe a keyword */
		    for(k=0; cppwords[k].len; k++ )
			if( cppwords[k].len == j
			    && !memcmp(cppwords[k].name, line+i,j) )
			    break;
		    if( cppwords[k].len ) {  /* really a keyword: set state */
			if( okay == 1 || cppwords[k].special )
			    memset( buffer+i, 7, j );
			okay++;
		    }
		}
		i += j-1; /*move on (and look for defined) */
	    }
	}
    }

    /* and then regular keywords */
    for(i=0; i < buflen; i++ ) {
	if( buffer[i] != 1 )
	    continue;
	if( isalpha(line[i]) || line[i]=='_' ) {
	    /* may be the start of an identifier, find end */
	    for(j=i+1; j < buflen && buffer[j] == 1
		       && (isalpha(line[j]) || line[j]=='_' || line[j]=='$'); )
		j++;
	    j -= i;
	    if( j >= 2 && j <= max_wordlen ) { /* maybe a keyword */
		for(k=0; words[k].len; k++ )
		    if( words[k].len == j && !memcmp(words[k].name, line+i,j) )
			break;
		if( words[k].len )  /* really a keyword: set state */
		    memset( buffer+i, 6, j );
	    }
	    i += j-1; /*move on*/
	}
    }
}

/****************
 * Die angegeben Zeile eines Buffers neu berechnen.
 * Hat diese Zeile keinen Status, so wird zurckgesucht bis eine
 * Zeile gefunden wird, die bereits einen status hat, oder aber bis die erste
 * Zeile gefunden ist. Alle untersuchten Zeilen werden auf den neuen State
 * gesetzt. Es wird nie ber die angegebene Zeile hinaus vorwrts berechnet,
 * aber der Anfangsstate der nchsten Zeile gesetzt.
 * Note: State 0 and 1 are never highlighted.
 *	 (0 := not initialized, 1 := start state)
 * Returns: NULL or a pointer to an attribute Buffer of at least len <buflen>
 *	    (this may be buffer but is usualy a cache entry)
 */

const byte *
HiliteLine( int fhd, ulong the_lnr, ushort bufstart,
	    byte *buffer, ushort buflen )
{
    ulong lnr = the_lnr;
    const char *p, *line;
    ushort n, len;
    int i, eof, state, start_state, next_state, shift, fwd=0;
    unsigned flags;
    ushort idx;
    const byte *ret_buffer;

    /*Info("HiliteLine(%lu) buflen=%u", the_lnr, buflen);*/
  get_line:
    eof = GetHiliteLine( fhd, lnr, &p, &len, &state, &flags);
    if( !p ) {
	/*Info("lnr=%lu, ready or error", lnr);*/
	return NULL; /* ready or error */
    }
    if( !state ) { /* we must go back */
	if( fwd )
	    /*Info("lnr=%lu, no state, going forward", lnr)*/;
	else if( !lnr ) {
	    state = 1;
	    /*Info("set first line to state 1, len=%u", len)*/;
	    SetHiliteState( fhd, 0, state );
	}
	else {
	    /*Info("lnr=%lu, no state, going back, len=%u", lnr, len)*/;
	    lnr--;
	    goto get_line;
	}
    }
    /*Info("lnr=%lu, len=%u, state=%d", lnr, len, state)*/;

    ret_buffer = buffer;
  #ifdef USE_CACHE
    if( fhd == cache_fhd ) {
	if( lnr < the_lnr )
	    /*Info("lnr=%lu, skipping cache, state=%d", lnr,state)*/;
	else {
	    for(i=0; i < DIM(cache); i++ )
		if( cache[i].used && cache[i].lnr == lnr ) { /* hit */
		    if( cache[i].state != state || buflen != cache[i].buflen
							    || (flags & 1) ) {
			cache[i].used = 0;
			/*Info("cache entry  %lu (%lu) cleared%s",
			      the_lnr, lnr, (flags&1)? " (not cacheable)":"");*/
			break; /* is invalid */
		    }
		    /*Info("cache hit at %lu (%lu) state=%d",
						the_lnr, lnr,state);*/
		    ret_buffer = cache[i].buffer;
		    goto next_line;
		}
	    /*Info("cache miss at %lu (%lu)", the_lnr, lnr);*/
	}
    }
    else {  /* prepare the cache */
	const char *s = GetFilesHiliteType(fhd);
	if( !s || !*s || !stricmp(s,"c") )
	    hilite_type = 0; /* default = C */
	else if( !stricmp(s,"cls") )
	    hilite_type = 1; /* Enfin CLS */
	else if( !stricmp(s,"srm") )
	    hilite_type = 2; /* Sermo */
	else if( !stricmp(s,"sgml") )
	    hilite_type = 3; /* SGML */
	else if( !stricmp(s,"scm") )
	    hilite_type = 4; /* scheme */
	else if( !stricmp(s,"tex") )
	    hilite_type = 5; /* TeX */
	else if( !stricmp(s,"ml") )
	    hilite_type = 6; /* Ocaml */
	else if( !stricmp(s,"perl") )
	    hilite_type = 7; /* Perl */
	else if( !stricmp(s,"sh") )
	    hilite_type = 8; /* sh */
	else
	    hilite_type = 0; /* unknown = default */
	/*Info("Reset cache old handle=%d new handle=%d (type=%s,%d)"
				      , cache_fhd, fhd, s, hilite_type);*/
	cache_fhd = fhd;
	for(i=0; i < DIM(cache); i++ ) {
	    cache[i].used = 0;
	    if( !cache[i].buffer ) {
		/* fixme: (we assume a fixed buffer size) */
		cache[i].bufsize = 80;
		cache[i].buffer = xmalloc( cache[i].bufsize );
	    }
	}
    }
  #endif
    /* calculate this line */
    idx = 0;
    shift = 0;
    start_state = state;
    next_state = 0;
    line = p;
    for(n=0; n < len; n++, p++ ) {
	if( shift )
	    shift--;
	else {
	    if( next_state ) {
		state = next_state;
		next_state = 0;
	    }
	    if( !hilite_type ) { /* default == C */
		if( state == 2 ) { /* in comment */
		    if( n+1 < len && *p == '*' && p[1] == '/' ) {
			next_state = 1;
			shift = 1;
		    }
		}
		else if( state == 3 ) { /* in string */
		    if( n+1 < len && *p == '\\' && p[1] == '\\' )
			shift = 1;
		    else if( n+1 < len && *p == '\\' && p[1] == '\"' )
			shift = 1;
		    else if( *p == '\"' )
			next_state = 1;
		}
		else if( n+1 < len && *p == '/' && p[1] == '*' ) {
		    state = 2; /* comment */
		    shift = 1;
		}
		else if( *p == '\"' && !(n && (p[-1]=='\\' || p[-1]=='\''))) {
		    state = 3; /* string */
		}
	    }
	    else if( hilite_type == 1) { /* ENFIN smalltalk */
		if( state == 2 ) { /* in comment */
		    if( *p == '\"' ) {
			next_state = 1;
		    }
		}
		else if( state == 3 ) { /* in string */
		    if( n+1 < len && *p == '\'' && p[1] == '\'' )
			shift = 1;
		    else if( *p == '\'' )
			next_state = 1;
		}
		else if( *p == '$' ) {  /* skip character constant */
		    shift = 1;
		}
		else if( *p == '\"' ) {
		    state = 2; /* comment */
		}
		else if( *p == '\'' ) {
		    state = 3; /* string */
		}
	    }
	    else if( hilite_type == 2) { /* Sermo */
		if( state == 2 ) { /* in comment */
		    if( n+1 < len && *p == '*' && p[1] == '/' ) {
			next_state = 1;
			shift = 1;
		    }
		}
		else if( state == 3 ) { /* in string */
		    if( n+1 < len && *p == '\'' && p[1] == '\'' )
			shift = 1;
		    else if( *p == '\'' )
			next_state = 1;
		}
		else if( n+1 < len && *p == '/' && p[1] == '*' ) {
		    state = 2; /* comment */
		    shift = 1;
		}
		else if( *p == '\'' ) {
		    state = 3; /* string */
		}
	    }
	    else if( hilite_type == 3) { /* SGML reference concrete syntax */
		/* States:
		 *  2 - in comment
		 *  3 - in tag
		 *  4 - in MD
		 *  5 - in MD string single quote
		 *  6 - in MD string double quote
		 *  7 - Attribute value with single quote
		 *  8 - Attribute value with double quote
		 *  9 - in tag after one slash
		 * 10 - Attribute value with single quote after state 9
		 * 11 - Attribute value with double quote after state 9
		 * 12 - entity in plain text
		 */
		switch( state ) {
		  case 2: /* in comment */
		    if( n+1 < len && *p == '-' && p[1] == '-' ) {
			if( n+2 < len && p[2] == '>' ) {
			    next_state = 1; /* back */
			    shift = 2;
			}
			else {
			    next_state = 4; /* back to MD */
			    shift = 1;
			}
		    }
		    break;
		  case 3: /* in tag */
		    if( *p == '>' )
			next_state = 1;
		    else if( n && *p == '/' && isalnum(p[-1]) )
			state = 1;
		    else if( *p == '\"' )
			state = 7;
		    else if( *p == '\'' )
			state = 8;
		    else if( *p == '/' )
			state = 9; /* slash seen */
		    break;
		  case 9: /* in tag, after a slash */
		    if( *p == '>' )
			next_state = 1;
		    else if( *p == '\"' )
			state = 10;
		    else if( *p == '\'' )
			state = 11;
		    break;
		  case 4:
		    if( n+1 < len && *p == '-' && p[1] == '-' ) {
			state = 2; /* comment */
			shift = 1;
		    }
		    else if( *p == '\"' )
			state = 5;
		    else if( *p == '\'' )
			state = 6;
		    else if( *p == '>' )
			next_state = 1;
		    break;
		  case 5:
		    if( *p == '\"' )
			next_state = 4;
		    break;
		  case 6:
		    if( *p == '\'' )
			next_state = 4;
		    break;
		  case 7:
		    if( *p == '\"' )
			next_state = 3;
		    break;
		  case 8:
		    if( *p == '\'' )
			next_state = 3;
		    break;
		  case 10:
		    if( *p == '\"' )
			next_state = 9;
		    break;
		  case 11:
		    if( *p == '\'' )
			next_state = 9;
		    break;
		  case 12: /* entity */
		    if( *p == ';' || !isalnum( *p ) )
			next_state = 1;
		    break;
		  default:
		    if( *p == '<' ) {
			if( n+1 < len && p[1] == '!' ) { /* markup declaration*/
			    if( n+3 < len && p[2] == '-' && p[3] == '-' ) {
				state = 2; /* direct to comment */
				shift = 3;
			    }
			    else {
				state = 4;
				shift = 1;
			    }
			}
			else
			    state = 3; /* tag */
		    }
		    else if( *p == '&' )
			state = 12; /* entity */
		    break;
		} /* end switch */
	    } /* end hilite type 3 */
	    else if( hilite_type == 4 ) { /* scheme */
		if( state == 2 ) { /* in comment */
		    if( n+1 >= len )
			next_state = 1;
		}
		else if( state == 3 ) { /* in string */
		    if( n+1 < len && *p == '\\' && p[1] == '\\' )
			shift = 1;
		    else if( n+1 < len && *p == '\\' && p[1] == '\"' )
			shift = 1;
		    else if( *p == '\"' )
			next_state = 1;
		}
		else if( state == 4 ) { /* in character constant etc. */
		    if( *p == ' '  || *p == '(' || *p == ')' )
			state = 1;
		    else if( n+1 >= len )
			next_state = 1;
		}
		else if( *p == ';' ) {
		    state = 2; /* comment */
		}
		else if( *p == '\"' && !(n && p[-1] == '\\')) {
		    state = 3; /* string */
		}
		else if( *p == '#' ) {
		    state = 4; /* character constant etc. */
		}
	    } /* end hilite type 4 */
	    else if( hilite_type == 5 ) { /* TeX */
		/* States for TeX:
		 *  2 - in comment
		 *  3 - keyword
		 *  6 - {...}
		 *  7 - special character
		 */
		if( state == 2 ) { /* in comment */
		    if( n+1 >= len )
			next_state = 1;
		}
		else if( state == 3 ) { /* in keyword */
		    if( n+1 >= len )   /* terminated by LF */
			next_state = 1;
		    else if( isspace(*p) || *p == '[') /*]*/
			state = 1;
		    else if( *p == '{' ) { /*}*/
			state = 7;
			next_state = 6;
		    }
		}
		else if( *p == '%' ) {
		    state = 2; /* comment */
		}
		else if( *p == '\\' ) {
		    if( n+1 < len && isalpha(p[1]) )
			state = 3; /* keyword */
		    else {
			shift = 1; /* skip special character */
		    }
		}
		else if( *p == '{' /*}*/ ) {
		    state = 7;
		    next_state = 6;
		}
		else if( *p == '}' /*{*/ ) {
		    state = 7;
		    next_state = 1;
		}
		else if( strchr( "#$%&~_^\\", *p ) ) {
		    state = 7; /* special character */
		    next_state = 1;
		}

	    } /* end hilite type 5 */
	    else if( hilite_type == 6) { /* Ocaml */
		if( state == 2 ) { /* in comment */
		    if( n+1 < len && *p == '*' && p[1] == ')' ) {
			next_state = 1;
			shift = 1;
		    }
		}
		else if( state == 3 ) { /* in string */
		    if( n+1 < len && *p == '\\' && p[1] == '\\' )
			shift = 1;
		    else if( n+1 < len && *p == '\\' && p[1] == '\"' )
			shift = 1;
		    else if( *p == '\"' )
			next_state = 1;
		}
		else if( n+1 < len && *p == '(' && p[1] == '*' ) {
		    state = 2; /* comment */
		    shift = 1;
		}
		else if( *p == '\"' && !(n && (p[-1]=='\\' || p[-1]=='\''))) {
		    state = 3; /* string */
		}
	    }
	    else if( hilite_type == 7 || hilite_type == 8 ) { /* perl/sh */
		if( state == 2 ) { /* in comment */
		    if( n+1 >= len )
			next_state = 1;
		}
		else if( state == 3 ) { /* in simple string */
		    if( n+1 < len && *p == '\\' && p[1] == '\'' )
			shift = 1;
		    else if( *p == '\'' )
			next_state = 1;
		}
		else if( state == 4 ) { /* in string 2 */
		    if( n+1 < len && *p == '\\' && p[1] == '\"' )
			shift = 1;
		    else if( *p == '\"' )
			next_state = 1;
		}
		else if( state == 5 ) { /* Perl inline dox */
		    if( !n && len >= 4 && !memcmp( p, "=cut", 4 ) ) {
			next_state = 1;
			shift = 3;
		    }
		}
		else if( n < len && *p == '#' && (!n || p[-1] != '$') ) {
		    state = 2; /* comment */
		    shift = 1;
		}
		else if( *p == '\'' ) {
		    state = 3; /* string */
		}
		else if( hilite_type == 7 &&*p == '\"' ) {
		    /* real quoting checking is too complicated */
		    state = 4; /* string */
		}
		else if( hilite_type == 7 && !n && *p == '='
			 && len > 1 && isalpha(p[1]) ) {
		    state = 5; /* inline dox */
		}
	    }
	}

	if( lnr == the_lnr && n >= bufstart && idx < buflen )
	    buffer[idx++] = state;
    }
    if( next_state )
	state = next_state;
    /* special handling for ER and TeX comments at line end */
    if( state == 12 || (hilite_type == 5 && state == 2) )
	state = 1;

    if( !hilite_type || hilite_type == 6 || hilite_type == 7 )
	find_c_keywords( hilite_type, line, len, buffer, buflen );
    State2Color(buffer, buflen);

  #ifdef USE_CACHE
    if( !(flags & 1) && lnr == the_lnr ) { /* cache this line */
	int again;
	for(again=2; again > 0; again--) { /* max. two times */
	    for(i=0; i < DIM(cache); i++ )
		if( !cache[i].used && buflen <= cache[i].bufsize ) {
		    cache[i].used = 1;
		    cache[i].lnr   = lnr;
		    cache[i].state = start_state;
		    cache[i].buflen = buflen;
		    memcpy(cache[i].buffer, buffer, buflen );
		    /*Info("caching line %lu (%lu)", the_lnr, lnr);*/
		    break;
		}
	    if( i >= DIM(cache) ) { /* the easy way: round robin releasing */
		static int counter = 0;
		if( counter >= DIM(cache) )
		    counter = 0;
		/*Info("clearing cache entry %d", counter);*/
		cache[counter++].used = 0;
	    }
	    else
		again--;
	}
    }
  #endif

    /* store the state of the next line */
    /*Info("lnr=%lu, endofline state=%d", lnr, state);*/
    SetHiliteState( fhd, lnr+1, state );

  next_line:
    if( lnr < the_lnr ) {
	fwd = 1;
	lnr++ ;
	if( !eof )  /* eof? oops */
	    goto get_line;
    }

    return ret_buffer;
}



/*** bottom of file ***/
