/* packet.c  - OpenPGP packet parsing
 *	Copyright (C) 1999 Werner Koch (dd9jn).
 *
 * This file is part of SFSV.
 *
 * SFSV is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * SFSV is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include "types.h"
#include "packet.h"
#include "sfsv.h"
#include "sha1.h"
#include "dsa.h"


static size_t
read_32( const byte *p )
{
    return p[0]<<24 | p[1] << 16 | p[2] << 8 | p[3];
}



/****************
 * This is the ELF signature verification function
 * Returns: Status of that ELF signature.
 */
int
sfsv_verify_elf_signature( const unsigned char *image, size_t imagelen )
{
    int n, rc;
    const byte *data;
    size_t off, datalen, pktlen;
    int lenbytes;
    SIG_packet sig;
    SHA1_CONTEXT sha1_ctx;

    rc = sfsv_locate_elf_signature( image, imagelen, &off, &datalen );
    if( rc )
	return rc;
    /* start digest calculation */
    sha1_init( &sha1_ctx );
    sha1_write( &sha1_ctx, image, off );
    sha1_write( &sha1_ctx, image+off+datalen, imagelen-off-datalen );

    /* parse the sig packet */
    data = image + off;
    /* we assume that this is a rfc2440 v3 signature */
    if( !datalen || !(*data & 0x80) )
	return SFSV_NO_SIG;

    if( (*data & 0x40) )
	return SFSV_BAD_PGP;  /* new CTB mode is not supported */
    if( ((*data>>2)&0xf) != 2 )
	return SFSV_BAD_PGP;  /* not a signature packet */
    lenbytes = ((*data&3)==3)? 0 : (1<<(*data & 3));
    data++; datalen--;
    if( !lenbytes || datalen < lenbytes )
	return SFSV_BAD_PGP;  /* unsupported length header or packet too short*/
    for( pktlen=0 ; lenbytes; lenbytes-- ) {
	pktlen <<= 8;
	pktlen |= *data++; datalen--;
    }
    if( pktlen > datalen || pktlen < 16 )
	return SFSV_BAD_PGP;  /* packet is too large or too short */

    if( *data != 3 )
	return SFSV_BAD_PGP;  /* only v3 signatures are supported */
    data++; datalen--;
    data++; datalen--; /* skip m5 len */

    sha1_write( &sha1_ctx, data, 1 );  /* hash the sig class */
    if( *data )
	return SFSV_BAD_PGP;  /* not a sig class 0 */
    data++; datalen--;

    sha1_write( &sha1_ctx, data, 4 );  /* hash the timestamp */
    sig.timestamp = read_32(data); data+=4; datalen-=4;

    sig.keyid[0] = read_32(data); data+=4; datalen-=4;
    sig.keyid[1] = read_32(data); data+=4; datalen-=4;

    if( *data != 17 )
	return SFSV_BAD_PGP;  /* not a DSA sig */
    data++; datalen--;
    if( *data != 2 )
	return SFSV_BAD_PGP;  /* not a SHA1 hash */
    data++; datalen--;

    if( pktlen < 6 )
	return SFSV_BAD_PGP;  /* packet is too short */

    data++; datalen--;	/* no need for the digest start bytes */
    data++; datalen--;

    n = bi_read( &sig.r, data, datalen );
    if( n <= 0 )
	return -1; /* bad MPI */
    data += n; datalen -= n;

    n = bi_read( &sig.s, data, datalen );
    if( n <= 0 )
	return -1; /* bad MPI */

    /* sig has now all the required data */
    sha1_final( &sha1_ctx );

    return verify_signature( sha1_read( &sha1_ctx ), &sig );
}

