/* elfsigtool.c - Utility to apply a rfc2440 signature to an ELF file
 *	Copyright (C) 1999 Werner Koch (dd9jn).
 *
 * This file is part of SFSV.
 *
 * SFSV is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * SFSV is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "types.h"
#include "sfsv.h"
#include "sha1.h"


#define PGMNAME "elfsigtool: "

char *
read_file( const char *filename, size_t *imagelen )
{
    FILE *fp = fopen( filename, "rb" );
    char *buffer;
    size_t n;
    struct stat st;

    if( !fp ) {
	fprintf(stderr,PGMNAME "can't open `%s': %s\n", filename, strerror(errno) );
	exit(1);
    }
    if( fstat(fileno(fp), &st) ) {
	fprintf(stderr, PGMNAME "fstat() failed: %s\n", strerror(errno) );
	exit(1);
    }

    n = st.st_size;
    buffer = malloc( n );
    if( !buffer ) {
	fprintf(stderr, PGMNAME "malloc failed - file too large?\n" );
	exit(1);
    }

    if( fread( buffer, n, 1, fp) != 1 ) {
	fprintf(stderr,PGMNAME "error reading `%s': %s\n", filename, strerror(errno) );
	free( buffer );
	exit(1);
    }

    fclose(fp );
    *imagelen = n;
    return buffer;
}



int
main( int argc, char **argv )
{
    SHA1_CONTEXT hash;
    int rc;
    char *image;
    size_t imagelen, sigoff, siglen;

    if( argc )	{
	argc--;
	argv++;
    }
    if( argc < 1 || argc > 2 ) {
	fprintf(stderr,"usage: elfsigtool elfobject [signature]\n");
	exit(1);
    }

    /* read the ELF object into memory */
    image = read_file( argv[0], &imagelen );
    /* and locate the special rfc2440 note */
    rc = sfsv_locate_elf_signature( image, imagelen, &sigoff, &siglen );
    if( rc ) {
	fprintf(stderr, PGMNAME "signature .note not found: rc=%d\n", rc );
	exit(1);
    }
    fprintf(stderr, PGMNAME "signature .note found at off=%lu size=%lu\n",
			    (ulong)sigoff, (ulong)siglen );

    /* now cat it */
    fwrite( image, sigoff, 1, stdout );
    if( argc > 1 ) { /* we have the file with the signature */
	char  *sfile;
	size_t slen;

	sfile = read_file( argv[1], &slen );
	fprintf(stderr, PGMNAME "replacing with signature of size=%lu\n", slen);
	if( slen > siglen ) {
	    fprintf(stderr,PGMNAME "not enough room for signature\n");
	    exit(1);
	}
	fwrite( sfile, slen, 1, stdout );
	free( sfile );
	for( ; slen < siglen; slen++ )
	    putchar( 0 );
    }
    if( fwrite( image+sigoff+siglen, imagelen-sigoff-siglen, 1, stdout ) != 1 ) {
	fprintf(stderr,PGMNAME "error writing to stdout: %s\n", strerror(errno));
	exit(1);
    }


    return 0;
}


