/* elfsigchk.c - Utility to check a rfc2440 signed ELF file
 *	Copyright (C) 1999 Werner Koch (dd9jn).
 *
 * This file is part of SFSV.
 *
 * SFSV is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * SFSV is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "types.h"
#include "sfsv.h"
#include "sha1.h"


#define PGMNAME "elfsigchk: "

char *
read_file( const char *filename, size_t *imagelen )
{
    FILE *fp = fopen( filename, "rb" );
    char *buffer;
    size_t n;
    struct stat st;

    if( !fp ) {
	fprintf(stderr,PGMNAME "can't open `%s': %s\n", filename, strerror(errno) );
	exit(1);
    }
    if( fstat(fileno(fp), &st) ) {
	fprintf(stderr, PGMNAME "fstat() failed: %s\n", strerror(errno) );
	exit(1);
    }

    n = st.st_size;
    buffer = malloc( n );
    if( !buffer ) {
	fprintf(stderr, PGMNAME "malloc failed - file too large?\n" );
	exit(1);
    }

    if( fread( buffer, n, 1, fp) != 1 ) {
	fprintf(stderr,PGMNAME "error reading `%s': %s\n", filename, strerror(errno) );
	free( buffer );
	exit(1);
    }

    fclose(fp );
    *imagelen = n;
    return buffer;
}



int
main( int argc, char **argv )
{
    SHA1_CONTEXT hash;
    int rc;
    char *image;
    size_t imagelen, sigoff, siglen;

    if( argc )	{
	argc--;
	argv++;
    }
    if( argc < 1 || argc > 2 ) {
	fprintf(stderr,"usage: elfsigchk elfobject [signature]\n");
	exit(1);
    }

    /* read the ELF object into memory */
    image = read_file( argv[0], &imagelen );
    /* and locate the special rfc2440 note */
    rc = sfsv_verify_elf_signature( image, imagelen);
  #ifdef ENABLE_DEBUG
    bi_stat();
  #endif
    if( rc ) {
	printf( "signature is NOT valid: %s\n",
		rc == SFSV_BAD_SIG ? "bad signature"
	      : rc == SFSV_NO_KEY  ? "public key not available"
	      : rc == SFSV_NO_SIG  ? "no signature found"
	      : rc == SFSV_BAD_ELF ? "invalid elf object"
	      : rc == SFSV_BAD_PGP ? "invalid signature packet"
	      : "unknown reason" );
	exit(1);
    }
    printf( "signature is valid.\n" );
    return 0;
}


